//---------------------------------------------------------------------------
// Copyright (C) 1999 Dallas Semiconductor Corporation, All Rights Reserved.
// 
// Permission is hereby granted, free of charge, to any person obtaining a 
// copy of this software and associated documentation files (the "Software"), 
// to deal in the Software without restriction, including without limitation 
// the rights to use, copy, modify, merge, publish, distribute, sublicense, 
// and/or sell copies of the Software, and to permit persons to whom the 
// Software is furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included 
// in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS 
// OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
// MERCHANTABILITY,  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. 
// IN NO EVENT SHALL DALLAS SEMICONDUCTOR BE LIABLE FOR ANY CLAIM, DAMAGES 
// OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, 
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR 
// OTHER DEALINGS IN THE SOFTWARE.
// 
// Except as contained in this notice, the name of Dallas Semiconductor 
// shall not be used except as stated in the Dallas Semiconductor 
// Branding Policy. 
//---------------------------------------------------------------------------
//
//  tstmlan.C - Test application to test MicroLan functions. No EPROM writes.
//
//  Version: 1.03
//
//  History: 1.00 -> 1.01  Change to use msDelay instead of Sleep. 
//
//           1.01 -> 1.02  Changed to generic OpenCOM/CloseCOM for easier 
//                           use with other platforms.  
//           1.02 -> 1.03  Removed caps in #includes for Linux capatibility
//                         Changed to use Aquire/Release 1-Wire Net functions
//

#include <stdio.h>
#include <stdlib.h>
#include "mlan.h"

// external MLan functions to test
extern int  MLanTouchByte(int);
extern int  MLanFirst(int,int);
extern int  MLanNext(int,int);
extern void MLanSerialNum(uchar *, int);
extern void MLanFamilySearchSetup(int);
extern void MLanSkipFamily(void);
extern int  MLanAccess(void);
extern int  MLanOverdriveAccess(void);
extern int  MLanVerify(int);
extern int  MLanBlock(int, uchar *, int);
extern int  MLanReadPacketStd(int, int, uchar *);
extern int  MLanWritePacketStd(int, uchar *, int, int, int);   
extern int  MLanSpeed(int);
extern int  MLanLevel(int);
extern int  MLanReadFile(uchar *, uchar *);
extern int  MLanFormatWriteFile(uchar *, int, uchar *);
extern int  MLanProgramPulse(void);
extern void msDelay(int);
extern int  Aquire1WireNet(char *, char *);
extern void Release1WireNet(char *);

// local funcitons
void PrintSerialNum(void);

//----------------------------------------------------------------------
//  Main Test
//
void main(short argc, char **argv)
{
   int PortNum=1,rslt,i,j,testcnt=0,length;
   uchar TempSerialNum[8];
   uchar tran_buffer[2000], filename[10];
   char return_msg[128];

   // check for required port name
   if (argc != 2)
   {
      printf("1-Wire Net name required on command line!\n"
             " (example: \"COM1\" (Win32 DS2480),\"/dev/cua0\" "
             "(Linux DS2480),\"1\" (Win32 TMEX)\n");
      exit(1);
   }

   // attempt to aquire the 1-Wire Net
   if (!Aquire1WireNet(argv[1], return_msg))
   {  
      printf("%s",return_msg);
      exit(1);
   }

   // success
   printf("%s",return_msg);

   //----------------------------------------
   // Introduction
   printf("\n/---------------------------------------------\n");
   printf("  The following is a test excersize of the\n"
          "  MicroLAN public domain library Version 1.03.\n\n"
          "  This test was run using with 2 DS1920's (DS1820),\n"
          "  1 DS1971 (DS2430), and 1 DS1996.\n\n");

   //----------------------------------------
   // First the devices on the MicroLAN
   printf("\n/---------------------------------------------\n");
   printf("TEST%d: Searching for devices on MicroLAN\n",testcnt++);

   // find the first device (all devices not just alarming)
   rslt = MLanFirst(TRUE, FALSE);
   while (rslt)
   {
      // print the Serial Number of the device just found
      PrintSerialNum();

      // find the next device
      rslt = MLanNext(TRUE, FALSE);
   }

   //----------------------------------------
   // now search for the part with a 0x0C family code (DS1996)
   printf("\n/---------------------------------------------\n");
   printf("TEST%d: Set to find first device with 0x0C family code\n",testcnt++);

   MLanFamilySearchSetup(0x0C);

   // find the next device
   printf("search result %d\n",MLanNext(TRUE, FALSE));

   // print the Serial Number of the device just found
   PrintSerialNum();
   
   //----------------------------------------
   // Access a device and read ram
   printf("\n/---------------------------------------------\n");
   printf("TEST%d: Access the current device and read ram\n",testcnt++);

   printf("MLanAccess %d\n",MLanAccess());

   printf("Read Ram 0xF0: %02X\n",MLanTouchByte(0xF0));
   printf("Address0 0x00: %02X\n",MLanTouchByte(0x00));
   printf("Address1 0x00: %02X\n",MLanTouchByte(0x00));

   printf("Page 0: ");
   for (i = 0; i < 32; i++)
      printf("%02X ",MLanTouchByte(0xFF));
   printf("\n");

   //----------------------------------------
   // Read ram with MLanBlock
   printf("\n/---------------------------------------------\n");
   printf("TEST%d: Read ram with MLanBlock\n",testcnt++);
   for (i = 0; i < 32; i++)
      tran_buffer[i] = 0xFF;

   printf("MLanBlock %d\n",MLanBlock(FALSE,tran_buffer,32));
   printf("Page 1: ");
   for (i = 0; i < 32; i++)
      printf("%02X ",tran_buffer[i]);
   printf("\n");

   //----------------------------------------
   // Write a packet in each page of DS1996
   printf("\n/---------------------------------------------\n");
   printf("TEST%d: Place the DS1996 into overdrive\n",testcnt++);
   printf("MLanOverdriveAccess %d\n",MLanOverdriveAccess());

   //----------------------------------------
   // Write 4 packets with MLanWritePacketStd 
   printf("\n/---------------------------------------------\n");
   printf("TEST%d: Write 4 packets with MLanWritePacketStd\n",testcnt++);
     
   for (j = 0; j < 4; j++)
   {
      for (i = 0; i < 29; i++)
         tran_buffer[i] = (uchar)i + j;

      printf("Write page %d: %d\n",j,MLanWritePacketStd(j,tran_buffer,29,FALSE,FALSE));

      for (i = 0; i < 29; i++)
         tran_buffer[i] = 0;
   
      length = MLanReadPacketStd(TRUE, j,tran_buffer);

      printf("Read page %d: %d\n",j,length);

      for (i = 0; i < length; i++)
         printf("%02X",tran_buffer[i]);
      printf("\n");
   }

   //----------------------------------------
   // Write a file to DS1996
   printf("\n/---------------------------------------------\n");
   printf("TEST%d: Format and write a file (in overdrive)\n",testcnt++);
   sprintf(filename,"DEMO");
   // set the data to write
   for (i = 0; i < 2000; i++)
      tran_buffer[i] = i % 255;
   printf("Format and write file DEMO.000 %d\n",MLanFormatWriteFile(filename, 2000, tran_buffer));
   // clear the buffer
   for (i = 0; i < 2000; i++)
      tran_buffer[i] = 0;
   printf("Read file DEMO.000 %d\n",MLanReadFile(filename, tran_buffer));
   // print the data result
   for (i = 0; i < 2000; i++)
   {
      if ((i % 0x20) == 0)
         printf("\n%03X    ",i);
      printf("%02X",tran_buffer[i]);
   }
   printf("\n");
  
   //----------------------------------------
   // Turn off overdrive
   printf("\n/---------------------------------------------\n");
   printf("TEST%d: Turn off overdrive\n",testcnt++);
   printf("Set MicroLAN speed to normal %d\n",MLanSpeed(MODE_NORMAL));

   //----------------------------------------
   // Verify a device
   printf("\n/---------------------------------------------\n");
   printf("TEST%d: Verify the current device\n",testcnt++);

   printf("MLanVerify (normal) %d\n",MLanVerify(FALSE));
   printf("MLanVerify (alarm)  %d\n",MLanVerify(TRUE));

   //----------------------------------------
   // Skip the first family code found
   printf("\n/---------------------------------------------\n");
   printf("TEST%d: Skip the first family code found\n",testcnt++);
   
   // find the next device
   printf("search result of MLanFirst %d\n",MLanFirst(TRUE, FALSE));

   // print the Serial Number of the device just found
   PrintSerialNum();

   // skip the first family type found
   MLanSkipFamily();
   printf("MLanSkipFamily called\n");

   // find the next device
   printf("search result of MLanNext %d\n",MLanNext(TRUE, FALSE));
   
   // print the Serial Number of the device just found
   PrintSerialNum();

   //----------------------------------------
   // Find first family code (DS1920) and read temperature
   printf("\n/---------------------------------------------\n");
   printf("TEST%d: Find first family code (DS1920) and read temperature\n",testcnt++);

   // find the next device
   printf("search result of MLanFirst %d\n",MLanFirst(TRUE, FALSE));

   // print the Serial Number of the device just found
   PrintSerialNum();

   // send the convert temperature command
   printf("Convert temperature command %02X\n",MLanTouchByte(0x44));

   // set the MicroLAN to strong pull-up
   printf("Set power delivery %d\n",MLanLevel(MODE_STRONG5));

   // sleep for 1 second
   msDelay(1000);

   // turn off the MicroLAN strong pull-up
   printf("Disable power delivery %d\n",MLanLevel(MODE_NORMAL));

   // read the DS1920 temperature value
   printf("Access the DS1920 %d\n",MLanAccess());
   tran_buffer[0] = 0xBE;
   tran_buffer[1] = 0xFF;
   tran_buffer[2] = 0xFF;
   printf("Block to read temperature %d\n",MLanBlock(FALSE,tran_buffer,3));
   // interpret the result
   printf("result: DS1920 temperature read: %d C\n", (tran_buffer[1] |
           (tran_buffer[2] << 8)) / 2);
  
   //----------------------------------------
   //  Verify the current device, could also be alarming
   printf("\n/---------------------------------------------\n");
   printf("TEST%d: Verify the current device, could also be alarming\n",testcnt++);

   printf("MLanVerify (normal) %d\n",MLanVerify(FALSE));
   printf("MLanVerify (alarm)  %d\n",MLanVerify(TRUE));

   //----------------------------------------
   // Test setting the Serial Number with MLanSerialNum
   printf("\n/---------------------------------------------\n");
   printf("TEST%d: Test setting the Serial Number with MLanSerialNum\n",testcnt++);

   // set the Serial Num to 0 to 7
   for (i = 0; i < 8; i++)
      TempSerialNum[i] = (uchar)i;
   MLanSerialNum(TempSerialNum,FALSE);

   // read back the Serial Number 
   PrintSerialNum();

   //----------------------------------------
   //  Verify the current device (should fail, no such device)
   printf("\n/---------------------------------------------\n");
   printf("TEST%d: Verify the current device (should fail, no such device)\n",testcnt++);

   printf("MLanVerify (normal) %d\n",MLanVerify(FALSE));
   printf("MLanVerify (alarm)  %d\n",MLanVerify(TRUE));

   // release the 1-Wire Net
   Release1WireNet(return_msg);
   printf("%s",return_msg);
   exit(0);
}


//----------------------------------------------------------------------
//  Read and print the Serial Number.
//
void PrintSerialNum(void)
{
   uchar TempSerialNumber[8];
   int i;

   MLanSerialNum(TempSerialNumber,TRUE);
   for (i = 7; i >= 0; i--)
      printf("%02X",TempSerialNumber[i]);
   printf("\n");
}
