#include "emu.h"
#include "video/konicdev.h"
#include "includes/tmnt.h"

TILE_GET_INFO_MEMBER(tmnt_state::glfgreat_get_roz_tile_info)
{
	UINT8 *rom = memregion("user1")->base();
	int code;

	tile_index += 0x40000 * m_glfgreat_roz_rom_bank;

	code = rom[tile_index + 0x80000] + 256 * rom[tile_index] + 256 * 256 * ((rom[tile_index / 4 + 0x100000] >> (2 * (tile_index & 3))) & 3);

	SET_TILE_INFO_MEMBER(0, code & 0x3fff, code >> 14, 0);
}

TILE_GET_INFO_MEMBER(tmnt_state::prmrsocr_get_roz_tile_info)
{
	UINT8 *rom = memregion("user1")->base();
	int code = rom[tile_index + 0x20000] + 256 * rom[tile_index];

	SET_TILE_INFO_MEMBER(0, code & 0x1fff, code >> 13, 0);
}



/***************************************************************************

  Callbacks for the K052109

***************************************************************************/

/* Missing in Action */

void mia_tile_callback( running_machine &machine, int layer, int bank, int *code, int *color, int *flags, int *priority )
{
	tmnt_state *state = machine.driver_data<tmnt_state>();
	*flags = (*color & 0x04) ? TILE_FLIPX : 0;
	if (layer == 0)
	{
		*code |= ((*color & 0x01) << 8);
		*color = state->m_layer_colorbase[layer] + ((*color & 0x80) >> 5) + ((*color & 0x10) >> 1);
	}
	else
	{
		*code |= ((*color & 0x01) << 8) | ((*color & 0x18) << 6) | (bank << 11);
		*color = state->m_layer_colorbase[layer] + ((*color & 0xe0) >> 5);
	}
}

void cuebrick_tile_callback( running_machine &machine, int layer, int bank, int *code, int *color, int *flags, int *priority )
{
	tmnt_state *state = machine.driver_data<tmnt_state>();

	if ((k052109_get_rmrd_line(state->m_k052109) == CLEAR_LINE) && (layer == 0))
	{
		*code |= ((*color & 0x01) << 8);
		*color = state->m_layer_colorbase[layer]  + ((*color & 0x80) >> 5) + ((*color & 0x10) >> 1);
	}
	else
	{
		*code |= ((*color & 0xf) << 8);
		*color = state->m_layer_colorbase[layer] + ((*color & 0xe0) >> 5);
	}
}

void tmnt_tile_callback( running_machine &machine, int layer, int bank, int *code, int *color, int *flags, int *priority )
{
	tmnt_state *state = machine.driver_data<tmnt_state>();
	*code |= ((*color & 0x03) << 8) | ((*color & 0x10) << 6) | ((*color & 0x0c) << 9) | (bank << 13);
	*color = state->m_layer_colorbase[layer] + ((*color & 0xe0) >> 5);
}

void ssbl_tile_callback( running_machine &machine, int layer, int bank, int *code, int *color, int *flags, int *priority )
{
	tmnt_state *state = machine.driver_data<tmnt_state>();
	if (layer == 0)
	{
		*code |= ((*color & 0x03) << 8) | ((*color & 0x10) << 6) | ((*color & 0x0c) << 9) | (bank << 13);
	}
	else
	{
		*code |= ((*color & 0x03) << 8) | ((*color & 0x10) << 6) | ((*color & 0x0c) << 9) | (bank << 13);
//      mame_printf_debug("L%d: bank %d code %x color %x\n", layer, bank, *code, *color);
	}

	*color = state->m_layer_colorbase[layer] + ((*color & 0xe0) >> 5);
}

void blswhstl_tile_callback( running_machine &machine, int layer, int bank, int *code, int *color, int *flags, int *priority )
{
	tmnt_state *state = machine.driver_data<tmnt_state>();

	/* (color & 0x02) is flip y handled internally by the 052109 */
	*code |= ((*color & 0x01) << 8) | ((*color & 0x10) << 5) | ((*color & 0x0c) << 8) | (bank << 12) | state->m_blswhstl_rombank << 14;
	*color = state->m_layer_colorbase[layer] + ((*color & 0xe0) >> 5);
}



/***************************************************************************

  Callbacks for the K051960

***************************************************************************/

void mia_sprite_callback( running_machine &machine, int *code, int *color, int *priority, int *shadow )
{
	tmnt_state *state = machine.driver_data<tmnt_state>();
	*color = state->m_sprite_colorbase + (*color & 0x0f);
}

void tmnt_sprite_callback( running_machine &machine, int *code, int *color, int *priority, int *shadow )
{
	tmnt_state *state = machine.driver_data<tmnt_state>();
	*code |= (*color & 0x10) << 9;
	*color = state->m_sprite_colorbase + (*color & 0x0f);
}

void punkshot_sprite_callback( running_machine &machine, int *code, int *color, int *priority_mask, int *shadow )
{
	tmnt_state *state = machine.driver_data<tmnt_state>();
	int pri = 0x20 | ((*color & 0x60) >> 2);
	if (pri <= state->m_layerpri[2])
		*priority_mask = 0;
	else if (pri > state->m_layerpri[2] && pri <= state->m_layerpri[1])
		*priority_mask = 0xf0;
	else if (pri > state->m_layerpri[1] && pri <= state->m_layerpri[0])
		*priority_mask = 0xf0 | 0xcc;
	else
		*priority_mask = 0xf0 | 0xcc | 0xaa;

	*code |= (*color & 0x10) << 9;
	*color = state->m_sprite_colorbase + (*color & 0x0f);
}

void thndrx2_sprite_callback( running_machine &machine, int *code, int *color, int *priority_mask, int *shadow )
{
	tmnt_state *state = machine.driver_data<tmnt_state>();
	int pri = 0x20 | ((*color & 0x60) >> 2);
	if (pri <= state->m_layerpri[2])
		*priority_mask = 0;
	else if (pri > state->m_layerpri[2] && pri <= state->m_layerpri[1])
		*priority_mask = 0xf0;
	else if (pri > state->m_layerpri[1] && pri <= state->m_layerpri[0])
		*priority_mask = 0xf0 | 0xcc;
	else
		*priority_mask = 0xf0 | 0xcc | 0xaa;

	*color = state->m_sprite_colorbase + (*color & 0x0f);
}


/***************************************************************************

  Callbacks for the K053245

***************************************************************************/

void lgtnfght_sprite_callback( running_machine &machine, int *code, int *color, int *priority_mask )
{
	tmnt_state *state = machine.driver_data<tmnt_state>();
	int pri = 0x20 | ((*color & 0x60) >> 2);
	if (pri <= state->m_layerpri[2])
		*priority_mask = 0;
	else if (pri > state->m_layerpri[2] && pri <= state->m_layerpri[1])
		*priority_mask = 0xf0;
	else if (pri > state->m_layerpri[1] && pri <= state->m_layerpri[0])
		*priority_mask = 0xf0 | 0xcc;
	else
		*priority_mask = 0xf0 | 0xcc | 0xaa;

	*color = state->m_sprite_colorbase + (*color & 0x1f);
}

void blswhstl_sprite_callback( running_machine &machine, int *code, int *color, int *priority_mask )
{
	tmnt_state *state = machine.driver_data<tmnt_state>();
#if 0
if (machine.input().code_pressed(KEYCODE_Q) && (*color & 0x20)) *color = rand();
if (machine.input().code_pressed(KEYCODE_W) && (*color & 0x40)) *color = rand();
if (machine.input().code_pressed(KEYCODE_E) && (*color & 0x80)) *color = rand();
#endif
	int pri = 0x20 | ((*color & 0x60) >> 2);
	if (pri <= state->m_layerpri[2])
		*priority_mask = 0;
	else if (pri > state->m_layerpri[2] && pri <= state->m_layerpri[1])
		*priority_mask = 0xf0;
	else if (pri > state->m_layerpri[1] && pri <= state->m_layerpri[0])
		*priority_mask = 0xf0 | 0xcc;
	else
		*priority_mask = 0xf0 | 0xcc | 0xaa;

	*color = state->m_sprite_colorbase + (*color & 0x1f);
}

void prmrsocr_sprite_callback( running_machine &machine, int *code, int *color, int *priority_mask )
{
	tmnt_state *state = machine.driver_data<tmnt_state>();
	int pri = 0x20 | ((*color & 0x60) >> 2);
	if (pri <= state->m_layerpri[2])
		*priority_mask = 0;
	else if (pri > state->m_layerpri[2] && pri <= state->m_layerpri[1])
		*priority_mask = 0xf0;
	else if (pri > state->m_layerpri[1] && pri <= state->m_layerpri[0])
		*priority_mask = 0xf0 | 0xcc;
	else
		*priority_mask = 0xf0 | 0xcc | 0xaa;

	*code |= state->m_prmrsocr_sprite_bank << 14;

	*color = state->m_sprite_colorbase + (*color & 0x1f);
}



/***************************************************************************

  Start the video hardware emulation.

***************************************************************************/

VIDEO_START_MEMBER(tmnt_state,cuebrick)
{
	m_layer_colorbase[0] = 0;
	m_layer_colorbase[1] = 32;
	m_layer_colorbase[2] = 40;
	m_sprite_colorbase = 16;
}

VIDEO_START_MEMBER(tmnt_state,mia)
{
	m_layer_colorbase[0] = 0;
	m_layer_colorbase[1] = 32;
	m_layer_colorbase[2] = 40;
	m_sprite_colorbase = 16;

	m_tmnt_priorityflag = 0;
	save_item(NAME(m_tmnt_priorityflag));
}

VIDEO_START_MEMBER(tmnt_state,tmnt)
{
	m_layer_colorbase[0] = 0;
	m_layer_colorbase[1] = 32;
	m_layer_colorbase[2] = 40;
	m_sprite_colorbase = 16;

	m_tmnt_priorityflag = 0;
	save_item(NAME(m_tmnt_priorityflag));

	palette_set_shadow_factor(machine(),0.75);
}

VIDEO_START_MEMBER(tmnt_state,lgtnfght)/* also tmnt2, ssriders */
{
	k05324x_set_z_rejection(m_k053245, 0);

	m_dim_c = m_dim_v = m_lastdim = m_lasten = 0;

	save_item(NAME(m_dim_c));
	save_item(NAME(m_dim_v));
	save_item(NAME(m_lastdim));
	save_item(NAME(m_lasten));
}

VIDEO_START_MEMBER(tmnt_state,glfgreat)
{
	m_roz_tilemap = &machine().tilemap().create(tilemap_get_info_delegate(FUNC(tmnt_state::glfgreat_get_roz_tile_info),this), TILEMAP_SCAN_ROWS, 16, 16, 512, 512);
	m_roz_tilemap->set_transparent_pen(0);

	m_glfgreat_roz_rom_bank = 0;
	m_glfgreat_roz_char_bank = 0;
	m_glfgreat_roz_rom_mode = 0;
	save_item(NAME(m_glfgreat_roz_rom_bank));
	save_item(NAME(m_glfgreat_roz_char_bank));
	save_item(NAME(m_glfgreat_roz_rom_mode));
}

VIDEO_START_MEMBER(tmnt_state,prmrsocr)
{
	m_roz_tilemap = &machine().tilemap().create(tilemap_get_info_delegate(FUNC(tmnt_state::prmrsocr_get_roz_tile_info),this), TILEMAP_SCAN_ROWS, 16, 16, 512, 256);
	m_roz_tilemap->set_transparent_pen(0);

	m_prmrsocr_sprite_bank = 0;
	m_glfgreat_roz_char_bank = 0;
	save_item(NAME(m_prmrsocr_sprite_bank));
	save_item(NAME(m_glfgreat_roz_char_bank));
}

VIDEO_START_MEMBER(tmnt_state,blswhstl)
{
	m_blswhstl_rombank = -1;
	save_item(NAME(m_blswhstl_rombank));
}


/***************************************************************************

  Memory handlers

***************************************************************************/

WRITE16_MEMBER(tmnt_state::tmnt_paletteram_word_w)
{
	COMBINE_DATA(m_generic_paletteram_16 + offset);
	offset &= ~1;

	data = (m_generic_paletteram_16[offset] << 8) | m_generic_paletteram_16[offset + 1];
	palette_set_color_rgb(machine(), offset / 2, pal5bit(data >> 0), pal5bit(data >> 5), pal5bit(data >> 10));
}



WRITE16_MEMBER(tmnt_state::tmnt_0a0000_w)
{
	if (ACCESSING_BITS_0_7)
	{
		/* bit 0/1 = coin counters */
		coin_counter_w(machine(), 0, data & 0x01);
		coin_counter_w(machine(), 1, data & 0x02);  /* 2 players version */

		/* bit 3 high then low triggers irq on sound CPU */
		if (m_last == 0x08 && (data & 0x08) == 0)
			m_audiocpu->set_input_line_and_vector(0, HOLD_LINE, 0xff);

		m_last = data & 0x08;

		/* bit 5 = irq enable */
		m_irq5_mask = data & 0x20;

		/* bit 7 = enable char ROM reading through the video RAM */
		k052109_set_rmrd_line(m_k052109, (data & 0x80) ? ASSERT_LINE : CLEAR_LINE);

		/* other bits unused */
	}
}

WRITE16_MEMBER(tmnt_state::punkshot_0a0020_w)
{
	if (ACCESSING_BITS_0_7)
	{
		/* bit 0 = coin counter */
		coin_counter_w(machine(), 0, data & 0x01);

		/* bit 2 = trigger irq on sound CPU */
		if (m_last == 0x04 && (data & 0x04) == 0)
			m_audiocpu->set_input_line_and_vector(0, HOLD_LINE, 0xff);

		m_last = data & 0x04;

		/* bit 3 = enable char ROM reading through the video RAM */
		k052109_set_rmrd_line(m_k052109, (data & 0x08) ? ASSERT_LINE : CLEAR_LINE);
	}
}

WRITE16_MEMBER(tmnt_state::lgtnfght_0a0018_w)
{
	if (ACCESSING_BITS_0_7)
	{
		/* bit 0,1 = coin counter */
		coin_counter_w(machine(), 0, data & 0x01);
		coin_counter_w(machine(), 1, data & 0x02);

		/* bit 2 = trigger irq on sound CPU */
		if (m_last == 0x00 && (data & 0x04) == 0x04)
			m_audiocpu->set_input_line_and_vector(0, HOLD_LINE, 0xff);

		m_last = data & 0x04;

		/* bit 3 = enable char ROM reading through the video RAM */
		k052109_set_rmrd_line(m_k052109, (data & 0x08) ? ASSERT_LINE : CLEAR_LINE);
	}
}

WRITE16_MEMBER(tmnt_state::blswhstl_700300_w)
{
	if (ACCESSING_BITS_0_7)
	{
		/* bit 0,1 = coin counter */
		coin_counter_w(machine(), 0,data & 0x01);
		coin_counter_w(machine(), 1,data & 0x02);

		/* bit 3 = enable char ROM reading through the video RAM */
		k052109_set_rmrd_line(m_k052109, (data & 0x08) ? ASSERT_LINE : CLEAR_LINE);

		/* bit 7 = select char ROM bank */
		if (m_blswhstl_rombank != ((data & 0x80) >> 7))
		{
			m_blswhstl_rombank = (data & 0x80) >> 7;
			machine().tilemap().mark_all_dirty();
		}

		/* other bits unknown */
	}
}


READ16_MEMBER(tmnt_state::glfgreat_rom_r)
{
	if (m_glfgreat_roz_rom_mode)
		return memregion("gfx3")->base()[m_glfgreat_roz_char_bank * 0x80000 + offset];
	else if (offset < 0x40000)
	{
		UINT8 *usr = memregion("user1")->base();
		return usr[offset + 0x80000 + m_glfgreat_roz_rom_bank * 0x40000] + 256 * usr[offset + m_glfgreat_roz_rom_bank * 0x40000];
	}
	else
		return memregion("user1")->base()[((offset & 0x3ffff) >> 2) + 0x100000 + m_glfgreat_roz_rom_bank * 0x10000];
}

WRITE16_MEMBER(tmnt_state::glfgreat_122000_w)
{
	if (ACCESSING_BITS_0_7)
	{
		/* bit 0,1 = coin counter */
		coin_counter_w(machine(), 0, data & 0x01);
		coin_counter_w(machine(), 1, data & 0x02);

		/* bit 4 = enable char ROM reading through the video RAM */
		k052109_set_rmrd_line(m_k052109, (data & 0x10) ? ASSERT_LINE : CLEAR_LINE);

		/* bit 5 = 53596 tile rom bank selection */
		if (m_glfgreat_roz_rom_bank != (data & 0x20) >> 5)
		{
			m_glfgreat_roz_rom_bank = (data & 0x20) >> 5;
			m_roz_tilemap->mark_all_dirty();
		}

		/* bit 6,7 = 53596 char bank selection for ROM test */
		m_glfgreat_roz_char_bank = (data & 0xc0) >> 6;

		/* other bits unknown */
	}
	if (ACCESSING_BITS_8_15)
	{
		/* bit 8 = 53596 char/rom selection for ROM test */
		m_glfgreat_roz_rom_mode = data & 0x100;
	}
}


WRITE16_MEMBER(tmnt_state::ssriders_eeprom_w)
{
	if (ACCESSING_BITS_0_7)
	{
		/* bit 0 is data */
		/* bit 1 is cs (active low) */
		/* bit 2 is clock (active high) */
		ioport("EEPROMOUT")->write(data, 0xff);

		/* bits 3-4 control palette dimming */
		/* 4 = DIMPOL = when set, negate SHAD */
		/* 3 = DIMMOD = when set, or BRIT with [negated] SHAD */
		m_dim_c = data & 0x18;

		/* bit 5 selects sprite ROM for testing in TMNT2 (bits 5-7, actually, according to the schematics) */
		k053244_bankselect(m_k053245, ((data & 0x20) >> 5) << 2);
	}
}

WRITE16_MEMBER(tmnt_state::ssriders_1c0300_w)
{
	if (ACCESSING_BITS_0_7)
	{
		/* bit 0,1 = coin counter */
		coin_counter_w(machine(), 0, data & 0x01);
		coin_counter_w(machine(), 1, data & 0x02);

		/* bit 3 = enable char ROM reading through the video RAM */
		k052109_set_rmrd_line(m_k052109, (data & 0x08) ? ASSERT_LINE : CLEAR_LINE);

		/* bits 4-6 control palette dimming (DIM0-DIM2) */
		m_dim_v = (data & 0x70) >> 4;
	}
}

WRITE16_MEMBER(tmnt_state::prmrsocr_122000_w)
{
	if (ACCESSING_BITS_0_7)
	{
		/* bit 0,1 = coin counter */
		coin_counter_w(machine(), 0, data & 0x01);
		coin_counter_w(machine(), 1, data & 0x02);

		/* bit 4 = enable char ROM reading through the video RAM */
		k052109_set_rmrd_line(m_k052109, (data & 0x10) ? ASSERT_LINE : CLEAR_LINE);

		/* bit 6 = sprite ROM bank */
		m_prmrsocr_sprite_bank = (data & 0x40) >> 6;
		k053244_bankselect(m_k053245, m_prmrsocr_sprite_bank << 2);

		/* bit 7 = 53596 region selector for ROM test */
		m_glfgreat_roz_char_bank = (data & 0x80) >> 7;

		/* other bits unknown (unused?) */
	}
}

READ16_MEMBER(tmnt_state::prmrsocr_rom_r)
{
	if(m_glfgreat_roz_char_bank)
		return memregion("gfx3")->base()[offset];
	else
	{
		UINT8 *usr = memregion("user1")->base();
		return 256 * usr[offset] + usr[offset + 0x020000];
	}
}

WRITE16_MEMBER(tmnt_state::tmnt_priority_w)
{
	if (ACCESSING_BITS_0_7)
	{
		/* bit 2/3 = priority; other bits unused */
		/* bit2 = PRI bit3 = PRI2
		      sprite/playfield priority is controlled by these two bits, by bit 3
		      of the background tile color code, and by the SHADOW sprite
		      attribute bit.
		      Priorities are encoded in a PROM (G19 for TMNT). However, in TMNT,
		      the PROM only takes into account the PRI and SHADOW bits.
		      PRI  Priority
		       0   bg fg spr text
		       1   bg spr fg text
		      The SHADOW bit, when set, torns a sprite into a shadow which makes
		      color below it darker (this is done by turning off three resistors
		      in parallel with the RGB output).

		      Note: the background color (color used when all of the four layers
		      are 0) is taken from the *foreground* palette, not the background
		      one as would be more intuitive.
		*/
		m_tmnt_priorityflag = (data & 0x0c) >> 2;
	}
}



/***************************************************************************

  Display refresh

***************************************************************************/

UINT32 tmnt_state::screen_update_mia(screen_device &screen, bitmap_ind16 &bitmap, const rectangle &cliprect)
{
	k052109_tilemap_update(m_k052109);

	k052109_tilemap_draw(m_k052109, bitmap, cliprect, 2, TILEMAP_DRAW_OPAQUE,0);
	if ((m_tmnt_priorityflag & 1) == 1) k051960_sprites_draw(m_k051960, bitmap, cliprect, 0, 0);
	k052109_tilemap_draw(m_k052109, bitmap, cliprect, 1, 0, 0);
	if ((m_tmnt_priorityflag & 1) == 0) k051960_sprites_draw(m_k051960, bitmap, cliprect, 0, 0);
	k052109_tilemap_draw(m_k052109, bitmap, cliprect, 0, 0, 0);

	return 0;
}

UINT32 tmnt_state::screen_update_tmnt(screen_device &screen, bitmap_ind16 &bitmap, const rectangle &cliprect)
{
	k052109_tilemap_update(m_k052109);

	k052109_tilemap_draw(m_k052109, bitmap, cliprect, 2, TILEMAP_DRAW_OPAQUE,0);
	if ((m_tmnt_priorityflag & 1) == 1) k051960_sprites_draw(m_k051960, bitmap, cliprect, 0, 0);
	k052109_tilemap_draw(m_k052109, bitmap, cliprect, 1, 0, 0);
	if ((m_tmnt_priorityflag & 1) == 0) k051960_sprites_draw(m_k051960, bitmap, cliprect, 0, 0);
	k052109_tilemap_draw(m_k052109, bitmap, cliprect, 0, 0, 0);

	return 0;
}


UINT32 tmnt_state::screen_update_punkshot(screen_device &screen, bitmap_ind16 &bitmap, const rectangle &cliprect)
{
	m_sprite_colorbase = k053251_get_palette_index(m_k053251, K053251_CI1);
	m_layer_colorbase[0] = k053251_get_palette_index(m_k053251, K053251_CI2);
	m_layer_colorbase[1] = k053251_get_palette_index(m_k053251, K053251_CI4);
	m_layer_colorbase[2] = k053251_get_palette_index(m_k053251, K053251_CI3);

	k052109_tilemap_update(m_k052109);

	m_sorted_layer[0] = 0;
	m_layerpri[0] = k053251_get_priority(m_k053251, K053251_CI2);
	m_sorted_layer[1] = 1;
	m_layerpri[1] = k053251_get_priority(m_k053251, K053251_CI4);
	m_sorted_layer[2] = 2;
	m_layerpri[2] = k053251_get_priority(m_k053251, K053251_CI3);

	konami_sortlayers3(m_sorted_layer, m_layerpri);

	machine().priority_bitmap.fill(0, cliprect);
	k052109_tilemap_draw(m_k052109, bitmap, cliprect, m_sorted_layer[0], TILEMAP_DRAW_OPAQUE, 1);
	k052109_tilemap_draw(m_k052109, bitmap, cliprect, m_sorted_layer[1], 0, 2);
	k052109_tilemap_draw(m_k052109, bitmap, cliprect, m_sorted_layer[2], 0, 4);

	k051960_sprites_draw(m_k051960, bitmap, cliprect, -1, -1);
	return 0;
}


UINT32 tmnt_state::screen_update_lgtnfght(screen_device &screen, bitmap_ind16 &bitmap, const rectangle &cliprect)
{
	int bg_colorbase;

	bg_colorbase = k053251_get_palette_index(m_k053251, K053251_CI0);
	m_sprite_colorbase = k053251_get_palette_index(m_k053251, K053251_CI1);
	m_layer_colorbase[0] = k053251_get_palette_index(m_k053251, K053251_CI2);
	m_layer_colorbase[1] = k053251_get_palette_index(m_k053251, K053251_CI4);
	m_layer_colorbase[2] = k053251_get_palette_index(m_k053251, K053251_CI3);

	k052109_tilemap_update(m_k052109);

	m_sorted_layer[0] = 0;
	m_layerpri[0] = k053251_get_priority(m_k053251, K053251_CI2);
	m_sorted_layer[1] = 1;
	m_layerpri[1] = k053251_get_priority(m_k053251, K053251_CI4);
	m_sorted_layer[2] = 2;
	m_layerpri[2] = k053251_get_priority(m_k053251, K053251_CI3);

	konami_sortlayers3(m_sorted_layer, m_layerpri);

	machine().priority_bitmap.fill(0, cliprect);
	bitmap.fill(16 * bg_colorbase, cliprect);
	k052109_tilemap_draw(m_k052109, bitmap, cliprect, m_sorted_layer[0], 0, 1);
	k052109_tilemap_draw(m_k052109, bitmap, cliprect, m_sorted_layer[1], 0, 2);
	k052109_tilemap_draw(m_k052109, bitmap, cliprect, m_sorted_layer[2], 0, 4);

	k053245_sprites_draw(m_k053245, bitmap, cliprect);
	return 0;
}


READ16_MEMBER(tmnt_state::glfgreat_ball_r)
{
#ifdef MAME_DEBUG
popmessage("%04x", m_glfgreat_pixel);
#endif
	/* if out of the ROZ layer palette range, it's in the water - return 0 */
	if (m_glfgreat_pixel < 0x400 || m_glfgreat_pixel >= 0x500)
		return 0;
	else
		return m_glfgreat_pixel & 0xff;
}

UINT32 tmnt_state::screen_update_glfgreat(screen_device &screen, bitmap_ind16 &bitmap, const rectangle &cliprect)
{
	int bg_colorbase;

	bg_colorbase = k053251_get_palette_index(m_k053251, K053251_CI0);
	m_sprite_colorbase  = k053251_get_palette_index(m_k053251, K053251_CI1);
	m_layer_colorbase[0] = k053251_get_palette_index(m_k053251, K053251_CI2);
	m_layer_colorbase[1] = k053251_get_palette_index(m_k053251, K053251_CI3) + 8;   /* weird... */
	m_layer_colorbase[2] = k053251_get_palette_index(m_k053251, K053251_CI4);

	k052109_tilemap_update(m_k052109);

	m_sorted_layer[0] = 0;
	m_layerpri[0] = k053251_get_priority(m_k053251, K053251_CI2);
	m_sorted_layer[1] = 1;
	m_layerpri[1] = k053251_get_priority(m_k053251, K053251_CI3);
	m_sorted_layer[2] = 2;
	m_layerpri[2] = k053251_get_priority(m_k053251, K053251_CI4);

	konami_sortlayers3(m_sorted_layer, m_layerpri);

	/* not sure about the 053936 priority, but it seems to work */

	machine().priority_bitmap.fill(0, cliprect);
	bitmap.fill(16 * bg_colorbase, cliprect);
	k052109_tilemap_draw(m_k052109, bitmap, cliprect, m_sorted_layer[0], 0, 1);

	if (m_layerpri[0] >= 0x30 && m_layerpri[1] < 0x30)
	{
		k053936_zoom_draw(m_k053936, bitmap, cliprect, m_roz_tilemap, 0, 1, 1);
		m_glfgreat_pixel = bitmap.pix16(0x80, 0x105);
	}

	k052109_tilemap_draw(m_k052109, bitmap, cliprect, m_sorted_layer[1], 0, 2);

	if (m_layerpri[1] >= 0x30 && m_layerpri[2] < 0x30)
	{
		k053936_zoom_draw(m_k053936, bitmap, cliprect, m_roz_tilemap, 0, 1, 1);
		m_glfgreat_pixel = bitmap.pix16(0x80, 0x105);
	}

	k052109_tilemap_draw(m_k052109, bitmap, cliprect, m_sorted_layer[2], 0, 4);

	if (m_layerpri[2] >= 0x30)
	{
		k053936_zoom_draw(m_k053936, bitmap, cliprect, m_roz_tilemap, 0, 1, 1);
		m_glfgreat_pixel = bitmap.pix16(0x80, 0x105);
	}

	k053245_sprites_draw(m_k053245, bitmap, cliprect);
	return 0;
}

UINT32 tmnt_state::screen_update_tmnt2(screen_device &screen, bitmap_ind16 &bitmap, const rectangle &cliprect)
{
	double brt;
	int i, newdim, newen, cb, ce;

	newdim = m_dim_v | ((~m_dim_c & 0x10) >> 1);
	newen  = (k053251_get_priority(m_k053251, 5) && k053251_get_priority(m_k053251, 5) != 0x3e);

	if (newdim != m_lastdim || newen != m_lasten)
	{
		brt = 1.0;
		if (newen)
			brt -= (1.0 - PALETTE_DEFAULT_SHADOW_FACTOR) * newdim / 8;
		m_lastdim = newdim;
		m_lasten = newen;

		/*
		    Only affect the background and sprites, not text layer.
		    Instead of dimming each layer we dim the entire palette
		    except text colors because palette bases may change
		    anytime and there's no guarantee a dimmed color will be
		    reset properly.
		*/

		// find the text layer's palette range
		cb = m_layer_colorbase[m_sorted_layer[2]] << 4;
		ce = cb + 128;

		// dim all colors before it
		for (i = 0; i < cb; i++)
			palette_set_pen_contrast(machine(), i, brt);

		// reset all colors in range
		for (i = cb; i < ce; i++)
			palette_set_pen_contrast(machine(), i, 1.0);

		// dim all colors after it
		for (i = ce; i < 2048; i++)
			palette_set_pen_contrast(machine(), i, brt);

		// toggle shadow/highlight
		if (~m_dim_c & 0x10)
			palette_set_shadow_mode(machine(), 1);
		else
			palette_set_shadow_mode(machine(), 0);
	}

	screen_update_lgtnfght(screen, bitmap, cliprect);
	return 0;
}


UINT32 tmnt_state::screen_update_thndrx2(screen_device &screen, bitmap_ind16 &bitmap, const rectangle &cliprect)
{
	int bg_colorbase;

	bg_colorbase = k053251_get_palette_index(m_k053251, K053251_CI0);
	m_sprite_colorbase = k053251_get_palette_index(m_k053251, K053251_CI1);
	m_layer_colorbase[0] = k053251_get_palette_index(m_k053251, K053251_CI2);
	m_layer_colorbase[1] = k053251_get_palette_index(m_k053251, K053251_CI4);
	m_layer_colorbase[2] = k053251_get_palette_index(m_k053251, K053251_CI3);

	k052109_tilemap_update(m_k052109);

	m_sorted_layer[0] = 0;
	m_layerpri[0] = k053251_get_priority(m_k053251, K053251_CI2);
	m_sorted_layer[1] = 1;
	m_layerpri[1] = k053251_get_priority(m_k053251, K053251_CI4);
	m_sorted_layer[2] = 2;
	m_layerpri[2] = k053251_get_priority(m_k053251, K053251_CI3);

	konami_sortlayers3(m_sorted_layer, m_layerpri);

	machine().priority_bitmap.fill(0, cliprect);
	bitmap.fill(16 * bg_colorbase, cliprect);
	k052109_tilemap_draw(m_k052109, bitmap, cliprect, m_sorted_layer[0], 0, 1);
	k052109_tilemap_draw(m_k052109, bitmap, cliprect, m_sorted_layer[1], 0, 2);
	k052109_tilemap_draw(m_k052109, bitmap, cliprect, m_sorted_layer[2], 0, 4);

	k051960_sprites_draw(m_k051960, bitmap, cliprect, -1, -1);
	return 0;
}



/***************************************************************************

  Housekeeping

***************************************************************************/

void tmnt_state::screen_eof_blswhstl(screen_device &screen, bool state)
{
	// on rising edge
	if (state)
	{
		k053245_clear_buffer(m_k053245);
	}
}
