%javamethodmodifiers getLibSBMLVersion "
/**
 * Returns the libSBML version as an integer: version 1.2.3 becomes 10203.
 * <p>
 * @return the libSBML version as an integer: version 1.2.3 becomes 10203.
 */
 public";

%javamethodmodifiers getLibSBMLDottedVersion "
/**
 * Returns the libSBML version as a string of the form \'1.2.3\'.
 * <p>
 * @return the libSBML version as a string of the form \'1.2.3\'.
 */
 public";

%javamethodmodifiers getLibSBMLVersionString "
/**
 * Returns the libSBML version as a string: version 1.2.3 becomes \'10203\'.
 * <p>
 * @return the libSBML version as a string: version 1.2.3 becomes \'10203\'.
 */
 public";

%typemap(javaimports) SBMLReader "
/** 
 * Methods for reading SBML from files and text strings.
 * <p>
 * <em style=\'color: #555\'>
This class of objects is defined by libSBML only and has no direct
equivalent in terms of SBML components.  This class is not prescribed by
the SBML specifications, although it is used to implement features
defined in SBML.
</em>

 * <p>
 * The {@link SBMLReader} class provides the main interface for reading SBML
 * content from files and strings.  The methods for reading SBML all return
 * an {@link SBMLDocument} object representing the results.
 * <p>
 * In the case of failures (such as if the SBML contains errors or a file
 * cannot be read), the errors will be recorded with the {@link SBMLErrorLog} object
 * kept in the {@link SBMLDocument} returned by {@link SBMLReader}.  Callers should check
 * for errors and warnings using the methods for this purpose provided on
 * {@link SBMLDocument}.
 * <p>
 * For convenience as well as easy access from other languages besides C++,
 * this file also defines two global functions, readSBML() and
 * readSBMLFromString().  They are equivalent to creating an {@link SBMLReader}
 * object and then calling
 * the {@link SBMLReader#readSBML(String filename)}  or
 * {@link SBMLReader#readSBMLFromString(String xml)}   methods, respectively.
 */
"

%javamethodmodifiers SBMLReader::SBMLReader "
  /**
   * Creates a new {@link SBMLReader} and returns it. 
   */
 public";

%javamethodmodifiers SBMLReader::readSBML(const std::string& filename) "
  /**
   * Reads an SBML document from a file.
   * <p>
   * If the file named <code>filename</code> does not exist or its content is not
   * valid SBML, one or more errors will be logged with the {@link SBMLDocument}
   * object returned by this method.  Callers can use the methods on
   * {@link SBMLDocument} such as SBMLDocument.getNumErrors() and
   * {@link SBMLDocument#getError(long n)}  to get the errors.  The object returned by
   * {@link SBMLDocument#getError(long n)}  is an {@link SBMLError} object, and it has methods to
   * get the error code, category, and severity level of the problem, as
   * well as a textual description of the problem.  The possible severity
   * levels range from informational messages to fatal errors; see the
   * documentation for {@link SBMLError} for more information.
   * <p>
   * If the file <code>filename</code> could not be read, the file-reading error will
   * appear first.  The error code can provide a clue about what happened.
   * For example, a file might be unreadable (either because it does not
   * actually exist or because the user does not have the necessary access
   * priviledges to read it) or some sort of file operation error may have
   * bee reported by the underlying operating system.  Callers can check
   * for these situations using code such as the following:
   * <div class=\'fragment\'><pre>
   * {@link SBMLReader}  reader = new {@link SBMLReader}();
   * {@link SBMLDocument}  doc  = reader.readSBML(filename);
   * 
   * if (doc-&gt;getNumErrors() &gt; 0)
   * {
   *   if (doc-&gt;getError(0)-&gt;getId() == XMLError.FileUnreadable)
   *   {
   *     // Handle case of unreadable file here.
   *   } 
   *   else if (doc-&gt;getError(0)-&gt;getId() == XMLError.FileOperationError)
   *   {
   *     // Handle case of other file error here.
   *   }
   *   else
   *   {
   *     // Handle other cases -- see error codes defined in XMLErrorCode_t
   *     // for other possible cases to check.
   *   }
   * }</pre></div>
   * <p>
   * If the given filename ends with the suffix <code>\'</code>.gz\' (for example, 
   * <code>\'myfile</code>.xml.gz\'), the file is assumed to be compressed in <em>gzip</em>
   * format and will be automatically decompressed upon reading.
   * Similarly, if the given filename ends with <code>\'</code>.zip\' or <code>\'</code>.bz2\', the
   * file is assumed to be compressed in <em>zip</em> or <em>bzip2</em> format
   * (respectively).  Files whose names lack these suffixes will be read
   * uncompressed.  Note that if the file is in <em>zip</em> format but the
   * archive contains more than one file, only the first file in the
   * archive will be read and the rest ignored.
   * <p>
   * @param filename the name or full pathname of the file to be read.
   * <p>
   * @return a pointer to the {@link SBMLDocument} created from the SBML content.
   * <p>
   * @see SBMLError
   * <p>
   * @note LibSBML versions 2.x and 3.x behave differently in error
   * handling in several respects.  One difference is how early some errors
   * are caught and whether libSBML continues processing a file in the face
   * of some early errors.  In general, libSBML 3.x stops parsing SBML
   * inputs sooner than libSBML 2.x in the face of XML errors because the
   * errors may invalidate any further SBML content.  For example, a
   * missing XML declaration at the beginning of the file was ignored by
   * libSBML 2.x but in version 3.x, it will cause libSBML to stop parsing
   * the rest of the input altogether.  While this behavior may seem more
   * severe and intolerant, it was necessary in order to provide uniform
   * behavior regardless of which underlying XML parser (Expat, Xerces,
   * libxml2) is being used by libSBML.  The XML parsers themselves behave
   * differently in their error reporting, and sometimes libSBML has to
   * resort to the lowest common denominator.
   * <p>
   * @note To read a gzip/zip file, libSBML needs to be configured and
   * linked with the <a href=\'http://www.zlib.net/\'>zlib</a> library at
   * compile time.  It also needs to be linked with the <a
   * href=\'\'>bzip2</a> library to read files in <em>bzip2</em> format.  (Both of
   * these are the default configurations for libSBML.)  Errors about
   * unreadable files will be logged if a compressed filename is given and
   * libSBML was <em>not</em> linked with the corresponding required library.
   * <p>
   * @note SBMLReader.hasZlib() and SBMLReader.hasBzip2() can be used to
   * check whether libSBML has been linked with each library.
   */
 public";

%javamethodmodifiers SBMLReader::readSBMLFromString(const std::string& xml) "
  /**
   * Reads an SBML document from the given XML string.
   * <p>
   * This method is flexible with respect to the presence of an XML
   * declaration at the beginning of the string.  In particular, if the
   * string in <code>xml</code> does not begin with the XML declaration
   * <code>&lt;?xml version=\'1.0\' encoding=\'UTF-8\'?&gt;</code>, then this
   * method will automatically prepend the declaration to <code>xml</code>.
   * <p>
   * This method will log a fatal error if the content given in the
   * parameter <code>xml</code> is not SBML.  See the method documentation for
   * {@link SBMLReader#readSBML(String filename)}  for an example of code for testing the returned
   * error code.
   * <p>
   * @param xml a string containing a full SBML model
   * <p>
   * @return a pointer to the {@link SBMLDocument} created from the SBML content.
   * <p>
   * @see SBMLReader#readSBML(String filename) 
   */
 public";

%javamethodmodifiers SBMLReader::hasZlib "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * libSBML is linked with zlib at compile time.
   * <p>
   * @return <code>true</code> if zlib is linked, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers SBMLReader::hasBzip2 "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * libSBML is linked with bzip2 at compile time.
   * <p>
   * @return <code>true</code> if bzip2 is linked, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers SBMLReader::readInternal(const char* content, bool isFile = true) "
  /**
   * Used by readSBML() and readSBMLFromString().
   * <p>
   * @docnote The native C++ implementation of this method defines a
   * default argument value.  In the documentation generated for different
   * libSBML language bindings, you may or may not see corresponding
   * arguments in the method declarations.  For example, in Java, a default
   * argument is handled by declaring two separate methods, with one of
   * them having the argument and the other one lacking the argument.
   * However, the libSBML documentation will be <em>identical</em> for both
   * methods.  Consequently, if you are reading this and do not see an
   * argument even though one is described, please look for descriptions of
   * other variants of this method near where this one appears in the
   * documentation.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers readSBML(const char *filename) "
/**
 * Reads an SBML document from the given file <code>filename</code>.
 * <p>
 * If <code>filename</code> does not exist, or it is not an SBML file, an error will
 * be logged in the error log of the {@link SBMLDocument} object returned by this
 * method.  Calling programs can inspect this error log to determine
 * the nature of the problem.  Please refer to the definition of
 * {@link SBMLDocument} for more information about the error reporting mechanism.
 * <p>
 * @return a pointer to the {@link SBMLDocument} read.
 */
 public";

%javamethodmodifiers readSBMLFromString(const char *xml) "
/**
 * Reads an SBML document from a string assumed to be in XML format.
 * <p>
 * If the string does not begin with XML declaration,
 *<div class=\'fragment\'><pre>
&lt;?xml version=\'1.0\' encoding=\'UTF-8\'?&gt;
</pre></div>
 * <p>
 * an XML declaration string will be prepended.
 * <p>
 * This method will report an error if the given string <code>xml</code> is not SBML.
 * The error will be logged in the error log of the {@link SBMLDocument} object
 * returned by this method.  Calling programs can inspect this error log to
 * determine the nature of the problem.  Please refer to the definition of
 * {@link SBMLDocument} for more information about the error reporting mechanism.
 * <p>
 * @return a pointer to the {@link SBMLDocument} read.
 */
 public";

%typemap(javaimports) SBMLWriter "
/** 
 * Methods for writing SBML to files and text strings.
 * <p>
 * <em style=\'color: #555\'>
This class of objects is defined by libSBML only and has no direct
equivalent in terms of SBML components.  This class is not prescribed by
the SBML specifications, although it is used to implement features
defined in SBML.
</em>

 * <p>
 * The {@link SBMLWriter} class is the converse of {@link SBMLReader}, and provides the
 * main interface for serializing SBML models into XML and writing the
 * result to files and text strings.  The methods for writing SBML all take
 * an {@link SBMLDocument} object and a destination.  They return a boolean value
 * to indicate success or failure.
 */
"

%javamethodmodifiers SBMLWriter::SBMLWriter "
  /**
   * Creates a new {@link SBMLWriter}.
   */
 public";

%javamethodmodifiers SBMLWriter::setProgramName(const std::string& name) "
  /**
   * Sets the name of this program, i.e., the program that is about to
   * write out the {@link SBMLDocument}.
   * <p>
   * If the program name and version are set (setProgramVersion()), the
   * following XML comment, intended for human consumption, will be written
   * at the beginning of the document:
   * <div class=\'fragment\'><pre>
   &lt;!-- Created by &lt;program name&gt; version &lt;program version&gt;
   on yyyy-MM-dd HH:mm with libsbml version &lt;libsbml version&gt;. --&gt;
</pre></div>
   * <p>
   * @param name the name of this program (where \'this program\' refers to
   * program in which libSBML is embedded, not libSBML itself!)
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <p>
   * @see #setProgramVersion(String version)
   */
 public";

%javamethodmodifiers SBMLWriter::setProgramVersion(const std::string& version) "
  /**
   * Sets the version of this program, i.e., the program that is about to
   * write out the {@link SBMLDocument}.
   * <p>
   * If the program version and name are set (setProgramName()), the
   * following XML comment, intended for human consumption, will be written
   * at the beginning of the document:
   * <div class=\'fragment\'><pre>
   &lt;!-- Created by &lt;program name&gt; version &lt;program version&gt;
   on yyyy-MM-dd HH:mm with libsbml version &lt;libsbml version&gt;. --&gt;
</pre></div>
   * <p>
   * @param version the version of this program (where \'this program\'
   * refers to program in which libSBML is embedded, not libSBML itself!)
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <p>
   * @see #setProgramName(String name)
   */
 public";

%javamethodmodifiers SBMLWriter::writeSBML(const SBMLDocument* d, const std::string& filename) "
  /**
   * Writes the given SBML document to filename.
   * <p>
   * If the given filename ends with the suffix <code>\'</code>.gz\' (for example, 
   * <code>\'myfile</code>.xml.gz\'), libSBML assumes the caller wants the file to be
   * written compressed in <em>gzip</em>.  Similarly, if the given filename ends
   * with <code>\'</code>.zip\' or <code>\'</code>.bz2\', libSBML assumes the caller wants the file
   * to be compressed in <em>zip</em> or <em>bzip2</em> format (respectively).  Files
   * whose names lack these suffixes will be written uncompressed.
   * <em>Special considerations for the zip format</em>: If the given
   * filename ends with <code>\'</code>.zip\', the file placed in the zip archive will
   * have the suffix <code>\'</code>.xml\' or <code>\'</code>.sbml\'.  For example, the file in
   * the zip archive will be named <code>\'test</code>.xml\' if the given filename is
   * <code>\'test</code>.xml.zip\' or <code>\'test</code>.zip\'.  Similarly, the filename in the
   * archive will be <code>\'test</code>.sbml\' if the given filename is 
   * <code>\'test</code>.sbml.zip\'.
   * <p>
   * @note To write a gzip/zip file, libSBML needs to be configured and
   * linked with the <a href=\'http://www.zlib.net/\'>zlib</a> library at
   * compile time.  It also needs to be linked with the <a
   * href=\'\'>bzip2</a> library to write files in <em>bzip2</em> format.  (Both
   * of these are the default configurations for libSBML.)  Errors about
   * unreadable files will be logged and this method will return <code>false</code>
   * if a compressed filename is given and libSBML was <em>not</em> linked with
   * the corresponding required library.
   * <p>
   * @note SBMLReader.hasZlib() and SBMLReader.hasBzip2() can be used to
   * check whether libSBML has been linked with each library.
   * <p>
   * @param d the SBML document to be written
   * <p>
   * @param filename the name or full pathname of the file where the SBML
   * is to be written. 
   * <p>
   * @return <code>true</code> on success and <code>false</code> if the filename could not be
   * opened for writing.
   */
 public";

%javamethodmodifiers SBMLWriter::writeSBML(const SBMLDocument* d, std::ostream& stream) "
  /**
   * Writes the given SBML document to the output stream.
   * <p>
   * @param d the SBML document to be written
   * <p>
   * @param stream the stream object where the SBML is to be written.
   * <p>
   * @return <code>true</code> on success and <code>false</code> if one of the underlying
   * parser components fail (rare).
   */
 public";

%javamethodmodifiers SBMLWriter::writeToString(const SBMLDocument* d) "
  /**
   * Writes the given SBML document to an in-memory string and returns a
   * pointer to it.
   * <p>
   * The string is owned by the caller and should be freed (with <code>free</code>())
   * when no longer needed.
   * <p>
   * @param d the SBML document to be written
   * <p>
   * @return the string on success and <code>0</code> if one of the underlying parser
   * components fail.
   */
 public";

%javamethodmodifiers SBMLWriter::hasZlib "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * underlying libSBML is linked with zlib.
   * <p>
   * LibSBML supports reading and writing files compressed with either
   * bzip2 or zip/gzip compression.  The facility depends on libSBML having
   * been compiled with the necessary support libraries.  This method
   * allows a calling program to inquire whether that is the case for the
   * copy of libSBML it is running.
   * <p>
   * @return <code>true</code> if libSBML is linked with zlib, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers SBMLWriter::hasBzip2 "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * underlying libSBML is linked with bzip2.
   * <p>
   * LibSBML supports reading and writing files compressed with either
   * bzip2 or zip/gzip compression.  The facility depends on libSBML having
   * been compiled with the necessary support libraries.  This method
   * allows a calling program to inquire whether that is the case for the
   * copy of libSBML it is running.
   * <p>
   * @return <code>true</code> if libSBML is linked with bzip2, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers SBMLWriter::hasBzip2 "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * underlying libSBML is linked with bzip2.
   * <p>
   * LibSBML supports reading and writing files compressed with either
   * bzip2 or zip/gzip compression.  The facility depends on libSBML having
   * been compiled with the necessary support libraries.  This method
   * allows a calling program to inquire whether that is the case for the
   * copy of libSBML it is running.
   * <p>
   * @return <code>true</code> if libSBML is linked with bzip2, <code>false</code> otherwise.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBMLWriter::hasBzip2 "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * underlying libSBML is linked with bzip2.
   * <p>
   * LibSBML supports reading and writing files compressed with either
   * bzip2 or zip/gzip compression.  The facility depends on libSBML having
   * been compiled with the necessary support libraries.  This method
   * allows a calling program to inquire whether that is the case for the
   * copy of libSBML it is running.
   * <p>
   * @return <code>true</code> if libSBML is linked with bzip2, <code>false</code> otherwise.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers writeSBML(const SBMLDocument_t *d, const char *filename) "
/**
 * Writes the given SBML document to filename.  This convenience function
 * is functionally equivalent to:
 * <p>
 *   SBMLWriter_writeSBML(SBMLWriter_create(), d, filename);
 * <p>
 * @return non-zero on success and zero if the filename could not be opened
 * for writing.
 */
 public";

%javamethodmodifiers writeSBMLToString(const SBMLDocument_t *d) "
/**
 * Writes the given SBML document to an in-memory string and returns a
 * pointer to it.  The string is owned by the caller and should be freed
 * (with free()) when no longer needed.  This convenience function is
 * functionally equivalent to:
 * <p>
 *   SBMLWriter_writeSBMLToString(SBMLWriter_create(), d);
 * <p>
 * @return the string on success and NULL if one of the underlying parser
 * components fail (rare).
 */
 public";

%javamethodmodifiers SBMLTypeCode_toString(SBMLTypeCode_t tc) "
/**
 * This method takes an SBML type code and returns a string representing
 * the code.
 * <p>
 * LibSBML attaches an
 * identifying code to every kind of SBML object.  These are known as
 * <em>SBML type codes</em>.  In other languages, the set of type codes
 * is stored in an enumeration; in the Java language interface for
 * libSBML, the type codes are defined as static integer constants in
 * interface class {@link libsbmlConstants}.  The names of the type codes
 * all begin with the characters <code>SBML_</code>. 
 * This method takes a type code as argument, and returns a string name
 * corresponding to that code.  For example, passing it the type code
 * <code>SBML_COMPARTMENT</code> will return the string
 * \'<code>{@link Compartment}</code>\'. 
 * <p>
 * @return a human readable name for the given SBMLTypeCode_t.
 * <p>
 * @note The caller does not own the returned string and is therefore not
 * allowed to modify it.
 */
 public";

%typemap(javaimports) SBase "
/** 
 * LibSBML implementation of SBase, the base class of all SBML objects.
 * <p>
 * Most components in SBML are derived from a single abstract base type,
 * {@link SBase}.  In addition to serving as the parent class for most other
 * classes of objects in SBML, this base type is designed to allow a
 * modeler or a software package to attach arbitrary information to each
 * major element or list in an SBML model.
 * <p>
 * {@link SBase} has an optional subelement called \'notes\'.  It is intended to
 * serve as a place for storing optional information intended to be seen by
 * humans.  An example use of the \'notes\' element would be to contain
 * formatted user comments about the model element in which the \'notes\'
 * element is enclosed.  There are certain conditions on the XHTML content
 * permitted inside the \'notes\' element; these are described separately
 * below.
 * <p>
 * {@link SBase} has another optional subelement called \'annotation\'.  Whereas the
 * \'notes\' element described above is a container for content to be shown
 * directly to humans, the \'annotation\' element is a container for optional
 * software-generated content <em>not</em> meant to be shown to humans.  The
 * element\'s content type is XML type <code>any</code>, allowing essentially
 * arbitrary data content.  SBML places only a few restrictions on the
 * organization of the content; these are intended to help software tools
 * read and write the data as well as help reduce conflicts between
 * annotations added by different tools.  They are described separately
 * below.
 * <p>
 * It is worth pointing out that the \'annotation\' element in the definition
 * of {@link SBase} exists in order that software developers may attach optional
 * application-specific data to the elements in an SBML model.  However, it
 * is important that this facility not be misused.  In particular, it is
 * <em>critical</em> that data essential to a model definition or that can
 * be encoded in existing SBML elements is <em>not</em> stored in
 * \'annotation\'. {@link Parameter} values, functional dependencies between model
 * elements, etc., should not be recorded as annotations.  It is crucial to
 * keep in mind the fact that data placed in annotations can be freely
 * ignored by software applications.  If such data affects the
 * interpretation of a model, then software interoperability is greatly
 * impeded.
 * <p>
 * Beginning with SBML Level 2, {@link SBase} also has an optional attribute named
 * \'metaid\' for supporting metadata annotations using RDF (Resource
 * Description Format). The attribute value has the data type 
 * <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>, the
 * XML identifier type, which means each \'metaid\' value must be globally
 * unique within an SBML file.  (Importantly, this uniqueness criterion
 * applies across any attribute with type 
 * <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>, not just the \'metaid\'
 * attribute used by SBML&mdash;something to be aware of if your
 * application-specific XML content inside the \'annotation\' subelement
 * happens to use <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>.)
 * The \'metaid\' value serves to identify a model
 * component for purposes such as referencing that component from metadata
 * placed within \'annotation\' subelements.
 * <p>
 * Beginning with SBML Level 2 Version 3, {@link SBase} also has an optional
 * attribute named \'sboTerm\' for supporting the use of the Systems Biology
 * Ontology.  In SBML proper, the data type of the attribute is a string of
 * the form SBO:NNNNNNN, where NNNNNNN is a seven digit integer number;
 * libSBML simplifies the representation by only storing the NNNNNNN
 * integer portion.  Thus, in libSBML, the \'sboTerm\' attribute on {@link SBase} has
 * data type <code>int</code>, and {@link SBO} identifiers are stored simply as integers.
 * {@link SBO} terms are a type of optional annotation, and each different class of
 * SBML object derived from {@link SBase} imposes its own requirements about the
 * values permitted for \'sboTerm\'.  Please consult the SBML Level 2 Version
 * 4 specification for more information about the use of {@link SBO} and the
 * \'sboTerm\' attribute.
 * <p>
 * Finally, note that, in the list of methods on {@link SBase}, there is no public
 * constructor because {@link SBase} is an abstract class.  The constructors reside
 * in the subclasses derived from {@link SBase}.
 * <p>
 * <p>
 * <h2>Requirements for the content of the \'notes\' subelement</h2>
 * <p>
 * The content of \'notes\' elements must be in XHTML 1.0 format.  (Plain
 * HTML would not be usable because whatever appears inside the \'notes\'
 * element must be compatible with XML, which HTML is not, and in any case,
 * the requirement for using XHTML does not prevent users from entering
 * plain-text content ,which they can do using the standard <code>&lt;pre&gt;
 * ... &lt;/pre&gt;</code> elements of [X]HTML.)
 * <p>
 * The XML content of a \'notes\' subelement must declare the use of the
 * XHTML XML namespace.  This can be done in multiple ways.  One way is to
 * place a namespace declaration for the appropriate namespace URI (which
 * for XHTML is <tt>\'http://www.w3.org/1999/xhtml\'</tt>) on the top-level
 * <code>sbml</code> element and then reference the namespace in the
 * \'notes\' element content using a prefix.  The following example
 * illustrates this approach:
 * <div class=\'fragment\'><pre>
&lt;sbml xmlns=\'http://www.sbml.org/sbml/level2/version4\' level=\'2\' version=\'4\'
      xmlns:xhtml=\'http://www.w3.org/1999/xhtml\'&gt;
  ...
  &lt;notes&gt;
    &lt;xhtml:body&gt;
      &lt;xhtml:center&gt;&lt;xhtml:h2&gt;A Simple Mitotic Oscillator&lt;/xhtml:h2&gt;&lt;/xhtml:center&gt;
      &lt;xhtml:p&gt;A minimal cascade model for the mitotic oscillator
      involving cyclin and cdc2 kinase&lt;/xhtml:p&gt;
    &lt;/xhtml:body&gt;
  &lt;/notes&gt;
  ...
</pre></div>
 * <p>
 * Another approach is to declare the XHTML namespace within the \'notes\'
 * element content itself, as in the following example:
 * <div class=\'fragment\'><pre>
...
&lt;notes&gt;
  &lt;body xmlns=\'http://www.w3.org/1999/xhtml\'&gt;

    &lt;center&gt;&lt;h2&gt;A Simple Mitotic Oscillator&lt;/h2&gt;&lt;/center&gt;

    A minimal cascade model for the mitotic oscillator
    involving cyclin and cdc2 kinase&lt;/p&gt;

  &lt;/body&gt;
&lt;/notes&gt;
...
</pre></div>
 * <p>
 * The <code>xmlns=\'http://www.w3.org/1999/xhtml\'</code> declaration on 
 * <code>body</code> as shown above changes the default XML namespace within it, such
 * that all of its content is by default in the XHTML namespace.  This is a
 * particularly convenient approach because it obviates the need to prefix
 * every element with a namespace prefix (i.e., <code>xhtml:</code>
 * in the previous case).  Other approaches are also possible.
 * <p>
 * SBML does not require the content of the \'notes\' subelement to be any
 * particular XHTML element; the content can be almost any well-formed
 * XHTML content.  SBML Level 2 Versions 2, 3 and 4 added some small
 * restrictions and clarifications for the allowable content in order to
 * promote greater interoperability between software tools.  The first
 * restriction comes from the requirements of XML: the \'notes\'
 * element must not contain an XML declaration nor a DOCTYPE declaration.
 * That is, \'notes\' must <em>not</em> contain
 * <div class=\'fragment\'><pre>
&lt;?xml version=\'1.0\' encoding=\'UTF-8\'?&gt;  
</pre></div>
 * nor (where the following is only one specific example of a DOCTYPE
 * declaration)
 * <div class=\'fragment\'><pre>
&lt;!DOCTYPE html PUBLIC \'-//W3C//DTD XHTML 1.0 Strict//EN\'
      \'http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd\'&gt;
</pre></div>
 * <p>
 * The second restriction is intended to balance freedom of content with
 * the complexity of implementing software that can interpret the content.
 * The content of the \'notes\' subelement in SBML can consist only of the
 * following possibilities:
 * <ul>
 * <li> A complete XHTML document (minus the XML and DOCTYPE declarations,
 * of course), that is, XHTML content beginning with the <code>html</code> tag.
 * <div class=\'fragment\'><pre>
&lt;notes&gt;
  &lt;html xmlns=\'http://www.w3.org/1999/xhtml\'&gt;
     ...
  &lt;/html&gt;
&lt;/notes&gt;
</pre></div>
 * <p>
 * <li> The <code>body</code> element from an XHTML document.  The following is an
 * example skeleton:
 * <div class=\'fragment\'><pre>
&lt;notes&gt;
    &lt;body xmlns=\'http://www.w3.org/1999/xhtml\'&gt;
      ...
    &lt;/body&gt;
&lt;/notes&gt;
</pre></div>
 * <p>
 * <li> Any XHTML content that would be permitted within a <code>body</code> element.
 * If this consists of multiple elements, each one must declare the XML
 * namespace separately.  The following is an example fragment:
 * <div class=\'fragment\'><pre>
&lt;notes&gt;
    &lt;p xmlns=\'http://www.w3.org/1999/xhtml\'&gt;
      ...
    &lt;/p&gt;
    &lt;p xmlns=\'http://www.w3.org/1999/xhtml\'&gt;
      ...
    &lt;/p&gt;
&lt;/notes&gt;
</pre></div>
 * </ul>
 * <p>
 * Another way to summarize the restrictions above is simply to say that
 * the content of an SBML \'notes\' element can be only be a complete <code>html</code>
 * element, a <code>body</code> element, or whatever is permitted inside a <code>body</code>
 * element.  In practice, this does not limit in any meaningful way what
 * can be placed inside a \'notes\' element; for example, if an application
 * or modeler wants to put a complete XHTML page, including a <code>head</code>
 * element, it can be done by putting in everything starting with the 
 * <code>html</code> container.  However, the restrictions above do make it somewhat
 * simpler to write software that can read and write the \'notes\' content.
 * <p>
 * <p>
 * <h2>Requirements for the content of the \'annotation\' subelement</h2>
 * <p>
 * At the outset, software developers should keep in mind that multiple
 * software tools may attempt to read and write annotation content.  To
 * reduce the potential for collisions between annotations written by
 * different applications, SBML Level 2 stipulates that tools must use XML
 * namespaces to specify the intended vocabulary of every annotation.  The
 * application\'s developers must choose a URI (Universal Resource
 * Identifier) reference that uniquely identifies the vocabulary the
 * application will use, and a prefix string for the annotations.
 * <p>
 * A important requirement is that application-specific annotation data is
 * entirely contained inside a single <em>top-level element</em> within the
 * SBML \'annotation\' subelement.  SBML Level 2 Versions 2, 3 and 4 place the
 * following restrictions on annotations:
 * <ul>
 * <li> Within a given SBML \'annotation\' element, there can only be one
 * top-level element using a given namespace.  An annotation element can
 * contain multiple top-level elements but each must be in a different
 * namespace.
 * <p>
 * <li> No top-level element in an \'annotation\' may use an SBML XML
 * namespace, either explicitly by referencing one of the SBML XML
 * namespace URIs or implicitly by failing to specify any namespace on the
 * annotation.
 * <p>
 * <li> The ordering of top-level elements within a given \'annotation\'
 * element is <em>not</em> significant.  An application should not expect
 * that its annotation content appears first in the \'annotation\' element,
 * nor in any other particular location.  Moreover, the ordering of
 * top-level annotation elements may be changed by different applications
 * as they read and write the same SBML file.
 * </ul>
 * <p>
 * The use of XML namespaces in this manner is intended to improve the
 * ability of multiple applications to place annotations on SBML model
 * elements with reduced risks of interference or name collisions.
 * Annotations stored by different simulation packages can therefore
 * coexist in the same model definition.  The rules governing the content
 * of \'annotation\' elements are designed to enable applications to easily
 * add, change, and remove their annotations from SBML elements while
 * simultaneously preserving annotations inserted by other applications
 * when mapping SBML from input to output.
 * <p>
 * As a further simplification for developers of software and to improve
 * software interoperability, applications are only required to preserve
 * other annotations (i.e., annotations they do not recognize) when those
 * annotations are self-contained entirely within \'annotation\', complete
 * with namespace declarations.  The following is an example:
 * <div class=\'fragment\'><pre>
&lt;annotation&gt;
    &lt;topLevelElement xmlns:\'URI\'&gt;
       ... content in the namespace identified by \'URI\' ...
    &lt;/topLevelElement&gt;
&lt;/annotation&gt;
</pre></div>
 * <p>
 * Some more examples hopefully will make these points more clear.  The
 * next example is invalid because it contains a top-level element in the
 * SBML XML namespace&mdash;this happens because no namespace is declared
 * for the <code>&lt;cytoplasm&gt;</code> element, which means by default it
 * falls into the enclosing SBML namespace:
 * <div class=\'fragment\'><pre>
&lt;annotation&gt;
    &lt;cytoplasm/&gt;
&lt;/annotation&gt;
</pre></div>
 * <p>
 * The following example is also invalid, this time because it contains two
 * top-level elements using the same XML namespace.  Note that it does not
 * matter that these are two different top-level elements
 * (<code>&lt;nodecolors&gt;</code> and <code>&lt;textcolors&gt;</code>); 
 * what matters is that these separate elements are both in the same 
 * namespace rather than having been collected and placed inside one 
 * overall container element for that namespace.
 * <div class=\'fragment\'><pre>
&lt;annotation&gt;
    &lt;mysim:nodecolors xmlns:mysim=\'http://www.mysim.org/ns\'
        mysim:bgcolor=\'green\' mysim:fgcolor=\'white\'/&gt;
    &lt;mysim:textcolors xmlns:mysim=\'http://www.mysim.org/ns\'
        mysim:bgcolor=\'green\' mysim:fgcolor=\'white\'/&gt;
&lt;/annotation&gt;
</pre></div>
 * <p>
 * On the other hand, the following example is valid:
 * <div class=\'fragment\'><pre>
&lt;annotation&gt;
    &lt;mysim:geometry xmlns:mysim=\'http://www.mysim.org/ns\'
             mysim:bgcolor=\'green\' mysim:fgcolor=\'white\'&gt;
        &lt;graph:node xmlns:graph=\'http://www.graph.org/ns\' 
             graph:x=\'4\' graph:y=\'5\' /&gt;
    &lt;/mysim:geometry&gt;
    &lt;othersim:icon xmlns:othersim=\'http://www.othersim.com/\'&gt;
        WS2002
    &lt;/othersim:icon&gt;
&lt;/annotation&gt;
</pre></div>
 * <p>
 * It is worth keeping in mind that although XML namespace names must be
 * URIs, they are (like all XML namespace names) <em>not required</em> to
 * be directly usable in the sense of identifying an actual, retrieval
 * document or resource on the Internet.  URIs such as
 * <code>\'http://www.mysim.org/\'</code> may appear as though they are
 * (e.g.) Internet addresses, but there are not the same thing.  This
 * style of URI strings, using a domain name and other parts, is only a
 * simple and commonly-used way of creating a unique name string.
 * <p>
 * <p>
 * <h2>Standard format for annotations linking data resources</h2>
 * <p>
 * SBML Level 2 Versions 2, 3 and 4 define a proposed regular format for
 * encoding two particular categories of annotations: (a) references to
 * controlled vocabulary terms and database identifiers which define and
 * describe biological and biochemical entities in a model; and (b)
 * descriptions of the provenance of a model, including its author(s) and
 * modification history.
 */
"

%javamethodmodifiers SBase::accept(SBMLVisitor& v) const "
  /**
   * Accepts the given {@link SBMLVisitor} for this instance of {@link SBase}.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used.
   * <p>
   * @return the result of calling <code>v.visit()</code>.
   */
 public";

%javamethodmodifiers SBase::clone() const "
  /**
   * Creates and returns a deep copy of this {@link SBase} object.
   * <p>
   * @return a (deep) copy of this {@link SBase} object
   */
 public";

%javamethodmodifiers SBase::getMetaId() const "
  /**
   * Returns the value of the \'metaid\' attribute of this object.
   * <p>
   * The optional attribute named \'metaid\', present on every major SBML
   * component type, is for supporting metadata annotations using RDF
   * (Resource Description Format). The attribute value has the data type
   * <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>, the XML
   * identifier type, which means each \'metaid\' value must be globally
   * unique within an SBML file.  (Importantly, this uniqueness criterion
   * applies across any attribute with type <a
   * href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>, not just the
   * \'metaid\' attribute used by SBML&mdash;something to be aware of if your
   * application-specific XML content inside the \'annotation\' subelement
   * happens to use <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>.)
   * The \'metaid\' value serves to identify a model
   * component for purposes such as referencing that component from
   * metadata placed within \'annotation\' subelements.
   * <p>
   * @return the metaid of this SBML object.
   * <p>
   * @see #isSetMetaId()
   * @see #setMetaId(String metaid)
   */
 public";

%javamethodmodifiers SBase::getMetaId "
  /**
   * Returns the value of the \'metaid\' attribute of this object.
   * <p>
   * The optional attribute named \'metaid\', present on every major SBML
   * component type, is for supporting metadata annotations using RDF
   * (Resource Description Format). The attribute value has the data type
   * <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>, the XML
   * identifier type, which means each \'metaid\' value must be globally
   * unique within an SBML file.  (Importantly, this uniqueness criterion
   * applies across any attribute with type <a
   * href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>, not just the
   * \'metaid\' attribute used by SBML&mdash;something to be aware of if your
   * application-specific XML content inside the \'annotation\' subelement
   * happens to use <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>.)
   * The \'metaid\' value serves to identify a model
   * component for purposes such as referencing that component from
   * metadata placed within \'annotation\' subelements.
   * <p>
   * @return the metaid of this SBML object.
   * <p>
   * @see #isSetMetaId()
   * @see #setMetaId(String metaid)
   */
 public";

%javamethodmodifiers SBase::getId() const "
  /**
   * Returns the value of the \'metaid\' attribute of this object.
   * <p>
   * The optional attribute named \'metaid\', present on every major SBML
   * component type, is for supporting metadata annotations using RDF
   * (Resource Description Format). The attribute value has the data type
   * <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>, the XML
   * identifier type, which means each \'metaid\' value must be globally
   * unique within an SBML file.  (Importantly, this uniqueness criterion
   * applies across any attribute with type <a
   * href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>, not just the
   * \'metaid\' attribute used by SBML&mdash;something to be aware of if your
   * application-specific XML content inside the \'annotation\' subelement
   * happens to use <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>.)
   * The \'metaid\' value serves to identify a model
   * component for purposes such as referencing that component from
   * metadata placed within \'annotation\' subelements.
   * <p>
   * @return the metaid of this SBML object.
   * <p>
   * @see #isSetMetaId()
   * @see #setMetaId(String metaid)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::getName() const "
  /**
   * Returns the value of the \'metaid\' attribute of this object.
   * <p>
   * The optional attribute named \'metaid\', present on every major SBML
   * component type, is for supporting metadata annotations using RDF
   * (Resource Description Format). The attribute value has the data type
   * <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>, the XML
   * identifier type, which means each \'metaid\' value must be globally
   * unique within an SBML file.  (Importantly, this uniqueness criterion
   * applies across any attribute with type <a
   * href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>, not just the
   * \'metaid\' attribute used by SBML&mdash;something to be aware of if your
   * application-specific XML content inside the \'annotation\' subelement
   * happens to use <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>.)
   * The \'metaid\' value serves to identify a model
   * component for purposes such as referencing that component from
   * metadata placed within \'annotation\' subelements.
   * <p>
   * @return the metaid of this SBML object.
   * <p>
   * @see #isSetMetaId()
   * @see #setMetaId(String metaid)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::getNotes "
  /**
   * Returns the content of the \'notes\' subelement of this object as
   * a tree of XML nodes.
   * <p>
   * The optional element named \'notes\', present on every major SBML
   * component type, is intended as a place for storing optional
   * information intended to be seen by humans.  An example use of the
   * \'notes\' element would be to contain formatted user comments
   * about the model element in which the \'notes\' element is
   * enclosed.  Every object derived directly or indirectly from type
   * {@link SBase} can have a separate value for \'notes\', allowing users
   * considerable freedom when adding comments to their models.
   * The format of \'notes\' elements must be XHTML 1.0.  The SBML
   * Level&nbsp;2 specification has considerable detail about how
   * \'notes\' element content must be handled; please refer to the
   * specifications.
   * <p>
   * The \'notes\' element content returned by this method will be in XML
   * form, but libSBML does not provide an object model specifically for
   * the content of notes.  Callers will need to traverse the XML tree
   * structure using the facilities available on {@link XMLNode} and related
   * objects.
   * <p>
   * @return the content of the \'notes\' subelement of this SBML object.
   * <p>
   * @see #getNotesString()
   * @see #isSetNotes()
   * @see #setNotes(XMLNode  notes)
   * @see #setNotes(String notes)
   * @see #appendNotes(XMLNode  notes)
   * @see #appendNotes(String notes)
   * @see #unsetNotes()
   */
 public";

%javamethodmodifiers SBase::getNotesString "
  /**
   * Returns the content of the \'notes\' subelement of this object as a
   * string.
   * <p>
   * The optional element named \'notes\', present on every major SBML
   * component type, is intended as a place for storing optional
   * information intended to be seen by humans.  An example use of the
   * \'notes\' element would be to contain formatted user comments
   * about the model element in which the \'notes\' element is
   * enclosed.  Every object derived directly or indirectly from type
   * {@link SBase} can have a separate value for \'notes\', allowing users
   * considerable freedom when adding comments to their models.
   * The format of \'notes\' elements must be XHTML 1.0.  The SBML
   * Level&nbsp;2 specification has considerable detail about how
   * \'notes\' element content must be handled; please refer to the
   * specifications.
   * <p>
   * @return the content of the \'notes\' subelement of this SBML object.
   * <p>
   * @see #getNotes()
   * @see #isSetNotes()
   * @see #setNotes(XMLNode  notes)
   * @see #setNotes(String notes)
   * @see #appendNotes(XMLNode  notes)
   * @see #appendNotes(String notes)
   * @see #unsetNotes()
   */
 public";

%javamethodmodifiers SBase::getAnnotation "
  /**
   * Returns the content of the \'annotation\' subelement of this object as
   * an XML node tree.
   * <p>
   * Whereas the {@link SBase} \'notes\' subelement is a container for content to be
   * shown directly to humans, the \'annotation\' element is a container for
   * optional software-generated content <em>not</em> meant to be shown to
   * humans.  Every object derived from {@link SBase} can have its own value for
   * \'annotation\'.  The element\'s content type is XML type
   * <code>any</code>, allowing essentially arbitrary well-formed XML data
   * content.
   * <p>
   * SBML places a few restrictions on the organization of the content of
   * annotations; these are intended to help software tools read and write
   * the data as well as help reduce conflicts between annotations added by
   * different tools.  Please see the SBML specifications for more details.
   * <p>
   * The annotations returned by this method will be in XML form.  LibSBML
   * provides an object model and related interfaces for certain specific
   * kinds of annotations, namely model history information and RDF
   * content.  See the {@link ModelHistory}, {@link CVTerm} and RDFAnnotationParser classes
   * for more information about the facilities available.
   * <p>
   * @return the annotation of this SBML object as an {@link XMLNode}
   * <p>
   * @see #getAnnotationString()
   * @see #isSetAnnotation()
   * @see #setAnnotation(XMLNode  annotation)
   * @see #setAnnotation(String annotation)
   * @see #appendAnnotation(XMLNode  annotation)
   * @see #appendAnnotation(String annotation)
   * @see #unsetAnnotation()
   */
 public";

%javamethodmodifiers SBase::getAnnotationString "
  /**
   * Returns the content of the \'annotation\' subelement of this object as a
   * character string.
   * <p>
   * Whereas the {@link SBase} \'notes\' subelement is a container for content to be
   * shown directly to humans, the \'annotation\' element is a container for
   * optional software-generated content <em>not</em> meant to be shown to
   * humans.  Every object derived from {@link SBase} can have its own value for
   * \'annotation\'.  The element\'s content type is XML type
   * <code>any</code>, allowing essentially arbitrary well-formed XML data
   * content.
   * <p>
   * SBML places a few restrictions on the organization of the content of
   * annotations; these are intended to help software tools read and write
   * the data as well as help reduce conflicts between annotations added by
   * different tools.  Please see the SBML specifications for more details.
   * <p>
   * The annotations returned by this method will be in string form.
   * <p>
   * @return the annotation string of this SBML object.
   * <p>
   * @see #getAnnotation()
   * @see #isSetAnnotation()
   * @see #setAnnotation(XMLNode  annotation)
   * @see #setAnnotation(String annotation)
   * @see #appendAnnotation(XMLNode  annotation)
   * @see #appendAnnotation(String annotation)
   * @see #unsetAnnotation()
   */
 public";

%javamethodmodifiers SBase::getNamespaces() const "
  /**
   * Returns a list of the XML Namespaces declared on this SBML document.
   * <p>
   * @return the XML Namespaces associated with this SBML object
   */
 public";

%javamethodmodifiers SBase::getSBMLDocument() const "
  /**
   * Returns the parent {@link SBMLDocument} object.
   * <p>
   * LibSBML uses the class {@link SBMLDocument} as a top-level container for
   * storing SBML content and data associated with it (such as warnings and
   * error messages).  An SBML model in libSBML is contained inside an
   * {@link SBMLDocument} object.  {@link SBMLDocument} corresponds roughly to the class
   * <i>Sbml</i> defined in the SBML Level&nbsp;2 specification, but it
   * does not have a direct correspondence in SBML Level&nbsp;1.  (But, it
   * is created by libSBML no matter whether the model is Level&nbsp;1 or
   * Level&nbsp;2.)
   * <p>
   * This method allows the {@link SBMLDocument} for the current object to be
   * retrieved.
   * <p>
   * @return the parent {@link SBMLDocument} object of this SBML object.
   */
 public";

%javamethodmodifiers SBase::getSBMLDocument "
  /**
   * Returns the parent {@link SBMLDocument} object.
   * <p>
   * LibSBML uses the class {@link SBMLDocument} as a top-level container for
   * storing SBML content and data associated with it (such as warnings and
   * error messages).  An SBML model in libSBML is contained inside an
   * {@link SBMLDocument} object.  {@link SBMLDocument} corresponds roughly to the class
   * <i>Sbml</i> defined in the SBML Level&nbsp;2 specification, but it
   * does not have a direct correspondence in SBML Level&nbsp;1.  (But, it
   * is created by libSBML no matter whether the model is Level&nbsp;1 or
   * Level&nbsp;2.)
   * <p>
   * This method allows the {@link SBMLDocument} for the current object to be
   * retrieved.
   * <p>
   * @return the parent {@link SBMLDocument} object of this SBML object.
   */
 public";

%javamethodmodifiers SBase::getParentSBMLObject "
  /**
   * Returns the parent SBML object.
   * <p>
   * This method is convenient when holding an object nested inside
   * other objects in an SBML model.  It allows direct access to the
   * <code>&lt;model&gt;</code> element containing it.
   * <p>
   * @return the parent SBML object of this SBML object.
   */
 public";

%javamethodmodifiers SBase::getAncestorOfType(SBMLTypeCode_t type) "
  /**
   * Returns the ancestor SBML object that corresponds to the 
   * given SBMLTypeCode_t.
   * <p>
   * This function allows any object to determine its exact
   * location/function within a model. For example a {@link StoichiometryMath}
   * object has ancestors of type {@link SpeciesReference},
   * ListOfProducts/ListOfReactants/ListOfModifiers, {@link Reaction},
   * {@link ListOfReactions} and {@link Model}; any of which can be accessed via this
   * function.
   * <p>
   * @param type the SBMLTypeCode_t of the ancestor to be returned.
   * <p>
   * @return the ancestor SBML object of this SBML object, that 
   * corresponds to the given SBMLTypeCode_t, NULL if none exists.
   */
 public";

%javamethodmodifiers SBase::getSBOTerm() const "
  /**
   * Returns the integer portion of the value of the \'sboTerm\' attribute of
   * this object.
   * <p>
   * In SBML Level 2 Versions 2, 3 and 4, the data type of the attribute is a
   * string of the form SBO:NNNNNNN, where NNNNNNN is a seven digit integer
   * number; libSBML simplifies the representation by only storing the
   * NNNNNNN integer portion.  Thus, in libSBML, the \'sboTerm\' attribute on
   * {@link SBase} has data type <code>int</code>, and {@link SBO} identifiers are stored simply as
   * integers.  {@link SBO} terms are a type of optional annotation, and each
   * different class of SBML object derived from {@link SBase} imposes its own
   * requirements about the values permitted for \'sboTerm\'.  Please consult
   * the SBML Level 2 Version 4 specification for more information about
   * the use of {@link SBO} and the \'sboTerm\' attribute.
   * <p>
   * @return the value of the \'sboTerm\' attribute as an integer, or <code>-1</code>
   * if the value is not set.
   */
 public";

%javamethodmodifiers SBase::getSBOTermID() const "
  /**
   * Returns the string representation of the \'sboTerm\' attribute of
   * this object.
   * <p>
   * In SBML Level 2 Versions 2, 3 and 4, the data type of the attribute is a
   * string of the form SBO:NNNNNNN, where NNNNNNN is a seven digit integer
   * number; libSBML simplifies the representation by only storing the
   * NNNNNNN integer portion.  Thus, in libSBML, the \'sboTerm\' attribute on
   * {@link SBase} has data type <code>int</code>, and {@link SBO} identifiers are stored simply as
   * integers.  This function recreates the string representation from the
   * stored value.  {@link SBO} terms are a type of optional annotation, and each
   * different class of SBML object derived from {@link SBase} imposes its own
   * requirements about the values permitted for \'sboTerm\'.  Please consult
   * the SBML Level 2 Version 4 specification for more information about
   * the use of {@link SBO} and the \'sboTerm\' attribute.
   * <p>
   * @return the value of the \'sboTerm\' attribute as a string of the form
   * SBO:NNNNNNN, or  an empty string
   * if the value is not set.
   */
 public";

%javamethodmodifiers SBase::getLine() const "
  /**
   * Returns the line number on which this object first appears in the XML
   * representation of the SBML document.
   * <p>
   * @return the line number of this SBML object.
   * <p>
   * @see #getColumn()
   */
 public";

%javamethodmodifiers SBase::getColumn() const "
  /**
   * Returns the column number on which this object first appears in the XML
   * representation of the SBML document.
   * <p>
   * @return the column number of this SBML object.
   * <p>
   * @see #getLine()
   */
 public";

%javamethodmodifiers SBase::isSetMetaId() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * object\'s \'metaid\' attribute has been set.
   * <p>
   * The optional attribute named \'metaid\', present on every major SBML
   * component type, is for supporting metadata annotations using RDF
   * (Resource Description Format). The attribute value has the data type
   * <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>, the XML
   * identifier type, which means each \'metaid\' value must be globally
   * unique within an SBML file.  (Importantly, this uniqueness criterion
   * applies across any attribute with type <a
   * href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>, not just the
   * \'metaid\' attribute used by SBML&mdash;something to be aware of if your
   * application-specific XML content inside the \'annotation\' subelement
   * happens to use <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>.)
   * The \'metaid\' value serves to identify a model
   * component for purposes such as referencing that component from
   * metadata placed within \'annotation\' subelements.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the \'metaid\' attribute of this SBML object has been
   * set, <code>false</code> otherwise.
   * <p>
   * @see #getMetaId()
   * @see #setMetaId(String metaid)
   */
 public";

%javamethodmodifiers SBase::isSetId() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * object\'s \'metaid\' attribute has been set.
   * <p>
   * The optional attribute named \'metaid\', present on every major SBML
   * component type, is for supporting metadata annotations using RDF
   * (Resource Description Format). The attribute value has the data type
   * <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>, the XML
   * identifier type, which means each \'metaid\' value must be globally
   * unique within an SBML file.  (Importantly, this uniqueness criterion
   * applies across any attribute with type <a
   * href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>, not just the
   * \'metaid\' attribute used by SBML&mdash;something to be aware of if your
   * application-specific XML content inside the \'annotation\' subelement
   * happens to use <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>.)
   * The \'metaid\' value serves to identify a model
   * component for purposes such as referencing that component from
   * metadata placed within \'annotation\' subelements.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the \'metaid\' attribute of this SBML object has been
   * set, <code>false</code> otherwise.
   * <p>
   * @see #getMetaId()
   * @see #setMetaId(String metaid)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::isSetName() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * object\'s \'metaid\' attribute has been set.
   * <p>
   * The optional attribute named \'metaid\', present on every major SBML
   * component type, is for supporting metadata annotations using RDF
   * (Resource Description Format). The attribute value has the data type
   * <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>, the XML
   * identifier type, which means each \'metaid\' value must be globally
   * unique within an SBML file.  (Importantly, this uniqueness criterion
   * applies across any attribute with type <a
   * href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>, not just the
   * \'metaid\' attribute used by SBML&mdash;something to be aware of if your
   * application-specific XML content inside the \'annotation\' subelement
   * happens to use <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>.)
   * The \'metaid\' value serves to identify a model
   * component for purposes such as referencing that component from
   * metadata placed within \'annotation\' subelements.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the \'metaid\' attribute of this SBML object has been
   * set, <code>false</code> otherwise.
   * <p>
   * @see #getMetaId()
   * @see #setMetaId(String metaid)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::isSetNotes() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * object\'s \'notes\' subelement exists and has content.
   * <p>
   * The optional element named \'notes\', present on every major SBML
   * component type, is intended as a place for storing optional
   * information intended to be seen by humans.  An example use of the
   * \'notes\' element would be to contain formatted user comments
   * about the model element in which the \'notes\' element is
   * enclosed.  Every object derived directly or indirectly from type
   * {@link SBase} can have a separate value for \'notes\', allowing users
   * considerable freedom when adding comments to their models.
   * The format of \'notes\' elements must be XHTML 1.0.  The SBML
   * Level&nbsp;2 specification has considerable detail about how
   * \'notes\' element content must be handled; please refer to the
   * specifications.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if a \'notes\' subelement exists, <code>false</code> otherwise.
   * <p>
   * @see #getNotes()
   * @see #getNotesString()
   * @see #setNotes(XMLNode  notes)
   * @see #setNotes(String notes)
   * @see #appendNotes(XMLNode  notes)
   * @see #appendNotes(String notes)
   * @see #unsetNotes()
   */
 public";

%javamethodmodifiers SBase::isSetAnnotation() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * object\'s \'annotation\' subelement exists and has content.
   * <p>
   * Whereas the {@link SBase} \'notes\' subelement is a container for content to be
   * shown directly to humans, the \'annotation\' element is a container for
   * optional software-generated content <em>not</em> meant to be shown to
   * humans.  Every object derived from {@link SBase} can have its own value for
   * \'annotation\'.  The element\'s content type is XML type
   * <code>any</code>, allowing essentially arbitrary well-formed XML data
   * content.
   * <p>
   * SBML places a few restrictions on the organization of the content of
   * annotations; these are intended to help software tools read and write
   * the data as well as help reduce conflicts between annotations added by
   * different tools.  Please see the SBML specifications for more details.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if a \'annotation\' subelement exists, <code>false</code>
   * otherwise.
   * <p>
   * @see #getAnnotation()
   * @see #getAnnotationString()
   * @see #setAnnotation(XMLNode  annotation)
   * @see #setAnnotation(String annotation)
   * @see #appendAnnotation(XMLNode  annotation)
   * @see #appendAnnotation(String annotation)
   * @see #unsetAnnotation()
   */
 public";

%javamethodmodifiers SBase::isSetSBOTerm() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * object\'s \'sboTerm\' attribute has been set.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the \'sboTerm\' attribute of this SBML object has been
   * set, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers SBase::setMetaId(const std::string& metaid) "
  /**
   * Sets the value of the \'metaid\' attribute of this object.
   * <p>
   * The string <code>metaid</code> is copied.  The value of <code>metaid</code> must be an
   * identifier conforming to the syntax defined by the XML 1.0 data type
   * <a href=\'http://www.w3.org/TR/REC-xml/#id\'>ID</a>.  Among other
   * things, this type requires that a value is unique among all the values
   * of type XML ID in an {@link SBMLDocument}.  Although SBML only uses <a
   * href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a> for the \'metaid\'
   * attribute, callers should be careful if they use
   * <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>\'s in XML
   * portions of a model that are not defined by SBML, such as in the
   * application-specific content of the \'annotation\' subelement.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param metaid the identifier string to use as the value of the
   * \'metaid\' attribute
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   * <li> LIBSBML_UNEXPECTED_ATTRIBUTE
   * <p>
   * @see #getMetaId()
   * @see #isSetMetaId()
   */
 public";

%javamethodmodifiers SBase::setId(const std::string& sid) "
  /**
   * Sets the value of the \'metaid\' attribute of this object.
   * <p>
   * The string <code>metaid</code> is copied.  The value of <code>metaid</code> must be an
   * identifier conforming to the syntax defined by the XML 1.0 data type
   * <a href=\'http://www.w3.org/TR/REC-xml/#id\'>ID</a>.  Among other
   * things, this type requires that a value is unique among all the values
   * of type XML ID in an {@link SBMLDocument}.  Although SBML only uses <a
   * href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a> for the \'metaid\'
   * attribute, callers should be careful if they use
   * <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>\'s in XML
   * portions of a model that are not defined by SBML, such as in the
   * application-specific content of the \'annotation\' subelement.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param metaid the identifier string to use as the value of the
   * \'metaid\' attribute
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   * <li> LIBSBML_UNEXPECTED_ATTRIBUTE
   * <p>
   * @see #getMetaId()
   * @see #isSetMetaId()
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::setName(const std::string& name) "
  /**
   * Sets the value of the \'metaid\' attribute of this object.
   * <p>
   * The string <code>metaid</code> is copied.  The value of <code>metaid</code> must be an
   * identifier conforming to the syntax defined by the XML 1.0 data type
   * <a href=\'http://www.w3.org/TR/REC-xml/#id\'>ID</a>.  Among other
   * things, this type requires that a value is unique among all the values
   * of type XML ID in an {@link SBMLDocument}.  Although SBML only uses <a
   * href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a> for the \'metaid\'
   * attribute, callers should be careful if they use
   * <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>\'s in XML
   * portions of a model that are not defined by SBML, such as in the
   * application-specific content of the \'annotation\' subelement.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param metaid the identifier string to use as the value of the
   * \'metaid\' attribute
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   * <li> LIBSBML_UNEXPECTED_ATTRIBUTE
   * <p>
   * @see #getMetaId()
   * @see #isSetMetaId()
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::setAnnotation(const XMLNode* annotation) "
  /**
   * Resets the value of the \'annotation\' subelement of this SBML object to a
   * copy of <code>annotation</code>.
   * <p>
   * Whereas the {@link SBase} \'notes\' subelement is a container for content to be
   * shown directly to humans, the \'annotation\' element is a container for
   * optional software-generated content <em>not</em> meant to be shown to
   * humans.  Every object derived from {@link SBase} can have its own value for
   * \'annotation\'.  The element\'s content type is XML type
   * <code>any</code>, allowing essentially arbitrary well-formed XML data
   * content.
   * <p>
   * SBML places a few restrictions on the organization of the content of
   * annotations; these are intended to help software tools read and write
   * the data as well as help reduce conflicts between annotations added by
   * different tools.  Please see the SBML specifications for more details.
   * <p>
   * Call this method will result in any existing content of the
   * \'annotation\' subelement to be discarded.  Unless you have taken steps
   * to first copy and reconstitute any existing annotations into the 
   * <code>annotation</code> that is about to be assigned, it is likely that performing
   * such wholesale replacement is unfriendly towards other software
   * applications whose annotations are discarded.  An alternative may be
   * to use {@link SBase#appendAnnotation(XMLNode  annotation)} or
   * {@link SBase#appendAnnotation(String annotation)}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param annotation an XML structure that is to be used as the new content
   * of the \'annotation\' subelement of this object
   * <p>
   * @see #getAnnotationString()
   * @see #isSetAnnotation()
   * @see #setAnnotation(String annotation)
   * @see #appendAnnotation(XMLNode  annotation)
   * @see #appendAnnotation(String annotation)
   * @see #unsetAnnotation()
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <p>
   * @see #getAnnotationString()
   * @see #isSetAnnotation()
   * @see #setAnnotation(String annotation)
   * @see #appendAnnotation(XMLNode  annotation)
   * @see #appendAnnotation(String annotation)
   * @see #unsetAnnotation()
   */
 public";

%javamethodmodifiers SBase::setAnnotation(const std::string& annotation) "
  /**
   * Sets the value of the \'annotation\' subelement of this SBML object to a
   * copy of <code>annotation</code> given as a character string.
   * <p>
   * Whereas the {@link SBase} \'notes\' subelement is a container for content to be
   * shown directly to humans, the \'annotation\' element is a container for
   * optional software-generated content <em>not</em> meant to be shown to
   * humans.  Every object derived from {@link SBase} can have its own value for
   * \'annotation\'.  The element\'s content type is XML type
   * <code>any</code>, allowing essentially arbitrary well-formed XML data
   * content.
   * <p>
   * SBML places a few restrictions on the organization of the content of
   * annotations; these are intended to help software tools read and write
   * the data as well as help reduce conflicts between annotations added by
   * different tools.  Please see the SBML specifications for more details.
   * <p>
   * Call this method will result in any existing content of the
   * \'annotation\' subelement to be discarded.  Unless you have taken steps
   * to first copy and reconstitute any existing annotations into the 
   * <code>annotation</code> that is about to be assigned, it is likely that performing
   * such wholesale replacement is unfriendly towards other software
   * applications whose annotations are discarded.  An alternative may be
   * to use {@link SBase#appendAnnotation(XMLNode  annotation)} or
   * {@link SBase#appendAnnotation(String annotation)}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param annotation an XML string that is to be used as the content
   * of the \'annotation\' subelement of this object
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_OPERATION_FAILED
   * <p>
   * @see #getAnnotationString()
   * @see #isSetAnnotation()
   * @see #setAnnotation(XMLNode  annotation)
   * @see #appendAnnotation(XMLNode  annotation)
   * @see #appendAnnotation(String annotation)
   * @see #unsetAnnotation()
   */
 public";

%javamethodmodifiers SBase::appendAnnotation(const XMLNode* annotation) "
  /**
   * Appends the annotation content given by <code>annotation</code> to any existing
   * content in the \'annotation\' subelement of this object.
   * <p>
   * Whereas the {@link SBase} \'notes\' subelement is a container for content to be
   * shown directly to humans, the \'annotation\' element is a container for
   * optional software-generated content <em>not</em> meant to be shown to
   * humans.  Every object derived from {@link SBase} can have its own value for
   * \'annotation\'.  The element\'s content type is XML type
   * <code>any</code>, allowing essentially arbitrary well-formed XML data
   * content.
   * <p>
   * SBML places a few restrictions on the organization of the content of
   * annotations; these are intended to help software tools read and write
   * the data as well as help reduce conflicts between annotations added by
   * different tools.  Please see the SBML specifications for more details.
   * <p>
   * Unlike {@link SBase#setAnnotation(XMLNode  annotation)} or
   * {@link SBase#setAnnotation(String annotation)}, this method
   * allows other annotations to be preserved when an application adds its
   * own data.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param annotation an XML structure that is to be copied and appended
   * to the content of the \'annotation\' subelement of this object
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_OPERATION_FAILED
   * <p>
   * @see #getAnnotationString()
   * @see #isSetAnnotation()
   * @see #setAnnotation(XMLNode  annotation)
   * @see #setAnnotation(String annotation)
   * @see #appendAnnotation(String annotation)
   * @see #unsetAnnotation()
   */
 public";

%javamethodmodifiers SBase::appendAnnotation(const std::string& annotation) "
  /**
   * Appends the annotation content given by <code>annotation</code> to any existing
   * content in the \'annotation\' subelement of this object.
   * <p>
   * Whereas the {@link SBase} \'notes\' subelement is a container for content to be
   * shown directly to humans, the \'annotation\' element is a container for
   * optional software-generated content <em>not</em> meant to be shown to
   * humans.  Every object derived from {@link SBase} can have its own value for
   * \'annotation\'.  The element\'s content type is XML type
   * <code>any</code>, allowing essentially arbitrary well-formed XML data
   * content.
   * <p>
   * SBML places a few restrictions on the organization of the content of
   * annotations; these are intended to help software tools read and write
   * the data as well as help reduce conflicts between annotations added by
   * different tools.  Please see the SBML specifications for more details.
   * <p>
   * Unlike {@link SBase#setAnnotation(XMLNode  annotation)} or
   * {@link SBase#setAnnotation(String annotation)}, this method
   * allows other annotations to be preserved when an application adds its
   * own data.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param annotation an XML string that is to be copied and appended
   * to the content of the \'annotation\' subelement of this object
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_OPERATION_FAILED
   * <p>
   * @see #getAnnotationString()
   * @see #isSetAnnotation()
   * @see #setAnnotation(XMLNode  annotation)
   * @see #setAnnotation(String annotation)
   * @see #appendAnnotation(XMLNode  annotation)
   * @see #unsetAnnotation()
   */
 public";

%javamethodmodifiers SBase::setNotes(const XMLNode* notes) "
  /**
   * Sets the value of the \'notes\' subelement of this SBML object to a copy
   * of the XML structure given by <code>notes</code>.
   * <p>
   * Any existing content of the \'notes\' subelement is discarded.
   * <p>
   * The optional element named \'notes\', present on every major SBML
   * component type, is intended as a place for storing optional
   * information intended to be seen by humans.  An example use of the
   * \'notes\' element would be to contain formatted user comments
   * about the model element in which the \'notes\' element is
   * enclosed.  Every object derived directly or indirectly from type
   * {@link SBase} can have a separate value for \'notes\', allowing users
   * considerable freedom when adding comments to their models.
   * The format of \'notes\' elements must be XHTML 1.0.  The SBML
   * Level&nbsp;2 specification has considerable detail about how
   * \'notes\' element content must be handled; please refer to the
   * specifications.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param notes an XML structure that is to be used as the content of the
   * \'notes\' subelement of this object
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_OBJECT
   * <p>
   * @see #getNotesString()
   * @see #isSetNotes()
   * @see #setNotes(String notes)
   * @see #appendNotes(XMLNode  notes)
   * @see #appendNotes(String notes)
   * @see #unsetNotes()
   */
 public";

%javamethodmodifiers SBase::setNotes(const std::string& notes) "
  /**
   * Sets the value of the \'notes\' subelement of this SBML object to a copy
   * of the string <code>notes</code>.
   * <p>
   * Any existing content of the \'notes\' subelement is discarded.
   * <p>
   * The optional element named \'notes\', present on every major SBML
   * component type, is intended as a place for storing optional
   * information intended to be seen by humans.  An example use of the
   * \'notes\' element would be to contain formatted user comments
   * about the model element in which the \'notes\' element is
   * enclosed.  Every object derived directly or indirectly from type
   * {@link SBase} can have a separate value for \'notes\', allowing users
   * considerable freedom when adding comments to their models.
   * The format of \'notes\' elements must be XHTML 1.0.  The SBML
   * Level&nbsp;2 specification has considerable detail about how
   * \'notes\' element content must be handled; please refer to the
   * specifications.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param notes an XML string that is to be used as the content of the
   * \'notes\' subelement of this object
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_OBJECT
   * <li> LIBSBML_OPERATION_FAILED
   * <p>
   * @see #getNotesString()
   * @see #isSetNotes()
   * @see #setNotes(XMLNode  notes)
   * @see #appendNotes(XMLNode  notes)
   * @see #appendNotes(String notes)
   * @see #unsetNotes()
   */
 public";

%javamethodmodifiers SBase::appendNotes(const XMLNode* notes) "
  /**
   * Appends notes content to the \'notes\' element attached to this
   * object.
   * <p>
   * The content in <code>notes</code> is copied.
   * <p>
   * The optional element named \'notes\', present on every major SBML
   * component type, is intended as a place for storing optional
   * information intended to be seen by humans.  An example use of the
   * \'notes\' element would be to contain formatted user comments
   * about the model element in which the \'notes\' element is
   * enclosed.  Every object derived directly or indirectly from type
   * {@link SBase} can have a separate value for \'notes\', allowing users
   * considerable freedom when adding comments to their models.
   * The format of \'notes\' elements must be XHTML 1.0.  The SBML
   * Level&nbsp;2 specification has considerable detail about how
   * \'notes\' element content must be handled; please refer to the
   * specifications.
   * <p>
   * @param notes an XML node structure that is to appended to the content
   * of the \'notes\' subelement of this object
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_OBJECT
   * <li> LIBSBML_OPERATION_FAILED
   * <p>
   * @see #getNotesString()
   * @see #isSetNotes()
   * @see #setNotes(XMLNode  notes)
   * @see #setNotes(String notes)
   * @see #appendNotes(String notes)
   * @see #unsetNotes()
   */
 public";

%javamethodmodifiers SBase::appendNotes(const std::string& notes) "
  /**
   * Appends notes content to the \'notes\' element attached to this
   * object.
   * <p>
   * The content in <code>notes</code> is copied.
   * <p>
   * The optional element named \'notes\', present on every major SBML
   * component type, is intended as a place for storing optional
   * information intended to be seen by humans.  An example use of the
   * \'notes\' element would be to contain formatted user comments
   * about the model element in which the \'notes\' element is
   * enclosed.  Every object derived directly or indirectly from type
   * {@link SBase} can have a separate value for \'notes\', allowing users
   * considerable freedom when adding comments to their models.
   * The format of \'notes\' elements must be XHTML 1.0.  The SBML
   * Level&nbsp;2 specification has considerable detail about how
   * \'notes\' element content must be handled; please refer to the
   * specifications.
   * <p>
   * @param notes an XML string that is to appended to the content of
   * the \'notes\' subelement of this object
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_OBJECT
   * <li> LIBSBML_OPERATION_FAILED
   * <p>
   * @see #getNotesString()
   * @see #isSetNotes()
   * @see #setNotes(XMLNode  notes)
   * @see #setNotes(String notes)
   * @see #appendNotes(XMLNode  notes)
   * @see #unsetNotes()
   */
 public";

%javamethodmodifiers SBase::setSBMLDocument(SBMLDocument* d) "
  /**
   * Sets the parent {@link SBMLDocument} of this SBML object.
   * <p>
   * @param d the {@link SBMLDocument} object to use
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::setParentSBMLObject(SBase* sb) "
  /**
   * Sets the parent SBML object of this SBML object.
   * <p>
   * @param sb the SBML object to use
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::setSBOTerm(int value) "
  /**
   * Sets the value of the \'sboTerm\' attribute.
   * <p>
   * In SBML Level 2 Versions 2, 3 and 4, the data type of the SBML \'sboTerm\'
   * attribute is a string of the form SBO:NNNNNNN, where NNNNNNN is a
   * seven digit integer number; libSBML simplifies the representation by
   * only storing the NNNNNNN integer portion.  Thus, in libSBML, the
   * \'sboTerm\' attribute on {@link SBase} has data type <code>int</code>, and {@link SBO} identifiers
   * are stored simply as integers.  {@link SBO} terms are a type of optional
   * annotation, and each different class of SBML object derived from {@link SBase}
   * imposes its own requirements about the values permitted for \'sboTerm\'.
   * Please consult the SBML Level 2 Version 4 specification for more
   * information about the use of {@link SBO} and the \'sboTerm\' attribute.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param value the NNNNNNN integer portion of the {@link SBO} identifier
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   * <li> LIBSBML_UNEXPECTED_ATTRIBUTE
   */
 public";

%javamethodmodifiers SBase::setSBOTerm(const std::string &sboid) "
  /**
   * Sets the value of the \'sboTerm\' attribute.
   * <p>
   * In SBML Level 2 Versions 2, 3 and 4, the data type of the SBML \'sboTerm\'
   * attribute is a string of the form SBO:NNNNNNN, where NNNNNNN is a
   * seven digit integer number; libSBML simplifies the representation by
   * only storing the NNNNNNN integer portion.  Thus, in libSBML, the
   * \'sboTerm\' attribute on {@link SBase} has data type <code>int</code>, and {@link SBO} identifiers
   * are stored simply as integers.  {@link SBO} terms are a type of optional
   * annotation, and each different class of SBML object derived from {@link SBase}
   * imposes its own requirements about the values permitted for \'sboTerm\'.
   * Please consult the SBML Level 2 Version 4 specification for more
   * information about the use of {@link SBO} and the \'sboTerm\' attribute.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param value the NNNNNNN integer portion of the {@link SBO} identifier
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   * <li> LIBSBML_UNEXPECTED_ATTRIBUTE
   */
 public";

%javamethodmodifiers SBase::setNamespaces(XMLNamespaces* xmlns) "
  /**
   * Sets the namespaces relevant of this SBML object.
   * <p>
   * @param xmlns the namespaces to set
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   */
 public";

%javamethodmodifiers SBase::unsetMetaId "
  /**
   * Unsets the value of the \'metaid\' attribute of this SBML object.
   * <p>
   * The optional attribute named \'metaid\', present on every major SBML
   * component type, is for supporting metadata annotations using RDF
   * (Resource Description Format). The attribute value has the data type
   * <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>, the XML
   * identifier type, which means each \'metaid\' value must be globally
   * unique within an SBML file.  (Importantly, this uniqueness criterion
   * applies across any attribute with type <a
   * href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>, not just the
   * \'metaid\' attribute used by SBML&mdash;something to be aware of if your
   * application-specific XML content inside the \'annotation\' subelement
   * happens to use <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>.)
   * The \'metaid\' value serves to identify a model
   * component for purposes such as referencing that component from
   * metadata placed within \'annotation\' subelements.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_OPERATION_FAILED
   */
 public";

%javamethodmodifiers SBase::unsetId "
  /**
   * Unsets the value of the \'id\' attribute of this SBML object.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <p>
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_OPERATION_FAILED
   */
 public";

%javamethodmodifiers SBase::unsetName "
  /**
   * Unsets the value of the \'name\' attribute of this SBML object.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <p>
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_OPERATION_FAILED
   */
 public";

%javamethodmodifiers SBase::unsetNotes "
  /**
   * Unsets the value of the \'notes\' subelement of this SBML object.
   * <p>
   * The optional element named \'notes\', present on every major SBML
   * component type, is intended as a place for storing optional
   * information intended to be seen by humans.  An example use of the
   * \'notes\' element would be to contain formatted user comments
   * about the model element in which the \'notes\' element is
   * enclosed.  Every object derived directly or indirectly from type
   * {@link SBase} can have a separate value for \'notes\', allowing users
   * considerable freedom when adding comments to their models.
   * The format of \'notes\' elements must be XHTML 1.0.  The SBML
   * Level&nbsp;2 specification has considerable detail about how
   * \'notes\' element content must be handled; please refer to the
   * specifications.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <p>
   * @see #getNotesString()
   * @see #isSetNotes()
   * @see #setNotes(XMLNode  notes)
   * @see #setNotes(String notes)
   * @see #appendNotes(XMLNode  notes)
   * @see #appendNotes(String notes)
   */
 public";

%javamethodmodifiers SBase::unsetAnnotation "
  /**
   * Unsets the value of the \'annotation\' subelement of this SBML object.
   * <p>
   * Whereas the {@link SBase} \'notes\' subelement is a container for content to be
   * shown directly to humans, the \'annotation\' element is a container for
   * optional software-generated content <em>not</em> meant to be shown to
   * humans.  Every object derived from {@link SBase} can have its own value for
   * \'annotation\'.  The element\'s content type is XML type
   * <code>any</code>, allowing essentially arbitrary well-formed XML data
   * content.
   * <p>
   * SBML places a few restrictions on the organization of the content of
   * annotations; these are intended to help software tools read and write
   * the data as well as help reduce conflicts between annotations added by
   * different tools.  Please see the SBML specifications for more details.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <p>
   * @see #getAnnotation()
   * @see #getAnnotationString()
   * @see #isSetAnnotation()
   * @see #setAnnotation(XMLNode  annotation)
   * @see #setAnnotation(String annotation)
   * @see #appendAnnotation(XMLNode  annotation)
   * @see #appendAnnotation(String annotation)
   */
 public";

%javamethodmodifiers SBase::unsetSBOTerm "
  /**
   * Unsets the value of the \'sboTerm\' attribute of this SBML object.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_UNEXPECTED_ATTRIBUTE
   */
 public";

%javamethodmodifiers SBase::addCVTerm(CVTerm * term) "
  /**
   * Adds a copy of the given {@link CVTerm} to this SBML object.
   * <p>
   * @param term the {@link CVTerm} to assign
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_OPERATION_FAILED
   * <li> LIBSBML_UNEXPECTED_ATTRIBUTE
   * <li> LIBSBML_INVALID_OBJECT
   * <p>
   * @note Since the CV Term uses the metaid of the object as a 
   * reference, if the object has no metaid set the {@link CVTerm} will
   * not be added.
   * <p>
   * @warning The fact that this method <em>copies</em> the object passed to it
   * means that the caller will be left holding a physically different
   * object instance than the one contained in <em>this</em> object.  Changes
   * made to the original object instance (such as resetting attribute
   * values) will <em>not affect the instance added here</em>.  In
   * addition, the caller should make sure to free the original object if
   * it is no longer being used, or else a memory leak will result.
   */
 public";

%javamethodmodifiers SBase::getCVTerms "
  /**
   * Returns a list of {@link CVTerm} objects in the annotations of this SBML
   * object.
   * <p>
   * @return the list of CVTerms for this SBML object.
   */
 public";

%javamethodmodifiers SBase::getCVTerms() const "
  /**
   * Returns a list of {@link CVTerm} objects in the annotations of this SBML
   * object.
   * <p>
   * @return the list of CVTerms for this SBML object.
   */
 public";

%javamethodmodifiers SBase::getNumCVTerms "
  /**
   * Returns the number of {@link CVTerm} objects in the annotations of this SBML
   * object.
   * <p>
   * @return the number of CVTerms for this SBML object.
   */
 public";

%javamethodmodifiers SBase::getCVTerm(unsigned int n) "
  /**
   * Returns the nth {@link CVTerm} in the list of CVTerms of this SBML
   * object.
   * <p>
   * @param n long the index of the {@link CVTerm} to retrieve
   * <p>
   * @return the nth {@link CVTerm} in the list of CVTerms for this SBML object.
   */
 public";

%javamethodmodifiers SBase::unsetCVTerms "
  /**
   * Clears the list of CVTerms of this SBML
   * object.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_OPERATION_FAILED
   */
 public";

%javamethodmodifiers SBase::getResourceBiologicalQualifier(std::string resource) "
  /**
   * Returns the BiologicalQualifier associated with this resource,
   * or BQB_UNKNOWN if the resource does not exist.
   * <p>
   * @param resource string representing the resource; e.g.,
   * \'http://www.geneontology.org/#GO:0005892\'
   * <p>
   * @return the #BiolQualifierType_t value associated with the resource
   */
 public";

%javamethodmodifiers SBase::getResourceModelQualifier(std::string resource) "
  /**
   * Returns the ModelQualifier associated with this resource,
   * or BQM_UNKNOWN if the resource does not exist.
   * <p>
   * @param resource string representing the resource; e.g.,
   * \'http://www.geneontology.org/#GO:0005892\'
   * <p>
   * @return the #ModelQualifierType_t value associated with the resource
   */
 public";

%javamethodmodifiers SBase::getModel() const "
  /**
   * Returns the {@link Model} object in which the current object is located.
   * <p>
   * @return the parent {@link Model} of this SBML object.
   */
 public";

%javamethodmodifiers SBase::getLevel() const "
  /**
   * Returns the SBML Level of the overall SBML document.
   * <p>
   * @return the SBML level of this SBML object.
   * <p>
   * @see #getVersion()
   */
 public";

%javamethodmodifiers SBase::getVersion() const "
  /**
   * Returns the Version within the SBML Level of the overall SBML document.
   * <p>
   * @return the SBML version of this SBML object.
   * <p>
   * @see #getLevel()
   */
 public";

%javamethodmodifiers SBase::getTypeCode() const "
  /**
   * Returns the libSBML type code for this object.
   * <p>
   * This method MAY return the typecode of this SBML object or it MAY
   * return SBML_UNKNOWN.  That is, subclasses of {@link SBase} are not required to
   * implement this method to return a typecode.  This method is meant
   * primarily for the LibSBML C interface where class and subclass
   * information is not readily available.
   * <p>
   * @return the #SBMLTypeCode_t value of this SBML object or SBML_UNKNOWN
   * (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers SBase::hasValidLevelVersionNamespaceCombination "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * object\'s level/version and namespace values correspond to a valid
   * SBML specification.
   * <p>
   * The valid combinations of SBML Level, Version and Namespace as of this release
   * of libSBML are the following: 
   * <ul>
   * <li> Level&nbsp;1 Version&nbsp;2 \'http://www.sbml.org/sbml/level1\'
   * <li> Level&nbsp;2 Version&nbsp;1 \'http://www.sbml.org/sbml/level2\'
   * <li> Level&nbsp;2 Version&nbsp;2 \'http://www.sbml.org/sbml/level2/version2\'
   * <li> Level&nbsp;2 Version&nbsp;3 \'http://www.sbml.org/sbml/level2/version3\'
   * <li> Level&nbsp;2 Version&nbsp;4 \'http://www.sbml.org/sbml/level2/version4\'
   * </ul>
   * <p>
   * @return <code>true</code> if the level, version and namespace values of this 
   * SBML object correspond to a valid set of values, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers SBase::getElementName() const "
  /**
   * Returns the XML element name of this object.
   * <p>
   * This is overridden by subclasses to return a string appropriate to the
   * SBML component.  For example, {@link Model} defines it as returning \'model\',
   * {@link CompartmentType} defines it as returning \'compartmentType\', etc.
   */
 public";

%javamethodmodifiers SBase::toSBML "
  /**
   * Returns a string that consists of the partial SBML describing this
   * object.  This is primarily provided for testing and debugging
   * purposes.  It may be removed in a future version of libSBML.
   * <p>
   * @return the partial SBML that describes this SBML object.
   */
 public";

%javamethodmodifiers SBase::read(XMLInputStream& stream) "
  /**
   * Reads (initializes) this SBML object by reading from XMLInputStream.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::write(XMLOutputStream& stream) const "
  /**
   * Writes (serializes) this SBML object by writing it to XMLOutputStream.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::writeElements(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write out their contained
   * SBML objects as XML elements.  Be sure to call your parents
   * implementation of this method as well.  For example:
   * <p>
   *   {@link SBase#writeElements(stream)};
   *   mReactans.write(stream);
   *   mProducts.write(stream);
   *   ...
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::hasRequiredAttributes() const "
  /**
   * Subclasses should override this method to write out their contained
   * SBML objects as XML elements.  Be sure to call your parents
   * implementation of this method as well.  For example:
   * <p>
   *   {@link SBase#writeElements(stream)};
   *   mReactans.write(stream);
   *   mProducts.write(stream);
   *   ...
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::hasRequiredElements() const "
  /**
   * Subclasses should override this method to write out their contained
   * SBML objects as XML elements.  Be sure to call your parents
   * implementation of this method as well.  For example:
   * <p>
   *   {@link SBase#writeElements(stream)};
   *   mReactans.write(stream);
   *   mProducts.write(stream);
   *   ...
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::setSBMLNamespaces(SBMLNamespaces * sbmlns) "
  /**
   * Subclasses should override this method to write out their contained
   * SBML objects as XML elements.  Be sure to call your parents
   * implementation of this method as well.  For example:
   * <p>
   *   {@link SBase#writeElements(stream)};
   *   mReactans.write(stream);
   *   mProducts.write(stream);
   *   ...
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::getSBMLNamespaces() const "
  /**
   * Subclasses should override this method to write out their contained
   * SBML objects as XML elements.  Be sure to call your parents
   * implementation of this method as well.  For example:
   * <p>
   *   {@link SBase#writeElements(stream)};
   *   mReactans.write(stream);
   *   mProducts.write(stream);
   *   ...
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::removeDuplicateAnnotations "
  /**
   * Subclasses should override this method to write out their contained
   * SBML objects as XML elements.  Be sure to call your parents
   * implementation of this method as well.  For example:
   * <p>
   *   {@link SBase#writeElements(stream)};
   *   mReactans.write(stream);
   *   mProducts.write(stream);
   *   ...
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::SBase(const std::string& id = "", const std::string& name = "", int sboTerm = -1) "
  /**
   * Only subclasses may create {@link SBase} objects.
   * <p>
   * @docnote The native C++ implementation of this method defines a
   * default argument value.  In the documentation generated for different
   * libSBML language bindings, you may or may not see corresponding
   * arguments in the method declarations.  For example, in Java, a default
   * argument is handled by declaring two separate methods, with one of
   * them having the argument and the other one lacking the argument.
   * However, the libSBML documentation will be <em>identical</em> for both
   * methods.  Consequently, if you are reading this and do not see an
   * argument even though one is described, please look for descriptions of
   * other variants of this method near where this one appears in the
   * documentation.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::SBase(unsigned int level, unsigned int version) "
  /**
   * Creates a new {@link SBase} object with the given sboTerm.
   * Only subclasses may create {@link SBase} objects.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::SBase(SBMLNamespaces* sbmlns) "
  /**
   * Creates a new {@link SBase} object with the given {@link SBMLNamespaces}.
   * Only subclasses may create {@link SBase} objects.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::SBase(const SBase& orig) "
  /**
  * Copy constructor. Creates a copy of this {@link SBase} object.
  <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::createObject(XMLInputStream& stream) "
  /**
   * Subclasses should override this method to create, store, and then
   * return an SBML object corresponding to the next {@link XMLToken} in the
   * XMLInputStream.
   * <p>
   * @return the SBML object corresponding to next {@link XMLToken} in the
   * XMLInputStream or NULL if the token was not recognized.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::readOtherXML(XMLInputStream& stream) "
  /**
   * Subclasses should override this method to read (and store) XHTML,
   * MathML, etc. directly from the XMLInputStream.
   * <p>
   * @return true if the subclass read from the stream, false otherwise.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::getElementPosition() const "
  /**
   * The SBML XML Schema is written such that the order of child elements
   * is significant.  LibSBML can read elements out of order.  If you
   * override this method to indicate the ordinal position of element with
   * respect to its siblings, libSBML will log an error if the element is
   * read out of order.
   * <p>
   * @return the ordinal position of the element with respect to its
   * siblings or -1 (default) to indicate the position is not significant.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::getErrorLog "
  /**
   * @return the {@link SBMLErrorLog} used to log errors during while reading and
   * validating SBML.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::logError(  unsigned int       id , const unsigned int level   = 2 , const unsigned int version = 3 , const std::string& details = "" ) "
  /**
   * Convenience method for easily logging problems from within method
   * implementations.
   * <p>
   * This is essentially a short form of getErrorLog()->logError(...)
   * <p>
   * @docnote The native C++ implementation of this method defines a
   * default argument value.  In the documentation generated for different
   * libSBML language bindings, you may or may not see corresponding
   * arguments in the method declarations.  For example, in Java, a default
   * argument is handled by declaring two separate methods, with one of
   * them having the argument and the other one lacking the argument.
   * However, the libSBML documentation will be <em>identical</em> for both
   * methods.  Consequently, if you are reading this and do not see an
   * argument even though one is described, please look for descriptions of
   * other variants of this method near where this one appears in the
   * documentation.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::logUnknownAttribute( std::string attribute, const unsigned int level, const unsigned int version, const std::string element ) "
  /**
   * Helper to log a common type of error.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::logUnknownElement( const std::string element, const unsigned int level, const unsigned int version ) "
  /**
   * Helper to log a common type of error.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::logEmptyString( std::string attribute, const unsigned int level, const unsigned int version, std::string element) "
  /**
   * Helper to log a common type of error.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::readAttributes(const XMLAttributes& attributes) "
  /**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.  For example:
   * <p>
   *   {@link SBase#writeAttributes(stream)};
   *   stream.writeAttribute( \'id\'  , mId   );
   *   stream.writeAttribute( \'name\', mName );
   *   ...
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::syncAnnotation "
  /**
   * Synchronizes the annotation of this SBML object.
   * <p>
   * Annotation element (XMLNode  mAnnotation) is synchronized with the 
   * current {@link CVTerm} objects (List  mCVTerm).
   * Currently, this method is called in getAnnotation, isSetAnnotation,
   * and writeElements methods.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::checkOrderAndLogError(SBase* object, int expected) "
  /**
   * Checks that SBML element has been read in the proper order.  If object
   * is not in the expected position, an error is logged.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::checkListOfPopulated(SBase* object) "
  /**
   * Checks that an SBML {@link ListOf} element has been populated.  
   * If a listOf element has been declared with no elements, 
   * an error is logged.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::checkUnitSyntax(unsigned int flag = 0) "
  /**
   * Checks the syntax of the unit attribute.
   * The syntax of an unit is of type UnitSId which is defined as:
   * <p>
   *  - letter .= \'a\'..\'z\',\'A\'..\'Z\'
   *  - digit  .= \'0\'..\'9\'
   *  - idChar .= letter | digit | \'_\'
   *  - UnitSId    .= ( letter | \'_\' ) idChar*
   * <p>
   * If the syntax of the unit attribute of this object is incorrect, 
   * an error is logged
   * <p>
   * @docnote The native C++ implementation of this method defines a
   * default argument value.  In the documentation generated for different
   * libSBML language bindings, you may or may not see corresponding
   * arguments in the method declarations.  For example, in Java, a default
   * argument is handled by declaring two separate methods, with one of
   * them having the argument and the other one lacking the argument.
   * However, the libSBML documentation will be <em>identical</em> for both
   * methods.  Consequently, if you are reading this and do not see an
   * argument even though one is described, please look for descriptions of
   * other variants of this method near where this one appears in the
   * documentation.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::checkAnnotation "
  /**
   * Checks the annotation does not declare an sbml namespace.
   * If the annotation declares an sbml namespace an error is logged.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::checkXHTML(const XMLNode *) "
  /**
   * Checks that the XHTML is valid.
   * If the xhtml does not conform to the specification of valid xhtml within
   * an sbml document, an error is logged.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::checkXHTML(const XMLNode *) "
  /**
   * Checks that the XHTML is valid.
   * If the xhtml does not conform to the specification of valid xhtml within
   * an sbml document, an error is logged.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::checkXHTML(const XMLNode *) "
  /**
   * Checks that the XHTML is valid.
   * If the xhtml does not conform to the specification of valid xhtml within
   * an sbml document, an error is logged.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::checkXHTML(const XMLNode *) "
  /**
   * Checks that the XHTML is valid.
   * If the xhtml does not conform to the specification of valid xhtml within
   * an sbml document, an error is logged.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::checkXHTML(const XMLNode *) "
  /**
   * Checks that the XHTML is valid.
   * If the xhtml does not conform to the specification of valid xhtml within
   * an sbml document, an error is logged.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::checkXHTML(const XMLNode *) "
  /**
   * Checks that the XHTML is valid.
   * If the xhtml does not conform to the specification of valid xhtml within
   * an sbml document, an error is logged.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::checkXHTML(const XMLNode *) "
  /**
   * Checks that the XHTML is valid.
   * If the xhtml does not conform to the specification of valid xhtml within
   * an sbml document, an error is logged.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::checkXHTML(const XMLNode *) "
  /**
   * Checks that the XHTML is valid.
   * If the xhtml does not conform to the specification of valid xhtml within
   * an sbml document, an error is logged.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::checkXHTML(const XMLNode *) "
  /**
   * Checks that the XHTML is valid.
   * If the xhtml does not conform to the specification of valid xhtml within
   * an sbml document, an error is logged.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::checkXHTML(const XMLNode *) "
  /**
   * Checks that the XHTML is valid.
   * If the xhtml does not conform to the specification of valid xhtml within
   * an sbml document, an error is logged.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::checkXHTML(const XMLNode *) "
  /**
   * Checks that the XHTML is valid.
   * If the xhtml does not conform to the specification of valid xhtml within
   * an sbml document, an error is logged.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::checkXHTML(const XMLNode *) "
  /**
   * Checks that the XHTML is valid.
   * If the xhtml does not conform to the specification of valid xhtml within
   * an sbml document, an error is logged.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::checkXHTML(const XMLNode *) "
  /**
   * Checks that the XHTML is valid.
   * If the xhtml does not conform to the specification of valid xhtml within
   * an sbml document, an error is logged.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::setSBaseFields(const XMLToken& element) "
  /**
   * Stores the location (line and column) and any XML namespaces (for
   * roundtripping) declared on this SBML (XML) element.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::readAnnotation(XMLInputStream& stream) "
  /**
   * @return true if read an <annotation> element from the stream
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::readNotes(XMLInputStream& stream) "
  /**
   * @return true if read a <notes> element from the stream
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBase::getHasBeenDeleted "
  /**
   * @return true if read a <notes> element from the stream
   <p>
 * @deprecated libSBML internal
 */
 public";

%typemap(javaimports) ListOf "
/** 
 * Parent class for the various SBML \'ListOfXYZ\' classes.
 * <p>
 * <em style=\'color: #555\'>
This class of objects is defined by libSBML only and has no direct
equivalent in terms of SBML components.  This class is not prescribed by
the SBML specifications, although it is used to implement features
defined in SBML.
</em>

 * <p>
 */
"

%javamethodmodifiers ListOf::ListOf "
  /**
   * Creates a new {@link ListOf}.
   */
 public";

%javamethodmodifiers ListOf::ListOf(const ListOf& orig) "
  /**
   * Copy constructor.  Creates a copy of this {@link ListOf}.
   */
 public";

%javamethodmodifiers ListOf::accept(SBMLVisitor& v) const "
  /**
   * Accepts the given {@link SBMLVisitor}.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used.
   * <p>
   * @return the result of calling <code>v.visit()</code>, which indicates
   * whether the Visitor would like to visit the next item in the
   * list.
   */
 public";

%javamethodmodifiers ListOf::clone() const "
  /**
   * Creates and returns a deep copy of this {@link ListOf}.
   * <p>
   * @return a (deep) copy of this {@link ListOf}.
   */
 public";

%javamethodmodifiers ListOf::append(const SBase* item) "
  /**
   * Adds item to the end of this {@link ListOf}.
   * <p>
   * This variant of the method makes a clone of the <code>item</code> handed to it.
   * This means that when the {@link ListOf} is destroyed, the original items will
   * not be destroyed.
   * <p>
   * @param item the item to be added to the list.
   * <p>
   * @see #appendAndOwn(SBase  item)
   */
 public";

%javamethodmodifiers ListOf::appendAndOwn(SBase* item) "
  /**
   * Adds item to the end of this {@link ListOf}.
   * <p>
   * This variant of the method does not clone the <code>item</code> handed to it;
   * instead, it assumes ownership of it.  This means that when the {@link ListOf}
   * is destroyed, the item will be destroyed along with it.
   * <p>
   * @param item the item to be added to the list.
   * <p>
   * @see #append(SBase  item)
   */
 public";

%javamethodmodifiers ListOf::get "
  /**
   * Get an item from the list.
   * <p>
   * @param n the index number of the item to get.
   * <p>
   * @return the nth item in this {@link ListOf} items.
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOf::get(unsigned int n) "
  /**
   * Get an item from the list.
   * <p>
   * @param n the index number of the item to get.
   * <p>
   * @return the nth item in this {@link ListOf} items.
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOf::get(const std::string& sid) const "
  /**
   * Get an item from the list based on its identifier.
   * <p>
   * @param sid a string representing the the identifier of the item to get.
   * <p>
   * @return item in this {@link ListOf} items with the given id or NULL if no such
   * item exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */
 public";

%javamethodmodifiers ListOf::get(const std::string& sid) "
  /**
   * Get an item from the list based on its identifier.
   * <p>
   * @param sid a string representing the the identifier of the item to get.
   * <p>
   * @return item in this {@link ListOf} items with the given id or NULL if no such
   * item exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */
 public";

%javamethodmodifiers ListOf::clear(bool doDelete = true) "
  /**
   * Removes all items in this {@link ListOf} object.
   * <p>
   * If doDelete is true (default), all items in this {@link ListOf} object are deleted
   * and cleared, and thus the caller doesn\'t have to delete those items.
   * Otherwise, all items are just cleared from this {@link ListOf} object and the caller 
   * is responsible for deleting all items (In this case, pointers to all items 
   * should be stored elsewhere before calling this function by the caller).
   * <p>
   * @param doDelete if true (default), all items are deleted and cleared.
   * Otherwise, all items are just cleared and not deleted. 
   */
 public";

%javamethodmodifiers ListOf::remove(unsigned int n) "
  /**
   * Removes the nth item from this {@link ListOf} items and returns a pointer to
   * it.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * <p>
   * @param n the index of the item to remove
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOf::remove(const std::string& sid) "
  /**
   * Removes item in this {@link ListOf} items with the given identifier.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * If none of the items in this list have the identifier <code>sid</code>, then 
   * <code>NULL</code> is returned.
   * <p>
   * @param sid the identifier of the item to remove
   * <p>
   * @return the item removed.  As mentioned above, the caller owns the
   * returned item.
   */
 public";

%javamethodmodifiers ListOf::size() const "
  /**
   * Get the size of this {@link ListOf}.
   * <p>
   * @return the number of items in this {@link ListOf} items.
   */
 public";

%javamethodmodifiers ListOf::setSBMLDocument(SBMLDocument* d) "
  /**
   * Sets the parent {@link SBMLDocument} of this SBML object.
   * <p>
   * @param d the {@link SBMLDocument} that should become the parent of this
   * {@link ListOf}.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers ListOf::setParentSBMLObject(SBase* sb) "
  /**
   * Sets the parent SBML object of this SBML object.
   * <p>
   * @param sb the SBML object to use
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers ListOf::getTypeCode() const "
  /**
   * Returns the libSBML type code for this object, namely, 
   * <code>SBML_LIST_OF</code>.
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for this object, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers ListOf::getItemTypeCode() const "
  /**
   * Get the type code of the objects contained in this {@link ListOf}.
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for the objects contained in this {@link ListOf}
   * instance, or <code>SBML_UNKNOWN</code> (default).
   */
 public";

%javamethodmodifiers ListOf::getElementName() const "
  /**
   * Returns the XML element name of this object, which for {@link ListOf}, is
   * always <code>\'listOf\'</code>.
   * <p>
   * @return the XML name of this element.
   */
 public";

%javamethodmodifiers ListOf::writeElements(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write out their contained
   * SBML objects as XML elements.  Be sure to call your parents
   * implementation of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers ListOf::readAttributes(const XMLAttributes& attributes) "
  /**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers ListOf::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.  For example:
   * <p>
   *   {@link SBase#writeAttributes(stream)};
   *   stream.writeAttribute( \'id\'  , mId   );
   *   stream.writeAttribute( \'name\', mName );
   *   ...
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers ListOf::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.  For example:
   * <p>
   *   {@link SBase#writeAttributes(stream)};
   *   stream.writeAttribute( \'id\'  , mId   );
   *   stream.writeAttribute( \'name\', mName );
   *   ...
   <p>
 * @deprecated libSBML internal
 */
 public";

%typemap(javaimports) Model "
/** 
 * LibSBML implementation of SBML\'s Model construct.
 * <p>
 * In an SBML model definition, a single object of class {@link Model} serves as
 * the overall container for the lists of the various model components.
 * All of the lists are optional, but if a given list container is present
 * within the model, the list must not be empty; that is, it must have
 * length one or more.  The following are the components and lists
 * permitted in different Levels and Versions of SBML as of this version
 * of libSBML (3.3):
 * <ul>
 * <li> In SBML Level 1, the components are: {@link UnitDefinition}, {@link Compartment},
 * {@link Species}, {@link Parameter}, {@link Rule}, and {@link Reaction}.  Instances of the classes are
 * placed inside instances of classes {@link ListOfUnitDefinitions},
 * {@link ListOfCompartments}, {@link ListOfSpecies}, {@link ListOfParameters}, {@link ListOfRules}, and
 * {@link ListOfReactions}.
 * <p>
 * <li> In SBML Level 2 Version 1, the components are: {@link FunctionDefinition},
 * {@link UnitDefinition}, {@link Compartment}, {@link Species}, {@link Parameter}, {@link Rule}, {@link Reaction} and
 * {@link Event}.  Instances of the classes are placed inside instances of classes
 * {@link ListOfFunctionDefinitions}, {@link ListOfUnitDefinitions}, {@link ListOfCompartments},
 * {@link ListOfSpecies}, {@link ListOfParameters}, {@link ListOfRules}, {@link ListOfReactions}, and
 * {@link ListOfEvents}.
 * <p>
 * <li> In SBML Level 2 Versions 2, 3 and 4, the components are:
 * {@link FunctionDefinition}, {@link UnitDefinition}, {@link CompartmentType}, {@link SpeciesType},
 * {@link Compartment}, {@link Species}, {@link Parameter}, {@link InitialAssignment}, {@link Rule}, {@link Constraint},
 * {@link Reaction} and {@link Event}.  Instances of the classes are placed inside
 * instances of classes {@link ListOfFunctionDefinitions}, {@link ListOfUnitDefinitions},
 * {@link ListOfCompartmentTypes}, {@link ListOfSpeciesTypes}, {@link ListOfCompartments},
 * {@link ListOfSpecies}, {@link ListOfParameters}, {@link ListOfInitialAssignments}, {@link ListOfRules},
 * {@link ListOfConstraints}, {@link ListOfReactions}, and {@link ListOfEvents}.
 * </ul>
 * <p>
 * Although all the lists are optional, there are dependencies between SBML
 * components such that defining some components requires defining others.
 * An example is that defining a species requires defining a compartment,
 * and defining a reaction requires defining a species.  The dependencies
 * are explained in more detail in the SBML specifications.
 * <p>
 * <h2>Approaches to creating objects using the libSBML API</h2>
 * <p>
 * LibSBML provides two main mechanisms for creating objects: class
 * constructors (e.g., <a href=\'org/sbml/libsbml/{@link Species}.html\'>{@link Species}()</a> ),
 * and <code>create<i>Object</i>()</code> methods (such as
 * Model.createSpecies()) provided by certain object classes such as
 * {@link Model}.  These multiple mechanisms are provided by libSBML for
 * flexibility and to support different use-cases, but they also have
 * different implications for the overall model structure.
 * <p>
 * In general, the recommended approach is to use the
 * <code>create<i>Object</i>()</code> methods.  These methods both create
 * an object <em>and</em> link it to the parent in one step.  Here is an
 * example:
 * <div class=\'fragment\'><pre>
// Create an {@link SBMLDocument} object in Level 2 Version 4 format:

{@link SBMLDocument}  sbmlDoc = new {@link SBMLDocument}(2, 4);

// Create a {@link Model} object inside the {@link SBMLDocument} object and set
// its identifier.  The call returns a pointer to the {@link Model} object
// created, and methods called on that object affect the attributes
// of the object attached to the model (as expected).

{@link Model}  model = sbmlDoc-&gt;createModel();
model-&gt;setId(\'MyModel\');

// Create a {@link Species} object inside the {@link Model} and set its identifier.
// Similar to the lines above, this call returns a pointer to the {@link Species}
// object created, and methods called on that object affect the attributes
// of the object attached to the model (as expected).

{@link Species} sp = model-&gt;createSpecies();
sp-&gt;setId(\'MySpecies\');
</pre></div>
 * <p>
 * The <code>create<i>Object</i>()</code> methods return a pointer to the
 * object created, but they also add the object to the relevant list of
 * object instances contained in the parent.  (These lists become the
 * <code>&lt;listOf<i>Object</i>s&gt;</code> elements in the finished XML
 * rendition of SBML.)  In the example above, Model.createSpecies() adds
 * the created species directly to the <code>&lt;listOfSpecies gt;</code>
 * list in the model.  Subsequently, methods called on the species change
 * the species in the model (which is what is expected in most situations).
 * <p>
 * By contrast, the other main way of creating an object and adding it to a
 * parent makes a <em>copy</em> of the object, and requires more care on the
 * part of the caller.  Here is an example of this alternative approach:
 * <div class=\'fragment\'><pre>
// Create a {@link Species} object and add it to the model.
// This uses the {@link Species} class constructor:

{@link Species} newsp = {@link Species}(\'MySpecies\');
model-&gt;addSpecies(newsp); // Warning! This makes a COPY inside \'model\'.

// addSpecies(...) copies the object, with the result that
// \'newsp\' still refers to the original.  The following may not
// do what is expected:

newsp.setId(\'NewId\');    // Warning -- doesn\'t change the species in \'model\'!

// If \'newsp\' object isn\'t going to be used further, it needs
// to be deleted to avoid a memory leak.

delete newsp;
</pre></div>
 * <p>
 * The key point of the example above is that, because the {@link Model#addSpecies(Species s)} 
 * call makes a copy of the object handed to it, care is needed both when
 * attempting to make changes to the object, and when the original object
 * is no longer needed.
 * <p>
 * <h2>Consistency and adherence to SBML specifications</h2>
 * <p>
 * To make it easier for applications to do whatever they need,
 * libSBML&nbsp;3.x is relatively lax when it comes to enforcing
 * correctness and completeness of models during model construction and
 * editing.  Essentially, libSBML <em>will</em> <em>not</em> in most cases check
 * automatically that a model\'s components have valid attribute values, or
 * that the overall model is consistent and free of errors&mdash;even
 * obvious errors such as duplication of identifiers.  This allows
 * applications great leeway in how they build their models, but it means
 * that software authors must take deliberate steps to ensure that the
 * model will be, in the end, valid SBML.  These steps include such things
 * as keeping track of the identifiers used in a model, manually performing
 * updates in certain situations where an entity is referenced in more than
 * one place (e.g., a species that is referenced by multiple
 * {@link SpeciesReference} objects), and so on.
 * <p>
 * That said, libSBML does provide powerful features for deliberately
 * performing validation of SBML when an application decides it is time to
 * do so.  The interfaces to these facilities are on the {@link SBMLDocument}
 * class, in the form of SBMLDocument.checkInternalConsistency() and
 * SBMLDocument.checkConsistency().  Please refer to the documentation for
 * {@link SBMLDocument} for more information about this.
 * <p>
 * While applications may play fast and loose and live like free spirits
 * during the construction and editing of SBML models, they should always
 * make sure to call SBMLDocument.checkInternalConsistency() and/or
 * SBMLDocument.checkConsistency() before writing out the final version of
 * an SBML model.
 */
"

%javamethodmodifiers Model::Model(unsigned int level, unsigned int version) "
  /**
   * Creates a new {@link Model} using the given SBML <code>level</code> and <code>version</code>
   * values.
   * <p>
   * @param level a long integer, the SBML Level to assign to this {@link Model}
   * <p>
   * @param version a long integer, the SBML Version to assign to this
   * {@link Model}
   * <p>
   * @note Once a {@link Model} has been added to an {@link SBMLDocument}, the <code>level</code>,
   * <code>version</code> for the document <em>override</em> those used
   * to create the {@link Model}.  Despite this, the ability to supply the values
   * at creation time is an important aid to creating valid SBML.  Knowledge of
   * the intented SBML Level and Version determine whether it is valid to
   * assign a particular value to an attribute, or whether it is valid to add
   * an object to an existing {@link SBMLDocument}.
   */
 public";

%javamethodmodifiers Model::Model(SBMLNamespaces* sbmlns) "
  /**
   * Creates a new {@link Model} using the given {@link SBMLNamespaces} object
   * <code>sbmlns</code>.
   * <p>
   * The {@link SBMLNamespaces} object encapsulates SBML Level/Version/namespaces
   * information.  It is used to communicate the SBML Level, Version, and
   * (in Level&nbsp;3) packages used in addition to SBML Level&nbsp; Core.
   * A common approach to using this class constructor is to create an
   * {@link SBMLNamespaces} object somewhere in a program, once, then pass it to
   * object constructors such as this one when needed.
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object.
   * <p>
   * @note Once a {@link Model} has been added to an {@link SBMLDocument}, the <code>level</code>,
   * <code>version</code> and <code>xmlns</code> namespaces for the document <em>override</em> those used
   * to create the {@link Model}.  Despite this, the ability to supply the values
   * at creation time is an important aid to creating valid SBML.  Knowledge of
   * the intented SBML Level and Version determine whether it is valid to
   * assign a particular value to an attribute, or whether it is valid to add
   * an object to an existing {@link SBMLDocument}.
   */
 public";

%javamethodmodifiers Model::Model(const Model& orig) "
  /**
  * Copy constructor; creates a (deep) copy of the given {@link Model} object.
  */
 public";

%javamethodmodifiers Model::accept(SBMLVisitor& v) const "
  /**
   * Accepts the given {@link SBMLVisitor} for this instance of {@link Constraint}.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used.
   * <p>
   * @return the result of calling <code>v.visit()</code>.
   */
 public";

%javamethodmodifiers Model::clone() const "
  /**
   * Creates and returns a deep copy of this {@link Model} object.
   * <p>
   * @return a (deep) copy of this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getId() const "
  /**
   * Returns the value of the \'id\' attribute of this {@link Model}.
   * <p>
   * @return the id of this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getName() const "
  /**
   * Returns the value of the \'name\' attribute of this {@link Model}.
   * <p>
   * @return the name of this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getModelHistory() const "
  /**
   * Returns the {@link ModelHistory} of this {@link Model}.
   * <p>
   * @return {@link ModelHistory} of this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getModelHistory "
  /**
   * Returns the {@link ModelHistory} of this {@link Model}.
   * <p>
   * @return {@link ModelHistory} of this {@link Model}.
   */
 public";

%javamethodmodifiers Model::isSetId() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link Model}\'s \'id\' attribute has been set.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the \'id\' attribute of this {@link Model} has been
   * set, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers Model::isSetName() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link Model}\'s \'name\' attribute has been set.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the \'name\' attribute of this {@link Model} has been
   * set, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers Model::isSetModelHistory "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link Model}\'s {@link ModelHistory} has been set.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the {@link ModelHistory} of this {@link Model}
   * has been set, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers Model::setId(const std::string& sid) "
  /**
   * Sets the value of the \'id\' attribute of this {@link Model}.
   * <p>
   * The string <code>sid</code> is copied.  Note that SBML has strict requirements
   * for the syntax of identifiers.  The following is summary of the
   * definition of the SBML identifier type <code>SId</code> (here expressed in an
   * extended form of BNF notation):
   * <div class=\'fragment\'><pre>
   *   letter .= \'a\'..\'z\',\'A\'..\'Z\'
   *   digit  .= \'0\'..\'9\'
   *   idChar .= letter | digit | \'_\'
   *   SId    .= ( letter | \'_\' ) idChar</pre></div>
   * The equality of SBML identifiers is determined by an exact character
   * sequence match; i.e., comparisons must be performed in a
   * case-sensitive manner.  In addition, there are a few conditions for
   * the uniqueness of identifiers in an SBML model.  Please consult the
   * SBML specifications for the exact formulations.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param sid the string to use as the identifier of this {@link Model}
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   */
 public";

%javamethodmodifiers Model::setName(const std::string& name) "
  /**
   * Sets the value of the \'name\' attribute of this {@link Model}.
   * <p>
   * The string in <code>name</code> is copied.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param name the new name for the {@link Model}
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   */
 public";

%javamethodmodifiers Model::setModelHistory(ModelHistory * history) "
  /**
   * Sets the {@link ModelHistory} of this {@link Model}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param history {@link ModelHistory} of this {@link Model}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_OBJECT
   */
 public";

%javamethodmodifiers Model::unsetId "
  /**
   * Unsets the value of the \'id\' attribute of this {@link Model}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_OPERATION_FAILED
   */
 public";

%javamethodmodifiers Model::unsetName "
  /**
   * Unsets the value of the \'name\' attribute of this {@link Model}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_OPERATION_FAILED
   */
 public";

%javamethodmodifiers Model::unsetModelHistory "
  /**
   * Unsets the {@link ModelHistory} of this {@link Model}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_OPERATION_FAILED
   */
 public";

%javamethodmodifiers Model::addFunctionDefinition(const FunctionDefinition* fd) "
  /**
   * Adds a copy of the given {@link FunctionDefinition} object to this {@link Model}.
   * <p>
   * @param fd the {@link FunctionDefinition} to add
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_LEVEL_MISMATCH
   * <li> LIBSBML_VERSION_MISMATCH
   * <li> LIBSBML_DUPLICATE_OBJECT_ID
   * <li> LIBSBML_OPERATION_FAILED
   * <p>
   * @note This method should be used with some caution.  The fact that
   * this method <em>copies</em> the object passed to it means that the caller
   * will be left holding a physically different object instance than the
   * one contained in this {@link Model}.  Changes made to the original object
   * instance (such as resetting attribute values) will <em>not affect the
   * instance in the {@link Model}</em>.  In addition, the caller should make sure
   * to free the original object if it is no longer being used, or else a
   * memory leak will result.  Please see Model.createFunctionDefinition()
   * for a method that does not lead to these issues.
   * <p>
   * @see #createFunctionDefinition()
   */
 public";

%javamethodmodifiers Model::addUnitDefinition(const UnitDefinition* ud) "
  /**
   * Adds a copy of the given {@link UnitDefinition} object to this {@link Model}.
   * <p>
   * @param ud the {@link UnitDefinition} object to add
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_LEVEL_MISMATCH
   * <li> LIBSBML_VERSION_MISMATCH
   * <li> LIBSBML_DUPLICATE_OBJECT_ID
   * <li> LIBSBML_OPERATION_FAILED
   * <p>
   * @note This method should be used with some caution.  The fact that
   * this method <em>copies</em> the object passed to it means that the caller
   * will be left holding a physically different object instance than the
   * one contained in this {@link Model}.  Changes made to the original object
   * instance (such as resetting attribute values) will <em>not affect the
   * instance in the {@link Model}</em>.  In addition, the caller should make sure
   * to free the original object if it is no longer being used, or else a
   * memory leak will result.  Please see Model.createUnitDefinition() for
   * a method that does not lead to these issues.
   * <p>
   * @see #createUnitDefinition()
   */
 public";

%javamethodmodifiers Model::addCompartmentType(const CompartmentType* ct) "
  /**
   * Adds a copy of the given {@link CompartmentType} object to this {@link Model}.
   * <p>
   * @param ct the {@link CompartmentType} object to add
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_LEVEL_MISMATCH
   * <li> LIBSBML_VERSION_MISMATCH
   * <li> LIBSBML_DUPLICATE_OBJECT_ID
   * <li> LIBSBML_OPERATION_FAILED
   * <p>
   * @note This method should be used with some caution.  The fact that
   * this method <em>copies</em> the object passed to it means that the caller
   * will be left holding a physically different object instance than the
   * one contained in this {@link Model}.  Changes made to the original object
   * instance (such as resetting attribute values) will <em>not affect the
   * instance in the {@link Model}</em>.  In addition, the caller should make sure
   * to free the original object if it is no longer being used, or else a
   * memory leak will result.  Please see Model.createCompartmentType()
   * for a method that does not lead to these issues.
   * <p>
   * @see #createCompartmentType()
   */
 public";

%javamethodmodifiers Model::addSpeciesType(const SpeciesType* st) "
  /**
   * Adds a copy of the given {@link SpeciesType} object to this {@link Model}.
   * <p>
   * @param st the {@link SpeciesType} object to add
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_LEVEL_MISMATCH
   * <li> LIBSBML_VERSION_MISMATCH
   * <li> LIBSBML_DUPLICATE_OBJECT_ID
   * <li> LIBSBML_OPERATION_FAILED
   * <p>
   * @note This method should be used with some caution.  The fact that
   * this method <em>copies</em> the object passed to it means that the caller
   * will be left holding a physically different object instance than the
   * one contained in this {@link Model}.  Changes made to the original object
   * instance (such as resetting attribute values) will <em>not affect the
   * instance in the {@link Model}</em>.  In addition, the caller should make sure
   * to free the original object if it is no longer being used, or else a
   * memory leak will result.  Please see Model.createSpeciesType() for a
   * method that does not lead to these issues.
   * <p>
   * @see #createSpeciesType()
   */
 public";

%javamethodmodifiers Model::addCompartment(const Compartment* c) "
  /**
   * Adds a copy of the given {@link Compartment} object to this {@link Model}.
   * <p>
   * @param c the {@link Compartment} object to add
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_LEVEL_MISMATCH
   * <li> LIBSBML_VERSION_MISMATCH
   * <li> LIBSBML_DUPLICATE_OBJECT_ID
   * <li> LIBSBML_OPERATION_FAILED
   * <p>
   * @note This method should be used with some caution.  The fact that
   * this method <em>copies</em> the object passed to it means that the caller
   * will be left holding a physically different object instance than the
   * one contained in this {@link Model}.  Changes made to the original object
   * instance (such as resetting attribute values) will <em>not affect the
   * instance in the {@link Model}</em>.  In addition, the caller should make sure
   * to free the original object if it is no longer being used, or else a
   * memory leak will result.  Please see Model.createCompartment() for a
   * method that does not lead to these issues.
   * <p>
   * @see #createCompartment()
   */
 public";

%javamethodmodifiers Model::addSpecies(const Species* s) "
  /**
   * Adds a copy of the given {@link Species} object to this {@link Model}.
   * <p>
   * @param s the {@link Species} object to add
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_LEVEL_MISMATCH
   * <li> LIBSBML_VERSION_MISMATCH
   * <li> LIBSBML_DUPLICATE_OBJECT_ID
   * <li> LIBSBML_OPERATION_FAILED
   * <p>
   * @note This method should be used with some caution.  The fact that
   * this method <em>copies</em> the object passed to it means that the caller
   * will be left holding a physically different object instance than the
   * one contained in this {@link Model}.  Changes made to the original object
   * instance (such as resetting attribute values) will <em>not affect the
   * instance in the {@link Model}</em>.  In addition, the caller should make sure
   * to free the original object if it is no longer being used, or else a
   * memory leak will result.  Please see Model.createSpecies() for a
   * method that does not lead to these issues.
   * <p>
   * @see #createSpecies()
   */
 public";

%javamethodmodifiers Model::addParameter(const Parameter* p) "
  /**
   * Adds a copy of the given {@link Parameter} object to this {@link Model}.
   * <p>
   * @param p the {@link Parameter} object to add
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_LEVEL_MISMATCH
   * <li> LIBSBML_VERSION_MISMATCH
   * <li> LIBSBML_DUPLICATE_OBJECT_ID
   * <li> LIBSBML_OPERATION_FAILED
   * <p>
   * @note This method should be used with some caution.  The fact that
   * this method <em>copies</em> the object passed to it means that the caller
   * will be left holding a physically different object instance than the
   * one contained in this {@link Model}.  Changes made to the original object
   * instance (such as resetting attribute values) will <em>not affect the
   * instance in the {@link Model}</em>.  In addition, the caller should make sure
   * to free the original object if it is no longer being used, or else a
   * memory leak will result.  Please see Model.createParameter() for a
   * method that does not lead to these issues.
   * <p>
   * @see #createParameter()
   */
 public";

%javamethodmodifiers Model::addInitialAssignment(const InitialAssignment* ia) "
  /**
   * Adds a copy of the given {@link InitialAssignment} object to this {@link Model}.
   * <p>
   * @param ia the {@link InitialAssignment} object to add
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_LEVEL_MISMATCH
   * <li> LIBSBML_VERSION_MISMATCH
   * <li> LIBSBML_DUPLICATE_OBJECT_ID
   * <li> LIBSBML_OPERATION_FAILED
   * <p>
   * @note This method should be used with some caution.  The fact that
   * this method <em>copies</em> the object passed to it means that the caller
   * will be left holding a physically different object instance than the
   * one contained in this {@link Model}.  Changes made to the original object
   * instance (such as resetting attribute values) will <em>not affect the
   * instance in the {@link Model}</em>.  In addition, the caller should make sure
   * to free the original object if it is no longer being used, or else a
   * memory leak will result.  Please see Model.createInitialAssignment()
   * for a method that does not lead to these issues.
   * <p>
   * @see #createInitialAssignment()
   */
 public";

%javamethodmodifiers Model::addRule(const Rule* r) "
  /**
   * Adds a copy of the given {@link Rule} object to this {@link Model}.
   * <p>
   * @param r the {@link Rule} object to add
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_LEVEL_MISMATCH
   * <li> LIBSBML_VERSION_MISMATCH
   * <li> LIBSBML_DUPLICATE_OBJECT_ID
   * <li> LIBSBML_OPERATION_FAILED
   * <p>
   * @note This method should be used with some caution.  The fact that
   * this method <em>copies</em> the object passed to it means that the caller
   * will be left holding a physically different object instance than the
   * one contained in this {@link Model}.  Changes made to the original object
   * instance (such as resetting attribute values) will <em>not affect the
   * instance in the {@link Model}</em>.  In addition, the caller should make sure
   * to free the original object if it is no longer being used, or else a
   * memory leak will result.  Please see the methods
   * Model.createAlgebraicRule(), Model.createAssignmentRule() and
   * Model.createRateRule() for methods that do not lead to these issues.
   * <p>
   * @see #createAlgebraicRule()
   * @see #createAssignmentRule()
   * @see #createRateRule()
   */
 public";

%javamethodmodifiers Model::addConstraint(const Constraint* c) "
  /**
   * Adds a copy of the given {@link Constraint} object to this {@link Model}.
   * <p>
   * @param c the {@link Constraint} object to add
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_LEVEL_MISMATCH
   * <li> LIBSBML_VERSION_MISMATCH
   * <li> LIBSBML_OPERATION_FAILED
   * <p>
   * @note This method should be used with some caution.  The fact that
   * this method <em>copies</em> the object passed to it means that the caller
   * will be left holding a physically different object instance than the
   * one contained in this {@link Model}.  Changes made to the original object
   * instance (such as resetting attribute values) will <em>not affect the
   * instance in the {@link Model}</em>.  In addition, the caller should make sure
   * to free the original object if it is no longer being used, or else a
   * memory leak will result.  Please see Model.createConstraint() for a
   * method that does not lead to these issues.
   * <p>
   * @see #createConstraint()
   */
 public";

%javamethodmodifiers Model::addReaction(const Reaction* r) "
  /**
   * Adds a copy of the given {@link Reaction} object to this {@link Model}.
   * <p>
   * @param r the {@link Reaction} object to add
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_LEVEL_MISMATCH
   * <li> LIBSBML_VERSION_MISMATCH
   * <li> LIBSBML_DUPLICATE_OBJECT_ID
   * <li> LIBSBML_OPERATION_FAILED
   * <p>
   * @note This method should be used with some caution.  The fact that
   * this method <em>copies</em> the object passed to it means that the caller
   * will be left holding a physically different object instance than the
   * one contained in this {@link Model}.  Changes made to the original object
   * instance (such as resetting attribute values) will <em>not affect the
   * instance in the {@link Model}</em>.  In addition, the caller should make sure
   * to free the original object if it is no longer being used, or else a
   * memory leak will result.  Please see Model.createReaction() for a
   * method that does not lead to these issues.
   * <p>
   * @see #createReaction()
   */
 public";

%javamethodmodifiers Model::addEvent(const Event* e) "
  /**
   * Adds a copy of the given {@link Event} object to this {@link Model}.
   * <p>
   * @param e the {@link Event} object to add
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_LEVEL_MISMATCH
   * <li> LIBSBML_VERSION_MISMATCH
   * <li> LIBSBML_DUPLICATE_OBJECT_ID
   * <li> LIBSBML_OPERATION_FAILED
   * <p>
   * @note This method should be used with some caution.  The fact that
   * this method <em>copies</em> the object passed to it means that the caller
   * will be left holding a physically different object instance than the
   * one contained in this {@link Model}.  Changes made to the original object
   * instance (such as resetting attribute values) will <em>not affect the
   * instance in the {@link Model}</em>.  In addition, the caller should make sure
   * to free the original object if it is no longer being used, or else a
   * memory leak will result.  Please see Model.createEvent() for a method
   * that does not lead to these issues.
   * <p>
   * @see #createEvent()
   */
 public";

%javamethodmodifiers Model::createFunctionDefinition "
  /**
   * Creates a new {@link FunctionDefinition} inside this {@link Model} and returns it.
   * <p>
   * @return the {@link FunctionDefinition} object created
   * <p>
   * @see #addFunctionDefinition(FunctionDefinition  fd)
   */
 public";

%javamethodmodifiers Model::createUnitDefinition "
  /**
   * Creates a new {@link UnitDefinition} inside this {@link Model} and returns it.
   * <p>
   * @return the {@link UnitDefinition} object created
   * <p>
   * @see #addUnitDefinition(UnitDefinition  ud)
   */
 public";

%javamethodmodifiers Model::createUnit "
  /**
   * Creates a new {@link Unit} object within the last {@link UnitDefinition} object
   * created in this model and returns a pointer to it.
   * <p>
   * The mechanism by which the {@link UnitDefinition} was created is not
   * significant.  If a {@link UnitDefinition} object does not exist in this model,
   * a new {@link Unit} is <em>not</em> created and NULL is returned instead.
   * <p>
   * @return the {@link Unit} object created
   * <p>
   * @see #addUnitDefinition(UnitDefinition  ud)
   */
 public";

%javamethodmodifiers Model::createCompartmentType "
  /**
   * Creates a new {@link CompartmentType} inside this {@link Model} and returns it.
   * <p>
   * @return the {@link CompartmentType} object created
   * <p>
   * @see #addCompartmentType(CompartmentType  ct)
   */
 public";

%javamethodmodifiers Model::createSpeciesType "
  /**
   * Creates a new {@link SpeciesType} inside this {@link Model} and returns it.
   * <p>
   * @return the {@link SpeciesType} object created
   * <p>
   * @see #addSpeciesType(SpeciesType  st)
   */
 public";

%javamethodmodifiers Model::createCompartment "
  /**
   * Creates a new {@link Compartment} inside this {@link Model} and returns it.
   * <p>
   * @return the {@link Compartment} object created
   * <p>
   * @see #addCompartment(Compartment c)
   */
 public";

%javamethodmodifiers Model::createSpecies "
  /**
   * Creates a new {@link Species} inside this {@link Model} and returns it.
   * <p>
   * @return the {@link Species} object created
   * <p>
   * @see #addSpecies(Species s)
   */
 public";

%javamethodmodifiers Model::createParameter "
  /**
   * Creates a new {@link Parameter} inside this {@link Model} and returns it.
   * <p>
   * @return the {@link Parameter} object created
   * <p>
   * @see #addParameter(Parameter p)
   */
 public";

%javamethodmodifiers Model::createInitialAssignment "
  /**
   * Creates a new {@link InitialAssignment} inside this {@link Model} and returns it.
   * <p>
   * @return the {@link InitialAssignment} object created
   * <p>
   * @see #addInitialAssignment(InitialAssignment  ia)
   */
 public";

%javamethodmodifiers Model::createAlgebraicRule "
  /**
   * Creates a new {@link AlgebraicRule} inside this {@link Model} and returns it.
   * <p>
   * @return the {@link AlgebraicRule} object created
   * <p>
   * @see #addRule(Rule  r)
   */
 public";

%javamethodmodifiers Model::createAssignmentRule "
  /**
   * Creates a new {@link AssignmentRule} inside this {@link Model} and returns it.
   * <p>
   * @return the {@link AssignmentRule} object created
   * <p>
   * @see #addRule(Rule  r)
   */
 public";

%javamethodmodifiers Model::createRateRule "
  /**
   * Creates a new {@link RateRule} inside this {@link Model} and returns it.
   * <p>
   * @return the {@link RateRule} object created
   * <p>
   * @see #addRule(Rule  r)
   */
 public";

%javamethodmodifiers Model::createConstraint "
  /**
   * Creates a new {@link Constraint} inside this {@link Model} and returns it.
   * <p>
   * @return the {@link Constraint} object created
   * <p>
   * @see #addConstraint(Constraint c)
   */
 public";

%javamethodmodifiers Model::createReaction "
  /**
   * Creates a new {@link Reaction} inside this {@link Model} and returns it.
   * <p>
   * @return the {@link Reaction} object created
   * <p>
   * @see #addReaction(Reaction r)
   */
 public";

%javamethodmodifiers Model::createReactant "
  /**
   * Creates a new {@link SpeciesReference} object for a reactant inside the last
   * {@link Reaction} object in this {@link Model}, and returns a pointer to it.
   * <p>
   * The mechanism by which the last {@link Reaction} object was created and added
   * to this {@link Model} is not significant.  It could have been created in a
   * variety of ways, for example using createReaction().  If a {@link Reaction}
   * does not exist for this model, a new {@link SpeciesReference} is <em>not</em>
   * created and NULL is returned instead.
   * <p>
   * @return the {@link SpeciesReference} object created
   */
 public";

%javamethodmodifiers Model::createProduct "
  /**
   * Creates a new {@link SpeciesReference} object for a product inside the last
   * {@link Reaction} object in this {@link Model}, and returns a pointer to it.
   * <p>
   * The mechanism by which the last {@link Reaction} object was created and added
   * to this {@link Model} is not significant.  It could have been created in a
   * variety of ways, for example using createReaction().  If a {@link Reaction}
   * does not exist for this model, a new {@link SpeciesReference} is <em>not</em>
   * created and NULL is returned instead.
   * <p>
   * @return the {@link SpeciesReference} object created
   */
 public";

%javamethodmodifiers Model::createModifier "
  /**
   * Creates a new {@link ModifierSpeciesReference} object for a modifier species
   * inside the last {@link Reaction} object in this {@link Model}, and returns a pointer
   * to it.
   * <p>
   * The mechanism by which the last {@link Reaction} object was created and added
   * to this {@link Model} is not significant.  It could have been created in a
   * variety of ways, for example using createReaction().  If a {@link Reaction}
   * does not exist for this model, a new {@link ModifierSpeciesReference} is 
   * <em>not</em> created and NULL is returned instead.
   * <p>
   * @return the {@link SpeciesReference} object created
   */
 public";

%javamethodmodifiers Model::createKineticLaw "
  /**
   * Creates a new {@link KineticLaw} inside the last {@link Reaction} object created in
   * this {@link Model}, and returns a pointer to it.
   * <p>
   * The mechanism by which the last {@link Reaction} object was created and added
   * to this {@link Model} is not significant.  It could have been created in a
   * variety of ways, for example using createReaction().  If a {@link Reaction}
   * does not exist for this model, or a {@link Reaction} exists but already has a
   * {@link KineticLaw}, a new {@link KineticLaw} is <em>not</em> created and NULL is returned
   * instead.
   * <p>
   * @return the {@link KineticLaw} object created
   */
 public";

%javamethodmodifiers Model::createKineticLawParameter "
  /**
   * Creates a new local {@link Parameter} inside the {@link KineticLaw} object of the last
   * {@link Reaction} created inside this {@link Model}, and returns a pointer to it.
   * <p>
   * The last {@link KineticLaw} object in this {@link Model} could have been created in a
   * variety of ways.  For example, it could have been added using
   * createKineticLaw(), or it could be the result of using
   * Reaction.createKineticLaw() on the {@link Reaction} object created by a
   * createReaction().  If a {@link Reaction} does not exist for this model, or the
   * last {@link Reaction} does not contain a {@link KineticLaw} object, a new {@link Parameter} is
   * <em>not</em> created and NULL is returned instead.
   * <p>
   * @return the {@link Parameter} object created
   */
 public";

%javamethodmodifiers Model::createEvent "
  /**
   * Creates a new {@link Event} inside this {@link Model} and returns it.
   * <p>
   * @return the {@link Event} object created
   */
 public";

%javamethodmodifiers Model::createEventAssignment "
  /**
   * Creates a new {@link EventAssignment} inside the last {@link Event} object created in
   * this {@link Model}, and returns a pointer to it.
   * <p>
   * The mechanism by which the last {@link Event} object in this model was created
   * is not significant.  It could have been created in a variety of ways,
   * for example by using createEvent().  If no {@link Event} object exists in this
   * {@link Model} object, a new {@link EventAssignment} is <em>not</em> created and NULL is
   * returned instead.
   * <p>
   * @return the {@link EventAssignment} object created
   */
 public";

%javamethodmodifiers Model::createTrigger "
  /**
   * Creates a new {@link Trigger} inside the last {@link Event} object created in
   * this {@link Model}, and returns a pointer to it.
   * <p>
   * The mechanism by which the last {@link Event} object in this model was created
   * is not significant.  It could have been created in a variety of ways,
   * for example by using createEvent().  If no {@link Event} object exists in this
   * {@link Model} object, a new {@link Trigger} is <em>not</em> created and NULL is
   * returned instead.
   * <p>
   * @return the {@link Trigger} object created
   */
 public";

%javamethodmodifiers Model::createDelay "
  /**
   * Creates a new {@link Delay} inside the last {@link Event} object created in
   * this {@link Model}, and returns a pointer to it.
   * <p>
   * The mechanism by which the last {@link Event} object in this model was created
   * is not significant.  It could have been created in a variety of ways,
   * for example by using createEvent().  If no {@link Event} object exists in this
   * {@link Model} object, a new {@link Delay} is <em>not</em> created and NULL is
   * returned instead.
   * <p>
   * @return the {@link Delay} object created
   */
 public";

%javamethodmodifiers Model::setAnnotation(const XMLNode* annotation) "
  /**
   * Sets the value of the \'annotation\' subelement of this SBML object to a
   * copy of <code>annotation</code>.
   * <p>
   * Any existing content of the \'annotation\' subelement is discarded.
   * Unless you have taken steps to first copy and reconstitute any
   * existing annotations into the <code>annotation</code> that is about to be
   * assigned, it is likely that performing such wholesale replacement is
   * unfriendly towards other software applications whose annotations are
   * discarded.  An alternative may be to use appendAnnotation().
   * <p>
   * @param annotation an XML structure that is to be used as the content
   * of the \'annotation\' subelement of this object
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <p>
   * @see #appendAnnotation(XMLNode  annotation)
   */
 public";

%javamethodmodifiers Model::setAnnotation(const std::string& annotation) "
  /**
   * Sets the value of the \'annotation\' subelement of this SBML object to a
   * copy of <code>annotation</code>.
   * <p>
   * Any existing content of the \'annotation\' subelement is discarded.
   * Unless you have taken steps to first copy and reconstitute any
   * existing annotations into the <code>annotation</code> that is about to be
   * assigned, it is likely that performing such wholesale replacement is
   * unfriendly towards other software applications whose annotations are
   * discarded.  An alternative may be to use appendAnnotation().
   * <p>
   * @param annotation an XML string that is to be used as the content
   * of the \'annotation\' subelement of this object
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_OPERATION_FAILED
   * <p>
   * @see #appendAnnotation(String annotation)
   */
 public";

%javamethodmodifiers Model::appendAnnotation(const XMLNode* annotation) "
  /**
   * Appends annotation content to any existing content in the \'annotation\'
   * subelement of this object.
   * <p>
   * The content in <code>annotation</code> is copied.  Unlike setAnnotation(), this
   * method allows other annotations to be preserved when an application
   * adds its own data.
   * <p>
   * @param annotation an XML structure that is to be copied and appended
   * to the content of the \'annotation\' subelement of this object
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_OPERATION_FAILED
   * <p>
   * @see #setAnnotation(XMLNode  annotation)
   */
 public";

%javamethodmodifiers Model::appendAnnotation(const std::string& annotation) "
  /**
   * Appends annotation content to any existing content in the \'annotation\'
   * subelement of this object.
   * <p>
   * The content in <code>annotation</code> is copied.  Unlike setAnnotation(), this 
   * method allows other annotations to be preserved when an application
   * adds its own data.
   * <p>
   * @param annotation an XML string that is to be copied and appended
   * to the content of the \'annotation\' subelement of this object
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_OPERATION_FAILED
   * <p>
   * @see #setAnnotation(String annotation)
   */
 public";

%javamethodmodifiers Model::getListOfFunctionDefinitions() const "
  /**
   * Get the {@link ListOfFunctionDefinitions} object in this {@link Model}.
   * <p>
   * @return the list of FunctionDefinitions for this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getListOfFunctionDefinitions "
  /**
   * Get the {@link ListOfFunctionDefinitions} object in this {@link Model}.
   * <p>
   * @return the list of FunctionDefinitions for this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getListOfUnitDefinitions() const "
  /**
   * Get the {@link ListOfUnitDefinitions} object in this {@link Model}.
   * <p>
   * @return the list of UnitDefinitions for this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getListOfUnitDefinitions "
  /**
   * Get the {@link ListOfUnitDefinitions} object in this {@link Model}.
   * <p>
   * @return the list of UnitDefinitions for this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getListOfCompartmentTypes() const "
  /**
   * Get the {@link ListOfCompartmentTypes} object in this {@link Model}.
   * <p>
   * @return the list of CompartmentTypes for this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getListOfCompartmentTypes "
  /**
   * Get the {@link ListOfCompartmentTypes} object in this {@link Model}.
   * <p>
   * @return the list of CompartmentTypes for this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getListOfSpeciesTypes() const "
  /**
   * Get the {@link ListOfSpeciesTypes} object in this {@link Model}.
   * <p>
   * @return the list of SpeciesTypes for this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getListOfSpeciesTypes "
  /**
   * Get the {@link ListOfSpeciesTypes} object in this {@link Model}.
   * <p>
   * @return the list of SpeciesTypes for this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getListOfCompartments() const "
  /**
   * Get the {@link ListOfCompartments} object in this {@link Model}.
   * <p>
   * @return the list of Compartments for this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getListOfCompartments "
  /**
   * Get the {@link ListOfCompartments} object in this {@link Model}.
   * <p>
   * @return the list of Compartments for this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getListOfSpecies() const "
  /**
   * Get the {@link ListOfSpecies} object in this {@link Model}.
   * <p>
   * @return the list of {@link Species} for this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getListOfSpecies "
  /**
   * Get the {@link ListOfSpecies} object in this {@link Model}.
   * <p>
   * @return the list of {@link Species} for this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getListOfParameters() const "
  /**
   * Get the {@link ListOfParameters} object in this {@link Model}.
   * <p>
   * @return the list of Parameters for this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getListOfParameters "
  /**
   * Get the {@link ListOfParameters} object in this {@link Model}.
   * <p>
   * @return the list of Parameters for this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getListOfInitialAssignments() const "
  /**
   * Get the {@link ListOfInitialAssignments} object in this {@link Model}.
   * <p>
   * @return the list of InitialAssignments for this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getListOfInitialAssignments "
  /**
   * Get the {@link ListOfInitialAssignments} object in this {@link Model}.
   * <p>
   * @return the list of {@link InitialAssignment} for this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getListOfRules() const "
  /**
   * Get the {@link ListOfRules} object in this {@link Model}.
   * <p>
   * @return the list of Rules for this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getListOfRules "
  /**
   * Get the {@link ListOfRules} object in this {@link Model}.
   * <p>
   * @return the list of Rules for this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getListOfConstraints() const "
  /**
   * Get the {@link ListOfConstraints} object in this {@link Model}.
   * <p>
   * @return the list of Constraints for this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getListOfConstraints "
  /**
   * Get the {@link ListOfConstraints} object in this {@link Model}.
   * <p>
   * @return the list of Constraints for this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getListOfReactions() const "
  /**
   * Get the {@link ListOfReactions} object in this {@link Model}.
   * <p>
   * @return the list of Reactions for this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getListOfReactions "
  /**
   * Get the {@link ListOfReactions} object in this {@link Model}.
   * <p>
   * @return the list of Reactions for this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getListOfEvents() const "
  /**
   * Get the {@link ListOfEvents} object in this {@link Model}.
   * <p>
   * @return the list of Events for this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getListOfEvents "
  /**
   * Get the {@link ListOfEvents} object in this {@link Model}.
   * <p>
   * @return the list of Events for this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getFunctionDefinition "
  /**
   * Get the nth FunctionDefinitions object in this {@link Model}.
   * <p>
   * @return the nth {@link FunctionDefinition} of this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getFunctionDefinition(unsigned int n) "
  /**
   * Get the nth FunctionDefinitions object in this {@link Model}.
   * <p>
   * @return the nth {@link FunctionDefinition} of this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getFunctionDefinition(const std::string& sid) const "
  /**
   * Get a {@link FunctionDefinition} object based on its identifier.
   * <p>
   * @return the {@link FunctionDefinition} in this {@link Model} with the identifier
   * <code>sid</code> or NULL if no such {@link FunctionDefinition} exists.
   */
 public";

%javamethodmodifiers Model::getFunctionDefinition(const std::string& sid) "
  /**
   * Get a {@link FunctionDefinition} object based on its identifier.
   * <p>
   * @return the {@link FunctionDefinition} in this {@link Model} with the identifier
   * <code>sid</code> or NULL if no such {@link FunctionDefinition} exists.
   */
 public";

%javamethodmodifiers Model::getUnitDefinition "
  /**
   * Get the nth {@link UnitDefinition} object in this {@link Model}.
   * <p>
   * @return the nth {@link UnitDefinition} of this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getUnitDefinition(unsigned int n) "
  /**
   * Get the nth {@link UnitDefinition} object in this {@link Model}.
   * <p>
   * @return the nth {@link UnitDefinition} of this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getUnitDefinition(const std::string& sid) const "
  /**
   * Get a {@link UnitDefinition} based on its identifier.
   * <p>
   * @return the {@link UnitDefinition} in this {@link Model} with the identifier <code>sid</code> or
   * NULL if no such {@link UnitDefinition} exists.
   */
 public";

%javamethodmodifiers Model::getUnitDefinition(const std::string& sid) "
  /**
   * Get a {@link UnitDefinition} based on its identifier.
   * <p>
   * @return the {@link UnitDefinition} in this {@link Model} with the identifier <code>sid</code> or
   * NULL if no such {@link UnitDefinition} exists.
   */
 public";

%javamethodmodifiers Model::getCompartmentType "
  /**
   * Get the nth {@link CompartmentType} object in this {@link Model}.
   * <p>
   * @return the nth {@link CompartmentType} of this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getCompartmentType(unsigned int n) "
  /**
   * Get the nth {@link CompartmentType} object in this {@link Model}.
   * <p>
   * @return the nth {@link CompartmentType} of this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getCompartmentType(const std::string& sid) const "
  /**
   * Get a {@link CompartmentType} object based on its identifier.
   * <p>
   * @return the {@link CompartmentType} in this {@link Model} with the identifier <code>sid</code>
   * or NULL if no such {@link CompartmentType} exists.
   */
 public";

%javamethodmodifiers Model::getCompartmentType(const std::string& sid) "
  /**
   * Get a {@link CompartmentType} object based on its identifier.
   * <p>
   * @return the {@link CompartmentType} in this {@link Model} with the identifier <code>sid</code>
   * or NULL if no such {@link CompartmentType} exists.
   */
 public";

%javamethodmodifiers Model::getSpeciesType "
  /**
   * Get the nth {@link SpeciesType} object in this {@link Model}.
   * <p>
   * @return the nth {@link SpeciesType} of this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getSpeciesType(unsigned int n) "
  /**
   * Get the nth {@link SpeciesType} object in this {@link Model}.
   * <p>
   * @return the nth {@link SpeciesType} of this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getSpeciesType(const std::string& sid) const "
  /**
   * Get a {@link SpeciesType} object based on its identifier.
   * <p>
   * @return the {@link SpeciesType} in this {@link Model} with the identifier <code>sid</code> or
   * NULL if no such {@link SpeciesType} exists.
   */
 public";

%javamethodmodifiers Model::getSpeciesType(const std::string& sid) "
  /**
   * Get a {@link SpeciesType} object based on its identifier.
   * <p>
   * @return the {@link SpeciesType} in this {@link Model} with the identifier <code>sid</code> or
   * NULL if no such {@link SpeciesType} exists.
   */
 public";

%javamethodmodifiers Model::getCompartment "
  /**
   * Get the nth {@link Compartment} object in this {@link Model}.
   * <p>
   * @return the nth {@link Compartment} of this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getCompartment(unsigned int n) "
  /**
   * Get the nth {@link Compartment} object in this {@link Model}.
   * <p>
   * @return the nth {@link Compartment} of this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getCompartment(const std::string& sid) const "
  /**
   * Get a {@link Compartment} object based on its identifier.
   * <p>
   * @return the {@link Compartment} in this {@link Model} with the identifier <code>sid</code> or
   * NULL if no such {@link Compartment} exists.
   */
 public";

%javamethodmodifiers Model::getCompartment(const std::string& sid) "
  /**
   * Get a {@link Compartment} object based on its identifier.
   * <p>
   * @return the {@link Compartment} in this {@link Model} with the identifier <code>sid</code> or
   * NULL if no such {@link Compartment} exists.
   */
 public";

%javamethodmodifiers Model::getSpecies "
  /**
   * Get the nth {@link Species} object in this {@link Model}.
   * <p>
   * @return the nth {@link Species} of this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getSpecies(unsigned int n) "
  /**
   * Get the nth {@link Species} object in this {@link Model}.
   * <p>
   * @return the nth {@link Species} of this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getSpecies(const std::string& sid) const "
  /**
   * Get a {@link Species} object based on its identifier.
   * <p>
   * @return the {@link Species} in this {@link Model} with the identifier <code>sid</code> or NULL
   * if no such {@link Species} exists.
   */
 public";

%javamethodmodifiers Model::getSpecies(const std::string& sid) "
  /**
   * Get a {@link Species} object based on its identifier.
   * <p>
   * @return the {@link Species} in this {@link Model} with the identifier <code>sid</code> or NULL
   * if no such {@link Species} exists.
   */
 public";

%javamethodmodifiers Model::getParameter "
  /**
   * Get the nth {@link Parameter} object in this {@link Model}.
   * <p>
   * @return the nth {@link Parameter} of this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getParameter(unsigned int n) "
  /**
   * Get the nth {@link Parameter} object in this {@link Model}.
   * <p>
   * @return the nth {@link Parameter} of this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getParameter(const std::string& sid) const "
  /**
   * Get a {@link Parameter} object based on its identifier.
   * <p>
   * @return the {@link Parameter} in this {@link Model} with the identifier <code>sid</code> or NULL
   * if no such {@link Parameter} exists.
   */
 public";

%javamethodmodifiers Model::getParameter(const std::string& sid) "
  /**
   * Get a {@link Parameter} object based on its identifier.
   * <p>
   * @return the {@link Parameter} in this {@link Model} with the identifier <code>sid</code> or NULL
   * if no such {@link Parameter} exists.
   */
 public";

%javamethodmodifiers Model::getInitialAssignment "
  /**
   * Get the nth {@link InitialAssignment} object in this {@link Model}.
   * <p>
   * @return the nth {@link InitialAssignment} of this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getInitialAssignment(unsigned int n) "
  /**
   * Get the nth {@link InitialAssignment} object in this {@link Model}.
   * <p>
   * @return the nth {@link InitialAssignment} of this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getInitialAssignment(const std::string& symbol) const "
  /**
   * Get an {@link InitialAssignment} object based on the symbol to which it
   * assigns a value.
   * <p>
   * @return the {@link InitialAssignment} in this {@link Model} with the given \'symbol\'
   * attribute value or NULL if no such {@link InitialAssignment} exists.
   */
 public";

%javamethodmodifiers Model::getInitialAssignment(const std::string& symbol) "
  /**
   * Get an {@link InitialAssignment} object based on the symbol to which it
   * assigns a value.
   * <p>
   * @return the {@link InitialAssignment} in this {@link Model} with the given \'symbol\'
   * attribute value or NULL if no such {@link InitialAssignment} exists.
   */
 public";

%javamethodmodifiers Model::getRule "
  /**
   * Get the nth {@link Rule} object in this {@link Model}.
   * <p>
   * @return the nth {@link Rule} of this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getRule(unsigned int n) "
  /**
   * Get the nth {@link Rule} object in this {@link Model}.
   * <p>
   * @return the nth {@link Rule} of this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getRule(const std::string& variable) const "
  /**
   * Get a {@link Rule} object based on the variable to which it assigns a value.
   * <p>
   * @return the {@link Rule} in this {@link Model} with the given \'variable\' attribute
   * value or NULL if no such {@link Rule} exists.
   */
 public";

%javamethodmodifiers Model::getRule(const std::string& variable) "
  /**
   * Get a {@link Rule} object based on the variable to which it assigns a value.
   * <p>
   * @return the {@link Rule} in this {@link Model} with the given \'variable\' attribute
   * value or NULL if no such {@link Rule} exists.
   */
 public";

%javamethodmodifiers Model::getConstraint "
  /**
   * Get the nth {@link Constraint} object in this {@link Model}.
   * <p>
   * @return the nth {@link Constraint} of this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getConstraint(unsigned int n) "
  /**
   * Get the nth {@link Constraint} object in this {@link Model}.
   * <p>
   * @return the nth {@link Constraint} of this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getReaction "
  /**
   * Get the nth {@link Reaction} object in this {@link Model}.
   * <p>
   * @return the nth {@link Reaction} of this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getReaction(unsigned int n) "
  /**
   * Get the nth {@link Reaction} object in this {@link Model}.
   * <p>
   * @return the nth {@link Reaction} of this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getReaction(const std::string& sid) const "
  /**
   * Get a {@link Reaction} object based on its identifier.
   * <p>
   * @return the {@link Reaction} in this {@link Model} with the identifier <code>sid</code> or NULL
   * if no such {@link Reaction} exists.
   */
 public";

%javamethodmodifiers Model::getReaction(const std::string& sid) "
  /**
   * Get a {@link Reaction} object based on its identifier.
   * <p>
   * @return the {@link Reaction} in this {@link Model} with the identifier <code>sid</code> or NULL
   * if no such {@link Reaction} exists.
   */
 public";

%javamethodmodifiers Model::getEvent "
  /**
   * Get the nth {@link Event} object in this {@link Model}.
   * <p>
   * @return the nth {@link Event} of this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getEvent(unsigned int n) "
  /**
   * Get the nth {@link Event} object in this {@link Model}.
   * <p>
   * @return the nth {@link Event} of this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getEvent(const std::string& sid) const "
  /**
   * Get an {@link Event} object based on its identifier.
   * <p>
   * @return the {@link Event} in this {@link Model} with the identifier <code>sid</code> or NULL if
   * no such {@link Event} exists.
   */
 public";

%javamethodmodifiers Model::getEvent(const std::string& sid) "
  /**
   * Get an {@link Event} object based on its identifier.
   * <p>
   * @return the {@link Event} in this {@link Model} with the identifier <code>sid</code> or NULL if
   * no such {@link Event} exists.
   */
 public";

%javamethodmodifiers Model::getNumFunctionDefinitions() const "
  /**
   * Get the number of {@link FunctionDefinition} objects in this {@link Model}.
   * <p>
   * @return the number of FunctionDefinitions in this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getNumUnitDefinitions() const "
  /**
   * Get the number of {@link UnitDefinition} objects in this {@link Model}.
   * <p>
   * @return the number of UnitDefinitions in this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getNumCompartmentTypes() const "
  /**
   * Get the number of {@link CompartmentType} objects in this {@link Model}.
   * <p>
   * @return the number of CompartmentTypes in this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getNumSpeciesTypes() const "
  /**
   * Get the number of {@link SpeciesType} objects in this {@link Model}.
   * <p>
   * @return the number of SpeciesTypes in this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getNumCompartments() const "
  /**
   * Get the number of {@link Compartment} objects in this {@link Model}.
   * <p>
   * @return the number of Compartments in this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getNumSpecies() const "
  /**
   * Get the number of Specie objects in this {@link Model}.
   * <p>
   * @return the number of {@link Species} in this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getNumSpeciesWithBoundaryCondition() const "
  /**
   * Get the number of {@link Species} in this {@link Model} having their
   * \'boundaryCondition\' attribute value set to <code>true</code>.
   * <p>
   * @return the number of {@link Species} in this {@link Model} with boundaryCondition set
   * to true.
   */
 public";

%javamethodmodifiers Model::getNumParameters() const "
  /**
   * Get the number of {@link Parameter} objects in this {@link Model}.
   * <p>
   * @return the number of Parameters in this {@link Model}.  Parameters defined in
   * KineticLaws are not included.
   */
 public";

%javamethodmodifiers Model::getNumInitialAssignments() const "
  /**
   * Get the number of {@link InitialAssignment} objects in this {@link Model}.
   * <p>
   * @return the number of InitialAssignments in this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getNumRules() const "
  /**
   * Get the number of {@link Rule} objects in this {@link Model}.
   * <p>
   * @return the number of Rules in this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getNumConstraints() const "
  /**
   * Get the number of {@link Constraint} objects in this {@link Model}.
   * <p>
   * @return the number of Constraints in this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getNumReactions() const "
  /**
   * Get the number of {@link Reaction} objects in this {@link Model}.
   * <p>
   * @return the number of Reactions in this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getNumEvents() const "
  /**
   * Get the number of {@link Event} objects in this {@link Model}.
   * <p>
   * @return the number of Events in this {@link Model}.
   */
 public";

%javamethodmodifiers Model::convertToL1(bool strict = false) "
  /**
   * Get the number of {@link Event} objects in this {@link Model}.
   * <p>
   * @return the number of Events in this {@link Model}.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::convertToL2 "
  /**
   * Get the number of {@link Event} objects in this {@link Model}.
   * <p>
   * @return the number of Events in this {@link Model}.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::isBoolean(const ASTNode* node) const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether the
   * given {@link ASTNode} is a boolean.
   * <p>
   * Often times, this question can be answered with the {@link ASTNode}\'s own
   * isBoolean() method, but if the AST is an expression that calls a
   * function defined in the {@link Model}\'s {@link ListOfFunctionDefinitions}, the model
   * is needed for lookup context.
   * <p>
   * @return true if the given {@link ASTNode} is a boolean.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::removeMetaId "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether the
   * given {@link ASTNode} is a boolean.
   * <p>
   * Often times, this question can be answered with the {@link ASTNode}\'s own
   * isBoolean() method, but if the AST is an expression that calls a
   * function defined in the {@link Model}\'s {@link ListOfFunctionDefinitions}, the model
   * is needed for lookup context.
   * <p>
   * @return true if the given {@link ASTNode} is a boolean.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::removeSBOTerms "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether the
   * given {@link ASTNode} is a boolean.
   * <p>
   * Often times, this question can be answered with the {@link ASTNode}\'s own
   * isBoolean() method, but if the AST is an expression that calls a
   * function defined in the {@link Model}\'s {@link ListOfFunctionDefinitions}, the model
   * is needed for lookup context.
   * <p>
   * @return true if the given {@link ASTNode} is a boolean.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::removeHasOnlySubstanceUnits "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether the
   * given {@link ASTNode} is a boolean.
   * <p>
   * Often times, this question can be answered with the {@link ASTNode}\'s own
   * isBoolean() method, but if the AST is an expression that calls a
   * function defined in the {@link Model}\'s {@link ListOfFunctionDefinitions}, the model
   * is needed for lookup context.
   * <p>
   * @return true if the given {@link ASTNode} is a boolean.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::removeSBOTermsNotInL2V2 "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether the
   * given {@link ASTNode} is a boolean.
   * <p>
   * Often times, this question can be answered with the {@link ASTNode}\'s own
   * isBoolean() method, but if the AST is an expression that calls a
   * function defined in the {@link Model}\'s {@link ListOfFunctionDefinitions}, the model
   * is needed for lookup context.
   * <p>
   * @return true if the given {@link ASTNode} is a boolean.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::convertToL2V1(bool strict = false) "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether the
   * given {@link ASTNode} is a boolean.
   * <p>
   * Often times, this question can be answered with the {@link ASTNode}\'s own
   * isBoolean() method, but if the AST is an expression that calls a
   * function defined in the {@link Model}\'s {@link ListOfFunctionDefinitions}, the model
   * is needed for lookup context.
   * <p>
   * @return true if the given {@link ASTNode} is a boolean.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::convertToL2V2(bool strict = false) "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether the
   * given {@link ASTNode} is a boolean.
   * <p>
   * Often times, this question can be answered with the {@link ASTNode}\'s own
   * isBoolean() method, but if the AST is an expression that calls a
   * function defined in the {@link Model}\'s {@link ListOfFunctionDefinitions}, the model
   * is needed for lookup context.
   * <p>
   * @return true if the given {@link ASTNode} is a boolean.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::removeDuplicateTopLevelAnnotations "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether the
   * given {@link ASTNode} is a boolean.
   * <p>
   * Often times, this question can be answered with the {@link ASTNode}\'s own
   * isBoolean() method, but if the AST is an expression that calls a
   * function defined in the {@link Model}\'s {@link ListOfFunctionDefinitions}, the model
   * is needed for lookup context.
   * <p>
   * @return true if the given {@link ASTNode} is a boolean.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::convertToL2Strict "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether the
   * given {@link ASTNode} is a boolean.
   * <p>
   * Often times, this question can be answered with the {@link ASTNode}\'s own
   * isBoolean() method, but if the AST is an expression that calls a
   * function defined in the {@link Model}\'s {@link ListOfFunctionDefinitions}, the model
   * is needed for lookup context.
   * <p>
   * @return true if the given {@link ASTNode} is a boolean.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::setSBMLDocument(SBMLDocument* d) "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether the
   * given {@link ASTNode} is a boolean.
   * <p>
   * Often times, this question can be answered with the {@link ASTNode}\'s own
   * isBoolean() method, but if the AST is an expression that calls a
   * function defined in the {@link Model}\'s {@link ListOfFunctionDefinitions}, the model
   * is needed for lookup context.
   * <p>
   * @return true if the given {@link ASTNode} is a boolean.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::setParentSBMLObject(SBase* sb) "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether the
   * given {@link ASTNode} is a boolean.
   * <p>
   * Often times, this question can be answered with the {@link ASTNode}\'s own
   * isBoolean() method, but if the AST is an expression that calls a
   * function defined in the {@link Model}\'s {@link ListOfFunctionDefinitions}, the model
   * is needed for lookup context.
   * <p>
   * @return true if the given {@link ASTNode} is a boolean.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::getTypeCode() const "
  /**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for this object, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers Model::getElementName() const "
  /**
   * Returns the XML element name of this object, which for {@link Model}, is
   * always <code>\'model\'</code>.
   * <p>
   * @return the name of this element, i.e., <code>\'model\'</code>.
   */
 public";

%javamethodmodifiers Model::getElementPosition() const "
  /**
   * @return the ordinal position of the element with respect to its
   * siblings or -1 (default) to indicate the position is not significant.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::writeElements(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write out their contained
   * SBML objects as XML elements.  Be sure to call your parents
   * implementation of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::getListOfLayouts() const "
  /**
   * Returns the {@link ListOf} Layouts for this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getListOfLayouts "
  /**
   * Returns the {@link ListOf} Layouts for this {@link Model}.
   */
 public";

%javamethodmodifiers Model::getLayout "
  /**
   * Returns the layout object that belongs to the given index. If the
   * index is invalid, NULL is returned.
   */
 public";

%javamethodmodifiers Model::getLayout(unsigned int index) "
  /**
   * Returns the layout object that belongs to the given index. If the
   * index is invalid, NULL is returned.
   */
 public";

%javamethodmodifiers Model::addLayout(const Layout* layout) "
  /**
   * Adds a copy of the layout object to the list of layouts.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   */
 public";

%javamethodmodifiers Model::createLayout "
  /**
   * Creates a new layout object and adds it to the list of layout objects
   * and returns it.
   */
 public";

%javamethodmodifiers Model::removeLayout(unsigned int n) "
  /**
   * Removes the nth Layout object from this {@link Model} object and
   * returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * <p>
   * @param n the index of the Layout object to remove
   * <p>
   * @return the Layout object removed.  As mentioned above, the caller owns the
   * returned object. NULL is returned if the given index is out of range.
   */
 public";

%javamethodmodifiers Model::populateListFormulaUnitsData "
  /**
   * Populates the list of FormulaDataUnits with the units derived 
   * for the model. The list contains elements of class
   * FormulaUnitsData. 
   * <p>
   * The first element of the list refers to the default units
   * of \'substance per time\' derived from the model and has the
   * unitReferenceId \'subs_per_time\'. This facilitates the comparison of units
   * derived from mathematical formula with the expected units.
   * <p>
   * The next elements of the list record the units of the 
   * compartments and species established from either explicitly
   * declared or default units.
   * <p>
   * The next elements record the units of any parameters.
   * <p>
   * Subsequent elements of the list record the units derived for
   * each mathematical expression encountered within the model.
   * <p>
   * @note This function is utilised by the {@link Unit} Consistency Validator.
   * The list is populated prior to running the validation and thus
   * the consistency of units can be checked by accessing the members
   * of the list and comparing the appropriate data.
   */
 public";

%javamethodmodifiers Model::isPopulatedListFormulaUnitsData "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether 
   * the list of FormulaUnitsData has been populated.
   * <p>
   * @return <code>true</code> if the list of FormulaUnitsData has been populated, 
   * <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers Model::addFormulaUnitsData(const FormulaUnitsData* fud) "
  /**
   * Adds a copy of the given FormulaUnitsData object to this {@link Model}.
   * <p>
   * @param fud the FormulaUnitsData to add
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::createFormulaUnitsData "
  /**
   * Creates a new FormulaUnitsData inside this {@link Model} and returns it.
   * <p>
   * @return the FormulaUnitsData object created
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::getFormulaUnitsData "
  /**
   * Get the nth FormulaUnitsData object in this {@link Model}.
   * <p>
   * @return the nth FormulaUnitsData of this {@link Model}.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::getFormulaUnitsData(unsigned int n) "
  /**
   * Get the nth FormulaUnitsData object in this {@link Model}.
   * <p>
   * @return the nth FormulaUnitsData of this {@link Model}.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::getFormulaUnitsData(const std::string& sid, SBMLTypeCode_t typecode) const "
  /**
   * Get a FormulaUnitsData object based on its unitReferenceId and typecode.
   * <p>
   * @return the FormulaUnitsData in this {@link Model} with the unitReferenceId <code>sid</code> 
   * and the SBMLTypeCode_t <code>typecode</code> or NULL
   * if no such FormulaUnitsData exists.
   * <p>
   * @note The SBMLTypecode_t parameter is necessary as the unitReferenceId
   * of the FormulaUnitsData need not be unique. For example if a {@link Species}
   * with id \'s\' is assigned by an {@link AssignmentRule} there will be two 
   * elements of the FormulaUnitsData list with the unitReferenceId \'s\'; 
   * one with
   * typecode \'SBML_SPECIES\' referring to the units related to the species, 
   * the other with typecode \'SBML_ASSIGNMENT_RULE\' referring to the units
   * derived from the math element of the {@link AssignmentRule}.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::getFormulaUnitsData(const std::string& sid, SBMLTypeCode_t) "
  /**
   * Get a FormulaUnitsData object based on its unitReferenceId and typecode.
   * <p>
   * @return the FormulaUnitsData in this {@link Model} with the unitReferenceId <code>sid</code> 
   * and the SBMLTypeCode_t <code>typecode</code> or NULL
   * if no such FormulaUnitsData exists.
   * <p>
   * @note The SBMLTypecode_t parameter is necessary as the unitReferenceId
   * of the FormulaUnitsData need not be unique. For example if a {@link Species}
   * with id \'s\' is assigned by an {@link AssignmentRule} there will be two 
   * elements of the FormulaUnitsData list with the unitReferenceId \'s\'; 
   * one with
   * typecode \'SBML_SPECIES\' referring to the units related to the species, 
   * the other with typecode \'SBML_ASSIGNMENT_RULE\' referring to the units
   * derived from the math element of the {@link AssignmentRule}.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::getNumFormulaUnitsData() const "
  /**
   * Get the number of FormulaUnitsData objects in this {@link Model}.
   * <p>
   * @return the number of FormulaUnitsData in this {@link Model}.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::getListFormulaUnitsData "
  /**
   * Get the list of FormulaUnitsData object in this {@link Model}.
   * <p>
   * @return the list of FormulaUnitsData for this {@link Model}.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::getListFormulaUnitsData() const "
  /**
   * Get the list of FormulaUnitsData object in this {@link Model}.
   * <p>
   * @return the list of FormulaUnitsData for this {@link Model}.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::hasRequiredElements() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required elements for this {@link Model} object
   * have been set.
   * <p>
   * @note The required elements for a {@link Model} object are:
   * listOfCompartments (L1 only); listOfSpecies (L1V1 only);
   * listOfReactions(L1V1 only)
   * <p>
   * @return a boolean value indicating whether all the required
   * elements for this object have been defined.
   */
 public";

%javamethodmodifiers Model::removeFunctionDefinition(unsigned int n) "
  /**
   * Removes the nth {@link FunctionDefinition} object from this {@link Model} object and 
   * returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * <p>
   * @param n the index of the {@link FunctionDefinition} object to remove
   * <p>
   * @return the {@link FunctionDefinition} object removed.  As mentioned above, 
   * the caller owns the returned item. NULL is returned if the given index 
   * is out of range.
   * <p>
   */
 public";

%javamethodmodifiers Model::removeFunctionDefinition(const std::string& sid) "
  /**
   * Removes the {@link FunctionDefinition} object with the given identifier from this {@link Model} 
   * object and returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * If none of the {@link FunctionDefinition} objects in this {@link Model} object have the identifier 
   * <code>sid</code>, then <code>NULL</code> is returned.
   * <p>
   * @param sid the identifier of the {@link FunctionDefinition} object to remove
   * <p>
   * @return the {@link FunctionDefinition} object removed.  As mentioned above, the 
   * caller owns the returned object. NULL is returned if no {@link FunctionDefinition}
   * object with the identifier exists in this {@link Model} object.
   */
 public";

%javamethodmodifiers Model::removeUnitDefinition(unsigned int n) "
  /**
   * Removes the nth {@link UnitDefinition} object from this {@link Model} object and
   * returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * <p>
   * @param n the index of the {@link UnitDefinition} object to remove
   * <p>
   * @return the {@link UnitDefinition} object removed.  As mentioned above, 
   * the caller owns the returned item. NULL is returned if the given index 
   * is out of range.
   * <p>
   */
 public";

%javamethodmodifiers Model::removeUnitDefinition(const std::string& sid) "
  /**
   * Removes the {@link UnitDefinition} object with the given identifier from this {@link Model}
   * object and returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * If none of the {@link UnitDefinition} objects in this {@link Model} object have the identifier 
   * <code>sid</code>, then <code>NULL</code> is returned.
   * <p>
   * @param sid the identifier of the {@link UnitDefinition} object to remove
   * <p>
   * @return the {@link UnitDefinition} object removed.  As mentioned above, the 
   * caller owns the returned object. NULL is returned if no {@link UnitDefinition}
   * object with the identifier exists in this {@link Model} object.
   */
 public";

%javamethodmodifiers Model::removeCompartmentType(unsigned int n) "
  /**
   * Removes the nth {@link CompartmentType} object from this {@link Model} object and
   * returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * <p>
   * @param n the index of the {@link CompartmentType} object to remove
   * <p>
   * @return the ComapartmentType object removed.  As mentioned above, 
   * the caller owns the returned item. NULL is returned if the given index 
   * is out of range.
   * <p>
   */
 public";

%javamethodmodifiers Model::removeCompartmentType(const std::string& sid) "
  /**
   * Removes the {@link CompartmentType} object with the given identifier from this {@link Model}
   * object and returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * If none of the {@link CompartmentType} objects in this {@link Model} object have the identifier 
   * <code>sid</code>, then <code>NULL</code> is returned.
   * <p>
   * @param sid the identifier of the object to remove
   * <p>
   * @return the {@link CompartmentType} object removed.  As mentioned above, the 
   * caller owns the returned object. NULL is returned if no {@link CompartmentType}
   * object with the identifier exists in this {@link Model} object.
   */
 public";

%javamethodmodifiers Model::removeSpeciesType(unsigned int n) "
  /**
   * Removes the nth {@link SpeciesType} object from this {@link Model} object and
   * returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * <p>
   * @param n the index of the {@link SpeciesType} object to remove
   * <p>
   * @return the {@link SpeciesType} object removed.  As mentioned above, 
   * the caller owns the returned item. NULL is returned if the given index 
   * is out of range.
   * <p>
   */
 public";

%javamethodmodifiers Model::removeSpeciesType(const std::string& sid) "
  /**
   * Removes the {@link SpeciesType} object with the given identifier from this {@link Model}
   * object and returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * If none of the {@link SpeciesType} objects in this {@link Model} object have the identifier 
   * <code>sid</code>, then <code>NULL</code> is returned.
   * <p>
   * @param sid the identifier of the {@link SpeciesType} object to remove
   * <p>
   * @return the {@link SpeciesType} object removed.  As mentioned above, the 
   * caller owns the returned object. NULL is returned if no {@link SpeciesType}
   * object with the identifier exists in this {@link Model} object.
   * <p>
   */
 public";

%javamethodmodifiers Model::removeCompartment(unsigned int n) "
  /**
   * Removes the nth {@link Compartment} object from this {@link Model} object and
   * returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * <p>
   * @param n the index of the {@link Compartment} object to remove
   * <p>
   * @return the {@link Compartment} object removed.  As mentioned above, 
   * the caller owns the returned item. NULL is returned if the given index 
   * is out of range.
   * <p>
   */
 public";

%javamethodmodifiers Model::removeCompartment(const std::string& sid) "
  /**
   * Removes the {@link Compartment} object with the given identifier from this {@link Model}
   * object and returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * If none of the {@link Compartment} objects in this {@link Model} object have the identifier 
   * <code>sid</code>, then <code>NULL</code> is returned.
   * <p>
   * @param sid the identifier of the {@link Compartment} object to remove
   * <p>
   * @return the {@link Compartment} object removed.  As mentioned above, the 
   * caller owns the returned object. NULL is returned if no {@link Compartment}
   * object with the identifier exists in this {@link Model} object.
   */
 public";

%javamethodmodifiers Model::removeSpecies(unsigned int n) "
  /**
   * Removes the nth {@link Species} object from this {@link Model} object and
   * returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * <p>
   * @param n the index of the {@link Species} object to remove
   * <p>
   * @return the {@link Species} object removed.  As mentioned above, 
   * the caller owns the returned item. NULL is returned if the given index 
   * is out of range.
   * <p>
   */
 public";

%javamethodmodifiers Model::removeSpecies(const std::string& sid) "
  /**
   * Removes the {@link Species} object with the given identifier from this {@link Model}
   * object and returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * If none of the {@link Species} objects in this {@link Model} object have the identifier 
   * <code>sid</code>, then <code>NULL</code> is returned.
   * <p>
   * @param sid the identifier of the {@link Species} object to remove
   * <p>
   * @return the {@link Species} object removed.  As mentioned above, the 
   * caller owns the returned object. NULL is returned if no {@link Species}
   * object with the identifier exists in this {@link Model} object.
   * <p>
   */
 public";

%javamethodmodifiers Model::removeParameter(unsigned int n) "
  /**
   * Removes the nth {@link Parameter} object from this {@link Model} object and
   * returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * <p>
   * @param n the index of the {@link Parameter} object to remove
   * <p>
   * @return the {@link Parameter} object removed.  As mentioned above, 
   * the caller owns the returned item. NULL is returned if the given index 
   * is out of range.
   * <p>
   */
 public";

%javamethodmodifiers Model::removeParameter(const std::string& sid) "
  /**
   * Removes the {@link Parameter} object with the given identifier from this {@link Model}
   * object and returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * If none of the {@link Parameter} objects in this {@link Model} object have the identifier 
   * <code>sid</code>, then <code>NULL</code> is returned.
   * <p>
   * @param sid the identifier of the {@link Parameter} object to remove
   * <p>
   * @return the {@link Parameter} object removed.  As mentioned above, the 
   * caller owns the returned object. NULL is returned if no {@link Parameter}
   * object with the identifier exists in this {@link Model} object.
   */
 public";

%javamethodmodifiers Model::removeInitialAssignment(unsigned int n) "
  /**
   * Removes the nth {@link InitialAssignment} object from this {@link Model} object and
   * returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * <p>
   * @param n the index of the {@link InitialAssignment} object to remove
   * <p>
   * @return the {@link InitialAssignment} object removed.  As mentioned above, 
   * the caller owns the returned item. NULL is returned if the given index 
   * is out of range.
   * <p>
   */
 public";

%javamethodmodifiers Model::removeInitialAssignment(const std::string& symbol) "
  /**
   * Removes the {@link InitialAssignment} object with the given \'symbol\' attribute 
   * from this {@link Model} object and returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * If none of the {@link InitialAssignment} objects in this {@link Model} object have the
   * \'symbol\' attribute <code>symbol</code>, then <code>NULL</code> is returned.
   * <p>
   * @param symbol the \'symbol\' attribute of the {@link InitialAssignment} object to remove
   * <p>
   * @return the {@link InitialAssignment} object removed.  As mentioned above, the 
   * caller owns the returned object. NULL is returned if no {@link InitialAssignment}
   * object with the \'symbol\' attribute exists in this {@link Model} object.
   */
 public";

%javamethodmodifiers Model::removeRule(unsigned int n) "
  /**
   * Removes the nth {@link Rule} object from this {@link Model} object and
   * returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * <p>
   * @param n the index of the {@link Rule} object to remove
   * <p>
   * @return the {@link Rule} object removed.  As mentioned above, 
   * the caller owns the returned item. NULL is returned if the given index 
   * is out of range.
   * <p>
   */
 public";

%javamethodmodifiers Model::removeRule(const std::string& variable) "
  /**
   * Removes the {@link Rule} object with the given \'variable\' attribute from this {@link Model} 
   * object and returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * If none of the {@link Rule} objects in this {@link Model} object have the \'variable\' attribute
   * <code>variable</code>, then <code>NULL</code> is returned.
   * <p>
   * @param variable the \'variable\' attribute of the {@link Rule} object to remove
   * <p>
   * @return the {@link Rule} object removed.  As mentioned above, the 
   * caller owns the returned object. NULL is returned if no {@link Rule}
   * object with the \'variable\' attribute exists in this {@link Model} object.
   */
 public";

%javamethodmodifiers Model::removeConstraint(unsigned int n) "
  /**
   * Removes the nth {@link Constraint} object from this {@link Model} object and
   * returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * <p>
   * @param n the index of the {@link Constraint} object to remove
   * <p>
   * @return the {@link Constraint} object removed.  As mentioned above, 
   * the caller owns the returned item. NULL is returned if the given index 
   * is out of range.
   * <p>
   */
 public";

%javamethodmodifiers Model::removeReaction(unsigned int n) "
  /**
   * Removes the nth {@link Reaction} object from this {@link Model} object and
   * returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * <p>
   * @param n the index of the {@link Reaction} object to remove
   * <p>
   * @return the {@link Reaction} object removed.  As mentioned above, 
   * the caller owns the returned item. NULL is returned if the given index 
   * is out of range.
   * <p>
   */
 public";

%javamethodmodifiers Model::removeReaction(const std::string& sid) "
  /**
   * Removes the {@link Reaction} object with the given identifier from this {@link Model}
   * object and returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * If none of the {@link Reaction} objects in this {@link Model} object have the identifier 
   * <code>sid</code>, then <code>NULL</code> is returned.
   * <p>
   * @param sid the identifier of the {@link Reaction} object to remove
   * <p>
   * @return the {@link Reaction} object removed.  As mentioned above, the 
   * caller owns the returned object. NULL is returned if no {@link Reaction}
   * object with the identifier exists in this {@link Model} object.
   * <p>
   */
 public";

%javamethodmodifiers Model::removeEvent(unsigned int n) "
  /**
   * Removes the nth {@link Event} object from this {@link Model} object and
   * returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * <p>
   * @param n the index of the {@link Event} object to remove
   * <p>
   * @return the {@link Event} object removed.  As mentioned above, 
   * the caller owns the returned item. NULL is returned if the given index 
   * is out of range.
   * <p>
   */
 public";

%javamethodmodifiers Model::removeEvent(const std::string& sid) "
  /**
   * Removes the {@link Event} object with the given identifier from this {@link Model}
   * object and returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * If none of the {@link Event} objects in this {@link Model} object have the identifier 
   * <code>sid</code>, then <code>NULL</code> is returned.
   * <p>
   * @param sid the identifier of the {@link Event} object to remove
   * <p>
   * @return the {@link Event} object removed.  As mentioned above, the 
   * caller owns the returned object. NULL is returned if no {@link Event}
   * object with the identifier exists in this {@link Model} object.
   * <p>
   */
 public";

%javamethodmodifiers Model::Model "
  /**
   * Removes the {@link Event} object with the given identifier from this {@link Model}
   * object and returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * If none of the {@link Event} objects in this {@link Model} object have the identifier 
   * <code>sid</code>, then <code>NULL</code> is returned.
   * <p>
   * @param sid the identifier of the {@link Event} object to remove
   * <p>
   * @return the {@link Event} object removed.  As mentioned above, the 
   * caller owns the returned object. NULL is returned if no {@link Event}
   * object with the identifier exists in this {@link Model} object.
   * <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::readOtherXML(XMLInputStream& stream) "
  /**
   * Subclasses should override this method to read (and store) XHTML,
   * MathML, etc. directly from the XMLInputStream.
   * <p>
   * @return true if the subclass read from the stream, false otherwise.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::createObject(XMLInputStream& stream) "
  /**
   * @return the SBML object corresponding to next {@link XMLToken} in the
   * XMLInputStream or NULL if the token was not recognized.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::readAttributes(const XMLAttributes& attributes) "
  /**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::syncAnnotation "
  /**
   * Synchronizes the annotation of this SBML object.
   * <p>
   * Annotation element (XMLNode  mAnnotation) is synchronized with the
   * current {@link CVTerm} objects (List  mCVTerm), {@link ModelHistory} object 
   * (ModelHistory  mHistory) and ListOfLayouts object (ListOfLayouts mLayouts).
   * Currently, this method is called in getAnnotation, isSetAnnotation,
   * and writeElements methods.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::syncAnnotation "
  /**
   * Synchronizes the annotation of this SBML object.
   * <p>
   * Annotation element (XMLNode  mAnnotation) is synchronized with the
   * current {@link CVTerm} objects (List  mCVTerm), {@link ModelHistory} object 
   * (ModelHistory  mHistory) and ListOfLayouts object (ListOfLayouts mLayouts).
   * Currently, this method is called in getAnnotation, isSetAnnotation,
   * and writeElements methods.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::syncAnnotation "
  /**
   * Synchronizes the annotation of this SBML object.
   * <p>
   * Annotation element (XMLNode  mAnnotation) is synchronized with the
   * current {@link CVTerm} objects (List  mCVTerm), {@link ModelHistory} object 
   * (ModelHistory  mHistory) and ListOfLayouts object (ListOfLayouts mLayouts).
   * Currently, this method is called in getAnnotation, isSetAnnotation,
   * and writeElements methods.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::syncAnnotation "
  /**
   * Synchronizes the annotation of this SBML object.
   * <p>
   * Annotation element (XMLNode  mAnnotation) is synchronized with the
   * current {@link CVTerm} objects (List  mCVTerm), {@link ModelHistory} object 
   * (ModelHistory  mHistory) and ListOfLayouts object (ListOfLayouts mLayouts).
   * Currently, this method is called in getAnnotation, isSetAnnotation,
   * and writeElements methods.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::syncAnnotation "
  /**
   * Synchronizes the annotation of this SBML object.
   * <p>
   * Annotation element (XMLNode  mAnnotation) is synchronized with the
   * current {@link CVTerm} objects (List  mCVTerm), {@link ModelHistory} object 
   * (ModelHistory  mHistory) and ListOfLayouts object (ListOfLayouts mLayouts).
   * Currently, this method is called in getAnnotation, isSetAnnotation,
   * and writeElements methods.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::syncAnnotation "
  /**
   * Synchronizes the annotation of this SBML object.
   * <p>
   * Annotation element (XMLNode  mAnnotation) is synchronized with the
   * current {@link CVTerm} objects (List  mCVTerm), {@link ModelHistory} object 
   * (ModelHistory  mHistory) and ListOfLayouts object (ListOfLayouts mLayouts).
   * Currently, this method is called in getAnnotation, isSetAnnotation,
   * and writeElements methods.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::syncAnnotation "
  /**
   * Synchronizes the annotation of this SBML object.
   * <p>
   * Annotation element (XMLNode  mAnnotation) is synchronized with the
   * current {@link CVTerm} objects (List  mCVTerm), {@link ModelHistory} object 
   * (ModelHistory  mHistory) and ListOfLayouts object (ListOfLayouts mLayouts).
   * Currently, this method is called in getAnnotation, isSetAnnotation,
   * and writeElements methods.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::syncAnnotation "
  /**
   * Synchronizes the annotation of this SBML object.
   * <p>
   * Annotation element (XMLNode  mAnnotation) is synchronized with the
   * current {@link CVTerm} objects (List  mCVTerm), {@link ModelHistory} object 
   * (ModelHistory  mHistory) and ListOfLayouts object (ListOfLayouts mLayouts).
   * Currently, this method is called in getAnnotation, isSetAnnotation,
   * and writeElements methods.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::syncAnnotation "
  /**
   * Synchronizes the annotation of this SBML object.
   * <p>
   * Annotation element (XMLNode  mAnnotation) is synchronized with the
   * current {@link CVTerm} objects (List  mCVTerm), {@link ModelHistory} object 
   * (ModelHistory  mHistory) and ListOfLayouts object (ListOfLayouts mLayouts).
   * Currently, this method is called in getAnnotation, isSetAnnotation,
   * and writeElements methods.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::syncAnnotation "
  /**
   * Synchronizes the annotation of this SBML object.
   * <p>
   * Annotation element (XMLNode  mAnnotation) is synchronized with the
   * current {@link CVTerm} objects (List  mCVTerm), {@link ModelHistory} object 
   * (ModelHistory  mHistory) and ListOfLayouts object (ListOfLayouts mLayouts).
   * Currently, this method is called in getAnnotation, isSetAnnotation,
   * and writeElements methods.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::syncAnnotation "
  /**
   * Synchronizes the annotation of this SBML object.
   * <p>
   * Annotation element (XMLNode  mAnnotation) is synchronized with the
   * current {@link CVTerm} objects (List  mCVTerm), {@link ModelHistory} object 
   * (ModelHistory  mHistory) and ListOfLayouts object (ListOfLayouts mLayouts).
   * Currently, this method is called in getAnnotation, isSetAnnotation,
   * and writeElements methods.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::syncAnnotation "
  /**
   * Synchronizes the annotation of this SBML object.
   * <p>
   * Annotation element (XMLNode  mAnnotation) is synchronized with the
   * current {@link CVTerm} objects (List  mCVTerm), {@link ModelHistory} object 
   * (ModelHistory  mHistory) and ListOfLayouts object (ListOfLayouts mLayouts).
   * Currently, this method is called in getAnnotation, isSetAnnotation,
   * and writeElements methods.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::syncAnnotation "
  /**
   * Synchronizes the annotation of this SBML object.
   * <p>
   * Annotation element (XMLNode  mAnnotation) is synchronized with the
   * current {@link CVTerm} objects (List  mCVTerm), {@link ModelHistory} object 
   * (ModelHistory  mHistory) and ListOfLayouts object (ListOfLayouts mLayouts).
   * Currently, this method is called in getAnnotation, isSetAnnotation,
   * and writeElements methods.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::syncAnnotation "
  /**
   * Synchronizes the annotation of this SBML object.
   * <p>
   * Annotation element (XMLNode  mAnnotation) is synchronized with the
   * current {@link CVTerm} objects (List  mCVTerm), {@link ModelHistory} object 
   * (ModelHistory  mHistory) and ListOfLayouts object (ListOfLayouts mLayouts).
   * Currently, this method is called in getAnnotation, isSetAnnotation,
   * and writeElements methods.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::syncAnnotation "
  /**
   * Synchronizes the annotation of this SBML object.
   * <p>
   * Annotation element (XMLNode  mAnnotation) is synchronized with the
   * current {@link CVTerm} objects (List  mCVTerm), {@link ModelHistory} object 
   * (ModelHistory  mHistory) and ListOfLayouts object (ListOfLayouts mLayouts).
   * Currently, this method is called in getAnnotation, isSetAnnotation,
   * and writeElements methods.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::syncAnnotation "
  /**
   * Synchronizes the annotation of this SBML object.
   * <p>
   * Annotation element (XMLNode  mAnnotation) is synchronized with the
   * current {@link CVTerm} objects (List  mCVTerm), {@link ModelHistory} object 
   * (ModelHistory  mHistory) and ListOfLayouts object (ListOfLayouts mLayouts).
   * Currently, this method is called in getAnnotation, isSetAnnotation,
   * and writeElements methods.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Model::syncAnnotation "
  /**
   * Synchronizes the annotation of this SBML object.
   * <p>
   * Annotation element (XMLNode  mAnnotation) is synchronized with the
   * current {@link CVTerm} objects (List  mCVTerm), {@link ModelHistory} object 
   * (ModelHistory  mHistory) and ListOfLayouts object (ListOfLayouts mLayouts).
   * Currently, this method is called in getAnnotation, isSetAnnotation,
   * and writeElements methods.
   <p>
 * @deprecated libSBML internal
 */
 public";

%typemap(javaimports) SBMLDocument "
/** 
 * Container for an SBML document and interface for global operations
 * on SBML documents.
 * <p>
 * LibSBML uses the class
 * {@link SBMLDocument} as a top-level container for storing SBML content and data
 * associated with it (such as warnings and error messages).  The two
 * primary means of reading an SBML model, {@link SBMLReader#readSBML(String
 * filename)} and {@link SBMLReader#readSBMLFromString(String xml)}, both return a
 * pointer to an {@link SBMLDocument} object.  From there, callers can inquire
 * about any errors encountered (e.g., using SBMLDocument.getNumErrors()),
 * access the {@link Model} object, and perform other actions such as
 * consistency-checking and model translation. 
 * <p>
 * When creating fresh models programmatically, the starting point is
 * typically the creation of an {@link SBMLDocument} object instance.  The
 * {@link SBMLDocument} constructor accepts arguments for the SBML Level and
 * Version of the model to be created.  After creating the {@link SBMLDocument}
 * object, calling programs then typically call SBMLDocument.createModel()
 * almost immediately, and then proceed to call the methods on the {@link Model}
 * object to fill out the model\'s contents.
 * <p>
 * {@link SBMLDocument} corresponds roughly to the class <i>Sbml</i> defined in the
 * SBML Level&nbsp;2 specification.  It does not have a direct
 * correspondence in SBML Level&nbsp;1.  (However, to make matters simpler
 * for applications, libSBML creates an {@link SBMLDocument} no matter whether the model
 * is Level&nbsp;1 or Level&nbsp;2.)  In its barest form, when written out
 * in XML format for SBML Level&nbsp;2 Version&nbsp;4, the corresponding
 * structure is the following:
 * <div class=\'fragment\'><pre>
&lt;sbml xmlns=\'http://www.sbml.org/sbml/level2/version4\' level=\'2\' version=\'4\'
  ...
&lt;/sbml&gt;
</pre></div>
 * <p>
 * {@link SBMLDocument} is derived from {@link SBase}, and therefore contains the usual {@link SBase}
 * attributes (in SBML Level&nbsp;2 Version&nbsp;4) of \'metaid\' and \'sboTerm\', as
 * well as the subelements \'notes\' and \'annotation\'.  It also contains the
 * attributes \'level\' and \'version\' indicating the Level and Version of the
 * SBML data structure.  These can be accessed using the methods defined by
 * the {@link SBase} class for that purpose.
 * <p>
 * <h2>Checking consistency and adherence to SBML specifications</h2>
 * <p>
 * One of the most important features of libSBML is its ability to perform
 * SBML validation to ensure that a model adheres to the SBML specification
 * for whatever Level+Version combination the model uses.  {@link SBMLDocument}
 * provides the methods for running consistency-checking and validation
 * rules on the SBML content.
 * <p>
 * First, a brief explanation of the rationale is in order.  In libSBML
 * versions up to and including the version 3.3.x series, the individual
 * methods for creating and setting attributes and other components are
 * quite lenient, and allow a caller to compose SBML entities that may not,
 * in the end, represent valid SBML.  This allows applications the freedom
 * to do things such as save incomplete models (which is useful when models
 * are being developed over long periods of time), but at the same time, it
 * means that a separate validation step is necessary when a calling
 * program finally wants to finish a complete SBML document.
 * <p>
 * The primary interface to this facility is {@link SBMLDocument}\'s
 * SBMLDocument.checkInternalConsistency() and
 * SBMLDocument.checkConsistency().  The former verifies the basic
 * internal consistency and syntax of an SBML document, and the latter
 * implements more elaborate validation rules (both those defined by the
 * SBML specifications, as well as additional rules offered by libSBML).
 * <p>
 * The checks
 * performed by SBMLDocument.checkInternalConsistency() are hardwired and
 * cannot be changed by calling programs, but the validation performed by
 * SBMLDocument.checkConsistency() is under program control using the
 * method {@link SBMLDocument#setConsistencyChecks(int categ, boolean onoff)}.
 * Applications can selectively disable specific kinds of checks that they
 * may not be interested by calling {@link SBMLDocument#setConsistencyChecks(int
 * categ, boolean onoff)} with appropriate parameters. 
 * <p>
 * These methods have slightly different relevance depending on whether a
 * model is created programmaticaly from scratch, or whether it is read in
 * from a file or data stream.  The following table summarizes the possible
 * scenarios.
 * <p>
 * <center>
 * <table border=\'0\' class=\'text-table width80 normal-font alt-row-colors\'>
 *  <tr style=\'background: lightgray; font-size: 14px;\'>
 *      <th align=\'left\' width=\'200\'>Scenario</th>
 *      <th align=\'left\'>Relevant methods</th>
 *  </tr>
 * <tr><td>Creating a model from scratch</td>
 * <td>Before writing out the model:<ol>
 * <p>
 * <li style=\'margin-bottom: 0.5em\'>Call
 * SBMLDocument.checkInternalConsistency(), then inquire about the results by
 * calling SBMLDocument.getNumErrors()</li>
 * <p>
 * <li style=\'margin-bottom: 0.5em\'>
 * Call {@link SBMLDocument#setConsistencyChecks(int categ, boolean onoff)}  to configure which checks
 * will be performed by SBMLDocument.checkConsistency()</li>
 * <p>
 * <li>Call SBMLDocument.checkConsistency(), then inquire about the results by
 * calling SBMLDocument.getNumErrors()</li>
 * </ol>
 * </td>
 * <tr><td>Reading a model from a file or data stream</td>
 * <td>After reading the model:<ol>
 * <p>
 * <li style=\'margin-bottom: 0.5em\'>Basic consistency checks will have been
 * performed automatically by libSBML upon reading the content&mdash;only need
 * to inquire about the results by using SBMLDocument.getNumErrors()</li>
 * <p>
 * <li style=\'margin-bottom: 0.5em\'>
 * Call {@link SBMLDocument#setConsistencyChecks(int categ, boolean onoff)}  to configure which
 * checks are performed by SBMLDocument.checkConsistency()</li>
 * <p>
 * <li>Call SBMLDocument.checkConsistency(), then inquire about the results
 * by calling SBMLDocument.getNumErrors()</li>
 * </ol>
 * </td>
 * </table>
 * </center>
 * <p>
 * <p>
 * <p>
 * <h2>Converting documents between Levels and Versions of SBML</h2>
 * <p>
 * LibSBML provides facilities for limited translation of SBML between
 * Levels and Versions of the SBML specifications.  The method for doing is
 * is {@link SBMLDocument#setLevelAndVersion(long lev, long ver)} .  In 
 * general, models can be converted upward without difficulty (e.g., from
 * SBML Level&nbsp;1 to Level&nbsp;2, or from an earlier Version of
 * Level&nbsp;2 to the latest Version of Level&nbsp;2).  Sometimes models
 * can be translated downward as well, if they do not use constructs
 * specific to more advanced Levels of SBML.
 * <p>
 * Calling {@link SBMLDocument#setLevelAndVersion(long lev, long ver)}  will not <em>necessarily</em> lead
 * to a successful conversion.  The method will return a boolean value
 * to indicate success or failure.  Callers must check the error log (see 
 * next section) attached to the {@link SBMLDocument} object after calling
 * {@link SBMLDocument#setLevelAndVersion(long lev, long ver)}  in order to assess whether any
 * problems arose.
 * <p>
 * If an application is interested in translating to a lower Level and/or
 * Version of SBML within a Level, the following methods allow for prior
 * assessment of whether there is sufficient compatibility to make a
 * translation possible:
 * <p>
 * <li> SBMLDocument.checkL1Compatibility(),
 * <li> SBMLDocument.checkL2v1Compatibility(),
 * <li> SBMLDocument.checkL2v2Compatibility(),
 * <li> SBMLDocument.checkL2v3Compatibility(), and
 * <li> SBMLDocument.checkL2v4Compatibility().
 * <p>
 * Some changes between Versions of SBML Level&nbsp;2 may lead to
 * unexpected behaviors when attempting conversions in either direction.
 * For example, SBML Level&nbsp;2 Version&nbsp;4 relaxed the requirement
 * for consistency in units of measurement between expressions annd
 * quantities in a model.  As a result, a model written in Version&nbsp;4,
 * if converted to Version&nbsp;3 with no other changes, may fail
 * validation as a Version&nbsp;3 model because Version&nbsp;3 imposed
 * stricter requirements on unit consistency.
 * <p>
 * <h2>Error handling</h2>
 * <p>
 * Upon reading a model, {@link SBMLDocument} logs any problems encountered while
 * reading the model from the file or data stream.  The log contains
 * objects that record diagnostic information about any notable issues that
 * arose.  Whether the problems are warnings or errors, they are both
 * reported through a single common interface involving the object class
 * {@link SBMLError}.
 * <p>
 * The methods SBMLDocument.getNumErrors(), {@link SBMLDocument#getError(long n)}  and
 * SBMLDocument.printErrors() allow callers to interact with the warnings
 * or errors logged.  Alternatively, callers may retrieve the entire log as
 * an {@link SBMLErrorLog} object using the method SBMLDocument.getErrorLog().
 * The {@link SBMLErrorLog} object provides some alternative methods for
 * interacting with the set of errors and warnings.  In either case,
 * applications typically should first call SBMLDocument.getNumErrors() to
 * find out if any issues have been logged after specific libSBML
 * operations such as the ones discussed in the sections above.  If they
 * have, then an application will should proceed to inspect the individual
 * reports using either the direct interfaces on {@link SBMLDocument} or using the
 * methods on the {@link SBMLErrorLog} object.
 * <p>
 * <p>
 * <p>
 * <p>
 * <hr style=\'margin: 1em auto\'>
 */
"

%javamethodmodifiers SBMLDocument::getDefaultLevel "
  /**
   * The default SBML Level of new {@link SBMLDocument} objects.
   * <p>
   * This \'default level\' corresponds to the most recent SBML specification
   * Level available at the time this version of libSBML was released.  For
   * this copy of libSBML, the value is <code>2</code>.  The default Level
   * is used by {@link SBMLDocument} if no Level is explicitly specified at the
   * time of the construction of an {@link SBMLDocument} instance. 
   * <p>
   * @return an integer indicating the most recent SBML specification Level
   * <p>
   * @see #getDefaultVersion()
   */
 public";

%javamethodmodifiers SBMLDocument::getDefaultVersion "
  /**
   * The default Version of new {@link SBMLDocument} objects.
   * <p>
   * This \'default version\' corresponds to the most recent SBML Version
   * within the most recent Level of SBML available at the time this
   * version of libSBML was released.  For this copy of libSBML, the value
   * is <code>4</code>.  The default Version is used by {@link SBMLDocument} if no
   * Version is explicitly specified at the time of the construction of an
   * {@link SBMLDocument} instance.
   * <p>
   * @return an integer indicating the most recent SBML specification
   * Version
   * <p>
   * @see #getDefaultLevel()
   */
 public";

%javamethodmodifiers SBMLDocument::SBMLDocument(unsigned int level = 0, unsigned int version = 0) "
  /**
   * Creates a new {@link SBMLDocument}, optionally with given values for the SBML
   * Level and Version.
   * <p>
   * If <em>both</em> the SBML Level and Version attributes are not
   * specified, the SBML document is treated as having the latest Level and
   * Version of SBML as determined by SBMLDocument.getDefaultLevel() and
   * SBMLDocument.getDefaultVersion(); <em>however</em>, the {@link SBMLDocument}
   * object is otherwise left blank.  In particular, the blank {@link SBMLDocument}
   * object has no associated XML attributes, including (but not limited
   * to) an XML namespace declaration.  The XML namespace declaration is
   * not added until the model is written out, <em>or</em> the method
   * {@link SBMLDocument#setLevelAndVersion(long lev, long ver)}  is called.  This may be important to keep in mind
   * if an application needs to add additional XML namespace declarations
   * on the <code>&lt;sbml&gt;</code> element.  Application writers should
   * either provide values for <code>level</code> and <code>version</code> on the call to this
   * constructor, or else call {@link SBMLDocument#setLevelAndVersion(long lev, long ver)}  shortly after creating
   * the {@link SBMLDocument} object.
   * <p>
   * @param level an integer for the SBML Level
   * <p>
   * @param version an integer for the Version within the SBML Level
   * <p>
   * @see SBMLDocument#setLevelAndVersion(long lev, long ver) 
   * @see #getDefaultLevel()
   * @see #getDefaultVersion()
   * <p>
   * @docnote The native C++ implementation of this method defines a
   * default argument value.  In the documentation generated for different
   * libSBML language bindings, you may or may not see corresponding
   * arguments in the method declarations.  For example, in Java, a default
   * argument is handled by declaring two separate methods, with one of
   * them having the argument and the other one lacking the argument.
   * However, the libSBML documentation will be <em>identical</em> for both
   * methods.  Consequently, if you are reading this and do not see an
   * argument even though one is described, please look for descriptions of
   * other variants of this method near where this one appears in the
   * documentation.
   */
 public";

%javamethodmodifiers SBMLDocument::SBMLDocument(const SBMLDocument& rhs) "
  /**
   * Copy constructor; creates a copy of this {@link SBMLDocument}.
   */
 public";

%javamethodmodifiers SBMLDocument::accept(SBMLVisitor& v) const "
  /**
   * Accepts the given {@link SBMLVisitor} for this instance of {@link SBMLDocument}.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used.
   * <p>
   * @return the result of calling <code>v.visit()</code>.
   */
 public";

%javamethodmodifiers SBMLDocument::clone() const "
  /**
   * Creates and returns a deep copy of this {@link SBMLDocument}.
   * <p>
   * @return a (deep) copy of this {@link SBMLDocument}.
   */
 public";

%javamethodmodifiers SBMLDocument::getModel() const "
  /**
   * Returns the {@link Model} object stored in this {@link SBMLDocument}.
   * <p>
   * It is important to note that this method <em>does not create</em> a
   * {@link Model} instance.  The model in the {@link SBMLDocument} must have been created
   * at some prior time, for example using SBMLDocument.createModel() 
   * or {@link SBMLDocument#setModel(Model m)} .
   * This method returns NULL if a model does not yet exist.
   * <p>
   * @return the {@link Model} contained in this {@link SBMLDocument}.
   * <p>
   * @see #createModel()
   */
 public";

%javamethodmodifiers SBMLDocument::getModel "
  /**
   * Returns the {@link Model} object stored in this {@link SBMLDocument}.
   * <p>
   * It is important to note that this method <em>does not create</em> a
   * {@link Model} instance.  The model in the {@link SBMLDocument} must have been created
   * at some prior time, for example using SBMLDocument.createModel() 
   * or {@link SBMLDocument#setModel(Model m)} .
   * This method returns NULL if a model does not yet exist.
   * <p>
   * @return the {@link Model} contained in this {@link SBMLDocument}.
   * <p>
   * @see #createModel()
   */
 public";

%javamethodmodifiers SBMLDocument::setLevelAndVersion(unsigned int level, unsigned int version, bool strict = true) "
  /**
   * Sets the SBML Level and Version of this {@link SBMLDocument} instance,
   * attempting to convert the model as needed.
   * <p>
   * This method is the principal way in libSBML to convert models between
   * Levels and Versions of SBML.  Generally, models can be converted
   * upward without difficulty (e.g., from SBML Level&nbsp;1 to
   * Level&nbsp;2, or from an earlier Version of Level&nbsp;2 to the latest
   * Version of Level&nbsp;2).  Sometimes models can be translated downward
   * as well, if they do not use constructs specific to more advanced
   * Levels of SBML.
   * <p>
   * Before calling this method, callers may check compatibility directly
   * using the methods SBMLDocument.checkL1Compatibility(),
   * SBMLDocument.checkL2v1Compatibility(),
   * SBMLDocument.checkL2v2Compatibility(),
   * SBMLDocument.checkL2v3Compatibility() and
   * SBMLDocument.checkL2v4Compatibility().
   * <p>
   * The valid combinations of SBML Level and Version as of this release
   * of libSBML are the following: 
   * <ul>
   * <li> Level&nbsp;1 Version&nbsp;2
   * <li> Level&nbsp;2 Version&nbsp;1
   * <li> Level&nbsp;2 Version&nbsp;2
   * <li> Level&nbsp;2 Version&nbsp;3
   * <li> Level&nbsp;2 Version&nbsp;4
   * </ul>
   * <p>
   * @param level the desired SBML Level
   * <p>
   * @param version the desired Version within the SBML Level
   * <p>
   * @param strict boolean indicating whether to check consistency
   * of both the source and target model when performing
   * conversion (defaults to <code> true <\code>)
   * <p>
   * @note Calling this method will not <em>necessarily</em> lead to a successful
   * conversion.  If the conversion fails, it will be logged in the error
   * list associated with this {@link SBMLDocument}.  Callers should consult
   * getNumErrors() to find out if the conversion succeeded without
   * problems.  For conversions from Level&nbsp;2 to Level&nbsp;1, callers
   * can also check the Level of the model after calling this method to
   * find out whether it is Level&nbsp;1.  (If the conversion to
   * Level&nbsp;1 failed, the Level of this model will be left unchanged.)
   * <p>
   * @see #checkL1Compatibility()
   * @see #checkL2v1Compatibility()
   * @see #checkL2v2Compatibility()
   * @see #checkL2v3Compatibility()
   * @see #checkL2v4Compatibility()
   * <p>
   * Strict conversion applies the additional criteria that both the source
   * and the target model must be consistent SBML.  Users can control the
   * consistency checks that are applied using the 
   * SBMLDocument.setConsistencyChecksForConversion function.  If either 
   * the source
   * or the potential target model have validation errors, the conversion
   * is not performed.  When a strict conversion is successful, the
   * underlying SBML object model is altered to reflect the new level
   * and version.  Thus information that cannot be converted (e.g. sboTerms)
   * will be lost.   
   */
 public";

%javamethodmodifiers SBMLDocument::setModel(const Model* m) "
  /**
   * Sets the {@link Model} for this {@link SBMLDocument} to a copy of the given {@link Model}.
   * <p>
   * @param m the new {@link Model} to use.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_LEVEL_MISMATCH
   * <li> LIBSBML_VERSION_MISMATCH
   * <p>
   * @see #createModel()
   * @see #getModel()
   */
 public";

%javamethodmodifiers SBMLDocument::createModel(const std::string& sid = "") "
  /**
   * Creates a new {@link Model} inside this {@link SBMLDocument}, and returns a pointer to
   * it.
   * <p>
   * In SBML Level&nbsp;2, the use of an identifier on a {@link Model} object is
   * optional.  This method takes an optional argument, <code>sid</code>, for setting
   * the identifier.  If not supplied, the identifier attribute on the
   * {@link Model} instance is not set.
   * <p>
   * @param sid the identifier of the new {@link Model} to create.
   * <p>
   * @docnote The native C++ implementation of this method defines a
   * default argument value.  In the documentation generated for different
   * libSBML language bindings, you may or may not see corresponding
   * arguments in the method declarations.  For example, in Java, a default
   * argument is handled by declaring two separate methods, with one of
   * them having the argument and the other one lacking the argument.
   * However, the libSBML documentation will be <em>identical</em> for both
   * methods.  Consequently, if you are reading this and do not see an
   * argument even though one is described, please look for descriptions of
   * other variants of this method near where this one appears in the
   * documentation.
   * <p>
   * @see #getModel()
   * @see SBMLDocument#setModel(Model m) 
   */
 public";

%javamethodmodifiers SBMLDocument::setConsistencyChecks(SBMLErrorCategory_t category, bool apply) "
  /**
   * Controls the consistency checks that are performed when
   * SBMLDocument.checkConsistency() is called.
   * <p>
   * This method works by adding or subtracting consistency checks from the
   * set of all possible checks that SBMLDocument.checkConsistency() knows
   * how to perform.  This method may need to be called multiple times in
   * order to achieve the desired combination of checks.  The first
   * argument (<code>category</code>) in a call to this method indicates the category
   * of consistency/error checks that are to be turned on or off, and the
   * second argument (<code>apply</code>, a boolean) indicates whether to turn it on
   * (value of <code>true</code>) or off (value of <code>false</code>).
   * <p>
   * The possible categories (values to the argument <code>category</code>) are the
   * set of values from the enumeration #SBMLErrorCategory_t.  The
   * following are the possible choices in libSBML version&nbsp;3.3.x:
   * <p>
   * <li> <code>LIBSBML_CAT_GENERAL_CONSISTENCY</code>: Correctness and consistency of
   * specific SBML language constructs.  Performing this set of checks is
   * highly recommended.  With respect to the SBML specification, these
   * concern failures in applying the validation rules numbered 2xxxx in
   * the Level&nbsp;2 Versions&nbsp;2, 3 and&nbsp;4 specifications.
   * <p>
   * <li> <code>LIBSBML_CAT_IDENTIFIER_CONSISTENCY</code>: Correctness and consistency
   * of identifiers used for model entities.  An example of inconsistency
   * would be using a species identifier in a reaction rate formula without
   * first having declared the species.  With respect to the SBML
   * specification, these concern failures in applying the validation rules
   * numbered 103xx in the Level&nbsp;2 Versions&nbsp;2, 3 and&nbsp;4
   * specifications.
   * <p>
   * <li> <code>LIBSBML_CAT_UNITS_CONSISTENCY</code>: Consistency of measurement units
   * associated with quantities in a model.  With respect to the SBML
   * specification, these concern failures in applying the validation rules
   * numbered 105xx in the Level&nbsp;2 Versions&nbsp;2, 3 and&nbsp;4
   * specifications.
   * <p>
   * <li> <code>LIBSBML_CAT_MATHML_CONSISTENCY</code>: Syntax of MathML constructs.
   * With respect to the SBML specification, these concern failures in
   * applying the validation rules numbered 102xx in the Level&nbsp;2
   * Versions&nbsp;2, 3 and&nbsp;4 specifications.
   * <p>
   * <li> <code>LIBSBML_CAT_SBO_CONSISTENCY</code>: Consistency and validity of {@link SBO}
   * identifiers (if any) used in the model.  With respect to the SBML
   * specification, these concern failures in applying the validation rules
   * numbered 107xx in the Level&nbsp;2 Versions&nbsp;2, 3 and&nbsp;4
   * specifications.
   * <p>
   * <li> <code>LIBSBML_CAT_OVERDETERMINED_MODEL</code>: Static analysis of
   * whether the system of equations implied by a model is mathematically
   * overdetermined.  With respect to the SBML specification, this is
   * validation rule #10601 in the SBML Level&nbsp;2 Versions&nbsp;2, 3
   * and&nbsp;4 specifications.
   * <p>
   * <li> <code>LIBSBML_CAT_MODELING_PRACTICE</code>: Additional checks for
   * recommended good modeling practice. (These are tests performed by
   * libSBML and do not have equivalent SBML validation rules.)
   * <p>
   * <em>By default, all validation checks are applied</em> to the model in
   * an {@link SBMLDocument} object <em>unless</em> {@link SBMLDocument#setConsistencyChecks(int categ, boolean onoff)}  is called to
   * indicate that only a subset should be applied.  Further, this default
   * (i.e., performing all checks) applies separately to <em>each new
   * {@link SBMLDocument} object</em> created.  In other words, each time a model
   * is read using {@link SBMLReader#readSBML(String filename)} , {@link SBMLReader#readSBMLFromString(String xml)} ,
   * or the global functions readSBML() and readSBMLFromString(), a new
   * {@link SBMLDocument} is created and for that document, a call to
   * SBMLDocument.checkConsistency() will default to applying all possible checks.
   * Calling programs must invoke {@link SBMLDocument#setConsistencyChecks(int categ, boolean onoff)}  for each such new
   * model if they wish to change the consistency checks applied.
   * <p>
   * @param category a value drawn from #SBMLErrorCategory_t indicating the
   * consistency checking/validation to be turned on or off
   * <p>
   * @param apply a boolean indicating whether the checks indicated by
   * <code>category</code> should be applied or not.
   * <p>
   * @see SBMLDocument#checkConsistency()
   */
 public";

%javamethodmodifiers SBMLDocument::setConsistencyChecksForConversion(SBMLErrorCategory_t category, bool apply) "
  /**
   * Controls the consistency checks that are performed when
   * {@link SBMLDocument#setLevelAndVersion(level, version)} is called.
   * <p>
   * This method works by adding or subtracting consistency checks from the
   * set of all possible checks that may be performed to avoid conversion
   * to or from an invalid document.  This method may need to be called 
   * multiple times in
   * order to achieve the desired combination of checks.  The first
   * argument (<code>category</code>) in a call to this method indicates the category
   * of consistency/error checks that are to be turned on or off, and the
   * second argument (<code>apply</code>, a boolean) indicates whether to turn it on
   * (value of <code>true</code>) or off (value of <code>false</code>).
   * <p>
   * The possible categories (values to the argument <code>category</code>) are the
   * set of values from the enumeration #SBMLErrorCategory_t.  The
   * following are the possible choices in libSBML version&nbsp;4.x.x:
   * <p>
   * <li> <code>LIBSBML_CAT_GENERAL_CONSISTENCY</code>: Correctness and consistency of
   * specific SBML language constructs.  Performing this set of checks is
   * highly recommended.  With respect to the SBML specification, these
   * concern failures in applying the validation rules numbered 2xxxx in
   * the Level&nbsp;2 Versions&nbsp;2, 3 and&nbsp;4 specifications.
   * <p>
   * <li> <code>LIBSBML_CAT_IDENTIFIER_CONSISTENCY</code>: Correctness and consistency
   * of identifiers used for model entities.  An example of inconsistency
   * would be using a species identifier in a reaction rate formula without
   * first having declared the species.  With respect to the SBML
   * specification, these concern failures in applying the validation rules
   * numbered 103xx in the Level&nbsp;2 Versions&nbsp;2, 3 and&nbsp;4
   * specifications.
   * <p>
   * <li> <code>LIBSBML_CAT_UNITS_CONSISTENCY</code>: Consistency of measurement units
   * associated with quantities in a model.  With respect to the SBML
   * specification, these concern failures in applying the validation rules
   * numbered 105xx in the Level&nbsp;2 Versions&nbsp;2, 3 and&nbsp;4
   * specifications.
   * <p>
   * <li> <code>LIBSBML_CAT_MATHML_CONSISTENCY</code>: Syntax of MathML constructs.
   * With respect to the SBML specification, these concern failures in
   * applying the validation rules numbered 102xx in the Level&nbsp;2
   * Versions&nbsp;2, 3 and&nbsp;4 specifications.
   * <p>
   * <li> <code>LIBSBML_CAT_SBO_CONSISTENCY</code>: Consistency and validity of {@link SBO}
   * identifiers (if any) used in the model.  With respect to the SBML
   * specification, these concern failures in applying the validation rules
   * numbered 107xx in the Level&nbsp;2 Versions&nbsp;2, 3 and&nbsp;4
   * specifications.
   * <p>
   * <li> <code>LIBSBML_CAT_OVERDETERMINED_MODEL</code>: Static analysis of
   * whether the system of equations implied by a model is mathematically
   * overdetermined.  With respect to the SBML specification, this is
   * validation rule #10601 in the SBML Level&nbsp;2 Versions&nbsp;2, 3
   * and&nbsp;4 specifications.
   * <p>
   * <li> <code>LIBSBML_CAT_MODELING_PRACTICE</code>: Additional checks for
   * recommended good modeling practice. (These are tests performed by
   * libSBML and do not have equivalent SBML validation rules.)
   * <p>
   * <em>By default, all validation checks are applied</em> to the model in
   * an {@link SBMLDocument} object <em>unless</em> {@link SBMLDocument#setConsistencyChecks(int categ, boolean onoff)}  is called to
   * indicate that only a subset should be applied.  Further, this default
   * (i.e., performing all checks) applies separately to <em>each new
   * {@link SBMLDocument} object</em> created.  In other words, each time a model
   * is read using {@link SBMLReader#readSBML(String filename)} , {@link SBMLReader#readSBMLFromString(String xml)} ,
   * or the global functions readSBML() and readSBMLFromString(), a new
   * {@link SBMLDocument} is created and for that document, a call to
   * SBMLDocument.checkConsistency() will default to applying all possible checks.
   * Calling programs must invoke {@link SBMLDocument#setConsistencyChecks(int categ, boolean onoff)}  for each such new
   * model if they wish to change the consistency checks applied.
   * <p>
   * @param category a value drawn from #SBMLErrorCategory_t indicating the
   * consistency checking/validation to be turned on or off
   * <p>
   * @param apply a boolean indicating whether the checks indicated by
   * <code>category</code> should be applied or not.
   * <p>
   * @see SBMLDocument#setLevelAndVersion()
   */
 public";

%javamethodmodifiers SBMLDocument::checkConsistency "
  /**
   * Performs consistency checking and validation on this SBML document.
   * <p>
   * If this method returns a nonzero value (meaning, one or more
   * consistency checks have failed for SBML document), the failures may be
   * due to warnings <em>or</em> errors.  Callers should inspect the severity
   * flag in the individual {@link SBMLError} objects returned by
   * {@link SBMLDocument#getError(long n)}  to determine the nature of the failures.
   * <p>
   * @return the number of failed checks (errors) encountered.
   * <p>
   * @see SBMLDocument#checkInternalConsistency()
   */
 public";

%javamethodmodifiers SBMLDocument::checkInternalConsistency "
  /**
   * Performs consistency checking on libSBML\'s internal representation of 
   * an SBML {@link Model}.
   * <p>
   * Callers should query the results of the consistency check by calling
   * {@link SBMLDocument#getError(long n)} .
   * <p>
   * @return the number of failed checks (errors) encountered.
   * <p>
   * The distinction between this method and
   * SBMLDocument.checkConsistency() is that this method reports on
   * fundamental syntactic and structural errors that violate the XML
   * Schema for SBML; by contrast, SBMLDocument.checkConsistency()
   * performs more elaborate model verifications and also validation
   * according to the validation rules written in the appendices of the
   * SBML Level&nbsp;2 Versions&nbsp;2&ndash;4 specification documents.
   * <p>
   * @see SBMLDocument#checkConsistency()
   */
 public";

%javamethodmodifiers SBMLDocument::checkL1Compatibility "
  /**
   * Performs a set of consistency checks on the document to establish
   * whether it is compatible with SBML Level&nbsp;1 and can be converted
   * to Level&nbsp;1.
   * <p>
   * Callers should query the results of the consistency check by calling
   * {@link SBMLDocument#getError(long n)} .
   * <p>
   * @return the number of failed checks (errors) encountered.
   */
 public";

%javamethodmodifiers SBMLDocument::checkL2v1Compatibility "
  /**
   * Performs a set of consistency checks on the document to establish
   * whether it is compatible with SBML Level&nbsp;2 Version&nbsp;1 and can
   * be converted to Level&nbsp;2 Version&nbsp;1.
   * <p>
   * Callers should query the results of the consistency check by calling
   * {@link SBMLDocument#getError(long n)} .
   * <p>
   * @return the number of failed checks (errors) encountered.
   */
 public";

%javamethodmodifiers SBMLDocument::checkL2v2Compatibility "
  /**
   * Performs a set of consistency checks on the document to establish
   * whether it is compatible with SBML Level&nbsp;2 Version&nbsp;2 and can
   * be converted to Level&nbsp;2 Version&nbsp;2.
   * <p>
   * Callers should query the results of the consistency check by calling
   * {@link SBMLDocument#getError(long n)} .
   * <p>
   * @return the number of failed checks (errors) encountered.
   */
 public";

%javamethodmodifiers SBMLDocument::checkL2v3Compatibility "
  /**
   * Performs a set of consistency checks on the document to establish
   * whether it is compatible with SBML Level&nbsp;2 Version&nbsp;3 and can
   * be converted to Level&nbsp;2 Version&nbsp;3.
   * <p>
   * Callers should query the results of the consistency check by calling
   * {@link SBMLDocument#getError(long n)} .
   * <p>
   * @return the number of failed checks (errors) encountered.
   */
 public";

%javamethodmodifiers SBMLDocument::checkL2v4Compatibility "
  /**
   * Performs a set of consistency checks on the document to establish
   * whether it is compatible with SBML Level&nbsp;2 Version&nbsp;4 and can
   * be converted to Level&nbsp;2 Version&nbsp;4.
   * <p>
   * Callers should query the results of the consistency check by calling
   * {@link SBMLDocument#getError(long n)} .
   * <p>
   * @return the number of failed checks (errors) encountered.
   */
 public";

%javamethodmodifiers SBMLDocument::getError "
  /**
   * Returns the nth error or warning encountered during parsing,
   * consistency checking, or attempted translation of this model.
   * <p>
   * Callers can use method XMLError.getSeverity() on the result to assess
   * the severity of the problem.  The possible severity levels range from
   * informational messages to fatal errors.
   * <p>
   * @return the error or warning indexed by integer <code>n</code>, or return 
   * <code>NULL</code> if <code>n &gt; (getNumErrors() - 1)</code>.
   * <p>
   * @param n the integer index of the error sought.
   * <p>
   * @see SBMLDocument#getNumErrors()
   */
 public";

%javamethodmodifiers SBMLDocument::getNumErrors() const "
  /**
   * Returns the number of errors or warnings encountered during parsing,
   * consistency checking, or attempted translation of this model.
   * <p>
   * @return the number of errors or warnings encountered
   */
 public";

%javamethodmodifiers SBMLDocument::printErrors(std::ostream& stream = std::cerr) const "
  /**
   * Prints to the given output stream all the errors or warnings
   * encountered during parsing, consistency checking, or attempted
   * translation of this model.
   * <p>
   * If no errors have occurred, i.e., <code>getNumErrors() == 0</code>, no
   * output will be sent to the stream.
   * <p>
   * The format of the output is:
   * <div class=\'fragment\'><pre>
   N error(s):
     line NNN: (id) message
</pre></div>
   * <p>
   * @docnote The native C++ implementation of this method defines a
   * default argument value.  In the documentation generated for different
   * libSBML language bindings, you may or may not see corresponding
   * arguments in the method declarations.  For example, in Java, a default
   * argument is handled by declaring two separate methods, with one of
   * them having the argument and the other one lacking the argument.
   * However, the libSBML documentation will be <em>identical</em> for both
   * methods.  Consequently, if you are reading this and do not see an
   * argument even though one is described, please look for descriptions of
   * other variants of this method near where this one appears in the
   * documentation.
   */
 public";

%javamethodmodifiers SBMLDocument::setSBMLDocument(SBMLDocument* d) "
  /**
   * No-op; it is provided for consistency with the method available on
   * other libSBML object classes but has no effect on {@link SBMLDocument}.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBMLDocument::getTypeCode() const "
  /**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for this object, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see SBMLDocument#getElementName()
   */
 public";

%javamethodmodifiers SBMLDocument::getElementName() const "
  /**
   * Returns the XML element name of this object, which for {@link SBMLDocument},
   * is always <code>\'sbml\'</code>.
   * <p>
   * @return the name of this element, i.e., <code>\'sbml\'</code>.
   */
 public";

%javamethodmodifiers SBMLDocument::getErrorLog "
  /**
   * Returns the list of errors or warnings logged during parsing, 
   * consistency checking, or attempted translation of this model.
   * <p>
   * @return the {@link SBMLErrorLog} used for this {@link SBMLDocument}
   * <p>
   * @see SBMLDocument#getNumErrors()
   */
 public";

%javamethodmodifiers SBMLDocument::getNamespaces() const "
  /**
   * Returns a list of XML Namespaces associated with the XML content
   * of this SBML document.
   * <p>
   * @return the XML Namespaces associated with this SBML object
   */
 public";

%javamethodmodifiers SBMLDocument::getElementPosition() const "
  /**
   * @return the ordinal position of the element with respect to its
   * siblings or -1 (default) to indicate the position is not significant.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBMLDocument::writeElements(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write out their contained
   * SBML objects as XML elements.  Be sure to call your parents
   * implementation of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBMLDocument::createObject(XMLInputStream& stream) "
  /**
   * @return the SBML object corresponding to next {@link XMLToken} in the
   * XMLInputStream or NULL if the token was not recognized.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBMLDocument::readAttributes(const XMLAttributes& attributes) "
  /**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBMLDocument::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBMLDocument::conversion_errors(unsigned int errors) "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBMLDocument::hasStrictUnits "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBMLDocument::hasStrictSBO "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBMLDocument::hasStrictSBO "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBMLDocument::hasStrictSBO "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBMLDocument::hasStrictSBO "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBMLDocument::hasStrictSBO "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBMLDocument::hasStrictSBO "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBMLDocument::hasStrictSBO "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBMLDocument::hasStrictSBO "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SBMLDocument::hasStrictSBO "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%typemap(javaimports) FunctionDefinition "
/** 
 * LibSBML implementation of SBML\'s FunctionDefinition construct.
 * <p>
 * The {@link FunctionDefinition} structure associates an identifier with a
 * function definition.  This identifier can then be used as the function
 * called in subsequent MathML content elsewhere in an SBML model.
 * <p>
 * {@link FunctionDefinition} has one required attribute, \'id\', to give the
 * function a unique identifier by which other parts of an SBML model
 * definition can refer to it.  A {@link FunctionDefinition} instance can also have
 * an optional \'name\' attribute of type <code>string</code>.  Identifiers and names
 * must be used according to the guidelines described in the SBML
 * specification (e.g., Section 3.3 in the Level 2 Version 4
 * specification).
 * <p>
 * {@link FunctionDefinition} has a required \'math\' subelement containing a MathML
 * expression defining the function body.  The content of this element can
 * only be a MathML \'lambda\' element.  The \'lambda\' element must begin with
 * zero or more \'bvar\' elements, followed by any other of the elements in
 * the MathML subset allowed in SBML Level 2 <em>except</em> \'lambda\' (i.e., a
 * \'lambda\' element cannot contain another \'lambda\' element).  This is the
 * only place in SBML where a \'lambda\' element can be used.  The function
 * defined by a {@link FunctionDefinition} is only available for use in other
 * MathML elements that <em>follow</em> the {@link FunctionDefinition} definition in the
 * model.  (These restrictions prevent recursive and mutually-recursive
 * functions from being expressed.)
 * <p>
 * A further restriction on the content of \'math\' is that it cannot contain
 * references to variables other than the variables declared to the
 * \'lambda\' itself.  That is, the contents of MathML \'ci\' elements inside
 * the body of the \'lambda\' can only be the variables declared by its
 * \'bvar\' elements, or the identifiers of other {@link FunctionDefinition}
 * instances earlier in the model.  This means must be written so that all
 * variables or parameters used in the MathML content are passed to them
 * via their function parameters.
 * <p>
 * @note Function definitions (also informally known as user-defined
 * functions) were introduced in SBML Level 2.  They have purposefully
 * limited capabilities.  A function cannot reference parameters or other
 * model quantities outside of itself; values must be passed as parameters
 * to the function.  Moreover, recursive and mutually-recursive functions
 * are not permitted.  The purpose of these limitations is to balance power
 * against complexity of implementation.  With the restrictions as they
 * are, function definitions could be implemented as textual
 * substitutions&mdash;they are simply macros.  Software implementations
 * therefore do not need the full function-definition machinery typically
 * associated with programming languages.
 * <p>
 * @note Another important point to note is {@link FunctionDefinition} does not
 * have a separate attribute for defining the units of the value returned
 * by the function.  The units associated with the function\'s return value,
 * when the function is called from within MathML expressions elsewhere in
 * SBML, are simply the overall units of the expression in
 * {@link FunctionDefinition}\'s \'math\' subelement when applied to the arguments
 * supplied in the call to the function.  Ascertaining these units requires
 * performing dimensional analysis on the expression.  (Readers may wonder
 * why there is no attribute.  The reason is that having a separate
 * attribute for declaring the units would not only be redundant, but also
 * lead to the potential for having conflicting information.  In the case
 * of a conflict between the declared units and those of the value actually
 * returned by the function, the only logical resolution rule would be to
 * assume that the correct units are those of the expression anyway.)
 * <p>
 * <!---------------------------------------------------------------------- -->
 * <p>
 */
"

%typemap(javaimports) ListOfFunctionDefinitions "
/** 
 * LibSBML implementation of SBML\'s ListOfFunctionDefinitions construct.
 * <p>
 * The various ListOf___ classes in SBML are merely containers used for
 * organizing the main components of an SBML model.  All are derived from
 * the abstract class {@link SBase}, and inherit the various attributes and
 * subelements of {@link SBase}, such as \'metaid\' as and \'annotation\'.  The
 * ListOf___ classes do not add any attributes of their own.
 * <p>
 * The relationship between the lists and the rest of an SBML model is
 * illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):
 * <p>
 * <center><img src=\'listof-illustration.jpg\'></center><br>
 * 
 * <p>
 * Readers may wonder about the motivations for using the ListOf___
 * containers.  A simpler approach in XML might be to place the components
 * all directly at the top level of the model definition.  We chose instead
 * to group them within XML elements named after {@link ListOf}<em>Classname</em>,
 * in part because we believe this helps organize the components and makes
 * visual reading of models in XML easier.  More importantly, the fact that
 * the container classes are derived from {@link SBase} means that software tools
 * can add information about the lists themselves into each list
 * container\'s \'annotation\'.
 * <p>
 * @see ListOfFunctionDefinitions
 * @see ListOfUnitDefinitions
 * @see ListOfCompartmentTypes
 * @see ListOfSpeciesTypes
 * @see ListOfCompartments
 * @see ListOfSpecies
 * @see ListOfParameters
 * @see ListOfInitialAssignments
 * @see ListOfRules
 * @see ListOfConstraints
 * @see ListOfReactions
 * @see ListOfEvents
 */
"

%javamethodmodifiers FunctionDefinition::FunctionDefinition(unsigned int level, unsigned int version) "
  /**
   * Creates a new {@link FunctionDefinition} using the given SBML <code>level</code> and <code>version</code>
   * values.
   * <p>
   * @param level a long integer, the SBML Level to assign to this {@link FunctionDefinition}
   * <p>
   * @param version a long integer, the SBML Version to assign to this
   * {@link FunctionDefinition}
   * <p>
   * @note Once a {@link FunctionDefinition} has been added to an {@link SBMLDocument}, the <code>level</code>,
   * <code>version</code> for the document <em>override</em> those used
   * to create the {@link FunctionDefinition}.  Despite this, the ability to supply the values
   * at creation time is an important aid to creating valid SBML.  Knowledge of
   * the intented SBML Level and Version determine whether it is valid to
   * assign a particular value to an attribute, or whether it is valid to add
   * an object to an existing {@link SBMLDocument}.
   */
 public";

%javamethodmodifiers FunctionDefinition::FunctionDefinition(SBMLNamespaces* sbmlns) "
  /**
   * Creates a new {@link FunctionDefinition} using the given {@link SBMLNamespaces} object
   * <code>sbmlns</code>.
   * <p>
   * The {@link SBMLNamespaces} object encapsulates SBML Level/Version/namespaces
   * information.  It is used to communicate the SBML Level, Version, and
   * (in Level&nbsp;3) packages used in addition to SBML Level&nbsp; Core.
   * A common approach to using this class constructor is to create an
   * {@link SBMLNamespaces} object somewhere in a program, once, then pass it to
   * object constructors such as this one when needed.
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object.
   * <p>
   * @note Once a {@link FunctionDefinition} has been added to an {@link SBMLDocument}, the <code>level</code>,
   * <code>version</code> and <code>xmlns</code> namespaces for the document <em>override</em> those used
   * to create the {@link FunctionDefinition}.  Despite this, the ability to supply the values
   * at creation time is an important aid to creating valid SBML.  Knowledge of
   * the intented SBML Level and Version determine whether it is valid to
   * assign a particular value to an attribute, or whether it is valid to add
   * an object to an existing {@link SBMLDocument}.
   */
 public";

%javamethodmodifiers FunctionDefinition::FunctionDefinition(const FunctionDefinition& orig) "
  /**
   * Copy constructor; creates a copy of this {@link FunctionDefinition}.
   */
 public";

%javamethodmodifiers FunctionDefinition::accept(SBMLVisitor& v) const "
  /**
   * Accepts the given {@link SBMLVisitor} for this instance of {@link FunctionDefinition}.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used.
   * <p>
   * @return the result of calling <code>v.visit()</code>, which indicates
   * whether the Visitor would like to visit the next {@link FunctionDefinition} in
   * the list of function definitions.
   */
 public";

%javamethodmodifiers FunctionDefinition::clone() const "
  /**
   * Creates and returns a deep copy of this {@link FunctionDefinition}.
   * <p>
   * @return a (deep) copy of this {@link FunctionDefinition}.
   */
 public";

%javamethodmodifiers FunctionDefinition::getId() const "
  /**
   * Returns the value of the \'id\' attribute of this {@link FunctionDefinition}.
   * <p>
   * @return the id of this {@link FunctionDefinition}.
   */
 public";

%javamethodmodifiers FunctionDefinition::getName() const "
  /**
   * Returns the value of the \'name\' attribute of this {@link FunctionDefinition}.
   * <p>
   * @return the name of this {@link FunctionDefinition}.
   */
 public";

%javamethodmodifiers FunctionDefinition::getMath() const "
  /**
   * Get the mathematical formula of this {@link FunctionDefinition}.
   * <p>
   * @return an {@link ASTNode}, the value of the \'math\' subelement of this
   * {@link FunctionDefinition}
   */
 public";

%javamethodmodifiers FunctionDefinition::isSetId() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link FunctionDefinition}\'s \'id\' attribute has been set.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the \'id\' attribute of this {@link FunctionDefinition} has been
   * set, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers FunctionDefinition::isSetName() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link FunctionDefinition}\'s \'name\' attribute has been set.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the \'name\' attribute of this {@link FunctionDefinition} has been
   * set, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers FunctionDefinition::isSetMath() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link FunctionDefinition}\'s \'math\' subelement contains a value.
   * <p>
   * @return <code>true</code> if the \'math\' for this {@link FunctionDefinition} has been set,
   * <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers FunctionDefinition::setId(const std::string& sid) "
  /**
   * Sets the value of the \'id\' attribute of this {@link FunctionDefinition}.
   * <p>
   * The string <code>sid</code> is copied.  Note that SBML has strict requirements
   * for the syntax of identifiers.  The following is summary of the
   * definition of the SBML identifier type <code>SId</code> (here expressed in an
   * extended form of BNF notation):
   * <div class=\'fragment\'><pre>
   *   letter .= \'a\'..\'z\',\'A\'..\'Z\'
   *   digit  .= \'0\'..\'9\'
   *   idChar .= letter | digit | \'_\'
   *   SId    .= ( letter | \'_\' ) idChar</pre></div>
   * The equality of SBML identifiers is determined by an exact character
   * sequence match; i.e., comparisons must be performed in a
   * case-sensitive manner.  In addition, there are a few conditions for
   * the uniqueness of identifiers in an SBML model.  Please consult the
   * SBML specifications for the exact formulations.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param sid the string to use as the identifier of this {@link FunctionDefinition}
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   */
 public";

%javamethodmodifiers FunctionDefinition::setName(const std::string& name) "
  /**
   * Sets the value of the \'name\' attribute of this {@link FunctionDefinition}.
   * <p>
   * The string in <code>name</code> is copied.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param name the new name for the {@link FunctionDefinition}
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   */
 public";

%javamethodmodifiers FunctionDefinition::setMath(const ASTNode* math) "
  /**
   * Sets the \'math\' subelement of this {@link FunctionDefinition} to the Abstract
   * Syntax Tree given in <code>math</code>.
   * <p>
   * @param math an AST containing the mathematical expression to
   * be used as the formula for this {@link FunctionDefinition}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_OBJECT
   */
 public";

%javamethodmodifiers FunctionDefinition::unsetName "
  /**
   * Unsets the value of the \'name\' attribute of this {@link FunctionDefinition}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_OPERATION_FAILED
   */
 public";

%javamethodmodifiers FunctionDefinition::getArgument "
  /**
   * Get the <code>n</code>th argument to this function.
   * <p>
   * Callers should first find out the number of arguments to the function
   * by calling getNumArguments().
   * <p>
   * @param n an integer index for the argument sought.
   * <p>
   * @return the nth argument (bound variable) passed to this
   * {@link FunctionDefinition}.
   * <p>
   * @see #getNumArguments()
   */
 public";

%javamethodmodifiers FunctionDefinition::getArgument(const std::string& name) const "
  /**
   * Get the argument named <code>name</code> to this {@link FunctionDefinition}.
   * <p>
   * @param name the exact name (case-sensitive) of the sought-after
   * argument
   * <p>
   * @return the argument (bound variable) having the given name, or NULL if
   * no such argument exists.
   */
 public";

%javamethodmodifiers FunctionDefinition::getBody() const "
  /**
   * Get the mathematical expression that is the body of this
   * {@link FunctionDefinition} object.
   * <p>
   * @return the body of this {@link FunctionDefinition} as an Abstract Syntax
   * Tree, or NULL if no body is defined.
   */
 public";

%javamethodmodifiers FunctionDefinition::getBody "
  /**
   * Get the mathematical expression that is the body of this
   * {@link FunctionDefinition} object.
   * <p>
   * @return the body of this {@link FunctionDefinition} as an Abstract Syntax
   * Tree, or NULL if no body is defined.
   */
 public";

%javamethodmodifiers FunctionDefinition::getNumArguments() const "
  /**
   * Get the number of arguments (bound variables) taken by this
   * {@link FunctionDefinition}.
   * <p>
   * @return the number of arguments (bound variables) that must be passed
   * to this {@link FunctionDefinition}.
   */
 public";

%javamethodmodifiers FunctionDefinition::getTypeCode() const "
  /**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for this object, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers FunctionDefinition::getElementName() const "
  /**
   * Returns the XML element name of this object, which for
   * {@link FunctionDefinition}, is always <code>\'functionDefinition\'</code>.
   * <p>
   * @return the name of this element, i.e., <code>\'functionDefinition\'</code>.
   */
 public";

%javamethodmodifiers FunctionDefinition::writeElements(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write out their contained
   * SBML objects as XML elements.  Be sure to call your parents
   * implementation of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers FunctionDefinition::hasRequiredAttributes() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link FunctionDefinition} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link FunctionDefinition} object are:
   * id
   * <p>
   * @return a boolean value indicating whether all the required
   * attributes for this object have been defined.
   */
 public";

%javamethodmodifiers FunctionDefinition::hasRequiredElements() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required elements for this {@link FunctionDefinition} object
   * have been set.
   * <p>
   * @note The required elements for a {@link FunctionDefinition} object are:
   * math
   * <p>
   * @return a boolean value indicating whether all the required
   * elements for this object have been defined.
   */
 public";

%javamethodmodifiers FunctionDefinition::FunctionDefinition "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required elements for this {@link FunctionDefinition} object
   * have been set.
   * <p>
   * @note The required elements for a {@link FunctionDefinition} object are:
   * math
   * <p>
   * @return a boolean value indicating whether all the required
   * elements for this object have been defined.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers FunctionDefinition::readOtherXML(XMLInputStream& stream) "
  /**
   * Subclasses should override this method to read (and store) XHTML,
   * MathML, etc. directly from the XMLInputStream.
   * <p>
   * @return true if the subclass read from the stream, false otherwise.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers FunctionDefinition::readAttributes(const XMLAttributes& attributes) "
  /**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers FunctionDefinition::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers FunctionDefinition::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers FunctionDefinition::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers FunctionDefinition::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers FunctionDefinition::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers FunctionDefinition::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers FunctionDefinition::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers FunctionDefinition::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers FunctionDefinition::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers FunctionDefinition::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers FunctionDefinition::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers FunctionDefinition::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers FunctionDefinition::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers FunctionDefinition::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers FunctionDefinition::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers FunctionDefinition::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers FunctionDefinition::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers FunctionDefinition::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers ListOfFunctionDefinitions::clone() const "
  /**
   * Creates and returns a deep copy of this {@link ListOfFunctionDefinitions} instance.
   * <p>
   * @return a (deep) copy of this {@link ListOfFunctionDefinitions}.
   */
 public";

%javamethodmodifiers ListOfFunctionDefinitions::getTypeCode() const "
  /**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for this object, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers ListOfFunctionDefinitions::getItemTypeCode() const "
  /**
   * Returns the libSBML type code for the objects contained in this {@link ListOf}
   * (i.e., {@link FunctionDefinition} objects, if the list is non-empty).
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for the objects contained in this {@link ListOf}
   * instance, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers ListOfFunctionDefinitions::getElementName() const "
  /**
   * Returns the XML element name of this object.
   * <p>
   * For {@link ListOfFunctionDefinitions}, the XML element name is 
   * <code>\'listOfFunctionDefinitions\'</code>.
   * <p>
   * @return the name of this element, i.e., <code>\'listOfFunctionDefinitions\'</code>.
   */
 public";

%javamethodmodifiers ListOfFunctionDefinitions::get(unsigned int n) "
  /**
   * Get a {@link FunctionDefinition} from the {@link ListOfFunctionDefinitions}.
   * <p>
   * @param n the index number of the {@link FunctionDefinition} to get.
   * <p>
   * @return the nth {@link FunctionDefinition} in this {@link ListOfFunctionDefinitions}.
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfFunctionDefinitions::get "
  /**
   * Get a {@link FunctionDefinition} from the {@link ListOfFunctionDefinitions}.
   * <p>
   * @param n the index number of the {@link FunctionDefinition} to get.
   * <p>
   * @return the nth {@link FunctionDefinition} in this {@link ListOfFunctionDefinitions}.
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfFunctionDefinitions::get(const std::string& sid) "
  /**
   * Get a {@link FunctionDefinition} from the {@link ListOfFunctionDefinitions}
   * based on its identifier.
   * <p>
   * @param sid a string representing the identifier 
   * of the {@link FunctionDefinition} to get.
   * <p>
   * @return {@link FunctionDefinition} in this {@link ListOfFunctionDefinitions}
   * with the given id or NULL if no such
   * {@link FunctionDefinition} exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfFunctionDefinitions::get(const std::string& sid) const "
  /**
   * Get a {@link FunctionDefinition} from the {@link ListOfFunctionDefinitions}
   * based on its identifier.
   * <p>
   * @param sid a string representing the identifier 
   * of the {@link FunctionDefinition} to get.
   * <p>
   * @return {@link FunctionDefinition} in this {@link ListOfFunctionDefinitions}
   * with the given id or NULL if no such
   * {@link FunctionDefinition} exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfFunctionDefinitions::remove(unsigned int n) "
  /**
   * Removes the nth item from this {@link ListOfFunctionDefinitions} items and returns a pointer to
   * it.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * <p>
   * @param n the index of the item to remove
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfFunctionDefinitions::remove(const std::string& sid) "
  /**
   * Removes item in this {@link ListOfFunctionDefinitions} items with the given identifier.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * If none of the items in this list have the identifier <code>sid</code>, then 
   * <code>NULL</code> is returned.
   * <p>
   * @param sid the identifier of the item to remove
   * <p>
   * @return the item removed.  As mentioned above, the caller owns the
   * returned item.
   */
 public";

%javamethodmodifiers ListOfFunctionDefinitions::getElementPosition() const "
  /**
   * Get the ordinal position of this element in the containing object
   * (which in this case is the {@link Model} object).
   * <p>
   * The ordering of elements in the XML form of SBML is generally fixed
   * for most components in SBML.  So, for example, the
   * {@link ListOfFunctionDefinitions} in a model is (in SBML Level 2 Version 4)
   * the first ListOf___.  (However, it differs for different Levels and
   * Versions of SBML.)
   * <p>
   * @return the ordinal position of the element with respect to its
   * siblings, or <code>-1</code> (default) to indicate the position is not significant.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers ListOfFunctionDefinitions::createObject(XMLInputStream& stream) "
  /**
   * @return the SBML object corresponding to next {@link XMLToken} in the
   * XMLInputStream or NULL if the token was not recognized.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers UnitKind_equals(UnitKind_t uk1, UnitKind_t uk2) "
/**
 * Tests for logical equality between two given UnitKind_t values.
 * <p>
 * This function behaves exactly like C\'s <tt>==</tt> operator, except for
 * the following two cases:
 * <p>
 * <li> UNIT_KIND_LITER <code>==</code> UNIT_KIND_LITRE
 * <li> UNIT_KIND_METER <code>==</code> UNIT_KIND_METRE
 * <p>
 * C equality comparison would yield false (because each of the above is a
 * distinct enumeration value), but UnitKind_equals() returns true.
 * <p>
 * @param uk1 a UnitKind_t enumeration value 
 * @param uk2 the UnitKind_t enumeration value to compare to <code>uk1</code>
 * <p>
 * @return nonzero (for true) if uk1 is logically equivalent to uk2,
 * zero (0) otherwise.
 * <p>
 * @note For more information about the UnitKind_t enumeration, please
 * refer to the class documentation for {@link Unit}.
 */
 public";

%javamethodmodifiers UnitKind_forName(const char *name) "
/**
 * Converts a string to its corresponding UnitKind_t enumeration value.
 * <p>
 * @param name a string, the name of a predefined base unit in SBML
 * <p>
 * @return a value from UnitKind_t corresponding to the given name
 * (determined in a case-insensitive manner).
 * <p>
 * @note For more information about the UnitKind_t enumeration, please
 * refer to the class documentation for {@link Unit}.
 */
 public";

%javamethodmodifiers UnitKind_toString(UnitKind_t uk) "
/**
 * Converts a UnitKind_t enumeration value to a text string equivalent.
 * <p>
 * @param uk the UnitKind_t value to convert
 * <p>
 * @return the name of the given UnitKind.
 * <p>
 * @note The string returned is a static data value.  The caller does not
 * own the returned string and is therefore not allowed to modify it.
 * <p>
 * @note For more information about the UnitKind_t enumeration, please
 * refer to the class documentation for {@link Unit}.
 */
 public";

%javamethodmodifiers UnitKind_isValidUnitKindString(const char *string, unsigned int level, unsigned int version) "
/**
 * Predicate for testing whether a given string corresponds to a
 * predefined UnitKind_t enumeration value.
 * <p>
 * @return nonzero (for true) if string is the name of a valid
 * UnitKind_t enumeration value, zero (0) otherwise.
 * <p>
 * @note For more information about the UnitKind_t enumeration, please
 * refer to the class documentation for {@link Unit}.
 */
 public";

%typemap(javaimports) Unit "
/** 
 * LibSBML implementation of SBML\'s Unit construct.
 * <p>
 * The SBML unit definition facility uses two classes of objects,
 * {@link UnitDefinition} and {@link Unit}.  The approach to defining units in SBML is
 * compositional; for example, <em>meter second<sup> &ndash;2</sup></em> is
 * constructed by combining a {@link Unit} object representing <em>meter</em> with
 * another {@link Unit} object representing <em>second<sup> &ndash;2</sup></em>.
 * The combination is wrapped inside a {@link UnitDefinition}, which provides for
 * assigning an identifier and optional name to the combination.  The
 * identifier can then be referenced from elsewhere in a model.  Thus, the
 * {@link UnitDefinition} class is the container, and {@link Unit} instances are placed
 * inside {@link UnitDefinition} instances.
 * <p>
 * A {@link Unit} structure has four attributes named \'kind\', \'exponent\', \'scale\'
 * and \'multiplier\'.  It represents a (possibly transformed) reference to a
 * base unit.  The attribute \'kind\' on {@link Unit} indicates the chosen base unit.
 * Its value in SBML Level&nbsp;2 Version&nbsp;4 must be one of the following
 * predefined strings:
 * <p>
 * <table align=\'center\' style=\'font-family: Courier, fixed; font-weight: bold; font-size: 12px;\' cellspacing=\'7\' border=\'0\'>
<tr><td>ampere</td><td>gram</td><td>katal</td><td>lux</td><td>pascal</td><td>tesla</td></tr>
<tr><td>becquerel</td><td>gray</td><td>kelvin</td><td>meter</td><td>radian</td><td>volt</td></tr>
<tr><td>candela</td><td>henry</td><td>kilogram</td><td>metre</td><td>second</td><td>watt</td></tr>
<tr><td>coulomb</td><td>hertz</td><td>liter</td><td>mole</td><td>siemens</td><td>weber</td></tr>
<tr><td>dimensionless</td><td>item</td><td>litre</td><td>newton</td><td>sievert</td></tr>
<tr><td>farad</td><td>joule</td><td>lumen</td><td>ohm</td><td>steradian</td></tr>
</table>

 * <p>
 * (See also the definition of the <code>UNIT_KIND_</code> constants in <a
 * class=\'el\' href=\'#UnitKind_t\'>the section below</a>.)  For SBML
 * Level&nbsp;2 Version&nbsp;1 models, the string <code>Celsius</code> is also
 * permitted.  Note also that the table above contains two names not
 * strictly defined in SBML Level&nbsp;2: <code>liter</code> and <code>meter</code>.  For
 * convenience, libSBML defines the American English spellings of these
 * units, although the SBML specification limits the valid unit names to
 * those defined by SI.
 * <p>
 * The optional attribute named \'exponent\' on {@link Unit} represents an exponent
 * on the unit.  Its default value is <code>1</code> (one).  A {@link Unit} structure also
 * has an optional attribute called \'scale\'; its value must be an integer
 * exponent for a power-of-ten multiplier used to set the scale of the
 * unit.  For example, a unit having a \'kind\' value of <code>gram</code> and a
 * \'scale\' value of <code>-3</code> signifies 10<sup>&nbsp;&ndash;3</sup>
 * &#215; gram, or milligrams.  The default value of \'scale\' is <code>0</code>
 * (zero), because 10<sup> 0</sup> = 1.  Lastly, the optional attribute
 * named \'multiplier\' can be used to multiply the kind unit by a
 * real-numbered factor; this enables the definition of units that are not
 * power-of-ten multiples of SI units.  For instance, a multiplier of
 * 0.3048 could be used to define <code>foot</code> as a measure of length in terms
 * of a <code>metre</code>.  The \'multiplier\' attribute has a default value of <code>1</code>
 * (one).
 * <p>
 * <h3><a class=\'anchor\' name=\'UnitKind_t\'>UnitKind_t</a></h3>
 * <p>
 * SBML defines a set of base units which serves as the starting point for
 * new unit definitions.  This set of base units consists of the SI units
 * and a small number of additional convenience units.
 * <p>
 * In SBML Level&nbsp;2 Versions before
 * Version&nbsp;3, there existed an enumeration of units called 
 * <code>UnitKind</code>.  In Version&nbsp;3, this enumeration was removed and the
 * identifier class <code>UnitSId</code> redefined to include the previous 
 * <code>UnitKind</code> values as reserved symbols.  This change has no net effect on
 * permissible models, their representation or their syntax.  The purpose
 * of the change in the SBML specification was simply to clean up an
 * inconsistency about the contexts in which these values were usable.
 * However, LibSBML maintains UnitKind in the form of of a set of static
 * integer constants whose names begin with the characters
 * <code>UNIT_KIND_</code>.  These constants are defined in the class
 * <code><a href=\'libsbmlConstants.html\'>libsbmlConstants</a></code> This
 * enumeration in libSBML has a small number of differences compared to the
 * SBML specifications: 
 * <p>
 * <ul>
 * <li> The alternate spelling <code>\'meter\'</code> is included in addition to the
 * official SI spelling <code>\'metre\'</code>.
 * <p>
 * <li> The alternate spelling <code>\'liter\'</code> is included in addition to the
 * official SI spelling <code>\'litre\'</code>.
 * <p>
 * <li> The unit <code>\'Celsius\'</code> is included because of its presence in
 * specifications of SBML prior to SBML Level&nbsp;2 Version&nbsp;3.
 * </ul>
 * <p>
 * The table below lists the unit
 * constants defined in libSBML, and their meanings. 
 * <p>
 * <center>
<table border=\'0\' class=\'text-table width80 normal-font alt-row-colors\'>
 <tr>
     <th align=\'left\' width=\'200\'>Enumerator</th>
     <th align=\'left\'>Meaning</th>
 </tr>
<tr><td><code>UNIT_KIND_AMPERE</code></td><td>The ampere unit</td></tr>
<tr><td><code>UNIT_KIND_BECQUEREL</code></td><td>The becquerel unit.</td></tr>
<tr><td><code>UNIT_KIND_CANDELA</code></td><td>The candela unit.</td></tr>
<tr><td><code>UNIT_KIND_CELSIUS</code></td><td>The Celsius unit.</td></tr>
<tr><td><code>UNIT_KIND_COULOMB</code></td><td>The coulomb unit.</td></tr>
<tr><td><code>UNIT_KIND_DIMENSIONLESS</code></td><td>A pseudo-unit
indicating a dimensionless quantity.  (This is in fact defined in the
SBML specification.)</td></tr>
<tr><td><code>UNIT_KIND_FARAD</code></td><td>The farad unit.</td></tr>
<tr><td><code>UNIT_KIND_GRAM</code></td><td>The gram unit.</td></tr>
<tr><td><code>UNIT_KIND_GRAY</code></td><td>The gray unit.</td></tr>
<tr><td><code>UNIT_KIND_HENRY</code></td><td>The henry unit.</td></tr>
<tr><td><code>UNIT_KIND_HERTZ</code></td><td>The hertz unit.</td></tr>
<tr><td><code>UNIT_KIND_ITEM</code></td><td>A pseudo-unit representing a
single \'thing\'. (This is in fact defined in the
SBML specification.)</td></tr>
<tr><td><code>UNIT_KIND_JOULE</code></td><td>The joule unit.</td></tr>
<tr><td><code>UNIT_KIND_KATAL</code></td><td>The katal unit.</td></tr>
<tr><td><code>UNIT_KIND_KELVIN</code></td><td>The kelvin unit.</td></tr>
<tr><td><code>UNIT_KIND_KILOGRAM</code></td><td>The kilogram unit.</td></tr>
<tr><td><code>UNIT_KIND_LITER</code></td><td>Alternate spelling of litre.</td></tr>
<tr><td><code>UNIT_KIND_LITRE</code></td><td>The litre unit.</td></tr>
<tr><td><code>UNIT_KIND_LUMEN</code></td><td>The lumen unit.</td></tr>
<tr><td><code>UNIT_KIND_LUX</code></td><td>The lux unit.</td></tr>
<tr><td><code>UNIT_KIND_METER</code></td><td>Alternate spelling of metre.</td></tr>
<tr><td><code>UNIT_KIND_METRE</code></td><td>The metre unit.</td></tr>
<tr><td><code>UNIT_KIND_MOLE</code></td><td>The mole unit.</td></tr>
<tr><td><code>UNIT_KIND_NEWTON</code></td><td>The newton unit.</td></tr>
<tr><td><code>UNIT_KIND_OHM</code></td><td>The ohm unit.</td></tr>
<tr><td><code>UNIT_KIND_PASCAL</code></td><td>The pascal unit.</td></tr>
<tr><td><code>UNIT_KIND_RADIAN</code></td><td>The radian unit.</td></tr>
<tr><td><code>UNIT_KIND_SECOND</code></td><td>The second unit.</td></tr>
<tr><td><code>UNIT_KIND_SIEMENS</code></td><td>The siemens unit.</td></tr>
<tr><td><code>UNIT_KIND_SIEVERT</code></td><td>The sievert unit.</td></tr>
<tr><td><code>UNIT_KIND_STERADIAN</code></td><td>The steradian unit.</td></tr>
<tr><td><code>UNIT_KIND_TESLA</code></td><td>The tesla unit.</td></tr>
<tr><td><code>UNIT_KIND_VOLT</code></td><td>The volt unit.</td></tr>
<tr><td><code>UNIT_KIND_WATT</code></td><td>The watt unit.</td></tr>
<tr><td><code>UNIT_KIND_WEBER</code></td><td>The weber unit.</td></tr>
<tr><td><code>UNIT_KIND_INVALID</code></td><td>Marker used by libSBML
to indicate an invalid or unset unit.</td></tr>
</table>
</center>

 * <p>
 * <!-- leave this next break as-is to work around some doxygen bug -->
 */
"

%typemap(javaimports) ListOfUnits "
/** 
 * LibSBML implementation of SBML\'s ListOfUnits construct.
 * <p>
 * The various ListOf___ classes in SBML are merely containers used for
 * organizing the main components of an SBML model.  All are derived from
 * the abstract class {@link SBase}, and inherit the various attributes and
 * subelements of {@link SBase}, such as \'metaid\' as and \'annotation\'.  The
 * ListOf___ classes do not add any attributes of their own.
 * <p>
 * {@link ListOfUnits} is entirely contained within {@link UnitDefinition}.
 */
"

%javamethodmodifiers Unit::Unit(unsigned int level, unsigned int version) "
  /**
   * Creates a new {@link Unit} using the given SBML <code>level</code> and <code>version</code>
   * values.
   * <p>
   * @param level a long integer, the SBML Level to assign to this {@link Unit}
   * <p>
   * @param version a long integer, the SBML Version to assign to this
   * {@link Unit}
   * <p>
   * @note Once a {@link Unit} has been added to an {@link SBMLDocument}, the <code>level</code>,
   * <code>version</code> for the document <em>override</em> those used
   * to create the {@link Unit}.  Despite this, the ability to supply the values
   * at creation time is an important aid to creating valid SBML.  Knowledge of
   * the intented SBML Level and Version determine whether it is valid to
   * assign a particular value to an attribute, or whether it is valid to add
   * an object to an existing {@link SBMLDocument}.
   */
 public";

%javamethodmodifiers Unit::Unit(SBMLNamespaces* sbmlns) "
  /**
   * Creates a new {@link Unit} using the given {@link SBMLNamespaces} object
   * <code>sbmlns</code>.
   * <p>
   * The {@link SBMLNamespaces} object encapsulates SBML Level/Version/namespaces
   * information.  It is used to communicate the SBML Level, Version, and
   * (in Level&nbsp;3) packages used in addition to SBML Level&nbsp; Core.
   * A common approach to using this class constructor is to create an
   * {@link SBMLNamespaces} object somewhere in a program, once, then pass it to
   * object constructors such as this one when needed.
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object.
   * <p>
   * @note Once a {@link Unit} has been added to an {@link SBMLDocument}, the <code>level</code>,
   * <code>version</code> and <code>xmlns</code> namespaces for the document <em>override</em> those used
   * to create the {@link Unit}.  Despite this, the ability to supply the values
   * at creation time is an important aid to creating valid SBML.  Knowledge of
   * the intented SBML Level and Version determine whether it is valid to
   * assign a particular value to an attribute, or whether it is valid to add
   * an object to an existing {@link SBMLDocument}.
   */
 public";

%javamethodmodifiers Unit::Unit(const Unit& orig) "
  /**
  * Copy constructor; creates a copy of this {@link Unit}.
  */
 public";

%javamethodmodifiers Unit::accept(SBMLVisitor& v) const "
  /**
   * Accepts the given {@link SBMLVisitor} for this instance of {@link Unit}.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used.
   * <p>
   * @return the result of calling <code>v.visit()</code>, which indicates
   * whether the Visitor would like to visit the next {@link Unit} in the list
   * of units within which this {@link Unit} is embedded (i.e., in the {@link ListOfUnits}
   * located in the enclosing {@link UnitDefinition} instance).
   */
 public";

%javamethodmodifiers Unit::clone() const "
  /**
   * Creates and returns a deep copy of this {@link Unit}.
   * <p>
   * @return a (deep) copy of this {@link Unit}.
   */
 public";

%javamethodmodifiers Unit::initDefaults "
  /**
   * Initializes the attributes of this {@link Unit} (except for \'kind\') to their
   * defaults values.
   * <p>
   * The default values are as follows:
   * <p>
   * <li> exponent   = 1
   * <li> scale      = 0
   * <li> multiplier = 1.0
   * <p>
   * The \'kind\' attribute is left unchanged.
   */
 public";

%javamethodmodifiers Unit::getKind() const "
  /**
   * Returns the \'kind\' of {@link Unit} this is.
   * <p>
   * @return the value of the \'kind\' attribute of this {@link Unit} as a value from
   * the <a class=\'el\' href=\'#UnitKind_t\'>UnitKind_t</a> enumeration
   */
 public";

%javamethodmodifiers Unit::getExponent() const "
  /**
   * Returns the value of the \'exponent\' attribute of this unit.
   * <p>
   * @return the \'exponent\' value of this {@link Unit}, as an integer
   */
 public";

%javamethodmodifiers Unit::getScale() const "
  /**
   * Returns the value of the \'scale\' attribute of this unit.
   * <p>
   * @return the \'scale\' value of this {@link Unit}, as an integer.
   */
 public";

%javamethodmodifiers Unit::getMultiplier() const "
  /**
   * Returns the value of the \'multiplier\' attribute of this {@link Unit}.
   * <p>
   * @return the \'multiplier\' value of this {@link Unit}, as a double
   */
 public";

%javamethodmodifiers Unit::getOffset() const "
  /**
   * Returns the value of the \'offset\' attribute of this {@link Unit}.
   * <p>
   * @warning The \'offset\' attribute is only available in SBML Level&nbsp;2
   * Version&nbsp;1.  This attribute is not present in SBML Level&nbsp;2
   * Version&nbsp;2 or above.  When producing SBML models using these later
   * specifications, modelers and software tools need to account for units
   * with offsets explicitly.  The SBML specification document offers a
   * number of suggestions for how to achieve this.  LibSBML methods such
   * as this one related to \'offset\' are retained for compatibility with
   * earlier versions of SBML Level&nbsp;2, but their use is strongly
   * discouraged.
   * <p>
   * @return the \'offset\' value of this {@link Unit}, as a double
   */
 public";

%javamethodmodifiers Unit::isAmpere() const "
  /**
   * Predicate for testing whether this {@link Unit} is of the kind <code>ampere</code>.
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>ampere</code>, <code>false</code>
   * otherwise. 
   */
 public";

%javamethodmodifiers Unit::isBecquerel() const "
  /**
   * Predicate for testing whether this {@link Unit} is of the kind <code>becquerel</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>becquerel</code>, <code>false</code>
   * otherwise. 
   */
 public";

%javamethodmodifiers Unit::isCandela() const "
  /**
   * Predicate for testing whether this {@link Unit} is of the kind <code>candela</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>candela</code>, <code>false</code>
   * otherwise. 
   */
 public";

%javamethodmodifiers Unit::isCelsius() const "
  /**
   * Predicate for testing whether this {@link Unit} is of the kind <code>Celsius</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>Celsius</code>, <code>false</code>
   * otherwise. 
   * <p>
   * @warning The predefined unit <code>Celsius</code> was removed from the list of
   * predefined units in SBML Level&nbsp;2 Version&nbsp;2 at the same time
   * that the \'offset\' attribute was removed from {@link Unit} definitions.
   * LibSBML methods such as this one related to <code>Celsius</code> are retained in
   * order to support SBML Level&nbsp;2 Version&nbsp;1, but their use is
   * strongly discouraged.
   */
 public";

%javamethodmodifiers Unit::isCoulomb() const "
  /**
   * Predicate for testing whether this {@link Unit} is of the kind <code>coulomb</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>coulomb</code>, <code>false</code>
   * otherwise. 
   */
 public";

%javamethodmodifiers Unit::isDimensionless() const "
  /**
   * Predicate for testing whether this {@link Unit} is of the kind 
   * <code>dimensionless</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>dimensionless</code>, <code>false</code>
   * <p>
   * otherwise.
   */
 public";

%javamethodmodifiers Unit::isFarad() const "
  /**
   * Predicate for testing whether this {@link Unit} is of the kind <code>farad</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>farad</code>, <code>false</code>
   * otherwise. 
   */
 public";

%javamethodmodifiers Unit::isGram() const "
  /**
   * Predicate for testing whether this {@link Unit} is of the kind <code>gram</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>gram</code>, <code>false</code>
   * otherwise. 
   */
 public";

%javamethodmodifiers Unit::isGray() const "
  /**
   * Predicate for testing whether this {@link Unit} is of the kind <code>gray</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>gray</code>, <code>false</code>
   * otherwise. 
   */
 public";

%javamethodmodifiers Unit::isHenry() const "
  /**
   * Predicate for testing whether this {@link Unit} is of the kind <code>henry</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>henry</code>, <code>false</code>
   * otherwise. 
   */
 public";

%javamethodmodifiers Unit::isHertz() const "
  /**
   * Predicate for testing whether this {@link Unit} is of the kind <code>hertz</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>hertz</code>, <code>false</code>
   * otherwise. 
   */
 public";

%javamethodmodifiers Unit::isItem() const "
  /**
   * Predicate for testing whether this {@link Unit} is of the kind <code>item</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>item</code>, <code>false</code>
   * otherwise. 
   */
 public";

%javamethodmodifiers Unit::isJoule() const "
  /**
   * Predicate for testing whether this {@link Unit} is of the kind <code>joule</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>joule</code>, <code>false</code>
   * otherwise. 
   */
 public";

%javamethodmodifiers Unit::isKatal() const "
  /**
   * Predicate for testing whether this {@link Unit} is of the kind <code>katal</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>katal</code>, <code>false</code>
   * otherwise. 
   */
 public";

%javamethodmodifiers Unit::isKelvin() const "
  /**
   * Predicate for testing whether this {@link Unit} is of the kind <code>kelvin</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>kelvin</code>, <code>false</code>
   * otherwise. 
   */
 public";

%javamethodmodifiers Unit::isKilogram() const "
  /**
   * Predicate for testing whether this {@link Unit} is of the kind <code>kilogram</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>kilogram</code>, <code>false</code>
   * otherwise. 
   */
 public";

%javamethodmodifiers Unit::isLitre() const "
  /**
   * Predicate for testing whether this {@link Unit} is of the kind <code>litre</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>litre</code> or \'liter\', 
   * <code>false</code> 
   * otherwise.
   */
 public";

%javamethodmodifiers Unit::isLumen() const "
  /**
   * Predicate for testing whether this {@link Unit} is of the kind <code>lumen</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>lumen</code>, <code>false</code>
   * otherwise. 
   */
 public";

%javamethodmodifiers Unit::isLux() const "
  /**
   * Predicate for testing whether this {@link Unit} is of the kind <code>lux</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>lux</code>, <code>false</code>
   * otherwise. 
   */
 public";

%javamethodmodifiers Unit::isMetre() const "
  /**
   * Predicate for testing whether this {@link Unit} is of the kind <code>metre</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>metre</code> or \'meter\', 
   * <code>false</code> 
   * otherwise.
   */
 public";

%javamethodmodifiers Unit::isMole() const "
  /**
   * Predicate for testing whether this {@link Unit} is of the kind <code>mole</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>mole</code>, <code>false</code>
   * otherwise. 
   */
 public";

%javamethodmodifiers Unit::isNewton() const "
  /**
   * Predicate for testing whether this {@link Unit} is of the kind <code>newton</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>newton</code>, <code>false</code>
   * otherwise. 
   */
 public";

%javamethodmodifiers Unit::isOhm() const "
  /**
   * Predicate for testing whether this {@link Unit} is of the kind <code>ohm</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>ohm</code>, <code>false</code>
   * otherwise. 
   */
 public";

%javamethodmodifiers Unit::isPascal() const "
  /**
   * Predicate for testing whether this {@link Unit} is of the kind <code>pascal</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>pascal</code>, <code>false</code>
   * otherwise. 
   */
 public";

%javamethodmodifiers Unit::isRadian() const "
  /**
   * Predicate for testing whether this {@link Unit} is of the kind <code>radian</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>radian</code>, <code>false</code>
   * otherwise. 
   */
 public";

%javamethodmodifiers Unit::isSecond() const "
  /**
   * Predicate for testing whether this {@link Unit} is of the kind <code>second</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>second</code>, <code>false</code>
   * otherwise. 
   */
 public";

%javamethodmodifiers Unit::isSiemens() const "
  /**
   * Predicate for testing whether this {@link Unit} is of the kind <code>siemens</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>siemens</code>, <code>false</code>
   * otherwise. 
   */
 public";

%javamethodmodifiers Unit::isSievert() const "
  /**
   * Predicate for testing whether this {@link Unit} is of the kind <code>sievert</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>sievert</code>, <code>false</code>
   * otherwise. 
   */
 public";

%javamethodmodifiers Unit::isSteradian() const "
  /**
   * Predicate for testing whether this {@link Unit} is of the kind <code>steradian</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>steradian</code>, <code>false</code>
   * otherwise. 
   */
 public";

%javamethodmodifiers Unit::isTesla() const "
  /**
   * Predicate for testing whether this {@link Unit} is of the kind <code>tesla</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>tesla</code>, <code>false</code>
   * otherwise. 
   */
 public";

%javamethodmodifiers Unit::isVolt() const "
  /**
   * Predicate for testing whether this {@link Unit} is of the kind <code>volt</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>volt</code>, <code>false</code>
   * otherwise. 
   */
 public";

%javamethodmodifiers Unit::isWatt() const "
  /**
   * Predicate for testing whether this {@link Unit} is of the kind <code>watt</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>watt</code>, <code>false</code>
   * otherwise. 
   */
 public";

%javamethodmodifiers Unit::isWeber() const "
  /**
   * Predicate for testing whether this {@link Unit} is of the kind <code>weber</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>weber</code>, <code>false</code>
   * otherwise. 
   */
 public";

%javamethodmodifiers Unit::isSetKind() const "
  /**
   * Predicate to test whether the \'kind\' attribute of this {@link Unit} has been set.
   * <p>
   * @return <code>true</code> if the \'kind\' attribute of this {@link Unit} has been set, 
   * <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers Unit::setKind(UnitKind_t kind) "
  /**
   * Sets the \'kind\' attribute value of this {@link Unit}.
   * <p>
   * @param kind a value from the <a class=\'el\'
   * href=\'#UnitKind_t\'>UnitKind_t</a> enumeration
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   */
 public";

%javamethodmodifiers Unit::setExponent(int value) "
  /**
   * Sets the \'exponent\' attribute value of this {@link Unit}.
   * <p>
   * @param value the integer to which the attribute \'exponent\' should be set
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   */
 public";

%javamethodmodifiers Unit::setScale(int value) "
  /**
   * Sets the \'scale\' attribute value of this {@link Unit}.
   * <p>
   * @param value the integer to which the attribute \'scale\' should be set
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   */
 public";

%javamethodmodifiers Unit::setMultiplier(double value) "
  /**
   * Sets the \'multipler\' attribute value of this {@link Unit}.
   * <p>
   * @param value the floating-point value to which the attribute
   * \'multiplier\' should be set
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_UNEXPECTED_ATTRIBUTE
   */
 public";

%javamethodmodifiers Unit::setOffset(double value) "
  /**
   * Sets the \'offset\' attribute value of this {@link Unit}.
   * <p>
   * @param value the float-point value to which the attribute \'offset\'
   * should set
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_UNEXPECTED_ATTRIBUTE
   * <p>
   * @warning The \'offset\' attribute is only available in SBML Level&nbsp;2
   * Version&nbsp;1.  This attribute is not present in SBML Level&nbsp;2
   * Version&nbsp;2 or above.  When producing SBML models using these later
   * specifications, modelers and software tools need to account for units
   * with offsets explicitly.  The SBML specification document offers a
   * number of suggestions for how to achieve this.  LibSBML methods such
   * as this one related to \'offset\' are retained for compatibility with
   * earlier versions of SBML Level&nbsp;2, but their use is strongly
   * discouraged.
   */
 public";

%javamethodmodifiers Unit::getTypeCode() const "
  /**
   * Returns the libSBML type code of this object instance.
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for this object, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers Unit::getElementName() const "
  /**
   * Returns the XML element name of this object, which for {@link Unit}, is
   * always <code>\'unit\'</code>.
   * <p>
   * @return the name of this element, i.e., <code>\'unit\'</code>. 
   */
 public";

%javamethodmodifiers Unit::isBuiltIn(const std::string& name, unsigned int level) "
  /**
   * Predicate to test whether a given string is the name of a
   * predefined SBML unit.
   * <p>
   * @param name a string to be tested against the predefined unit names
   * <p>
   * @param level the Level of SBML for which the determination should be
   * made.  This is necessary because there are a few small differences
   * in allowed units between SBML Level&nbsp;1 and Level&nbsp;2.
   * <p>
   * @return <code>true</code> if <code>name</code> is one of the five SBML predefined unit
   * identifiers (<code>\'substance\'</code>, <code>\'volume\'</code>, <code>\'area\'</code>, <code>\'length\'</code> or 
   * <code>\'time\'</code>), <code>false</code> otherwise.
   * <p>
   * @note The predefined unit identifiers <code>\'length\'</code> and <code>\'area\'</code> were
   * added in Level&nbsp;2 Version&nbsp;1
   */
 public";

%javamethodmodifiers Unit::isUnitKind(const std::string& name, unsigned int level, unsigned int version) "
  /**
   * Predicate to test whether a given string is the name of a valid
   * base unit in SBML (such as <code>\'gram\'</code> or <code>\'mole\'</code>).
   * <p>
   * This method exists because prior to SBML Level&nbsp;2 Version&nbsp;3,
   * an enumeration called <code>UnitKind</code> was defined by SBML.  This enumeration
   * was removed in SBML Level&nbsp;2 Version&nbsp;3 and its values were
   * folded into the space of values of a type called <code>UnitSId</code>.  This method
   * therefore has less significance in SBML Level&nbsp;2 Version&nbsp;3
   * and Level&nbsp;2 Version&nbsp;4, but remains for backward
   * compatibility and support for reading models in older Versions of
   * Level&nbsp;2.
   * <p>
   * @param name a string to be tested
   * <p>
   * @param level a long integer representing the SBML specification
   * Level 
   * <p>
   * @param version a long integer representing the SBML specification
   * Version
   * <p>
   * @return <code>true</code> if name is a valid UnitKind, <code>false</code> otherwise
   * <p>
   * @note The allowed unit names differ between SBML Levels&nbsp;1
   * and&nbsp;2 and again slightly between Level&nbsp;2 Versions&nbsp;1
   * and&nbsp;2.
   */
 public";

%javamethodmodifiers Unit::areIdentical(Unit * unit1, Unit * unit2) "
  /** 
  * Predicate returning <code>true</code> or <code>false</code> depending on whether two
  * {@link Unit} objects are identical.
  * <p>
  * Two {@link Unit} objects are considered to be <em>identical</em> if they match in
  * all attributes.  (Contrast this to the method
  * {@link Unit#areEquivalent(Unit  unit1, Unit  unit2)}, which compares
  * {@link Unit} objects only with respect to certain attributes.)
  * <p>
  * @param unit1 the first {@link Unit} object to compare
  * @param unit2 the second {@link Unit} object to compare
  * <p>
  * @return <code>true</code> if all the attributes of unit1 are identical
  * to the attributes of unit2, <code>false</code> otherwise.
  * <p>
  * @see #areEquivalent(Unit  unit1, Unit  unit2)
  */
 public";

%javamethodmodifiers Unit::areEquivalent(Unit * unit1, Unit * unit2) "
  /** 
  * Predicate returning <code>true</code> or <code>false</code> depending on whether 
  * {@link Unit} objects are equivalent.
  * <p>
  * Two {@link Unit} objects are considered to be <em>equivalent</em> if their \'kind\'
  * and \'exponent\' attributes are equal.  (Contrast this to the method
  * {@link Unit#areIdentical(Unit  unit1, Unit  unit2)}, which compares {@link Unit}
  * objects with respect to all attributes, not just the kind and
  * exponent.)
  * <p>
  * @param unit1 the first {@link Unit} object to compare
  * @param unit2 the second {@link Unit} object to compare
  * <p>
  * @return <code>true</code> if the \'kind\' and \'exponent\' attributes of unit1 are
  * identical to the kind and exponent attributes of unit2, <code>false</code>
  * otherwise.
  * <p>
  * @see #areIdentical(Unit  unit1, Unit  unit2)
  */
 public";

%javamethodmodifiers Unit::removeScale(Unit * unit) "
  /** 
  * Manipulates the attributes of the {@link Unit} to express the unit with the 
  * value of the scale attribute reduced to zero.
  * <p>
  * For example, 1 millimetre can be expressed as a {@link Unit} with kind=
  * <code>\'metre\'</code> multiplier=<code>\'1\'</code> scale=<code>\'-3\'</code> exponent=<code>\'1\'</code>. It can also be
  * expressed as a {@link Unit} with kind=<code>\'metre\'</code>
  * multiplier=<code>\'0.001\'</code> scale=<code>\'0\'</code> exponent=<code>\'1\'</code>.
  * <p>
  * @param unit the {@link Unit} object to manipulate.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
  */
 public";

%javamethodmodifiers Unit::merge(Unit * unit1, Unit * unit2) "
  /** 
  * Merges two {@link Unit} objects with the same \'kind\' attribute value into a
  * single {@link Unit}.
  * <p>
  * For example, the following,
  * <div class=\'fragment\'><pre>
  * &lt;unit kind=\'metre\' exponent=\'2\'/&gt;
  * &lt;unit kind=\'metre\' exponent=\'1\'/&gt;</pre></div>
  * would be merged to become
  * <div class=\'fragment\'><pre>
  * &lt;unit kind=\'metre\' exponent=\'3\'/&gt;</pre></div>
  * <p>
  * @param unit1 the first {@link Unit} object; the result of the operation is
  * left as a new version of this unit, modified in-place.
  * <p>
  * @param unit2 the second {@link Unit} object to merge with the first
  */
 public";

%javamethodmodifiers Unit::convertToSI(const Unit * unit) "
  /**
  * Returns a {@link UnitDefinition} object which contains the argument {@link Unit}
  * converted to the appropriate SI unit.
  * <p>
  * @param unit the {@link Unit} object to convert to SI
  * <p>
  * @return a {@link UnitDefinition} object containing the SI unit.
  */
 public";

%javamethodmodifiers Unit::hasRequiredAttributes() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link Unit} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link Unit} object are:
   * kind
   * <p>
   * @return a boolean value indicating whether all the required
   * elements for this object have been defined.
   */
 public";

%javamethodmodifiers Unit::Unit "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link Unit} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link Unit} object are:
   * kind
   * <p>
   * @return a boolean value indicating whether all the required
   * elements for this object have been defined.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Unit::readAttributes(const XMLAttributes& attributes) "
  /**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Unit::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Unit::isL1UnitKind(const std::string& name) "
  /**
   * Predicate to test whether a given string is the name of a valid
   * base unit in SBML Level 1 (such as <code>\'gram\'</code> or <code>\'mole\'</code>)
   * <p>
   * @param name a string to be tested
   * <p>
   * @return <code>true</code> if name is a valid UnitKind, <code>false</code> otherwise
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Unit::isL2V1UnitKind(const std::string& name) "
  /**
   * Predicate to test whether a given string is the name of a valid base
   * unit in SBML Level&nbsp;2 Version&nbsp;1 (such as <code>\'gram\'</code> or 
   * <code>\'mole\'</code>)
   * <p>
   * @param name a string to be tested
   * <p>
   * @return <code>true</code> if name is a valid UnitKind, <code>false</code> otherwise
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Unit::isL2UnitKind(const std::string& name) "
  /**
   * Predicate to test whether a given string is the name of a valid base
   * unit in SBML Level&nbsp;2 Version&nbsp;2, 3 or 4 (such as <code>\'gram\'</code> or 
   * <code>\'mole\'</code>)
   * <p>
   * @param name a string to be tested
   * <p>
   * @return <code>true</code> if name is a valid UnitKind, <code>false</code> otherwise
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Unit::isL2UnitKind(const std::string& name) "
  /**
   * Predicate to test whether a given string is the name of a valid base
   * unit in SBML Level&nbsp;2 Version&nbsp;2, 3 or 4 (such as <code>\'gram\'</code> or 
   * <code>\'mole\'</code>)
   * <p>
   * @param name a string to be tested
   * <p>
   * @return <code>true</code> if name is a valid UnitKind, <code>false</code> otherwise
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Unit::isL2UnitKind(const std::string& name) "
  /**
   * Predicate to test whether a given string is the name of a valid base
   * unit in SBML Level&nbsp;2 Version&nbsp;2, 3 or 4 (such as <code>\'gram\'</code> or 
   * <code>\'mole\'</code>)
   * <p>
   * @param name a string to be tested
   * <p>
   * @return <code>true</code> if name is a valid UnitKind, <code>false</code> otherwise
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Unit::isL2UnitKind(const std::string& name) "
  /**
   * Predicate to test whether a given string is the name of a valid base
   * unit in SBML Level&nbsp;2 Version&nbsp;2, 3 or 4 (such as <code>\'gram\'</code> or 
   * <code>\'mole\'</code>)
   * <p>
   * @param name a string to be tested
   * <p>
   * @return <code>true</code> if name is a valid UnitKind, <code>false</code> otherwise
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Unit::isL2UnitKind(const std::string& name) "
  /**
   * Predicate to test whether a given string is the name of a valid base
   * unit in SBML Level&nbsp;2 Version&nbsp;2, 3 or 4 (such as <code>\'gram\'</code> or 
   * <code>\'mole\'</code>)
   * <p>
   * @param name a string to be tested
   * <p>
   * @return <code>true</code> if name is a valid UnitKind, <code>false</code> otherwise
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Unit::isL2UnitKind(const std::string& name) "
  /**
   * Predicate to test whether a given string is the name of a valid base
   * unit in SBML Level&nbsp;2 Version&nbsp;2, 3 or 4 (such as <code>\'gram\'</code> or 
   * <code>\'mole\'</code>)
   * <p>
   * @param name a string to be tested
   * <p>
   * @return <code>true</code> if name is a valid UnitKind, <code>false</code> otherwise
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Unit::isL2UnitKind(const std::string& name) const "
  /**
   * Predicate to test whether a given string is the name of a valid base
   * unit in SBML Level&nbsp;2 Version&nbsp;2, 3 or 4 (such as <code>\'gram\'</code> or 
   * <code>\'mole\'</code>)
   * <p>
   * @param name a string to be tested
   * <p>
   * @return <code>true</code> if name is a valid UnitKind, <code>false</code> otherwise
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Unit::isL2UnitKind(const std::string& name) const "
  /**
   * Predicate to test whether a given string is the name of a valid base
   * unit in SBML Level&nbsp;2 Version&nbsp;2, 3 or 4 (such as <code>\'gram\'</code> or 
   * <code>\'mole\'</code>)
   * <p>
   * @param name a string to be tested
   * <p>
   * @return <code>true</code> if name is a valid UnitKind, <code>false</code> otherwise
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Unit::isL2UnitKind(const std::string& name) const "
  /**
   * Predicate to test whether a given string is the name of a valid base
   * unit in SBML Level&nbsp;2 Version&nbsp;2, 3 or 4 (such as <code>\'gram\'</code> or 
   * <code>\'mole\'</code>)
   * <p>
   * @param name a string to be tested
   * <p>
   * @return <code>true</code> if name is a valid UnitKind, <code>false</code> otherwise
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Unit::isL2UnitKind(const std::string& name) const "
  /**
   * Predicate to test whether a given string is the name of a valid base
   * unit in SBML Level&nbsp;2 Version&nbsp;2, 3 or 4 (such as <code>\'gram\'</code> or 
   * <code>\'mole\'</code>)
   * <p>
   * @param name a string to be tested
   * <p>
   * @return <code>true</code> if name is a valid UnitKind, <code>false</code> otherwise
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Unit::isL2UnitKind(const std::string& name) const "
  /**
   * Predicate to test whether a given string is the name of a valid base
   * unit in SBML Level&nbsp;2 Version&nbsp;2, 3 or 4 (such as <code>\'gram\'</code> or 
   * <code>\'mole\'</code>)
   * <p>
   * @param name a string to be tested
   * <p>
   * @return <code>true</code> if name is a valid UnitKind, <code>false</code> otherwise
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Unit::isL2UnitKind(const std::string& name) const "
  /**
   * Predicate to test whether a given string is the name of a valid base
   * unit in SBML Level&nbsp;2 Version&nbsp;2, 3 or 4 (such as <code>\'gram\'</code> or 
   * <code>\'mole\'</code>)
   * <p>
   * @param name a string to be tested
   * <p>
   * @return <code>true</code> if name is a valid UnitKind, <code>false</code> otherwise
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Unit::isL2UnitKind(const std::string& name) const "
  /**
   * Predicate to test whether a given string is the name of a valid base
   * unit in SBML Level&nbsp;2 Version&nbsp;2, 3 or 4 (such as <code>\'gram\'</code> or 
   * <code>\'mole\'</code>)
   * <p>
   * @param name a string to be tested
   * <p>
   * @return <code>true</code> if name is a valid UnitKind, <code>false</code> otherwise
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Unit::isL2UnitKind(const std::string& name) const "
  /**
   * Predicate to test whether a given string is the name of a valid base
   * unit in SBML Level&nbsp;2 Version&nbsp;2, 3 or 4 (such as <code>\'gram\'</code> or 
   * <code>\'mole\'</code>)
   * <p>
   * @param name a string to be tested
   * <p>
   * @return <code>true</code> if name is a valid UnitKind, <code>false</code> otherwise
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Unit::isL2UnitKind(const std::string& name) const "
  /**
   * Predicate to test whether a given string is the name of a valid base
   * unit in SBML Level&nbsp;2 Version&nbsp;2, 3 or 4 (such as <code>\'gram\'</code> or 
   * <code>\'mole\'</code>)
   * <p>
   * @param name a string to be tested
   * <p>
   * @return <code>true</code> if name is a valid UnitKind, <code>false</code> otherwise
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Unit::isL2UnitKind(const std::string& name) const "
  /**
   * Predicate to test whether a given string is the name of a valid base
   * unit in SBML Level&nbsp;2 Version&nbsp;2, 3 or 4 (such as <code>\'gram\'</code> or 
   * <code>\'mole\'</code>)
   * <p>
   * @param name a string to be tested
   * <p>
   * @return <code>true</code> if name is a valid UnitKind, <code>false</code> otherwise
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Unit::isL2UnitKind(const std::string& name) const "
  /**
   * Predicate to test whether a given string is the name of a valid base
   * unit in SBML Level&nbsp;2 Version&nbsp;2, 3 or 4 (such as <code>\'gram\'</code> or 
   * <code>\'mole\'</code>)
   * <p>
   * @param name a string to be tested
   * <p>
   * @return <code>true</code> if name is a valid UnitKind, <code>false</code> otherwise
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Unit::isL2UnitKind(const std::string& name) const "
  /**
   * Predicate to test whether a given string is the name of a valid base
   * unit in SBML Level&nbsp;2 Version&nbsp;2, 3 or 4 (such as <code>\'gram\'</code> or 
   * <code>\'mole\'</code>)
   * <p>
   * @param name a string to be tested
   * <p>
   * @return <code>true</code> if name is a valid UnitKind, <code>false</code> otherwise
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Unit::isL2UnitKind(const std::string& name) const "
  /**
   * Predicate to test whether a given string is the name of a valid base
   * unit in SBML Level&nbsp;2 Version&nbsp;2, 3 or 4 (such as <code>\'gram\'</code> or 
   * <code>\'mole\'</code>)
   * <p>
   * @param name a string to be tested
   * <p>
   * @return <code>true</code> if name is a valid UnitKind, <code>false</code> otherwise
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Unit::isL2UnitKind(const std::string& name) const "
  /**
   * Predicate to test whether a given string is the name of a valid base
   * unit in SBML Level&nbsp;2 Version&nbsp;2, 3 or 4 (such as <code>\'gram\'</code> or 
   * <code>\'mole\'</code>)
   * <p>
   * @param name a string to be tested
   * <p>
   * @return <code>true</code> if name is a valid UnitKind, <code>false</code> otherwise
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers ListOfUnits::clone() const "
  /**
   * Creates and returns a deep copy of this {@link ListOfUnits}.
   * <p>
   * @return a (deep) copy of this {@link ListOfUnits}.
   */
 public";

%javamethodmodifiers ListOfUnits::getTypeCode() const "
  /**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for this object, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers ListOfUnits::getItemTypeCode() const "
  /**
   * Returns the libSBML type code for the objects contained in this {@link ListOf}
   * (i.e., {@link Unit} objects, if the list is non-empty).
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for the objects contained in this {@link ListOf}
   * instance, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers ListOfUnits::getElementName() const "
  /**
   * Returns the XML element name of this object.
   * <p>
   * For {@link ListOfUnits}, the XML element name is <code>\'listOfUnits\'</code>.
   * <p>
   * @return the name of this element, i.e., <code>\'listOfUnits\'</code>.
   */
 public";

%javamethodmodifiers ListOfUnits::get(unsigned int n) "
  /**
   * Get a {@link Unit} from the {@link ListOfUnits}.
   * <p>
   * @param n the index number of the {@link Unit} to get.
   * <p>
   * @return the nth {@link Unit} in this {@link ListOfUnits}.
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfUnits::get "
  /**
   * Get a {@link Unit} from the {@link ListOfUnits}.
   * <p>
   * @param n the index number of the {@link Unit} to get.
   * <p>
   * @return the nth {@link Unit} in this {@link ListOfUnits}.
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfUnits::remove(unsigned int n) "
  /**
   * Removes the nth item from this {@link ListOfUnits} items and returns a pointer to
   * it.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * <p>
   * @param n the index of the item to remove
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfUnits::getElementPosition() const "
  /**
   * Get the ordinal position of this element in the containing object
   * (which in this case is the {@link Model} object).
   * <p>
   * @return the ordinal position of the element with respect to its
   * siblings, or <code>-1</code> (default) to indicate the position is not significant.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers ListOfUnits::createObject(XMLInputStream& stream) "
  /**
   * Create a {@link ListOfUnits} object corresponding to the next token
   * in the XML input stream.
   * <p>
   * @return the SBML object corresponding to next {@link XMLToken} in the
   * XMLInputStream, or <code>NULL</code> if the token was not recognized.
   <p>
 * @deprecated libSBML internal
 */
 public";

%typemap(javaimports) UnitDefinition "
/** 
 * LibSBML implementation of SBML\'s UnitDefinition construct.
 * <p>
 * Units of measurement may be supplied in a number of contexts in an SBML
 * model.  The units of the following mathematical entities can be
 * specified explicitly: the size of a compartment, the initial amount of a
 * species, and the units of constant and variable parameter values.  The
 * overall units of any mathematical formula appearing in SBML are those
 * that arise naturally from the components and mathematical expressions
 * comprising the formula, or in other words, the units obtained by doing
 * dimensional analysis on the formula.
 * <p>
 * Rather than requiring a complete unit definition on every construct,
 * SBML provides a facility for defining units that can be referenced
 * throughout a model.  In addition, every kind of SBML mathematical
 * entity has units assigned to it from a set of predefined defaults (listed
 * below); by redefining these predefined default units, it is possible to
 * change the units used throughout a model in a simple and consistent
 * manner.
 * <p>
 * The SBML unit definition facility uses two classes of objects,
 * {@link UnitDefinition} and {@link Unit}.  The approach to defining units in SBML is
 * compositional; for example, <em>meter second<sup> &ndash;2</sup></em> is
 * constructed by combining a {@link Unit} object representing <em>meter</em> with
 * another {@link Unit} object representing <em>second<sup> &ndash;2</sup></em>.
 * The combination is wrapped inside a {@link UnitDefinition}, which provides for
 * assigning an identifier and optional name to the combination.  The
 * identifier can then be referenced from elsewhere in a model.  Thus, the
 * {@link UnitDefinition} class is the container, and {@link Unit} instances are placed
 * inside {@link UnitDefinition} instances.
 * <p>
 * <h2>Summary of the {@link UnitDefinition} construct</h2>
 * <p>
 * {@link UnitDefinition} in SBML Level&nbsp;2 has two attributes and one
 * subelement.  The two attributes are \'id\' and \'name\', and the subelement
 * is {@link ListOfUnits}.
 * <p>
 * The required attribute \'id\' and optional attribute \'name\' are both
 * strings.  The \'id\' attribute is used to give the defined unit a unique
 * identifier by which other parts of an SBML model definition can refer to
 * it.  The \'name\' attribute is intended to be used for giving the unit
 * definition an optional human-readable name.
 * <p>
 * There are two important restrictions about the use of unit definition
 * \'id\' values in SBML Level&nbsp;2 Versions&nbsp;3&ndash;4:
 * <ul>
 * <li> The \'id\' of a {@link UnitDefinition} must <em>not</em> contain a value from the
 *   list of reserved base unit names (i.e., the strings <code>gram</code>, <code>liter</code>,
 *   etc.; see the definition of {@link Unit} for the complete list).  This
 *   constraint simply prevents the redefinition of the base units.
 * <p>
 * <li> There is a set of reserved identifiers for the predefined
 *   units in SBML; these identifiers are <code>substance</code>, <code>volume</code>, <code>area</code>,
 *   <code>length</code>, and <code>time</code>.  Using one of these values for the attribute
 *   \'id\' of a {@link UnitDefinition} has the effect of redefining the model-wide
 *   default units for the corresponding quantities.  The list of
 *   predefined units is given in the table below:
 * <p>
 *   <center>
<table border=\'0\' class=\'text-table width80 normal-font alt-row-colors\'>
 <tr>
     <th align=\'left\'>Identifier</th>
     <th align=\'left\'>Possible scalable units</th>
     <th align=\'left\'>Default units</th>
 </tr>
<tr><td><code>substance</code></td><td>mole, item, gram, kilogram, dimensionless</td><td>mole</td></tr>
<tr><td><code>volume</code></td><td>litre, cubic metre, dimensionless</td><td>litre</td></tr>
<tr><td><code>area</code></td><td>square metre, dimensionless</td><td>square metre</td></tr>
<tr><td><code>length</code></td><td>metre, dimensionless</td><td>metre</td></tr>
<tr><td><code>time</code></td><td>second, dimensionless</td><td>second</td></tr>
</table>
</center>

 * <p>
 *   Also, two limitations are imposed on redefining the predefined
 *   unit <code>substance</code>, <code>volume</code>, <code>area</code>, <code>length</code>, and <code>time</code>:
 *   (1) The {@link UnitDefinition} of a predefined SBML unit can only
 *      contain a single {@link Unit} object within it.
 *   (2) The value of the \'kind\' attribute in 
 *      a {@link Unit} instance must be drawn from one of the values
 *      in the second column of the table above.
 * </ul>
 * <p>
 * A {@link UnitDefinition} must contain exactly one {@link ListOfUnits}, and this list
 * must contain one or more {@link Unit} definitions; see the definitions of these
 * other object classes for more information about them.  The following
 * example illustrates a complete unit definition (when written in XML)
 * when they all the pieces are combined together.  This defines \'mmls\'
 * to be millimoles per litre per second.
 * <div class=\'fragment\'><pre>
 * &lt;listOfUnitDefinitions&gt;
 *     &lt;unitDefinition id=\'mmls\'&gt;
 *         &lt;listOfUnits&gt;
 *             &lt;unit kind=\'mole\'   scale=\'-3\'/&gt;
 *             &lt;unit kind=\'litre\'  exponent=\'-1\'/&gt;
 *             &lt;unit kind=\'second\' exponent=\'-1\'/&gt;
 *         &lt;/listOfUnits&gt;
 *     &lt;/unitDefinition&gt;
 * &lt;/listOfUnitDefinitions&gt;</pre></div>
 * <p>
 * <p>
 * <h2>Further comments about SBML\'s unit definition system</h2>
 * <p>
 * The vast majority of modeling situations requiring new SBML unit
 * definitions involve simple multiplicative combinations of base units and
 * factors.  An example of this might be <em>moles per litre per
 * second</em>.  What distinguishes these sorts of simpler unit definitions
 * from more complex ones is that they may be expressed without the use of
 * an additive offset from a zero point.  The use of offsets complicates
 * all unit definition systems, yet in the domain of SBML the real-life
 * cases requiring offsets are few (and in fact, to the best of our
 * knowledge, only involve temperature).  Consequently, the SBML unit
 * system has been consciously designed in a way that attempts to simplify
 * implementation of unit support for the most common cases in systems
 * biology.
 * <p>
 * As of SBML Level&nbsp;2 Version&nbsp;2, {@link Unit} no longer has the
 * attribute called \'offset\' introduced in SBML Level&nbsp;2
 * Version&nbsp;1.  It turned out that the general case involving units
 * with offsets was incorrectly defined, and few (if any) developers even
 * attempted to support offset-based units in their software.  In the
 * development of Level&nbsp;2 Version&nbsp;2, a consensus among SBML
 * developers emerged that a fully generalized unit scheme is <em>so</em>
 * confusing and complicated that it actually <em>impedes</em> interoperability.
 * SBML Level&nbsp;2 Version&nbsp;2, Version&nbsp;3 and Version&nbsp;4 acknowledge this
 * reality by reducing and simplifying the unit system, specifically by
 * removing the \'offset\' attribute on {@link Unit} and <code>Celsius</code> as a pre-defined
 * unit.
 * <p>
 * The following guidelines suggest methods for handling units that do
 * require the use of zero offsets for their definitions:
 * <ul>
 * <li> <em>Handling Celsius</em>.  A model in which certain quantities are
 *   temperatures measured in degrees Celsius can be converted
 *   straightforwardly to a model in which those temperatures are in
 *   kelvin.  A software tool could do this by performing a straightforward
 *   substitution using the following relationship: T<sub> kelvin</sub> =
 *   T<sub> Celsius</sub> + 273.15.  In every mathematical formula of the
 *   model where a quantity (call it <em>x</em>) in degrees Celsius appears,
 *   replace <em>x</em> with x<sub> k</sub>+ 273.15, where x<sub> k</sub> is now
 *   in kelvin.  An alternative approach would be to use a
 *   {@link FunctionDefinition} to define a function encapsulating this
 *   relationship above and then using that in the rest of the model as
 *   needed.  Since Celsius is a commonly-used unit, software tools could
 *   help users by providing users with the ability to express temperatures
 *   in Celsius in the tools\' interfaces, and making substitutions
 *   automatically when writing out the SBML.
 * <p>
 * <li> <em>Other units requiring offsets</em>.  One approach to handling
 *   other kinds of units is to use a {@link FunctionDefinition} to define a function
 *   encapsulating the necessary mathematical relationship, then
 *   substituting a call to this function wherever the original quantity
 *   appeared in the model.  For example, here is a possible definition for
 *   converting Fahrenheit to Celsius degrees:
 *   <div class=\'fragment\'><pre>
 * &lt;functionDefinition id=\'Fahrenheit_to_kelvin\'&gt;
 *     &lt;math xmlns=\'http://www.w3.org/1998/Math/MathML\'&gt;
 *         &lt;lambda&gt;
 *             &lt;bvar&gt;&lt;ci&gt; temp_in_fahrenheit &lt;/ci&gt;&lt;/bvar&gt;
 *             &lt;apply&gt;
 *                 &lt;divide/&gt;
 *                 &lt;apply&gt;
 *                     &lt;plus/&gt;
 *                     &lt;ci&gt; temp_in_fahrenheit &lt;/ci&gt;
 *                     &lt;cn&gt; 459.67 &lt;/cn&gt;
 *                 &lt;/apply&gt;
 *                 &lt;cn&gt; 1.8 &lt;/cn&gt;
 *             &lt;/apply&gt;
 *         &lt;/lambda&gt;
 *     &lt;/math&gt;
 * &lt;/functionDefinition&gt;</pre></div>
 * <p>
 * <li> An alternative approach not requiring the use of function definitions
 *   is to use an {@link AssignmentRule} for each variable in Fahrenheit units.
 *   The {@link AssignmentRule} could compute the conversion from Fahrenheit to
 *   (say) kelvin, assign its value to a variable (in Kelvin units), and
 *   then that variable could be used elsewhere in the model.
 * <p>
 * <li> Still another approach is to rewrite the mathematical formulas of a
 *   model to directly incorporate the conversion formula wherever the
 *   original quantity appeared.
 * </ul>
 * <p>
 * Please consult the SBML specifications for more information about this
 * and other issues involving units.
 * <p>
 * <!-- leave this next break as-is to work around some doxygen bug -->
 */
"

%typemap(javaimports) ListOfUnitDefinitions "
/** 
 * LibSBML implementation of SBML\'s ListOfUnitDefinitions construct.
 * <p>
 * The various ListOf___ classes in SBML are merely containers used for
 * organizing the main components of an SBML model.  All are derived from
 * the abstract class {@link SBase}, and inherit the various attributes and
 * subelements of {@link SBase}, such as \'metaid\' as and \'annotation\'.  The
 * ListOf___ classes do not add any attributes of their own.
 * <p>
 * The relationship between the lists and the rest of an SBML model is
 * illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):
 * <p>
 * <center><img src=\'listof-illustration.jpg\'></center><br>
 * 
 * <p>
 * Readers may wonder about the motivations for using the ListOf___
 * containers.  A simpler approach in XML might be to place the components
 * all directly at the top level of the model definition.  We chose instead
 * to group them within XML elements named after {@link ListOf}<em>Classname</em>,
 * in part because we believe this helps organize the components and makes
 * visual reading of models in XML easier.  More importantly, the fact that
 * the container classes are derived from {@link SBase} means that software tools
 * can add information about the lists themselves into each list
 * container\'s \'annotation\'.
 * <p>
 * @see ListOfFunctionDefinitions
 * @see ListOfUnitDefinitions
 * @see ListOfCompartmentTypes
 * @see ListOfSpeciesTypes
 * @see ListOfCompartments
 * @see ListOfSpecies
 * @see ListOfParameters
 * @see ListOfInitialAssignments
 * @see ListOfRules
 * @see ListOfConstraints
 * @see ListOfReactions
 * @see ListOfEvents
 */
"

%javamethodmodifiers UnitDefinition::UnitDefinition(unsigned int level, unsigned int version) "
  /**
   * Creates a new {@link UnitDefinition} using the given SBML <code>level</code> and <code>version</code>
   * values.
   * <p>
   * @param level a long integer, the SBML Level to assign to this {@link UnitDefinition}
   * <p>
   * @param version a long integer, the SBML Version to assign to this
   * {@link UnitDefinition}
   * <p>
   * @note Once a {@link UnitDefinition} has been added to an {@link SBMLDocument}, the <code>level</code>,
   * <code>version</code> for the document <em>override</em> those used
   * to create the {@link UnitDefinition}.  Despite this, the ability to supply the values
   * at creation time is an important aid to creating valid SBML.  Knowledge of
   * the intented SBML Level and Version determine whether it is valid to
   * assign a particular value to an attribute, or whether it is valid to add
   * an object to an existing {@link SBMLDocument}.
   */
 public";

%javamethodmodifiers UnitDefinition::UnitDefinition(SBMLNamespaces* sbmlns) "
  /**
   * Creates a new {@link UnitDefinition} using the given {@link SBMLNamespaces} object
   * <code>sbmlns</code>.
   * <p>
   * The {@link SBMLNamespaces} object encapsulates SBML Level/Version/namespaces
   * information.  It is used to communicate the SBML Level, Version, and
   * (in Level&nbsp;3) packages used in addition to SBML Level&nbsp; Core.
   * A common approach to using this class constructor is to create an
   * {@link SBMLNamespaces} object somewhere in a program, once, then pass it to
   * object constructors such as this one when needed.
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object.
   * <p>
   * @note Once a {@link UnitDefinition} has been added to an {@link SBMLDocument}, the <code>level</code>,
   * <code>version</code> and <code>xmlns</code> namespaces for the document <em>override</em> those used
   * to create the {@link UnitDefinition}.  Despite this, the ability to supply the values
   * at creation time is an important aid to creating valid SBML.  Knowledge of
   * the intented SBML Level and Version determine whether it is valid to
   * assign a particular value to an attribute, or whether it is valid to add
   * an object to an existing {@link SBMLDocument}.
   */
 public";

%javamethodmodifiers UnitDefinition::UnitDefinition(const UnitDefinition& orig) "
  /**
  * Copy constructor; creates a copy of this {@link UnitDefinition}.
  */
 public";

%javamethodmodifiers UnitDefinition::accept(SBMLVisitor& v) const "
  /**
   * Accepts the given {@link SBMLVisitor} for this instance of {@link UnitDefinition}.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used.
   * <p>
   * @return the result of calling <code>v.visit()</code>, which indicates
   * whether the Visitor would like to visit the next {@link UnitDefinition} in the
   * list of units within which this {@link UnitDefinition} is embedded (i.e., in
   * the {@link ListOfUnitDefinitions} located in the enclosing {@link Model} instance).
   */
 public";

%javamethodmodifiers UnitDefinition::clone() const "
  /**
   * Creates and returns a deep copy of this {@link UnitDefinition}.
   * <p>
   * @return a (deep) copy of this {@link UnitDefinition}.
   */
 public";

%javamethodmodifiers UnitDefinition::getId() const "
  /**
   * Returns the value of the \'id\' attribute of this {@link UnitDefinition}.
   * <p>
   * @return the id of this {@link UnitDefinition}.
   */
 public";

%javamethodmodifiers UnitDefinition::getName() const "
  /**
   * Returns the value of the \'name\' attribute of this {@link UnitDefinition}.
   * <p>
   * @return the name of this {@link UnitDefinition}.
   */
 public";

%javamethodmodifiers UnitDefinition::isSetId() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link UnitDefinition}\'s \'id\' attribute has been set.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the \'id\' attribute of this {@link UnitDefinition} has been
   * set, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers UnitDefinition::isSetName() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link UnitDefinition}\'s \'name\' attribute has been set.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the \'name\' attribute of this {@link UnitDefinition} has been
   * set, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers UnitDefinition::setId(const std::string& sid) "
  /**
   * Sets the value of the \'id\' attribute of this {@link UnitDefinition}.
   * <p>
   * The string <code>sid</code> is copied.  Note that SBML has strict requirements
   * for the syntax of identifiers.  The following is summary of the
   * definition of the SBML identifier type <code>SId</code> (here expressed in an
   * extended form of BNF notation):
   * <div class=\'fragment\'><pre>
   *   letter .= \'a\'..\'z\',\'A\'..\'Z\'
   *   digit  .= \'0\'..\'9\'
   *   idChar .= letter | digit | \'_\'
   *   SId    .= ( letter | \'_\' ) idChar</pre></div>
   * The equality of SBML identifiers is determined by an exact character
   * sequence match; i.e., comparisons must be performed in a
   * case-sensitive manner.  In addition, there are a few conditions for
   * the uniqueness of identifiers in an SBML model.  Please consult the
   * SBML specifications for the exact formulations.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param sid the string to use as the identifier of this {@link UnitDefinition}
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   */
 public";

%javamethodmodifiers UnitDefinition::setName(const std::string& name) "
  /**
   * Sets the value of the \'name\' attribute of this {@link UnitDefinition}.
   * <p>
   * The string in <code>name</code> is copied.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param name the new name for the {@link UnitDefinition}
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   */
 public";

%javamethodmodifiers UnitDefinition::unsetName "
  /**
   * Unsets the value of the \'name\' attribute of this {@link UnitDefinition}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_OPERATION_FAILED
   */
 public";

%javamethodmodifiers UnitDefinition::isVariantOfArea() const "
  /**
   * Convenience function for testing if a given unit definition is a
   * variant of the predefined unit identifier <code>\'area\'</code>.
   * <p>
   * @return <code>true</code> if this {@link UnitDefinition} is a variant of the predefined
   * unit <code>area</code>, meaning square metres with only abritrary variations
   * in scale or multiplier values; <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers UnitDefinition::isVariantOfLength() const "
  /**
   * Convenience function for testing if a given unit definition is a
   * variant of the predefined unit identifier <code>\'length\'</code>.
   * <p>
   * @return <code>true</code> if this {@link UnitDefinition} is a variant of the predefined
   * unit <code>length</code>, meaning metres with only abritrary variations in scale
   * or multiplier values; <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers UnitDefinition::isVariantOfSubstance() const "
  /**
   * Convenience function for testing if a given unit definition is a
   * variant of the predefined unit identifier <code>\'substance\'</code>.
   * <p>
   * @return <code>true</code> if this {@link UnitDefinition} is a variant of the predefined
   * unit <code>substance</code>, meaning moles or items (and grams or kilograms from
   * SBML Level&nbsp;2 Version&nbsp;2 onwards) with only abritrary variations
   * in scale or multiplier values; <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers UnitDefinition::isVariantOfTime() const "
  /**
   * Convenience function for testing if a given unit definition is a
   * variant of the predefined unit identifier <code>\'time\'</code>.
   * <p>
   * @return <code>true</code> if this {@link UnitDefinition} is a variant of the predefined
   * unit <code>time</code>, meaning seconds with only abritrary variations in scale or
   * multiplier values; <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers UnitDefinition::isVariantOfVolume() const "
  /**
   * Convenience function for testing if a given unit definition is a
   * variant of the predefined unit identifier <code>\'volume\'</code>.
   * <p>
   * @return <code>true</code> if this {@link UnitDefinition} is a variant of the predefined
   * unit <code>volume</code>, meaning litre or cubic metre with only abritrary
   * variations in scale or multiplier values; <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers UnitDefinition::isVariantOfDimensionless() const "
  /**
   * Convenience function for testing if a given unit definition is a
   * variant of the unit <code>\'dimensionless\'</code>.
   * <p>
   * @return <code>true</code> if this {@link UnitDefinition} is a variant of 
   * <code>dimensionless</code>, meaning dimensionless with only abritrary variations in
   * scale or multiplier values; <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers UnitDefinition::isVariantOfMass() const "
  /**
   * Convenience function for testing if a given unit definition is a
   * variant of the predefined unit identifier <code>\'mass\'</code>.
   * <p>
   * @return <code>true</code> if this {@link UnitDefinition} is a variant of mass units,
   * meaning gram or kilogram with only abritrary variations in scale or
   * multiplier values; <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers UnitDefinition::isVariantOfSubstancePerTime() const "
  /**
   * Convenience function for testing if a given unit definition is a
   * variant of the predefined unit <code>\'substance\'</code> divided by the predefined
   * unit <code>\'time\'</code>.
   * <p>
   * @return <code>true</code> if this {@link UnitDefinition} is a variant of the predefined
   * unit <code>substance</code> per predefined unit <code>time</code>, meaning it contains two
   * units one of which is a variant of substance and the other is a
   * variant of time which an exponent of -1; <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers UnitDefinition::addUnit(const Unit* u) "
  /**
   * Adds a copy of the given {@link Unit} to this {@link UnitDefinition}.
   * <p>
   * @param u the {@link Unit} instance to add to this {@link UnitDefinition}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_LEVEL_MISMATCH
   * <li> LIBSBML_VERSION_MISMATCH
   * <li> LIBSBML_DUPLICATE_OBJECT_ID
   * <li> LIBSBML_OPERATION_FAILED
   * <p>
   * @note This method should be used with some caution.  The fact that
   * this method <em>copies</em> the object passed to it means that the caller
   * will be left holding a physically different object instance than the
   * one contained in this {@link UnitDefinition}.  Changes made to the original
   * object instance (such as resetting attribute values) will <em>not
   * affect the instance in the {@link UnitDefinition}</em>.  In addition, the
   * caller should make sure to free the original object if it is no longer
   * being used, or else a memory leak will result.  Please see
   * UnitDefinition.createUnit() for a method that does not lead to these
   * issues.
   * <p>
   * @see #createUnit()
   */
 public";

%javamethodmodifiers UnitDefinition::createUnit "
  /**
   * Creates a new and empty {@link Unit}, adds it to this {@link UnitDefinition}\'s list of
   * units, and returns it.
   * <p>
   * @return a newly constructed (and empty) {@link Unit} instance.
   * <p>
   * @note It is worth emphasizing that the attribute \'kind\' value of a
   * {@link Unit} is a required attribute for a valid {@link Unit} definition.  The
   * createUnit() method does not assign a valid kind to the constructed
   * unit (instead, it sets the \'kind\' to <code>UNIT_KIND_INVALID</code>).  Callers
   * are cautioned to set the newly-constructed {@link Unit}\'s kind using
   * Unit.setKind() soon after calling this method.
   * <p>
   * @see #addUnit(Unit  u)
   */
 public";

%javamethodmodifiers UnitDefinition::getListOfUnits() const "
  /**
   * Returns the list of Units for this {@link UnitDefinition} instance.
   * @return the {@link ListOfUnits} value for this {@link UnitDefinition}.
   */
 public";

%javamethodmodifiers UnitDefinition::getListOfUnits "
  /**
   * Returns the list of Units for this {@link UnitDefinition} instance.
   * @return the {@link ListOfUnits} value for this {@link UnitDefinition}.
   */
 public";

%javamethodmodifiers UnitDefinition::getUnit(unsigned int n) "
  /**
   * Returns a specific {@link Unit} instance belonging to this {@link UnitDefinition}.
   * <p>
   * @param n an integer, the index of the {@link Unit} to be returned.
   * <p>
   * @return the nth {@link Unit} of this {@link UnitDefinition}
   * <p>
   * @see #getNumUnits()
   */
 public";

%javamethodmodifiers UnitDefinition::getUnit "
  /**
   * Returns a specific {@link Unit} instance belonging to this {@link UnitDefinition}.
   * <p>
   * @param n an integer, the index of the {@link Unit} to be returned.
   * <p>
   * @return the nth {@link Unit} of this {@link UnitDefinition}
   */
 public";

%javamethodmodifiers UnitDefinition::getNumUnits() const "
  /**
   * Returns the number of {@link Unit} objects contained within this
   * {@link UnitDefinition}.
   * <p>
   * @return an integer representing the number of Units in this
   * {@link UnitDefinition}.
   */
 public";

%javamethodmodifiers UnitDefinition::removeUnit(unsigned int n) "
  /**
   * Removes the nth {@link Unit} object from this {@link UnitDefinition} object and
   * returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * <p>
   * @param n the index of the {@link Unit} object to remove
   * <p>
   * @return the {@link Unit} object removed.  As mentioned above, 
   * the caller owns the returned item. NULL is returned if the given index 
   * is out of range.
   * <p>
   */
 public";

%javamethodmodifiers UnitDefinition::setSBMLDocument(SBMLDocument* d) "
  /**
   * Sets the parent {@link SBMLDocument} of this SBML object.
   * <p>
   * @param d the {@link SBMLDocument} to use
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers UnitDefinition::setParentSBMLObject(SBase* sb) "
  /**
   * Sets the parent SBML object of this SBML object.
   * <p>
   * @param sb the SBML object to use
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers UnitDefinition::getTypeCode() const "
  /**
   * Returns the libSBML type code for this object instance.
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for this object, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers UnitDefinition::getElementName() const "
  /**
   * Returns the XML element name of this object, which for {@link UnitDefinition},
   * is always <code>\'unitDefinition\'</code>.
   * <p>
   * @return the name of this element, i.e., <code>\'unitDefinition\'</code>.
   */
 public";

%javamethodmodifiers UnitDefinition::simplify(UnitDefinition * ud) "
  /** 
  * Simplifies the {@link UnitDefinition} so that any {@link Unit} objects occurring within
  * the {@link ListOfUnits} occurs only once.
  * <p>
  * For example, the following definition,
  * <div class=\'fragment\'><pre>
  * &lt;unitDefinition&gt;
  *  &lt;listOfUnits&gt;
  *    &lt;unit kind=\'metre\' exponent=\'1\'/&gt;
  *    &lt;unit kind=\'metre\' exponent=\'2\'/&gt;
  *  &lt;/listOfUnits&gt;
  * &lt;unitDefinition&gt;</pre></div>
  * will be simplified to 
  * <div class=\'fragment\'><pre>
  * &lt;unitDefinition&gt;
  *   &lt;listOfUnits&gt;
  *     &lt;unit kind=\'metre\' exponent=\'3\'/&gt;
  *   &lt;/listOfUnits&gt;
  * &lt;unitDefinition&gt;</pre></div>
  * <p>
  * @param ud the {@link UnitDefinition} object to be simplified.
  */
 public";

%javamethodmodifiers UnitDefinition::reorder(UnitDefinition * ud) "
  /** 
   * Orders alphabetically the {@link Unit} objects within the {@link ListOfUnits} of a
   * {@link UnitDefinition}.
   * <p>
   * @param ud the {@link UnitDefinition} object whose units are to be reordered.
   */
 public";

%javamethodmodifiers UnitDefinition::convertToSI(const UnitDefinition *) "
  /**
   * Convert a given {@link UnitDefinition} into a new {@link UnitDefinition} object
   * that uses SI units.
   * <p>
   * @param ud the {@link UnitDefinition} object to convert to SI
   * <p>
   * @return a new {@link UnitDefinition} object representing the results of the
   * conversion.
   */
 public";

%javamethodmodifiers UnitDefinition::areIdentical(const UnitDefinition * ud1, const UnitDefinition * ud2) "
  /** 
   * Predicate returning <code>true</code> or <code>false</code> depending on whether two
   * {@link UnitDefinition} objects are identical.
   * <p>
   * For the purposes of performing this comparison, two {@link UnitDefinition}
   * objects are considered identical when they contain identical lists of
   * {@link Unit} objects.  Pairs of {@link Unit} objects in the lists are in turn
   * considered identical if they satisfy the predicate
   * Unit.areIdentical().  The predicate compares every attribute of the
   * {@link Unit} objects.
   * <p>
   * @param ud1 the first {@link UnitDefinition} object to compare
   * @param ud2 the second {@link UnitDefinition} object to compare
   * <p>
   * @return <code>true</code> if all the {@link Unit} objects in ud1 are identical to the
   * {@link Unit} objects of ud2, <code>false</code> otherwise.
   * <p>
   * @see #areEquivalent(UnitDefinition  ud1, UnitDefinition  ud2)
   * @see Unit#areIdentical(Unit  unit1, Unit  unit2)
   */
 public";

%javamethodmodifiers UnitDefinition::areEquivalent(const UnitDefinition *ud1 , const UnitDefinition * ud2) "
  /** 
   * Predicate returning <code>true</code> or <code>false</code> depending on whether two
   * {@link UnitDefinition} objects are equivalent.
   * <p>
   * For the purposes of performing this comparison, two {@link UnitDefinition}
   * objects are considered equivalent when they contain <em>equivalent</em>
   * list of {@link Unit} objects.  {@link Unit} objects are in turn considered equivalent
   * if they satisfy the predicate Unit.areEquivalent().  The predicate
   * tests a subset of the objects\'s attributes.
   * <p>
   * @param ud1 the first {@link UnitDefinition} object to compare
   * <p>
   * @param ud2 the second {@link UnitDefinition} object to compare
   * <p>
   * @return <code>true</code> if all the {@link Unit} objects in ud1 are equivalent
   * to the {@link Unit} objects in ud2, <code>false</code> otherwise.
   * <p>
   * @see #areIdentical(UnitDefinition  ud1, UnitDefinition  ud2)
   * @see Unit#areEquivalent(Unit  unit1, Unit  unit2)
   */
 public";

%javamethodmodifiers UnitDefinition::areIdenticalSIUnits(const UnitDefinition * ud1, const UnitDefinition * ud2) "
  /** 
   * Predicate returning <code>true</code> or <code>false</code> depending on whether two
   * {@link UnitDefinition} objects are equivalent.
   * <p>
   * For the purposes of performing this comparison, two {@link UnitDefinition}
   * objects are considered equivalent when they contain <em>equivalent</em>
   * list of {@link Unit} objects.  {@link Unit} objects are in turn considered equivalent
   * if they satisfy the predicate Unit.areEquivalent().  The predicate
   * tests a subset of the objects\'s attributes.
   * <p>
   * @param ud1 the first {@link UnitDefinition} object to compare
   * <p>
   * @param ud2 the second {@link UnitDefinition} object to compare
   * <p>
   * @return <code>true</code> if all the {@link Unit} objects in ud1 are equivalent
   * to the {@link Unit} objects in ud2, <code>false</code> otherwise.
   * <p>
   * @see #areIdentical(UnitDefinition  ud1, UnitDefinition  ud2)
   * @see Unit#areEquivalent(Unit  unit1, Unit  unit2)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers UnitDefinition::combine(UnitDefinition * ud1, UnitDefinition * ud2) "
  /** 
   * Combines two {@link UnitDefinition} objects into a single {@link UnitDefinition}
   * object which expresses the product of the units of the two
   * {@link UnitDefinition}\'s.
   * <p>
   * @param ud1 the first {@link UnitDefinition} object 
   * @param ud2 the second {@link UnitDefinition} object
   * <p>
   * @return a {@link UnitDefinition} which represents the product of the 
   * units of the two argument UnitDefinitions.
   */
 public";

%javamethodmodifiers UnitDefinition::printUnits(const UnitDefinition * ud, bool compact = false) "
  /** 
   * Returns a string that expresses the unit definition
   * represented by this {@link UnitDefinition} object.
   * <p>
   * For example printUnits applied to
   * <div class=\'fragment\'><pre>
   * &lt;unitDefinition&gt;
   *  &lt;listOfUnits&gt;
   *    &lt;unit kind=\'metre\' exponent=\'1\'/&gt;
   *    &lt;unit kind=\'second\' exponent=\'-2\'/&gt;
   *  &lt;/listOfUnits&gt;
   * &lt;unitDefinition&gt;</pre></div>
   * will return the string <code>\'metre (exponent = 1, multiplier = 1, scale = 0) 
   * second (exponent = -2, multiplier = 1, scale = 0)\'</code> or, if 
   * compact = true, the string <code>\'(1 metre)^1 (1 second)^-2\'</code>
   * This may be useful for printing unit information to
   * human users, or in debugging, or other situations.
   * <p>
   * @param ud the {@link UnitDefinition} object
   * @param compact boolean indicating whether the compact form
   * should be used (defaults to false)
   * <p>
   * @return a string expressing the unit definition
   */
 public";

%javamethodmodifiers UnitDefinition::writeElements(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write out their contained
   * SBML objects as XML elements.  Be sure to call your parents
   * implementation of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers UnitDefinition::hasRequiredAttributes() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link UnitDefinition} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link UnitDefinition} object are:
   * id (name in L1)
   * <p>
   * @return a boolean value indicating whether all the required
   * attributes for this object have been defined.
   */
 public";

%javamethodmodifiers UnitDefinition::hasRequiredElements() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required elements for this {@link UnitDefinition} object
   * have been set.
   * <p>
   * @note The required elements for a {@link Constraint} object are:
   * listOfUnits (L2 only)
   * <p>
   * @return a boolean value indicating whether all the required
   * elements for this object have been defined.
   */
 public";

%javamethodmodifiers UnitDefinition::UnitDefinition "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required elements for this {@link UnitDefinition} object
   * have been set.
   * <p>
   * @note The required elements for a {@link Constraint} object are:
   * listOfUnits (L2 only)
   * <p>
   * @return a boolean value indicating whether all the required
   * elements for this object have been defined.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers UnitDefinition::createObject(XMLInputStream& stream) "
  /**
   * @return the SBML object corresponding to next {@link XMLToken} in the
   * XMLInputStream or NULL if the token was not recognized.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers UnitDefinition::readAttributes(const XMLAttributes& attributes) "
  /**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers UnitDefinition::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers UnitDefinition::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers UnitDefinition::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers UnitDefinition::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers UnitDefinition::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers UnitDefinition::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers UnitDefinition::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers UnitDefinition::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers UnitDefinition::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers UnitDefinition::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers UnitDefinition::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers UnitDefinition::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers UnitDefinition::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers UnitDefinition::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers UnitDefinition::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers UnitDefinition::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers UnitDefinition::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers UnitDefinition::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers ListOfUnitDefinitions::clone() const "
  /**
   * Creates and returns a deep copy of this {@link ListOfUnitDefinitions} instance.
   * <p>
   * @return a (deep) copy of this {@link ListOfUnitDefinitions}.
   */
 public";

%javamethodmodifiers ListOfUnitDefinitions::getTypeCode() const "
  /**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for this object, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers ListOfUnitDefinitions::getItemTypeCode() const "
  /**
   * Returns the libSBML type code for the objects contained in this {@link ListOf}
   * (i.e., {@link UnitDefinition} objects, if the list is non-empty).
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for the objects contained in this {@link ListOf}
   * instance, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers ListOfUnitDefinitions::getElementName() const "
  /**
   * Returns the XML element name of this object.
   * <p>
   * For {@link ListOfUnitDefinitions}, the XML element name is 
   * <code>\'listOfUnitDefinitions\'</code>.
   * <p>
   * @return the name of this element, i.e., <code>\'listOfUnitDefinitions\'</code>.
   */
 public";

%javamethodmodifiers ListOfUnitDefinitions::get(unsigned int n) "
  /**
   * Get a {@link UnitDefinition} from the {@link ListOfUnitDefinitions}.
   * <p>
   * @param n the index number of the {@link UnitDefinition} to get.
   * <p>
   * @return the nth {@link UnitDefinition} in this {@link ListOfUnitDefinitions}.
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfUnitDefinitions::get "
  /**
   * Get a {@link UnitDefinition} from the {@link ListOfUnitDefinitions}.
   * <p>
   * @param n the index number of the {@link UnitDefinition} to get.
   * <p>
   * @return the nth {@link UnitDefinition} in this {@link ListOfUnitDefinitions}.
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfUnitDefinitions::get(const std::string& sid) "
  /**
   * Get a {@link UnitDefinition} from the {@link ListOfUnitDefinitions}
   * based on its identifier.
   * <p>
   * @param sid a string representing the identifier 
   * of the {@link UnitDefinition} to get.
   * <p>
   * @return {@link UnitDefinition} in this {@link ListOfUnitDefinitions}
   * with the given id or NULL if no such
   * {@link UnitDefinition} exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfUnitDefinitions::get(const std::string& sid) const "
  /**
   * Get a {@link UnitDefinition} from the {@link ListOfUnitDefinitions}
   * based on its identifier.
   * <p>
   * @param sid a string representing the identifier 
   * of the {@link UnitDefinition} to get.
   * <p>
   * @return {@link UnitDefinition} in this {@link ListOfUnitDefinitions}
   * with the given id or NULL if no such
   * {@link UnitDefinition} exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfUnitDefinitions::remove(unsigned int n) "
  /**
   * Removes the nth item from this {@link ListOfUnitDefinitions} items and returns a pointer to
   * it.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * <p>
   * @param n the index of the item to remove
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfUnitDefinitions::remove(const std::string& sid) "
  /**
   * Removes item in this {@link ListOfUnitDefinitions} items with the given identifier.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * If none of the items in this list have the identifier <code>sid</code>, then 
   * <code>NULL</code> is returned.
   * <p>
   * @param sid the identifier of the item to remove
   * <p>
   * @return the item removed.  As mentioned above, the caller owns the
   * returned item.
   */
 public";

%javamethodmodifiers ListOfUnitDefinitions::getElementPosition() const "
  /**
   * Get the ordinal position of this element in the containing object
   * (which in this case is the {@link Model} object).
   * <p>
   * The ordering of elements in the XML form of SBML is generally fixed
   * for most components in SBML.  So, for example, the
   * {@link ListOfUnitDefinitions} in a model is (in SBML Level&nbsp;2
   * Version&nbsp;4) the second ListOf___.  (However, it differs for
   * different Levels and Versions of SBML.)
   * <p>
   * @return the ordinal position of the element with respect to its
   * siblings, or <code>-1</code> (default) to indicate the position is not significant.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers ListOfUnitDefinitions::createObject(XMLInputStream& stream) "
  /**
   * @return the SBML object corresponding to next {@link XMLToken} in the
   * XMLInputStream or NULL if the token was not recognized.
   <p>
 * @deprecated libSBML internal
 */
 public";

%typemap(javaimports) CompartmentType "
/** 
 * LibSBML implementation of SBML\'s CompartmentType construct.
 * <p>
 * A <em>compartment type</em> in SBML is a grouping construct used to
 * establish a relationship between multiple {@link Compartment} objects.
 * In SBML Level 2 Versions 2, 3 and 4, a compartment type only has an
 * identity, and this identity can only be used to indicate that particular
 * compartments belong to this type.  This may be useful for conveying a
 * modeling intention, such as when a model contains many similar
 * compartments, either by their biological function or the reactions they
 * carry.  Without a compartment type construct, it would be impossible in
 * the language of SBML to indicate that all of the compartments share an
 * underlying conceptual relationship because each SBML compartment must be
 * given a unique and separate identity.  {@link Compartment} types have no
 * mathematical meaning in SBML Level 2&mdash;they have no effect on a
 * model\'s mathematical interpretation.  Simulators and other numerical
 * analysis software may ignore {@link CompartmentType} definitions and references
 * to them in a model.
 * <p>
 * There is no mechanism in SBML for representing hierarchies of
 * compartment types.  One {@link CompartmentType} instance cannot be the subtype
 * of another {@link CompartmentType} instance; SBML provides no means of defining
 * such relationships.
 * <p>
 * As with other major structures in SBML, {@link CompartmentType} has a mandatory
 * attribute, \'id\', used to give the compartment type an identifier.  The
 * identifier must be a text string conforming to the identifer syntax
 * permitted in SBML.  {@link CompartmentType} also has an optional \'name\'
 * attribute, of type <code>string</code>.  The \'id\' and \'name\' must be used
 * according to the guidelines described in the SBML specification (e.g.,
 * Section 3.3 in the Level 2 Version 4 specification).
 * <p>
 * {@link CompartmentType} was introduced in SBML Level 2 Version 2.  It is not
 * available in earlier versions of Level 2 nor in any version of Level 1.
 * <p>
 * @see Compartment
 * @see ListOfCompartmentTypes
 * @see SpeciesType
 * @see ListOfSpeciesTypes
 * <p>
 * <p>
 */
"

%typemap(javaimports) ListOfCompartmentTypes "
/** 
 * LibSBML implementation of SBML\'s ListOfCompartmentTypes construct.
 * <p>
 * The various ListOf___ classes in SBML are merely containers used for
 * organizing the main components of an SBML model.  All are derived from
 * the abstract class {@link SBase}, and inherit the various attributes and
 * subelements of {@link SBase}, such as \'metaid\' as and \'annotation\'.  The
 * ListOf___ classes do not add any attributes of their own.
 * <p>
 * The relationship between the lists and the rest of an SBML model is
 * illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):
 * <p>
 * <center><img src=\'listof-illustration.jpg\'></center><br>
 * 
 * <p>
 * Readers may wonder about the motivations for using the ListOf___
 * containers.  A simpler approach in XML might be to place the components
 * all directly at the top level of the model definition.  We chose instead
 * to group them within XML elements named after {@link ListOf}<em>Classname</em>,
 * in part because we believe this helps organize the components and makes
 * visual reading of models in XML easier.  More importantly, the fact that
 * the container classes are derived from {@link SBase} means that software tools
 * can add information about the lists themselves into each list
 * container\'s \'annotation\'.
 * <p>
 * @see ListOfFunctionDefinitions
 * @see ListOfUnitDefinitions
 * @see ListOfCompartmentTypes
 * @see ListOfSpeciesTypes
 * @see ListOfCompartments
 * @see ListOfSpecies
 * @see ListOfParameters
 * @see ListOfInitialAssignments
 * @see ListOfRules
 * @see ListOfConstraints
 * @see ListOfReactions
 * @see ListOfEvents
 */
"

%javamethodmodifiers CompartmentType::CompartmentType(unsigned int level, unsigned int version) "
  /**
   * Creates a new {@link CompartmentType} using the given SBML <code>level</code> and <code>version</code>
   * values.
   * <p>
   * @param level a long integer, the SBML Level to assign to this {@link CompartmentType}
   * <p>
   * @param version a long integer, the SBML Version to assign to this
   * {@link CompartmentType}
   * <p>
   * @note Once a {@link CompartmentType} has been added to an {@link SBMLDocument}, the <code>level</code>,
   * <code>version</code> for the document <em>override</em> those used
   * to create the {@link CompartmentType}.  Despite this, the ability to supply the values
   * at creation time is an important aid to creating valid SBML.  Knowledge of
   * the intented SBML Level and Version determine whether it is valid to
   * assign a particular value to an attribute, or whether it is valid to add
   * an object to an existing {@link SBMLDocument}.
   */
 public";

%javamethodmodifiers CompartmentType::CompartmentType(SBMLNamespaces* sbmlns) "
  /**
   * Creates a new {@link CompartmentType} using the given {@link SBMLNamespaces} object
   * <code>sbmlns</code>.
   * <p>
   * The {@link SBMLNamespaces} object encapsulates SBML Level/Version/namespaces
   * information.  It is used to communicate the SBML Level, Version, and
   * (in Level&nbsp;3) packages used in addition to SBML Level&nbsp; Core.
   * A common approach to using this class constructor is to create an
   * {@link SBMLNamespaces} object somewhere in a program, once, then pass it to
   * object constructors such as this one when needed.
   * <p>
   * It is worth emphasizing that although this constructor does not take
   * an identifier argument, in SBML Level&nbsp;2 and beyond, the \'id\'
   * (identifier) attribute of a {@link CompartmentType} is required to have a value.
   * Thus, callers are cautioned to assign a value after calling this
   * constructor.  Setting the identifier can be accomplished using the
   * method {@link SBase#setId(String id)} .
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object.
   * <p>
   * @note Once a {@link CompartmentType} has been added to an {@link SBMLDocument}, the <code>level</code>,
   * <code>version</code> and <code>xmlns</code> namespaces for the document <em>override</em> those used
   * to create the {@link CompartmentType}.  Despite this, the ability to supply the values
   * at creation time is an important aid to creating valid SBML.  Knowledge of
   * the intented SBML Level and Version determine whether it is valid to
   * assign a particular value to an attribute, or whether it is valid to add
   * an object to an existing {@link SBMLDocument}.
   */
 public";

%javamethodmodifiers CompartmentType::CompartmentType(const CompartmentType& orig) "
  /**
   * Copy constructor; creates a copy of this {@link CompartmentType}.
   */
 public";

%javamethodmodifiers CompartmentType::accept(SBMLVisitor& v) const "
  /**
   * Accepts the given {@link SBMLVisitor} for this instance of {@link CompartmentType}.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used.
   * <p>
   * @return the result of calling <code>v.visit()</code>, which indicates
   * whether the Visitor would like to visit the next {@link CompartmentType} in
   * the list of compartment types.
   */
 public";

%javamethodmodifiers CompartmentType::clone() const "
  /**
   * Creates and returns a deep copy of this {@link CompartmentType}.
   * <p>
   * @return a (deep) copy of this {@link CompartmentType}.
   */
 public";

%javamethodmodifiers CompartmentType::getId() const "
  /**
   * Returns the value of the \'id\' attribute of this {@link CompartmentType}.
   * <p>
   * @return the id of this {@link CompartmentType}.
   */
 public";

%javamethodmodifiers CompartmentType::getName() const "
  /**
   * Returns the value of the \'name\' attribute of this {@link CompartmentType}.
   * <p>
   * @return the name of this {@link CompartmentType}.
   */
 public";

%javamethodmodifiers CompartmentType::isSetId() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link CompartmentType}\'s \'id\' attribute has been set.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the \'id\' attribute of this {@link CompartmentType} has been
   * set, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers CompartmentType::isSetName() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link CompartmentType}\'s \'name\' attribute has been set.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the \'name\' attribute of this CompartmentTypeType has been
   * set, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers CompartmentType::setId(const std::string& sid) "
  /**
   * Sets the value of the \'id\' attribute of this {@link CompartmentType}.
   * <p>
   * The string <code>sid</code> is copied.  Note that SBML has strict requirements
   * for the syntax of identifiers.  The following is summary of the
   * definition of the SBML identifier type <code>SId</code> (here expressed in an
   * extended form of BNF notation):
   * <div class=\'fragment\'><pre>
   *   letter .= \'a\'..\'z\',\'A\'..\'Z\'
   *   digit  .= \'0\'..\'9\'
   *   idChar .= letter | digit | \'_\'
   *   SId    .= ( letter | \'_\' ) idChar</pre></div>
   * The equality of SBML identifiers is determined by an exact character
   * sequence match; i.e., comparisons must be performed in a
   * case-sensitive manner.  In addition, there are a few conditions for
   * the uniqueness of identifiers in an SBML model.  Please consult the
   * SBML specifications for the exact formulations.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param sid the string to use as the identifier of this {@link CompartmentType}
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   */
 public";

%javamethodmodifiers CompartmentType::setName(const std::string& name) "
  /**
   * Sets the value of the \'name\' attribute of this {@link CompartmentType}.
   * <p>
   * The string in <code>name</code> is copied.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param name the new name for the {@link CompartmentType}
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   */
 public";

%javamethodmodifiers CompartmentType::unsetName "
  /**
   * Unsets the value of the \'name\' attribute of this {@link CompartmentType}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_OPERATION_FAILED
   */
 public";

%javamethodmodifiers CompartmentType::getTypeCode() const "
  /**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for this object, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers CompartmentType::getElementName() const "
  /**
   * Returns the XML element name of this object, which for
   * {@link CompartmentType}, is always <code>\'compartmentType\'</code>.
   * <p>
   * @return the name of this element, i.e., <code>\'compartmentType\'</code>.
   */
 public";

%javamethodmodifiers CompartmentType::hasRequiredAttributes() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link CompartmentType} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link CompartmentType} object are:
   * id
   * <p>
   * @return a boolean value indicating whether all the required
   * attributes for this object have been defined.
   */
 public";

%javamethodmodifiers CompartmentType::CompartmentType "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link CompartmentType} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link CompartmentType} object are:
   * id
   * <p>
   * @return a boolean value indicating whether all the required
   * attributes for this object have been defined.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers CompartmentType::readAttributes(const XMLAttributes& attributes) "
  /**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * <p>
   * @param attributes the {@link XMLAttributes} to use.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers CompartmentType::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * <p>
   * @param stream the XMLOutputStream to use.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers CompartmentType::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * <p>
   * @param stream the XMLOutputStream to use.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers CompartmentType::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * <p>
   * @param stream the XMLOutputStream to use.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers CompartmentType::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * <p>
   * @param stream the XMLOutputStream to use.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers CompartmentType::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * <p>
   * @param stream the XMLOutputStream to use.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers CompartmentType::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * <p>
   * @param stream the XMLOutputStream to use.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers CompartmentType::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * <p>
   * @param stream the XMLOutputStream to use.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers CompartmentType::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * <p>
   * @param stream the XMLOutputStream to use.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers CompartmentType::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * <p>
   * @param stream the XMLOutputStream to use.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers CompartmentType::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * <p>
   * @param stream the XMLOutputStream to use.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers CompartmentType::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * <p>
   * @param stream the XMLOutputStream to use.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers CompartmentType::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * <p>
   * @param stream the XMLOutputStream to use.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers CompartmentType::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * <p>
   * @param stream the XMLOutputStream to use.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers CompartmentType::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * <p>
   * @param stream the XMLOutputStream to use.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers CompartmentType::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * <p>
   * @param stream the XMLOutputStream to use.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers CompartmentType::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * <p>
   * @param stream the XMLOutputStream to use.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers CompartmentType::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * <p>
   * @param stream the XMLOutputStream to use.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers ListOfCompartmentTypes::clone() const "
  /**
   * Creates and returns a deep copy of this {@link ListOfCompartmentTypes} instance.
   * <p>
   * @return a (deep) copy of this {@link ListOfCompartmentTypes}.
   */
 public";

%javamethodmodifiers ListOfCompartmentTypes::getTypeCode() const "
  /**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for this object, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers ListOfCompartmentTypes::getItemTypeCode() const "
  /**
   * Returns the libSBML type code for the objects contained in this {@link ListOf}
   * (i.e., {@link CompartmentType} objects, if the list is non-empty).
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for the objects contained in this {@link ListOf}
   * instance, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers ListOfCompartmentTypes::getElementName() const "
  /**
   * Returns the XML element name of this object.
   * <p>
   * For {@link ListOfCompartmentTypes}, the XML element name is 
   * <code>\'listOfCompartmentTypes\'</code>.
   * <p>
   * @return the name of this element, i.e., <code>\'listOfCompartmentTypes\'</code>.
   */
 public";

%javamethodmodifiers ListOfCompartmentTypes::get(unsigned int n) "
  /**
   * Get a {@link CompartmentType} from the {@link ListOfCompartmentTypes}.
   * <p>
   * @param n the index number of the {@link CompartmentType} to get.
   * <p>
   * @return the nth {@link CompartmentType} in this {@link ListOfCompartmentTypes}.
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfCompartmentTypes::get "
  /**
   * Get a {@link CompartmentType} from the {@link ListOfCompartmentTypes}.
   * <p>
   * @param n the index number of the {@link CompartmentType} to get.
   * <p>
   * @return the nth {@link CompartmentType} in this {@link ListOfCompartmentTypes}.
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfCompartmentTypes::get(const std::string& sid) "
  /**
   * Get a {@link CompartmentType} from the {@link ListOfCompartmentTypes}
   * based on its identifier.
   * <p>
   * @param sid a string representing the identifier 
   * of the {@link CompartmentType} to get.
   * <p>
   * @return {@link CompartmentType} in this {@link ListOfCompartmentTypes}
   * with the given id or NULL if no such
   * {@link CompartmentType} exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfCompartmentTypes::get(const std::string& sid) const "
  /**
   * Get a {@link CompartmentType} from the {@link ListOfCompartmentTypes}
   * based on its identifier.
   * <p>
   * @param sid a string representing the identifier 
   * of the {@link CompartmentType} to get.
   * <p>
   * @return {@link CompartmentType} in this {@link ListOfCompartmentTypes}
   * with the given id or NULL if no such
   * {@link CompartmentType} exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfCompartmentTypes::remove(unsigned int n) "
  /**
   * Removes the nth item from this {@link ListOfCompartmentTypes} items and returns a pointer to
   * it.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * <p>
   * @param n the index of the item to remove
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfCompartmentTypes::remove(const std::string& sid) "
  /**
   * Removes item in this {@link ListOfCompartmentTypes} items with the given identifier.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * If none of the items in this list have the identifier <code>sid</code>, then 
   * <code>NULL</code> is returned.
   * <p>
   * @param sid the identifier of the item to remove
   * <p>
   * @return the item removed.  As mentioned above, the caller owns the
   * returned item.
   */
 public";

%javamethodmodifiers ListOfCompartmentTypes::getElementPosition() const "
  /**
   * Get the ordinal position of this element in the containing object
   * (which in this case is the {@link Model} object).
   * <p>
   * The ordering of elements in the XML form of SBML is generally fixed
   * for most components in SBML.  For example, the
   * {@link ListOfCompartmentTypes} in a model (in SBML Level 2 Version 4) is the
   * third ListOf___.  (However, it differs for different Levels and
   * Versions of SBML, so calling code should not hardwire this number.)
   * <p>
   * @return the ordinal position of the element with respect to its
   * siblings, or <code>-1</code> (default) to indicate the position is not significant.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers ListOfCompartmentTypes::createObject(XMLInputStream& stream) "
  /**
   * Create a {@link ListOfCompartmentTypes} object corresponding to the next token
   * in the XML input stream.
   * <p>
   * @return the SBML object corresponding to next {@link XMLToken} in the
   * XMLInputStream, or <code>NULL</code> if the token was not recognized.
   <p>
 * @deprecated libSBML internal
 */
 public";

%typemap(javaimports) SpeciesType "
/** 
 * LibSBML implementation of SBML\'s SpeciesType construct.
 * <p>
 * The term <em>species</em> <em>type</em> refers to reacting entities independent of
 * location.  These include simple ions (e.g., protons, calcium), simple
 * molecules (e.g., glucose, ATP), large molecules (e.g., RNA,
 * polysaccharides, and proteins), and others.
 * <p>
 * {@link SpeciesType} structures are included in SBML to enable {@link Species} of the
 * same type to be related together.  It is a conceptual construct; the
 * existence of {@link SpeciesType} objects in a model has no effect on the
 * model\'s numerical interpretation.  Except for the requirement for
 * uniqueness of species/species type combinations located in compartments,
 * simulators and other numerical analysis software may ignore {@link SpeciesType}
 * definitions and references to them in a model.
 * <p>
 * There is no mechanism in SBML for representing hierarchies of species
 * types.  One {@link SpeciesType} object cannot be the subtype of another
 * {@link SpeciesType} object; SBML provides no means of defining such
 * relationships.
 * <p>
 * As with other major structures in SBML, {@link SpeciesType} has a mandatory
 * attribute, \'id\', used to give the species type an identifier.  The
 * identifier must be a text string conforming to the identifer syntax
 * permitted in SBML.  {@link SpeciesType} also has an optional \'name\' attribute,
 * of type <code>string</code>.  The \'id\' and \'name\' must be used according to the
 * guidelines described in the SBML specification (e.g., Section 3.3 in
 * the Level 2 Version 4 specification).
 * <p>
 * {@link SpeciesType} was introduced in SBML Level 2 Version 2.  It is not
 * available in earlier versions of Level 2 nor in any version of Level 1.
 * <p>
 * <!---------------------------------------------------------------------- -->
 * <p>
 */
"

%typemap(javaimports) ListOfSpeciesTypes "
/** 
 * LibSBML implementation of SBML\'s ListOfSpeciesTypes construct.
 * <p>
 * The various ListOf___ classes in SBML are merely containers used for
 * organizing the main components of an SBML model.  All are derived from
 * the abstract class {@link SBase}, and inherit the various attributes and
 * subelements of {@link SBase}, such as \'metaid\' as and \'annotation\'.  The
 * ListOf___ classes do not add any attributes of their own.
 * <p>
 * The relationship between the lists and the rest of an SBML model is
 * illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):
 * <p>
 * <center><img src=\'listof-illustration.jpg\'></center><br>
 * 
 * <p>
 * Readers may wonder about the motivations for using the ListOf___
 * containers.  A simpler approach in XML might be to place the components
 * all directly at the top level of the model definition.  We chose instead
 * to group them within XML elements named after {@link ListOf}<em>Classname</em>,
 * in part because we believe this helps organize the components and makes
 * visual reading of models in XML easier.  More importantly, the fact that
 * the container classes are derived from {@link SBase} means that software tools
 * can add information about the lists themselves into each list
 * container\'s \'annotation\'.
 * <p>
 * @see ListOfFunctionDefinitions
 * @see ListOfUnitDefinitions
 * @see ListOfCompartmentTypes
 * @see ListOfSpeciesTypes
 * @see ListOfCompartments
 * @see ListOfSpecies
 * @see ListOfParameters
 * @see ListOfInitialAssignments
 * @see ListOfRules
 * @see ListOfConstraints
 * @see ListOfReactions
 * @see ListOfEvents
 */
"

%javamethodmodifiers SpeciesType::SpeciesType(unsigned int level, unsigned int version) "
  /**
   * Creates a new {@link SpeciesType} using the given SBML <code>level</code> and <code>version</code>
   * values.
   * <p>
   * @param level a long integer, the SBML Level to assign to this {@link SpeciesType}
   * <p>
   * @param version a long integer, the SBML Version to assign to this
   * {@link SpeciesType}
   * <p>
   * @note Once a {@link SpeciesType} has been added to an {@link SBMLDocument}, the <code>level</code>,
   * <code>version</code> for the document <em>override</em> those used
   * to create the {@link SpeciesType}.  Despite this, the ability to supply the values
   * at creation time is an important aid to creating valid SBML.  Knowledge of
   * the intented SBML Level and Version determine whether it is valid to
   * assign a particular value to an attribute, or whether it is valid to add
   * an object to an existing {@link SBMLDocument}.
   */
 public";

%javamethodmodifiers SpeciesType::SpeciesType(SBMLNamespaces* sbmlns) "
  /**
   * Creates a new {@link SpeciesType} using the given {@link SBMLNamespaces} object
   * <code>sbmlns</code>.
   * <p>
   * The {@link SBMLNamespaces} object encapsulates SBML Level/Version/namespaces
   * information.  It is used to communicate the SBML Level, Version, and
   * (in Level&nbsp;3) packages used in addition to SBML Level&nbsp; Core.
   * A common approach to using this class constructor is to create an
   * {@link SBMLNamespaces} object somewhere in a program, once, then pass it to
   * object constructors such as this one when needed.
   * <p>
   * It is worth emphasizing that although this constructor does not take
   * an identifier argument, in SBML Level&nbsp;2 and beyond, the \'id\'
   * (identifier) attribute of a {@link SpeciesType} object is required to have a value.
   * Thus, callers are cautioned to assign a value after calling this
   * constructor.  Setting the identifier can be accomplished using the
   * method {@link SBase#setId(String id)} .
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object.
   * <p>
   * @note Once a {@link SpeciesType} has been added to an {@link SBMLDocument}, the <code>level</code>,
   * <code>version</code> and <code>xmlns</code> namespaces for the document <em>override</em> those used
   * to create the {@link SpeciesType}.  Despite this, the ability to supply the values
   * at creation time is an important aid to creating valid SBML.  Knowledge of
   * the intented SBML Level and Version determine whether it is valid to
   * assign a particular value to an attribute, or whether it is valid to add
   * an object to an existing {@link SBMLDocument}.
   */
 public";

%javamethodmodifiers SpeciesType::SpeciesType(const SpeciesType& orig) "
  /**
  * Copy constructor; creates a copy of this {@link SpeciesType}.
  */
 public";

%javamethodmodifiers SpeciesType::accept(SBMLVisitor& v) const "
  /**
   * Accepts the given {@link SBMLVisitor} for this instance of {@link SpeciesType}.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used.
   * <p>
   * @return the result of calling <code>v.visit()</code>, which indicates
   * whether the Visitor would like to visit the next {@link SpeciesType} in
   * the list of compartment types.
   */
 public";

%javamethodmodifiers SpeciesType::clone() const "
  /**
   * Creates and returns a deep copy of this {@link SpeciesType}.
   * <p>
   * @return a (deep) copy of this {@link SpeciesType}.
   */
 public";

%javamethodmodifiers SpeciesType::getId() const "
  /**
   * Returns the value of the \'id\' attribute of this {@link SpeciesType}.
   * <p>
   * @return the id of this {@link SpeciesType}.
   */
 public";

%javamethodmodifiers SpeciesType::getName() const "
  /**
   * Returns the value of the \'name\' attribute of this {@link SpeciesType}.
   * <p>
   * @return the name of this {@link SpeciesType}.
   */
 public";

%javamethodmodifiers SpeciesType::isSetId() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link SpeciesType}\'s \'id\' attribute has been set.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the \'id\' attribute of this {@link SpeciesType} has been
   * set, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers SpeciesType::isSetName() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link SpeciesType}\'s \'name\' attribute has been set.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the \'name\' attribute of this {@link SpeciesType} has been
   * set, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers SpeciesType::setId(const std::string& sid) "
  /**
   * Sets the value of the \'id\' attribute of this {@link SpeciesType}.
   * <p>
   * The string <code>sid</code> is copied.  Note that SBML has strict requirements
   * for the syntax of identifiers.  The following is summary of the
   * definition of the SBML identifier type <code>SId</code> (here expressed in an
   * extended form of BNF notation):
   * <div class=\'fragment\'><pre>
   *   letter .= \'a\'..\'z\',\'A\'..\'Z\'
   *   digit  .= \'0\'..\'9\'
   *   idChar .= letter | digit | \'_\'
   *   SId    .= ( letter | \'_\' ) idChar</pre></div>
   * The equality of SBML identifiers is determined by an exact character
   * sequence match; i.e., comparisons must be performed in a
   * case-sensitive manner.  In addition, there are a few conditions for
   * the uniqueness of identifiers in an SBML model.  Please consult the
   * SBML specifications for the exact formulations.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param sid the string to use as the identifier of this {@link SpeciesType}
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   */
 public";

%javamethodmodifiers SpeciesType::setName(const std::string& name) "
  /**
   * Sets the value of the \'name\' attribute of this {@link SpeciesType}.
   * <p>
   * The string in <code>name</code> is copied.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param name the new name for the {@link SpeciesType}
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   */
 public";

%javamethodmodifiers SpeciesType::unsetName "
  /**
   * Unsets the value of the \'name\' attribute of this {@link SpeciesType}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_OPERATION_FAILED
   */
 public";

%javamethodmodifiers SpeciesType::getTypeCode() const "
  /**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for this object, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers SpeciesType::getElementName() const "
  /**
   * Returns the XML element name of this object, which for
   * {@link SpeciesType}, is always <code>\'compartmentType\'</code>.
   * <p>
   * @return the name of this element, i.e., <code>\'compartmentType\'</code>.
   */
 public";

%javamethodmodifiers SpeciesType::hasRequiredAttributes() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link SpeciesType} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link SpeciesType} object are:
   * id
   * <p>
   * @return a boolean value indicating whether all the required
   * attributes for this object have been defined.
   */
 public";

%javamethodmodifiers SpeciesType::SpeciesType "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link SpeciesType} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link SpeciesType} object are:
   * id
   * <p>
   * @return a boolean value indicating whether all the required
   * attributes for this object have been defined.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SpeciesType::readAttributes(const XMLAttributes& attributes) "
  /**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * <p>
   * @param attributes the {@link XMLAttributes} to use.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SpeciesType::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * <p>
   * @param stream the XMLOutputStream to use.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SpeciesType::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * <p>
   * @param stream the XMLOutputStream to use.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SpeciesType::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * <p>
   * @param stream the XMLOutputStream to use.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SpeciesType::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * <p>
   * @param stream the XMLOutputStream to use.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SpeciesType::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * <p>
   * @param stream the XMLOutputStream to use.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SpeciesType::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * <p>
   * @param stream the XMLOutputStream to use.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SpeciesType::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * <p>
   * @param stream the XMLOutputStream to use.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SpeciesType::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * <p>
   * @param stream the XMLOutputStream to use.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SpeciesType::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * <p>
   * @param stream the XMLOutputStream to use.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SpeciesType::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * <p>
   * @param stream the XMLOutputStream to use.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SpeciesType::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * <p>
   * @param stream the XMLOutputStream to use.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SpeciesType::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * <p>
   * @param stream the XMLOutputStream to use.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SpeciesType::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * <p>
   * @param stream the XMLOutputStream to use.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SpeciesType::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * <p>
   * @param stream the XMLOutputStream to use.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SpeciesType::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * <p>
   * @param stream the XMLOutputStream to use.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SpeciesType::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * <p>
   * @param stream the XMLOutputStream to use.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers SpeciesType::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * <p>
   * @param stream the XMLOutputStream to use.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers ListOfSpeciesTypes::clone() const "
  /**
   * Creates and returns a deep copy of this {@link ListOfSpeciesTypes} instance.
   * <p>
   * @return a (deep) copy of this {@link ListOfSpeciesTypes}.
   */
 public";

%javamethodmodifiers ListOfSpeciesTypes::getTypeCode() const "
  /**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for this object, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers ListOfSpeciesTypes::getItemTypeCode() const "
  /**
   * Returns the libSBML type code for the objects contained in this {@link ListOf}
   * (i.e., {@link SpeciesType} objects, if the list is non-empty).
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for the objects contained in this {@link ListOf}
   * instance, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers ListOfSpeciesTypes::getElementName() const "
  /**
   * Returns the XML element name of this object.
   * <p>
   * For {@link ListOfSpeciesTypes}, the XML element name is 
   * <code>\'listOfSpeciesTypes\'</code>.
   * <p>
   * @return the name of this element, i.e., <code>\'listOfSpeciesTypes\'</code>.
   */
 public";

%javamethodmodifiers ListOfSpeciesTypes::get(unsigned int n) "
  /**
   * Get a {@link SpeciesType} from the {@link ListOfSpeciesTypes}.
   * <p>
   * @param n the index number of the {@link SpeciesType} to get.
   * <p>
   * @return the nth {@link SpeciesType} in this {@link ListOfSpeciesTypes}.
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfSpeciesTypes::get "
  /**
   * Get a {@link SpeciesType} from the {@link ListOfSpeciesTypes}.
   * <p>
   * @param n the index number of the {@link SpeciesType} to get.
   * <p>
   * @return the nth {@link SpeciesType} in this {@link ListOfSpeciesTypes}.
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfSpeciesTypes::get(const std::string& sid) "
  /**
   * Get a {@link SpeciesType} from the {@link ListOfSpeciesTypes}
   * based on its identifier.
   * <p>
   * @param sid a string representing the identifier 
   * of the {@link SpeciesType} to get.
   * <p>
   * @return {@link SpeciesType} in this {@link ListOfSpeciesTypes}
   * with the given id or NULL if no such
   * {@link SpeciesType} exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfSpeciesTypes::get(const std::string& sid) const "
  /**
   * Get a {@link SpeciesType} from the {@link ListOfSpeciesTypes}
   * based on its identifier.
   * <p>
   * @param sid a string representing the identifier 
   * of the {@link SpeciesType} to get.
   * <p>
   * @return {@link SpeciesType} in this {@link ListOfSpeciesTypes}
   * with the given id or NULL if no such
   * {@link SpeciesType} exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfSpeciesTypes::remove(unsigned int n) "
  /**
   * Removes the nth item from this {@link ListOfSpeciesTypes} items and returns a pointer to
   * it.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * <p>
   * @param n the index of the item to remove
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfSpeciesTypes::remove(const std::string& sid) "
  /**
   * Removes item in this {@link ListOfSpeciesTypes} items with the given identifier.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * If none of the items in this list have the identifier <code>sid</code>, then 
   * <code>NULL</code> is returned.
   * <p>
   * @param sid the identifier of the item to remove
   * <p>
   * @return the item removed.  As mentioned above, the caller owns the
   * returned item.
   */
 public";

%javamethodmodifiers ListOfSpeciesTypes::getElementPosition() const "
  /**
   * Get the ordinal position of this element in the containing object
   * (which in this case is the {@link Model} object).
   * <p>
   * The ordering of elements in the XML form of SBML is generally fixed
   * for most components in SBML.  For example, the
   * {@link ListOfSpeciesTypes} in a model (in SBML Level 2 Version 4) is the
   * third ListOf___.  (However, it differs for different Levels and
   * Versions of SBML, so calling code should not hardwire this number.)
   * <p>
   * @return the ordinal position of the element with respect to its
   * siblings, or <code>-1</code> (default) to indicate the position is not significant.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers ListOfSpeciesTypes::createObject(XMLInputStream& stream) "
  /**
   * Create a {@link ListOfSpeciesTypes} object corresponding to the next token
   * in the XML input stream.
   * <p>
   * @return the SBML object corresponding to next {@link XMLToken} in the
   * XMLInputStream, or <code>NULL</code> if the token was not recognized.
   <p>
 * @deprecated libSBML internal
 */
 public";

%typemap(javaimports) Compartment "
/** 
 * LibSBML implementation of SBML\'s Compartment construct.
 * <p>
 * A compartment in SBML represents a bounded space in which species are
 * located.  Compartments do not necessarily have to correspond to actual
 * structures inside or outside of a biological cell.
 * <p>
 * It is important to note that although compartments are optional in the
 * overall definition of {@link Model}, every species in an SBML model must be
 * located in a compartment.  This in turn means that if a model defines
 * any species, the model must also define at least one compartment.  The
 * reason is simply that species represent physical things, and therefore
 * must exist <em>somewhere</em>.  Compartments represent the <em>somewhere</em>.
 * <p>
 * {@link Compartment} has one required attribute, \'id\', to give the compartment a
 * unique identifier by which other parts of an SBML model definition can
 * refer to it.  A compartment can also have an optional \'name\' attribute
 * of type <code>string</code>.  Identifiers and names must be used according to the
 * guidelines described in the SBML specification (e.g., Section 3.3 in
 * the Level 2 Version 4 specification).
 * <p>
 * Each compartment in a model may optionally be designated as belonging to
 * a particular compartment type.  The optional attribute \'compartmentType\'
 * is used identify the compartment type represented by the {@link Compartment}
 * structure.  The \'compartmentType\' attribute\'s value must be the
 * identifier of a {@link CompartmentType} instance defined in the model.  If the
 * \'compartmentType\' attribute is not present on a particular compartment
 * definition, a unique virtual compartment type is assumed for that
 * compartment, and no other compartment can belong to that compartment
 * type.  The values of \'compartmentType\' attributes on compartments have
 * no effect on the numerical interpretation of a model.  Simulators and
 * other numerical analysis software may ignore \'compartmentType\'
 * attributes.
 * <p>
 * {@link Compartment} also has an optional attribute \'spatialDimensions\', whose
 * value must be a positive integer indicating the number of spatial
 * dimensions possessed by the compartment.  The maximum value is <code>3</code>,
 * meaning a three-dimensional structure (a volume).  Other permissible
 * values are <code>2</code> (for a two-dimensional area), <code>1</code> (for a
 * one-dimensional curve), and <code>0</code> (for a point).  The default value of
 * \'spatialDimensions\' is <code>3</code>.
 * <p>
 * {@link Compartment} has another optional attribute named \'size\', representing
 * the initial total size of the compartment.  The \'size\' attribute must be
 * a floating-point value and may represent a volume (if the compartment is
 * a three-dimensional one), or an area (if the compartment is
 * two-dimensional), or a length (if the compartment is one-dimensional).
 * There is no default value of compartment size in SBML Level 2.  In
 * particular, a missing \'size\' value <em>does not imply that the
 * compartment size is 1</em>.  (This is unlike the definition of
 * compartment \'volume\' in SBML Level 1.)  When the compartment\'s
 * \'spatialDimensions\' attribute does not have a value of <code>0</code>, a missing
 * value of \'size\' for a given compartment signifies that the value either
 * is unknown, or to be obtained from an external source, or determined by
 * an {@link InitialAssignment}, {@link AssignmentRule}, {@link AlgebraicRule} or {@link RateRule}
 * elsewhere in the model.  The \'size\' attribute must not be present if the
 * \'spatialDimensions\' attribute has a value of <code>0</code>; otherwise, a logical
 * inconsistency would exist because a zero-dimensional object cannot have
 * a physical size.
 * <p>
 * The units associated with a compartment\'s \'size\' attribute value may be
 * set using the optional {@link Compartment} attribute \'units\'.  The default
 * units, and the kinds of units allowed as values of the attribute
 * \'units\', interact with the number of spatial dimensions of the
 * compartment.  The value of the \'units\' attribute of a {@link Compartment} object
 * must be one of the base units (see {@link Unit}), or the predefined unit
 * identifiers <code>volume</code>, <code>area</code>, <code>length</code> or <code>dimensionless</code>, or a new
 * unit defined by a {@link UnitDefinition} object in the enclosing {@link Model}, subject
 * to the restrictions detailed in the following table:
 * <p>
 * <center>
<table border=\'0\' class=\'text-table width80 normal-font alt-row-colors\'>
<caption class=\'top-caption\'>Restrictions on values permitted for
compartment <code>size</code> and <code>units</code> attributes.</caption>
 <tr>
     <th align=\'left\' valign=\'bottom\'>
       Value of<br><code>spatialDimensions</code>
     </th>
     <th align=\'left\' valign=\'bottom\'>
       <code>size</code><br>allowed?
     </th>
     <th align=\'left\' valign=\'bottom\'>
       <code>units</code><br>allowed?
     </th>
     <th align=\'left\' valign=\'bottom\'>
	  Allowable kinds of units
     </th>
     <th align=\'left\' valign=\'bottom\'>
	  Default value of attribute <code>units</code>
     </th>
 </tr>
 <tr>
     <td>\'<code>3</code>\'</td>
     <td>yes</td>
     <td>yes</td>
     <td>units of volume, or <code>dimensionless</code></td>
     <td>\'<code>volume</code>\'</td>
 </tr>
 <tr>
     <td>\'<code>2</code>\'</td>
     <td>yes</td>
     <td>yes</td>
     <td>units of area, or <code>dimensionless</code></td>
     <td>\'<code>area</code>\'</td>
 </tr>
 <tr>
     <td>\'<code>1</code>\'</td>
     <td>yes</td>
     <td>yes</td>
     <td>units of length, or <code>dimensionless</code></td>
     <td>\'<code>length</code>\'</td>
 </tr>
 <tr>
     <td>\'<code>0</code>\'</td>
     <td>no</td>
     <td>no</td>
     <td>(no units allowed)</td>
     <td></td>
 </tr>
 </tr>
</table>
</center>

 * <p>
 * In SBML Level 2, the units of the compartment size, as defined by the
 * \'units\' attribute or (if \'units\' is not set) the default value listed in
 * the table above, are used in the following ways when the compartment has
 * a \'spatialDimensions\' value greater than <code>0</code>:
 * <ul>
 * <li> The value of the \'units\' attribute is used as the units of the
 * compartment identifier when the identifier appears as a numerical
 * quantity in a mathematical formula expressed in MathML.
 * <p>
 * <li> The <code>math</code> element of an {@link AssignmentRule} or {@link InitialAssignment}
 * referring to this compartment must have identical units.
 * <p>
 * <li> In {@link RateRule} objects that set the rate of change of the compartment\'s
 * size, the units of the rule\'s <code>math</code> element must be identical to the
 * compartment\'s \'units\' attribute divided by the default <em>time</em> units.
 * (In other words, the units for the rate of change of compartment size
 * are <em>compartment size</em>/<em>time</em> units.
 * <p>
 * <li> When a {@link Species} is to be treated in terms of concentrations or
 * density, the units of the spatial size portion of the concentration
 * value (i.e., the denominator in the units formula <em>substance</em>/
 * <em>size</em>) are those indicated by the value of the \'units\' attribute on the
 * compartment in which the species is located.
 * </ul>
 * <p>
 * Compartments with \'spatialDimensions\'=<code>0</code> require special treatment in
 * this framework.  If a compartment has no size or dimensional units, how
 * should such a compartment\'s identifier be interpreted when it appears in
 * mathematical formulas?  The answer is that such a compartment\'s
 * identifier should not appear in mathematical formulas in the first
 * place&mdash;it has no value, and its value cannot change.  Note also
 * that a zero-dimensional compartment is a point, and species located at
 * points can only be described in terms of amounts, not
 * spatially-dependent measures such as concentration.  Since SBML
 * {@link KineticLaw} formulas are already in terms of <em>substance</em>/<em>time</em> and
 * not (say) <em>concentration</em>/<em>time</em>, volume or other factors in
 * principle are not needed for species located in zero-dimensional
 * compartments.
 * <p>
 * {@link Compartment} has another optional attribute named \'constant\'.  This takes
 * a boolean value indicating whether the compartment\'s size stays constant
 * or can vary during a simulation.  A value of <code>false</code> indicates the
 * compartment\'s \'size\' can be changed by other constructs in SBML.  A
 * value of <code>true</code> indicates the compartment\'s \'size\' cannot be changed by
 * any other construct except {@link InitialAssignment}.  In the special case of
 * \'spatialDimensions\'=<code>0</code>, the value cannot be changed by
 * {@link InitialAssignment} either.  The default value for the \'constant\'
 * attribute is <code>true</code> because in the most common modeling scenarios at
 * the time of this writing, compartment sizes remain constant.  The
 * \'constant\' attribute must default to or be set to <code>true</code> if the value
 * of the \'spatialDimensions\' attribute is <code>0</code>, because a zero-dimensional
 * compartment cannot ever have a size.
 * <p>
 * Finally, {@link Compartment} has an optional attribute named \'outside\', whose
 * value can be the identifier of another {@link Compartment} object defined in the
 * enclosing {@link Model} object.  Doing so means that the other compartment
 * contains it or is outside of it.  This enables the representation of
 * simple topological relationships between compartments, for those
 * simulation systems that can make use of the information (e.g., for
 * drawing simple diagrams of compartments).  There are two restrictions on
 * the containment relationships in SBML.  First, because a compartment
 * with \'spatialDimensions\' of <code>0</code> has no size, such a compartment cannot
 * act as the container of any other compartment <em>except</em> compartments
 * that <em>also</em> have \'spatialDimensions\' values of <code>0</code>.  Second, the
 * directed graph formed by representing {@link Compartment} structures as vertexes
 * and the \'outside\' attribute values as edges must be acyclic.  The latter
 * condition is imposed to prevent a compartment from being contained
 * inside itself.  In the absence of a value for \'outside\', compartment
 * definitions in SBML Level 2 do not have any implied spatial
 * relationships between each other.
 * <p>
 * It is worth noting that in SBML, there is no relationship between
 * compartment sizes when compartment positioning is expressed using the
 * \'outside\' attribute.  The size of a given compartment does not in any
 * sense include the sizes of other compartments having it as the value of
 * their \'outside\' attributes.  In other words, if a compartment <em>B</em> has
 * the identifier of compartment <em>A</em> as its \'outside\' attribute value,
 * the size of <em>A</em> does not include the size of <em>B</em>.  The compartment
 * sizes are separate.
 * <p>
 * @note the \'size\' attribute on a compartment must be defined as optional;
 * however, <em>it is extremely good practice to specify values for
 * compartment sizes</em> when such values are available.  There are three
 * major technical reasons for this.  First, if the model contains any
 * species whose initial amounts are given in terms of concentrations, and
 * there is at least one reaction in the model referencing such a species,
 * then the model is numerically incomplete if it lacks a value for the
 * size of the compartment in which the species is located.  The reason is
 * simply that SBML {@link Reaction} objects defined in units of
 * <em>substance</em>/<em>time</em>, not concentration per time, and
 * thus the compartment size must at some point be used to convert from
 * species concentration to substance units.  Second, models ideally should
 * be instantiable in a variety of simulation frameworks.  A commonly-used
 * one is the discrete stochastic framework in which species are
 * represented as item counts (e.g., molecule counts).  If species\' initial
 * quantities are given in terms of concentrations or densities, it is
 * impossible to convert the values to item counts without knowing
 * compartment sizes.  Third, if a model contains multiple compartments
 * whose sizes are not all identical to each other, it is impossible to
 * quantify the reaction rate expressions without knowing the compartment
 * volumes.  The reason for the latter is again that reaction rates in SBML
 * are defined in terms of <em>substance</em>/<em>time</em>, and when
 * species quantities are given in terms of concentrations or densities,
 * the compartment sizes become factors in the reaction rate expressions.
 * <p>
 * <!-- leave this next break as-is to work around some doxygen bug -->
 */
"

%typemap(javaimports) ListOfCompartments "
/** 
 * LibSBML implementation of SBML\'s ListOfCompartments construct.
 * <p>
 * The various ListOf___ classes in SBML are merely containers used for
 * organizing the main components of an SBML model.  All are derived from
 * the abstract class {@link SBase}, and inherit the various attributes and
 * subelements of {@link SBase}, such as \'metaid\' as and \'annotation\'.  The
 * ListOf___ classes do not add any attributes of their own.
 * <p>
 * The relationship between the lists and the rest of an SBML model is
 * illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):
 * <p>
 * <center><img src=\'listof-illustration.jpg\'></center><br>
 * 
 * <p>
 * Readers may wonder about the motivations for using the ListOf___
 * containers.  A simpler approach in XML might be to place the components
 * all directly at the top level of the model definition.  We chose instead
 * to group them within XML elements named after {@link ListOf}<em>Classname</em>,
 * in part because we believe this helps organize the components and makes
 * visual reading of models in XML easier.  More importantly, the fact that
 * the container classes are derived from {@link SBase} means that software tools
 * can add information about the lists themselves into each list
 * container\'s \'annotation\'.
 * <p>
 * @see ListOfFunctionDefinitions
 * @see ListOfUnitDefinitions
 * @see ListOfCompartmentTypes
 * @see ListOfSpeciesTypes
 * @see ListOfCompartments
 * @see ListOfSpecies
 * @see ListOfParameters
 * @see ListOfInitialAssignments
 * @see ListOfRules
 * @see ListOfConstraints
 * @see ListOfReactions
 * @see ListOfEvents
 */
"

%javamethodmodifiers Compartment::Compartment(unsigned int level, unsigned int version) "
  /**
   * Creates a new {@link Compartment} using the given SBML <code>level</code> and <code>version</code>
   * values.
   * <p>
   * @param level a long integer, the SBML Level to assign to this {@link Compartment}
   * <p>
   * @param version a long integer, the SBML Version to assign to this
   * {@link Compartment}
   * <p>
   * @note Once a {@link Compartment} has been added to an {@link SBMLDocument}, the <code>level</code>,
   * <code>version</code> for the document <em>override</em> those used
   * to create the {@link Compartment}.  Despite this, the ability to supply the values
   * at creation time is an important aid to creating valid SBML.  Knowledge of
   * the intented SBML Level and Version determine whether it is valid to
   * assign a particular value to an attribute, or whether it is valid to add
   * an object to an existing {@link SBMLDocument}.
   */
 public";

%javamethodmodifiers Compartment::Compartment(SBMLNamespaces* sbmlns) "
  /**
   * Creates a new {@link Compartment} using the given {@link SBMLNamespaces} object 
   * <code>sbmlns</code>.
   * <p>
   * The {@link SBMLNamespaces} object encapsulates SBML Level/Version/namespaces
   * information.  It is used to communicate the SBML Level, Version, and
   * (in Level&nbsp;3) packages used in addition to SBML Level&nbsp; Core.
   * A common approach to using this class constructor is to create an
   * {@link SBMLNamespaces} object somewhere in a program, once, then pass it to
   * object constructors such as this one when needed.
   * <p>
   * It is worth emphasizing that although this constructor does not take
   * an identifier argument, in SBML Level&nbsp;2 and beyond, the \'id\'
   * (identifier) attribute of a {@link Compartment} is required to have a value.
   * Thus, callers are cautioned to assign a value after calling this
   * constructor.  Setting the identifier can be accomplished using the
   * method {@link SBase#setId(String id)} .
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object.
   * <p>
   * @note Once a {@link Compartment} has been added to an {@link SBMLDocument}, the <code>level</code>,
   * <code>version</code> and <code>xmlns</code> namespaces for the document <em>override</em> those used
   * to create the {@link Compartment}.  Despite this, the ability to supply the values
   * at creation time is an important aid to creating valid SBML.  Knowledge of
   * the intented SBML Level and Version determine whether it is valid to
   * assign a particular value to an attribute, or whether it is valid to add
   * an object to an existing {@link SBMLDocument}.
   */
 public";

%javamethodmodifiers Compartment::Compartment(const Compartment& orig) "
  /**
   * Copy constructor; creates a copy of a {@link Compartment}.
   * <p>
   * @param orig the {@link Compartment} instance to copy.
   */
 public";

%javamethodmodifiers Compartment::accept(SBMLVisitor& v) const "
  /**
   * Accepts the given {@link SBMLVisitor} for this instance of {@link Compartment}.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used.
   * <p>
   * @return the result of calling <code>v.visit()</code>, which indicates
   * whether the Visitor would like to visit the next {@link Compartment} in the
   * list of compartments within which this {@link Compartment} is embedded (i.e.,
   * the {@link ListOfCompartments} in the parent {@link Model}).
   */
 public";

%javamethodmodifiers Compartment::clone() const "
  /**
   * Creates and returns a deep copy of this {@link Compartment}.
   * <p>
   * @return a (deep) copy of this {@link Compartment}.
   */
 public";

%javamethodmodifiers Compartment::initDefaults "
  /**
   * Initializes the fields of this {@link Compartment} to the defaults defined in
   * the specification of the relevant Level/Version of SBML.
   * <ul>
   * <li> (SBML Level 1 only) sets attribute \'volume\' to <code>1</code>.0
   * <li> (SBML Level 2 only) sets attribute \'spatialDimensions\' to <code>3</code>
   * <li> (SBML Level 2 only) sets attribute \'constant\' to <code>1</code> (true)
   * </ul>
   */
 public";

%javamethodmodifiers Compartment::getId() const "
  /**
   * Returns the value of the \'id\' attribute of this {@link Compartment}.
   * <p>
   * @return the id of this {@link Compartment}.
   */
 public";

%javamethodmodifiers Compartment::getName() const "
  /**
   * Returns the value of the \'name\' attribute of this {@link Compartment}.
   * <p>
   * @return the name of this {@link Compartment}.
   */
 public";

%javamethodmodifiers Compartment::getCompartmentType() const "
  /**
   * Get the compartment type of this {@link Compartment}, as indicated by the
   * {@link Compartment} object\'s \'compartmentType\' attribute value.
   * <p>
   * @return the value of the \'compartmentType\' attribute of this
   * {@link Compartment} as a string.
   */
 public";

%javamethodmodifiers Compartment::getSpatialDimensions() const "
  /**
   * Get the number of spatial dimensions of this {@link Compartment} object.
   * <p>
   * @return the value of the \'spatialDimensions\' attribute of this
   * {@link Compartment} as a long integereger
   */
 public";

%javamethodmodifiers Compartment::getSize() const "
  /**
   * Get the size of this {@link Compartment}
   * <p>
   * This method is identical to getVolume().  In SBML Level 1,
   * compartments are always three-dimensional constructs and only have
   * volumes, whereas in SBML Level 2, compartments may be other than
   * three-dimensional and therefore the \'volume\' attribute is named \'size\'
   * in Level 2.  LibSBML provides both getSize() and getVolume() for
   * easier compatibility between SBML Levels.
   * <p>
   * @return the value of the \'size\' attribute (\'volume\' in Level 1) of
   * this {@link Compartment} as a float-point number.
   * <p>
   * @see #isSetSize()
   */
 public";

%javamethodmodifiers Compartment::getVolume() const "
  /**
   * (For SBML Level 1) Get the volume of this {@link Compartment}
   * <p>
   * This method is identical to getSize().  In SBML Level 1, compartments
   * are always three-dimensional constructs and only have volumes, whereas
   * in SBML Level 2, compartments may be other than three-dimensional and
   * therefore the \'volume\' attribute is named \'size\' in Level 2.  LibSBML
   * provides both getSize() and getVolume() for easier compatibility
   * between SBML Levels.
   * <p>
   * @return the value of the \'volume\' attribute (\'size\' in Level 2) of
   * this {@link Compartment}, as a floating-point number.
   * <p>
   * @see #isSetVolume()
   */
 public";

%javamethodmodifiers Compartment::getUnits() const "
  /**
   * Get the units of this compartment\'s size or volume.
   * <p>
   * @return the value of the \'units\' attribute of this {@link Compartment}.
   */
 public";

%javamethodmodifiers Compartment::getOutside() const "
  /**
   * Get the identifier, if any, of the compartment that is designated
   * as being outside of this one.
   * <p>
   * @return the value of the \'outside\' attribute of this {@link Compartment}.
   */
 public";

%javamethodmodifiers Compartment::getConstant() const "
  /**
   * Get the value of the \'constant\' attribute of this {@link Compartment}.
   * <p>
   * @return <code>true</code> if this {@link Compartment}\'s size is flagged as being
   * constant, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers Compartment::isSetId() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link Compartment}\'s \'id\' attribute has been set.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the \'id\' attribute of this {@link Compartment} has been
   * set, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers Compartment::isSetName() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link Compartment}\'s \'name\' attribute has been set.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the \'name\' attribute of this {@link Compartment} has been
   * set, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers Compartment::isSetCompartmentType() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link Compartment}\'s \'compartmentType\' attribute has been set.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the \'compartmentType\' attribute of this {@link Compartment}
   * has been set, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers Compartment::isSetSize() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link Compartment}\'s \'size\' attribute has been set.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * This method is similar but not identical to isSetVolume().  The latter
   * should be used in the context of SBML Level 1 models instead of
   * isSetSize() because isSetVolume() performs extra processing to take
   * into account the difference in default values between SBML Levels 1
   * and 2.
   * <p>
   * @return <code>true</code> if the \'size\' attribute (\'volume\' in Level) of this
   * {@link Compartment} has been set, <code>false</code> otherwise.
   * <p>
   * @see #isSetVolume()
   * @see #setSize(double value)
   */
 public";

%javamethodmodifiers Compartment::isSetVolume() const "
  /**
   * (For SBML Level 1) Predicate returning <code>true</code> or <code>false</code> depending
   * on whether this {@link Compartment}\'s \'volume\' attribute has been set.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * This method is similar but not identical to isSetSize().  The latter
   * should not be used in the context of SBML Level 1 models because this
   * method (isSetVolume()) performs extra processing to take into account
   * the difference in default values between SBML Levels 1 and 2.
   * <p>
   * @return <code>true</code> if the \'volume\' attribute (\'size\' in L2) of this
   * {@link Compartment} has been set, <code>false</code> otherwise.
   * <p>
   * @see #isSetSize()
   * @see #setVolume(double value)
   * <p>
   * @note In SBML Level 1, a compartment\'s volume has a default value (
   * <code>1</code>.0) and therefore this method will always return <code>true</code>.  In Level
   * 2, a compartment\'s size (the equivalent of SBML Level 1\'s \'volume\') is
   * optional and has no default value, and therefore may or may not be
   * set.
   */
 public";

%javamethodmodifiers Compartment::isSetUnits() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link Compartment}\'s \'units\' attribute has been set.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the \'units\' attribute of this {@link Compartment} has been
   * set, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers Compartment::isSetOutside() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link Compartment}\'s \'outside\' attribute has been set.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the \'outside\' attribute of this {@link Compartment} has
   * been set, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers Compartment::setId(const std::string& sid) "
  /**
   * Sets the value of the \'id\' attribute of this {@link Compartment}.
   * <p>
   * The string <code>sid</code> is copied.  Note that SBML has strict requirements
   * for the syntax of identifiers.  The following is summary of the
   * definition of the SBML identifier type <code>SId</code> (here expressed in an
   * extended form of BNF notation):
   * <div class=\'fragment\'><pre>
   *   letter .= \'a\'..\'z\',\'A\'..\'Z\'
   *   digit  .= \'0\'..\'9\'
   *   idChar .= letter | digit | \'_\'
   *   SId    .= ( letter | \'_\' ) idChar</pre></div>
   * The equality of SBML identifiers is determined by an exact character
   * sequence match; i.e., comparisons must be performed in a
   * case-sensitive manner.  In addition, there are a few conditions for
   * the uniqueness of identifiers in an SBML model.  Please consult the
   * SBML specifications for the exact formulations.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param sid the string to use as the identifier of this {@link Compartment}
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   */
 public";

%javamethodmodifiers Compartment::setName(const std::string& name) "
  /**
   * Sets the value of the \'name\' attribute of this {@link Compartment}.
   * <p>
   * The string in <code>name</code> is copied.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param name the new name for the {@link Compartment}
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   */
 public";

%javamethodmodifiers Compartment::setCompartmentType(const std::string& sid) "
  /**
   * Sets the \'compartmentType\' attribute of this {@link Compartment}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param sid the identifier of a {@link CompartmentType} object defined
   * elsewhere in this {@link Model}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   * <li> LIBSBML_UNEXPECTED_ATTRIBUTE
   */
 public";

%javamethodmodifiers Compartment::setSpatialDimensions(unsigned int value) "
  /**
   * Sets the \'spatialDimensions\' attribute of this {@link Compartment}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * If <code>value</code> is not one of <code>0</code>, <code>1</code>, <code>2</code>, or <code>3</code>, this method will
   * have no effect (i.e., the \'spatialDimensions\' attribute will not be
   * set).
   * <p>
   * @param value a long integereger indicating the number of dimensions
   * of this compartment.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   * <li> LIBSBML_UNEXPECTED_ATTRIBUTE
   */
 public";

%javamethodmodifiers Compartment::setSize(double value) "
  /**
   * Sets the \'size\' attribute (or \'volume\' in SBML Level 1) of this
   * {@link Compartment}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * This method is identical to setVolume() and is provided for
   * compatibility between SBML Level 1 and Level 2.
   * <p>
   * @param value a <code>double</code> representing the size of this compartment
   * instance in whatever units are in effect for the compartment.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   */
 public";

%javamethodmodifiers Compartment::setVolume(double value) "
  /**
   * Sets the \'volume\' attribute (or \'size\' in SBML Level 2) of this
   * {@link Compartment}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * This method is identical to setVolume() and is provided for
   * compatibility between SBML Level 1 and Level 2.
   * <p>
   * @param value a <code>double</code> representing the volume of this compartment
   * instance in whatever units are in effect for the compartment.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   */
 public";

%javamethodmodifiers Compartment::setUnits(const std::string& sid) "
  /**
   * Sets the \'units\' attribute of this {@link Compartment}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param sid the identifier of the defined units to use.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   */
 public";

%javamethodmodifiers Compartment::setOutside(const std::string& sid) "
  /**
   * Sets the \'outside\' attribute of this {@link Compartment}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param sid the identifier of a compartment that encloses this one.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   */
 public";

%javamethodmodifiers Compartment::setConstant(bool value) "
  /**
   * Sets the value of the \'constant\' attribute of this {@link Compartment}.
   * <p>
   * @param value a boolean indicating whether the size/volume of this
   * compartment should be considered constant (<code>true</code>) or variable (
   * <code>false</code>)
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_UNEXPECTED_ATTRIBUTE
   */
 public";

%javamethodmodifiers Compartment::unsetName "
  /**
   * Unsets the value of the \'name\' attribute of this {@link Compartment}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_OPERATION_FAILED
   */
 public";

%javamethodmodifiers Compartment::unsetCompartmentType "
  /**
   * Unsets the value of the \'compartmentType\' attribute of this
   * {@link Compartment}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @see #setCompartmentType(String sid)
   * @see #isSetCompartmentType()
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_OPERATION_FAILED
   */
 public";

%javamethodmodifiers Compartment::unsetSize "
  /**
   * Unsets the value of the \'size\' attribute of this {@link Compartment}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   */
 public";

%javamethodmodifiers Compartment::unsetVolume "
  /**
   * (For SBML Level 1) Unsets the value of the \'volume\' attribute of this
   * {@link Compartment}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * In SBML Level 1, a {@link Compartment} volume has a default value (1.0) and
   * therefore <em>should always be set</em>.  In Level 2, \'size\' is
   * optional with no default value and as such may or may not be set.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   */
 public";

%javamethodmodifiers Compartment::unsetUnits "
  /**
   * Unsets the value of the \'units\' attribute of this {@link Compartment}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_OPERATION_FAILED
   */
 public";

%javamethodmodifiers Compartment::unsetOutside "
  /**
   * Unsets the value of the \'outside\' attribute of this {@link Compartment}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_OPERATION_FAILED
   */
 public";

%javamethodmodifiers Compartment::getDerivedUnitDefinition "
  /**
   * Constructs and returns a {@link UnitDefinition} that corresponds to the units
   * of this {@link Compartment}\'s designated size.
   * <p>
   * Compartments in SBML have an attribute (\'units\') for declaring the
   * units of measurement intended for the value of the compartment\'s size.
   * In the absence of a value given for this attribute, the units are
   * taken from the model\'s definition of <code>\'volume\'</code>, <code>\'area\'</code>, 
   * <code>\'length\'</code> units, depending on the value given to this {@link Compartment}\'s
   * \'size\' attribute, or (if the {@link Model} does not redefine them) the
   * corresponding SBML default units for those quantities.  Following that
   * procedure, the method getDerivedUnitDefinition() returns a
   * {@link UnitDefinition} based on the interpreted units of this compartment\'s
   * size.
   * <p>
   * Note that the functionality that facilitates unit analysis depends 
   * on the model as a whole.  Thus, in cases where the object has not 
   * been added to a model or the model itself is incomplete,
   * unit analysis is not possible and this method will return NULL.
   * <p>
   * Note also that unit declarations for {@link Compartment} are in terms of the
   * <em>identifier</em> of a unit, but this method returns a {@link UnitDefinition}
   * object, not a unit identifier.  It does this by constructing an
   * appropriate {@link UnitDefinition}.  Callers may find this particularly useful
   * when used in conjunction with the helper methods on {@link UnitDefinition} for
   * comparing different {@link UnitDefinition} objects.
   * <p>
   * @return a {@link UnitDefinition} that expresses the units of this 
   * {@link Compartment}.
   * <p>
   * @see #getUnits()
   */
 public";

%javamethodmodifiers Compartment::getDerivedUnitDefinition() const "
  /**
   * Constructs and returns a {@link UnitDefinition} that corresponds to the units
   * of this {@link Compartment}\'s designated size.
   * <p>
   * Compartments in SBML have an attribute (\'units\') for declaring the
   * units of measurement intended for the value of the compartment\'s size.
   * In the absence of a value given for this attribute, the units are
   * taken from the model\'s definition of <code>\'volume\'</code>, <code>\'area\'</code>, 
   * <code>\'length\'</code> units, depending on the value given to this {@link Compartment}\'s
   * \'size\' attribute, or (if the {@link Model} does not redefine them) the
   * corresponding SBML default units for those quantities.  Following that
   * procedure, the method getDerivedUnitDefinition() returns a
   * {@link UnitDefinition} based on the interpreted units of this compartment\'s
   * size.
   * <p>
   * Note that the functionality that facilitates unit analysis depends 
   * on the model as a whole.  Thus, in cases where the object has not 
   * been added to a model or the model itself is incomplete,
   * unit analysis is not possible and this method will return NULL.
   * <p>
   * Note also that unit declarations for {@link Compartment} are in terms of the
   * <em>identifier</em> of a unit, but this method returns a {@link UnitDefinition}
   * object, not a unit identifier.  It does this by constructing an
   * appropriate {@link UnitDefinition}.  Callers may find this particularly useful
   * when used in conjunction with the helper methods on {@link UnitDefinition} for
   * comparing different {@link UnitDefinition} objects.
   * <p>
   * @return a {@link UnitDefinition} that expresses the units of this 
   * {@link Compartment}.
   * <p>
   * @see #getUnits()
   */
 public";

%javamethodmodifiers Compartment::getTypeCode() const "
  /**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for this object, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers Compartment::getElementName() const "
  /**
   * Returns the XML element name of this object, which for {@link Compartment}, is
   * always <code>\'compartment\'</code>.
   * <p>
   * @return the name of this element, i.e., <code>\'compartment\'</code>.
   */
 public";

%javamethodmodifiers Compartment::hasRequiredAttributes() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link Compartment} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link Compartment} object are:
   * id (name in L1)
   * <p>
   * @return a boolean value indicating whether all the required
   * attributes for this object have been defined.
   */
 public";

%javamethodmodifiers Compartment::Compartment "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link Compartment} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link Compartment} object are:
   * id (name in L1)
   * <p>
   * @return a boolean value indicating whether all the required
   * attributes for this object have been defined.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Compartment::readAttributes(const XMLAttributes& attributes) "
  /**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Compartment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Compartment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Compartment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Compartment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Compartment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Compartment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Compartment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Compartment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Compartment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Compartment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Compartment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Compartment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Compartment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Compartment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Compartment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Compartment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Compartment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Compartment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Compartment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Compartment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Compartment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Compartment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Compartment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Compartment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers ListOfCompartments::clone() const "
  /**
   * Creates and returns a deep copy of this {@link ListOfCompartments} instance.
   * <p>
   * @return a (deep) copy of this {@link ListOfCompartments}.
   */
 public";

%javamethodmodifiers ListOfCompartments::getTypeCode() const "
  /**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for this object, or <code>SBML_UNKNOWN</code> (default).
<p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers ListOfCompartments::getItemTypeCode() const "
  /**
   * Returns the libSBML type code for the objects contained in this {@link ListOf}
   * (i.e., {@link Compartment} objects, if the list is non-empty).
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for the objects contained in this {@link ListOf}
   * instance, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers ListOfCompartments::getElementName() const "
  /**
   * Returns the XML element name of this object.
   * <p>
   * For {@link ListOfCompartments}, the XML element name is <code>\'listOfCompartments\'</code>.
   * <p>
   * @return the name of this element, i.e., <code>\'listOfCompartments\'</code>.
   */
 public";

%javamethodmodifiers ListOfCompartments::get(unsigned int n) "
  /**
   * Get a {@link Compartment} from the {@link ListOfCompartments}.
   * <p>
   * @param n the index number of the {@link Compartment} to get.
   * <p>
   * @return the nth {@link Compartment} in this {@link ListOfCompartments}.
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfCompartments::get "
  /**
   * Get a {@link Compartment} from the {@link ListOfCompartments}.
   * <p>
   * @param n the index number of the {@link Compartment} to get.
   * <p>
   * @return the nth {@link Compartment} in this {@link ListOfCompartments}.
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfCompartments::get(const std::string& sid) "
  /**
   * Get a {@link Compartment} from the {@link ListOfCompartments}
   * based on its identifier.
   * <p>
   * @param sid a string representing the identifier 
   * of the {@link Compartment} to get.
   * <p>
   * @return {@link Compartment} in this {@link ListOfCompartments}
   * with the given id or NULL if no such
   * {@link Compartment} exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfCompartments::get(const std::string& sid) const "
  /**
   * Get a {@link Compartment} from the {@link ListOfCompartments}
   * based on its identifier.
   * <p>
   * @param sid a string representing the identifier 
   * of the {@link Compartment} to get.
   * <p>
   * @return {@link Compartment} in this {@link ListOfCompartments}
   * with the given id or NULL if no such
   * {@link Compartment} exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfCompartments::remove(unsigned int n) "
  /**
   * Removes the nth item from this {@link ListOfCompartments} items and returns a pointer to
   * it.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * <p>
   * @param n the index of the item to remove
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfCompartments::remove(const std::string& sid) "
  /**
   * Removes item in this {@link ListOfCompartments} items with the given identifier.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * If none of the items in this list have the identifier <code>sid</code>, then 
   * <code>NULL</code> is returned.
   * <p>
   * @param sid the identifier of the item to remove
   * <p>
   * @return the item removed.  As mentioned above, the caller owns the
   * returned item.
   */
 public";

%javamethodmodifiers ListOfCompartments::getElementPosition() const "
  /**
   * Get the ordinal position of this element in the containing object
   * (which in this case is the {@link Model} object).
   * <p>
   * The ordering of elements in the XML form of SBML is generally fixed
   * for most components in SBML.  So, for example, the {@link ListOfCompartments}
   * in a model is (in SBML Level 2 Version 4) the fifth ListOf___.
   * (However, it differs for different Levels and Versions of SBML.)
   * <p>
   * @return the ordinal position of the element with respect to its
   * siblings, or <code>-1</code> (default) to indicate the position is not significant.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers ListOfCompartments::createObject(XMLInputStream& stream) "
  /**
   * @return the SBML object corresponding to next {@link XMLToken} in the
   * XMLInputStream or NULL if the token was not recognized.
   <p>
 * @deprecated libSBML internal
 */
 public";

%typemap(javaimports) Species "
/** 
 * LibSBML implementation of SBML\'s Species construct.
 * <p>
 * A <em>species</em> refers to a pool of reacting entities of a specific
 * <em>species type</em> that take part in reactions and are located in a
 * specific <em>compartment</em>.  The {@link Species} data structure is intended to
 * represent these pools.  A {@link Species} definition has several parts: an
 * optional identifier (defined by the attribute \'id\'), an optional name
 * (defined by the attribute \'name\'), a required attribute \'compartment\',
 * and optional attributes \'speciesType\', \'initialAmount\',
 * \'initialConcentration\', \'substanceUnits\', \'hasOnlySubstanceUnits\',
 * \'boundaryCondition\', \'charge\' and \'constant\'.  These various parts are
 * described next.
 * <p>
 * As with other major structures in SBML, {@link Species} has a mandatory
 * attribute, \'id\', used to give the species type an identifier in the
 * model.  The identifier must be a text string conforming to the identifer
 * syntax permitted in SBML.  {@link Species} also has an optional \'name\'
 * attribute, of type <code>string</code>.  The \'id\' and \'name\' must be used
 * according to the guidelines described in the SBML specification (e.g.,
 * Section 3.3 in the Level&nbsp;2 Version&nbsp;4 specification).
 * <p>
 * The required attribute \'compartment\' is used to identify the compartment
 * in which the species is located.  The attribute\'s value must be the
 * identifier of an existing {@link Compartment} structure.  It is important to
 * note that there is no default value for the \'compartment\' attribute on
 * {@link Species}; every species in an SBML model must be assigned a compartment,
 * and consequently, a model must define at least one compartment if that
 * model contains any species.
 * <p>
 * Each species in a model may optionally be designated as belonging to a
 * particular species type.  The optional attribute \'speciesType\' is used
 * to identify the species type of the chemical entities that make up the
 * pool represented by the {@link Species} structure.  The attribute\'s value must
 * be the identifier of an existing {@link SpeciesType} structure.  If the
 * \'speciesType\' attribute is not present on a particular species
 * definition, it means the pool contains chemical entities of a type
 * unique to that pool; in effect, a virtual species type is assumed for
 * that species, and no other species can belong to that species type.  The
 * value of \'speciesType\' attributes on species have no effect on the
 * numerical interpretation of a model; simulators and other numerical
 * analysis software may ignore \'speciesType\' attributes.
 * <p>
 * There can be only one species of a given species type in any given
 * compartment of a model.  More specifically, for all {@link Species} structures
 * having a value for the \'speciesType\' attribute, the pair
 * <center>
 * (\'speciesType\' attribute value, \'compartment\' attribute value)
 * </center>
 * <p>
 * must be unique across the set of all {@link Species} structures in a model.
 * <p>
 * <p>
 * <h2>The initial amount and concentration of a species</h2>
 * <p>
 * The optional attributes \'initialAmount\' and \'initialConcentration\', both
 * having a data type of <code>double</code>, are used to set the initial quantity of
 * the species in the compartment where the species is located.  These
 * attributes are mutually exclusive; i.e., <em>only one</em> can have a
 * value on any given instance of a {@link Species} structure.  Missing
 * \'initialAmount\' and \'initialConcentration\' values implies that their
 * values either are unknown, or to be obtained from an external source, or
 * determined by an {@link InitialAssignment} or {@link Rule} object elsewhere in the
 * model.  In the case where a species\' compartment has a
 * \'spatialDimensions\' value of <code>0</code> (zero), the species cannot have a
 * value for \'initialConcentration\' because the concepts of concentration
 * and density break down when a container has zero dimensions.
 * <p>
 * A species\' initial quantity is set by the \'initialAmount\' or
 * \'initialConcentration\' attributes exactly once.  If the species\'
 * \'constant\' attribute is <code>true</code> (the default), then the size is fixed
 * and cannot be changed except by an {@link InitialAssignment}.  These methods
 * differ in that the \'initialAmount\' and \'initialConcentration\' attributes
 * can only be used to set the species quantity to a literal scalar value,
 * whereas {@link InitialAssignment} allows the value to be set using an arbitrary
 * mathematical expression.  If the species\' \'constant\' attribute is 
 * <code>false</code>, the species\' quantity value may be overridden by an
 * {@link InitialAssignment} or changed by {@link AssignmentRule} or {@link AlgebraicRule}, and in
 * addition, for <em>t &lt; 0</em>, it may also be changed by a {@link RateRule} or
 * {@link Event}. (However, some constructs are mutually exclusive; see the SBML
 * specification for more details.)  It is not an error to define
 * \'initialAmount\' or \'initialConcentration\' on a species and also redefine
 * the value using an {@link InitialAssignment}, but the \'initialAmount\' or
 * \'initialConcentration\' setting in that case is ignored.
 * <p>
 * <h2>The units of a species\' amount or concentration</h2>
 * <p>
 * The units associated with a species\' quantity, referred to as the
 * <em>units of the species</em>, are determined via the optional
 * attributes \'substanceUnits\' and \'hasOnlySubstanceUnits\', in combination
 * with the units of the size defined for the compartment object in which
 * the species are located.  The way this is done is as follows.
 * <p>
 * The units of the value in the \'initialConcentration\' attribute are 
 * <em>substance</em>/<em>size</em> units, where the units of <em>substance</em> are those
 * defined by the \'substanceUnits\' attribute and the <em>size</em> units are
 * those given in the definition of the size of the {@link Compartment} in which
 * the species is located.  The units of the value in the \'initialAmount\'
 * attribute are determined by the \'substanceUnits\' attribute of the
 * species structure.  The role of the attribute \'hasOnlySubstanceUnits\' is
 * to indicate whether the units of the species, when the species
 * identifier appears in mathematical formulas, are intended to be
 * concentration or amount.  The attribute takes on boolean values and
 * defaults to <code>false</code>.  Although it may seem as though this intention
 * could be determined based on whether \'initialConcentration\' or
 * \'initialAmount\' is set, the fact that these two attributes are optional
 * means that a separate flag is needed.  (Consider the situation where
 * neither is set, and instead the species\' quantity is established by an
 * {@link InitialAssignment} or {@link AssignmentRule}.)
 * <p>
 * The possible values of <em>units of the species</em> are summarized in
 * the following table.  (The dependence on the number of spatial
 * dimensions of the compartment is due to the fact that a zero-dimensional
 * compartment cannot support concentrations or densities.)
 * <p>
 * <center>
<table border=\'0\' class=\'text-table width80 normal-font alt-row-colors\'>
<caption class=\'top-caption\'>Interpretation of species\' units.</caption>
 <tr>
     <th align=\'left\' valign=\'bottom\'>
       Value of<br><code>hasOnlySubstanceUnits</code>
     </th>
     <th align=\'left\' valign=\'bottom\'>
       <em>Units of the species</em> when<br><code>spatialDimensions</code> is greater than 0
     </th>
     <th align=\'left\' valign=\'bottom\'>
       <em>Units of the species</em> when<br><code>spatialDimensions</code> is 0
     </th>
 </tr>
 <tr>
     <td><code>false</code> (default)</td>
     <td><em>substance/size</em></td>
     <td><em>substance</em></td>
 </tr>
 <tr>
     <td><code>true</code> (default)</td>
     <td><em>substance</em></td>
     <td><em>substance</em></td>
 </tr>
</table>
</center>

 * <p>
 * The value assigned to \'substanceUnits\' must be chosen from one of the
 * following possibilities: one of the base unit identifiers defined in
 * SBML; the built-in unit identifier <code>\'substance\'</code>; or the identifier of
 * a new unit defined in the list of unit definitions in the enclosing
 * {@link Model} structure.  The chosen units for \'substanceUnits\' must be be 
 * <code>\'dimensionless\'</code>, <code>\'mole\'</code>, <code>\'item\'</code>, <code>\'kilogram\'</code>, <code>\'gram\'</code>, or
 * units derived from these.  The \'substanceUnits\' attribute defaults to
 * the the built-in unit <code>\'substance\'</code>.
 * <p>
 * The <em>units of the species</em> are used in the following ways:
 * <ul>
 * <li> The species identifier has these units when the identifier appears
 * as a numerical quantity in a mathematical formula expressed in MathML.
 * <p>
 * <li> The \'math\' subelement of an {@link AssignmentRule} or {@link InitialAssignment}
 * referring to this species must have identical units.
 * <p>
 * <li> In {@link RateRule} structures that set the rate of change of the species\'
 * quantity, the units of the rule\'s \'math\' subelement must be identical to
 * the <em>units of the species</em> divided by the model\'s <em>time</em> units.
 * </ul>
 * <p>
 * <h2>The \'constant\' and \'boundaryCondition\' attributes</h2>
 * <p>
 * The {@link Species} structure has two optional boolean attributes named
 * \'constant\' and \'boundaryCondition\', used to indicate whether and how the
 * quantity of that species can vary during a simulation.  The following
 * table shows how to interpret the combined values of these attributes.
 * <p>
 * <center>
<table border=\'0\' class=\'text-table width80 normal-font alt-row-colors\'>
<caption class=\'top-caption\'>Interpretation
of species\' <code>constant</code> and <code>boundaryCondition</code>
attributes.</caption>
 <tr>
     <th align=\'left\' valign=\'bottom\'>
       <code>constant</code><br>value
     </th>
     <th align=\'left\' valign=\'bottom\'>
       <code>boundaryCondition</code><br>value
     </th>
     <th align=\'left\' valign=\'bottom\'>
       Can have<br>assignment<br>or rate rule?
     </th>
     <th align=\'left\' valign=\'bottom\'>
       Can be<br>reactant<br>or product?
     </th>
     <th align=\'left\' valign=\'bottom\'>
       {@link Species}\' quantity<br>can be changed by
     </th>
 </tr>
 <tr>
     <td><code>true</code></td>
     <td><code>true</code></td>
     <td>no</td>
     <td>yes</td>
     <td>(never changes)</td>
 </tr>
 <tr>
     <td><code>false</code></td>
     <td><code>true</code></td>
     <td>yes</td>
     <td>yes</td>
     <td>rules and events</td>
 </tr>
 <tr>
     <td><code>true</code></td>
     <td><code>false</code></td>
     <td>no</td>
     <td>no</td>
     <td>(never changes)</td>
 </tr>
 <tr>
     <td><code>false</code></td>
     <td><code>false</code></td>
     <td>yes</td>
     <td>yes</td>
     <td>reactions <em>or</em> rules (but not both at the same time), and events</td>
 </tr>
</table>
</center>

 * <p>
 * By default, when a species is a product or reactant of one or more
 * reactions, its quantity is determined by those reactions.  In SBML, it
 * is possible to indicate that a given species\' quantity is <em>not</em>
 * determined by the set of reactions even when that species occurs as a
 * product or reactant; i.e., the species is on the <em>boundary</em> of
 * the reaction system, and its quantity is not determined by the
 * reactions.  The boolean attribute \'boundaryCondition\' can be used to
 * indicate this.  The value of the attribute defaults to <code>false</code>,
 * indicating the species <em>is</em> part of the reaction system.
 * <p>
 * The \'constant\' attribute indicates whether the species\' quantity can be
 * changed at all, regardless of whether by reactions, rules, or constructs
 * other than {@link InitialAssignment}.  The default value is <code>false</code>, indicating
 * that the species\' quantity can be changed, since the purpose of most
 * simulations is precisely to calculate changes in species quantities.
 * Note that the initial quantity of a species can be set by an
 * {@link InitialAssignment} irrespective of the value of the \'constant\' attribute.
 * <p>
 * In practice, a \'boundaryCondition\' value of <code>true</code> means a differential
 * equation derived from the reaction definitions should not be generated
 * for the species.  However, the species\' quantity may still be changed by
 * {@link AssignmentRule}, {@link RateRule}, {@link AlgebraicRule}, {@link Event}, and {@link InitialAssignment}
 * constructs if its \'constant\' attribute is <code>false</code>.  Conversely, if the
 * species\' \'constant\' attribute is <code>true</code>, then its value cannot be
 * changed by anything except {@link InitialAssignment}.
 * <p>
 * A species having \'boundaryCondition\'=<code>false</code> and \'constant\'=<code>false</code>
 * can appear as a product and/or reactant of one or more reactions in the
 * model.  If the species is a reactant or product of a reaction, it must
 * <em>not</em> also appear as the target of any {@link AssignmentRule} or {@link RateRule}
 * structure in the model.  If instead the species has
 * \'boundaryCondition\'=<code>false</code> and \'constant\'=<code>true</code>, then it cannot
 * appear as a reactant or product, or as the target of any
 * {@link AssignmentRule}, {@link RateRule} or {@link EventAssignment} structure in the model.
 * <p>
 * @warning In versions of SBML Level&nbsp;2 before Version&nbsp;3, the class
 * {@link Species} included an attribute called \'spatialSizeUnits, which allowed
 * explicitly setting the units of size for initial concentration.  LibSBML
 * retains this attribute for compatibility with older definitions of
 * Level&nbsp;2, but its use is strongly discouraged because it is
 * incompatible with Level&nbsp;2 Version&nbsp;3 andLevel&nbsp;2 Version&nbsp;4.
 * <p>
 * <!-- leave this next break as-is to work around some doxygen bug -->
 */
"

%typemap(javaimports) ListOfSpecies "
/** 
 * LibSBML implementation of SBML\'s ListOfSpecies construct.
 * <p>
 * The various ListOf___ classes in SBML are merely containers used for
 * organizing the main components of an SBML model.  All are derived from
 * the abstract class {@link SBase}, and inherit the various attributes and
 * subelements of {@link SBase}, such as \'metaid\' as and \'annotation\'.  The
 * ListOf___ classes do not add any attributes of their own.
 * <p>
 * The relationship between the lists and the rest of an SBML model is
 * illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):
 * <p>
 * <center><img src=\'listof-illustration.jpg\'></center><br>
 * 
 * <p>
 * Readers may wonder about the motivations for using the ListOf___
 * containers.  A simpler approach in XML might be to place the components
 * all directly at the top level of the model definition.  We chose instead
 * to group them within XML elements named after {@link ListOf}<em>Classname</em>,
 * in part because we believe this helps organize the components and makes
 * visual reading of models in XML easier.  More importantly, the fact that
 * the container classes are derived from {@link SBase} means that software tools
 * can add information about the lists themselves into each list
 * container\'s \'annotation\'.
 * <p>
 * @see ListOfFunctionDefinitions
 * @see ListOfUnitDefinitions
 * @see ListOfCompartmentTypes
 * @see ListOfSpeciesTypes
 * @see ListOfCompartments
 * @see ListOfSpecies
 * @see ListOfParameters
 * @see ListOfInitialAssignments
 * @see ListOfRules
 * @see ListOfConstraints
 * @see ListOfReactions
 * @see ListOfEvents
 */
"

%javamethodmodifiers Species::Species(unsigned int level, unsigned int version) "
  /**
   * Creates a new {@link Species} using the given SBML <code>level</code> and <code>version</code>
   * values.
   * <p>
   * @param level a long integer, the SBML Level to assign to this {@link Species}
   * <p>
   * @param version a long integer, the SBML Version to assign to this
   * {@link Species}
   * <p>
   * @note Once a {@link Species} has been added to an {@link SBMLDocument}, the <code>level</code>,
   * <code>version</code> for the document <em>override</em> those used
   * to create the {@link Species}.  Despite this, the ability to supply the values
   * at creation time is an important aid to creating valid SBML.  Knowledge of
   * the intented SBML Level and Version determine whether it is valid to
   * assign a particular value to an attribute, or whether it is valid to add
   * an object to an existing {@link SBMLDocument}.
   */
 public";

%javamethodmodifiers Species::Species(SBMLNamespaces* sbmlns) "
  /**
   * Creates a new {@link Species} using the given {@link SBMLNamespaces} object
   * <code>sbmlns</code>.
   * <p>
   * The {@link SBMLNamespaces} object encapsulates SBML Level/Version/namespaces
   * information.  It is used to communicate the SBML Level, Version, and
   * (in Level&nbsp;3) packages used in addition to SBML Level&nbsp; Core.
   * A common approach to using this class constructor is to create an
   * {@link SBMLNamespaces} object somewhere in a program, once, then pass it to
   * object constructors such as this one when needed.
   * <p>
   * It is worth emphasizing that although this constructor does not take
   * an identifier argument, in SBML Level&nbsp;2 and beyond, the \'id\'
   * (identifier) attribute of a {@link Species} is required to have a value.
   * Thus, callers are cautioned to assign a value after calling this
   * constructor.  Setting the identifier can be accomplished using the
   * method {@link SBase#setId(String id)} .
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object.
   * <p>
   * @note Once a {@link Species} has been added to an {@link SBMLDocument}, the <code>level</code>,
   * <code>version</code> and <code>xmlns</code> namespaces for the document <em>override</em> those used
   * to create the {@link Species}.  Despite this, the ability to supply the values
   * at creation time is an important aid to creating valid SBML.  Knowledge of
   * the intented SBML Level and Version determine whether it is valid to
   * assign a particular value to an attribute, or whether it is valid to add
   * an object to an existing {@link SBMLDocument}.
   */
 public";

%javamethodmodifiers Species::Species(const Species& orig) "
  /**
  * Copy constructor; creates a copy of this {@link Species}.
  */
 public";

%javamethodmodifiers Species::accept(SBMLVisitor& v) const "
  /**
   * Accepts the given {@link SBMLVisitor} for this instance of {@link Species}.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used.
   * <p>
   * @return the result of calling <code>v.visit()</code>.
   */
 public";

%javamethodmodifiers Species::clone() const "
  /**
   * Creates and returns a deep copy of this {@link Species}.
   * <p>
   * @return a (deep) copy of this {@link Species}.
   */
 public";

%javamethodmodifiers Species::initDefaults "
  /**
   * Initializes the fields of this {@link Species} to the defaults defined
   * in the specification of the relevant Level/Version of SBML.
   * <ul>
   * <li> sets \'boundaryCondition\' to <code>1</code> (true)
   * <li> (Level&nbsp;2 only) sets \'constant\' to <code>0</code> (false)
   * <li> (Level&nbsp;2 only) sets \'hasOnlySubstanceUnits\' to <code>0</code> (false)
   * </ul>
   */
 public";

%javamethodmodifiers Species::getId() const "
  /**
   * Returns the value of the \'id\' attribute of this {@link Species}.
   * <p>
   * @return the id of this {@link Species}.
   */
 public";

%javamethodmodifiers Species::getName() const "
  /**
   * Returns the value of the \'name\' attribute of this {@link Species}.
   * <p>
   * @return the name of this {@link Species}.
   */
 public";

%javamethodmodifiers Species::getSpeciesType() const "
  /**
   * Get the species type of this {@link Species}, as indicated by the
   * {@link Species} object\'s \'speciesType\' attribute value.
   * <p>
   * @return the value of the \'speciesType\' attribute of this
   * {@link Species} as a string.
   */
 public";

%javamethodmodifiers Species::getCompartment() const "
  /**
   * Get the compartment in which this species is located.
   * <p>
   * @return the value of the \'compartment\' attribute of this {@link Species}, as a
   * string.
   */
 public";

%javamethodmodifiers Species::getInitialAmount() const "
  /**
   * Get the value of the \'initialAmount\' attribute.
   * <p>
   * @return the initialAmount of this {@link Species}, as a float-point number.
   */
 public";

%javamethodmodifiers Species::getInitialConcentration() const "
  /**
   * Get the value of the \'initialConcentration\' attribute.
   * <p>
   * @return the initialConcentration of this {@link Species},, as a float-point
   * number.
   */
 public";

%javamethodmodifiers Species::getSubstanceUnits() const "
  /**
   * Get the value of the \'substanceUnit\' attribute.
   * <p>
   * @return the substanceUnits of this {@link Species}, as a string.
   */
 public";

%javamethodmodifiers Species::getSpatialSizeUnits() const "
  /**
   * Get the value of the \'spatialSizeUnits\' attribute.
   * <p>
   * @return the spatialSizeUnits of this {@link Species}.
   * <p>
   * @warning In versions of SBML Level~2 before Version&nbsp;3, the class
   * {@link Species} included an attribute called \'spatialSizeUnits\', which allowed
   * explicitly setting the units of size for initial concentration.  This
   * attribute was removed in SBML Level&nbsp;2 Version&nbsp;3.  LibSBML
   * retains this attribute for compatibility with older definitions of
   * Level&nbsp;2, but its use is strongly discouraged because it is
   * incompatible with Level&nbsp;2 Version&nbsp;3 and Level&nbsp;2 Version&nbsp;4.
   */
 public";

%javamethodmodifiers Species::getUnits() const "
  /**
   * (SBML Level&nbsp;1 only) Get the value of the \'units\' attribute.
   * <p>
   * @return the units of this {@link Species} (L1 only).
   */
 public";

%javamethodmodifiers Species::getHasOnlySubstanceUnits() const "
  /**
   * Get the value of the \'hasOnlySubstanceUnits\' attribute.
   * <p>
   * @return <code>true</code> if this {@link Species}\' \'hasOnlySubstanceUnits\' attribute
   * value is nonzero, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers Species::getBoundaryCondition() const "
  /**
   * Get the value of the \'boundaryCondition\' attribute.
   * <p>
   * @return <code>true</code> if this {@link Species}\' \'boundaryCondition\' attribute value
   * is nonzero, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers Species::getCharge() const "
  /**
   * Get the value of the \'charge\' attribute.
   * <p>
   * @return the charge of this {@link Species}.
   * <p>
   * @note Beginning in SBML Level&nbsp;2 Version&nbsp;2, the \'charge\'
   * attribute on {@link Species} is deprecated and its use strongly discouraged.
   * Its presence is considered a misfeature in earlier definitions of SBML
   * because its implications for the mathematics of a model were never
   * defined, and in any case, no known modeling system ever used it.
   * Instead, models take account of charge values directly in their
   * definitions of species by (for example) having separate species
   * identities for the charged and uncharged versions of the same species.
   * This allows the condition to affect model mathematics directly.
   * LibSBML retains this method for easier compatibility with SBML
   * Level&nbsp;1.
   */
 public";

%javamethodmodifiers Species::getConstant() const "
  /**
   * Get the value of the \'constant\' attribute.
   * <p>
   * @return <code>true</code> if this {@link Species}\'s \'constant\' attribute value is
   * nonzero, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers Species::isSetId() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link Species}\'s \'id\' attribute has been set.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the \'id\' attribute of this {@link Species} has been
   * set, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers Species::isSetName() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link Species}\'s \'name\' attribute has been set.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the \'name\' attribute of this {@link Species} has been
   * set, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers Species::isSetSpeciesType() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link Species}\'s \'speciesType\' attribute has been set.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the \'speciesType\' attribute of this {@link Species} has
   * been set, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers Species::isSetCompartment() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link Species}\'s \'compartment\' attribute has been set.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the \'compartment\' attribute of this {@link Species} has
   * been set, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers Species::isSetInitialAmount() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link Species}\'s \'initialAmount\' attribute has been set.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the \'initialAmount\' attribute of this {@link Species} has
   * been set, <code>false</code> otherwise.
   * <p>
   * @note In SBML Level&nbsp;1, {@link Species}\' \'initialAmount\' is required and
   * therefore <em>should always be set</em>.  (However, in Level&nbsp;1, the
   * attribute has no default value either, so this method will not return
   * <code>true</code> until a value has been assigned.)  In SBML Level&nbsp;2,
   * \'initialAmount\' is optional and as such may or may not be set.
   */
 public";

%javamethodmodifiers Species::isSetInitialConcentration() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link Species}\'s \'initialConcentration\' attribute has been set.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the \'initialConcentration\' attribute of this {@link Species} has
   * been set, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers Species::isSetSubstanceUnits() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link Species}\'s \'substanceUnits\' attribute has been set.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the \'substanceUnits\' attribute of this {@link Species} has
   * been set, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers Species::isSetSpatialSizeUnits() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link Species}\'s \'spatialSizeUnits\' attribute has been set.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the \'spatialSizeUnits\' attribute of this {@link Species} has
   * been set, <code>false</code> otherwise.
   * <p>
   * @warning In versions of SBML Level~2 before Version&nbsp;3, the class
   * {@link Species} included an attribute called \'spatialSizeUnits\', which allowed
   * explicitly setting the units of size for initial concentration.  This
   * attribute was removed in SBML Level&nbsp;2 Version&nbsp;3.  LibSBML
   * retains this attribute for compatibility with older definitions of
   * Level&nbsp;2, but its use is strongly discouraged because it is
   * incompatible with Level&nbsp;2 Version&nbsp;3 and Level&nbsp;2 Version&nbsp;4.
   */
 public";

%javamethodmodifiers Species::isSetUnits() const "
  /**
   * (SBML Level&nbsp;1 only) Predicate returning <code>true</code> or <code>false</code> depending
   * on whether this {@link Species}\'s \'units\' attribute has been set.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the \'units\' attribute of this {@link Species} has
   * been set, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers Species::isSetCharge() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link Species}\'s \'charge\' attribute has been set.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the \'charge\' attribute of this {@link Species} has
   * been set, <code>false</code> otherwise.
   * <p>
   * @note Beginning in SBML Level&nbsp;2 Version&nbsp;2, the \'charge\'
   * attribute on {@link Species} in SBML is deprecated and its use strongly
   * discouraged.  Its presence is considered a misfeature in earlier
   * definitions of SBML because its implications for the mathematics of a
   * model were never defined, and in any case, no known modeling system
   * ever used it.  Instead, models take account of charge values directly
   * in their definitions of species by (for example) having separate
   * species identities for the charged and uncharged versions of the same
   * species.  This allows the condition to affect model mathematics
   * directly.  LibSBML retains this method for easier compatibility with
   * SBML Level&nbsp;1.
   */
 public";

%javamethodmodifiers Species::setId(const std::string& sid) "
  /**
   * Sets the value of the \'id\' attribute of this {@link Species}.
   * <p>
   * The string <code>sid</code> is copied.  Note that SBML has strict requirements
   * for the syntax of identifiers.  The following is summary of the
   * definition of the SBML identifier type <code>SId</code> (here expressed in an
   * extended form of BNF notation):
   * <div class=\'fragment\'><pre>
   *   letter .= \'a\'..\'z\',\'A\'..\'Z\'
   *   digit  .= \'0\'..\'9\'
   *   idChar .= letter | digit | \'_\'
   *   SId    .= ( letter | \'_\' ) idChar</pre></div>
   * The equality of SBML identifiers is determined by an exact character
   * sequence match; i.e., comparisons must be performed in a
   * case-sensitive manner.  In addition, there are a few conditions for
   * the uniqueness of identifiers in an SBML model.  Please consult the
   * SBML specifications for the exact formulations.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param sid the string to use as the identifier of this {@link Species}
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   */
 public";

%javamethodmodifiers Species::setName(const std::string& name) "
  /**
   * Sets the value of the \'name\' attribute of this {@link Species}.
   * <p>
   * The string in <code>name</code> is copied.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param name the new name for the {@link Species}
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   */
 public";

%javamethodmodifiers Species::setSpeciesType(const std::string& sid) "
  /**
   * Sets the \'speciesType\' attribute of this {@link Species}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param sid the identifier of a {@link SpeciesType} object defined elsewhere
   * in this {@link Model}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   * <li> LIBSBML_UNEXPECTED_ATTRIBUTE
   */
 public";

%javamethodmodifiers Species::setCompartment(const std::string& sid) "
  /**
   * Sets the \'compartment\' attribute of this {@link Species}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param sid the identifier of a {@link Compartment} object defined elsewhere
   * in this {@link Model}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   */
 public";

%javamethodmodifiers Species::setInitialAmount(double value) "
  /**
   * Sets the \'initialAmount\' attribute of this {@link Species} and marks the field
   * as set.
   * <p>
   * This method also unsets the \'initialConcentration\' attribute.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param value the value to which the \'initialAmount\' attribute should
   * be set.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   */
 public";

%javamethodmodifiers Species::setInitialConcentration(double value) "
  /**
   * Sets the \'initialConcentration\' attribute of this {@link Species} and marks
   * the field as set.
   * <p>
   * This method also unsets the \'initialAmount\' attribute.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param value the value to which the \'initialConcentration\' attribute
   * should be set.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_UNEXPECTED_ATTRIBUTE
   */
 public";

%javamethodmodifiers Species::setSubstanceUnits(const std::string& sid) "
  /**
   * Sets the \'substanceUnits\' attribute of this {@link Species}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param sid the identifier of the unit to use.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   */
 public";

%javamethodmodifiers Species::setSpatialSizeUnits(const std::string& sid) "
  /**
   * Sets the \'spatialSizeUnits\' attribute of this {@link Species}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param sid the identifier of the unit to use.
   * <p>
   * @warning In versions of SBML Level~2 before Version&nbsp;3, the class
   * {@link Species} included an attribute called \'spatialSizeUnits\', which allowed
   * explicitly setting the units of size for initial concentration.  This
   * attribute was removed in SBML Level&nbsp;2 Version&nbsp;3.  LibSBML
   * retains this attribute for compatibility with older definitions of
   * Level&nbsp;2, but its use is strongly discouraged because it is
   * incompatible with Level&nbsp;2 Version&nbsp;3 and Level&nbsp;2 Version&nbsp;4.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   * <li> LIBSBML_UNEXPECTED_ATTRIBUTE
   */
 public";

%javamethodmodifiers Species::setUnits(const std::string& sname) "
  /**
   * (SBML Level&nbsp;1 only) Sets the units of this {@link Species}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param sname the identifier of the unit to use.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
  */
 public";

%javamethodmodifiers Species::setHasOnlySubstanceUnits(bool value) "
  /**
   * Sets the \'hasOnlySubstanceUnits\' attribute of this {@link Species}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param value boolean value for the \'hasOnlySubstanceUnits\' attribute.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_UNEXPECTED_ATTRIBUTE
   */
 public";

%javamethodmodifiers Species::setBoundaryCondition(bool value) "
  /**
   * Sets the \'boundaryCondition\' attribute of this {@link Species}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param value boolean value for the \'boundaryCondition\' attribute.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   */
 public";

%javamethodmodifiers Species::setCharge(int value) "
  /**
   * Sets the \'charge\' attribute of this {@link Species}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param value an integer to which to set the \'charge\' to.
   * <p>
   * @note Beginning in SBML Level&nbsp;2 Version&nbsp;2, the \'charge\'
   * attribute on {@link Species} in SBML is deprecated and its use strongly
   * discouraged.  Its presence is considered a misfeature in earlier
   * definitions of SBML because its implications for the mathematics of a
   * model were never defined, and in any case, no known modeling system
   * ever used it.  Instead, models take account of charge values directly
   * in their definitions of species by (for example) having separate
   * species identities for the charged and uncharged versions of the same
   * species.  This allows the condition to affect model mathematics
   * directly.  LibSBML retains this method for easier compatibility with
   * SBML Level&nbsp;1.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_UNEXPECTED_ATTRIBUTE
   */
 public";

%javamethodmodifiers Species::setConstant(bool value) "
  /**
   * Sets the \'constant\' attribute of this {@link Species}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param value a boolean value for the \'constant\' attribute
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_UNEXPECTED_ATTRIBUTE
   */
 public";

%javamethodmodifiers Species::unsetName "
  /**
   * Unsets the value of the \'name\' attribute of this {@link Species}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_OPERATION_FAILED
   */
 public";

%javamethodmodifiers Species::unsetSpeciesType "
  /**
   * Unsets the \'speciesType\' attribute value of this {@link Species}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_OPERATION_FAILED
   */
 public";

%javamethodmodifiers Species::unsetInitialAmount "
  /**
   * Unsets the \'initialAmount\' attribute value of this {@link Species}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   */
 public";

%javamethodmodifiers Species::unsetInitialConcentration "
  /**
   * Unsets the \'initialConcentration\' attribute value of this {@link Species}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   */
 public";

%javamethodmodifiers Species::unsetSubstanceUnits "
  /**
   * Unsets the \'substanceUnits\' attribute value of this {@link Species}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_OPERATION_FAILED
   */
 public";

%javamethodmodifiers Species::unsetSpatialSizeUnits "
  /**
   * Unsets the \'spatialSizeUnits\' attribute value of this {@link Species}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_OPERATION_FAILED
   * <p>
   * @warning In versions of SBML Level~2 before Version&nbsp;3, the class
   * {@link Species} included an attribute called \'spatialSizeUnits\', which allowed
   * explicitly setting the units of size for initial concentration.  This
   * attribute was removed in SBML Level&nbsp;2 Version&nbsp;3.  LibSBML
   * retains this attribute for compatibility with older definitions of
   * Level&nbsp;2, but its use is strongly discouraged because it is
   * incompatible with Level&nbsp;2 Version&nbsp;3 and Level&nbsp;2 Version&nbsp;4.
   */
 public";

%javamethodmodifiers Species::unsetUnits "
  /**
   * (SBML Level&nbsp;1 only) Unsets the \'units\' attribute value of this {@link Species}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_OPERATION_FAILED
   */
 public";

%javamethodmodifiers Species::unsetCharge "
  /**
   * Unsets the \'charge\' attribute value of this {@link Species}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_OPERATION_FAILED
   * <p>
   * @note Beginning in SBML Level&nbsp;2 Version&nbsp;2, the \'charge\'
   * attribute on {@link Species} in SBML is deprecated and its use strongly
   * discouraged.  Its presence is considered a misfeature in earlier
   * definitions of SBML because its implications for the mathematics of a
   * model were never defined, and in any case, no known modeling system
   * ever used it.  Instead, models take account of charge values directly
   * in their definitions of species by (for example) having separate
   * species identities for the charged and uncharged versions of the same
   * species.  This allows the condition to affect model mathematics
   * directly.  LibSBML retains this method for easier compatibility with
   * SBML Level&nbsp;1.
   */
 public";

%javamethodmodifiers Species::getDerivedUnitDefinition "
  /**
   * Constructs and returns a {@link UnitDefinition} that corresponds to the units
   * of this {@link Species}\' amount or concentration.
   * <p>
   * {@link Species} in SBML have an attribute (\'substanceUnits\') for declaring the
   * units of measurement intended for the species\' amount or concentration
   * (depending on which one applies).  In the absence of a value given for
   * \'substanceUnits\', the units are taken from the enclosing {@link Model}\'s
   * definition of <code>\'substance\'</code> or <code>\'substance\'</code>/<em>(size of the
   * compartment)</em> in which the species is located, or finally, if
   * these are not redefined by the {@link Model}, the relevant SBML default units
   * for those quantities.  Following that procedure, the method
   * getDerivedUnitDefinition() returns a {@link UnitDefinition} based on the
   * interpreted units of this species\'s amount or concentration.
   * <p>
   * Note that the functionality that facilitates unit analysis depends 
   * on the model as a whole.  Thus, in cases where the object has not 
   * been added to a model or the model itself is incomplete,
   * unit analysis is not possible and this method will return NULL.
   * <p>
   * Note also that unit declarations for {@link Species} are in terms of the 
   * <em>identifier</em> of a unit, but this method returns a {@link UnitDefinition} object,
   * not a unit identifier.  It does this by constructing an appropriate
   * {@link UnitDefinition}.  Callers may find this particularly useful when used
   * in conjunction with the helper methods on {@link UnitDefinition} for comparing
   * different {@link UnitDefinition} objects.
   * <p>
   * In SBML Level&nbsp;2 specifications prior to Version&nbsp;3, {@link Species}
   * includes an additional attribute named \'spatialSizeUnits\', which
   * allows explicitly setting the units of size for initial concentration.
   * The getDerivedUnitDefinition() takes this into account for models
   * expressed in SBML Level&nbsp;2 Versions&nbsp;1 and&nbsp;2.
   * <p>
   * @return a {@link UnitDefinition} that expresses the units of this 
   * {@link Species}.
   * <p>
   * @see #getSubstanceUnits()
   */
 public";

%javamethodmodifiers Species::getDerivedUnitDefinition() const "
  /**
   * Constructs and returns a {@link UnitDefinition} that corresponds to the units
   * of this {@link Species}\' amount or concentration.
   * <p>
   * {@link Species} in SBML have an attribute (\'substanceUnits\') for declaring the
   * units of measurement intended for the species\' amount or concentration
   * (depending on which one applies).  In the absence of a value given for
   * \'substanceUnits\', the units are taken from the enclosing {@link Model}\'s
   * definition of <code>\'substance\'</code> or <code>\'substance\'</code>/<em>(size of the
   * compartment)</em> in which the species is located, or finally, if
   * these are not redefined by the {@link Model}, the relevant SBML default units
   * for those quantities.  Following that procedure, the method
   * getDerivedUnitDefinition() returns a {@link UnitDefinition} based on the
   * interpreted units of this species\'s amount or concentration.
   * <p>
   * Note that the functionality that facilitates unit analysis depends 
   * on the model as a whole.  Thus, in cases where the object has not 
   * been added to a model or the model itself is incomplete,
   * unit analysis is not possible and this method will return NULL.
   * <p>
   * Note also that unit declarations for {@link Species} are in terms of the 
   * <em>identifier</em> of a unit, but this method returns a {@link UnitDefinition} object,
   * not a unit identifier.  It does this by constructing an appropriate
   * {@link UnitDefinition}.  Callers may find this particularly useful when used
   * in conjunction with the helper methods on {@link UnitDefinition} for comparing
   * different {@link UnitDefinition} objects.
   * <p>
   * In SBML Level&nbsp;2 specifications prior to Version&nbsp;3, {@link Species}
   * includes an additional attribute named \'spatialSizeUnits\', which
   * allows explicitly setting the units of size for initial concentration.
   * The getDerivedUnitDefinition() takes this into account for models
   * expressed in SBML Level&nbsp;2 Versions&nbsp;1 and&nbsp;2.
   * <p>
   * @return a {@link UnitDefinition} that expresses the units of this 
   * {@link Species}.
   * <p>
   * @see #getSubstanceUnits()
   */
 public";

%javamethodmodifiers Species::getTypeCode() const "
  /**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for this object, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers Species::getElementName() const "
  /**
   * Returns the XML element name of this object, which for {@link Species}, is
   * always <code>\'species\'</code>.
   * <p>
   * @return the name of this element, i.e., <code>\'species\'</code>.
   */
 public";

%javamethodmodifiers Species::hasRequiredAttributes() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link Species} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link Species} object are:
   * id (name L1); compartment; initialAmount (L1 only)
   * <p>
   * @return a boolean value indicating whether all the required
   * attributes for this object have been defined.
   */
 public";

%javamethodmodifiers Species::Species "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link Species} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link Species} object are:
   * id (name L1); compartment; initialAmount (L1 only)
   * <p>
   * @return a boolean value indicating whether all the required
   * attributes for this object have been defined.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Species::readAttributes(const XMLAttributes& attributes) "
  /**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Species::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Species::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Species::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Species::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Species::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Species::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Species::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Species::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Species::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Species::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Species::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Species::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Species::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Species::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Species::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Species::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Species::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Species::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Species::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Species::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Species::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Species::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Species::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Species::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Species::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Species::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Species::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Species::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Species::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Species::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers ListOfSpecies::clone() const "
  /**
   * Creates and returns a deep copy of this ListOfSpeciess instance.
   * <p>
   * @return a (deep) copy of this ListOfSpeciess.
   */
 public";

%javamethodmodifiers ListOfSpecies::getTypeCode() const "
  /**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for this object, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers ListOfSpecies::getItemTypeCode() const "
  /**
   * Returns the libSBML type code for the objects contained in this {@link ListOf}
   * (i.e., {@link Species} objects, if the list is non-empty).
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for the objects contained in this {@link ListOf}
   * instance, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers ListOfSpecies::getElementName() const "
  /**
   * Returns the XML element name of this object.
   * <p>
   * For ListOfSpeciess, the XML element name is <code>\'listOfSpeciess\'</code>.
   * <p>
   * @return the name of this element, i.e., <code>\'listOfSpeciess\'</code>.
   */
 public";

%javamethodmodifiers ListOfSpecies::get(unsigned int n) "
  /**
   * Get a {@link Species} from the {@link ListOfSpecies}.
   * <p>
   * @param n the index number of the {@link Species} to get.
   * <p>
   * @return the nth {@link Species} in this {@link ListOfSpecies}.
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfSpecies::get "
  /**
   * Get a {@link Species} from the {@link ListOfSpecies}.
   * <p>
   * @param n the index number of the {@link Species} to get.
   * <p>
   * @return the nth {@link Species} in this {@link ListOfSpecies}.
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfSpecies::get(const std::string& sid) "
  /**
   * Get a {@link Species} from the {@link ListOfSpecies}
   * based on its identifier.
   * <p>
   * @param sid a string representing the identifier 
   * of the {@link Species} to get.
   * <p>
   * @return {@link Species} in this {@link ListOfSpecies}
   * with the given id or NULL if no such
   * {@link Species} exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfSpecies::get(const std::string& sid) const "
  /**
   * Get a {@link Species} from the {@link ListOfSpecies}
   * based on its identifier.
   * <p>
   * @param sid a string representing the identifier 
   * of the {@link Species} to get.
   * <p>
   * @return {@link Species} in this {@link ListOfSpecies}
   * with the given id or NULL if no such
   * {@link Species} exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfSpecies::remove(unsigned int n) "
  /**
   * Removes the nth item from this ListOfSpeciess items and returns a pointer to
   * it.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * <p>
   * @param n the index of the item to remove
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfSpecies::remove(const std::string& sid) "
  /**
   * Removes item in this ListOfSpeciess items with the given identifier.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * If none of the items in this list have the identifier <code>sid</code>, then 
   * <code>NULL</code> is returned.
   * <p>
   * @param sid the identifier of the item to remove
   * <p>
   * @return the item removed.  As mentioned above, the caller owns the
   * returned item.
   */
 public";

%javamethodmodifiers ListOfSpecies::getElementPosition() const "
  /**
   * Get the ordinal position of this element in the containing object
   * (which in this case is the {@link Model} object).
   * <p>
   * The ordering of elements in the XML form of SBML is generally fixed
   * for most components in SBML.  So, for example, the ListOfSpeciess in
   * a model is (in SBML Level&nbsp;2 Version&nbsp;4) the sixth
   * ListOf___.  (However, it differs for different Levels and Versions of
   * SBML.)
   * <p>
   * @return the ordinal position of the element with respect to its
   * siblings, or <code>-1</code> (default) to indicate the position is not significant.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers ListOfSpecies::createObject(XMLInputStream& stream) "
  /**
   * @return the SBML object corresponding to next {@link XMLToken} in the
   * XMLInputStream or NULL if the token was not recognized.
   <p>
 * @deprecated libSBML internal
 */
 public";

%typemap(javaimports) Parameter "
/** 
 * LibSBML implementation of SBML\'s Parameter construct.
 * <p>
 * A {@link Parameter} is used in SBML to define a symbol associated with a value;
 * this symbol can then be used in mathematical formulas in a model.  By
 * default, parameters have constant value for the duration of a
 * simulation, and for this reason are called <em>parameters</em> instead of 
 * <em>variables</em> in SBML, although it is crucial to understand that <em>%SBML
 * parameters represent both concepts</em>.  Whether a given SBML
 * parameter is intended to be constant or variable is indicated by the
 * value of its \'constant\' attribute.
 * <p>
 * SBML\'s {@link Parameter} has one required attribute, \'id\', to give the
 * parameter a unique identifier by which other parts of an SBML model
 * definition can refer to it.  A parameter can also have an optional
 * \'name\' attribute of type <code>string</code>.  Identifiers and names must be used
 * according to the guidelines described in the SBML specification (e.g.,
 * Section 3.3 in the Level&nbsp;2 Version&nbsp;4 specification).
 * <p>
 * The optional attribute \'value\' determines the value (of type <code>double</code>)
 * assigned to the identifier.  A missing value for \'value\' implies that
 * the value either is unknown, or to be obtained from an external source,
 * or determined by an initial assignment.  The units associated with the
 * value of the parameter are specified by the attribute named \'units\'.
 * The value assigned to the parameter\'s \'units\' attribute must be chosen
 * from one of the following possibilities: one of the base unit
 * identifiers defined in SBML; one of the built-in unit identifiers 
 * <code>\'substance\'</code>, <code>\'time\'</code>, <code>\'volume\'</code>, <code>\'area\'</code> or <code>\'length\'</code>; or the
 * identifier of a new unit defined in the list of unit definitions in the
 * enclosing {@link Model} structure.  There are no constraints on the units that
 * can be chosen from these sets.  There are no default units for
 * parameters.  Please consult the SBML specification documents for more
 * details about the meanings and implications of the various unit choices.
 * <p>
 * The {@link Parameter} structure has an optional boolean attribute named
 * \'constant\' that indicates whether the parameter\'s value can vary during
 * a simulation.  The attribute\'s default value is <code>true</code>.  A value of 
 * <code>false</code> indicates the parameter\'s value can be changed by {@link Rule} constructs
 * and that the \'value\' attribute is actually intended to be the initial
 * value of the parameter. Parameters local to a reaction (that is, those
 * defined within the {@link KineticLaw} structure of a {@link Reaction}) cannot be changed
 * by rules and therefore are implicitly always constant; thus, parameter
 * definitions within {@link Reaction} structures should <em>not</em> have their
 * \'constant\' attribute set to <code>false</code>.
 * <p>
 * What if a global parameter has its \'constant\' attribute set to <code>false</code>,
 * but the model does not contain any rules, events or other constructs
 * that ever change its value over time?  Although the model may be
 * suspect, this situation is not strictly an error.  A value of <code>false</code>
 * for \'constant\' only indicates that a parameter <em>can</em> change value, not
 * that it <em>must</em>.
 * <p>
 * As with all other major SBML components, {@link Parameter} is derived from
 * {@link SBase}, and the methods defined on {@link SBase} are available on {@link Parameter}.
 * <p>
 * @see ListOfParameters
 * @see KineticLaw
 * <p>
 * @note The use of the term <em>parameter</em> in SBML sometimes leads to
 * confusion among readers who have a particular notion of what something
 * called \'parameter\' should be.  It has been the source of heated debate,
 * but despite this, no one has yet found an adequate replacement term that
 * does not have different connotations to different people and hence leads
 * to confusion among <em>some</em> subset of users.  Perhaps it would have been
 * better to have two constructs, one called <em>constants</em> and the other
 * called <em>variables</em>.  The current approach in SBML is simply more
 * parsimonious, using a single {@link Parameter} construct with the boolean flag
 * \'constant\' indicating which flavor it is.  In any case, readers are
 * implored to look past their particular definition of a <em>parameter</em> and
 * simply view SBML\'s {@link Parameter} as a single mechanism for defining both
 * constants and (additional) variables in a model.  (We write 
 * <em>additional</em> because the species in a model are usually considered to be
 * the central variables.)  After all, software tools are not required to
 * expose to users the actual names of particular SBML constructs, and
 * thus tools can present to their users whatever terms their designers
 * feel best matches their target audience.
 * <p>
 * <!-- leave this next break as-is to work around some doxygen bug -->
 */
"

%typemap(javaimports) ListOfParameters "
/** 
 * LibSBML implementation of SBML\'s ListOfParameters construct.
 * <p>
 * The various ListOf___ classes in SBML are merely containers used for
 * organizing the main components of an SBML model.  All are derived from
 * the abstract class {@link SBase}, and inherit the various attributes and
 * subelements of {@link SBase}, such as \'metaid\' as and \'annotation\'.  The
 * ListOf___ classes do not add any attributes of their own.
 * <p>
 * The relationship between the lists and the rest of an SBML model is
 * illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):
 * <p>
 * <center><img src=\'listof-illustration.jpg\'></center><br>
 * 
 * <p>
 * Readers may wonder about the motivations for using the ListOf___
 * containers.  A simpler approach in XML might be to place the components
 * all directly at the top level of the model definition.  We chose instead
 * to group them within XML elements named after {@link ListOf}<em>Classname</em>,
 * in part because we believe this helps organize the components and makes
 * visual reading of models in XML easier.  More importantly, the fact that
 * the container classes are derived from {@link SBase} means that software tools
 * can add information about the lists themselves into each list
 * container\'s \'annotation\'.
 * <p>
 * @see ListOfFunctionDefinitions
 * @see ListOfUnitDefinitions
 * @see ListOfCompartmentTypes
 * @see ListOfSpeciesTypes
 * @see ListOfCompartments
 * @see ListOfSpecies
 * @see ListOfParameters
 * @see ListOfInitialAssignments
 * @see ListOfRules
 * @see ListOfConstraints
 * @see ListOfReactions
 * @see ListOfEvents
 */
"

%javamethodmodifiers Parameter::Parameter(unsigned int level, unsigned int version) "
  /**
   * Creates a new {@link Parameter} using the given SBML <code>level</code> and <code>version</code>
   * values.
   * <p>
   * @param level a long integer, the SBML Level to assign to this {@link Parameter}
   * <p>
   * @param version a long integer, the SBML Version to assign to this
   * {@link Parameter}
   * <p>
   * @note Once a {@link Parameter} has been added to an {@link SBMLDocument}, the <code>level</code>,
   * <code>version</code> for the document <em>override</em> those used
   * to create the {@link Parameter}.  Despite this, the ability to supply the values
   * at creation time is an important aid to creating valid SBML.  Knowledge of
   * the intented SBML Level and Version determine whether it is valid to
   * assign a particular value to an attribute, or whether it is valid to add
   * an object to an existing {@link SBMLDocument}.
   */
 public";

%javamethodmodifiers Parameter::Parameter(SBMLNamespaces* sbmlns) "
  /**
   * Creates a new {@link Parameter} using the given {@link SBMLNamespaces} object
   * <code>sbmlns</code>.
   * <p>
   * The {@link SBMLNamespaces} object encapsulates SBML Level/Version/namespaces
   * information.  It is used to communicate the SBML Level, Version, and
   * (in Level&nbsp;3) packages used in addition to SBML Level&nbsp; Core.
   * A common approach to using this class constructor is to create an
   * {@link SBMLNamespaces} object somewhere in a program, once, then pass it to
   * object constructors such as this one when needed.
   * <p>
   * It is worth emphasizing that although this constructor does not take
   * an identifier argument, in SBML Level&nbsp;2 and beyond, the \'id\'
   * (identifier) attribute of a {@link Parameter} is required to have a value.
   * Thus, callers are cautioned to assign a value after calling this
   * constructor if no identifier is provided as an argument.  Setting the
   * identifier can be accomplished using the method {@link SBase#setId(String id)}.
   * <p>
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object.
   * <p>
   * @note Once a {@link Parameter} has been added to an {@link SBMLDocument}, the <code>level</code>,
   * <code>version</code> and <code>xmlns</code> namespaces for the document <em>override</em> those used
   * to create the {@link Parameter}.  Despite this, the ability to supply the values
   * at creation time is an important aid to creating valid SBML.  Knowledge of
   * the intented SBML Level and Version determine whether it is valid to
   * assign a particular value to an attribute, or whether it is valid to add
   * an object to an existing {@link SBMLDocument}.
   */
 public";

%javamethodmodifiers Parameter::Parameter(const Parameter& orig) "
  /**
   * Copy constructor; creates a copy of a {@link Parameter}.
   * <p>
   * @param orig the {@link Parameter} instance to copy.
   */
 public";

%javamethodmodifiers Parameter::accept(SBMLVisitor& v) const "
  /**
   * Accepts the given {@link SBMLVisitor} for this instance of {@link Parameter}.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used.
   * <p>
   * @return the result of calling <code>v.visit()</code>, which indicates
   * whether the Visitor would like to visit the next {@link Parameter} in the list
   * of parameters within which this {@link Parameter} is embedded (i.e., either
   * the list of parameters in the parent {@link Model} or the list of parameters
   * in the enclosing {@link KineticLaw}).
   */
 public";

%javamethodmodifiers Parameter::clone() const "
  /**
   * Creates and returns a deep copy of this {@link Parameter}.
   * <p>
   * @return a (deep) copy of this {@link Parameter}.
   */
 public";

%javamethodmodifiers Parameter::initDefaults "
  /**
   * Initializes the fields of this {@link Parameter} to the defaults defined in
   * the specification of the relevant Level/Version of SBML.
   * <p>
   * The exact actions of this are as follows:
   * <ul>
   * <li> (%SBML Level&nbsp;2 only) set the \'constant\' attribute to <code>true</code>.
   * </ul>
   */
 public";

%javamethodmodifiers Parameter::getId() const "
  /**
   * Returns the value of the \'id\' attribute of this {@link Parameter}.
   * <p>
   * @return the id of this {@link Parameter}.
   */
 public";

%javamethodmodifiers Parameter::getName() const "
  /**
   * Returns the value of the \'name\' attribute of this {@link Parameter}.
   * <p>
   * @return the name of this {@link Parameter}.
   */
 public";

%javamethodmodifiers Parameter::getValue() const "
  /**
   * Gets the numerical value of this {@link Parameter}.
   * <p>
   * @return the value of the \'value\' attribute of this {@link Parameter}, as a
   * number of type <code>double</code>.
   * <p>
   * @see #isSetValue()
   * <p>
   * @note <b>It is crucial</b> that callers not blindly call
   * Parameter.getValue() without first checking with
   * Parameter.isSetValue() to determine whether a value has been set.
   * Otherwise, the value return by Parameter.getValue() may not actually
   * represent a value assigned to the parameter.
   */
 public";

%javamethodmodifiers Parameter::getUnits() const "
  /**
   * Gets the units defined for this {@link Parameter}
   * <p>
   * @return the value of the \'units\' attribute of this {@link Parameter}, as a
   * string.
   */
 public";

%javamethodmodifiers Parameter::getConstant() const "
  /**
   * Gets the value of the \'constant\' attribute of this {@link Parameter} instance.
   * <p>
   * Note that in SBML Level&nbsp;2 and beyond, the default value of
   * {@link Parameter}\'s \'constant\' attribute is <code>true</code>.  Since a boolean value
   * can only be <code>true</code> or <code>false</code>, there is no \'isSetConstant()\'
   * method as is available for the other attributes on {@link Parameter}, because
   * \'unset\' is not an option.
   * <p>
   * @return <code>true</code> if this {@link Parameter} has been declared as being constant,
   * <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers Parameter::isSetId() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link Parameter}\'s \'id\' attribute has been set.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the \'id\' attribute of this {@link Parameter} has been
   * set, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers Parameter::isSetName() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link Parameter}\'s \'name\' attribute has been set.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the \'name\' attribute of this {@link Parameter} has been
   * set, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers Parameter::isSetValue() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether the
   * \'value\' attribute of this {@link Parameter} has been set.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * In SBML definitions after SBML Level&nbsp;1 Version&nbsp;1,
   * parameter values are optional and have no defaults.  If a model read
   * from a file does not contain a setting for the \'value\' attribute of a
   * parameter, its value is considered unset; it does not default to any
   * particular value.  Similarly, when a {@link Parameter} object is created in
   * libSBML, it has no value until given a value.  The
   * Parameter.isSetValue() method allows calling applications to
   * determine whether a given parameter\'s value has ever been set.
   * <p>
   * In SBML Level&nbsp;1 Version&nbsp;1, parameters are required to have
   * values and therefore, the value of a {@link Parameter} <b>should always be
   * set</b>.  In Level&nbsp;1 Version&nbsp;2 and beyond, the value is
   * optional and as such, the \'value\' attribute may or may not be set.
   * <p>
   * @return <code>true</code> if the value of this {@link Parameter} has been set,
   * <code>false</code> otherwise.
   * <p>
   * @see #getValue()
   * <p>
   * @note <b>It is crucial</b> that callers not blindly call
   * Parameter.getValue() without first checking with
   * Parameter.isSetValue() to determine whether a value has been set.
   * Otherwise, the value return by Parameter.getValue() may not actually
   * represent a value assigned to the parameter.
   */
 public";

%javamethodmodifiers Parameter::isSetUnits() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether the
   * \'units\' attribute of this {@link Parameter} has been set.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the \'units\' attribute of this {@link Parameter} has been
   * set, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers Parameter::setId(const std::string& sid) "
  /**
   * Sets the value of the \'id\' attribute of this {@link Parameter}.
   * <p>
   * The string <code>sid</code> is copied.  Note that SBML has strict requirements
   * for the syntax of identifiers.  The following is summary of the
   * definition of the SBML identifier type <code>SId</code> (here expressed in an
   * extended form of BNF notation):
   * <div class=\'fragment\'><pre>
   *   letter .= \'a\'..\'z\',\'A\'..\'Z\'
   *   digit  .= \'0\'..\'9\'
   *   idChar .= letter | digit | \'_\'
   *   SId    .= ( letter | \'_\' ) idChar</pre></div>
   * The equality of SBML identifiers is determined by an exact character
   * sequence match; i.e., comparisons must be performed in a
   * case-sensitive manner.  In addition, there are a few conditions for
   * the uniqueness of identifiers in an SBML model.  Please consult the
   * SBML specifications for the exact formulations.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param sid the string to use as the identifier of this {@link Parameter}
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   */
 public";

%javamethodmodifiers Parameter::setName(const std::string& name) "
  /**
   * Sets the value of the \'name\' attribute of this {@link Parameter}.
   * <p>
   * The string in <code>name</code> is copied.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param name the new name for the {@link Parameter}
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   */
 public";

%javamethodmodifiers Parameter::setValue(double value) "
  /**
   * Sets the \'value\' attribute of this {@link Parameter} to the given <code>double</code>
   * value and marks the attribute as set.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param value a <code>double</code>, the value to assign
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   */
 public";

%javamethodmodifiers Parameter::setUnits(const std::string& units) "
  /**
   * Sets the \'units\' attribute of this {@link Parameter} to a copy of the given
   * units identifier <code>units</code>.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param units a string, the identifier of the units to assign to this
   * {@link Parameter} instance
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   */
 public";

%javamethodmodifiers Parameter::setConstant(bool flag) "
  /**
   * Sets the \'constant\' attribute of this {@link Parameter} to the given boolean
   * <code>flag</code>.
   * <p>
   * @param flag a boolean, the value for the \'constant\' attribute of this
   * {@link Parameter} instance
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_UNEXPECTED_ATTRIBUTE
   */
 public";

%javamethodmodifiers Parameter::unsetName "
  /**
   * Unsets the value of the \'name\' attribute of this {@link Parameter}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_OPERATION_FAILED
   */
 public";

%javamethodmodifiers Parameter::unsetValue "
  /**
   * Unsets the \'value\' attribute of this {@link Parameter} instance.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <p>
   * In SBML Level&nbsp;1 Version&nbsp;1, parameters are required to have
   * values and therefore, the value of a {@link Parameter} <b>should always be
   * set</b>.  In SBML Level&nbsp;1 Version&nbsp;2 and beyond, the value
   * is optional and as such, the \'value\' attribute may or may not be set.
   */
 public";

%javamethodmodifiers Parameter::unsetUnits "
  /**
   * Unsets the \'units\' attribute of this {@link Parameter} instance.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_OPERATION_FAILED
   */
 public";

%javamethodmodifiers Parameter::getDerivedUnitDefinition "
  /**
   * Constructs and returns a {@link UnitDefinition} that corresponds to the units
   * of this {@link Parameter}\'s value.
   * <p>
   * Parameters in SBML have an attribute (\'units\') for declaring the units
   * of measurement intended for the parameter\'s value.  <b>No defaults are
   * defined</b> by SBML in the absence of a definition for \'units\'.  The
   * Parameter.getDerivedUnitDefinition() method returns a {@link UnitDefinition}
   * object based on the units declared for this {@link Parameter} using its
   * \'units\' attribute, or it returns NULL if no units have been declared.
   * <p>
   * Note that the functionality that facilitates unit analysis depends 
   * on the model as a whole.  Thus, in cases where the object has not 
   * been added to a model or the model itself is incomplete,
   * unit analysis is not possible and this method will return NULL.
   * <p>
   * Note that unit declarations for {@link Parameter} are in terms of the 
   * <em>identifier</em> of a unit, but this method returns a {@link UnitDefinition} object,
   * not a unit identifier.  It does this by constructing an appropriate
   * {@link UnitDefinition} even when the value of the \'units\' attribute is one of
   * the predefined SBML units <code>\'substance\'</code>, <code>\'volume\'</code>, <code>\'area\'</code>, 
   * <code>\'length\'</code> or <code>\'time\'</code>.  Callers may find this particularly useful
   * when used in conjunction with the helper methods on {@link UnitDefinition}
   * for comparing different {@link UnitDefinition} objects.
   * <p>
   * @return a {@link UnitDefinition} that expresses the units of this 
   * {@link Parameter}.
   */
 public";

%javamethodmodifiers Parameter::getDerivedUnitDefinition() const "
  /**
   * Constructs and returns a {@link UnitDefinition} that corresponds to the units
   * of this {@link Parameter}\'s value.
   * <p>
   * Parameters in SBML have an attribute (\'units\') for declaring the units
   * of measurement intended for the parameter\'s value.  <b>No defaults are
   * defined</b> by SBML in the absence of a definition for \'units\'.  The
   * Parameter.getDerivedUnitDefinition() method returns a {@link UnitDefinition}
   * object based on the units declared for this {@link Parameter} using its
   * \'units\' attribute, or it returns NULL if no units have been declared.
   * <p>
   * Note that the functionality that facilitates unit analysis depends 
   * on the model as a whole.  Thus, in cases where the object has not 
   * been added to a model or the model itself is incomplete,
   * unit analysis is not possible and this method will return NULL.
   * <p>
   * Note that unit declarations for {@link Parameter} are in terms of the 
   * <em>identifier</em> of a unit, but this method returns a {@link UnitDefinition} object,
   * not a unit identifier.  It does this by constructing an appropriate
   * {@link UnitDefinition} even when the value of the \'units\' attribute is one of
   * the predefined SBML units <code>\'substance\'</code>, <code>\'volume\'</code>, <code>\'area\'</code>, 
   * <code>\'length\'</code> or <code>\'time\'</code>.  Callers may find this particularly useful
   * when used in conjunction with the helper methods on {@link UnitDefinition}
   * for comparing different {@link UnitDefinition} objects.
   * <p>
   * @return a {@link UnitDefinition} that expresses the units of this 
   * {@link Parameter}.
   */
 public";

%javamethodmodifiers Parameter::getTypeCode() const "
  /**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for this object, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers Parameter::getElementName() const "
  /**
   * Returns the XML element name of this object, which for {@link Parameter}, is
   * always <code>\'parameter\'</code>.
   * <p>
   * @return the name of this element, i.e., <code>\'parameter\'</code>.
   */
 public";

%javamethodmodifiers Parameter::hasRequiredAttributes() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link Parameter} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link Parameter} object are:
   * id (name in L1); value (L1V1 only)
   * <p>
   * @return a boolean value indicating whether all the required
   * attributes for this object have been defined.
   */
 public";

%javamethodmodifiers Parameter::Parameter "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link Parameter} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link Parameter} object are:
   * id (name in L1); value (L1V1 only)
   * <p>
   * @return a boolean value indicating whether all the required
   * attributes for this object have been defined.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Parameter::readAttributes(const XMLAttributes& attributes) "
  /**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Parameter::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Parameter::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Parameter::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Parameter::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Parameter::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Parameter::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Parameter::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Parameter::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Parameter::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Parameter::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Parameter::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Parameter::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Parameter::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Parameter::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Parameter::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Parameter::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Parameter::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Parameter::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Parameter::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Parameter::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Parameter::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers ListOfParameters::clone() const "
  /**
   * Creates and returns a deep copy of this {@link ListOfParameters} instance.
   * <p>
   * @return a (deep) copy of this {@link ListOfParameters}.
   */
 public";

%javamethodmodifiers ListOfParameters::getTypeCode() const "
  /**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for this object, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers ListOfParameters::getItemTypeCode() const "
  /**
   * Returns the libSBML type code for the objects contained in this {@link ListOf}
   * (i.e., {@link Parameter} objects, if the list is non-empty).
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for the objects contained in this {@link ListOf}
   * instance, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers ListOfParameters::getElementName() const "
  /**
   * Returns the XML element name of this object.
   * <p>
   * For {@link ListOfParameters}, the XML element name is <code>\'listOfParameters\'</code>.
   * <p>
   * @return the name of this element, i.e., <code>\'listOfParameters\'</code>.
   */
 public";

%javamethodmodifiers ListOfParameters::get(unsigned int n) "
  /**
   * Get a {@link Parameter} from the {@link ListOfParameters}.
   * <p>
   * @param n the index number of the {@link Parameter} to get.
   * <p>
   * @return the nth {@link Parameter} in this {@link ListOfParameters}.
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfParameters::get "
  /**
   * Get a {@link Parameter} from the {@link ListOfParameters}.
   * <p>
   * @param n the index number of the {@link Parameter} to get.
   * <p>
   * @return the nth {@link Parameter} in this {@link ListOfParameters}.
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfParameters::get(const std::string& sid) "
  /**
   * Get a {@link Parameter} from the {@link ListOfParameters}
   * based on its identifier.
   * <p>
   * @param sid a string representing the identifier 
   * of the {@link Parameter} to get.
   * <p>
   * @return {@link Parameter} in this {@link ListOfParameters}
   * with the given id or NULL if no such
   * {@link Parameter} exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfParameters::get(const std::string& sid) const "
  /**
   * Get a {@link Parameter} from the {@link ListOfParameters}
   * based on its identifier.
   * <p>
   * @param sid a string representing the identifier 
   * of the {@link Parameter} to get.
   * <p>
   * @return {@link Parameter} in this {@link ListOfParameters}
   * with the given id or NULL if no such
   * {@link Parameter} exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfParameters::remove(unsigned int n) "
  /**
   * Removes the nth item from this {@link ListOfParameters} items and returns a pointer to
   * it.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * <p>
   * @param n the index of the item to remove
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfParameters::remove(const std::string& sid) "
  /**
   * Removes item in this {@link ListOfParameters} items with the given identifier.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * If none of the items in this list have the identifier <code>sid</code>, then 
   * <code>NULL</code> is returned.
   * <p>
   * @param sid the identifier of the item to remove
   * <p>
   * @return the item removed.  As mentioned above, the caller owns the
   * returned item.
   */
 public";

%javamethodmodifiers ListOfParameters::getElementPosition() const "
  /**
   * Get the ordinal position of this element in the containing object
   * (which in this case is the {@link Model} object).
   * <p>
   * The ordering of elements in the XML form of SBML is generally fixed
   * for most components in SBML.  So, for example, the {@link ListOfParameters}
   * in a model is (in SBML Level&nbsp;2 Version&nbsp;4) the seventh
   * ListOf___.  (However, it differs for different Levels and Versions of
   * SBML.)
   * <p>
   * @return the ordinal position of the element with respect to its
   * siblings, or <code>-1</code> (default) to indicate the position is not significant.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers ListOfParameters::createObject(XMLInputStream& stream) "
  /**
   * Create a {@link ListOfParameters} object corresponding to the next token in
   * the XML input stream.
   * <p>
   * @return the SBML object corresponding to next {@link XMLToken} in the
   * XMLInputStream, or <code>NULL</code> if the token was not recognized.
   <p>
 * @deprecated libSBML internal
 */
 public";

%typemap(javaimports) InitialAssignment "
/** 
 * LibSBML implementation of SBML\'s InitialAssignment construct.
 * <p>
 * SBML Level 2 Versions 2, 3 and 4 provide two ways of assigning initial
 * values to entities in a model.  The simplest and most basic is to set
 * the values of the appropriate attributes in the relevant components; for
 * example, the initial value of a model parameter (whether it is a
 * constant or a variable) can be assigned by setting its \'value\' attribute
 * directly in the model definition.  However, this approach is not
 * suitable when the value must be calculated, because the initial value
 * attributes on different components such as species, compartments, and
 * parameters are single values and not mathematical expressions.  In those
 * situations, the {@link InitialAssignment} construct can be used; it permits the
 * calculation of the value of a constant or the initial value of a
 * variable from the values of <em>other</em> quantities in a model.
 * <p>
 * As explained below, the provision of {@link InitialAssignment} does not mean
 * that models necessarily must use this construct when defining initial
 * values of quantities in a model.  If a value can be set directly using
 * the relevant attribute of a component in a model, then that
 * approach may be more efficient and more portable to other software
 * tools.  {@link InitialAssignment} should be used when the other mechanism is
 * insufficient for the needs of a particular model.
 * <p>
 * The {@link InitialAssignment} construct has some similarities to {@link AssignmentRule}.
 * The main differences are: (a) an {@link InitialAssignment} can set the value of
 * a constant whereas an {@link AssignmentRule} cannot, and (b) unlike
 * {@link AssignmentRule}, an {@link InitialAssignment} definition only applies up to and
 * including the beginning of simulation time, i.e., <em>t &#8804; 0</em>,
 * while an {@link AssignmentRule} applies at all times.
 * <p>
 * {@link InitialAssignment} has a required attribute, \'symbol\', whose value must
 * follow the guidelines for identifiers described in the SBML
 * specification (e.g., Section 3.3 in the Level 2 Version 4
 * specification).  The value of this attribute in an {@link InitialAssignment}
 * object can be the identifier of a {@link Compartment}, {@link Species} or global
 * {@link Parameter} elsewhere in the model.  The {@link InitialAssignment} defines the
 * initial value of the constant or variable referred to by the \'symbol\'
 * attribute.  (The attribute\'s name is \'symbol\' rather than \'variable\'
 * because it may assign values to constants as well as variables in a
 * model.)  Note that an initial assignment cannot be made to reaction
 * identifiers, that is, the \'symbol\' attribute value of an
 * {@link InitialAssignment} cannot be an identifier that is the \'id\' attribute
 * value of a {@link Reaction} object in the model.  This is identical to a
 * restriction placed on rules.
 * <p>
 * {@link InitialAssignment} also has a required \'math\' subelement that contains a
 * MathML expression used to calculate the value of the constant or the
 * initial value of the variable.  The units of the value computed by the
 * formula in the \'math\' subelement should (in SBML Level&nbsp;2
 * Version&nbsp;4) or must (in previous Versions) be identical to be the
 * units associated with the identifier given in the \'symbol\' attribute.
 * (That is, the units are the units of the species, compartment, or
 * parameter, as appropriate for the kind of object identified by the value
 * of \'symbol\'.)
 * <p>
 * {@link InitialAssignment} was introduced in SBML Level 2 Version 2.  It is not
 * available in earlier versions of Level 2 nor in any version of Level 1.
 * <p>
 * <h2>Semantics of Initial Assignments</h2>
 * <p>
 * The value calculated by an {@link InitialAssignment} object overrides the value
 * assigned to the given symbol by the object defining that symbol.  For
 * example, if a compartment\'s \'size\' attribute is set in its definition,
 * and the model also contains an {@link InitialAssignment} having that
 * compartment\'s identifier as its \'symbol\' attribute value, then the
 * interpretation is that the \'size\' assigned in the {@link Compartment} object
 * should be ignored and the value assigned based on the computation
 * defined in the {@link InitialAssignment}.  Initial assignments can take place
 * for {@link Compartment}, {@link Species} and global {@link Parameter} objects regardless of the
 * value of their \'constant\' attribute.
 * <p>
 * The actions of all {@link InitialAssignment} objects are in general terms
 * the same, but differ in the precise details depending on the type
 * of variable being set:
 * <ul>
 * <li> <em>In the case of a species</em>, an {@link InitialAssignment} sets the
 * referenced species\' initial quantity (concentration or amount of
 * substance) to the value determined by the formula in the \'math\'
 * subelement.    The overall units of the formula should (in SBML
 * Level&nbsp;2 Version&nbsp;4) or must (in previous Versions) be the same
 * as the units specified for the species.
 * <p>
 * <li> <em>In the case of a compartment</em>, an {@link InitialAssignment} sets
 * the referenced compartment\'s initial size to the size determined by the
 * formula in \'math\'.  The overall units of the formula should (in SBML
 * Level&nbsp;2 Version&nbsp;4) or must (in previous Versions) be the same
 * as the units specified for the size of the compartment.
 * <p>
 * <li> <em>In the case of a parameter</em>, an {@link InitialAssignment} sets the
 * referenced parameter\'s initial value to that determined by the formula
 * in \'math\'.  The overall units of the formula should (in SBML
 * Level&nbsp;2 Version&nbsp;4) or must (in previous Versions) be the same
 * as the units defined for the parameter.  </ul>
 * <p>
 * In the context of a simulation, initial assignments establish values
 * that are in effect prior to and including the start of simulation time,
 * i.e., <em>t &#8804; 0</em>.  Section 3.4.8 in the SBML Level 2
 * Version 4 specification provides information about the interpretation of
 * assignments, rules, and entity values for simulation time up to and
 * including the start time <em>t = 0</em>; this is important for
 * establishing the initial conditions of a simulation if the model
 * involves expressions containing the <em>delay</em> \'csymbol\'.
 * <p>
 * There cannot be two initial assignments for the same symbol in a model;
 * that is, a model must not contain two or more {@link InitialAssignment} objects
 * that both have the same identifier as their \'symbol\' attribute value.  A
 * model must also not define initial assignments <em>and</em> assignment
 * rules for the same entity.  That is, there cannot be <em>both</em> an
 * {@link InitialAssignment} and an {@link AssignmentRule} for the same symbol in a model,
 * because both kinds of constructs apply prior to and at the start of
 * simulated time&mdash;allowing both to exist for a given symbol would
 * result in indeterminism).
 * <p>
 * The ordering of {@link InitialAssignment} objects is not significant.  The
 * combined set of {@link InitialAssignment}, {@link AssignmentRule} and {@link KineticLaw}
 * objects form a set of assignment statements that must be considered as a
 * whole.  The combined set of assignment statements should not contain
 * algebraic loops: a chain of dependency between these statements should
 * terminate.  (More formally, consider the directed graph of assignment
 * statements where nodes are a model\'s assignment statements and directed
 * arcs exist for each occurrence of a symbol in an assignment statement
 * \'math\' attribute.  The directed arcs in this graph start from the
 * statement assigning the symbol and end at the statement that contains
 * the symbol in their math elements.  Such a graph must be acyclic.)
 * <p>
 * Finally, it is worth being explicit about the expected behavior in the
 * following situation.  Suppose (1) a given symbol has a value <em>x</em>
 * assigned to it in its definition, and (2) there is an initial assignment
 * having the identifier as its \'symbol\' value and reassigning the value to
 * <em>y</em>, <em>and</em> (3) the identifier is also used in the
 * mathematical formula of a second initial assignment.  What value should
 * the second initial assignment use?  It is <em>y</em>, the value assigned
 * to the symbol by the first initial assignment, not whatever value was
 * given in the symbol\'s definition.  This follows directly from the
 * behavior described above: if an {@link InitialAssignment} object exists for a
 * given symbol, then the symbol\'s value is overridden by that initial
 * assignment.
 * <p>
 * <!---------------------------------------------------------------------- -->
 * <p>
 */
"

%typemap(javaimports) ListOfInitialAssignments "
/** 
 * LibSBML implementation of SBML\'s ListOfInitialAssignments construct.
 * <p>
 * The various ListOf___ classes in SBML are merely containers used for
 * organizing the main components of an SBML model.  All are derived from
 * the abstract class {@link SBase}, and inherit the various attributes and
 * subelements of {@link SBase}, such as \'metaid\' as and \'annotation\'.  The
 * ListOf___ classes do not add any attributes of their own.
 * <p>
 * The relationship between the lists and the rest of an SBML model is
 * illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):
 * <p>
 * <center><img src=\'listof-illustration.jpg\'></center><br>
 * 
 * <p>
 * Readers may wonder about the motivations for using the ListOf___
 * containers.  A simpler approach in XML might be to place the components
 * all directly at the top level of the model definition.  We chose instead
 * to group them within XML elements named after {@link ListOf}<em>Classname</em>,
 * in part because we believe this helps organize the components and makes
 * visual reading of models in XML easier.  More importantly, the fact that
 * the container classes are derived from {@link SBase} means that software tools
 * can add information about the lists themselves into each list
 * container\'s \'annotation\'.
 * <p>
 * @see ListOfFunctionDefinitions
 * @see ListOfUnitDefinitions
 * @see ListOfCompartmentTypes
 * @see ListOfSpeciesTypes
 * @see ListOfCompartments
 * @see ListOfSpecies
 * @see ListOfParameters
 * @see ListOfInitialAssignments
 * @see ListOfRules
 * @see ListOfConstraints
 * @see ListOfReactions
 * @see ListOfEvents
 */
"

%javamethodmodifiers InitialAssignment::InitialAssignment(unsigned int level, unsigned int version) "
  /**
   * Creates a new {@link InitialAssignment} using the given SBML <code>level</code> and <code>version</code>
   * values.
   * <p>
   * @param level a long integer, the SBML Level to assign to this {@link InitialAssignment}
   * <p>
   * @param version a long integer, the SBML Version to assign to this
   * {@link InitialAssignment}
   * <p>
   * @note Once a {@link InitialAssignment} has been added to an {@link SBMLDocument}, the <code>level</code>,
   * <code>version</code> for the document <em>override</em> those used
   * to create the {@link InitialAssignment}.  Despite this, the ability to supply the values
   * at creation time is an important aid to creating valid SBML.  Knowledge of
   * the intented SBML Level and Version determine whether it is valid to
   * assign a particular value to an attribute, or whether it is valid to add
   * an object to an existing {@link SBMLDocument}.
   */
 public";

%javamethodmodifiers InitialAssignment::InitialAssignment(SBMLNamespaces* sbmlns) "
  /**
   * Creates a new {@link InitialAssignment} using the given {@link SBMLNamespaces} object
   * <code>sbmlns</code>.
   * <p>
   * The {@link SBMLNamespaces} object encapsulates SBML Level/Version/namespaces
   * information.  It is used to communicate the SBML Level, Version, and
   * (in Level&nbsp;3) packages used in addition to SBML Level&nbsp; Core.
   * A common approach to using this class constructor is to create an
   * {@link SBMLNamespaces} object somewhere in a program, once, then pass it to
   * object constructors such as this one when needed.
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object.
   * <p>
   * @note Once a {@link InitialAssignment} has been added to an {@link SBMLDocument}, the <code>level</code>,
   * <code>version</code> and <code>xmlns</code> namespaces for the document <em>override</em> those used
   * to create the {@link InitialAssignment}.  Despite this, the ability to supply the values
   * at creation time is an important aid to creating valid SBML.  Knowledge of
   * the intented SBML Level and Version determine whether it is valid to
   * assign a particular value to an attribute, or whether it is valid to add
   * an object to an existing {@link SBMLDocument}.
   */
 public";

%javamethodmodifiers InitialAssignment::InitialAssignment(const InitialAssignment& orig) "
  /**
   * Copy constructor; creates a copy of this {@link InitialAssignment}.
   */
 public";

%javamethodmodifiers InitialAssignment::accept(SBMLVisitor& v) const "
  /**
   * Accepts the given {@link SBMLVisitor} for this instance of {@link InitialAssignment}.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used.
   * <p>
   * @return the result of calling <code>v.visit()</code>, which indicates
   * whether the Visitor would like to visit the next {@link InitialAssignment} in
   * the list of compartment types.
   */
 public";

%javamethodmodifiers InitialAssignment::clone() const "
  /**
   * Creates and returns a deep copy of this {@link InitialAssignment}.
   * <p>
   * @return a (deep) copy of this {@link InitialAssignment}.
   */
 public";

%javamethodmodifiers InitialAssignment::getSymbol() const "
  /**
   * Get the value of the \'symbol\' attribute of this {@link InitialAssignment}.
   * <p>
   * @return the identifier string stored as the \'symbol\' attribute value
   * in this {@link InitialAssignment}.
   */
 public";

%javamethodmodifiers InitialAssignment::getMath() const "
  /**
   * Get the mathematical formula of this {@link InitialAssignment}.
   * <p>
   * @return an {@link ASTNode}, the value of the \'math\' subelement of this
   * {@link InitialAssignment}
   */
 public";

%javamethodmodifiers InitialAssignment::isSetSymbol() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link InitialAssignment}\'s \'symbol\' attribute has been set.
   * <p>
   * @return <code>true</code> if the \'symbol\' attribute of this {@link InitialAssignment}
   * has been set, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers InitialAssignment::isSetMath() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link InitialAssignment}\'s \'math\' subelement contains a value.
   * <p>
   * @return <code>true</code> if the \'math\' for this {@link InitialAssignment} has been set,
   * <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers InitialAssignment::setSymbol(const std::string& sid) "
  /**
   * Sets the \'symbol\' attribute value of this {@link InitialAssignment}.
   * <p>
   * @param sid the identifier of a {@link Species}, {@link Compartment} or {@link Parameter}
   * object defined elsewhere in this {@link Model}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   */
 public";

%javamethodmodifiers InitialAssignment::setMath(const ASTNode* math) "
  /**
   * Sets the \'math\' subelement of this {@link InitialAssignment}.
   * <p>
   * The AST passed in <code>math</code> is copied.
   * <p>
   * @param math an AST containing the mathematical expression to
   * be used as the formula for this {@link InitialAssignment}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_OBJECT
   */
 public";

%javamethodmodifiers InitialAssignment::getDerivedUnitDefinition "
  /**
   * Calculates and returns a {@link UnitDefinition} that expresses the units
   * of measurement assumed for the \'math\' expression of this
   * {@link InitialAssignment}.
   * <p>
   * The units are calculated based on the mathematical expression in the
   * {@link InitialAssignment} and the model quantities referenced by
   * <code>&lt;ci&gt;</code> elements used within that expression.  The
   * getDerivedUnitDefinition() method returns the calculated units.
   * <p>
   * Note that the functionality that facilitates unit analysis depends 
   * on the model as a whole.  Thus, in cases where the object has not 
   * been added to a model or the model itself is incomplete,
   * unit analysis is not possible and this method will return NULL.
   * <p>
   * @warning Note that it is possible the \'math\' expression in the
   * {@link InitialAssignment} contains pure numbers or parameters with undeclared
   * units.  In those cases, it is not possible to calculate the units of
   * the overall expression without making assumptions.  LibSBML does not
   * make assumptions about the units, and getDerivedUnitDefinition() only
   * returns the units as far as it is able to determine them.  For
   * example, in an expression <em>X + Y</em>, if <em>X</em> has
   * unambiguously-defined units and <em>Y</em> does not, it will return
   * the units of <em>X</em>.  <strong>It is important that callers also
   * invoke the method</strong> containsUndeclaredUnits() <strong>to
   * determine whether this situation holds</strong>.  Callers may wish to
   * take suitable actions in those scenarios.
   * <p>
   * @return a {@link UnitDefinition} that expresses the units of the math 
   * expression of this {@link InitialAssignment}.
   * <p>
   * @see #containsUndeclaredUnits()
   */
 public";

%javamethodmodifiers InitialAssignment::getDerivedUnitDefinition() const "
  /**
   * Calculates and returns a {@link UnitDefinition} that expresses the units
   * of measurement assumed for the \'math\' expression of this
   * {@link InitialAssignment}.
   * <p>
   * The units are calculated based on the mathematical expression in the
   * {@link InitialAssignment} and the model quantities referenced by
   * <code>&lt;ci&gt;</code> elements used within that expression.  The
   * getDerivedUnitDefinition() method returns the calculated units.
   * <p>
   * Note that the functionality that facilitates unit analysis depends 
   * on the model as a whole.  Thus, in cases where the object has not 
   * been added to a model or the model itself is incomplete,
   * unit analysis is not possible and this method will return NULL.
   * <p>
   * @warning Note that it is possible the \'math\' expression in the
   * {@link InitialAssignment} contains pure numbers or parameters with undeclared
   * units.  In those cases, it is not possible to calculate the units of
   * the overall expression without making assumptions.  LibSBML does not
   * make assumptions about the units, and getDerivedUnitDefinition() only
   * returns the units as far as it is able to determine them.  For
   * example, in an expression <em>X + Y</em>, if <em>X</em> has
   * unambiguously-defined units and <em>Y</em> does not, it will return
   * the units of <em>X</em>.  <strong>It is important that callers also
   * invoke the method</strong> containsUndeclaredUnits() <strong>to
   * determine whether this situation holds</strong>.  Callers may wish to
   * take suitable actions in those scenarios.
   * <p>
   * @return a {@link UnitDefinition} that expresses the units of the math 
   * expression of this {@link InitialAssignment}.
   * <p>
   * @see #containsUndeclaredUnits()
   */
 public";

%javamethodmodifiers InitialAssignment::containsUndeclaredUnits "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether 
   * the math expression of this {@link InitialAssignment} contains
   * parameters/numbers with undeclared units.
   * <p>
   * @return <code>true</code> if the math expression of this {@link InitialAssignment}
   * includes parameters/numbers 
   * with undeclared units, <code>false</code> otherwise.
   * <p>
   * @note A return value of <code>true</code> indicates that the {@link UnitDefinition}
   * returned by getDerivedUnitDefinition() may not accurately represent
   * the units of the expression.
   * <p>
   * @see #getDerivedUnitDefinition()
   */
 public";

%javamethodmodifiers InitialAssignment::containsUndeclaredUnits() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether 
   * the math expression of this {@link InitialAssignment} contains
   * parameters/numbers with undeclared units.
   * <p>
   * @return <code>true</code> if the math expression of this {@link InitialAssignment}
   * includes parameters/numbers 
   * with undeclared units, <code>false</code> otherwise.
   * <p>
   * @note A return value of <code>true</code> indicates that the {@link UnitDefinition}
   * returned by getDerivedUnitDefinition() may not accurately represent
   * the units of the expression.
   * <p>
   * @see #getDerivedUnitDefinition()
   */
 public";

%javamethodmodifiers InitialAssignment::getTypeCode() const "
  /**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for this object, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers InitialAssignment::getElementName() const "
  /**
   * Returns the XML element name of this object, which for
   * {@link InitialAssignment}, is always <code>\'initialAssignment\'</code>.
   * <p>
   * @return the name of this element, i.e., <code>\'initialAssignment\'</code>.
   */
 public";

%javamethodmodifiers InitialAssignment::writeElements(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write out their contained
   * SBML objects as XML elements.  Be sure to call your parents
   * implementation of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers InitialAssignment::hasRequiredAttributes() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link InitialAssignment} object
   * have been set.
   * <p>
   * @note The required attributes for an {@link InitialAssignment} object are:
   * symbol
   * <p>
   * @return a boolean value indicating whether all the required
   * attributes for this object have been defined.
   */
 public";

%javamethodmodifiers InitialAssignment::hasRequiredElements() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required elements for this {@link InitialAssignment} object
   * have been set.
   * <p>
   * @note The required elements for a {@link InitialAssignment} object are:
   * math
   * <p>
   * @return a boolean value indicating whether all the required
   * elements for this object have been defined.
   */
 public";

%javamethodmodifiers InitialAssignment::getId() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required elements for this {@link InitialAssignment} object
   * have been set.
   * <p>
   * @note The required elements for a {@link InitialAssignment} object are:
   * math
   * <p>
   * @return a boolean value indicating whether all the required
   * elements for this object have been defined.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers InitialAssignment::InitialAssignment "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required elements for this {@link InitialAssignment} object
   * have been set.
   * <p>
   * @note The required elements for a {@link InitialAssignment} object are:
   * math
   * <p>
   * @return a boolean value indicating whether all the required
   * elements for this object have been defined.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers InitialAssignment::readOtherXML(XMLInputStream& stream) "
  /**
   * Subclasses should override this method to read (and store) XHTML,
   * MathML, etc. directly from the XMLInputStream.
   * <p>
   * @return true if the subclass read from the stream, false otherwise.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers InitialAssignment::readAttributes(const XMLAttributes& attributes) "
  /**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers InitialAssignment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers InitialAssignment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers InitialAssignment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers InitialAssignment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers InitialAssignment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers InitialAssignment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers InitialAssignment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers InitialAssignment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers InitialAssignment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers InitialAssignment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers InitialAssignment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers InitialAssignment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers InitialAssignment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers InitialAssignment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers InitialAssignment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers InitialAssignment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers InitialAssignment::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers ListOfInitialAssignments::clone() const "
  /**
   * Creates and returns a deep copy of this {@link ListOfInitialAssignments} instance.
   * <p>
   * @return a (deep) copy of this {@link ListOfInitialAssignments}.
   */
 public";

%javamethodmodifiers ListOfInitialAssignments::getTypeCode() const "
  /**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for this object, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers ListOfInitialAssignments::getItemTypeCode() const "
  /**
   * Returns the libSBML type code for the objects contained in this {@link ListOf}
   * (i.e., {@link InitialAssignment} objects, if the list is non-empty).
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for the objects contained in this {@link ListOf}
   * instance, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers ListOfInitialAssignments::getElementName() const "
  /**
   * Returns the XML element name of this object.
   * <p>
   * For {@link ListOfInitialAssignments}, the XML element name is 
   * <code>\'listOfInitialAssignments\'</code>.
   * <p>
   * @return the name of this element, i.e., <code>\'listOfInitialAssignments\'</code>.
   */
 public";

%javamethodmodifiers ListOfInitialAssignments::get(unsigned int n) "
  /**
   * Get a {@link InitialAssignment} from the {@link ListOfInitialAssignments}.
   * <p>
   * @param n the index number of the {@link InitialAssignment} to get.
   * <p>
   * @return the nth {@link InitialAssignment} in this {@link ListOfInitialAssignments}.
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfInitialAssignments::get "
  /**
   * Get a {@link InitialAssignment} from the {@link ListOfInitialAssignments}.
   * <p>
   * @param n the index number of the {@link InitialAssignment} to get.
   * <p>
   * @return the nth {@link InitialAssignment} in this {@link ListOfInitialAssignments}.
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfInitialAssignments::get(const std::string& sid) "
  /**
   * Get a {@link InitialAssignment} from the {@link ListOfInitialAssignments}
   * based on its identifier.
   * <p>
   * @param sid a string representing the identifier 
   * of the {@link InitialAssignment} to get.
   * <p>
   * @return {@link InitialAssignment} in this {@link ListOfInitialAssignments}
   * with the given id or NULL if no such
   * {@link InitialAssignment} exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfInitialAssignments::get(const std::string& sid) const "
  /**
   * Get a {@link InitialAssignment} from the {@link ListOfInitialAssignments}
   * based on its identifier.
   * <p>
   * @param sid a string representing the identifier 
   * of the {@link InitialAssignment} to get.
   * <p>
   * @return {@link InitialAssignment} in this {@link ListOfInitialAssignments}
   * with the given id or NULL if no such
   * {@link InitialAssignment} exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfInitialAssignments::remove(unsigned int n) "
  /**
   * Removes the nth item from this {@link ListOfInitialAssignments} items and returns a pointer to
   * it.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * <p>
   * @param n the index of the item to remove
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfInitialAssignments::remove(const std::string& sid) "
  /**
   * Removes item in this {@link ListOfInitialAssignments} items with the given identifier.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * If none of the items in this list have the identifier <code>sid</code>, then 
   * <code>NULL</code> is returned.
   * <p>
   * @param sid the identifier of the item to remove
   * <p>
   * @return the item removed.  As mentioned above, the caller owns the
   * returned item.
   */
 public";

%javamethodmodifiers ListOfInitialAssignments::getElementPosition() const "
  /**
   * Get the ordinal position of this element in the containing object
   * (which in this case is the {@link Model} object).
   * <p>
   * The ordering of elements in the XML form of SBML is generally fixed
   * for most components in SBML.  So, for example, the
   * {@link ListOfInitialAssignments} in a model is (in SBML Level 2 Version 4)
   * the eigth ListOf___.  (However, it differs for different Levels and
   * Versions of SBML.)
   * <p>
   * @return the ordinal position of the element with respect to its
   * siblings, or <code>-1</code> (default) to indicate the position is not significant.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers ListOfInitialAssignments::createObject(XMLInputStream& stream) "
  /**
   * @return the SBML object corresponding to next {@link XMLToken} in the
   * XMLInputStream or NULL if the token was not recognized.
   <p>
 * @deprecated libSBML internal
 */
 public";

%typemap(javaimports) Rule "
/** 
 * LibSBML implementation of SBML\'s Rule construct.
 * <p>
 * In SBML, <em>rules</em> provide additional ways to define the values of
 * variables in a model, their relationships, and the dynamical behaviors
 * of those variables.  They enable encoding relationships that cannot be
 * expressed using {@link Reaction} nor {@link InitialAssignment} objects alone.
 * <p>
 * The libSBML implementation of rules mirrors the SBML Level&nbsp;2
 * Version&nbsp;4 definition, with {@link Rule} being the parent class of three
 * subclasses as explained below.  The {@link Rule} class itself cannot be
 * instantiated by user programs and has no constructor; only the
 * subclasses {@link AssignmentRule}, {@link AlgebraicRule} and {@link RateRule} can be
 * instantiated directly.
 * <p>
 * <h2>General summary of SBML rules</h2>
<p>
In SBML Level&nbsp;2, rules are separated into three subclasses for the
benefit of model analysis software.  The three subclasses are based on
the following three different possible functional forms (where <em>x</em> is
a variable, <em>f</em> is some arbitrary function returning a numerical
result, <b>V</b> is a vector of variables that does not include <em>x</em>,
and <b>W</b> is a vector of variables that may include <em>x</em>):

<p>
<center>
<table border=\'0\' cellpadding=\'0\' style=\'font-size: small\'>
<tr><td width=\'120px\'><em>Algebraic:</em></td><td width=\'250px\'>left-hand side is zero</td><td><em>0 = f(<b>W</b>)</em></td></tr>
<tr><td><em>Assignment:</em></td><td>left-hand side is a scalar:</td><td><em>x = f(<b>V</b>)</em></td></tr>
<tr><td><em>Rate:</em></td><td>left-hand side is a rate-of-change:</td><td><em>dx/dt = f(<b>W</b>)</em></td></tr>
</table>
</center>

<p>
In their general form given above, there is little to distinguish
between <em>assignment</em> and <em>algebraic</em> rules.  They are treated as
separate cases for the following reasons:
<p>
<ul>
<li> <em>Assignment</em> rules can simply be evaluated to calculate
intermediate values for use in numerical methods.  They are statements
of equality that hold at all times.  (For assignments that are only
performed once, see {@link InitialAssignment}.)<p>

<li> SBML needs to place restrictions on assignment rules, for example
the restriction that assignment rules cannot contain algebraic loops.<p>

<li> Some simulators do not contain numerical solvers capable of solving
unconstrained algebraic equations, and providing more direct forms such
as assignment rules may enable those simulators to process models they
could not process if the same assignments were put in the form of
general algebraic equations;<p>

<li> Those simulators that <em>can</em> solve these algebraic equations make a
distinction between the different categories listed above; and<p>

<li> Some specialized numerical analyses of models may only be applicable
to models that do not contain <em>algebraic</em> rules.
</ul>

<p> The approach taken to covering these cases in SBML is to define an
abstract {@link Rule} structure containing a subelement, \'math\', to hold the
right-hand side expression, then to derive subtypes of {@link Rule} that add
attributes to distinguish the cases of algebraic, assignment and rate
rules.  The \'math\' subelement must contain a MathML expression defining the
mathematical formula of the rule.  This MathML formula must return a
numerical value.  The formula can be an arbitrary expression referencing
the variables and other entities in an SBML model.

<p> Each of the three subclasses of {@link Rule} (AssignmentRule, {@link AlgebraicRule},
{@link RateRule}) inherit the the \'math\' subelement and other fields from {@link SBase}.
The {@link AssignmentRule} and {@link RateRule} classes add an additional attribute,
\'variable\'.  See the definitions of {@link AssignmentRule}, {@link AlgebraicRule} and
{@link RateRule} for details about the structure and interpretation of each one.

<h2>Additional restrictions on SBML rules</h2>

<p> An important design goal of SBML rule semantics is to ensure that a
model\'s simulation and analysis results will not be dependent on when or
how often rules are evaluated.  To achieve this, SBML needs to place two
restrictions on rule use.  The first concerns algebraic loops in the system
of assignments in a model, and the second concerns overdetermined systems.

<h3>A model must not contain algebraic loops</h3>

<p> The combined set of {@link InitialAssignment}, {@link AssignmentRule} and {@link KineticLaw}
objects in a model constitute a set of assignment statements that should be
considered as a whole.  (A {@link KineticLaw} object is counted as an assignment
because it assigns a value to the symbol contained in the \'id\' attribute of
the {@link Reaction} object in which it is defined.)  This combined set of
assignment statements must not contain algebraic loops&mdash;dependency
chains between these statements must terminate.  To put this more formally,
consider a directed graph in which nodes are assignment statements and
directed arcs exist for each occurrence of an SBML species, compartment or
parameter symbol in an assignment statement\'s \'math\' subelement.  Let the
directed arcs point from the statement assigning the symbol to the
statements that contain the symbol in their \'math\' subelement expressions.
This graph must be acyclic.

<p> SBML does not specify when or how often rules should be evaluated.
Eliminating algebraic loops ensures that assignment statements can be
evaluated any number of times without the result of those evaluations
changing.  As an example, consider the set of equations <em>x = x + 1</em>,
<em>y = z + 200</em> and <em>z = y + 100</em>.  If this set of equations
were interpreted as a set of assignment statements, it would be invalid
because the rule for <em>x</em> refers to <em>x</em> (exhibiting one type
of loop), and the rule for <em>y</em> refers to <em>z</em> while the rule
for <em>z</em> refers back to <em>y</em> (exhibiting another type of loop).
Conversely, the following set of equations would constitute a valid set of
assignment statements: <em>x = 10</em>, <em>y = z + 200</em>, and <em>z = x
+ 100</em>.

<h3>A model must not be overdetermined</h3>

<p> An SBML model must not be overdetermined; that is, a model must not
define more equations than there are unknowns in a model.  An SBML model
that does not contain {@link AlgebraicRule} structures cannot be overdetermined.

<p> LibSBML 3.3 implements the static analysis procedure described in
Appendix D of the SBML Level&nbsp;2 Version&nbsp;4 specification for
assessing whether a model is overdetermined.

<p> (In summary, assessing whether a given continuous, deterministic,
mathematical model is overdetermined does not require dynamic analysis; it
can be done by analyzing the system of equations created from the model.
One approach is to construct a bipartite graph in which one set of vertices
represents the variables and the other the set of vertices represents the
equations.  Place edges between vertices such that variables in the system
are linked to the equations that determine them.  For algebraic equations,
there will be edges between the equation and each variable occurring in the
equation.  For ordinary differential equations (such as those defined by
rate rules or implied by the reaction rate definitions), there will be a
single edge between the equation and the variable determined by that
differential equation.  A mathematical model is overdetermined if the
maximal matchings of the bipartite graph contain disconnected vertexes
representing equations.  If one maximal matching has this property, then
all the maximal matchings will have this property; i.e., it is only
necessary to find one maximal matching.)

<h3><a class=\'anchor\' name=\'RuleType_t\'>RuleType_t</a></h3>

<p> SBML Level 1 uses a different scheme than SBML Level&nbsp;2 for
distinguishing rules; specifically, it uses an attribute whose value is
drawn from an enumeration.  LibSBML supports this using methods that work
with the RuleType_t enumeration.

<p>
<center>
<table width=\'90%\' cellspacing=\'1\' cellpadding=\'1\' border=\'0\' class=\'normal-font\'>
 <tr style=\'background: lightgray\' class=\'normal-font\'>
     <td><strong>Enumerator</strong></td>
     <td><strong>Meaning</strong></td>
 </tr>
<tr><td><em>RULE_TYPE_RATE</em></td><td>Indicates the rule is a \'rate\' rule.</td>
<tr><td><em>RULE_TYPE_SCALAR</em></td><td>Indicates the rule is a \'scalar\' rule.</td>
<tr><td><em>RULE_TYPE_UNKNOWN</em></td><td>Indicates the rule type is unknown or not
yet set.</td>
</table>
</center>


 * <p>
 * <!-- leave this next break as-is to work around some doxygen bug -->
 */
"

%typemap(javaimports) AlgebraicRule "
/** 
 * LibSBML implementation of SBML\'s AlgebraicRule construct.
 * <p>
 * The rule type {@link AlgebraicRule} is derived from the parent class {@link Rule}.  It
 * is used to express equations that are neither assignments of model
 * variables nor rates of change.  {@link AlgebraicRule} does not add any
 * attributes to the basic {@link Rule}; its role is simply to distinguish this
 * case from the other cases.
 * <p>
 * In the context of a simulation, algebraic rules are in effect at all
 * times, <em>t</em> &#8805; <em>0</em>.  For purposes of evaluating
 * expressions that involve the delay \'csymbol\' (see the SBML
 * specification), algebraic rules are considered to apply also at
 * <em>t</em> &#8804; <em>0</em>.  The SBML Level&nbsp;2 Version&nbsp;4
 * specification provides additional information about the semantics of
 * assignments, rules, and entity values for simulation time <em>t</em>
 * &#8804; <em>0</em>.
 * <p>
 * The ability to define arbitrary algebraic expressions in an SBML model
 * introduces the possibility that a model is mathematically overdetermined
 * by the overall system of equations constructed from its rules and
 * reactions.  An SBML model must not be overdetermined; see the
 * description of {@link Rule} and also the SBML Level&nbsp;2 Version&nbsp;4
 * specification.  An SBML model that does not contain {@link AlgebraicRule}
 * structures cannot be overdetermined.
 * <p>
 * Assessing whether a given continuous, deterministic, mathematical model
 * is overdetermined does not require dynamic analysis; it can be done by
 * analyzing the system of equations created from the model.  One approach
 * is to construct a bipartite graph in which one set of vertices
 * represents the variables and the other the set of vertices represents
 * the equations.  Place edges between vertices such that variables in the
 * system are linked to the equations that determine them.  For algebraic
 * equations, there will be edges between the equation and each variable
 * occurring in the equation.  For ordinary differential equations (such as
 * those defined by rate rules or implied by the reaction rate
 * definitions), there will be a single edge between the equation and the
 * variable determined by that differential equation.  A mathematical model
 * is overdetermined if the maximal matchings of the bipartite graph
 * contain disconnected vertexes representing equations.  (If one maximal
 * matching has this property, then all the maximal matchings will have
 * this property; i.e., it is only necessary to find one maximal matching.)
 * Appendix D of the SBML Level&nbsp;2 Version&nbsp;4 specification
 * describes a method of applying this procedure to specific SBML data
 * objects.
 * <p>
 * <h2>General summary of SBML rules</h2>
<p>
In SBML Level&nbsp;2, rules are separated into three subclasses for the
benefit of model analysis software.  The three subclasses are based on
the following three different possible functional forms (where <em>x</em> is
a variable, <em>f</em> is some arbitrary function returning a numerical
result, <b>V</b> is a vector of variables that does not include <em>x</em>,
and <b>W</b> is a vector of variables that may include <em>x</em>):

<p>
<center>
<table border=\'0\' cellpadding=\'0\' style=\'font-size: small\'>
<tr><td width=\'120px\'><em>Algebraic:</em></td><td width=\'250px\'>left-hand side is zero</td><td><em>0 = f(<b>W</b>)</em></td></tr>
<tr><td><em>Assignment:</em></td><td>left-hand side is a scalar:</td><td><em>x = f(<b>V</b>)</em></td></tr>
<tr><td><em>Rate:</em></td><td>left-hand side is a rate-of-change:</td><td><em>dx/dt = f(<b>W</b>)</em></td></tr>
</table>
</center>

<p>
In their general form given above, there is little to distinguish
between <em>assignment</em> and <em>algebraic</em> rules.  They are treated as
separate cases for the following reasons:
<p>
<ul>
<li> <em>Assignment</em> rules can simply be evaluated to calculate
intermediate values for use in numerical methods.  They are statements
of equality that hold at all times.  (For assignments that are only
performed once, see {@link InitialAssignment}.)<p>

<li> SBML needs to place restrictions on assignment rules, for example
the restriction that assignment rules cannot contain algebraic loops.<p>

<li> Some simulators do not contain numerical solvers capable of solving
unconstrained algebraic equations, and providing more direct forms such
as assignment rules may enable those simulators to process models they
could not process if the same assignments were put in the form of
general algebraic equations;<p>

<li> Those simulators that <em>can</em> solve these algebraic equations make a
distinction between the different categories listed above; and<p>

<li> Some specialized numerical analyses of models may only be applicable
to models that do not contain <em>algebraic</em> rules.
</ul>

<p> The approach taken to covering these cases in SBML is to define an
abstract {@link Rule} structure containing a subelement, \'math\', to hold the
right-hand side expression, then to derive subtypes of {@link Rule} that add
attributes to distinguish the cases of algebraic, assignment and rate
rules.  The \'math\' subelement must contain a MathML expression defining the
mathematical formula of the rule.  This MathML formula must return a
numerical value.  The formula can be an arbitrary expression referencing
the variables and other entities in an SBML model.

<p> Each of the three subclasses of {@link Rule} (AssignmentRule, {@link AlgebraicRule},
{@link RateRule}) inherit the the \'math\' subelement and other fields from {@link SBase}.
The {@link AssignmentRule} and {@link RateRule} classes add an additional attribute,
\'variable\'.  See the definitions of {@link AssignmentRule}, {@link AlgebraicRule} and
{@link RateRule} for details about the structure and interpretation of each one.

<h2>Additional restrictions on SBML rules</h2>

<p> An important design goal of SBML rule semantics is to ensure that a
model\'s simulation and analysis results will not be dependent on when or
how often rules are evaluated.  To achieve this, SBML needs to place two
restrictions on rule use.  The first concerns algebraic loops in the system
of assignments in a model, and the second concerns overdetermined systems.

<h3>A model must not contain algebraic loops</h3>

<p> The combined set of {@link InitialAssignment}, {@link AssignmentRule} and {@link KineticLaw}
objects in a model constitute a set of assignment statements that should be
considered as a whole.  (A {@link KineticLaw} object is counted as an assignment
because it assigns a value to the symbol contained in the \'id\' attribute of
the {@link Reaction} object in which it is defined.)  This combined set of
assignment statements must not contain algebraic loops&mdash;dependency
chains between these statements must terminate.  To put this more formally,
consider a directed graph in which nodes are assignment statements and
directed arcs exist for each occurrence of an SBML species, compartment or
parameter symbol in an assignment statement\'s \'math\' subelement.  Let the
directed arcs point from the statement assigning the symbol to the
statements that contain the symbol in their \'math\' subelement expressions.
This graph must be acyclic.

<p> SBML does not specify when or how often rules should be evaluated.
Eliminating algebraic loops ensures that assignment statements can be
evaluated any number of times without the result of those evaluations
changing.  As an example, consider the set of equations <em>x = x + 1</em>,
<em>y = z + 200</em> and <em>z = y + 100</em>.  If this set of equations
were interpreted as a set of assignment statements, it would be invalid
because the rule for <em>x</em> refers to <em>x</em> (exhibiting one type
of loop), and the rule for <em>y</em> refers to <em>z</em> while the rule
for <em>z</em> refers back to <em>y</em> (exhibiting another type of loop).
Conversely, the following set of equations would constitute a valid set of
assignment statements: <em>x = 10</em>, <em>y = z + 200</em>, and <em>z = x
+ 100</em>.

<h3>A model must not be overdetermined</h3>

<p> An SBML model must not be overdetermined; that is, a model must not
define more equations than there are unknowns in a model.  An SBML model
that does not contain {@link AlgebraicRule} structures cannot be overdetermined.

<p> LibSBML 3.3 implements the static analysis procedure described in
Appendix D of the SBML Level&nbsp;2 Version&nbsp;4 specification for
assessing whether a model is overdetermined.

<p> (In summary, assessing whether a given continuous, deterministic,
mathematical model is overdetermined does not require dynamic analysis; it
can be done by analyzing the system of equations created from the model.
One approach is to construct a bipartite graph in which one set of vertices
represents the variables and the other the set of vertices represents the
equations.  Place edges between vertices such that variables in the system
are linked to the equations that determine them.  For algebraic equations,
there will be edges between the equation and each variable occurring in the
equation.  For ordinary differential equations (such as those defined by
rate rules or implied by the reaction rate definitions), there will be a
single edge between the equation and the variable determined by that
differential equation.  A mathematical model is overdetermined if the
maximal matchings of the bipartite graph contain disconnected vertexes
representing equations.  If one maximal matching has this property, then
all the maximal matchings will have this property; i.e., it is only
necessary to find one maximal matching.)

<h3><a class=\'anchor\' name=\'RuleType_t\'>RuleType_t</a></h3>

<p> SBML Level 1 uses a different scheme than SBML Level&nbsp;2 for
distinguishing rules; specifically, it uses an attribute whose value is
drawn from an enumeration.  LibSBML supports this using methods that work
with the RuleType_t enumeration.

<p>
<center>
<table width=\'90%\' cellspacing=\'1\' cellpadding=\'1\' border=\'0\' class=\'normal-font\'>
 <tr style=\'background: lightgray\' class=\'normal-font\'>
     <td><strong>Enumerator</strong></td>
     <td><strong>Meaning</strong></td>
 </tr>
<tr><td><em>RULE_TYPE_RATE</em></td><td>Indicates the rule is a \'rate\' rule.</td>
<tr><td><em>RULE_TYPE_SCALAR</em></td><td>Indicates the rule is a \'scalar\' rule.</td>
<tr><td><em>RULE_TYPE_UNKNOWN</em></td><td>Indicates the rule type is unknown or not
yet set.</td>
</table>
</center>


 * <p>
 * <!-- leave this next break as-is to work around some doxygen bug -->
 */
"

%typemap(javaimports) AssignmentRule "
/** 
 * LibSBML implementation of SBML\'s AssignmentRule construct.
 * <p>
 * The rule type {@link AssignmentRule} is derived from the parent class {@link Rule}.  It
 * is used to express equations that set the values of variables.  The
 * left-hand side (the attribute named \'variable\') of an assignment rule
 * can refer to the identifier of a {@link Species}, {@link Compartment}, or {@link Parameter}
 * object in the model (but not a {@link Reaction}).  The entity identified must
 * not have its \'constant\' attribute set to <code>true</code>.  The effects of an
 * {@link AssignmentRule} are in general terms the same, but differ in the precise
 * details depending on the type of variable being set:
 * <ul>
 * <li> <em>In the case of a species</em>, an {@link AssignmentRule} sets the
 * referenced species\' quantity (<em>concentration</em> or <em>amount of
 * substance</em>) to the value determined by the formula in the subelement
 * \'math\' of the {@link AssignmentRule} object.  The units of the formula in \'math\'
 * should (in SBML Level&nbsp;2 Version&nbsp;4) or must (in previous
 * Versions) be the same as the <em>units of the species</em> for the
 * species identified by the \'variable\' attribute of the {@link AssignmentRule}.
 * <em>Restrictions</em>: There must not be both an {@link AssignmentRule}
 * \'variable\' attribute and a {@link SpeciesReference} \'species\' attribute having
 * the same value, unless that species has its \'boundaryCondition\'
 * attribute set to <code>true</code>.  In other words, an assignment rule cannot be
 * defined for a species that is created or destroyed in a reaction unless
 * that species is defined as a boundary condition in the model.
 * <p>
 * <li> <em>In the case of a compartment</em>, an {@link AssignmentRule} sets the
 * referenced compartment\'s size to the value determined by the formula in
 * the \'math\' subelement of the {@link AssignmentRule} object.  The overall units
 * of the formula in \'math\' should (in SBML Level&nbsp;2 Version&nbsp;4) or
 * must (in previous Versions) be the same as the units of the size of the
 * compartment.
 * <p>
 * <li> <em>In the case of a parameter</em>, an {@link AssignmentRule} sets the
 * referenced parameter\'s value to that determined by the formula in the
 * \'math\' subelement of the {@link AssignmentRule} object.  The overall units of
 * the formula in the \'math\' subelement should (in SBML Level&nbsp;2
 * Version&nbsp;4) or must (in previous Versions) be the same as the units
 * defined for the parameter.  </ul>
 * <p>
 * In the context of a simulation, assignment rules are in effect at all
 * times, <em>t</em> &#8805; <em>0</em>.  For purposes of evaluating
 * expressions that involve the <em>delay</em> \'csymbol\' (see the SBML
 * Level&nbsp;2 specification), assignment rules are considered to apply
 * also at <em>t</em> &#8804; <em>0</em>.  The SBML Level&nbsp;2
 * Version&nbsp;4 specification provides additional information about the
 * semantics of assignments, rules, and entity values for simulation time
 * <em>t</em> &#8804; <em>0</em>.
 * <p>
 * A model must not contain more than one {@link AssignmentRule} or {@link RateRule}
 * object having the same value of \'variable\'; in other words, in the set
 * of all assignment rules and rate rules in an SBML model, each variable
 * appearing in the left-hand sides can only appear once.  This simply
 * follows from the fact that an indeterminate system would result if a
 * model contained more than one assignment rule for the same variable or
 * both an assignment rule and a rate rule for the same variable.
 * <p>
 * Similarly, a model must also not contain <em>both</em> an {@link AssignmentRule}
 * and an {@link InitialAssignment} for the same variable, because both kinds of
 * constructs apply prior to and at the start of simulation time, i.e.,
 * <em>t</em> &#8804; <em>0</em>.  If a model contained both an initial
 * assignment and an assignment rule for the same variable, an
 * indeterminate system would result.
 * <p>
 * The value calculated by an {@link AssignmentRule} object overrides the value
 * assigned to the given symbol by the object defining that symbol.  For
 * example, if a {@link Compartment}\'s \'size\' attribute value is set in its
 * definition, and the model also contains an {@link AssignmentRule} having that
 * compartment\'s \'id\' as its \'variable\' value, then the \'size\' assigned in
 * the {@link Compartment} definition is ignored and the value assigned based on
 * the computation defined in the {@link AssignmentRule}.  This does <em>not</em>
 * mean that a definition for a given symbol can be omitted if there is an
 * {@link AssignmentRule} object for it.  For example, there must be a {@link Parameter}
 * definition for a given parameter if there is an {@link AssignmentRule} for that
 * parameter.  It is only a question of which value definition takes
 * precedence.
 * <p>
 * <h2>General summary of SBML rules</h2>
<p>
In SBML Level&nbsp;2, rules are separated into three subclasses for the
benefit of model analysis software.  The three subclasses are based on
the following three different possible functional forms (where <em>x</em> is
a variable, <em>f</em> is some arbitrary function returning a numerical
result, <b>V</b> is a vector of variables that does not include <em>x</em>,
and <b>W</b> is a vector of variables that may include <em>x</em>):

<p>
<center>
<table border=\'0\' cellpadding=\'0\' style=\'font-size: small\'>
<tr><td width=\'120px\'><em>Algebraic:</em></td><td width=\'250px\'>left-hand side is zero</td><td><em>0 = f(<b>W</b>)</em></td></tr>
<tr><td><em>Assignment:</em></td><td>left-hand side is a scalar:</td><td><em>x = f(<b>V</b>)</em></td></tr>
<tr><td><em>Rate:</em></td><td>left-hand side is a rate-of-change:</td><td><em>dx/dt = f(<b>W</b>)</em></td></tr>
</table>
</center>

<p>
In their general form given above, there is little to distinguish
between <em>assignment</em> and <em>algebraic</em> rules.  They are treated as
separate cases for the following reasons:
<p>
<ul>
<li> <em>Assignment</em> rules can simply be evaluated to calculate
intermediate values for use in numerical methods.  They are statements
of equality that hold at all times.  (For assignments that are only
performed once, see {@link InitialAssignment}.)<p>

<li> SBML needs to place restrictions on assignment rules, for example
the restriction that assignment rules cannot contain algebraic loops.<p>

<li> Some simulators do not contain numerical solvers capable of solving
unconstrained algebraic equations, and providing more direct forms such
as assignment rules may enable those simulators to process models they
could not process if the same assignments were put in the form of
general algebraic equations;<p>

<li> Those simulators that <em>can</em> solve these algebraic equations make a
distinction between the different categories listed above; and<p>

<li> Some specialized numerical analyses of models may only be applicable
to models that do not contain <em>algebraic</em> rules.
</ul>

<p> The approach taken to covering these cases in SBML is to define an
abstract {@link Rule} structure containing a subelement, \'math\', to hold the
right-hand side expression, then to derive subtypes of {@link Rule} that add
attributes to distinguish the cases of algebraic, assignment and rate
rules.  The \'math\' subelement must contain a MathML expression defining the
mathematical formula of the rule.  This MathML formula must return a
numerical value.  The formula can be an arbitrary expression referencing
the variables and other entities in an SBML model.

<p> Each of the three subclasses of {@link Rule} (AssignmentRule, {@link AlgebraicRule},
{@link RateRule}) inherit the the \'math\' subelement and other fields from {@link SBase}.
The {@link AssignmentRule} and {@link RateRule} classes add an additional attribute,
\'variable\'.  See the definitions of {@link AssignmentRule}, {@link AlgebraicRule} and
{@link RateRule} for details about the structure and interpretation of each one.

<h2>Additional restrictions on SBML rules</h2>

<p> An important design goal of SBML rule semantics is to ensure that a
model\'s simulation and analysis results will not be dependent on when or
how often rules are evaluated.  To achieve this, SBML needs to place two
restrictions on rule use.  The first concerns algebraic loops in the system
of assignments in a model, and the second concerns overdetermined systems.

<h3>A model must not contain algebraic loops</h3>

<p> The combined set of {@link InitialAssignment}, {@link AssignmentRule} and {@link KineticLaw}
objects in a model constitute a set of assignment statements that should be
considered as a whole.  (A {@link KineticLaw} object is counted as an assignment
because it assigns a value to the symbol contained in the \'id\' attribute of
the {@link Reaction} object in which it is defined.)  This combined set of
assignment statements must not contain algebraic loops&mdash;dependency
chains between these statements must terminate.  To put this more formally,
consider a directed graph in which nodes are assignment statements and
directed arcs exist for each occurrence of an SBML species, compartment or
parameter symbol in an assignment statement\'s \'math\' subelement.  Let the
directed arcs point from the statement assigning the symbol to the
statements that contain the symbol in their \'math\' subelement expressions.
This graph must be acyclic.

<p> SBML does not specify when or how often rules should be evaluated.
Eliminating algebraic loops ensures that assignment statements can be
evaluated any number of times without the result of those evaluations
changing.  As an example, consider the set of equations <em>x = x + 1</em>,
<em>y = z + 200</em> and <em>z = y + 100</em>.  If this set of equations
were interpreted as a set of assignment statements, it would be invalid
because the rule for <em>x</em> refers to <em>x</em> (exhibiting one type
of loop), and the rule for <em>y</em> refers to <em>z</em> while the rule
for <em>z</em> refers back to <em>y</em> (exhibiting another type of loop).
Conversely, the following set of equations would constitute a valid set of
assignment statements: <em>x = 10</em>, <em>y = z + 200</em>, and <em>z = x
+ 100</em>.

<h3>A model must not be overdetermined</h3>

<p> An SBML model must not be overdetermined; that is, a model must not
define more equations than there are unknowns in a model.  An SBML model
that does not contain {@link AlgebraicRule} structures cannot be overdetermined.

<p> LibSBML 3.3 implements the static analysis procedure described in
Appendix D of the SBML Level&nbsp;2 Version&nbsp;4 specification for
assessing whether a model is overdetermined.

<p> (In summary, assessing whether a given continuous, deterministic,
mathematical model is overdetermined does not require dynamic analysis; it
can be done by analyzing the system of equations created from the model.
One approach is to construct a bipartite graph in which one set of vertices
represents the variables and the other the set of vertices represents the
equations.  Place edges between vertices such that variables in the system
are linked to the equations that determine them.  For algebraic equations,
there will be edges between the equation and each variable occurring in the
equation.  For ordinary differential equations (such as those defined by
rate rules or implied by the reaction rate definitions), there will be a
single edge between the equation and the variable determined by that
differential equation.  A mathematical model is overdetermined if the
maximal matchings of the bipartite graph contain disconnected vertexes
representing equations.  If one maximal matching has this property, then
all the maximal matchings will have this property; i.e., it is only
necessary to find one maximal matching.)

<h3><a class=\'anchor\' name=\'RuleType_t\'>RuleType_t</a></h3>

<p> SBML Level 1 uses a different scheme than SBML Level&nbsp;2 for
distinguishing rules; specifically, it uses an attribute whose value is
drawn from an enumeration.  LibSBML supports this using methods that work
with the RuleType_t enumeration.

<p>
<center>
<table width=\'90%\' cellspacing=\'1\' cellpadding=\'1\' border=\'0\' class=\'normal-font\'>
 <tr style=\'background: lightgray\' class=\'normal-font\'>
     <td><strong>Enumerator</strong></td>
     <td><strong>Meaning</strong></td>
 </tr>
<tr><td><em>RULE_TYPE_RATE</em></td><td>Indicates the rule is a \'rate\' rule.</td>
<tr><td><em>RULE_TYPE_SCALAR</em></td><td>Indicates the rule is a \'scalar\' rule.</td>
<tr><td><em>RULE_TYPE_UNKNOWN</em></td><td>Indicates the rule type is unknown or not
yet set.</td>
</table>
</center>


 * <p>
 * <!-- leave this next break as-is to work around some doxygen bug -->
 */
"

%typemap(javaimports) RateRule "
/** 
 * LibSBML implementation of SBML\'s RateRule construct.
 * <p>
 * The rule type {@link RateRule} is derived from the parent class {@link Rule}.  It is
 * used to express equations that determine the rates of change of
 * variables.  The left-hand side (the \'variable\' attribute) can refer to
 * the identifier of a species, compartment, or parameter (but not a
 * reaction).  The entity identified must have its \'constant\' attribute set
 * to <code>false</code>.  The effects of a {@link RateRule} are in general terms the same,
 * but differ in the precise details depending on which variable is being
 * set:
 * <ul>
 * <li> <em>In the case of a species</em>, a {@link RateRule} sets the rate of
 * change of the species\' quantity (<em>concentration</em> or <em>amount of
 * substance</em>) to the value determined by the formula in the \'math\'
 * subelement of the {@link RateRule} object.  The overall units of the formula in
 * \'math\' should (in SBML Level&nbsp;2 Version&nbsp;4) or must (in previous
 * Versions) be <em>species quantity</em>/<em>time</em>, where the
 * <em>time</em> units are the SBML predefined units of time and the
 * <em>species quantity</em> units are the <em>units of the species</em>.
 * <em>Restrictions</em>: There must not be both a {@link RateRule} \'variable\'
 * attribute and a {@link SpeciesReference} \'species\' attribute having the same
 * value, unless that species has its \'boundaryCondition\' attribute is set
 * to <code>true</code>.  This means a rate rule cannot be defined for a species that
 * is created or destroyed in a reaction, unless that species is defined as
 * a boundary condition in the model.
 * <p>
 * <li> <em>In the case of a compartment</em>, a {@link RateRule} sets the rate of
 * change of the compartment\'s size to the value determined by the formula
 * in the \'math\' subelement of the {@link RateRule} object.  The overall units of
 * the formula should (in SBML Level&nbsp;2 Version&nbsp;4) or must (in previous
 * Versions) be <em>size</em>/<em>time</em>, where the <em>time</em>
 * units are the SBML predefined units of time and the <em>size</em> units
 * are the units of size on the compartment.
 * <p>
 * <li> <em>In the case of a parameter</em>, a {@link RateRule} sets the rate of
 * change of the parameter\'s value to that determined by the formula in the
 * \'math\' subelement of the {@link RateRule} object.  The overall units of the
 * formula should (in SBML Level&nbsp;2 Version&nbsp;4) or must (in previous
 * Versions) be <em>x</em>/<em>time</em>, where <em>x</em> are the units
 * of the parameter.
 * </ul>
 * <p>
 * In the context of a simulation, rate rules are in effect for simulation
 * time <em>t</em> &lt; <em>0</em>.  The SBML Level&nbsp;2 Version&nbsp;4
 * specification provides additional information about the semantics of
 * assignments, rules, and entity values for simulation time <em>t</em>
 * &#8804; <em>0</em>.
 * <p>
 * As mentioned in the description of {@link AssignmentRule}, a model must not
 * contain more than one {@link RateRule} or {@link AssignmentRule} object having the same
 * value of \'variable\'; in other words, in the set of all assignment rules
 * and rate rules in an SBML model, each variable appearing in the
 * left-hand sides can only appear once.  This simply follows from the fact
 * that an indeterminate system would result if a model contained more than
 * one assignment rule for the same variable or both an assignment rule and
 * a rate rule for the same variable.
 * <p>
 * <h2>General summary of SBML rules</h2>
<p>
In SBML Level&nbsp;2, rules are separated into three subclasses for the
benefit of model analysis software.  The three subclasses are based on
the following three different possible functional forms (where <em>x</em> is
a variable, <em>f</em> is some arbitrary function returning a numerical
result, <b>V</b> is a vector of variables that does not include <em>x</em>,
and <b>W</b> is a vector of variables that may include <em>x</em>):

<p>
<center>
<table border=\'0\' cellpadding=\'0\' style=\'font-size: small\'>
<tr><td width=\'120px\'><em>Algebraic:</em></td><td width=\'250px\'>left-hand side is zero</td><td><em>0 = f(<b>W</b>)</em></td></tr>
<tr><td><em>Assignment:</em></td><td>left-hand side is a scalar:</td><td><em>x = f(<b>V</b>)</em></td></tr>
<tr><td><em>Rate:</em></td><td>left-hand side is a rate-of-change:</td><td><em>dx/dt = f(<b>W</b>)</em></td></tr>
</table>
</center>

<p>
In their general form given above, there is little to distinguish
between <em>assignment</em> and <em>algebraic</em> rules.  They are treated as
separate cases for the following reasons:
<p>
<ul>
<li> <em>Assignment</em> rules can simply be evaluated to calculate
intermediate values for use in numerical methods.  They are statements
of equality that hold at all times.  (For assignments that are only
performed once, see {@link InitialAssignment}.)<p>

<li> SBML needs to place restrictions on assignment rules, for example
the restriction that assignment rules cannot contain algebraic loops.<p>

<li> Some simulators do not contain numerical solvers capable of solving
unconstrained algebraic equations, and providing more direct forms such
as assignment rules may enable those simulators to process models they
could not process if the same assignments were put in the form of
general algebraic equations;<p>

<li> Those simulators that <em>can</em> solve these algebraic equations make a
distinction between the different categories listed above; and<p>

<li> Some specialized numerical analyses of models may only be applicable
to models that do not contain <em>algebraic</em> rules.
</ul>

<p> The approach taken to covering these cases in SBML is to define an
abstract {@link Rule} structure containing a subelement, \'math\', to hold the
right-hand side expression, then to derive subtypes of {@link Rule} that add
attributes to distinguish the cases of algebraic, assignment and rate
rules.  The \'math\' subelement must contain a MathML expression defining the
mathematical formula of the rule.  This MathML formula must return a
numerical value.  The formula can be an arbitrary expression referencing
the variables and other entities in an SBML model.

<p> Each of the three subclasses of {@link Rule} (AssignmentRule, {@link AlgebraicRule},
{@link RateRule}) inherit the the \'math\' subelement and other fields from {@link SBase}.
The {@link AssignmentRule} and {@link RateRule} classes add an additional attribute,
\'variable\'.  See the definitions of {@link AssignmentRule}, {@link AlgebraicRule} and
{@link RateRule} for details about the structure and interpretation of each one.

<h2>Additional restrictions on SBML rules</h2>

<p> An important design goal of SBML rule semantics is to ensure that a
model\'s simulation and analysis results will not be dependent on when or
how often rules are evaluated.  To achieve this, SBML needs to place two
restrictions on rule use.  The first concerns algebraic loops in the system
of assignments in a model, and the second concerns overdetermined systems.

<h3>A model must not contain algebraic loops</h3>

<p> The combined set of {@link InitialAssignment}, {@link AssignmentRule} and {@link KineticLaw}
objects in a model constitute a set of assignment statements that should be
considered as a whole.  (A {@link KineticLaw} object is counted as an assignment
because it assigns a value to the symbol contained in the \'id\' attribute of
the {@link Reaction} object in which it is defined.)  This combined set of
assignment statements must not contain algebraic loops&mdash;dependency
chains between these statements must terminate.  To put this more formally,
consider a directed graph in which nodes are assignment statements and
directed arcs exist for each occurrence of an SBML species, compartment or
parameter symbol in an assignment statement\'s \'math\' subelement.  Let the
directed arcs point from the statement assigning the symbol to the
statements that contain the symbol in their \'math\' subelement expressions.
This graph must be acyclic.

<p> SBML does not specify when or how often rules should be evaluated.
Eliminating algebraic loops ensures that assignment statements can be
evaluated any number of times without the result of those evaluations
changing.  As an example, consider the set of equations <em>x = x + 1</em>,
<em>y = z + 200</em> and <em>z = y + 100</em>.  If this set of equations
were interpreted as a set of assignment statements, it would be invalid
because the rule for <em>x</em> refers to <em>x</em> (exhibiting one type
of loop), and the rule for <em>y</em> refers to <em>z</em> while the rule
for <em>z</em> refers back to <em>y</em> (exhibiting another type of loop).
Conversely, the following set of equations would constitute a valid set of
assignment statements: <em>x = 10</em>, <em>y = z + 200</em>, and <em>z = x
+ 100</em>.

<h3>A model must not be overdetermined</h3>

<p> An SBML model must not be overdetermined; that is, a model must not
define more equations than there are unknowns in a model.  An SBML model
that does not contain {@link AlgebraicRule} structures cannot be overdetermined.

<p> LibSBML 3.3 implements the static analysis procedure described in
Appendix D of the SBML Level&nbsp;2 Version&nbsp;4 specification for
assessing whether a model is overdetermined.

<p> (In summary, assessing whether a given continuous, deterministic,
mathematical model is overdetermined does not require dynamic analysis; it
can be done by analyzing the system of equations created from the model.
One approach is to construct a bipartite graph in which one set of vertices
represents the variables and the other the set of vertices represents the
equations.  Place edges between vertices such that variables in the system
are linked to the equations that determine them.  For algebraic equations,
there will be edges between the equation and each variable occurring in the
equation.  For ordinary differential equations (such as those defined by
rate rules or implied by the reaction rate definitions), there will be a
single edge between the equation and the variable determined by that
differential equation.  A mathematical model is overdetermined if the
maximal matchings of the bipartite graph contain disconnected vertexes
representing equations.  If one maximal matching has this property, then
all the maximal matchings will have this property; i.e., it is only
necessary to find one maximal matching.)

<h3><a class=\'anchor\' name=\'RuleType_t\'>RuleType_t</a></h3>

<p> SBML Level 1 uses a different scheme than SBML Level&nbsp;2 for
distinguishing rules; specifically, it uses an attribute whose value is
drawn from an enumeration.  LibSBML supports this using methods that work
with the RuleType_t enumeration.

<p>
<center>
<table width=\'90%\' cellspacing=\'1\' cellpadding=\'1\' border=\'0\' class=\'normal-font\'>
 <tr style=\'background: lightgray\' class=\'normal-font\'>
     <td><strong>Enumerator</strong></td>
     <td><strong>Meaning</strong></td>
 </tr>
<tr><td><em>RULE_TYPE_RATE</em></td><td>Indicates the rule is a \'rate\' rule.</td>
<tr><td><em>RULE_TYPE_SCALAR</em></td><td>Indicates the rule is a \'scalar\' rule.</td>
<tr><td><em>RULE_TYPE_UNKNOWN</em></td><td>Indicates the rule type is unknown or not
yet set.</td>
</table>
</center>


 * <p>
 * <!-- leave this next break as-is to work around some doxygen bug -->
 */
"

%typemap(javaimports) ListOfRules "
/** 
 * LibSBML implementation of SBML\'s ListOfRules construct.
 * <p>
 * The various ListOf___ classes in SBML are merely containers used for
 * organizing the main components of an SBML model.  All are derived from
 * the abstract class {@link SBase}, and inherit the various attributes and
 * subelements of {@link SBase}, such as \'metaid\' as and \'annotation\'.  The
 * ListOf___ classes do not add any attributes of their own.
 * <p>
 * The relationship between the lists and the rest of an SBML model is
 * illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):
 * <p>
 * <center><img src=\'listof-illustration.jpg\'></center><br>
 * 
 * <p>
 * Readers may wonder about the motivations for using the ListOf___
 * containers.  A simpler approach in XML might be to place the components
 * all directly at the top level of the model definition.  We chose instead
 * to group them within XML elements named after {@link ListOf}<em>Classname</em>,
 * in part because we believe this helps organize the components and makes
 * visual reading of models in XML easier.  More importantly, the fact that
 * the container classes are derived from {@link SBase} means that software tools
 * can add information about the lists themselves into each list
 * container\'s \'annotation\'.
 * <p>
 * @see ListOfFunctionDefinitions
 * @see ListOfUnitDefinitions
 * @see ListOfCompartmentTypes
 * @see ListOfSpeciesTypes
 * @see ListOfCompartments
 * @see ListOfSpecies
 * @see ListOfParameters
 * @see ListOfInitialAssignments
 * @see ListOfRules
 * @see ListOfConstraints
 * @see ListOfReactions
 * @see ListOfEvents
 */
"

%javamethodmodifiers Rule::Rule(const Rule& orig) "
  /**
   * Copy constructor; creates a copy of this {@link Rule}.
   */
 public";

%javamethodmodifiers Rule::accept(SBMLVisitor& v) const "
  /**
   * Accepts the given {@link SBMLVisitor} for this instance of {@link Rule}.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used.
   * <p>
   * @return the result of calling <code>v.visit()</code>.
   */
 public";

%javamethodmodifiers Rule::clone() const "
  /**
   * Creates and returns a deep copy of this {@link Rule}.
   * <p>
   * @return a (deep) copy of this {@link Rule}.
   */
 public";

%javamethodmodifiers Rule::getFormula() const "
  /**
   * Returns the mathematical expression of this {@link Rule} in text-string form.
   * <p>
   * @return the formula for this {@link Rule}.
   */
 public";

%javamethodmodifiers Rule::getMath() const "
  /**
   * Get the mathematical formula of this {@link Rule}.
   * <p>
   * @return an {@link ASTNode}, the value of the \'math\' subelement of this {@link Rule}
   */
 public";

%javamethodmodifiers Rule::getVariable() const "
  /**
   * Get the value of the \'variable\' attribute of this {@link Rule}.
   * <p>
   * This applies to {@link AssignmentRule} and {@link RateRule}, which have explicit
   * left-hand sides in their equations.  {@link AlgebraicRule} does not have a
   * variable field.
   * <p>
   * @return the identifier string stored as the \'variable\' attribute value
   * in this {@link Rule}.
   */
 public";

%javamethodmodifiers Rule::getUnits() const "
  /**
   * (SBML Level&nbsp;1 ParameterRule only) Returns the units for the
   * mathematical formula of this {@link Rule}.
   * <p>
   * @return the identifier of the units for the expression of this {@link Rule}
   */
 public";

%javamethodmodifiers Rule::isSetFormula() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link Rule}\'s mathematical expression has been set.
   * <p>
   * This method is equivalent to isSetMath().  This version is present for
   * easier compatibility with SBML Level&nbsp;1, in which mathematical
   * formulas were written in text-string form.
   * <p>
   * @return <code>true</code> if the mathematical formula for this {@link Rule} has been
   * set, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers Rule::isSetMath() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link Rule}\'s mathematical expression has been set.
   * <p>
   * This method is equivalent to isSetFormula().
   * <p>
   * @return <code>true</code> if the formula (or equivalently the math) for this
   * {@link Rule} has been set, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers Rule::isSetVariable() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link Rule}\'s \'variable\' attribute has been set.
   * <p>
   * @return <code>true</code> if the variable of this {@link Rule} has been set, <code>false</code>
   * otherwise.
   */
 public";

%javamethodmodifiers Rule::isSetUnits() const "
  /**
   * (SBML Level&nbsp;1 ParameterRule only) Predicate returning <code>true</code> or 
   * <code>false</code> depending on whether this {@link Rule}\'s \'units\' attribute has been set.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the units for this {@link Rule} has been set, <code>false</code>
   * otherwise
   */
 public";

%javamethodmodifiers Rule::setFormula(const std::string& formula) "
  /**
   * Sets the \'math\' subelement of this {@link Rule} to an expression in
   * text-string form.
   * <p>
   * This is equivalent to setMath().  The provision of using text-string
   * formulas is retained for easier SBML Level&nbsp;1 compatibility.  The
   * formula is converted to an {@link ASTNode} internally.
   * <p>
   * @param formula a mathematical formula in text-string form.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_OBJECT
   */
 public";

%javamethodmodifiers Rule::setMath(const ASTNode* math) "
  /**
   * Sets the \'math\' subelement of this {@link Rule} to a copy of the given
   * {@link ASTNode}.
   * <p>
   * @param math the {@link ASTNode} structure of the mathematical formula.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_OBJECT
   */
 public";

%javamethodmodifiers Rule::setVariable(const std::string& sid) "
  /**
   * Sets the \'variable\' attribute of this {@link Rule}.
   * <p>
   * @param sid the identifier of a {@link Compartment}, {@link Species} or {@link Parameter}
   * elsewhere in the enclosing {@link Model} object.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   * <li> LIBSBML_UNEXPECTED_ATTRIBUTE
   */
 public";

%javamethodmodifiers Rule::setUnits(const std::string& sname) "
  /**
   * (SBML Level&nbsp;1 ParameterRule only) Sets the units for this {@link Rule}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param sname the identifier of the units
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   * <li> LIBSBML_UNEXPECTED_ATTRIBUTE
   */
 public";

%javamethodmodifiers Rule::unsetUnits "
  /**
   * (SBML Level&nbsp;1 ParameterRule only) Unsets the \'units\' for this {@link Rule}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_OPERATION_FAILED
   */
 public";

%javamethodmodifiers Rule::getDerivedUnitDefinition "
  /**
   * Calculates and returns a {@link UnitDefinition} that expresses the units of
   * measurement assumed for the \'math\' expression of this {@link Rule}.
   * <p>
   * The units are calculated based on the mathematical expression in the
   * {@link Rule} and the model quantities referenced by <code>&lt;ci&gt;</code>
   * elements used within that expression.  The getDerivedUnitDefinition()
   * method returns the calculated units.
   * <p>
   * Note that the functionality that facilitates unit analysis depends 
   * on the model as a whole.  Thus, in cases where the object has not 
   * been added to a model or the model itself is incomplete,
   * unit analysis is not possible and this method will return NULL.
   * <p>
   * @warning Note that it is possible the \'math\' expression in the {@link Rule}
   * contains pure numbers or parameters with undeclared units.  In those
   * cases, it is not possible to calculate the units of the overall
   * expression without making assumptions.  LibSBML does not make
   * assumptions about the units, and getDerivedUnitDefinition() only
   * returns the units as far as it is able to determine them.  For
   * example, in an expression <em>X + Y</em>, if <em>X</em> has
   * unambiguously-defined units and <em>Y</em> does not, it will return
   * the units of <em>X</em>.  <strong>It is important that callers also
   * invoke the method</strong> containsUndeclaredUnits() <strong>to
   * determine whether this situation holds</strong>.  Callers may wish to
   * take suitable actions in those scenarios.
   * <p>
   * @return a {@link UnitDefinition} that expresses the units of the math 
   * expression of this {@link Rule}.
   * <p>
   * @see #containsUndeclaredUnits()
   */
 public";

%javamethodmodifiers Rule::getDerivedUnitDefinition() const "
  /**
   * Calculates and returns a {@link UnitDefinition} that expresses the units of
   * measurement assumed for the \'math\' expression of this {@link Rule}.
   * <p>
   * The units are calculated based on the mathematical expression in the
   * {@link Rule} and the model quantities referenced by <code>&lt;ci&gt;</code>
   * elements used within that expression.  The getDerivedUnitDefinition()
   * method returns the calculated units.
   * <p>
   * Note that the functionality that facilitates unit analysis depends 
   * on the model as a whole.  Thus, in cases where the object has not 
   * been added to a model or the model itself is incomplete,
   * unit analysis is not possible and this method will return NULL.
   * <p>
   * @warning Note that it is possible the \'math\' expression in the {@link Rule}
   * contains pure numbers or parameters with undeclared units.  In those
   * cases, it is not possible to calculate the units of the overall
   * expression without making assumptions.  LibSBML does not make
   * assumptions about the units, and getDerivedUnitDefinition() only
   * returns the units as far as it is able to determine them.  For
   * example, in an expression <em>X + Y</em>, if <em>X</em> has
   * unambiguously-defined units and <em>Y</em> does not, it will return
   * the units of <em>X</em>.  <strong>It is important that callers also
   * invoke the method</strong> containsUndeclaredUnits() <strong>to
   * determine whether this situation holds</strong>.  Callers may wish to
   * take suitable actions in those scenarios.
   * <p>
   * @return a {@link UnitDefinition} that expresses the units of the math 
   * expression of this {@link Rule}.
   * <p>
   * @see #containsUndeclaredUnits()
   */
 public";

%javamethodmodifiers Rule::containsUndeclaredUnits "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether 
   * the math expression of this {@link Rule} contains
   * parameters/numbers with undeclared units.
   * <p>
   * @return <code>true</code> if the math expression of this {@link Rule}
   * includes parameters/numbers 
   * with undeclared units, <code>false</code> otherwise.
   * <p>
   * @note A return value of <code>true</code> indicates that the {@link UnitDefinition}
   * returned by getDerivedUnitDefinition() may not accurately represent
   * the units of the expression.
   * <p>
   * @see #getDerivedUnitDefinition()
   */
 public";

%javamethodmodifiers Rule::containsUndeclaredUnits() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether 
   * the math expression of this {@link Rule} contains
   * parameters/numbers with undeclared units.
   * <p>
   * @return <code>true</code> if the math expression of this {@link Rule}
   * includes parameters/numbers 
   * with undeclared units, <code>false</code> otherwise.
   * <p>
   * @note A return value of <code>true</code> indicates that the {@link UnitDefinition}
   * returned by getDerivedUnitDefinition() may not accurately represent
   * the units of the expression.
   * <p>
   * @see #getDerivedUnitDefinition()
   */
 public";

%javamethodmodifiers Rule::getType() const "
  /**
   * (SBML Level&nbsp;1) Get the type of rule this is.
   * <p>
   * @return the rule type (a value drawn from the enumeration <a
   * class=\'el\' href=\'#RuleType_t\'>RuleType_t</a>) of this {@link Rule}.  The value
   * will be either <code>RULE_TYPE_RATE</code> or <code>RULE_TYPE_SCALAR</code>.
   */
 public";

%javamethodmodifiers Rule::isAlgebraic() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link Rule} is an {@link AlgebraicRule}.
   * <p>
   * @return <code>true</code> if this {@link Rule} is an {@link AlgebraicRule}, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers Rule::isAssignment() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link Rule} is an {@link AssignmentRule}.
   * <p>
   * @return <code>true</code> if this {@link Rule} is an {@link AssignmentRule}, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers Rule::isCompartmentVolume() const "
  /**
   * (SBML Level&nbsp;1 only) Predicate returning <code>true</code> or <code>false</code> depending
   * on whether this {@link Rule} is an CompartmentVolumeRule.
   * <p>
   * @return <code>true</code> if this {@link Rule} is a CompartmentVolumeRule, <code>false</code>
   * otherwise.
   */
 public";

%javamethodmodifiers Rule::isParameter() const "
  /**
   * (SBML Level&nbsp;1 only) Predicate returning <code>true</code> or <code>false</code> depending
   * on whether this {@link Rule} is an ParameterRule.
   * <p>
   * @return <code>true</code> if this {@link Rule} is a ParameterRule, <code>false</code>
   * otherwise.
   */
 public";

%javamethodmodifiers Rule::isRate() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this {@link Rule}
   * is a {@link RateRule} (SBML Level&nbsp;2) or has a \'type\' attribute value of 
   * <code>\'rate\'</code> (SBML Level&nbsp;1).
   * <p>
   * @return <code>true</code> if this {@link Rule} is a {@link RateRule} (Level&nbsp;2) or has
   * type \'rate\' (Level&nbsp;1), <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers Rule::isScalar() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this {@link Rule}
   * is an {@link AssignmentRule} (SBML Level&nbsp;2) or has a \'type\' attribute value of
   * <code>\'scalar\'</code> (SBML Level&nbsp;1).
   * <p>
   * @return <code>true</code> if this {@link Rule} is an {@link AssignmentRule} (Level&nbsp;2) or has
   * type \'scalar\' (Level&nbsp;1), <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers Rule::isSpeciesConcentration() const "
  /**
   * (SBML Level&nbsp;1 only) Predicate returning <code>true</code> or <code>false</code> depending
   * on whether this {@link Rule} is an SpeciesConcentrationRule.
   * <p>
   * @return <code>true</code> if this {@link Rule} is a SpeciesConcentrationRule, <code>false</code>
   * otherwise.
   */
 public";

%javamethodmodifiers Rule::getTypeCode() const "
  /**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for this object, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers Rule::getL1TypeCode() const "
  /**
   * Returns  the SBML Level&nbsp;1 type code for this {@link Rule}, or <code>SBML_UNNOWN</code>.
   * <p>
   * @return the SBML Level&nbsp;1 typecode for this {@link Rule} (<code>SBML_COMPARTMENT_VOLUME_RULE</code>,
   * <code>SBML_PARAMETER_RULE</code>, or <code>SBML_SPECIES_CONCENTRATION_RULE</code>) or <code>SBML_UNKNOWN</code>
   * (default).
   */
 public";

%javamethodmodifiers Rule::getElementName() const "
  /**
   * Returns the XML element name of this object, which can be any
   * of a number of different strings depending on the SBML Level and the
   * kind of rule this is.
   * <p>
   * The rules as of libSBML 3.0.0 are the following:
   * <ul>
   * <li> (Level&nbsp;2) RateRule: returns <code>\'rateRule\'</code>
   * <li> (Level&nbsp;2) AssignmentRule: returns <code>\'assignmentRule\'</code> 
   * <li> (Level&nbsp;2) AlgebraicRule: returns <code>\'algebraicRule\'</code>
   * <li> (Level&nbsp;1 Version&nbsp;1) SpecieConcentrationRule: returns <code>\'specieConcentrationRule\'</code>
   * <li> (Level&nbsp;1 Version&nbsp;2) SpeciesConcentrationRule: returns <code>\'speciesConcentrationRule\'</code>
   * <li> (Level&nbsp;1) CompartmentVolumeRule: returns <code>\'compartmentVolumeRule\'</code>
   * <li> (Level&nbsp;1) ParameterRule: returns <code>\'parameterRule\'</code>
   * <li> Unknown rule type: returns <code>\'unknownRule\'</code>
   * </ul>
   * <p>
   * @return the name of this element
   */
 public";

%javamethodmodifiers Rule::writeElements(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write out their contained
   * SBML objects as XML elements.  Be sure to call your parents
   * implementation of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Rule::setL1TypeCode(SBMLTypeCode_t type) "
  /**
   * Sets the SBML Level&nbsp;1 typecode for this {@link Rule}.
   * <p>
   * @param type the SBML Level&nbsp;1 typecode for this {@link Rule} 
   * (<code>SBML_COMPARTMENT_VOLUME_RULE</code>, <code>SBML_PARAMETER_RULE</code>, 
   * or <code>SBML_SPECIES_CONCENTRATION_RULE</code>).
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   * <p>
   */
 public";

%javamethodmodifiers Rule::hasRequiredElements() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether all the
   * required elements for this {@link Rule} object have been set.
   * <p>
   * @note The required elements for a {@link Rule} object are: math
   * <p>
   * @return a boolean value indicating whether all the required
   * elements for this object have been defined.
   */
 public";

%javamethodmodifiers Rule::hasRequiredAttributes() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether all the
   * required attributes for this {@link Rule} object have been set.
   * <p>
   * @note The required elements for a {@link Rule} object are: math
   * <p>
   * @return a boolean value indicating whether all the required
   * elements for this object have been defined.
   */
 public";

%javamethodmodifiers Rule::getInternalId() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether all the
   * required attributes for this {@link Rule} object have been set.
   * <p>
   * @note The required elements for a {@link Rule} object are: math
   * <p>
   * @return a boolean value indicating whether all the required
   * elements for this object have been defined.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Rule::setInternalId(std::string id) "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether all the
   * required attributes for this {@link Rule} object have been set.
   * <p>
   * @note The required elements for a {@link Rule} object are: math
   * <p>
   * @return a boolean value indicating whether all the required
   * elements for this object have been defined.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Rule::getId() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether all the
   * required attributes for this {@link Rule} object have been set.
   * <p>
   * @note The required elements for a {@link Rule} object are: math
   * <p>
   * @return a boolean value indicating whether all the required
   * elements for this object have been defined.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Rule::Rule(  SBMLTypeCode_t      type , unsigned int        level , unsigned int        version ) "
  /**
   * Only subclasses may create Rules.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Rule::Rule(  SBMLTypeCode_t      type , SBMLNamespaces *    sbmlns ) "
  /**
   * Only subclasses may create Rules.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Rule::readOtherXML(XMLInputStream& stream) "
  /**
   * Subclasses should override this method to read (and store) XHTML,
   * MathML, etc. directly from the XMLInputStream.
   * <p>
   * @return true if the subclass read from the stream, false otherwise.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Rule::readAttributes(const XMLAttributes& attributes) "
  /**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Rule::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Rule::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Rule::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Rule::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Rule::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Rule::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Rule::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Rule::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Rule::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers AlgebraicRule::AlgebraicRule(unsigned int level, unsigned int version) "
  /**
   * Creates a new {@link AlgebraicRule} using the given SBML <code>level</code> and <code>version</code>
   * values.
   * <p>
   * @param level a long integer, the SBML Level to assign to this {@link AlgebraicRule}
   * <p>
   * @param version a long integer, the SBML Version to assign to this
   * {@link AlgebraicRule}
   * <p>
   * @note Once a {@link AlgebraicRule} has been added to an {@link SBMLDocument}, the <code>level</code>,
   * <code>version</code> for the document <em>override</em> those used
   * to create the {@link AlgebraicRule}.  Despite this, the ability to supply the values
   * at creation time is an important aid to creating valid SBML.  Knowledge of
   * the intented SBML Level and Version determine whether it is valid to
   * assign a particular value to an attribute, or whether it is valid to add
   * an object to an existing {@link SBMLDocument}.
   */
 public";

%javamethodmodifiers AlgebraicRule::AlgebraicRule(SBMLNamespaces* sbmlns) "
  /**
   * Creates a new {@link AlgebraicRule} using the given {@link SBMLNamespaces} object
   * <code>sbmlns</code>.
   * <p>
   * The {@link SBMLNamespaces} object encapsulates SBML Level/Version/namespaces
   * information.  It is used to communicate the SBML Level, Version, and
   * (in Level&nbsp;3) packages used in addition to SBML Level&nbsp; Core.
   * A common approach to using this class constructor is to create an
   * {@link SBMLNamespaces} object somewhere in a program, once, then pass it to
   * object constructors such as this one when needed.
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object.
   * <p>
   * @note Once a {@link AlgebraicRule} has been added to an {@link SBMLDocument}, the <code>level</code>,
   * <code>version</code> and <code>xmlns</code> namespaces for the document <em>override</em> those used
   * to create the {@link AlgebraicRule}.  Despite this, the ability to supply the values
   * at creation time is an important aid to creating valid SBML.  Knowledge of
   * the intented SBML Level and Version determine whether it is valid to
   * assign a particular value to an attribute, or whether it is valid to add
   * an object to an existing {@link SBMLDocument}.
   */
 public";

%javamethodmodifiers AlgebraicRule::clone() const "
  /**
   * Creates and returns a deep copy of this {@link Rule}.
   * <p>
   * @return a (deep) copy of this {@link Rule}.
   */
 public";

%javamethodmodifiers AlgebraicRule::accept(SBMLVisitor& v) const "
  /**
   * Accepts the given {@link SBMLVisitor} for this instance of {@link AlgebraicRule}.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used.
   * <p>
   * @return the result of calling <code>v.visit()</code>.
   */
 public";

%javamethodmodifiers AlgebraicRule::setInternalIdOnly "
  /**
   * sets the mInternalIdOnly flag
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers AlgebraicRule::getInternalIdOnly() const "
  /**
   * sets the mInternalIdOnly flag
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers AlgebraicRule::hasRequiredAttributes() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link AlgebraicRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link AlgebraicRule} object are:
   * formula (L1 only)
   */
 public";

%javamethodmodifiers AlgebraicRule::AlgebraicRule "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link AlgebraicRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link AlgebraicRule} object are:
   * formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers AlgebraicRule::AlgebraicRule "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link AlgebraicRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link AlgebraicRule} object are:
   * formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers AlgebraicRule::AlgebraicRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link AlgebraicRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link AlgebraicRule} object are:
   * formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers AlgebraicRule::AlgebraicRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link AlgebraicRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link AlgebraicRule} object are:
   * formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers AlgebraicRule::AlgebraicRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link AlgebraicRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link AlgebraicRule} object are:
   * formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers AlgebraicRule::AlgebraicRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link AlgebraicRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link AlgebraicRule} object are:
   * formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers AlgebraicRule::AlgebraicRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link AlgebraicRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link AlgebraicRule} object are:
   * formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers AlgebraicRule::AlgebraicRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link AlgebraicRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link AlgebraicRule} object are:
   * formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers AlgebraicRule::AlgebraicRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link AlgebraicRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link AlgebraicRule} object are:
   * formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers AlgebraicRule::AlgebraicRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link AlgebraicRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link AlgebraicRule} object are:
   * formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers AlgebraicRule::AlgebraicRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link AlgebraicRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link AlgebraicRule} object are:
   * formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers AlgebraicRule::AlgebraicRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link AlgebraicRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link AlgebraicRule} object are:
   * formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers AlgebraicRule::AlgebraicRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link AlgebraicRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link AlgebraicRule} object are:
   * formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers AlgebraicRule::AlgebraicRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link AlgebraicRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link AlgebraicRule} object are:
   * formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers AlgebraicRule::AlgebraicRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link AlgebraicRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link AlgebraicRule} object are:
   * formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers AlgebraicRule::AlgebraicRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link AlgebraicRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link AlgebraicRule} object are:
   * formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers AssignmentRule::AssignmentRule(unsigned int level, unsigned int version) "
  /**
   * Creates a new {@link AssignmentRule} using the given SBML <code>level</code> and <code>version</code>
   * values.
   * <p>
   * @param level a long integer, the SBML Level to assign to this {@link AssignmentRule}
   * <p>
   * @param version a long integer, the SBML Version to assign to this
   * {@link AssignmentRule}
   * <p>
   * @note Once a {@link AssignmentRule} has been added to an {@link SBMLDocument}, the <code>level</code>,
   * <code>version</code> for the document <em>override</em> those used
   * to create the {@link AssignmentRule}.  Despite this, the ability to supply the values
   * at creation time is an important aid to creating valid SBML.  Knowledge of
   * the intented SBML Level and Version determine whether it is valid to
   * assign a particular value to an attribute, or whether it is valid to add
   * an object to an existing {@link SBMLDocument}.
   */
 public";

%javamethodmodifiers AssignmentRule::AssignmentRule(SBMLNamespaces* sbmlns) "
  /**
   * Creates a new {@link AssignmentRule} using the given {@link SBMLNamespaces} object
   * <code>sbmlns</code>.
   * <p>
   * The {@link SBMLNamespaces} object encapsulates SBML Level/Version/namespaces
   * information.  It is used to communicate the SBML Level, Version, and
   * (in Level&nbsp;3) packages used in addition to SBML Level&nbsp; Core.
   * A common approach to using this class constructor is to create an
   * {@link SBMLNamespaces} object somewhere in a program, once, then pass it to
   * object constructors such as this one when needed.
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object.
   * <p>
   * @note Once a {@link AssignmentRule} has been added to an {@link SBMLDocument}, the <code>level</code>,
   * <code>version</code> and <code>xmlns</code> namespaces for the document <em>override</em> those used
   * to create the {@link AssignmentRule}.  Despite this, the ability to supply the values
   * at creation time is an important aid to creating valid SBML.  Knowledge of
   * the intented SBML Level and Version determine whether it is valid to
   * assign a particular value to an attribute, or whether it is valid to add
   * an object to an existing {@link SBMLDocument}.
   */
 public";

%javamethodmodifiers AssignmentRule::clone() const "
  /**
   * Creates and returns a deep copy of this {@link Rule}.
   * <p>
   * @return a (deep) copy of this {@link Rule}.
   */
 public";

%javamethodmodifiers AssignmentRule::accept(SBMLVisitor& v) const "
  /**
   * Accepts the given {@link SBMLVisitor} for this instance of {@link AssignmentRule}.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used.
   * <p>
   * @return the result of calling <code>v.visit()</code>.
   */
 public";

%javamethodmodifiers AssignmentRule::hasRequiredAttributes() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link AssignmentRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link AssignmentRule} object are:
   * variable (compartment/species/name in L1); formula (L1 only)
   */
 public";

%javamethodmodifiers AssignmentRule::AssignmentRule "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link AssignmentRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link AssignmentRule} object are:
   * variable (compartment/species/name in L1); formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers AssignmentRule::AssignmentRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link AssignmentRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link AssignmentRule} object are:
   * variable (compartment/species/name in L1); formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers AssignmentRule::AssignmentRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link AssignmentRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link AssignmentRule} object are:
   * variable (compartment/species/name in L1); formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers AssignmentRule::AssignmentRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link AssignmentRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link AssignmentRule} object are:
   * variable (compartment/species/name in L1); formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers AssignmentRule::AssignmentRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link AssignmentRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link AssignmentRule} object are:
   * variable (compartment/species/name in L1); formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers AssignmentRule::AssignmentRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link AssignmentRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link AssignmentRule} object are:
   * variable (compartment/species/name in L1); formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers AssignmentRule::AssignmentRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link AssignmentRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link AssignmentRule} object are:
   * variable (compartment/species/name in L1); formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers AssignmentRule::AssignmentRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link AssignmentRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link AssignmentRule} object are:
   * variable (compartment/species/name in L1); formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers AssignmentRule::AssignmentRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link AssignmentRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link AssignmentRule} object are:
   * variable (compartment/species/name in L1); formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers AssignmentRule::AssignmentRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link AssignmentRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link AssignmentRule} object are:
   * variable (compartment/species/name in L1); formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers AssignmentRule::AssignmentRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link AssignmentRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link AssignmentRule} object are:
   * variable (compartment/species/name in L1); formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers AssignmentRule::AssignmentRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link AssignmentRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link AssignmentRule} object are:
   * variable (compartment/species/name in L1); formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers AssignmentRule::AssignmentRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link AssignmentRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link AssignmentRule} object are:
   * variable (compartment/species/name in L1); formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers AssignmentRule::AssignmentRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link AssignmentRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link AssignmentRule} object are:
   * variable (compartment/species/name in L1); formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers AssignmentRule::AssignmentRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link AssignmentRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link AssignmentRule} object are:
   * variable (compartment/species/name in L1); formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers RateRule::RateRule(unsigned int level, unsigned int version) "
  /**
   * Creates a new {@link RateRule} using the given SBML <code>level</code> and <code>version</code>
   * values.
   * <p>
   * @param level a long integer, the SBML Level to assign to this {@link RateRule}
   * <p>
   * @param version a long integer, the SBML Version to assign to this
   * {@link RateRule}
   * <p>
   * @note Once a {@link RateRule} has been added to an {@link SBMLDocument}, the <code>level</code>,
   * <code>version</code> for the document <em>override</em> those used
   * to create the {@link RateRule}.  Despite this, the ability to supply the values
   * at creation time is an important aid to creating valid SBML.  Knowledge of
   * the intented SBML Level and Version determine whether it is valid to
   * assign a particular value to an attribute, or whether it is valid to add
   * an object to an existing {@link SBMLDocument}.
   */
 public";

%javamethodmodifiers RateRule::RateRule(SBMLNamespaces* sbmlns) "
  /**
   * Creates a new {@link RateRule} using the given {@link SBMLNamespaces} object
   * <code>sbmlns</code>.
   * <p>
   * The {@link SBMLNamespaces} object encapsulates SBML Level/Version/namespaces
   * information.  It is used to communicate the SBML Level, Version, and
   * (in Level&nbsp;3) packages used in addition to SBML Level&nbsp; Core.
   * A common approach to using this class constructor is to create an
   * {@link SBMLNamespaces} object somewhere in a program, once, then pass it to
   * object constructors such as this one when needed.
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object.
   * <p>
   * @note Once a {@link RateRule} has been added to an {@link SBMLDocument}, the <code>level</code>,
   * <code>version</code> and <code>xmlns</code> namespaces for the document <em>override</em> those used
   * to create the {@link RateRule}.  Despite this, the ability to supply the values
   * at creation time is an important aid to creating valid SBML.  Knowledge of
   * the intented SBML Level and Version determine whether it is valid to
   * assign a particular value to an attribute, or whether it is valid to add
   * an object to an existing {@link SBMLDocument}.
   */
 public";

%javamethodmodifiers RateRule::clone() const "
  /**
   * Creates and returns a deep copy of this {@link Rule}.
   * <p>
   * @return a (deep) copy of this {@link Rule}.
   */
 public";

%javamethodmodifiers RateRule::accept(SBMLVisitor& v) const "
  /**
   * Accepts the given {@link SBMLVisitor}.
   * <p>
   * @return the result of calling <code>v.visit()</code>, which indicates
   * whether or not the Visitor would like to visit the {@link Model}\'s next {@link Rule}
   * (if available).
   */
 public";

%javamethodmodifiers RateRule::hasRequiredAttributes() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link RateRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link RateRule} object are:
   * variable (compartment/species/name in L1); formula (L1 only)
   */
 public";

%javamethodmodifiers RateRule::RateRule "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link RateRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link RateRule} object are:
   * variable (compartment/species/name in L1); formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers RateRule::RateRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link RateRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link RateRule} object are:
   * variable (compartment/species/name in L1); formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers RateRule::RateRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link RateRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link RateRule} object are:
   * variable (compartment/species/name in L1); formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers RateRule::RateRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link RateRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link RateRule} object are:
   * variable (compartment/species/name in L1); formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers RateRule::RateRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link RateRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link RateRule} object are:
   * variable (compartment/species/name in L1); formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers RateRule::RateRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link RateRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link RateRule} object are:
   * variable (compartment/species/name in L1); formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers RateRule::RateRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link RateRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link RateRule} object are:
   * variable (compartment/species/name in L1); formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers RateRule::RateRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link RateRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link RateRule} object are:
   * variable (compartment/species/name in L1); formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers RateRule::RateRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link RateRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link RateRule} object are:
   * variable (compartment/species/name in L1); formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers RateRule::RateRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link RateRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link RateRule} object are:
   * variable (compartment/species/name in L1); formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers RateRule::RateRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link RateRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link RateRule} object are:
   * variable (compartment/species/name in L1); formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers RateRule::RateRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link RateRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link RateRule} object are:
   * variable (compartment/species/name in L1); formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers RateRule::RateRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link RateRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link RateRule} object are:
   * variable (compartment/species/name in L1); formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers RateRule::RateRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link RateRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link RateRule} object are:
   * variable (compartment/species/name in L1); formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers RateRule::RateRule() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link RateRule} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link RateRule} object are:
   * variable (compartment/species/name in L1); formula (L1 only)
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers ListOfRules::clone() const "
  /**
   * Creates and returns a deep copy of this {@link ListOfRules} instance.
   * <p>
   * @return a (deep) copy of this {@link ListOfRules}.
   */
 public";

%javamethodmodifiers ListOfRules::getTypeCode() const "
  /**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for this object, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers ListOfRules::getItemTypeCode() const "
  /**
   * Returns the libSBML type code for the objects contained in this {@link ListOf}
   * (i.e., {@link Rule} objects, if the list is non-empty).
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for the objects contained in this {@link ListOf}
   * instance, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers ListOfRules::getElementName() const "
  /**
   * Returns the XML element name of this object.
   * <p>
   * For {@link ListOfRules}, the XML element name is <code>\'listOfRules\'</code>.
   * <p>
   * @return the name of this element, i.e., <code>\'listOfRules\'</code>.
   */
 public";

%javamethodmodifiers ListOfRules::get(unsigned int n) "
  /**
   * Get a {@link Rule} from the {@link ListOfRules}.
   * <p>
   * @param n the index number of the {@link Rule} to get.
   * <p>
   * @return the nth {@link Rule} in this {@link ListOfRules}.
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfRules::get "
  /**
   * Get a {@link Rule} from the {@link ListOfRules}.
   * <p>
   * @param n the index number of the {@link Rule} to get.
   * <p>
   * @return the nth {@link Rule} in this {@link ListOfRules}.
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfRules::get(const std::string& sid) "
  /**
   * Get a {@link Rule} from the {@link ListOfRules}
   * based on its identifier.
   * <p>
   * @param sid a string representing the identifier 
   * of the {@link Rule} to get.
   * <p>
   * @return {@link Rule} in this {@link ListOfRules}
   * with the given id or NULL if no such
   * {@link Rule} exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfRules::get(const std::string& sid) const "
  /**
   * Get a {@link Rule} from the {@link ListOfRules}
   * based on its identifier.
   * <p>
   * @param sid a string representing the identifier 
   * of the {@link Rule} to get.
   * <p>
   * @return {@link Rule} in this {@link ListOfRules}
   * with the given id or NULL if no such
   * {@link Rule} exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfRules::remove(unsigned int n) "
  /**
   * Removes the nth item from this {@link ListOfRules} items and returns a pointer to
   * it.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * <p>
   * @param n the index of the item to remove
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfRules::remove(const std::string& sid) "
  /**
   * Removes item in this {@link ListOfRules} items with the given identifier.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * If none of the items in this list have the identifier <code>sid</code>, then 
   * <code>NULL</code> is returned.
   * <p>
   * @param sid the identifier of the item to remove
   * <p>
   * @return the item removed.  As mentioned above, the caller owns the
   * returned item.
   */
 public";

%javamethodmodifiers ListOfRules::getElementPosition() const "
  /**
   * Get the ordinal position of this element in the containing object
   * (which in this case is the {@link Model} object).
   * <p>
   * The ordering of elements in the XML form of SBML is generally fixed
   * for most components in SBML.
   * <p>
   * @return the ordinal position of the element with respect to its
   * siblings, or <code>-1</code> (default) to indicate the position is not significant.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers ListOfRules::createObject(XMLInputStream& stream) "
  /**
   * @return the SBML object corresponding to next {@link XMLToken} in the
   * XMLInputStream or NULL if the token was not recognized.
   <p>
 * @deprecated libSBML internal
 */
 public";

%typemap(javaimports) Constraint "
/** 
 * LibSBML implementation of SBML\'s Constraint construct.
 * <p>
 * The {@link Constraint} object class was introduced in SBML Level 2 Version 2 as
 * a mechanism for stating the assumptions under which a model is designed
 * to operate.  The <em>constraints</em> are statements about permissible
 * values of different quantities in a model.  Constraints are not used to
 * compute dynamical values for simulation or analysis, but rather, they
 * serve an advisory role for simulation/analysis tools.
 * <p>
 * SBML\'s {@link Constraint} object class has one required attribute, \'id\', to
 * give the parameter a unique identifier by which other parts of an SBML
 * model definition can refer to it.  A {@link Constraint} object can also have an
 * optional \'name\' attribute of type <code>string</code>.  Identifiers and names must
 * be used according to the guidelines described in the SBML specification
 * (e.g., Section 3.3 in the Level 2 Version 4 specification).  
 * <p>
 * {@link Constraint} has one required subelement, \'math\', containing a MathML
 * formula defining the condition of the constraint.  This formula must
 * return a boolean value of <code>true</code> when the model is a <em>valid</em>
 * state.  The formula can be an arbitrary expression referencing the
 * variables and other entities in an SBML model.  The evaluation of \'math\'
 * and behavior of constraints are described in more detail below.
 * <p>
 * A {@link Constraint} structure also has an optional subelement called \'message\'.
 * This can contain a message in XHTML format that may be displayed to the
 * user when the condition of the formula in the \'math\' subelement
 * evaluates to a value of <code>false</code>.  Software tools are not required to
 * display the message, but it is recommended that they do so as a matter
 * of best practice.  The XHTML content within a \'message\' subelement must
 * follow the same restrictions as for the \'notes\' element on {@link SBase}
 * described in in the SBML Level 2 specification.  For example, \'message\'
 * must not contain an XML declaration or a DOCTYPE declaration, and the
 * permitted content can only take one of the following general forms: (1)
 * a complete XHTML document beginning with the element
 * <code>&lt;html&gt;</code> and ending with <code>&lt;/xhtml&gt;</code>; 
 * (2) the body portion of a document beginning with the element
 * <code>&lt;body&gt;</code> and ending with <code>&lt;/body&gt;</code>; or
 * (3) XHTML content that is permitted within a <code>&lt;body&gt;</code> ...
 * <code>&lt;/body&gt;</code> elements.  The appendix of the SBML Level 2
 * Version 4 specification document describes one approach to reading the
 * \'message\' subelement content.
 * <p>
 * {@link Constraint} was introduced in SBML Level 2 Version 2.  It is not
 * available in earlier versions of Level 2 nor in any version of Level 1.
 * <p>
 * <h2>Semantics of Constraints</h2>
 * <p>
 * In the context of a simulation, a {@link Constraint} has effect at all times
 * <em>t &#8805; 0</em>.  Each {@link Constraint}\'s \'math\' subelement is first
 * evaluated after any {@link InitialAssignment} definitions in a model at <em>t =
 * 0</em> and can conceivably trigger at that point.  (In other words, a
 * simulation could fail a constraint immediately.)
 * <p>
 * {@link Constraint} structures <em>cannot and should not</em> be used to compute
 * the dynamical behavior of a model as part of, for example, simulation.
 * Constraints may be used as input to non-dynamical analysis, for instance
 * by expressing flux constraints for flux balance analysis.
 * <p>
 * The results of a simulation of a model containing a constraint are
 * invalid from any simulation time at and after a point when the function
 * given by the \'math\' subelement returns a value of <code>false</code>.  Invalid
 * simulation results do not make a prediction of the behavior of the
 * biochemical reaction network represented by the model.  The precise
 * behavior of simulation tools is left undefined with respect to
 * constraints.  If invalid results are detected with respect to a given
 * constraint, the \'message\' subelement may optionally be displayed to the
 * user.  The simulation tool may also halt the simulation or clearly
 * delimit in output data the simulation time point at which the simulation
 * results become invalid.
 * <p>
 * SBML does not impose restrictions on duplicate {@link Constraint} definitions or
 * the order of evaluation of {@link Constraint} objects in a model.  It is
 * possible for a model to define multiple constraints all with the same
 * mathematical expression.  Since the failure of any constraint indicates
 * that the model simulation has entered an invalid state, a system is not
 * required to attempt to detect whether other constraints in the model
 * have failed once any one constraint has failed.
 * <p>
 * <!---------------------------------------------------------------------- -->
 * <p>
 */
"

%typemap(javaimports) ListOfConstraints "
/** 
 * LibSBML implementation of SBML\'s ListOfConstraints construct.
 * <p>
 * The various ListOf___ classes in SBML are merely containers used for
 * organizing the main components of an SBML model.  All are derived from
 * the abstract class {@link SBase}, and inherit the various attributes and
 * subelements of {@link SBase}, such as \'metaid\' as and \'annotation\'.  The
 * ListOf___ classes do not add any attributes of their own.
 * <p>
 * The relationship between the lists and the rest of an SBML model is
 * illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):
 * <p>
 * <center><img src=\'listof-illustration.jpg\'></center><br>
 * 
 * <p>
 * Readers may wonder about the motivations for using the ListOf___
 * containers.  A simpler approach in XML might be to place the components
 * all directly at the top level of the model definition.  We chose instead
 * to group them within XML elements named after {@link ListOf}<em>Classname</em>,
 * in part because we believe this helps organize the components and makes
 * visual reading of models in XML easier.  More importantly, the fact that
 * the container classes are derived from {@link SBase} means that software tools
 * can add information about the lists themselves into each list
 * container\'s \'annotation\'.
 * <p>
 * @see ListOfFunctionDefinitions
 * @see ListOfUnitDefinitions
 * @see ListOfCompartmentTypes
 * @see ListOfSpeciesTypes
 * @see ListOfCompartments
 * @see ListOfSpecies
 * @see ListOfParameters
 * @see ListOfInitialAssignments
 * @see ListOfRules
 * @see ListOfConstraints
 * @see ListOfReactions
 * @see ListOfEvents
 */
"

%javamethodmodifiers Constraint::Constraint(unsigned int level, unsigned int version) "
  /**
   * Creates a new {@link Constraint} using the given SBML <code>level</code> and <code>version</code>
   * values.
   * <p>
   * @param level a long integer, the SBML Level to assign to this {@link Constraint}
   * <p>
   * @param version a long integer, the SBML Version to assign to this
   * {@link Constraint}
   * <p>
   * @note Once a {@link Constraint} has been added to an {@link SBMLDocument}, the <code>level</code>,
   * <code>version</code> for the document <em>override</em> those used
   * to create the {@link Constraint}.  Despite this, the ability to supply the values
   * at creation time is an important aid to creating valid SBML.  Knowledge of
   * the intented SBML Level and Version determine whether it is valid to
   * assign a particular value to an attribute, or whether it is valid to add
   * an object to an existing {@link SBMLDocument}.
   */
 public";

%javamethodmodifiers Constraint::Constraint(SBMLNamespaces* sbmlns) "
  /**
   * Creates a new {@link Constraint} using the given {@link SBMLNamespaces} object
   * <code>sbmlns</code>.
   * <p>
   * The {@link SBMLNamespaces} object encapsulates SBML Level/Version/namespaces
   * information.  It is used to communicate the SBML Level, Version, and
   * (in Level&nbsp;3) packages used in addition to SBML Level&nbsp; Core.
   * A common approach to using this class constructor is to create an
   * {@link SBMLNamespaces} object somewhere in a program, once, then pass it to
   * object constructors such as this one when needed.
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object.
   * <p>
   * @note Once a {@link Constraint} has been added to an {@link SBMLDocument}, the <code>level</code>,
   * <code>version</code> and <code>xmlns</code> namespaces for the document <em>override</em> those used
   * to create the {@link Constraint}.  Despite this, the ability to supply the values
   * at creation time is an important aid to creating valid SBML.  Knowledge of
   * the intented SBML Level and Version determine whether it is valid to
   * assign a particular value to an attribute, or whether it is valid to add
   * an object to an existing {@link SBMLDocument}.
   */
 public";

%javamethodmodifiers Constraint::Constraint(const Constraint& orig) "
  /**
   * Copy constructor; creates a copy of this {@link Constraint}.
   */
 public";

%javamethodmodifiers Constraint::accept(SBMLVisitor& v) const "
  /**
   * Accepts the given {@link SBMLVisitor} for this instance of {@link Constraint}.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used.
   * <p>
   * @return the result of calling <code>v.visit()</code>, which indicates
   * whether the Visitor would like to visit the next {@link Constraint} in the
   * list of constraints within which this {@link Constraint} is embedded (i.e., in
   * the {@link ListOfConstraints} located in the enclosing {@link Model} instance).
   */
 public";

%javamethodmodifiers Constraint::clone() const "
  /**
   * Creates and returns a deep copy of this {@link Constraint}.
   * <p>
   * @return a (deep) copy of this {@link Constraint}.
   */
 public";

%javamethodmodifiers Constraint::getMessage() const "
  /**
   * Get the message, if any, associated with this {@link Constraint}
   * <p>
   * @return the message for this {@link Constraint}, as an {@link XMLNode}.
   */
 public";

%javamethodmodifiers Constraint::getMessageString() const "
  /**
   * Get the message string, if any, associated with this {@link Constraint}
   * <p>
   * @return the message for this {@link Constraint}, as a string.
   */
 public";

%javamethodmodifiers Constraint::getMath() const "
  /**
   * Get the mathematical expression of this {@link Constraint}
   * <p>
   * @return the math for this {@link Constraint}, as an {@link ASTNode}.
   */
 public";

%javamethodmodifiers Constraint::isSetMessage() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether a
   * message has been defined for this {@link Constraint}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the message of this {@link Constraint} has been set,
   * <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers Constraint::isSetMath() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether a
   * mathematical formula has been defined for this {@link Constraint}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the \'math\' subelement for this {@link Constraint} has been
   * set, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers Constraint::setMessage(const XMLNode* xhtml) "
  /**
   * Sets the message of this {@link Constraint}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * The {@link XMLNode} tree passed in <code>xhtml</code> is copied.
   * <p>
   * @param xhtml an XML tree containing XHTML content.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_OBJECT
   */
 public";

%javamethodmodifiers Constraint::setMath(const ASTNode* math) "
  /**
   * Sets the mathematical expression of this {@link Constraint} to a copy of the
   * AST given as <code>math</code>.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param math an {@link ASTNode} expression to be assigned as the \'math\'
   * subelement of this {@link Constraint}
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_OBJECT
   */
 public";

%javamethodmodifiers Constraint::unsetMessage "
  /**
   * Unsets the \'message\' subelement of this {@link Constraint}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_OPERATION_FAILED
   */
 public";

%javamethodmodifiers Constraint::getTypeCode() const "
  /**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for this object, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers Constraint::getElementName() const "
  /**
   * Returns the XML element name of this object, which for {@link Constraint}, is
   * always <code>\'constraint\'</code>.
   * <p>
   * @return the name of this element, i.e., <code>\'constraint\'</code>.
   */
 public";

%javamethodmodifiers Constraint::writeElements(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write out their contained
   * SBML objects as XML elements.  Be sure to call your parents
   * implementation of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Constraint::hasRequiredElements() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required elements for this {@link Constraint} object
   * have been set.
   * <p>
   * @note The required elements for a {@link Constraint} object are:
   * math
   * <p>
   * @return a boolean value indicating whether all the required
   * elements for this object have been defined.
   */
 public";

%javamethodmodifiers Constraint::Constraint "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required elements for this {@link Constraint} object
   * have been set.
   * <p>
   * @note The required elements for a {@link Constraint} object are:
   * math
   * <p>
   * @return a boolean value indicating whether all the required
   * elements for this object have been defined.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Constraint::readOtherXML(XMLInputStream& stream) "
  /**
   * Subclasses should override this method to read (and store) XHTML,
   * MathML, etc. directly from the XMLInputStream.
   * <p>
   * @return true if the subclass read from the stream, false otherwise.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Constraint::readAttributes(const XMLAttributes& attributes) "
  /**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Constraint::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Constraint::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Constraint::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Constraint::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Constraint::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Constraint::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Constraint::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Constraint::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Constraint::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Constraint::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Constraint::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Constraint::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Constraint::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Constraint::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Constraint::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Constraint::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Constraint::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers ListOfConstraints::clone() const "
  /**
   * Creates and returns a deep copy of this {@link ListOfConstraints} instance.
   * <p>
   * @return a (deep) copy of this {@link ListOfConstraints}.
   */
 public";

%javamethodmodifiers ListOfConstraints::getTypeCode() const "
  /**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for this object, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers ListOfConstraints::getItemTypeCode() const "
  /**
   * Returns the libSBML type code for the objects contained in this {@link ListOf}
   * (i.e., {@link Constraint} objects, if the list is non-empty).
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for the objects contained in this {@link ListOf}
   * instance, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers ListOfConstraints::getElementName() const "
  /**
   * Returns the XML element name of this object.
   * <p>
   * For {@link ListOfConstraints}, the XML element name is <code>\'listOfConstraints\'</code>.
   * <p>
   * @return the name of this element, i.e., <code>\'listOfConstraints\'</code>.
   */
 public";

%javamethodmodifiers ListOfConstraints::get(unsigned int n) "
  /**
   * Get a {@link Constraint} from the {@link ListOfConstraints}.
   * <p>
   * @param n the index number of the {@link Constraint} to get.
   * <p>
   * @return the nth {@link Constraint} in this {@link ListOfConstraints}.
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfConstraints::get "
  /**
   * Get a {@link Constraint} from the {@link ListOfConstraints}.
   * <p>
   * @param n the index number of the {@link Constraint} to get.
   * <p>
   * @return the nth {@link Constraint} in this {@link ListOfConstraints}.
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfConstraints::remove(unsigned int n) "
  /**
   * Removes the nth item from this {@link ListOfConstraints} items and returns a
   * pointer to it.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * <p>
   * @param n the index of the item to remove
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfConstraints::getElementPosition() const "
  /**
   * Get the ordinal position of this element in the containing object
   * (which in this case is the {@link Model} object).
   * <p>
   * The ordering of elements in the XML form of SBML is generally fixed
   * for most components in SBML.  So, for example, the {@link ListOfConstraints}
   * in a model is (in SBML Level 2 Version 4) the tenth ListOf___.
   * (However, it differs for different Levels and Versions of SBML.)
   * <p>
   * @return the ordinal position of the element with respect to its
   * siblings, or <code>-1</code> (default) to indicate the position is not significant.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers ListOfConstraints::createObject(XMLInputStream& stream) "
  /**
   * @return the SBML object corresponding to next {@link XMLToken} in the
   * XMLInputStream or NULL if the token was not recognized.
   <p>
 * @deprecated libSBML internal
 */
 public";

%typemap(javaimports) Reaction "
/** 
 * LibSBML implementation of SBML\'s Reaction construct.
 * <p>
 * A <em>reaction</em> represents any transformation, transport or binding
 * process, typically a chemical reaction, that can change the quantity of
 * one or more species.  In SBML, a reaction is defined primarily in terms
 * of the participating reactants and products (and their corresponding
 * stoichiometries), along with optional modifier species, an optional rate
 * at which the reaction takes place, and optional parameters.  These
 * various parts of a reaction are recorded in the {@link Reaction} object and its
 * supporting object classes: {@link KineticLaw}, {@link SpeciesReference},
 * {@link ModifierSpeciesReference}, {@link ListOfSpeciesReferences},
 * ListOfModifierSpeciesReferences, and {@link StoichiometryMath}.  It also uses
 * {@link Parameter} and {@link ListOfParameters}.
 * <p>
 * As with other major object in SBML, {@link Reaction} has a mandatory attribute,
 * \'id\', used to give the compartment type an identifier.  The identifier
 * must be a text string conforming to the identifer syntax permitted in
 * SBML.  The reaction \'id\' identifier can be used in mathematical
 * formulas elsewhere in an SBML model to represent the rate of that
 * reaction; this usage is explained below.  {@link Reaction} also has an optional
 * \'name\' attribute, of type <code>string</code>.  The \'id\' and \'name\' must be used
 * according to the guidelines described in the SBML specification (e.g.,
 * Section 3.3 in the Level 2 Version 4 specification).
 * <p>
 * The species participating as reactants, products, and/or modifiers in a
 * reaction are declared using lists of {@link SpeciesReference} and/or
 * {@link ModifierSpeciesReference} instances stored in subelements
 * \'listOfReactants\', \'listOfProducts\' and \'listOfModifiers\'.  Certain
 * restrictions are placed on the appearance of species in reaction
 * definitions:
 * <ul>
 * <li> The ability of a species to appear as a reactant or product of any
 * reaction in a model is governed by certain flags in that species\'
 * definition; see the definition of {@link Species} for more information.
 * <p>
 * <li> Any species appearing in the mathematical formula of the subelement
 * \'kineticLaw\' (described below) of a {@link Reaction} must be declared in at
 * least one of that {@link Reaction}\'s lists of reactants, products, and/or
 * modifiers.  Put another way, it is an error for a reaction\'s kinetic law
 * formula to refer to species that have not been declared for that
 * reaction.
 * <p>
 * <li> A reaction definition can contain an empty list of reactants
 * <em>or</em> an empty list of products, but it must have at least one
 * reactant or product; in other words, a reaction without any reactant or
 * product species is not permitted.  (This restriction does not apply to
 * modifier species, which remain optional in all cases.)
 * </ul>
 * <p>
 * A reaction can contain up to one {@link KineticLaw} object in a subelement named
 * \'kineticLaw\'.  It defines the speed at which the process defined by the
 * reaction takes place.  The description of {@link KineticLaw} provides more
 * details about its use.  Note that although the inclusion of a {@link KineticLaw}
 * object in an instance of a {@link Reaction} component is optional, there is no
 * useful default that can be substituted in place of a missing rate
 * expression in a reaction.  Moreover, a reaction\'s rate cannot be defined
 * in any other way in SBML&mdash;{@link InitialAssignment}, {@link AssignmentRule},
 * {@link RateRule}, {@link AlgebraicRule}, {@link Event}, and other constructs in SBML cannot be
 * used to set the reaction rate separately.  Nevertheless, for some
 * modeling applications, reactions without any defined rate can be
 * perfectly acceptable.
 * <p>
 * {@link Reaction} also has an optional boolean attribute named \'reversible\' for
 * indicating whether the reaction is reversible.  The default is <code>true</code>.
 * To say that a reaction is <em>reversible</em> is to say it can proceed in
 * either the forward or the reverse direction.  Although the reversibility
 * of a reaction can sometimes be deduced by inspecting its rate
 * expression, this is not always the case, especially for complicated
 * expressions.  Moreover, the need in SBML to allow rate expressions
 * (i.e., {@link KineticLaw}) to be optional leads to the need for a separate flag
 * indicating reversibility.  Note that labeling a reaction as irreversible
 * is an assertion that the reaction always proceeds in the given forward
 * direction.  (Why else would it be flagged as irreversible?)  This
 * implies the rate expression in the {@link KineticLaw} always has a non-negative
 * value during simulations.  Software tools could provide a means of
 * optionally testing that this condition holds.  The presence of
 * reversibility information in two places (i.e., the rate expression and
 * the \'reversible\' attribute on {@link Reaction}) leaves open the possibility that
 * a model could contain contradictory information, but the creation of
 * such a model would be an error on the part of the software generating
 * it.
 * <p>
 * Finally, {@link Reaction} has another optional boolean attribute called \'fast\'.
 * It is used to indicate that a reaction occurs on a vastly faster time
 * scale than others in a system.  Readers are directed to the SBML Level
 * 2 Version 4 specification, which provides more detail about the
 * conditions under which a reaction can be considered to be fast in this
 * sense.  The attribute\'s default value is <code>false</code>.  SBML Level 1 and
 * Level 2 Version 1 incorrectly claimed that software tools could ignore
 * this attribute if they did not implement support for the corresponding
 * concept; however, further research in SBML has revealed that this is
 * not true, and \'fast\' <em>cannot be ignored</em> if it is set to <code>true</code>.
 * SBML Level 2 Versions 2, 3 and 4 therefore stipulate that if a model has
 * any reactions with \'fast\' set to <code>true</code>, a software tool must be able
 * to respect the attribute or else indicate to the user that it does not
 * have the capacity to do so.  Analysis software cannot ignore the value
 * of the \'fast\' attribute because doing so may lead to different results
 * as compared to a software system that <em>does</em> make use of \'fast\'.
 * <p>
 * Readers are urged to read the SBML specification for more details about
 * the proper use of {@link Reaction}.
 * <p>
 * <p>
 */
"

%typemap(javaimports) ListOfReactions "
/** 
 * LibSBML implementation of SBML\'s ListOfReactions construct.
 * <p>
 * The various ListOf___ classes in SBML are merely containers used for
 * organizing the main components of an SBML model.  All are derived from
 * the abstract class {@link SBase}, and inherit the various attributes and
 * subelements of {@link SBase}, such as \'metaid\' as and \'annotation\'.  The
 * ListOf___ classes do not add any attributes of their own.
 * <p>
 * The relationship between the lists and the rest of an SBML model is
 * illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):
 * <p>
 * <center><img src=\'listof-illustration.jpg\'></center><br>
 * 
 * <p>
 * Readers may wonder about the motivations for using the ListOf___
 * containers.  A simpler approach in XML might be to place the components
 * all directly at the top level of the model definition.  We chose instead
 * to group them within XML elements named after {@link ListOf}<em>Classname</em>,
 * in part because we believe this helps organize the components and makes
 * visual reading of models in XML easier.  More importantly, the fact that
 * the container classes are derived from {@link SBase} means that software tools
 * can add information about the lists themselves into each list
 * container\'s \'annotation\'.
 * <p>
 * @see ListOfFunctionDefinitions
 * @see ListOfUnitDefinitions
 * @see ListOfCompartmentTypes
 * @see ListOfSpeciesTypes
 * @see ListOfCompartments
 * @see ListOfSpecies
 * @see ListOfParameters
 * @see ListOfInitialAssignments
 * @see ListOfRules
 * @see ListOfConstraints
 * @see ListOfReactions
 * @see ListOfEvents
 */
"

%javamethodmodifiers Reaction::Reaction(unsigned int level, unsigned int version) "
  /**
   * Creates a new {@link Reaction} using the given SBML <code>level</code> and <code>version</code>
   * values.
   * <p>
   * @param level a long integer, the SBML Level to assign to this {@link Reaction}
   * <p>
   * @param version a long integer, the SBML Version to assign to this
   * {@link Reaction}
   * <p>
   * @note Once a {@link Reaction} has been added to an {@link SBMLDocument}, the <code>level</code>,
   * <code>version</code> for the document <em>override</em> those used
   * to create the {@link Reaction}.  Despite this, the ability to supply the values
   * at creation time is an important aid to creating valid SBML.  Knowledge of
   * the intented SBML Level and Version determine whether it is valid to
   * assign a particular value to an attribute, or whether it is valid to add
   * an object to an existing {@link SBMLDocument}.
   */
 public";

%javamethodmodifiers Reaction::Reaction(SBMLNamespaces* sbmlns) "
  /**
   * Creates a new {@link Reaction} using the given {@link SBMLNamespaces} object
   * <code>sbmlns</code>.
   * <p>
   * The {@link SBMLNamespaces} object encapsulates SBML Level/Version/namespaces
   * information.  It is used to communicate the SBML Level, Version, and
   * (in Level&nbsp;3) packages used in addition to SBML Level&nbsp; Core.
   * A common approach to using this class constructor is to create an
   * {@link SBMLNamespaces} object somewhere in a program, once, then pass it to
   * object constructors such as this one when needed.
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object.
   * <p>
   * @note Once a {@link Reaction} has been added to an {@link SBMLDocument}, the <code>level</code>,
   * <code>version</code> and <code>xmlns</code> namespaces for the document <em>override</em> those used
   * to create the {@link Reaction}.  Despite this, the ability to supply the values
   * at creation time is an important aid to creating valid SBML.  Knowledge of
   * the intented SBML Level and Version determine whether it is valid to
   * assign a particular value to an attribute, or whether it is valid to add
   * an object to an existing {@link SBMLDocument}.
   */
 public";

%javamethodmodifiers Reaction::Reaction(const Reaction& orig) "
  /**
   * Copy constructor; creates a copy of this {@link Reaction}.
   */
 public";

%javamethodmodifiers Reaction::accept(SBMLVisitor& v) const "
  /**
   * Accepts the given {@link SBMLVisitor} for this instance of {@link Reaction}.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used.
   * <p>
   * @return the result of calling <code>v.visit()</code>.
   */
 public";

%javamethodmodifiers Reaction::clone() const "
  /**
   * Creates and returns a deep copy of this {@link Reaction}.
   * <p>
   * @return a (deep) copy of this {@link Reaction}.
   */
 public";

%javamethodmodifiers Reaction::initDefaults "
  /**
   * Initializes certain attributes of this {@link Reaction} object to default
   * values.
   * <ul>
   * <li> Sets the \'reversible\' attribute to <code>true</code>
   * <li> (SBML Level 1 only) Sets the \'fast\' attribute to <code>false</code>
   * </ul>
   * <p>
   * @warning The \'fast\' attribute must be used with care.  SBML
   * definitions before SBML Level 2 Version 2 incorrectly indicated that
   * software tools could ignore this attribute if they did not implement
   * support for the corresponding concept; however, further research in
   * SBML has revealed that this is not true, and \'fast\' <em>cannot be
   * ignored</em> if it is set to <code>true</code>.  SBML Level 2 Versions 2, 3 and 4
   * therefore stipulate that if a model has any reactions with \'fast\' set
   * to <code>true</code>, a software tool must be able to respect the attribute or
   * else indicate to the user that it does not have the capacity to do so.
   * Readers are directed to the SBML Level 2 Version 4 specification,
   * which provides more detail about the conditions under which a reaction
   * can be considered to be fast in this sense.
   */
 public";

%javamethodmodifiers Reaction::getId() const "
  /**
   * Returns the value of the \'id\' attribute of this {@link Reaction}.
   * <p>
   * @return the id of this {@link Reaction}.
   */
 public";

%javamethodmodifiers Reaction::getName() const "
  /**
   * Returns the value of the \'name\' attribute of this {@link Reaction}.
   * <p>
   * @return the name of this {@link Reaction}.
   */
 public";

%javamethodmodifiers Reaction::getKineticLaw() const "
  /**
   * Returns the {@link KineticLaw} object contained in this {@link Reaction}.
   * <p>
   * @return the {@link KineticLaw} instance.
   */
 public";

%javamethodmodifiers Reaction::getKineticLaw "
  /**
   * Returns the {@link KineticLaw} object contained in this {@link Reaction}.
   * <p>
   * @return the {@link KineticLaw} instance.
   */
 public";

%javamethodmodifiers Reaction::getReversible() const "
  /**
   * Returns the value of the \'reversible\' attribute on the {@link Reaction} as a
   * boolean value.
   * <p>
   * @return the reversibility status of this {@link Reaction}.
   */
 public";

%javamethodmodifiers Reaction::getFast() const "
  /**
   * Returns the value of the \'fast\' attribute of this {@link Reaction}.
   * <p>
   * @return the \'fast\' status of this {@link Reaction}.
   * <p>
   * @warning SBML definitions before SBML Level 2 Version 2 incorrectly
   * indicated that software tools could ignore this attribute if they did
   * not implement support for the corresponding concept; however, further
   * research in SBML has revealed that this is not true, and \'fast\'
   * <em>cannot be ignored</em> if it is set to <code>true</code>.  SBML Level 2
   * Versions 2, 3 and 4 therefore stipulate that if a model has any reactions
   * with \'fast\' set to <code>true</code>, a software tool must be able to respect
   * the attribute or else indicate to the user that it does not have the
   * capacity to do so.  Readers are directed to the SBML Level 2 Version
   * 4 specification, which provides more detail about the conditions under
   * which a reaction can be considered to be fast in this sense.
   */
 public";

%javamethodmodifiers Reaction::isSetId() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link Reaction}\'s \'id\' attribute has been set.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the \'id\' attribute of this {@link Reaction} has been
   * set, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers Reaction::isSetName() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link Reaction}\'s \'name\' attribute has been set.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the \'name\' attribute of this {@link Reaction} has been
   * set, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers Reaction::isSetKineticLaw() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link Reaction} contains a kinetic law object.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if a {@link KineticLaw} is present in this {@link Reaction},, <code>false</code>
   * otherwise.
   */
 public";

%javamethodmodifiers Reaction::isSetFast() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on the value of
   * the \'fast\' attribute on this {@link Reaction}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the \'fast\' attribute is true, <code>false</code> otherwise.
   * <p>
   * @warning SBML definitions before SBML Level 2 Version 2 incorrectly
   * indicated that software tools could ignore this attribute if they did
   * not implement support for the corresponding concept; however, further
   * research in SBML has revealed that this is not true, and \'fast\'
   * <em>cannot be ignored</em> if it is set to <code>true</code>.  SBML Level 2
   * Versions 2, 3 and 4 therefore stipulate that if a model has any reactions
   * with \'fast\' set to <code>true</code>, a software tool must be able to respect
   * the attribute or else indicate to the user that it does not have the
   * capacity to do so.  Readers are directed to the SBML Level 2 Version
   * 4 specification, which provides more detail about the conditions under
   * which a reaction can be considered to be fast in this sense.  Note
   * also that in SBML Level 1, \'fast\' is defined as optional with a
   * default of <code>false</code>, which means it is effectively always set.
   */
 public";

%javamethodmodifiers Reaction::setId(const std::string& sid) "
  /**
   * Sets the value of the \'id\' attribute of this {@link Reaction}.
   * <p>
   * The string <code>sid</code> is copied.  Note that SBML has strict requirements
   * for the syntax of identifiers.  The following is summary of the
   * definition of the SBML identifier type <code>SId</code> (here expressed in an
   * extended form of BNF notation):
   * <div class=\'fragment\'><pre>
   *   letter .= \'a\'..\'z\',\'A\'..\'Z\'
   *   digit  .= \'0\'..\'9\'
   *   idChar .= letter | digit | \'_\'
   *   SId    .= ( letter | \'_\' ) idChar</pre></div>
   * The equality of SBML identifiers is determined by an exact character
   * sequence match; i.e., comparisons must be performed in a
   * case-sensitive manner.  In addition, there are a few conditions for
   * the uniqueness of identifiers in an SBML model.  Please consult the
   * SBML specifications for the exact formulations.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param sid the string to use as the identifier of this {@link Reaction}
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   */
 public";

%javamethodmodifiers Reaction::setName(const std::string& name) "
  /**
   * Sets the value of the \'name\' attribute of this {@link Reaction}.
   * <p>
   * The string in <code>name</code> is copied.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param name the new name for the {@link Reaction}
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   */
 public";

%javamethodmodifiers Reaction::setKineticLaw(const KineticLaw* kl) "
  /**
   * Sets the \'kineticLaw\' subelement of this {@link Reaction} to a copy of the
   * given {@link KineticLaw} object.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param kl the {@link KineticLaw} object to use.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_LEVEL_MISMATCH
   * <li> LIBSBML_VERSION_MISMATCH
  */
 public";

%javamethodmodifiers Reaction::setReversible(bool value) "
  /**
   * Sets the value of the \'reversible\' attribute of this {@link Reaction}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param value the value of the \'reversible\' attribute.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   */
 public";

%javamethodmodifiers Reaction::setFast(bool value) "
  /**
   * Sets the value of the \'fast\' attribute of this {@link Reaction}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param value the value of the \'fast\' attribute.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <p>
   * @warning SBML definitions before SBML Level 2 Version 2 incorrectly
   * indicated that software tools could ignore this attribute if they did
   * not implement support for the corresponding concept; however, further
   * research in SBML has revealed that this is not true, and \'fast\'
   * <em>cannot be ignored</em> if it is set to <code>true</code>.  SBML Level 2
   * Versions 2, 3 and 4 therefore stipulate that if a model has any reactions
   * with \'fast\' set to <code>true</code>, a software tool must be able to respect
   * the attribute or else indicate to the user that it does not have the
   * capacity to do so.  Readers are directed to the SBML Level 2 Version
   * 4 specification, which provides more detail about the conditions under
   * which a reaction can be considered to be fast in this sense.
   */
 public";

%javamethodmodifiers Reaction::unsetName "
  /**
   * Unsets the value of the \'name\' attribute of this {@link Reaction}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_OPERATION_FAILED
   */
 public";

%javamethodmodifiers Reaction::unsetKineticLaw "
  /**
   * Unsets the \'kineticLaw\' subelement of this {@link Reaction}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_OPERATION_FAILED
   */
 public";

%javamethodmodifiers Reaction::unsetFast "
  /**
   * Unsets the value of the \'fast\' attribute of this {@link Reaction}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_OPERATION_FAILED
   * <p>
   * @warning In SBML Level 1, \'fast\' is optional with a default of 
   * <code>false</code>, which means it is effectively always set (and reset to <code>false</code>
   * if this method is called).  Further, SBML definitions before SBML
   * Level 2 Version 2 incorrectly indicated that software tools could
   * ignore this attribute if they did not implement support for the
   * corresponding concept; however, further research in SBML has revealed
   * that this is not true, and \'fast\' <em>cannot be ignored</em> if it is
   * set to <code>true</code>.  SBML Level 2 Versions 2, 3 and 4 therefore stipulate
   * that if a model has any reactions with \'fast\' set to <code>true</code>, a
   * software tool must be able to respect the attribute or else indicate
   * to the user that it does not have the capacity to do so.  Readers are
   * directed to the SBML Level 2 Version 4 specification, which provides
   * more detail about the conditions under which a reaction can be
   * considered to be fast in this sense.
   */
 public";

%javamethodmodifiers Reaction::addReactant(const SpeciesReference* sr) "
  /**
   * Adds a given {@link SpeciesReference} object as a reactant in this {@link Reaction}.
   * <p>
   * The {@link SpeciesReference} instance in <code>sr</code> is copied.
   * <p>
   * @param sr a {@link SpeciesReference} object referring to a {@link Species} in the
   * enclosing {@link Model}
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_LEVEL_MISMATCH
   * <li> LIBSBML_VERSION_MISMATCH
   * <li> LIBSBML_OPERATION_FAILED
   * <p>
   * @note This method should be used with some caution.  The fact that
   * this method <em>copies</em> the object passed to it means that the caller
   * will be left holding a physically different object instance than the
   * one contained in this {@link Reaction}.  Changes made to the original object
   * instance (such as resetting attribute values) will <em>not affect the
   * instance in the {@link Reaction}</em>.  In addition, the caller should make
   * sure to free the original object if it is no longer being used, or
   * else a memory leak will result.  Please see Reaction.createReactant()
   * for a method that does not lead to these issues.
   * <p>
   * @see #createReactant()
   */
 public";

%javamethodmodifiers Reaction::addProduct(const SpeciesReference* sr) "
  /**
   * Adds a given {@link SpeciesReference} object as a product in this {@link Reaction}.
   * <p>
   * The {@link SpeciesReference} instance in <code>sr</code> is copied.
   * <p>
   * @param sr a {@link SpeciesReference} object referring to a {@link Species} in the
   * enclosing {@link Model}
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_LEVEL_MISMATCH
   * <li> LIBSBML_VERSION_MISMATCH
   * <li> LIBSBML_OPERATION_FAILED
   * <p>
   * @note This method should be used with some caution.  The fact that
   * this method <em>copies</em> the object passed to it means that the caller
   * will be left holding a physically different object instance than the
   * one contained in this {@link Reaction}.  Changes made to the original object
   * instance (such as resetting attribute values) will <em>not affect the
   * instance in the {@link Reaction}</em>.  In addition, the caller should make
   * sure to free the original object if it is no longer being used, or
   * else a memory leak will result.  Please see Reaction.createProduct()
   * for a method that does not lead to these issues.
   * <p>
   * @see #createProduct()
   */
 public";

%javamethodmodifiers Reaction::addModifier(const ModifierSpeciesReference* msr) "
  /**
   * Adds a given {@link ModifierSpeciesReference} object as a product in this
   * {@link Reaction}.
   * <p>
   * The {@link ModifierSpeciesReference} instance in <code>msr</code> is copied.
   * <p>
   * @param msr a {@link ModifierSpeciesReference} object referring to a {@link Species} in
   * the enclosing {@link Model}
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_UNEXPECTED_ATTRIBUTE
   * <li> LIBSBML_LEVEL_MISMATCH
   * <li> LIBSBML_VERSION_MISMATCH
   * <li> LIBSBML_OPERATION_FAILED
   * <p>
   * @note This method should be used with some caution.  The fact that
   * this method <em>copies</em> the object passed to it means that the caller
   * will be left holding a physically different object instance than the
   * one contained in this {@link Reaction}.  Changes made to the original object
   * instance (such as resetting attribute values) will <em>not affect the
   * instance in the {@link Reaction}</em>.  In addition, the caller should make
   * sure to free the original object if it is no longer being used, or
   * else a memory leak will result.  Please see Reaction.createModifier()
   * for a method that does not lead to these issues.
   * <p>
   * @see #createModifier()
   */
 public";

%javamethodmodifiers Reaction::createReactant "
  /**
   * Creates a new {@link SpeciesReference}, adds it to this {@link Reaction}\'s list of
   * reactants, and returns it.
   * <p>
   * @return a new {@link SpeciesReference} object.
   */
 public";

%javamethodmodifiers Reaction::createProduct "
  /**
   * Creates a new {@link SpeciesReference}, adds it to this {@link Reaction}\'s list of
   * products, and returns it.
   * <p>
   * @return a new {@link SpeciesReference} object.
   */
 public";

%javamethodmodifiers Reaction::createModifier "
  /**
   * Creates a new {@link ModifierSpeciesReference}, adds it to this {@link Reaction}\'s
   * list of modifiers and returns it.
   * <p>
   * @return a new {@link ModifierSpeciesReference} object.
   */
 public";

%javamethodmodifiers Reaction::createKineticLaw "
  /**
   * Creates a new {@link KineticLaw} object, installs it as this {@link Reaction}\'s
   * \'kineticLaw\' subelement, and returns it.
   * <p>
   * If this {@link Reaction} had a previous {@link KineticLaw}, it will be destroyed.
   * <p>
   * @return the new {@link KineticLaw} object
   */
 public";

%javamethodmodifiers Reaction::getListOfReactants() const "
  /**
   * Returns the list of reactants in this {@link Reaction} object.
   * <p>
   * @return the {@link ListOfSpeciesReferences} containing the references to the
   * species acting as reactants in this reaction
   */
 public";

%javamethodmodifiers Reaction::getListOfReactants "
  /**
   * Returns the list of reactants in this {@link Reaction} object.
   * <p>
   * @return the {@link ListOfSpeciesReferences} containing the references to the
   * species acting as reactants in this reaction
   */
 public";

%javamethodmodifiers Reaction::getListOfProducts() const "
  /**
   * Returns the list of products in this {@link Reaction} object.
   * <p>
   * @return the {@link ListOfSpeciesReferences} containing the references to the
   * species acting as products in this reaction
   */
 public";

%javamethodmodifiers Reaction::getListOfProducts "
  /**
   * Returns the list of products in this {@link Reaction} object.
   * <p>
   * @return the {@link ListOfSpeciesReferences} containing the references to the
   * species acting as products in this reaction
   */
 public";

%javamethodmodifiers Reaction::getListOfModifiers() const "
  /**
   * Returns the list of modifiers in this {@link Reaction} object.
   * <p>
   * @return the {@link ListOfSpeciesReferences} containing the references to the
   * species acting as modifiers in this reaction
   */
 public";

%javamethodmodifiers Reaction::getListOfModifiers "
  /**
   * Returns the list of modifiers in this {@link Reaction} object.
   * <p>
   * @return the {@link ListOfSpeciesReferences} containing the references to the
   * species acting as modifiers in this reaction
   */
 public";

%javamethodmodifiers Reaction::getReactant "
  /**
   * Returns the nth reactant species (as a {@link SpeciesReference} object) in 
   * the list of reactants in this {@link Reaction}.
   * <p>
   * Callers should first call getNumReactants() to find out how many
   * reactants there are, to avoid using an invalid index number.
   * <p>
   * @param n the index of the reactant sought.
   * <p>
   * @return the nth reactant (as a {@link SpeciesReference} object) of this
   * {@link Reaction}.
   */
 public";

%javamethodmodifiers Reaction::getReactant(unsigned int n) "
  /**
   * Returns the nth reactant species (as a {@link SpeciesReference} object) 
   * in the list of reactants in this {@link Reaction}.
   * <p>
   * Callers should first call getNumReactants() to find out how many
   * reactants there are, to avoid using an invalid index number.
   * <p>
   * @param n the index of the reactant sought.
   * <p>
   * @return the nth reactant (as a {@link SpeciesReference} object) of this
   * {@link Reaction}.
   */
 public";

%javamethodmodifiers Reaction::getReactant(const std::string& species) const "
  /**
   * Returns the reactant species (as a {@link SpeciesReference} object) having 
   * a specific identifier in this {@link Reaction}.
   * <p>
   * @param species the identifier of the reactant {@link Species} (\'species\' 
   * attribute of the reactant {@link SpeciesReference} object)
   * <p>
   * @return a {@link SpeciesReference} object, or NULL if no species with the
   * given identifier <code>species</code> appears as a reactant in this {@link Reaction}.
   */
 public";

%javamethodmodifiers Reaction::getReactant(const std::string& species) "
  /**
   * Returns the reactant species (as a {@link SpeciesReference} object) having 
   * a specific identifier in this {@link Reaction}.
   * <p>
   * @param species the identifier of the reactant {@link Species} (\'species\' 
   * attribute of the reactant {@link SpeciesReference} object)
   * <p>
   * @return a {@link SpeciesReference} object, or NULL if no species with the
   * given identifier <code>species</code> appears as a reactant in this {@link Reaction}.
   */
 public";

%javamethodmodifiers Reaction::getProduct "
  /**
   * Returns the nth product species (as a {@link SpeciesReference} object) in 
   * the list of products in this {@link Reaction}.
   * <p>
   * Callers should first call getNumProducts() to find out how many
   * products there are, to avoid using an invalid index number.
   * <p>
   * @param n the index of the product sought.
   * <p>
   * @return the nth product (as a {@link SpeciesReference} object) of this
   * {@link Reaction}.
   */
 public";

%javamethodmodifiers Reaction::getProduct(unsigned int n) "
  /**
   * Returns the nth product species (as a {@link SpeciesReference} object) 
   * in the list of products in this {@link Reaction}.
   * <p>
   * Callers should first call getNumProducts() to find out how many
   * products there are, to avoid using an invalid index number.
   * <p>
   * @param n the index of the product sought.
   * <p>
   * @return the nth product (as a {@link SpeciesReference} object) of this
   * {@link Reaction}.
   */
 public";

%javamethodmodifiers Reaction::getProduct(const std::string& species) const "
  /**
   * Returns the product species (as a {@link SpeciesReference} object) having 
   * a specific identifier in this {@link Reaction}.
   * <p>
   * @param species the identifier of the product {@link Species} (\'species\'
   * attribute of the product {@link SpeciesReference} object)
   * <p>
   * @return a {@link SpeciesReference} object, or NULL if no species with the
   * given identifier <code>species</code> appears as a product in this {@link Reaction}.
   */
 public";

%javamethodmodifiers Reaction::getProduct(const std::string& species) "
  /**
   * Returns the product species (as a {@link SpeciesReference} object) having 
   * a specific identifier in this {@link Reaction}.
   * <p>
   * @param species the identifier of the product {@link Species} (\'species\'
   * attribute of the product {@link SpeciesReference} object)
   * <p>
   * @return a {@link SpeciesReference} object, or NULL if no species with the
   * given identifier <code>species</code> appears as a product in this {@link Reaction}.
   */
 public";

%javamethodmodifiers Reaction::getModifier "
  /**
   * Returns the nth modifier species (as a {@link ModifierSpeciesReference} object) 
   * in the list of modifiers of this {@link Reaction}.
   * <p>
   * Callers should first call getNumModifiers() to find out how many
   * modifiers there are, to avoid using an invalid index number.
   * <p>
   * @param n the index of the modifier species sought
   * <p>
   * @return the nth modifier (as a {@link ModifierSpeciesReference} object) of
   * this {@link Reaction}.
   */
 public";

%javamethodmodifiers Reaction::getModifier(unsigned int n) "
  /**
   * Returns the nth modifier species (as a {@link ModifierSpeciesReference} object) 
   * in the list of modifiers of this {@link Reaction}.
   * <p>
   * Callers should first call getNumModifiers() to find out how many
   * modifiers there are, to avoid using an invalid index number.
   * <p>
   * @param n the index of the modifier species sought
   * <p>
   * @return the nth modifier (as a {@link ModifierSpeciesReference} object) of
   * this {@link Reaction}.
   */
 public";

%javamethodmodifiers Reaction::getModifier(const std::string& species) const "
  /**
   * Returns the modifier species (as a {@link ModifierSpeciesReference} object) 
   * having a specific identifier in this {@link Reaction}.
   * <p>
   * @param species the identifier of the modifier {@link Species} (\'species\' 
   * attribute of the {@link ModifierSpeciesReference} object)
   * <p>
   * @return a {@link ModifierSpeciesReference} object, or NULL if no species with
   * the given identifier <code>species</code> appears as a modifier in this
   * {@link Reaction}.
   */
 public";

%javamethodmodifiers Reaction::getModifier(const std::string& species) "
  /**
   * Returns the modifier species (as a {@link ModifierSpeciesReference} object) 
   * having a specific identifier in this {@link Reaction}.
   * <p>
   * @param species the identifier of the modifier {@link Species} (\'species\' 
   * attribute of the {@link ModifierSpeciesReference} object)
   * <p>
   * @return a {@link ModifierSpeciesReference} object, or NULL if no species with
   * the given identifier <code>species</code> appears as a modifier in this
   * {@link Reaction}.
   */
 public";

%javamethodmodifiers Reaction::getNumReactants() const "
  /**
   * Returns the number of reactant species in this {@link Reaction}.
   * <p>
   * @return the number of reactants in this {@link Reaction}.
   */
 public";

%javamethodmodifiers Reaction::getNumProducts() const "
  /**
   * Returns the number of product species in this {@link Reaction}.
   * <p>
   * @return the number of products in this {@link Reaction}.
   */
 public";

%javamethodmodifiers Reaction::getNumModifiers() const "
  /**
   * Returns the number of modifier species in this {@link Reaction}.
   * <p>
   * @return the number of modifiers in this {@link Reaction}.
   */
 public";

%javamethodmodifiers Reaction::removeReactant(unsigned int n) "
  /**
   * Removes the nth reactant species (SpeciesReference object) in the list of 
   * reactants in this {@link Reaction} and returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * The caller should first call getNumReactants() to find out how many
   * reactants there are, to avoid using an invalid index number.
   * <p>
   * @param n the index of the reactant {@link SpeciesReference} object to remove
   * <p>
   * @return the removed reactant {@link SpeciesReference} object, or NULL if the 
   * given index is out of range.
   */
 public";

%javamethodmodifiers Reaction::removeReactant(const std::string& species) "
  /**
   * Removes the reactant species (SpeciesReference object) having the given  
   * \'species\' attribute in this {@link Reaction} and returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * <p>
   * @param species the \'species\' attribute of the reactant {@link SpeciesReference} 
   * object
   * <p>
   * @return the removed reactant {@link SpeciesReference} object, or NULL if no 
   * reactant {@link SpeciesReference} object with the given \'species\' attribute 
   * <code>species</code> exists in this {@link Reaction}.
   */
 public";

%javamethodmodifiers Reaction::removeProduct(unsigned int n) "
  /**
   * Removes the nth product species (SpeciesReference object) in the list of 
   * products in this {@link Reaction} and returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * The caller should first call getNumProducts() to find out how many
   * products there are, to avoid using an invalid index number.
   * <p>
   * @param n the index of the product {@link SpeciesReference} object to remove
   * <p>
   * @return the removed product {@link SpeciesReference} object, or NULL if the 
   * given index is out of range.
   */
 public";

%javamethodmodifiers Reaction::removeProduct(const std::string& species) "
  /**
   * Removes the product species (SpeciesReference object) having the given  
   * \'species\' attribute in this {@link Reaction} and returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * <p>
   * @param species the \'species\' attribute of the product {@link SpeciesReference} 
   * object
   * <p>
   * @return the removed product {@link SpeciesReference} object, or NULL if no 
   * product {@link SpeciesReference} object with the given \'species\' attribute 
   * <code>species</code> exists in this {@link Reaction}.
   */
 public";

%javamethodmodifiers Reaction::removeModifier(unsigned int n) "
  /**
   * Removes the nth modifier species (ModifierSpeciesReference object) in 
   * the list of  modifiers in this {@link Reaction} and returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * The caller should first call getNumModifiers() to find out how many
   * modifiers there are, to avoid using an invalid index number.
   * <p>
   * @param n the index of the {@link ModifierSpeciesReference} object to remove
   * <p>
   * @return the removed {@link ModifierSpeciesReference} object, or NULL if the 
   * given index is out of range.
   */
 public";

%javamethodmodifiers Reaction::removeModifier(const std::string& species) "
  /**
   * Removes the modifier species (ModifierSpeciesReference object) having 
   * the given \'species\' attribute in this {@link Reaction} and returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * <p>
   * @param species the \'species\' attribute of the {@link ModifierSpeciesReference} 
   * object
   * <p>
   * @return the removed {@link ModifierSpeciesReference} object, or NULL if no 
   * {@link ModifierSpeciesReference} object with the given \'species\' attribute @p 
   * species exists in this {@link Reaction}.
   */
 public";

%javamethodmodifiers Reaction::setSBMLDocument(SBMLDocument* d) "
  /**
   * Sets the parent {@link SBMLDocument} of this SBML object.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Reaction::setParentSBMLObject(SBase* sb) "
  /**
   * Sets the parent SBML object of this SBML object.
   * <p>
   * @param sb the SBML object to use
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Reaction::getTypeCode() const "
  /**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for this object, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers Reaction::getElementName() const "
  /**
   * Returns the XML element name of this object, which for {@link Reaction}, is
   * always <code>\'reaction\'</code>.
   * <p>
   * @return the name of this element, i.e., <code>\'reaction\'</code>.
   */
 public";

%javamethodmodifiers Reaction::writeElements(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write out their contained
   * SBML objects as XML elements.  Be sure to call your parents
   * implementation of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Reaction::hasRequiredAttributes() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link Reaction} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link Reaction} object are:
   * id
   * <p>
   * @return a boolean value indicating whether all the required
   * attributes for this object have been defined.
   */
 public";

%javamethodmodifiers Reaction::Reaction "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether
   * all the required attributes for this {@link Reaction} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link Reaction} object are:
   * id
   * <p>
   * @return a boolean value indicating whether all the required
   * attributes for this object have been defined.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Reaction::createObject(XMLInputStream& stream) "
  /**
   * @return the SBML object corresponding to next {@link XMLToken} in the
   * XMLInputStream or NULL if the token was not recognized.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Reaction::readAttributes(const XMLAttributes& attributes) "
  /**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Reaction::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Reaction::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Reaction::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Reaction::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Reaction::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Reaction::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Reaction::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Reaction::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Reaction::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Reaction::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Reaction::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Reaction::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Reaction::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Reaction::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Reaction::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Reaction::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Reaction::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Reaction::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Reaction::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Reaction::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Reaction::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Reaction::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Reaction::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers Reaction::writeAttributes(XMLOutputStream& stream) const "
  /**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers ListOfReactions::clone() const "
  /**
   * Creates and returns a deep copy of this {@link ListOfReactions} instance.
   * <p>
   * @return a (deep) copy of this {@link ListOfReactions}.
   */
 public";

%javamethodmodifiers ListOfReactions::getTypeCode() const "
  /**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for this object, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers ListOfReactions::getItemTypeCode() const "
  /**
   * Returns the libSBML type code for the objects contained in this {@link ListOf}
   * (i.e., {@link Reaction} objects, if the list is non-empty).
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for the objects contained in this {@link ListOf}
   * instance, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers ListOfReactions::getElementName() const "
  /**
   * Returns the XML element name of this object, which for
   * {@link ListOfReactions}, is always <code>\'listOfReactions\'</code>.
   * <p>
   * @return the name of this element, i.e., <code>\'listOfReactions\'</code>.
   */
 public";

%javamethodmodifiers ListOfReactions::get(unsigned int n) "
  /**
   * Get a {@link Reaction} from the {@link ListOfReactions}.
   * <p>
   * @param n the index number of the {@link Reaction} to get.
   * <p>
   * @return the nth {@link Reaction} in this {@link ListOfReactions}.
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfReactions::get "
  /**
   * Get a {@link Reaction} from the {@link ListOfReactions}.
   * <p>
   * @param n the index number of the {@link Reaction} to get.
   * <p>
   * @return the nth {@link Reaction} in this {@link ListOfReactions}.
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfReactions::get(const std::string& sid) "
  /**
   * Get a {@link Reaction} from the {@link ListOfReactions}
   * based on its identifier.
   * <p>
   * @param sid a string representing the identifier 
   * of the {@link Reaction} to get.
   * <p>
   * @return {@link Reaction} in this {@link ListOfReactions}
   * with the given id or NULL if no such
   * {@link Reaction} exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfReactions::get(const std::string& sid) const "
  /**
   * Get a {@link Reaction} from the {@link ListOfReactions}
   * based on its identifier.
   * <p>
   * @param sid a string representing the identifier 
   * of the {@link Reaction} to get.
   * <p>
   * @return {@link Reaction} in this {@link ListOfReactions}
   * with the given id or NULL if no such
   * {@link Reaction} exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfReactions::remove(unsigned int n) "
  /**
   * Removes the nth item from this {@link ListOfReactions} items and returns a pointer to
   * it.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * <p>
   * @param n the index of the item to remove
   * <p>
   * @see #size()
   */
 public";

%javamethodmodifiers ListOfReactions::remove(const std::string& sid) "
  /**
   * Removes item in this {@link ListOfReactions} items with the given identifier.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * If none of the items in this list have the identifier <code>sid</code>, then 
   * <code>NULL</code> is returned.
   * <p>
   * @param sid the identifier of the item to remove
   * <p>
   * @return the item removed.  As mentioned above, the caller owns the
   * returned item.
   */
 public";

%javamethodmodifiers ListOfReactions::getElementPosition() const "
  /**
   * @return the ordinal position of the element with respect to its
   * siblings or -1 (default) to indicate the position is not significant.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers ListOfReactions::createObject(XMLInputStream& stream) "
  /**
   * @return the SBML object corresponding to next {@link XMLToken} in the
   * XMLInputStream or NULL if the token was not recognized.
   <p>
 * @deprecated libSBML internal
 */
 public";

%typemap(javaimports) KineticLaw "
/** 
 * LibSBML implementation of SBML\'s KineticLaw construct.
 * <p>
 * An object of class {@link KineticLaw} is used to describe the rate at which the
 * process defined by a given {@link Reaction} takes place.  {@link KineticLaw} has
 * subelements called \'math\' (for MathML content) and \'listOfParameters\'
 * (of class {@link ListOfParameters}), in addition to the attributes and
 * subelements it inherits from {@link SBase}.
 * <p>
 * {@link KineticLaw}\'s \'math\' subelement for holding a MathML formula defines the
 * rate of the reaction.  The formula may refer to other entities in a
 * model (Compartment, {@link Species}, {@link Parameter}, {@link FunctionDefinition}, {@link Reaction}),
 * but the only {@link Species} identifiers that can be used in this formula are
 * those declared in the lists of reactants, products and modifiers in the
 * {@link Reaction} structure.  {@link Parameter} identifiers may be taken from either the
 * {@link KineticLaw}\'s list of local parameters (discussed below) or the
 * parameters defined globally on the {@link Model} instance.
 * <p>
 * {@link KineticLaw}\'s \'listOfParameters\' subelement can contain a list of one or
 * more {@link Parameter} objects defining new parameters whose identifiers can be
 * used in the \'math\' formula of that {@link KineticLaw} instance.  The {@link Parameter}
 * objects so defined are only visible within the {@link KineticLaw}; they cannot
 * be accessed outside.  Moreover, if a {@link Parameter} has the same identifier
 * as a global {@link Parameter} object defined in the enclosing {@link Model}, the
 * definition inside the {@link KineticLaw} takes precedence (i.e., it shadows the
 * global definition).
 * <p>
 * The type of structure used to define a parameter inside {@link KineticLaw} is
 * the same {@link Parameter} structure used to define global parameters.  This
 * simplifies the SBML language and reduces the number of unique types of
 * data objects.  However, there is a difference between local and global
 * parameters: in the case of parameters defined locally to a {@link KineticLaw},
 * there is no means by which the parameter values can be changed.
 * Consequently, such parameters\' values are always constant, and the
 * \'constant\' attribute in their definitions must always have a value of 
 * <code>true</code> (either explicitly or left to its default value).  See the
 * definition of {@link Parameter} for more information about the \'constant\'
 * attribute.
 * <p>
 * <h2>Units of rate expressions in {@link KineticLaw}</h2>
 * <p>
 * It is important to make clear that a so-called kinetic law in SBML is
 * <em>not</em> identical to a traditional rate law.  The reason is that SBML
 * must support multicompartment models, and the units normally used in
 * traditional rate laws as well as some conventional single-compartment
 * modeling packages are problematic when used for defining reactions
 * between multiple compartments.  When modeling species as continuous
 * amounts (e.g., concentrations), the rate laws used are traditionally
 * expressed in terms of <em>amount of substance concentration per
 * time</em>, embodying a tacit assumption that reactants and products are
 * all located in a single, constant volume.  Attempting to describe
 * reactions between multiple volumes using
 * <em>concentration</em>/<em>time</em> (which is to say,
 * <em>substance</em>/<em>volume</em>/<em>time</em>) quickly leads to
 * difficulties, as explained in the SBML Level&nbsp;2 Version&nbsp;2, 3
 * and&nbsp;4 specifications.
 * <p>
 * The need to support multicompartment models requires that the reaction
 * rates in SBML to be expressed in terms of
 * <em>substance</em>/<em>time</em>, rather than the more typical
 * <em>substance</em>/<em>size</em>/<em>time</em>.  As a result, modelers
 * and software tools in general cannot insert textbook rate laws
 * unmodified as the rate expression in the \'math\' subelement of a
 * {@link KineticLaw}.  The unusual term \'kinetic law\' was chosen to alert users to
 * this difference.
 * <p>
 * Please consult the SBML specification document for more information
 * about the definition reaction rates in SBML.
 * <p>
 * @note Before SBML Level&nbsp;2 Version&nbsp;2, the SBML specification
 * included two additional attributes on {@link KineticLaw} called \'substanceUnits\'
 * and \'timeUnits\'.  They were removed beginning with SBML Level&nbsp;2
 * Version&nbsp;2 because further research determined they introduced many
 * problems.  The most significant problem was that their use could easily
 * lead to the creation of valid models whose reactions nevertheless could
 * not be integrated into a system of equations without outside knowledge
 * for converting the quantities used.  Examination of real-life models
 * revealed that a common reason for using \'substanceUnits\' on {@link KineticLaw}
 * was to set the units of all reactions to the same set of substance
 * units, something that is better achieved by using {@link UnitDefinition} to
 * redefine <code>\'substance\'</code> for the whole {@link Model}.
 */
"

%javamethodmodifiers KineticLaw::KineticLaw(unsigned int level, unsigned int version) "
  /**
   * Creates a new {@link KineticLaw} using the given SBML <code>level</code> and <code>version</code>
   * values.
   * <p>
   * @param level a long integer, the SBML Level to assign to this {@link KineticLaw}
   * <p>
   * @param version a long integer, the SBML Version to assign to this
   * {@link KineticLaw}
   * <p>
   * @note Once a {@link KineticLaw} has been added to an {@link SBMLDocument}, the <code>level</code>,
   * <code>version</code> for the document <em>override</em> those used
   * to create the {@link KineticLaw}.  Despite this, the ability to supply the values
   * at creation time is an important aid to creating valid SBML.  Knowledge of
   * the intented SBML Level and Version determine whether it is valid to
   * assign a particular value to an attribute, or whether it is valid to add
   * an object to an existing {@link SBMLDocument}.
   */
 public";

%javamethodmodifiers KineticLaw::KineticLaw(SBMLNamespaces* sbmlns) "
  /**
   * Creates a new {@link KineticLaw} using the given {@link SBMLNamespaces} object
   * <code>sbmlns</code>.
   * <p>
   * The {@link SBMLNamespaces} object encapsulates SBML Level/Version/namespaces
   * information.  It is used to communicate the SBML Level, Version, and
   * (in Level&nbsp;3) packages used in addition to SBML Level&nbsp; Core.
   * A common approach to using this class constructor is to create an
   * {@link SBMLNamespaces} object somewhere in a program, once, then pass it to
   * object constructors such as this one when needed.
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object.
   * <p>
   * @note Once a {@link KineticLaw} has been added to an {@link SBMLDocument}, the <code>level</code>,
   * <code>version</code> and <code>xmlns</code> namespaces for the document <em>override</em> those used
   * to create the {@link KineticLaw}.  Despite this, the ability to supply the values
   * at creation time is an important aid to creating valid SBML.  Knowledge of
   * the intented SBML Level and Version determine whether it is valid to
   * assign a particular value to an attribute, or whether it is valid to add
   * an object to an existing {@link SBMLDocument}.
   */
 public";

%javamethodmodifiers KineticLaw::KineticLaw(const KineticLaw& orig) "
  /**
   * Copy constructor; creates a copy of this {@link KineticLaw}.
   */
 public";

%javamethodmodifiers KineticLaw::accept(SBMLVisitor& v) const "
  /**
   * Accepts the given {@link SBMLVisitor} for this instance of {@link KineticLaw}.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used.
   * <p>
   * @return the result of calling <code>v.visit()</code>.
   */
 public";

%javamethodmodifiers KineticLaw::clone() const "
  /**
   * Creates and returns a deep copy of this {@link KineticLaw} object.
   * <p>
   * @return a (deep) copy of this {@link KineticLaw}.
   */
 public";

%javamethodmodifiers KineticLaw::getFormula() const "
  /**
   * Returns the mathematical formula for this {@link KineticLaw} object and return
   * it as as a text string.
   * <p>
   * This is fundamentally equivalent to getMath().  This variant is
   * provided principally for compatibility compatibility with SBML Level
   * 1.
   * <p>
   * @return a string representing the formula of this {@link KineticLaw}.
   * <p>
   * @see #getMath()
   * <p>
   * @note SBML Level&nbsp;1 uses a text-string format for mathematical
   * formulas.  SBML Level&nbsp;2 uses MathML, an XML format for
   * representing mathematical expressions.  LibSBML provides an Abstract
   * Syntax Tree API for working with mathematical expressions; this API is
   * more powerful than working with formulas directly in text form, and
   * ASTs can be translated into either MathML or the text-string syntax.
   * The libSBML methods that accept text-string formulas directly (such as
   * this constructor) are provided for SBML Level&nbsp;1 compatibility,
   * but developers are encouraged to use the AST mechanisms.
   */
 public";

%javamethodmodifiers KineticLaw::getMath() const "
  /**
   * Returns the mathematical formula for this {@link KineticLaw} object and return
   * it as as an AST.
   * <p>
   * This is fundamentally equivalent to getFormula().  The latter is
   * provided principally for compatibility compatibility with SBML Level
   * 1, which represented mathematical formulas in text-string form.
   * <p>
   * @return the {@link ASTNode} representation of the mathematical formula.
   * <p>
   * @see #getFormula()
   */
 public";

%javamethodmodifiers KineticLaw::getTimeUnits() const "
  /**
   * Returns the value of the \'timeUnits\' attribute of this {@link KineticLaw}
   * object.
   * <p>
   * @return the \'timeUnits\' attribute value
   * <p>
   * @warning In SBML Level&nbsp;2 Version&nbsp;2, the \'timeUnits\' and
   * \'substanceUnits\' attributes were removed.  For compatibility with new
   * versions of SBML, users are cautioned to avoid these attributes.
   */
 public";

%javamethodmodifiers KineticLaw::getSubstanceUnits() const "
  /**
   * Returns the value of the \'substanceUnits\' attribute of this {@link KineticLaw}
   * object.
   * <p>
   * @return the \'substanceUnits\' attribute value
   * <p>
   * @warning In SBML Level&nbsp;2 Version&nbsp;2, the \'timeUnits\' and
   * \'substanceUnits\' attributes were removed.  For compatibility with new
   * versions of SBML, users are cautioned to avoid these attributes.
   */
 public";

%javamethodmodifiers KineticLaw::isSetFormula() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link KineticLaw}\'s \'formula\' attribute has been set
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * This is functionally identical to the method isSetMath().  It is
   * provided in order to mirror the parallel between getFormula() and
   * getMath().
   * <p>
   * @return <code>true</code> if the formula (meaning the <code>math</code> subelement) of
   * this {@link KineticLaw} has been set, <code>false</code> otherwise.
   * <p>
   * @note SBML Level&nbsp;1 uses a text-string format for mathematical
   * formulas.  SBML Level&nbsp;2 uses MathML, an XML format for
   * representing mathematical expressions.  LibSBML provides an Abstract
   * Syntax Tree API for working with mathematical expressions; this API is
   * more powerful than working with formulas directly in text form, and
   * ASTs can be translated into either MathML or the text-string syntax.
   * The libSBML methods that accept text-string formulas directly (such as
   * this constructor) are provided for SBML Level&nbsp;1 compatibility,
   * but developers are encouraged to use the AST mechanisms.
   */
 public";

%javamethodmodifiers KineticLaw::isSetMath() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * Kinetic\'s \'math\' subelement has been set
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * This is identical to the method isSetFormula().  It is provided
   * in order to mirror the parallel between getFormula() and getMath().
   * <p>
   * @return <code>true</code> if the formula (meaning the <code>math</code> subelement) of
   * this {@link KineticLaw} has been set, <code>false</code> otherwise.
   */
 public";

%javamethodmodifiers KineticLaw::isSetTimeUnits() const "
 /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link SpeciesReference}\'s \'timeUnits\' attribute has been set
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the \'timeUnits\' attribute of this {@link KineticLaw} object
   * has been set, <code>false</code> otherwise.
   * <p>
   * @warning In SBML Level&nbsp;2 Version&nbsp;2, the \'timeUnits\' and
   * \'substanceUnits\' attributes were removed.  For compatibility with new
   * versions of SBML, users are cautioned to avoid these attributes.
   */
 public";

%javamethodmodifiers KineticLaw::isSetSubstanceUnits() const "
 /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether this
   * {@link SpeciesReference}\'s \'substanceUnits\' attribute has been set
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return <code>true</code> if the \'substanceUnits\' attribute of this {@link KineticLaw}
   * object has been set, <code>false</code> otherwise.
   * <p>
   * @warning In SBML Level&nbsp;2 Version&nbsp;2, the \'timeUnits\' and
   * \'substanceUnits\' attributes were removed.  For compatibility with new
   * versions of SBML, users are cautioned to avoid these attributes.
   */
 public";

%javamethodmodifiers KineticLaw::setFormula(const std::string& formula) "
  /**
   * Sets the mathematical expression of this {@link KineticLaw} instance to the
   * given <code>formula</code>.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * The given <code>formula</code> string is copied.  Internally, libSBML stores the
   * mathematical expression as an {@link ASTNode}.
   * <p>
   * @param formula the mathematical expression to use, represented in
   * text-string form.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_OBJECT
   * <p>
   * @note SBML Level&nbsp;1 uses a text-string format for mathematical
   * formulas.  SBML Level&nbsp;2 uses MathML, an XML format for representing
   * mathematical expressions.  LibSBML provides an Abstract Syntax Tree
   * API for working with mathematical expressions; this API is more
   * powerful than working with formulas directly in text form, and ASTs
   * can be translated into either MathML or the text-string syntax.  The
   * libSBML methods that accept text-string formulas directly (such as
   * this constructor) are provided for SBML Level&nbsp;1 compatibility, but
   * developers are encouraged to use the AST mechanisms.
   */
 public";

%javamethodmodifiers KineticLaw::setMath(const ASTNode* math) "
  /**
   * Sets the mathematical expression of this {@link KineticLaw} instance to a copy
   * of the given {@link ASTNode}.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * This is fundamentally identical to setFormula().  The latter is
   * provided principally for compatibility compatibility with SBML Level
   * 1, which represented mathematical formulas in text-string form.
   * <p>
   * @param math an {@link ASTNode} representing a formula tree.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_OBJECT
  */
 public";

%javamethodmodifiers KineticLaw::setTimeUnits(const std::string& sid) "
  /**
   * Sets the \'timeUnits\' attribute of this {@link KineticLaw} object to a copy of
   * the identifier in <code>sid</code>.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param sid the identifier of the units to use.
   * <p>
   * @warning In SBML Level&nbsp;2 Version&nbsp;2, the \'timeUnits\' and
   * \'substanceUnits\' attributes were removed.  For compatibility with new
   * versions of SBML, users are cautioned to avoid these attributes.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   * <li> LIBSBML_UNEXPECTED_ATTRIBUTE
   */
 public";

%javamethodmodifiers KineticLaw::setSubstanceUnits(const std::string& sid) "
  /**
   * Sets the \'substanceUnits\' attribute of this {@link KineticLaw} object to a copy
   * of the identifier given in <code>sid</code>.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @param sid the identifier of the units to use.
   * <p>
   * @warning In SBML Level&nbsp;2 Version&nbsp;2, the \'timeUnits\' and
   * \'substanceUnits\' attributes were removed.  For compatibility with new
   * versions of SBML, users are cautioned to avoid these attributes.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_INVALID_ATTRIBUTE_VALUE
   * <li> LIBSBML_UNEXPECTED_ATTRIBUTE
   */
 public";

%javamethodmodifiers KineticLaw::unsetTimeUnits "
  /**
   * Unsets the \'timeUnits\' attribugte of this {@link KineticLaw} object.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_OPERATION_FAILED
   * <p>
   * @warning In SBML Level&nbsp;2 Version&nbsp;2, the \'timeUnits\' and
   * \'substanceUnits\' attributes were removed.  For compatibility with new
   * versions of SBML, users are cautioned to avoid these attributes.
   */
 public";

%javamethodmodifiers KineticLaw::unsetSubstanceUnits "
  /**
   * Unsets the \'substanceUnits\' attribute of this {@link KineticLaw} object.
   * <p>
   * <em>Some words of explanation about the</em>
<code>set</code>/<code>unset</code>/<code>isSet</code> <em>methods</em>:
SBML Levels 1 and 2 define certain attributes on some classes of objects as
optional.  This requires an application to be careful about the distinction
between two cases: (1) a given attribute has <em>never</em> been set to a
value, and therefore should be assumed to have the SBML-defined default
value, and (2) a given attribute has been set to a value, but the value
happens to be an empty string.  LibSBML supports these distinctions by
providing methods to set, unset, and query the status of attributes that
are optional.  The methods have names of the form
<code>set</code><i>Attribute</i><code>(...)</code>,
<code>unset</code><i>Attribute</i><code>()</code>, and
<code>isSet</code><i>Attribute</i><code>()</code>, where <i>Attribute</i>
is the the name of the optional attribute in question.

   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_OPERATION_FAILED
   * <p>
   * @warning In SBML Level&nbsp;2 Version&nbsp;2, the \'timeUnits\' and
   * \'substanceUnits\' attributes were removed.  For compatibility with new
   * versions of SBML, users are cautioned to avoid these attributes.
   */
 public";

%javamethodmodifiers KineticLaw::addParameter(const Parameter* p) "
  /**
   * Adds a copy of the given {@link Parameter} object to the list of local
   * parameters in this {@link KineticLaw}.
   * <p>
   * @param p the {@link Parameter} to add
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <li> LIBSBML_OPERATION_SUCCESS
   * <li> LIBSBML_LEVEL_MISMATCH
   * <li> LIBSBML_VERSION_MISMATCH
   * <li> LIBSBML_DUPLICATE_OBJECT_ID
   * <li> LIBSBML_OPERATION_FAILED
   * <p>
   * @note This method should be used with some caution.  The fact that
   * this method <em>copies</em> the object passed to it means that the caller
   * will be left holding a physically different object instance than the
   * one contained in this {@link KineticLaw}.  Changes made to the original object
   * instance (such as resetting attribute values) will <em>not affect the
   * instance in the {@link KineticLaw}</em>.  In addition, the caller should make
   * sure to free the original object if it is no longer being used, or
   * else a memory leak will result.  Please see
   * KineticLaw.createParameter() for ab method that does not lead to
   * these issues.
   * <p>
   * @see #createParameter()
   */
 public";

%javamethodmodifiers KineticLaw::createParameter "
  /**
   * Creates a new {@link Parameter} object, adds it to this {@link KineticLaw}\'s list of
   * local parameters, and returns the {@link Parameter} object created.
   * <p>
   * @return a new {@link Parameter} object instance
   * <p>
   * @see #addParameter(Parameter  p)
   */
 public";

%javamethodmodifiers KineticLaw::getListOfParameters() const "
  /**
   * Returns the list of local parameters in this {@link KineticLaw} object.
   * <p>
   * @return the list of Parameters for this {@link KineticLaw}.
   */
 public";

%javamethodmodifiers KineticLaw::getListOfParameters "
  /**
   * Returns the list of local parameters in this {@link KineticLaw} object.
   * <p>
   * @return the list of Parameters for this {@link KineticLaw}.
   */
 public";

%javamethodmodifiers KineticLaw::getParameter "
  /**
   * Returns the nth {@link Parameter} object in the list of local parameters in
   * this {@link KineticLaw} instance.
   * <p>
   * @param n the index of the {@link Parameter} object sought
   * <p>
   * @return the nth {@link Parameter} of this {@link KineticLaw}.
   */
 public";

%javamethodmodifiers KineticLaw::getParameter(unsigned int n) "
  /**
   * Returns the nth {@link Parameter} object in the list of local parameters in
   * this {@link KineticLaw} instance.
   * <p>
   * @param n the index of the {@link Parameter} object sought
   * <p>
   * @return the nth {@link Parameter} of this {@link KineticLaw}.
   */
 public";

%javamethodmodifiers KineticLaw::getParameter(const std::string& sid) const "
  /**
   * Returns a local parameter based on its identifier.
   * <p>
   * @param sid the identifier of the {@link Parameter} being sought.
   * <p>
   * @return the {@link Parameter} object in this {@link KineticLaw} instace having the
   * given \'id\', or <code>NULL</code> if no such {@link Parameter} exists.
   */
 public";

%javamethodmodifiers KineticLaw::getParameter(const std::string& sid) "
  /**
   * Returns a local parameter based on its identifier.
   * <p>
   * @param sid the identifier of the {@link Parameter} being sought.
   * <p>
   * @return the {@link Parameter} object in this {@link KineticLaw} instace having the
   * given \'id\', or <code>NULL</code> if no such {@link Parameter} exists.
   */
 public";

%javamethodmodifiers KineticLaw::getNumParameters() const "
  /**
   * Returns the number of local parameters in this {@link KineticLaw} instance.
   * <p>
   * @return the number of Parameters in this {@link KineticLaw}.
   */
 public";

%javamethodmodifiers KineticLaw::getDerivedUnitDefinition "
  /**
   * Calculates and returns a {@link UnitDefinition} that expresses the units
   * of measurement assumed for the \'math\' expression of this
   * {@link KineticLaw}.
   * <p>
   * The units are calculated based on the mathematical expression in the
   * {@link KineticLaw} and the model quantities referenced by
   * <code>&lt;ci&gt;</code> elements used within that expression.  The
   * getDerivedUnitDefinition() method returns the calculated units.
   * <p>
   * Note that the functionality that facilitates unit analysis depends 
   * on the model as a whole.  Thus, in cases where the object has not 
   * been added to a model or the model itself is incomplete,
   * unit analysis is not possible and this method will return NULL.
   * <p>
   * @warning Note that it is possible the \'math\' expression in the
   * {@link KineticLaw} contains pure numbers or parameters with undeclared
   * units.  In those cases, it is not possible to calculate the units of
   * the overall expression without making assumptions.  LibSBML does not
   * make assumptions about the units, and getDerivedUnitDefinition() only
   * returns the units as far as it is able to determine them.  For
   * example, in an expression <em>X + Y</em>, if <em>X</em> has
   * unambiguously-defined units and <em>Y</em> does not, it will return
   * the units of <em>X</em>.  <strong>It is important that callers also
   * invoke the method</strong> containsUndeclaredUnits() <strong>to
   * determine whether this situation holds</strong>.  Callers may wish to
   * take suitable actions in those scenarios.
   * <p>
   * @return a {@link UnitDefinition} that expresses the units of the math 
   * expression of this {@link KineticLaw}.
   * <p>
   * @see #containsUndeclaredUnits()
   */
 public";

%javamethodmodifiers KineticLaw::getDerivedUnitDefinition() const "
  /**
   * Calculates and returns a {@link UnitDefinition} that expresses the units
   * of measurement assumed for the \'math\' expression of this
   * {@link KineticLaw}.
   * <p>
   * The units are calculated based on the mathematical expression in the
   * {@link KineticLaw} and the model quantities referenced by
   * <code>&lt;ci&gt;</code> elements used within that expression.  The
   * getDerivedUnitDefinition() method returns the calculated units.
   * <p>
   * Note that the functionality that facilitates unit analysis depends 
   * on the model as a whole.  Thus, in cases where the object has not 
   * been added to a model or the model itself is incomplete,
   * unit analysis is not possible and this method will return NULL.
   * <p>
   * @warning Note that it is possible the \'math\' expression in the
   * {@link KineticLaw} contains pure numbers or parameters with undeclared
   * units.  In those cases, it is not possible to calculate the units of
   * the overall expression without making assumptions.  LibSBML does not
   * make assumptions about the units, and getDerivedUnitDefinition() only
   * returns the units as far as it is able to determine them.  For
   * example, in an expression <em>X + Y</em>, if <em>X</em> has
   * unambiguously-defined units and <em>Y</em> does not, it will return
   * the units of <em>X</em>.  <strong>It is important that callers also
   * invoke the method</strong> containsUndeclaredUnits() <strong>to
   * determine whether this situation holds</strong>.  Callers may wish to
   * take suitable actions in those scenarios.
   * <p>
   * @return a {@link UnitDefinition} that expresses the units of the math 
   * expression of this {@link KineticLaw}.
   * <p>
   * @see #containsUndeclaredUnits()
   */
 public";

%javamethodmodifiers KineticLaw::containsUndeclaredUnits "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether 
   * the math expression of this {@link KineticLaw} contains
   * parameters/numbers with undeclared units.
   * <p>
   * @return <code>true</code> if the math expression of this {@link KineticLaw}
   * includes parameters/numbers 
   * with undeclared units, <code>false</code> otherwise.
   * <p>
   * @note A return value of <code>true</code> indicates that the {@link UnitDefinition}
   * returned by getDerivedUnitDefinition() may not accurately represent
   * the units of the expression.
   * <p>
   * @see #getDerivedUnitDefinition()
   */
 public";

%javamethodmodifiers KineticLaw::containsUndeclaredUnits() const "
  /**
   * Predicate returning <code>true</code> or <code>false</code> depending on whether 
   * the math expression of this {@link KineticLaw} contains
   * parameters/numbers with undeclared units.
   * <p>
   * @return <code>true</code> if the math expression of this {@link KineticLaw}
   * includes parameters/numbers 
   * with undeclared units, <code>false</code> otherwise.
   * <p>
   * @note A return value of <code>true</code> indicates that the {@link UnitDefinition}
   * returned by getDerivedUnitDefinition() may not accurately represent
   * the units of the expression.
   * <p>
   * @see #getDerivedUnitDefinition()
   */
 public";

%javamethodmodifiers KineticLaw::removeParameter(unsigned int n) "
  /**
   * Removes the nth {@link Parameter} object in the list of local parameters 
   * in this {@link KineticLaw} instance and returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * <p>
   * @param n the index of the {@link Parameter} object to remove
   * <p>
   * @return the {@link Parameter} object removed.  As mentioned above, 
   * the caller owns the returned item. NULL is returned if the given index 
   * is out of range.
   */
 public";

%javamethodmodifiers KineticLaw::removeParameter(const std::string& sid) "
  /**
   * Removes a {@link Parameter} object with the given identifier in the list of
   * local parameters in this {@link KineticLaw} instance and returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * <p>
   * @param sid the identifier of the {@link Parameter} to remove
   * <p>
   * @return the {@link Parameter} object removed.  As mentioned above, the 
   * caller owns the returned object. NULL is returned if no {@link Parameter}
   * object with the identifier exists in this {@link KineticLaw} instance.
   */
 public";

%javamethodmodifiers KineticLaw::setSBMLDocument(SBMLDocument* d) "
  /**
   * Sets the parent {@link SBMLDocument} of this SBML object.
   * <p>
   * @param d the {@link SBMLDocument} to use.
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers KineticLaw::setParentSBMLObject(SBase* sb) "
  /**
   * Sets the parent SBML object of this SBML object.
   * <p>
   * @param sb the SBML object to use
   <p>
 * @deprecated libSBML internal
 */
 public";

%javamethodmodifiers KineticLaw::getTypeCode() const "
  /**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an
   * identifying code to every kind of SBML object.  These are known as
   * <em>SBML type codes</em>.  In other languages, the set of type codes
   * is stored in an enumeration; in the Java language interface for
   * libSBML, the type codes are defined as static integer constants in
   * interface class {@link libsbmlConstants}.  The names of the type codes
   * all begin with the characters <code>SBML_</code>. 
   * <p>
   * @return the SBML type code for this object, or <code>SBML_UNKNOWN</code> (default).
   * <p>
   * @see #getElementName()
   */
 public";

%javamethodmodifiers KineticLaw::getElementName() const "
  /**
   * Returns the XML element name of this object, which for {@link Species}, is
   * always <code>\'kineticLaw\'</code>.
   * <p>
   * @return the name of this element, i.e., <code>\'kineticLaw\'</code>.
   */
 public";

%javamethodmodifiers KineticLaw::getElementPosition() const "
  /**
   * @return the ordinal position of the element with respect to its
   * siblings or -1 (default) to indicate the positi