// Copyright 2001-2003 Erwin Bolwidt. All rights reserved.
// See the file LICENSE.txt in this package for information about licensing.
package org.jaxup.xupdate;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import org.jaxen.JaxenException;
import org.jaxen.Navigator;
import org.jaxen.SimpleNamespaceContext;
import org.jaxen.SimpleVariableContext;
import org.jaxen.XPath;
import org.jaxen.pattern.Pattern;
import org.jaxup.UpdateException;
import org.jaxup.Updater;
import org.saxpath.SAXPathException;

/** 
 * XUpdate engine. This class can, using the instructions in an XUpdate
 * instruction document, change another XML document in-place. For small changes
 * to large XML files, this may be preferable over a full XSLT translation. It
 * is especially efficient if the XML document is either kept in memory for a
 * long time, or is stored in a persistent database.
 *  
 * @author Erwin Bolwidt
 */
public class XUpdate
{
    /**
     * Namespace for xupdate command elements.
     */
    public final static String NS_XUPDATE = "http://www.xmldb.org/xupdate";

    /**
     * Internal code for xupdate "append" operation.
     */
    private final static int OP_APPEND = 0;
    /**
     * Internal code for xupdate "update" operation.
    */
    private final static int OP_UPDATE = 1;
    /**
     * Internal code for xupdate "insert-before" operation.
     */
    private final static int OP_INSERT_BEFORE = 2;
    /**
     * Internal code for xupdate "insert-after" operation.
     */
    private final static int OP_INSERT_AFTER = 3;

    /**
     * XML element name for element that contains xupdate instructions.
     */
    private final static String MODIFICATIONS = "modifications";
    private final static String CMD_APPEND = "append";
    private final static String CMD_UPDATE = "update";
    private final static String CMD_INSERT_BEFORE = "insert-before";
    private final static String CMD_INSERT_AFTER = "insert-after";
    private final static String CMD_REMOVE = "remove";
    private final static String CMD_IF = "if";
    private final static String CMD_VARIABLE = "variable";
    private static final String XPR_TEXT = "text";
    private static final String XPR_COMMENT = "comment";
    private static final String XPR_PROC_INSTR = "processing-instruction";
    private static final String XPR_ELEMENT = "element";
    private static final String XPR_ATTRIBUTE = "attribute";
    private static final String XPR_VALUE_OF = "value-of";
    private static final String AT_CHILD = "child";
    private static final String AT_NAMESPACE = "namespace";
    private static final String AT_TEST = "test";
    private static final String AT_SELECT = "select";
    private static final String AT_NAME = "name";

    /**
     * XPath variables.
     */
    private SimpleVariableContext variables = new SimpleVariableContext();

    /**
     * The updater for the document being updated.
     */
    private Updater docUpdater;

    /**
     * The navigator for the document being updated.
     */
    private Navigator docNavigator;

    /**
     * The navigator for the xupdate command file.
     */
    private Navigator xuNavigator;

    /**
     * Creates an xupdate engine. The source documents that are updated,
     * <i>and</i> the xml documents containing the xupdate instructions, are
     * both stored in the same data model, e.g. DOM, JDOM, DOM4J, EXML.
     * 
     * @param updater An implementation of the Updater interface to update
     * source and rad xupdate instruction documents.
     */
    public XUpdate(Updater docUpdater)
    {
        this.docUpdater = docUpdater;
        this.docNavigator = docUpdater.getNavigator();
        this.xuNavigator = docUpdater.getNavigator();
    }

    /**
     * Creates an xupdate engine. The source documents that are updated
     * and the xml documents containing the xupdate instructions may be stored
     * in different data models, i.e. the source document in W3C DOM and the
     * xupdate instructions in JDOM.
     *
     *
     * @param docUpdater An implementation of the Updater interface to update
     * source documents.
     *                     
     * @param xuNavigator An implementation of the Navigator interface to read
     * xupdate instruction documents.
     */
    public XUpdate(Updater docUpdater, Navigator xuNavigator)
    {
        this.docUpdater = docUpdater;
        this.docNavigator = docUpdater.getNavigator();
        this.xuNavigator = xuNavigator;
    }

    /**
     * Runs a series of xupdate instructions to manipulate a given source
     * document.
     *
     * @param sourceDoc The source document that is going to be changed.
     * @param xupdateContainer An element node that contains xupdate
     * instructions as its children. Ordinarily, this would be the document
     * element of an xupdate XML document.
     */
    public void runUpdate(Object sourceDoc, Object xupdateContainer)
        throws JaxenException, SAXPathException, UpdateException
    {
        executeUpdateInstructions(sourceDoc, xupdateContainer);
    }

    /**
     * Runs a series of xupdate instructions to manipulate a given source
     * document.
     *
     * @param sourceDoc The source document that is going to be changed.
     * @param xupdateContainer An element node that contains xupdate
     * instructions as its children. Ordinarily, this would be the document
     * element of an xupdate XML document.
     */
    public void runUpdateDocument(Object sourceDoc, Object xupdateDocument)
        throws JaxenException, SAXPathException, UpdateException
    {
        if (!docNavigator.isDocument(xupdateDocument))
        {
            throw new IllegalArgumentException(
                "xupdateDocument is not a document in " + "the Navigator's document model");
        }
        for (Iterator docChildren = docNavigator.getChildAxisIterator(xupdateDocument);
            docChildren.hasNext();
            )
        {
            Object child = docChildren.next();
            if (!docNavigator.isElement(child))
            {
                continue;
            }
            if (!NS_XUPDATE.equals(docNavigator.getElementNamespaceUri(child)))
            {
                throw new XUpdateException(
                    "Parameter is not a valid xupdate document, " + "wrong namespace for document element");
            }
            if (!MODIFICATIONS.equals(docNavigator.getElementName(child)))
            {
                throw new XUpdateException(
                    "Parameter is not a valid xupdate document, "
                        + "document element name is not <"
                        + MODIFICATIONS
                        + ">");
            }
            executeUpdateInstructions(sourceDoc, child);
        }
    }

    /**
     * Handles an xupdate 'modifications' or 'if' instruction container.
     */
    protected void executeUpdateInstructions(Object doc, Object instructionsContainer)
        throws JaxenException, SAXPathException, UpdateException
    {
        Iterator children = xuNavigator.getChildAxisIterator(instructionsContainer);
        while (children.hasNext())
        {
            Object node = children.next();
            // Ignore non-elements
            if (!xuNavigator.isElement(node))
                continue;
            String uri = xuNavigator.getElementNamespaceUri(node);
            if (!NS_XUPDATE.equals(uri))
            {
                throw new JaxenException("Element with unsupported namespace " + uri + ": " + node);
            }

            String lname = xuNavigator.getElementName(node);
            if (CMD_REMOVE.equals(lname))
                remove(doc, node);
            else if (CMD_APPEND.equals(lname))
                creation(doc, node, OP_APPEND);
            else if (CMD_UPDATE.equals(lname))
                creation(doc, node, OP_UPDATE);
            else if (CMD_INSERT_BEFORE.equals(lname))
                creation(doc, node, OP_INSERT_BEFORE);
            else if (CMD_INSERT_AFTER.equals(lname))
                creation(doc, node, OP_INSERT_AFTER);
            else if (CMD_IF.equals(lname))
                conditional(doc, node);
            else if (CMD_VARIABLE.equals(lname))
                variable(doc, node);
            else
            {
                throw new JaxenException("Unsupported xupdate element: " + node);
            }
        }
    }

    /**
     * Returns attribute without namespace with the specified NCName in
     * specified element, or null if there is no such attribute.
     */
    protected static Object getAttributeNode(Navigator nav, Object element, String name)
        throws JaxenException
    {
        Iterator attrs = nav.getAttributeAxisIterator(element);
        while (attrs.hasNext())
        {
            Object attr = attrs.next();
            if (nav.getAttributeNamespaceUri(attr) == null && name.equals(nav.getAttributeName(attr)))
            {
                return attr;
            }
        }
        return null;
    }

    /**
     * Parses an xpath expression in the context of an element with regard to
     * namespace bindings.
     * 
     * @param xpath The xpath expression.
     * @param xuElement The element in the xupdate document from which the
     * namespace bindings should be used in the xpath expression.
     * @return The parsed xpath expression, bound to the variable context of
     * this xupdate engine, and with its namespaces bound like the given
     * element.
     */
    protected XPath parseXPath(String xpath, Object xuElement) throws JaxenException, SAXPathException
    {
        XPath xpathExpr = docNavigator.parseXPath(xpath);
        xpathExpr.setVariableContext(variables);
        SimpleNamespaceContext nsc = new SimpleNamespaceContext();
        nsc.addElementNamespaces(xuNavigator, xuElement);
        xpathExpr.setNamespaceContext(nsc);
        return xpathExpr;
    }

    /**
     * Splits a Qname into a prefix and a localname, and returns these as an
     * array of two strings.
     */
    protected static String[] splitQName(String qname) throws JaxenException
    {
        int colon = qname.indexOf(':');
        if (colon == -1)
        {
            return new String[] { "", qname };
        }
        else
        {
            return new String[] { qname.substring(0, colon), qname.substring(colon + 1)};
        }
    }

    protected void variable(Object doc, Object instrNode)
        throws JaxenException, SAXPathException, UpdateException
    {
        Object nameAttr = getAttributeNode(xuNavigator, instrNode, AT_NAME);
        if (nameAttr == null)
        {
            throw new JaxenException("No name attribute for " + instrNode);
        }
        // Interpret variable name as QName
        String qname = xuNavigator.getAttributeStringValue(nameAttr);
        String[] split = splitQName(qname);
        String lname = split[1];
        String uri = xuNavigator.translateNamespacePrefixToUri(split[0], instrNode);
        if (uri != null && uri.length() == 0)
        {
            uri = null;
        }
        Object selectAttr = getAttributeNode(xuNavigator, instrNode, AT_SELECT);
        if (selectAttr == null)
        { // Instantiate template
            List instance = instantiateTemplate(doc, instrNode, null);
            variables.setVariableValue(uri, lname, instance);
        }
        else
        {
            String xpathStr = xuNavigator.getAttributeStringValue(selectAttr);
            XPath xpath = parseXPath(xpathStr, instrNode);
            List nodes = xpath.selectNodes(doc);
            // Is this the right select, to also get strings etc. as result?
            variables.setVariableValue(uri, lname, nodes);
        }
    }

    /** 
     * Handles an xupdate 'if' instruction.
     */
    protected void conditional(Object doc, Object instrNode)
        throws JaxenException, SAXPathException, UpdateException
    {
        Object testAttr = getAttributeNode(xuNavigator, instrNode, AT_TEST);
        if (testAttr == null)
        {
            throw new JaxenException("No test attribute for " + instrNode);
        }

        String xpathStr = xuNavigator.getAttributeStringValue(testAttr);
        XPath xpath = parseXPath(xpathStr, instrNode);
        boolean test = xpath.booleanValueOf(doc);
        if (test)
        {
            executeUpdateInstructions(doc, instrNode);
        }
    }

    /** 
     * Handles an xupdate 'remove' instruction.
     */
    protected void remove(Object doc, Object removeNode)
        throws JaxenException, UpdateException, SAXPathException
    {
        Object selectAttr = getAttributeNode(xuNavigator, removeNode, AT_SELECT);
        if (selectAttr == null)
        {
            throw new JaxenException("No select attribute for " + removeNode);
        }

        String xpathStr = xuNavigator.getAttributeStringValue(selectAttr);
        XPath xpath = parseXPath(xpathStr, removeNode);
        List nodes = xpath.selectNodes(doc);
        if (nodes == null)
        {
            return;
        }
        for (Iterator iter = nodes.iterator(); iter.hasNext();)
        {
            docUpdater.remove(iter.next());
        }
    }

    protected Object evaluateElement(Object doc, Object instrNode, Object selectContext)
        throws JaxenException, SAXPathException, UpdateException
    {

        Object nameAttr = getAttributeNode(xuNavigator, instrNode, AT_NAME);
        if (nameAttr == null)
        {
            throw new JaxenException("No name attribute for " + instrNode);
        }
        // Interpret variable name as QName
        String qname = xuNavigator.getAttributeStringValue(nameAttr);
        String[] split = splitQName(qname);
        String uri;
        Object namespace = getAttributeNode(xuNavigator, instrNode, AT_NAMESPACE);
        if (namespace != null)
        {
            uri = xuNavigator.getAttributeStringValue(namespace);
        }
        else
        {
            uri = xuNavigator.translateNamespacePrefixToUri(split[0], instrNode);
        }
        Object element = docUpdater.createElement(doc, uri, qname);
        List fragment = instantiateTemplate(doc, instrNode, selectContext);
        Iterator iter = fragment.iterator();
        while (iter.hasNext())
        {
            Object node = iter.next();
            if (docNavigator.isAttribute(node))
            {
                docUpdater.setAttribute(element, node);
            }
            else if (docNavigator.isNamespace(node))
            {
                docUpdater.setNamespace(element, node);
            }
            else
            {
                docUpdater.appendChild(element, node, -1);
            }
        }

        return element;
    }

    protected String getNodeListStringValue(List nodeList)
    {
        StringBuffer result = new StringBuffer();
        for (Iterator i = nodeList.iterator(); i.hasNext();)
        {
            Object node = i.next();
            if (docNavigator.isText(node))
            {
                result.append(docNavigator.getTextStringValue(node));
            }
            else if (docNavigator.isAttribute(node))
            {
                result.append(docNavigator.getAttributeStringValue(node));
            }
            else if (docNavigator.isElement(node))
            {
                result.append(docNavigator.getElementStringValue(node));
            }
        }
        return result.toString();
    }

    /**
     * Evaluates an attribute creation instruction in a template.
     */
    protected Object evaluateAttribute(Object doc, Object instrNode, Object selectContext)
        throws JaxenException, SAXPathException, UpdateException
    {
        Object nameAttr = getAttributeNode(xuNavigator, instrNode, AT_NAME);
        if (nameAttr == null)
        {
            throw new JaxenException("No name attribute for " + instrNode);
        }

        // Interpret variable name as QName
        String qname = xuNavigator.getAttributeStringValue(nameAttr);
        String[] split = splitQName(qname);
        String uri;
        Object namespace = getAttributeNode(xuNavigator, instrNode, AT_NAMESPACE);
        if (namespace != null)
        {
            uri = xuNavigator.getAttributeStringValue(namespace);
        }
        else
        {
            uri = xuNavigator.translateNamespacePrefixToUri(split[0], instrNode);
        }
        List fragment = instantiateTemplate(doc, instrNode, selectContext);
        String value = getNodeListStringValue(fragment);
        // xuNavigator.getElementStringValue(instrNode);
        return docUpdater.createAttribute(doc, uri, qname, value);
    }

    /**
     * Evaluates a processing-instruction creation instruction in a template.
     */
    protected Object evaluateProcessingInstruction(Object doc, Object instrNode)
        throws JaxenException, SAXPathException, UpdateException
    {
        Object nameAttr = getAttributeNode(xuNavigator, instrNode, AT_NAME);
        if (nameAttr == null)
        {
            throw new JaxenException("No name attribute for " + instrNode);
        }
        String target = xuNavigator.getAttributeStringValue(nameAttr);
        String value = xuNavigator.getElementStringValue(instrNode);
        return docUpdater.createProcessingInstruction(doc, target, value);
    }

    /**
     * Evaluates a comment creation instruction in a template.
     */
    protected Object evaluateComment(Object doc, Object instrNode)
        throws JaxenException, SAXPathException, UpdateException
    {
        String value = xuNavigator.getElementStringValue(instrNode);
        return docUpdater.createComment(doc, value);
    }

    /**
     * Evaluates a text creation instruction in a template.
     */
    protected Object evaluateText(Object doc, Object instrNode)
        throws JaxenException, SAXPathException, UpdateException
    {
        String value = xuNavigator.getElementStringValue(instrNode);
        return docUpdater.createText(doc, value);
    }

    /**
     * Evaluates a value-of instruction in a template, which will instantiate a
     * copy of the result of an xpath expression, possibly containing xpath
     * variables.
     * @param selectContext If this value-of is evaluated in the context of
     * another select, such as the select attribute from an append, update, etc.
     * instruction, then this parameter contains this node. If no context is
     * known, it is null.
     */
    protected List evaluateValueOf(Object doc, Object instrNode, Object selectContext)
        throws JaxenException, SAXPathException, UpdateException
    {
        Object selectAttr = getAttributeNode(xuNavigator, instrNode, AT_SELECT);
        if (selectAttr == null)
        {
            throw new JaxenException("No select attribute for " + instrNode);
        }

        String xpathStr = xuNavigator.getAttributeStringValue(selectAttr);
        XPath xpath = parseXPath(xpathStr, instrNode);
        List queryResult = xpath.selectNodes(selectContext == null ? doc : selectContext);
        if (queryResult == null)
        {
            return Collections.EMPTY_LIST;
        }
        List valueOfResult = new ArrayList();
        Iterator toCopy = queryResult.iterator();
        while (toCopy.hasNext())
        {
            Object literal = toCopy.next();
            Object copy = NodeCopier.copyLiteral(xuNavigator, docUpdater, doc, literal, true);
            if (copy != null)
            {
                valueOfResult.add(copy);
            }
        }

        return valueOfResult;
    }

    /**
     * Handles an xupdate 'append', 'update', 'insert-before' and
     * 'insert-after' instruction.
     * @param type OP_APPEND, OP_UPDATE, OP_INSERT_BEFORE, OP_INSERT_AFTER
     */
    protected void creation(Object doc, Object instrNode, int type)
        throws JaxenException, SAXPathException, UpdateException
    {

        // Parse select attribute
        Object selectAttr = getAttributeNode(xuNavigator, instrNode, AT_SELECT);
        if (selectAttr == null)
        {
            throw new JaxenException("No select attribute for " + instrNode);
        }

        String xpathStr = xuNavigator.getAttributeStringValue(selectAttr);
        XPath xpath = parseXPath(xpathStr, instrNode);
        List nodes = xpath.selectNodes(doc);
        if (nodes == null)
            return;

        for (Iterator i = nodes.iterator(); i.hasNext();)
        {
            Object targetNode = i.next();
            singleNodeCreation(doc, instrNode, targetNode, type);
        }
    }

    /**
     * Handles an xupdate 'append', 'update', 'insert-before' and
     * 'insert-after' instruction for a single node.
     * @param type OP_APPEND, OP_UPDATE, OP_INSERT_BEFORE, OP_INSERT_AFTER
     */
    protected void singleNodeCreation(Object doc, Object instrNode, Object targetNode, int type)
        throws JaxenException, SAXPathException, UpdateException
    {
        if (targetNode == null)
        {
            return;
        }

        // For an update, the current child elements must be removed first
        if (type == OP_UPDATE)
        {
            removeChildren(targetNode);
        }

        // Append index
        int childIndex = -1;

        // Parse child attribute for append
        if (type == OP_APPEND)
        {
            Object childAttr = getAttributeNode(xuNavigator, instrNode, AT_CHILD);
            if (childAttr != null)
            {
                String childXPathStr = xuNavigator.getAttributeStringValue(childAttr);
                XPath childXPath = parseXPath(childXPathStr, instrNode);
                childIndex = childXPath.numberValueOf(targetNode).intValue();
            }
        }

        List fragment = instantiateTemplate(doc, instrNode, targetNode);

        // When trying to update an attribute, the whole fragment is taken as a string and
        // the attribute is copied but given the the value of the string.
        if (docNavigator.isAttribute(targetNode))
        {
            switch (type)
            {
                case OP_UPDATE :
                    {
                        String newValue = getNodeListStringValue(fragment);
                        docUpdater.setAttributeValue(targetNode, newValue);
                        break;
                    }
                case OP_INSERT_AFTER :
                case OP_INSERT_BEFORE :
                    addFragmentToTarget(type, fragment, targetNode, childIndex);
                    break;
            }
        }
        else
        {
            addFragmentToTarget(type, fragment, targetNode, childIndex);
        }

    }

    protected void addFragmentToTarget(int type, List fragment, Object targetNode, int childIndex)
        throws UpdateException, JaxenException
    {
        // Fragment now contains instantiated template 
        for (Iterator iter = fragment.iterator(); iter.hasNext();)
        {
            Object node = iter.next();
            switch (type)
            {
                case OP_APPEND :
                case OP_UPDATE :

                    if (docNavigator.isAttribute(node))
                    {
                        docUpdater.setAttribute(targetNode, node);
                    }
                    else
                    {
                        docUpdater.appendChild(targetNode, node, childIndex);
                    }

                    break;
                case OP_INSERT_BEFORE :
                    if (docNavigator.isAttribute(node))
                    {
                        // Inserts new attribute after an existing attribute. Order is not
                        // defined for attributes, so "after" is a bad term, but that's how
                        // it is in xupdate.
                        Object owner = docNavigator.getParentNode(targetNode);
                        docUpdater.setAttribute(owner, node);
                    }
                    else
                    {
                        docUpdater.insertBefore(targetNode, node);
                    }

                    break;
                case OP_INSERT_AFTER :
                    if (docNavigator.isAttribute(node))
                    {
                        // Inserts new attribute after an existing attribute. Order is not
                        // defined for attributes, so "after" is a bad term, but that's how
                        // it is in xupdate.
                        Object owner = docNavigator.getParentNode(targetNode);
                        docUpdater.setAttribute(owner, node);
                    }
                    else
                    {
                        docUpdater.insertAfter(targetNode, node);
                        // Make sure that the insertion remains in the order of
                        // the template
                        targetNode = node;
                    }
                    break;
            }
        }
    }

    /**
     * Removes the children of given element, if it has any children.
     */
    protected void removeChildren(Object element) throws JaxenException, UpdateException
    {
        int type = docNavigator.getNodeType(element);
        if (type != Pattern.DOCUMENT_NODE && type != Pattern.ELEMENT_NODE)
        {
            // Can only remove children of document or element.
            return;
        }

        // To avoid ConcurrentModificationException's, first copy all children to a local list,
        // and then remove all children collected in the local list.
        List childList = new ArrayList();
        for (Iterator children = docNavigator.getChildAxisIterator(element);
            children != null && children.hasNext();
            )
        {
            childList.add(children.next());
        }
        for (Iterator children = childList.iterator(); children.hasNext();)
        {
            docUpdater.remove(children.next());
        }
    }

    /**
     * @param selectContext If this value-of is evaluated in the context of
     * another select, such as the select attribute from an append, update, etc.
     * instruction, then this parameter contains this node. If no context is
     * known, it is null.
     */
    protected List instantiateTemplate(Object doc, Object templateHolder, Object selectContext)
        throws JaxenException, SAXPathException, UpdateException
    {
        List result = new ArrayList();

        Iterator templateNodes = xuNavigator.getChildAxisIterator(templateHolder);
        templateNodes = new NormalizedTextNodeIterator(docUpdater, templateNodes);
        while (templateNodes.hasNext())
        {
            Object templateNode = templateNodes.next();

            if (!xuNavigator.isElement(templateNode))
            {
                // Copy non-elements as literals.
                Object node = NodeCopier.copyLiteral(xuNavigator, docUpdater, doc, templateNode, true);
                if (node != null)
                {
                    result.add(node);
                }
                continue;
            }

            String uri = xuNavigator.getElementNamespaceUri(templateNode);
            String lname = xuNavigator.getElementName(templateNode);
            if (!NS_XUPDATE.equals(uri))
            {
                // All elements of non-xupdate namespace are interpreted 
                // as literals to be copied.
                // However if the element is in the xupdate namespace, it
                // is an instruction, not a literal element.
                Object node = NodeCopier.copyLiteral(xuNavigator, docUpdater, doc, templateNode, true);
                if (node != null)
                {
                    result.add(node);
                }
            }
            else if (XPR_VALUE_OF.equals(lname))
            {
                result.addAll(evaluateValueOf(doc, templateNode, selectContext));
            }
            else if (XPR_ATTRIBUTE.equals(lname))
            {
                result.add(evaluateAttribute(doc, templateNode, selectContext));
            }
            else if (XPR_ELEMENT.equals(lname))
            {
                result.add(evaluateElement(doc, templateNode, selectContext));
            }
            else if (XPR_PROC_INSTR.equals(lname))
            {
                result.add(evaluateProcessingInstruction(doc, templateNode));
            }
            else if (XPR_COMMENT.equals(lname))
            {
                result.add(evaluateComment(doc, templateNode));
            }
            else if (XPR_TEXT.equals(lname))
            {
                result.add(evaluateText(doc, templateNode));
            }
            else
            {
                throw new JaxenException("Unsupported xupdate element: " + templateNode);
            }
        }
        return result;
    }

}
