//Copyright 2002-2003 Erwin Bolwidt. All rights reserved.
//See the file LICENSE.txt in this package for information about licensing.
package org.jaxup.xupdate;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.StringTokenizer;

import org.jaxen.Navigator;
import org.jaxup.InvalidContextException;
import org.jaxup.Updater;

/**
 * Node iterator that concatenates adjacent text nodes and normalizes the
 * resulting text before returning it as a new text node. The new text node is
 * not inserted in the document, so this should be taken into account when using
 * this iterator.
 * 
 * @author Erwin Bolwidt
 */
public class NormalizedTextNodeIterator implements Iterator
{
    private Updater updater;
    private Navigator navigator;
    private Iterator nodeIterator;

    /** 
     * A node that was retrieved from the nodeIterator but didn't need to be
     * returned yet. If peekNode is a text node, it is already normalized, and
     * was stored there by a previous call to hasNext().
     */
    private Object peekNode;

    public NormalizedTextNodeIterator(Updater updater, Iterator nodeIterator)
    {
        this.updater = updater;
        this.navigator = updater.getNavigator();
        this.nodeIterator = nodeIterator;
    }

    public boolean hasNext()
    {
        try
        {
            peekNode = next();
            return true;
        }
        catch (NoSuchElementException e)
        {
            return false;
        }
    }

    public Object next()
    {
        if (peekNode != null)
        {
            Object result = peekNode;
            peekNode = null;
            return result;
        }

        if (!nodeIterator.hasNext())
        {
            throw new NoSuchElementException("next");
        }

        Object nextNode = nodeIterator.next();
        if (!navigator.isText(nextNode))
        {
            return nextNode;
        }

        List textNodes = new ArrayList();
        textNodes.add(nextNode);
        slurpText(textNodes);

        String text = normalizeText(textNodes);

        if (text.length() == 0)
        {
            if (peekNode != null)
            {
                Object result = peekNode;
                peekNode = null;
                return result;
            }
            else
            {
                throw new NoSuchElementException("empty text");
            }
        }

        try
        {
            return updater.createText(nextNode, text);
        }
        catch (InvalidContextException e)
        {
            throw new NoSuchElementException("Invalid context: " + e.getMessage());
        }
    }

    private String normalizeText(String text)
    {
        StringBuffer result = new StringBuffer();
        StringTokenizer tokens = new StringTokenizer(text, " \t\n\r");
        int count = tokens.countTokens();
        for (int i = 0; tokens.hasMoreTokens(); i++)
        {
            String token = tokens.nextToken();
            result.append(token);
            if (i + 1 != count)
            {
                result.append(' ');
            }
        }

        return result.toString();
    }

    private String normalizeText(List textNodes)
    {
        StringBuffer result = new StringBuffer();
        for (int i = 0, l = textNodes.size(); i < l; i++)
        {
            String value = navigator.getTextStringValue(textNodes.get(i));
            result.append(value);
        }
        return normalizeText(result.toString());
    }

    private void slurpText(List textNodes)
    {
        while (nodeIterator.hasNext())
        {
            Object nextNode = nodeIterator.next();
            if (!navigator.isText(nextNode))
            {
                this.peekNode = nextNode;
                break;
            }
            textNodes.add(nextNode);
        }
    }

    public void remove()
    {
        throw new UnsupportedOperationException("remove operation not supported on this iterator");
    }
}
