package org.jaxup.xupdate;

import java.util.Iterator;

import org.jaxen.JaxenException;
import org.jaxen.Navigator;
import org.jaxup.UpdateException;
import org.jaxup.Updater;

public class NodeCopier
{
    /**
     * Namespace prefix for "xml:" namespace.
     */
    private final static String PFX_XML = "xml";

    /**
     * The complete attribute name for declaring the "xml:" namespace.
     */
    private final static String AT_XMLNS_XML = "xmlns:xml";

    private NodeCopier()
    {
    }

    /**
     * Copies a node in the xupdate document document as an unplaced node for the source document.
     * This ignores comments, XUpdate namespace declarations, etc.
     */
    public static Object copyLiteral(
        Navigator from,
        Updater to,
        Object toDoc,
        Object original,
        boolean template)
        throws JaxenException, UpdateException
    {
        if (from.isText(original))
        {
            return copyText(from, to, toDoc, original);
        }
        else if (from.isComment(original))
        {
            // When copying templates, this is a literal comment in the xupdate file.
            // Comments should be comments; i.e. ignored by a processor. Even though it could
            // be copied. Comments can be created in the data document with
            // xu:comment.
            if (template)
            {
                return null;
            }
            else
            {
                return copyComment(from, to, toDoc, original);
            }
        }
        else if (from.isAttribute(original))
        {
            return copyAttribute(from, to, toDoc, original);
        }
        else if (from.isProcessingInstruction(original))
        {
            // Copy processing instruction
            String target = from.getProcessingInstructionTarget(original);
            String data = from.getProcessingInstructionData(original);
            return to.createProcessingInstruction(toDoc, target, data);
        }
        else if (from.isNamespace(original))
        {
            // Copy namespace
            String prefix = from.getNamespacePrefix(original);
            String uri = from.getNamespaceStringValue(original);
            // Never copy declaration of namespace with xml prefix
            if (prefix.equals(PFX_XML))
            {
                return null;
            }
            return to.createNamespace(toDoc, prefix, uri);
        }
        else if (from.isElement(original))
        {
            return copyElement(from, to, toDoc, original, template);
        }
        else
        {
            // Convert to text node 
            if (original == null)
            {
                return null;
            }
            String stringLit = original.toString();
            if (stringLit.length() == 0)
            {
                return null;
            }
            return to.createText(toDoc, stringLit);
        }
    }

    public static Object copyText(Navigator from, Updater to, Object toDoc, Object original)
        throws UpdateException
    {
        String text = from.getTextStringValue(original);
        return to.createText(toDoc, text);
    }

    public static Object copyComment(Navigator from, Updater to, Object toDoc, Object original)
        throws UpdateException
    {
        String text = from.getCommentStringValue(original);
        return to.createComment(toDoc, text);
    }

    /**
     * Copies an attribute completely.
     */
    public static Object copyAttribute(Navigator from, Updater to, Object toDoc, Object original)
        throws UpdateException
    {
        // Copy attribute
        String uri = from.getAttributeNamespaceUri(original);
        String qname = from.getAttributeQName(original);

        // Never copy xmlns:xml declaration.
        if (qname.equals(AT_XMLNS_XML))
        {
            return null;
        }
        String value = from.getAttributeStringValue(original);
        return to.createAttribute(toDoc, uri, qname, value);
    }

    /**
     * Copies an attribute but assigns it a new value.
     */
    public static Object copyAttribute(
        Navigator from,
        Updater to,
        Object toDoc,
        Object original,
        String newValue)
        throws UpdateException
    {
        // Copy attribute
        String uri = from.getAttributeNamespaceUri(original);
        String qname = from.getAttributeQName(original);

        // Never copy xmlns:xml declaration.
        if (qname.equals(AT_XMLNS_XML))
        {
            return null;
        }
        return to.createAttribute(toDoc, uri, qname, newValue);
    }

    public static Object copyElement(
        Navigator from,
        Updater to,
        Object toDoc,
        Object original,
        boolean template)
        throws JaxenException, UpdateException
    {
        // Copy element
        String uri = from.getElementNamespaceUri(original);
        String qname = from.getElementQName(original);
        Object element = to.createElement(toDoc, uri, qname);

        // Copy attributes
        Iterator attributes = from.getAttributeAxisIterator(original);
        while (attributes.hasNext())
        {
            Object attrLiteral = attributes.next();
            Object copied = copyLiteral(from, to, toDoc, attrLiteral, template);
            if (copied != null)
            {
                to.setAttribute(element, copied);
            }
        }

        // Copy namespaces
        Iterator namespaces = from.getNamespaceAxisIterator(original);
        while (namespaces.hasNext())
        {
            Object nsLiteral = namespaces.next();
            String nsURI = from.getNamespaceStringValue(nsLiteral);
            // Ignore XUPDATE namespace when copying template
            if (template && XUpdate.NS_XUPDATE.equals(nsURI))
            {
                continue;
            }
            Object copied = copyLiteral(from, to, toDoc, nsLiteral, template);
            if (copied != null)
            {
                to.setNamespace(element, copied);
            }
        }

        // Copy children
        Iterator children = from.getChildAxisIterator(original);
        while (children.hasNext())
        {
            Object childLiteral = children.next();
            Object copied = copyLiteral(from, to, toDoc, childLiteral, template);
            if (copied != null)
            {
                to.appendChild(element, copied, -1);
            }
        }
        return element;
    }
}
