// Copyright 2001-2003 Erwin Bolwidt. All rights reserved.
// See the file LICENSE.txt in this package for information about licensing.
package org.jaxup.dom4j;

import java.util.Iterator;
import java.util.List;

import org.dom4j.Attribute;
import org.dom4j.Branch;
import org.dom4j.Document;
import org.dom4j.DocumentFactory;
import org.dom4j.Element;
import org.dom4j.Namespace;
import org.dom4j.Node;
import org.dom4j.QName;
import org.dom4j.Text;
import org.jaxen.Navigator;
import org.jaxen.dom4j.DocumentNavigator;
import org.jaxup.InvalidContextException;
import org.jaxup.UpdateException;
import org.jaxup.Updater;

/** 
 * XML document creating and updating methods for the DOM4J document object model.
 *
 * @author Erwin Bolwidt
 */
public class Dom4JDocumentUpdater implements Updater
{
    private DocumentFactory factory = DocumentFactory.getInstance();

    private final static Navigator modifiedNavigator = new DocumentNavigator()
    {
        public Iterator getChildAxisIterator(Object contextNode)
        {
            Iterator original = super.getChildAxisIterator(contextNode);
            if (original == null)
            {
                return null;
            }
            return new TypeFilteredIterator(this, original, TypeFilteredIterator.TYPE_NAMESPACE);
        }
    };

    public Object createDocument() throws UpdateException
    {
        return factory.createDocument();
    }

    public static Navigator getModifiedNavigator()
    {
        return modifiedNavigator;
    }

    public Navigator getNavigator()
    {
        return modifiedNavigator;
    }

    public Object createComment(Object contextNode, String comment) throws InvalidContextException
    {
        if (!(contextNode instanceof Node))
        {
            throw new InvalidContextException("Context is not a DOM4J node");
        }
        return factory.createComment(comment);
    }

    public Object createText(Object contextNode, String text) throws InvalidContextException
    {
        if (!(contextNode instanceof Node))
        {
            throw new InvalidContextException("Context is not a DOM4J node");
        }
        return factory.createText(text);
    }

    public Object createElement(Object contextNode, String uri, String qname) throws InvalidContextException
    {
        if (!(contextNode instanceof Node))
        {
            throw new InvalidContextException("Context is not a DOM4J node");
        }
        return factory.createElement(qname, uri);
    }

    public Object createNamespace(Object contextNode, String prefix, String uri)
        throws InvalidContextException
    {
        if (!(contextNode instanceof Node))
        {
            throw new InvalidContextException("Context is not a DOM4J node");
        }
        return factory.createNamespace(prefix, uri);
    }

    public Object createAttribute(Object contextNode, String uri, String qname, String value)
        throws InvalidContextException
    {
        if (!(contextNode instanceof Node))
        {
            throw new InvalidContextException("Context is not a DOM4J node");
        }
        QName domQName = QName.get(qname, uri);
        return factory.createAttribute(null, domQName, value);
    }

    public Object createProcessingInstruction(Object contextNode, String target, String data)
        throws InvalidContextException
    {
        if (!(contextNode instanceof Node))
        {
            throw new InvalidContextException("Context is not a DOM4J node");
        }
        return factory.createProcessingInstruction(target, data);
    }

    public void insertBefore(Object refNode, Object node) throws UpdateException
    {
        if (!(refNode instanceof Node) || !(node instanceof Node))
        {
            throw new InvalidContextException("Node is not a DOM4J node");
        }
        Node domRefNode = (Node) refNode;
        Node domNode = (Node) node;
        Branch parent = domRefNode.getParent();
        if (parent == null)
        {
            parent = domRefNode.getDocument();
        }
        if (parent == null)
        {
            throw new InvalidContextException("refNode has no parent");
        }

        List siblings = parent.content();
        int refIndex = siblings.indexOf(refNode);
        if (refIndex == -1)
        {
            throw new UpdateException("refNode is not a sibling of node");
        }
        siblings.add(refIndex, domNode);
    }

    public void insertAfter(Object refNode, Object node) throws UpdateException
    {
        if (!(refNode instanceof Node) || !(node instanceof Node))
        {
            throw new InvalidContextException("Node is not a DOM4J node");
        }
        Node domRefNode = (Node) refNode;
        Node domNode = (Node) node;
        Branch parent = domRefNode.getParent();
        if (parent == null)
        {
            parent = domRefNode.getDocument();
        }
        if (parent == null)
        {
            throw new InvalidContextException("refNode has no parent: " + refNode);
        }

        List siblings = parent.content();
        int refIndex = siblings.indexOf(refNode);
        if (refIndex == -1)
        {
            throw new UpdateException("refNode is not a sibling of node");
        }

        siblings.add(refIndex + 1, domNode);
    }

    public void appendChild(Object element, Object child, int position) throws UpdateException
    {
        if (!(element instanceof Node) || !(child instanceof Node))
        {
            throw new InvalidContextException("Node is not a DOM4J node");
        }
        Node domChild = (Node) child;
        Branch parent = (Branch) element;

        if (parent instanceof Document && domChild instanceof Text)
        {
            Text t = (Text) domChild;
            // Can't add text to a Document directly, but whitespace can be
            // silently ignored.
            if (t.getText().trim().length() == 0)
            {
                return;
            }
        }

        if (position == -1)
        {
            parent.add(domChild);
            return;
        }

        List siblings = parent.content();
        siblings.add(position, domChild);
    }

    public void remove(Object node) throws UpdateException
    {
        if (!(node instanceof Node))
        {
            throw new InvalidContextException("Node is not a DOM4J node");
        }
        Node domNode = (Node) node;
        Branch parent = domNode.getParent();
        if (parent == null)
        {
            parent = domNode.getDocument();
        }
        if (parent == null)
        {
            throw new InvalidContextException("refNode has no parent");
        }

        parent.remove(domNode);
    }

    public void setAttribute(Object element, Object attribute) throws UpdateException
    {
        if (!(element instanceof Element) || !(attribute instanceof Attribute))
        {
            throw new InvalidContextException("Invalid DOM4J node types");
        }
        Attribute domAttribute = (Attribute) attribute;
        Element domElement = (Element) element;
        domElement.add(domAttribute);
    }

    public void setNamespace(Object element, Object namespace) throws UpdateException
    {
        if (!(element instanceof Element) || !(namespace instanceof Namespace))
        {
            throw new InvalidContextException("Invalid DOM4J node types");
        }
        Namespace domNamespace = (Namespace) namespace;
        Element domElement = (Element) element;
        domElement.add(domNamespace);
    }

    public void setAttributeValue(Object attribute, String value) throws UpdateException
    {
        if (!(attribute instanceof Attribute))
        {
            throw new InvalidContextException("Invalid DOM4J node type");
        }
        Attribute domAttribute = (Attribute) attribute;
        domAttribute.setValue(value);
    }
}
