// Copyright 2001-2003 Erwin Bolwidt. All rights reserved.
// See the file LICENSE.txt in this package for information about licensing.
package org.jaxup;

import org.jaxen.Navigator;

/** 
 * XML document updating and creating functionality, with an interface
 * based on the XPath data model. The interface is intended to be  sufficiently
 * high-level, that implementations for different concrete  document object
 * models can be implemented.
 *
 * @author Erwin Bolwidt
 */
public interface Updater
{
    /** 
     * Creates an empty XML document.
     * @throws UpdateException A document could not be created.
     */
    Object createDocument() throws UpdateException;

    /**
     * Creates a comment node with the given text as the comment.
     * 
     * @param contextNode A node in the document in which the comment will be
     * used.
     * @param comment The text of the comment to create.
     * @return A comment node that can be used in the same document as the
     * contextNode.
     * @throws InvalidContextException The contextNode parameter is not of a
     * class or interface that the document model recognizes for a node.
     */
    Object createComment(Object contextNode, String comment) throws InvalidContextException;

    /**
     * Creates a text node of the given text.
     * 
     * @param contextNode A node in the document in which the text will be used.
     * @param comment The text to create.
     * @return A text node that can be used in the same document as the
     * contextNode.
     * @throws InvalidContextException The contextNode parameter is not of a
     * class or interface that the document model recognizes for a node.
     */
    Object createText(Object contextNode, String text) throws InvalidContextException;

    /**
     * Creates an element node. Note that the namespace declaration attributes
     * that would bind the prefix of the qname to the namespace uri are not
     * automatically created. If they are not created separately, the document
     * cannot be serialized in to a valid xml document that has the same meaning
     * as the document tree that is in memory.
     * 
     * @param contextNode A node in the document in which the element will be
     * used.
     * @param uri The namespace uri of the new element, or <code>null</code>
     * @param qname The qualified name (prefix, semicolon, localname) of the new
     * element.
     * @return An element node that can be used in the same document as the
     * contextNode.
     * @throws InvalidContextException The contextNode parameter is not of a
     * class or interface that the document model recognizes for a node.
     */
    Object createElement(Object contextNode, String uri, String qname) throws InvalidContextException;

    /**
     * Creates a namespace node. A namespace node is a virtual node that is
     * used in XPath and in XML Infoset. It has no equivalent in a serialized
     * XML document, but is instead derived from the namespace declaration
     * attributes in the serialized XML document. Care needs to be taken to
     * create the namespace declaration attributes when creating namespace
     * nodes, otherwise the serialized XML document may not be a valid
     * representation of the in-memory document.
     * 
     * @param contextNode A node in the document in which the namespace will be
     * used.
     * @param prefix The namespace prefix that will be bound to the namespace
     * uri.
     * @param uri The namespace uri of the new namespace node.
     * @return A namespace node that can be used in the same document as the
     * contextNode.
     * @throws InvalidContextException The contextNode parameter is not of a
     * class or interface that the document model recognizes for a node.
     */
    Object createNamespace(Object contextNode, String prefix, String uri) throws InvalidContextException;

    /**
     * Creates an attribute node. Note that the namespace declaration attributes
     * that would bind the prefix of the qname to the namespace uri are not
     * automatically created. If they are not created separately, the document
     * cannot be serialized in to a valid xml document that has the same meaning
     * as the document tree that is in memory.
     * 
     * @param contextNode A node in the document in which the attribute will be
     * used.
     * @param uri The namespace uri of the new attribute, or <code>null</code>
     * to indicate that the attribute will not be in a namespace.
     * @param qname The qualified name (prefix, semicolon, localname) of the new
     * attribute.
     * @param value The value of the new attribute.
     * @return An attribute node that can be used in the same document as the
     * contextNode.
     * @throws InvalidContextException The contextNode parameter is not of a
     * class or interface that the document model recognizes for a node.
     */
    Object createAttribute(Object contextNode, String uri, String qname, String value)
        throws InvalidContextException;

    /**
     * Creates a processing-instruction node of the given target and data.
     * Format: &lt;?<i>target</i> <i>data</i>?&gt;
     * @param contextNode A node in the document in which the text will be used.
     * @param target The target for the processing instruction to create.
     * @param data The data for the processing instruction to create. The format
     * of the data depends on the syntax defined by the target; many targets
     * define a syntax simular to the attribute list of an element node.
     * @return A processing-instruction node that can be used in the same
     * document as the contextNode.
     * @throws InvalidContextException The contextNode parameter is not of a
     * class or interface that the document model recognizes for a node.
     */
    Object createProcessingInstruction(Object contextNode, String target, String data)
        throws InvalidContextException;

    /**
     * Inserts a node into the document before the refNode, as a sibling of
     * the ref Node.
     * 
     * @param refNode The reference node, before which the new node will be
     * inserted as a sibling.
     * @param node The node that will be inserted.
     * @throws InvalidNodeException A parameter that represents a node is not
     * of a class or interface that the document model recognizes for the
     * expected type of node.
     * @throws UpdateException Another exception occurred during update.
     */
    void insertBefore(Object refNode, Object node) throws UpdateException;

    /**
     * Inserts a node into the document after the refNode, as a sibling of the
     * ref Node.
     * 
     * @param refNode The reference node, after which the new node will be
     * inserted as a sibling.
     * @param node The node that will be inserted.
     * @throws InvalidNodeException A parameter that represents a node is not
     * of a class or interface that the document model recognizes for the
     * expected type of node.
     * @throws UpdateException Another exception occurred during update.
     */
    void insertAfter(Object refNode, Object node) throws UpdateException;

    /**
     * Inserts a node into the document by appending it to the children of
     * the parent, or if the position parameter isn't -1, by inserting it at
     * that position in the list of children of the parent.
     * 
     * @param parent The parent node, which must be either a document
     * or element node.
     * @param child The node that will be appended / inserted.
     * @param position The position in the child list where the child node will
     * be inserted, or -1 to append the new child to the list of children.
     * @throws InvalidNodeException A parameter that represents a node is not
     * of a class or interface that the document model recognizes for the
     * expected type of node.
     * @throws UpdateException Another exception occurred during update.
     */
    void appendChild(Object parent, Object child, int position) throws UpdateException;

    /**
     * Removes a node from the document that it is in.
     * 
     * @param node The node to remove.
     * @throws InvalidNodeException A parameter that represents a node is not
     * of a class or interface that the document model recognizes for the
     * expected type of node.
     * @throws UpdateException Another exception occurred during update.
     */
    void remove(Object node) throws UpdateException;

    /**
     * Sets an attribute on an element node, replacing an existing
     * attribute with the same name.
     * 
     * @param element The element node to set the attribute on.
     * @param attribute The attribute node to set on the element.
     * @throws InvalidNodeException A parameter that represents a node is not
     * of a class or interface that the document model recognizes for the
     * expected type of node.
     * @throws UpdateException Another exception occurred during update.
     */
    void setAttribute(Object element, Object attribute) throws UpdateException;

    /**
     * Changes the value of an existing attribute.
     */
    void setAttributeValue(Object attribute, String value) throws UpdateException;

    /**
     * Binds a namespace node to an element node.
     * 
     * @param element The element node to bind the namespace node to.
     * @param namespace The namespace node to bind to the element node.
     * @throws InvalidNodeException A parameter that represents a node is not
     * of a class or interface that the document model recognizes for the
     * expected type of node.
     * @throws UpdateException Another exception occurred during update.
     */
    void setNamespace(Object element, Object namespace) throws UpdateException;

    /** 
     * Returns a jaxen Navigator instance that can be used to navigate the
     * document that is being updated by this Updater.
     * @return A Navigator instance for the same document object model as this
     * updater.
     * @see Navigator
     */
    Navigator getNavigator();
}
