/*
 * Javolution - Java(TM) Solution for Real-Time and Embedded Systems
 * Copyright (C) 2005 - Javolution (http://javolution.org/)
 * All rights reserved.
 * 
 * Permission to use, copy, modify, and distribute this software is
 * freely granted, provided that this notice is preserved.
 */
package _templates.javolution.lang;

import _templates.javolution.context.LogContext;
import _templates.javolution.context.SecurityContext;
import _templates.javolution.text.TextFormat;
import _templates.javolution.util.FastTable;
import _templates.javolution.xml.XMLBinding;
import _templates.javolution.xml.XMLFormat;
import _templates.javolution.xml.XMLFormat.InputElement;
import _templates.javolution.xml.XMLFormat.OutputElement;
import _templates.javolution.xml.XMLObjectReader;
import _templates.javolution.xml.stream.XMLStreamException;
import java.io.InputStream;
import java.util.Enumeration;
import java.util.Properties;

/**
 *  <p> This class facilitates separation of concerns between the configuration
 *      logic and the application code.</p>

 *  <p> Does your class need to know or has to assume that the configuration is
 *      coming from system properties ??</p>
 *
 *  <p> The response is obviously NO!</p>
 *
 *  <p> Let's compare the following examples:<pre>
 *      class Document {
 *          private static final Font DEFAULT_FONT
 *              = Font.decode(System.getProperty("DEFAULT_FONT") != null ?
 *                  System.getProperty("DEFAULT_FONT") : "Arial-BOLD-18");
 *          ...
 *      }</pre>
 *      With the following (using this class):<pre>
 *      class Document {
 *          public static final Configurable<Font> DEFAULT_FONT
 *              = new Configurable<Font>(new Font("Arial", Font.BOLD, 18)) {};
 *          ...
 *      }</pre>
 *      Not only the second example is cleaner, but the actual configuration
 *      data can come from anywhere, for example from the OSGI Configuration
 *      Admin package (<code>org.osgi.service.cm</code>).
 *      Low level code does not need to know.</p>
 *
 * <p>  Configurable instances have the same textual representation as their
 *      current values. For example:<pre>
 *       public static final Configurable<String> AIRPORT_TABLE
 *            = new Configurable<String>("Airports") {};
 *       ...
 *       String sql = "SELECT * FROM " + AIRPORT_TABLE
 *           // AIRPORT_TABLE.get() is superfluous
 *           + " WHERE State = '" + state  + "'";</pre>
 *      </p>
 *
 *  <p> Unlike system properties (or any static mapping), configuration
 *      parameters may not be known until run-time or may change dynamically.
 *      They may depend upon the current run-time platform,
 *      the number of cpus, etc. Configuration parameters may also be retrieved
 *      from external resources such as databases, XML files,
 *      external servers, system properties, etc.<pre>
 *      public abstract class FastComparator<T> implements Comparator<T>, Serializable  {
 *          public static final Configurable<Boolean> REHASH_SYSTEM_HASHCODE
 *              = new Configurable<Boolean>(isPoorSystemHash()) {}; // Test system hashcode.
 *      ...
 *      public abstract class ConcurrentContext extends Context {
 *          public static final Configurable<Integer> MAXIMUM_CONCURRENCY
 *              = new Configurable<Integer>(Runtime.getRuntime().availableProcessors() - 1) {};
 *                  // No algorithm parallelization on single-processor machines.
 *     ...
 *     public abstract class XMLInputFactory {
 *          public static final Configurable<Class<? extends XMLInputFactory>> DEFAULT
 *              = new Configurable<Class<? extends XMLInputFactory>>(XMLInputFactory.Default.class) {};
 *                  // Default class implementation is a private class.
 *     ...
 *     </pre></p>
 *
 *  <p> Dynamic {@link #configure configuration} is allowed/disallowed based
 *      upon the current {SecurityContext}. Configurables are automatically
 *      {@link Configurable#notifyChange notified} of
 *      any changes in their configuration values.</p>
 *
 * <p>  Unlike system properties, configurable can be
 *      used in applets or unsigned webstart applications.</p>
 *
 *  <p> Here is an example of configuration of a web application from 
 *      a property file:[code]
 *      public class Configuration implements ServletContextListener {
 *          public void contextInitialized(ServletContextEvent sce) {
 *              try {
 *                  ServletContext ctx = sce.getServletContext();
 *               
 *                  // Loads properties.
 *                  Properties properties = new Properties();
 *                  properties.load(ctx.getResourceAsStream("WEB-INF/config/configuration.properties"));
 *               
 *                  // Reads properties superceeding default values.
 *                  Configurable.read(properties);
 *                  
 *              } catch (Exception ex) {
 *                  LogContext.error(ex);
 *              }
 *          }
 *      }[/code]
 *      This listener is registered in the <code>web.xml</code> file:[code]
 *      <web-app>
 *          <listener>
 *              <listener-class>mypackage.Configuration</listener-class>
 *           </listener>
 *      </web-app>[/code]
 *      The property file contains the full names of the configurable static
 *      fields and the textual representation of their new values:[code]
 *      # File configuration.properties
 *      javolution.util.FastComparator#REHASH_SYSTEM_HASHCODE = true
 *      javolution.context.ConcurrentContext#MAXIMUM_CONCURRENCY = 0
 *      javolution.xml.stream.XMLInputFactory#DEFAULT = com.foo.bar.XMLInputFactoryImpl
 *      [/code]</p>
 *      
 *  <p> Here is an example of configuration from a xml file:[code]
 *      FileInputStream xml = new FileInputStream("configuration.xml");
 *      Configurable.read(xml);[/code]
 *      and the configuration file:[code]
 *      <Configuration>
 *          <Configurable name="javolution.util.FastComparator#REHASH_SYSTEM_HASHCODE" />
 *          <java.lang.Boolean value="true" />
 *          <Configurable name="javolution.context.ConcurrentContext#MAXIMUM_CONCURRENCY" />
 *          <java.lang.Integer value="0" />
 *          <Configurable name="javolution.xml.stream.XMLInputFactory#DEFAULT" />
 *          <java.lang.Class name="com.foo.bar.XMLInputFactoryImpl" />
 *      </Configuration>[/code]</p>
 *       
 * @author  <a href="mailto:jean-marie@dautelle.com">Jean-Marie Dautelle</a>
 * @version 5.4, November 2, 2009
 */
public abstract class Configurable/*<T>*/ {

    /**
     * Holds the default xml representation of a configurable. Because
     * configurable instances are unique. Deserialisation of a configurable
     * returns this unique instance (typically public static).
     * The default xml representation is very simple it holds the configurable
     * full name. For example:[code]
     *   <javolution.lang.Configurable name="javolution.context.ConcurrentContext#MAXIMUM_CONCURRENCY" />
     * [/code]
     * XML mapping of configurables and their new values can be read directly
     * using the {@link #read(InputStream)} method.
     */
    protected static final XMLFormat XML = new XMLFormat(Configurable.class) {

        public Object newInstance(Class cls, InputElement xml) throws XMLStreamException {
            return Configurable.getInstance(xml.getAttribute("name", ""));
        }

        public void write(Object c, OutputElement xml) throws XMLStreamException {
            Configurable cfg = (Configurable) c;
            xml.setAttribute("name", cfg.getName());
        }

        public void read(InputElement xml, Object c) throws XMLStreamException {
            // Nothing else to read.
        }
    };

    /**
     * Holds the current value (never null).
     */
    private volatile Object/*{T}*/ _current;

    /**
     * Holds the default value (never null).
     */
    private final Object/*{T}*/ _default;

    /**
     * Creates a new configurable having the specified default value.
     *
     * @param defaultValue the default value.
     * @throws IllegalArgumentException if <code>defaultValue</code> is
     *         <code>null</code>.
     */
    public Configurable(Object/*{T}*/ defaultValue) {
        if (defaultValue == null) {
            throw new IllegalArgumentException("Default value cannot be null");
        }
        _default = defaultValue;
        _current = defaultValue;
    }

    /**
     * Returns the current value for this configurable.
     * 
     * @return the current value (always different from <code>null</code>).
     */
    public Object/*{T}*/ get() {
        return _current;
    }

    /**
     * Returns the default value for this configurable.
     * 
     * @return the default value (always different from <code>null</code>).
     */
    public Object/*{T}*/ getDefault() {
        return _default;
    }

    /**
     * Notifies this configurable that its runtime value is going to be changed.
     * The default implementation does nothing.
     *
     * @param oldValue the previous value.
     * @param newValue the new value.
     * @throws UnsupportedOperationException if dynamic reconfiguration of 
     *         this configurable is not allowed (regardless of the security
     *         context).
     */
    protected void notifyChange(Object/*{T}*/ oldValue, Object/*{T}*/ newValue) throws _templates.java.lang.UnsupportedOperationException {
    }

    /**
     * @deprecated To be replaced by {@link #notifyChange(java.lang.Object, java.lang.Object)}
     */
    protected void notifyChange() {
    }

    /**
     * Returns the string representation of the value of this configurable.
     * 
     * @return <code>String.valueOf(this.get())</code>
     */
    public String toString() {
        return String.valueOf(_current);
    }

    /**
     * Returns the field name of this configurable (for example <code>
     * "javolution.context.ConcurrentContext#MAXIMUM_CONCURRENCY"</code>)
     * for {@link javolution.context.ConcurrentContext#MAXIMUM_CONCURRENCY}.
     *
     *  @return this configurable name or <code>null</code> if the name
     *          of this configurable cannot be determined (e.g. J2ME).
     */
    public String getName() {
        /** @JVM-1.4+@
        Class cfgClass = this.getClass();
        Class cls = cfgClass.getEnclosingClass();
        java.lang.reflect.Field[] fields = cls.getDeclaredFields();
        try {
            for (int i = 0; i < fields.length; i++) {
                java.lang.reflect.Field field = fields[i];
                if (field.get(null) == this) // Found it.
                    return cls.getName() + '#' + field.getName();
            }
        } catch (IllegalAccessException ex) {
            LogContext.error(ex);
        }
        /**/
        return null;
    }

    /**
     * Returns the configurable instance having the specified name.
     * For example:[code]
     *     Configurable cfg = Configurable.getInstance("javolution.context.ConcurrentContext#MAXIMUM_CONCURRENCY")
     * [/code] returns {@link javolution.context.ConcurrentContext#MAXIMUM_CONCURRENCY}.
     *
     * @param  name the name of the configurable to retrieve.
     * @return the corresponding configurable or <code>null</code> if it
     *         cannot be found (e.g. J2ME platform).
     */
    public static Configurable getInstance(String name) {
        /** @JVM-1.4+@
        int sep = name.lastIndexOf('#');
        if (sep < 0)
        return null;
        String className = name.substring(0, sep);
        String fieldName = name.substring(sep + 1);
        Class cls = Reflection.getClass(className);
        if (cls == null)
        return null;
        try {
        Configurable cfg = (Configurable) cls.getDeclaredField(fieldName).get(null);
        return cfg;
        } catch (Exception ex) {
        LogContext.error(ex);
        }
        /**/
        return null;
    }

    /**
     * Sets the run-time value of the specified configurable. If the
     * configurable value is different from the previous one, then
     * {@link #notifyChange} is called. This method
     * raises a <code>SecurityException</code> if the specified
     * configurable cannot be {@link SecurityContext#isConfigurable
     * reconfigured}.
     *
     * @param  cfg the configurable being configured.
     * @param  newValue the new run-time value.
     * @throws IllegalArgumentException if <code>value</code> is
     *         <code>null</code>.
     * @throws SecurityException if the specified configurable cannot
     *         be modified.
     */
    public static /*<T>*/ void configure(Configurable/*<T>*/ cfg,
            Object/*{T}*/ newValue) throws SecurityException {
        if (newValue == null) {
            throw new IllegalArgumentException("Default value cannot be null");
        }
        SecurityContext policy = (SecurityContext) SecurityContext.getCurrent();
        if (!policy.isConfigurable(cfg)) {
            throw new SecurityException(
                    "Configuration disallowed by SecurityContext");
        }
        Object/*{T}*/ oldValue = cfg._current;
        if (!newValue.equals(oldValue)) {
            LogContext.info("Configurable ", cfg.getName(), " set to ", newValue);
            cfg.notifyChange(oldValue, newValue);
            cfg._current = newValue;
            cfg.notifyChange();
        }
    }

    /**
     * Convenience method to read the specified properties and reconfigure
     * accordingly.[code]
     *     // Load configurables from system properties.
     *     Configurable.read(System.getProperties());[/code]
     * Configurables are identified by their field names. The textual
     * representation of their value is defined by
     * {@link _templates.javolution.text.TextFormat#getInstance(Class)}
     * text format}. For example:[code]
     *      javolution.util.FastComparator#REHASH_SYSTEM_HASHCODE = true
     *      javolution.context.ConcurrentContext#MAXIMUM_CONCURRENCY = 0
     *      javolution.xml.stream.XMLInputFactory#DEFAULT = com.foo.bar.XMLInputFactoryImpl
     * [/code]
     * Conversion of <code>String</code> values to actual object is
     * performed using {@link _templates.javolution.text.TextFormat#getInstance(Class)}.
     *
     * <p><b>Note:</b> OSGI based framework should ensure that class loaders
     *    of configurable instances are known to the {@link Reflection} utility
     *    class.</p>
     *
     * @param properties the properties.
     * @throws _templates.java.lang.UnsupportedOperationException on J2ME
     */
    public static void read(Properties properties) {
        Enumeration e = properties.keys();
        while (e.hasMoreElements()) {
            String name = (String) e.nextElement();
            String textValue = properties.getProperty(name);
            Configurable cfg = Configurable.getInstance(name);
            if (cfg == null) 
                continue;
            // Use the default value to retrieve the configurable type
            // and the associated textual format.
            Class type = cfg.getDefault().getClass();
            TextFormat format = TextFormat.getInstance(type);
            if (format == null) {
                LogContext.error("Cannot find TextFormat for instance of " + type);
                continue;
            }
            Object newValue = format.parse(_templates.javolution.Javolution.j2meToCharSeq(textValue));
            Configurable.configure(cfg, newValue);
        }
    }

    /**
     * Convenience method to read configurable values from the specified 
     * xml stream. This method uses
     * <a href="http://javolution.org/target/site/apidocs/javolution/xml/package-summary.html">
     * Javolution XML</a> facility to perform the deserialization.
     * Here is an example of XML configuration file.[code]
     * <Configuration>
     *      <Configurable name="javolution.util.FastComparator#REHASH_SYSTEM_HASHCODE" />
     *      <java.lang.Boolean value="true" />
     *      <Configurable name="javolution.context.ConcurrentContext#MAXIMUM_CONCURRENCY" />
     *      <java.lang.Integer value="0" />
     *      <Configurable name="javolution.xml.stream.XMLInputFactory#DEFAULT" />
     *      <java.lang.Class name="com.foo.bar.XMLInputFactoryImpl" />
     * </Configuration>[/code]
     * The default XML format for predefined types is held by
     * {@link _templates.javolution.xml.XMLBinding}.
     *
     * <p><b>Note:</b> OSGI based framework should ensure that class loaders
     *    of configurable instances are known to the {@link Reflection} utility.
     *    </p>
     *
     * @param inputStream the input stream holding the xml configuration.
     */
    public static void read(InputStream inputStream) {
        try {
            XMLObjectReader reader = XMLObjectReader.newInstance(inputStream);
            XMLBinding binding = new XMLBinding();
            binding.setAlias(Configurable.class, "Configurable");
            reader.setBinding(binding);
            FastTable configuration = (FastTable) reader.read("Configuration", FastTable.class);
            for (int i = 0; i < configuration.size();) {
                Configurable cfg = (Configurable) configuration.get(i++);
                Object newValue = configuration.get(i++);
                Configurable.configure(cfg, newValue);
            }
        } catch (Exception ex) {
            LogContext.error(ex);
        }
    }
}
