package com.jaspersoft.jasperserver.war.cascade.handlers.converters;

import com.jaspersoft.jasperserver.api.common.util.rd.DateRangeFactory;
import net.sf.jasperreports.types.date.FixedDate;
import net.sf.jasperreports.types.date.RelativeDateRange;
import com.jaspersoft.jasperserver.war.util.CalendarFormatProvider;
import net.sf.jasperreports.types.date.DateRange;
import org.junit.Test;
import org.unitils.UnitilsJUnit4;
import org.unitils.mock.Mock;
import org.unitils.mock.MockUnitils;
import org.unitils.util.ReflectionUtils;

import java.lang.reflect.Field;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.GregorianCalendar;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;

public class DateRangeConverterTest extends UnitilsJUnit4 {

    @Test
    public void ensureRelativeDateRangeReturnedForRangeExpressions() throws Exception {
        DateRangeDataConverter dateRangeConverter = getDateRangeConverter();
        DateRange dr = dateRangeConverter.stringToValue("WEEK-1");
        assertNotNull(dr);
        assertEquals(RelativeDateRange.class, dr.getClass());
    }

    @Test
    public void ensureSingleRelativeReturnedDateForDateExpressions() throws Exception {
        DateRangeDataConverter dateRangeConverter = getDateRangeConverter();
        DateRange dr = dateRangeConverter.stringToValue("DAY+1");
        assertNotNull(dr);
        assertEquals(RelativeDateRange.class, dr.getClass());
    }

    @Test
    public void ensureFixedDateReturnedForDateExpressions() throws Exception {
        DateRangeDataConverter dateRangeConverter = getDateRangeConverter();
        DateRange dr = dateRangeConverter.stringToValue("2012-07-01");
        assertNotNull(dr);
        assertEquals(FixedDate.class, dr.getClass());

        assertEquals(new GregorianCalendar(2012, Calendar.JULY, 1).getTime(), dr.getStart());
        assertEquals(new GregorianCalendar(2012, Calendar.JULY, 1).getTime(), dr.getEnd());
    }

    @Test
    public void ensureNullReturnedForEmptyString() throws Exception {
        DateRangeDataConverter dateRangeConverter = getDateRangeConverter();
        DateRange dr = dateRangeConverter.stringToValue("");
        assertNull(dr);
    }

    @Test
    public void ensureEmptyStringReturnedForNull() throws Exception {
        DateRangeDataConverter dateRangeConverter = getDateRangeConverter();
        String str = dateRangeConverter.valueToString(null);
        assertEquals(str, "");
    }

    @Test
    public void ensureRelativeDateRangeExpressionMatches() throws Exception {
        DateRangeDataConverter dateRangeConverter = getDateRangeConverter();
        String expression = "WEEK-1";
        String str = dateRangeConverter.valueToString(DateRangeFactory.getInstance(expression));
        assertEquals(expression, str);
    }

    @Test
    public void ensureSingleRelativeDateExpressionMatches() throws Exception {
        DateRangeDataConverter dateRangeConverter = getDateRangeConverter();
        String expression = "DAY-100";
        String str = dateRangeConverter.valueToString(DateRangeFactory.getInstance(expression));
        assertEquals(expression, str);
    }

    @Test
    public void ensureFixedDateExpressionMatches() throws Exception {
        DateRangeDataConverter dateRangeConverter = getDateRangeConverter();
        String expression = "2012-07-12";
        String str = dateRangeConverter.valueToString(DateRangeFactory.getInstance(expression));
        assertEquals(expression, str);
    }

    @Test
    public void ensureFixedDatetimeExpressionMatches() throws Exception {
        DateRangeDataConverter dateRangeConverter = getDateRangeConverter();
        String expression = "2012-07-12 12:20:01";
        String dateConverterFormat = "2012-07-12";
        String str = dateRangeConverter.valueToString(DateRangeFactory.getInstance(expression));
        assertEquals(dateConverterFormat, str);
    }

    private DateRangeDataConverter getDateRangeConverter() {
        DateRangeDataConverter dateRangeConverter = new DateRangeDataConverter();
        injectDependencyToPrivateField(dateRangeConverter, "calendarFormatProvider", createCalendarFormatProvider());

        return dateRangeConverter;
    }

    public static CalendarFormatProvider createCalendarFormatProvider() {
        Mock<CalendarFormatProvider> calendarFormatProviderMock = MockUnitils.createMock(CalendarFormatProvider.class);
        calendarFormatProviderMock.returns(new SimpleDateFormat("yyyy-MM-dd")).getDateFormat();
        calendarFormatProviderMock.returns(new SimpleDateFormat("yyyy-MM-dd HH:mm")).getDatetimeFormat();

        return calendarFormatProviderMock.getMock();
    }

    public static void injectDependencyToPrivateField(Object holder, String name, Object value) {
        Field field = ReflectionUtils.getFieldWithName(holder.getClass(), name, false);
        ReflectionUtils.setFieldValue(holder, field, value);
    }

}
