package com.jaspersoft.jasperserver.war.amazon;

import com.amazonaws.AmazonClientException;
import com.amazonaws.auth.AWSCredentials;
import com.amazonaws.auth.PropertiesCredentials;
import com.jaspersoft.jasperserver.api.engine.jasperreports.util.AwsCredentialUtil;
import com.jaspersoft.jasperserver.war.amazon.client.AwsDataSourceService;
import com.jaspersoft.jasperserver.war.amazon.client.AwsDataSourceServiceImpl;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author vsabadosh
 */
public class AwsDataSourceServiceTest {
    private AwsDataSourceService awsDataSourceService;
    private AWSCredentials longTermCredentials = null;
    private final String ROLE_ARN =
            "arn:aws:iam::544542564254:role/RDS-ReadOnly-role";
    
    private final String mysql = "mysql";

    @Before
    public void initAction() throws Exception {
        awsDataSourceService = new AwsDataSourceServiceImpl();
        Map<String, Map<String, Object>> jdbcConnectionMap = new HashMap<String, Map<String, Object>>();
        Map<String, Object> mapMysql = new HashMap<String, Object>();
        mapMysql.put("jdbcUrl", "jdbc:mysql://%s");
        mapMysql.put("jdbcDriverClass", "com.mysql.jdbc.Driver");

        Map<String, Object> mapPostgresql = new HashMap<String, Object>();
        mapPostgresql.put("jdbcUrl", "jdbc:postgresql://%s");
        mapPostgresql.put("jdbcDriverClass", "org.postgresql.Driver");

        Map<String, Object> mapOracle = new HashMap<String, Object>();
        mapOracle.put("jdbcUrl", "jdbc:oracle:thin:@%s");
        mapOracle.put("jdbcDriverClass", "oracle.jdbc.OracleDriver");

        Map<String, Object> mapSQlServer = new HashMap<String, Object>();
        mapSQlServer.put("jdbcUrl", "jdbc:sqlserver://%s");
        mapSQlServer.put("jdbcDriverClass", "com.microsoft.sqlserver.jdbc.SQLServerDriver");

        jdbcConnectionMap.put("mysql", mapMysql);
        jdbcConnectionMap.put("postgresql",  mapPostgresql);
        jdbcConnectionMap.put("oracle",  mapOracle);
        jdbcConnectionMap.put("sqlserver",  mapSQlServer);

        ((AwsDataSourceServiceImpl)awsDataSourceService).setJdbcConnectionMap(jdbcConnectionMap);
        try {
           longTermCredentials = new PropertiesCredentials(AwsDataSourceServiceTest.class.getResourceAsStream("AwsCredentials.properties"));
        } catch (Exception ex) {
            System.out.println("WARN: Could not load access key and private key");
            System.out.println("WARN: Please provide properties file with Amazon access key and secret key!!!");
        }
    }

    @Test
    public void testGetRDSInstancesByCredentialsWithOutRoleARN() throws Exception {
        if (longTermCredentials != null) {
            AWSCredentials awsCredentialUtil = AwsCredentialUtil.getAWSCredentials(longTermCredentials.getAWSAccessKeyId(), longTermCredentials.getAWSSecretKey(), null);
            List rdsList = awsDataSourceService.getAwsDBInstances(awsCredentialUtil, "RDS", "us-west-2.amazonaws.com");
            Assert.assertEquals("RDS count", 0, rdsList.size());
        }
    }

    @Test
    public void testGetRDSInstancesByCredentialsWithRoleARN() throws Exception {
        if (longTermCredentials != null) {
            AWSCredentials awsCredentialUtil = AwsCredentialUtil.getAWSCredentials(longTermCredentials.getAWSAccessKeyId(), longTermCredentials.getAWSSecretKey(), ROLE_ARN);
            List rdsList = awsDataSourceService.getAwsDBInstances(awsCredentialUtil, "RDS", null);
            Assert.assertEquals("RDS count", 2, rdsList.size());
        }
    }

    @Test
    public void testGetRDSInstancesByDefaultCredentials() throws Exception {
        if (longTermCredentials != null) {
            try {
                List rdsList = awsDataSourceService.getAwsDBInstances(null, "RDS", null);
                Assert.assertEquals("RDS count", 0, rdsList.size());
            } catch (AmazonClientException ex) {
                System.out.println("WARN : " + ex.getMessage());
                System.out.println("WARN : Will work only on Amazon EC2 instance with predefined roles");
            }
        }
    }

    @Test
    public void testGetRedshiftInstancesByCredentialsWithOutRoleARN() throws Exception {
        if (longTermCredentials != null) {
            AWSCredentials awsCredentialUtil = AwsCredentialUtil.getAWSCredentials(longTermCredentials.getAWSAccessKeyId(), longTermCredentials.getAWSSecretKey(), null);
            List rdsList = awsDataSourceService.getAwsDBInstances(awsCredentialUtil, "Redshift", null);
            Assert.assertEquals("Redshift count", 0, rdsList.size());
        }
    }

}
