/*
 * Copyright (C) 2005 - 2012 Jaspersoft Corporation. All rights reserved.
 * http://www.jaspersoft.com.
 * Unless you have purchased  a commercial license agreement from Jaspersoft,
 * the following license terms  apply:
 *
 * This program is free software: you can redistribute it and/or  modify
 * it under the terms of the GNU Affero General Public License  as
 * published by the Free Software Foundation, either version 3 of  the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero  General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public  License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package com.jaspersoft.jasperserver.remote.services.impl;

import com.jaspersoft.jasperserver.api.common.domain.ExecutionContext;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Folder;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Resource;
import com.jaspersoft.jasperserver.api.metadata.common.domain.client.FolderImpl;
import com.jaspersoft.jasperserver.api.metadata.common.service.RepositoryService;
import com.jaspersoft.jasperserver.remote.exception.ResourceNotFoundException;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;
import org.springframework.security.AccessDeniedException;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;

import java.util.LinkedList;
import java.util.List;

/**
 * <p></p>
 *
 * @author Zakhar.Tomchenco
 * @version $Id$
 */
public class BatchRepositoryServiceImplTest {
    @InjectMocks
    private BatchRepositoryServiceImpl service;
    @Mock
    private RepositoryService repositoryService;

    final Resource res = new FolderImpl();
    final String uri = "/test";
    final List<String> uris = new LinkedList<String>();

    @BeforeClass
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);
    }

    @BeforeMethod
    public void cleanUp(){
        Mockito.reset(repositoryService);
        res.setURIString(uri);
        uris.clear();
        uris.add(uri);
    }

    @Test
    public void testDeleteResources_Resource() throws Exception {
        Mockito.when(repositoryService.getResource(Mockito.isNull(ExecutionContext.class), Mockito.eq(uri))).thenReturn(res);

        service.deleteResources(uris);

        Mockito.verify(repositoryService).deleteResource(Mockito.isNull(ExecutionContext.class),Mockito.eq(uri));
    }

    @Test
    public void testDeleteResources_Folder() throws Exception {
        Mockito.when(repositoryService.getResource(Mockito.isNull(ExecutionContext.class), Mockito.eq(uri))).thenReturn(null);
        Mockito.when(repositoryService.getFolder(Mockito.isNull(ExecutionContext.class), Mockito.eq(uri))).thenReturn((Folder)res);

        service.deleteResources(uris);

        Mockito.verify(repositoryService).deleteFolder(Mockito.isNull(ExecutionContext.class),Mockito.eq(uri));
    }

    @Test(expectedExceptions = {com.jaspersoft.jasperserver.remote.exception.AccessDeniedException.class})
    public void testDeleteResources_NoPermission_Resource() throws Exception {
        Mockito.when(repositoryService.getResource(Mockito.isNull(ExecutionContext.class), Mockito.eq(uri))).thenReturn(res);
        Mockito.doThrow(new AccessDeniedException("NO")).when(repositoryService).deleteResource(Mockito.isNull(ExecutionContext.class), Mockito.eq(uri));

        service.deleteResources(uris);
    }
    @Test(expectedExceptions = {com.jaspersoft.jasperserver.remote.exception.AccessDeniedException.class})
    public void testDeleteResources_NoPermission_Folder() throws Exception {
        Mockito.when(repositoryService.getResource(Mockito.isNull(ExecutionContext.class), Mockito.eq(uri))).thenReturn(null);
        Mockito.when(repositoryService.getFolder(Mockito.isNull(ExecutionContext.class), Mockito.eq(uri))).thenReturn((Folder)res);
        Mockito.doThrow(new AccessDeniedException("NO")).when(repositoryService).deleteFolder(Mockito.isNull(ExecutionContext.class), Mockito.eq(uri));

        service.deleteResources(uris);
    }

    @Test
    public void testDeleteResources_Multiple() throws Exception {
        Resource res1 = new FolderImpl();
        String res1uri = "/a";

        Resource res2 = new FolderImpl();
        String res2uri = "/b";

        uris.add(res1uri);
        uris.add(res2uri);


        Mockito.when(repositoryService.getResource(Mockito.isNull(ExecutionContext.class), Mockito.eq(uri))).thenReturn(res);
        Mockito.when(repositoryService.getResource(Mockito.isNull(ExecutionContext.class), Mockito.eq(res1uri))).thenReturn(res1);
        Mockito.when(repositoryService.getResource(Mockito.isNull(ExecutionContext.class), Mockito.eq(res2uri))).thenReturn(res2);

        service.deleteResources(uris);

        Mockito.verify(repositoryService).deleteResource(Mockito.isNull(ExecutionContext.class),Mockito.eq(uri));
        Mockito.verify(repositoryService).deleteResource(Mockito.isNull(ExecutionContext.class),Mockito.eq(res1uri));
        Mockito.verify(repositoryService).deleteResource(Mockito.isNull(ExecutionContext.class),Mockito.eq(res2uri));
    }
}
