/*
* Copyright (C) 2005 - 2009 Jaspersoft Corporation. All rights  reserved.
* http://www.jaspersoft.com.
*
* Unless you have purchased  a commercial license agreement from Jaspersoft,
* the following license terms  apply:
*
* This program is free software: you can redistribute it and/or  modify
* it under the terms of the GNU Affero General Public License  as
* published by the Free Software Foundation, either version 3 of  the
* License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU Affero  General Public License for more details.
*
* You should have received a copy of the GNU Affero General Public  License
* along with this program.&nbsp; If not, see <http://www.gnu.org/licenses/>.
*/
package com.jaspersoft.jasperserver.remote.resources.converters;

import com.jaspersoft.jasperserver.api.metadata.common.domain.Folder;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Resource;
import com.jaspersoft.jasperserver.api.metadata.common.domain.ResourceLookup;
import com.jaspersoft.jasperserver.api.metadata.common.domain.client.FolderImpl;
import com.jaspersoft.jasperserver.api.metadata.common.domain.client.ResourceLookupImpl;
import com.jaspersoft.jasperserver.api.metadata.common.service.ResourceFactory;
import com.jaspersoft.jasperserver.api.metadata.user.domain.ObjectPermission;
import com.jaspersoft.jasperserver.api.metadata.user.domain.client.ObjectPermissionImpl;
import com.jaspersoft.jasperserver.dto.resources.ClientAdhocDataView;
import com.jaspersoft.jasperserver.dto.resources.ClientResourceLookup;
import com.jaspersoft.jasperserver.remote.exception.IllegalParameterValueException;
import com.jaspersoft.jasperserver.remote.exception.MandatoryParameterNotFoundException;
import com.jaspersoft.jasperserver.remote.exception.xml.ErrorDescriptor;
import com.jaspersoft.jasperserver.remote.services.PermissionsService;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.invocation.InvocationOnMock;
import org.mockito.stubbing.Answer;
import org.springframework.security.Authentication;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;

import javax.security.sasl.AuthenticationException;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlType;
import java.text.DateFormat;
import java.text.FieldPosition;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;

import static org.mockito.Matchers.eq;
import static org.mockito.Mockito.*;
import static org.testng.Assert.*;

/**
 * <p></p>
 *
 * @author Yaroslav.Kovalchyk
 * @version $Id: ResourceConverterTest.java 29489 2013-03-11 15:59:46Z ztomchenco $
 */
public class ResourceConverterTest {
    private static final String TEST_CLIENT_OBJECT_NAME = "testClientObjectName";
    @InjectMocks
    private ResourceConverterImpl converter;
    @Mock
    private PermissionsService permissionsService;

    private final ObjectPermission permission = new ObjectPermissionImpl();

    @BeforeMethod
    public void resetConverter() {
        converter = mock(ResourceConverterImpl.class);
        permission.setPermissionMask(1);
        MockitoAnnotations.initMocks(this);
        when(permissionsService.getEffectivePermission(any(Resource.class), any(Authentication.class))).thenReturn(permission);
    }

    @Test(expectedExceptions = {MandatoryParameterNotFoundException.class})
    public void toClient_EmptyLabel() throws Exception {
        when(converter.genericFieldsToServer(any(ClientResourceLookup.class), any(Resource.class))).thenCallRealMethod();

        converter.genericFieldsToServer(new ClientAdhocDataView(), null);
    }

    @Test
    public void internalToClient() throws ParseException {
        ResourceLookup serverObject = new ResourceLookupImpl();
        final String testResourceType = "testResourceType";
        final String testResourceUri = "/test/resource/uri";
        final SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSZ");
        final String testDescription = "testDescription";
        final String testLabel = "testLabel";
        final int testVersion = 123;
        final Date creationDate = simpleDateFormat.parse("1996-05-25T22:15:26.235-0700");
        final Date updateDate = simpleDateFormat.parse("1999-08-31T03:21:55.156-0300");
        final String creationDateString = "creationDate";
        final String updateDateString = "updateDate";

        permission.setPermissionMask(1);
        serverObject.setResourceType("ignoredResourceType");
        serverObject.setURIString(testResourceUri);
        serverObject.setCreationDate(creationDate);
        serverObject.setUpdateDate(updateDate);
        serverObject.setDescription(testDescription);
        serverObject.setLabel(testLabel);
        serverObject.setVersion(testVersion);
        final ClientResourceLookup expectedClientObject = new ClientResourceLookup();
        DateFormat dateFormatMock = mock(DateFormat.class);
        when(dateFormatMock.format(eq(creationDate), any(StringBuffer.class), any(FieldPosition.class))).thenReturn(new StringBuffer(creationDateString));
        when(dateFormatMock.format(eq(updateDate), any(StringBuffer.class), any(FieldPosition.class))).thenReturn(new StringBuffer(updateDateString));
        when(converter.getDateTimeFormat()).thenReturn(dateFormatMock);
        when(converter.getClientResourceType()).thenReturn(testResourceType);
        when(converter.genericFieldsToClient(expectedClientObject, serverObject)).thenCallRealMethod();
        final ClientResourceLookup clientObject = converter.genericFieldsToClient(expectedClientObject, serverObject);
        assertSame(clientObject, expectedClientObject);
        assertEquals(clientObject.getResourceType(), testResourceType);
        assertEquals(clientObject.getCreationDate(), creationDateString);
        assertEquals(clientObject.getUpdateDate(), updateDateString);
        assertEquals(clientObject.getDescription(), testDescription);
        assertEquals(clientObject.getLabel(), testLabel);
        assertEquals(clientObject.getUri(), testResourceUri);
        assertEquals(clientObject.getVersion().intValue(), testVersion);
        assertEquals(clientObject.getPermissionMask().intValue(), permission.getPermissionMask());
    }

    @Test
    public void toServer_updateMode() throws Exception {
        final String testResourceType = "testResourceType";
        final String testResourceUri = "/test/resource/uri";
        final SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSZ");
        final String testDescription = "testDescription";
        final String testLabel = "testLabel";
        final int testVersion = 123;
        final String creationDateString = "creationDate";
        final String updateDateString = "updateDate";
        final Date creationDate = simpleDateFormat.parse("1996-05-25T22:15:26.235-0700");
        final Date updateDate = simpleDateFormat.parse("1999-08-31T03:21:55.156-0300");
        ClientResourceLookup clientResourceLookup = new ClientResourceLookup();
        clientResourceLookup.setDescription(testDescription);
        clientResourceLookup.setLabel(testLabel);
        clientResourceLookup.setResourceType("resourceTypeToIgnore");
        clientResourceLookup.setUpdateDate(updateDateString);
        clientResourceLookup.setUri(testResourceUri);
        clientResourceLookup.setVersion(testVersion);
        clientResourceLookup.setCreationDate(creationDateString);
        ResourceLookup serverObject = new ResourceLookupImpl();
        DateFormat dateFormatMock = mock(DateFormat.class);
        when(dateFormatMock.parse(creationDateString)).thenReturn(creationDate);
        when(dateFormatMock.parse(updateDateString)).thenReturn(updateDate);
        when(converter.getDateTimeFormat()).thenReturn(dateFormatMock);
        when(converter.getServerResourceType()).thenReturn(testResourceType);
        when(converter.toServer(clientResourceLookup, serverObject)).thenCallRealMethod();
        when(converter.genericFieldsToServer(clientResourceLookup, serverObject)).thenCallRealMethod();
        when(converter.resourceSpecificFieldsToServer(clientResourceLookup, serverObject)).thenReturn(serverObject);
        final Resource resource = converter.toServer(clientResourceLookup, serverObject);
        assertSame(resource, serverObject);
        assertEquals(resource.getURIString(), clientResourceLookup.getUri());
        assertEquals(resource.getDescription(), clientResourceLookup.getDescription());
        assertEquals(resource.getLabel(), clientResourceLookup.getLabel());
        assertNull(resource.getResourceType());
        assertEquals(resource.getUpdateDate(), updateDate);
        assertEquals(resource.getCreationDate(), creationDate);
        assertEquals(resource.getVersion(), clientResourceLookup.getVersion().intValue());
    }

    @Test
    public void toServer_invalidDate() throws Exception {
        ClientResourceLookup clientObject = new ClientResourceLookup();
        final String invalidCreationDate = "invalidCreationDate";
        clientObject.setCreationDate(invalidCreationDate);
        clientObject.setLabel(invalidCreationDate);
        final String invalidUpdateDate = "invalidUpdateDate";
        final ResourceLookupImpl resultToUpdate = new ResourceLookupImpl();
        when(converter.toServer(clientObject, resultToUpdate)).thenCallRealMethod();
        when(converter.genericFieldsToServer(clientObject, resultToUpdate)).thenCallRealMethod();
        when(converter.resourceSpecificFieldsToServer(clientObject, resultToUpdate)).thenReturn(resultToUpdate);
        DateFormat dateFormatMock = mock(DateFormat.class);
        when(dateFormatMock.parse(invalidCreationDate)).thenThrow(ParseException.class);
        when(converter.getDateTimeFormat()).thenReturn(dateFormatMock);
        // check invalid creation date
        IllegalParameterValueException exception = null;
        try {
            converter.toServer(clientObject, resultToUpdate);
        } catch (IllegalParameterValueException e) {
            exception = e;
        }
        assertNotNull(exception);
        ErrorDescriptor errorDescriptor = exception.getErrorDescriptor();
        assertNotNull(errorDescriptor);
        String[] parameters = errorDescriptor.getParameters();
        assertNotNull(parameters);
        assertEquals(parameters.length, 2);
        assertEquals(parameters[0], "creationDate");
        assertEquals(parameters[1], invalidCreationDate);
        // reset mock and check invalid update date
        exception = null;
        clientObject.setCreationDate(null);
        clientObject.setUpdateDate(invalidUpdateDate);
        when(dateFormatMock.parse(invalidUpdateDate)).thenThrow(ParseException.class);
        try {
            converter.toServer(clientObject, resultToUpdate);
        } catch (IllegalParameterValueException e) {
            exception = e;
        }
        assertNotNull(exception);
        errorDescriptor = exception.getErrorDescriptor();
        assertNotNull(errorDescriptor);
        parameters = errorDescriptor.getParameters();
        assertNotNull(parameters);
        assertEquals(parameters.length, 2);
        assertEquals(parameters[0], "updateDate");
        assertEquals(parameters[1], invalidUpdateDate);
    }

    @Test
    public void toClient() {
        ResourceLookup serverObject = new ResourceLookupImpl();
        serverObject.setURIString("/test/uri");
        ClientResourceLookup expectedClientObject = new ClientResourceLookup();
        when(converter.getNewClientObjectInstance()).thenReturn(expectedClientObject);
        when(converter.toClient(serverObject)).thenCallRealMethod();
        converter.toClient(serverObject);
        verify(converter).genericFieldsToClient(expectedClientObject, serverObject);
    }

    @Test
    public void getNewClientObjectInstance() {
        ResourceConverterImpl<ResourceLookup, ClientResourceLookup> converter = new ResourceConverterImpl<ResourceLookup, ClientResourceLookup>() {
            @Override
            protected ResourceLookup resourceSpecificFieldsToServer(ClientResourceLookup clientObject, ResourceLookup resultToUpdate) throws IllegalParameterValueException, MandatoryParameterNotFoundException {
                return resultToUpdate;
            }

            @Override
            protected ClientResourceLookup resourceSpecificFieldsToClient(ClientResourceLookup client, ResourceLookup serverObject) {
                return client;
            }
        };
        final ClientResourceLookup newClientObjectInstance = converter.getNewClientObjectInstance();
        assertNotNull(newClientObjectInstance);
    }

    @Test
    public void getClientResourceType_defaultClassName() {
        when(converter.getClientTypeClass()).thenReturn(TestClientObjectWithDefaultName.class);
        when(converter.getClientResourceType()).thenCallRealMethod();
        final String classSimpleName = TestClientObjectWithDefaultName.class.getSimpleName();
        final String defaultClientType = classSimpleName.replaceFirst("^.", classSimpleName.substring(0, 1).toLowerCase());
        assertEquals(converter.getClientResourceType(), defaultClientType);
    }

    @Test
    public void getClientResourceType_specifiedName() {
        when(converter.getClientTypeClass()).thenReturn(TestClientObjectWithSpecifiedName.class);
        when(converter.getClientResourceType()).thenCallRealMethod();
        assertEquals(converter.getClientResourceType(), TEST_CLIENT_OBJECT_NAME);
    }

    @Test
    public void getClientResourceType_clientWithXmlTypeAnnotation() {
        when(converter.getClientTypeClass()).thenReturn(TestClientObjectWithXmlTypeAnnotation.class);
        when(converter.getClientResourceType()).thenCallRealMethod();
        assertEquals(converter.getClientResourceType(), TEST_CLIENT_OBJECT_NAME);
    }

    @Test
    public void getClientTypeClass_rawConverter() {

        final ResourceConverterImpl resourceConverter = new ResourceConverterImpl() {
            @Override
            protected Resource resourceSpecificFieldsToServer(ClientResourceLookup clientObject, Resource resultToUpdate) throws IllegalParameterValueException, MandatoryParameterNotFoundException {
                return resultToUpdate;
            }

            @Override
            protected ClientResourceLookup resourceSpecificFieldsToClient(ClientResourceLookup client, Resource serverObject) {
                return client;
            }
        };
        IllegalArgumentException exception = null;
        try {
            resourceConverter.getClientTypeClass();
        } catch (IllegalArgumentException ex) {
            exception = ex;
        }
        assertNotNull(exception);
    }

    @Test
    public void getClientTypeClass() {
        final ResourceConverterImpl resourceConverter = new ResourceConverterImpl<ResourceLookup, ClientResourceLookup>() {
            @Override
            protected ResourceLookup resourceSpecificFieldsToServer(ClientResourceLookup clientObject, ResourceLookup resultToUpdate) throws IllegalParameterValueException, MandatoryParameterNotFoundException {
                return resultToUpdate;
            }

            @Override
            protected ClientResourceLookup resourceSpecificFieldsToClient(ClientResourceLookup client, ResourceLookup serverObject) {
                return client;
            }
        };
        final Class<?> clientTypeClass = resourceConverter.getClientTypeClass();
        assertTrue(clientTypeClass == ClientResourceLookup.class);
    }

    @Test
    public void toServer() throws Exception {
        ClientResourceLookup expectedClientObject = new ClientResourceLookup();
        Resource expectedNewObject = new FolderImpl();
        when(converter.toServer(expectedClientObject, expectedNewObject)).thenAnswer(new Answer<Object>() {
            @Override
            public Object answer(InvocationOnMock invocationOnMock) throws Throwable {
                return invocationOnMock.getArguments()[1];
            }
        });
        when(converter.getNewResourceInstance()).thenReturn(expectedNewObject);
        when(converter.toServer(expectedClientObject)).thenCallRealMethod();
        final Resource resource = converter.toServer(expectedClientObject);
        assertSame(resource, expectedNewObject);
    }

    @Test
    public void getNewResourceInstance() {
        when(converter.getNewResourceInstance()).thenCallRealMethod();
        final String expectedResourceTypeName = "testResourceTypeName";
        when(converter.getServerResourceType()).thenReturn(expectedResourceTypeName);
        converter.objectFactory = mock(ResourceFactory.class);
        final FolderImpl expectedFolder = new FolderImpl();
        when(converter.objectFactory.newResource(null, expectedResourceTypeName)).thenReturn(expectedFolder);
        final Object folder = converter.getNewResourceInstance();
        assertSame(folder, expectedFolder);
    }

    @Test
    public void getServerResourceType() {
        // correct resource converter for folder resources
        ResourceConverterImpl<Folder, ClientResourceLookup> folderConverter = new ResourceConverterImpl<Folder, ClientResourceLookup>() {
            @Override
            protected Folder resourceSpecificFieldsToServer(ClientResourceLookup clientObject, Folder resultToUpdate) throws IllegalParameterValueException, MandatoryParameterNotFoundException {
                return resultToUpdate;
            }

            @Override
            protected ClientResourceLookup resourceSpecificFieldsToClient(ClientResourceLookup client, Folder serverObject) {
                return client;
            }
        };
        final String extractedResourceName = folderConverter.getServerResourceType();
        assertEquals(extractedResourceName, Folder.class.getName());

        // incorrect resource converter for unknown resource type
        ResourceConverterImpl<?, ?> rawConverter = new ResourceConverterImpl() {
            @Override
            protected Resource resourceSpecificFieldsToServer(ClientResourceLookup clientObject, Resource resultToUpdate) throws IllegalParameterValueException, MandatoryParameterNotFoundException {
                return resultToUpdate;
            }

            @Override
            protected ClientResourceLookup resourceSpecificFieldsToClient(ClientResourceLookup client, Resource serverObject) {
                return client;
            }
        };
        IllegalArgumentException exception = null;
        try {
            rawConverter.getServerResourceType();
        } catch (IllegalArgumentException ex) {
            exception = ex;
        }
        assertNotNull(exception);
    }


    @XmlRootElement(name = TEST_CLIENT_OBJECT_NAME)
    private class TestClientObjectWithSpecifiedName extends ClientResourceLookup {
    }

    @XmlRootElement
    private class TestClientObjectWithDefaultName extends ClientResourceLookup {
    }

    @XmlType(name = TEST_CLIENT_OBJECT_NAME)
    private class TestClientObjectWithXmlTypeAnnotation extends ClientResourceLookup {
    }
}
