/*
* Copyright (C) 2005 - 2009 Jaspersoft Corporation. All rights  reserved.
* http://www.jaspersoft.com.
*
* Unless you have purchased  a commercial license agreement from Jaspersoft,
* the following license terms  apply:
*
* This program is free software: you can redistribute it and/or  modify
* it under the terms of the GNU Affero General Public License  as
* published by the Free Software Foundation, either version 3 of  the
* License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU Affero  General Public License for more details.
*
* You should have received a copy of the GNU Affero General Public  License
* along with this program.&nbsp; If not, see <http://www.gnu.org/licenses/>.
*/
package com.jaspersoft.jasperserver.remote.resources.converters;

import com.jaspersoft.jasperserver.api.metadata.common.domain.DataType;
import com.jaspersoft.jasperserver.api.metadata.common.domain.client.DataTypeImpl;
import com.jaspersoft.jasperserver.dto.resources.ClientDataType;
import com.jaspersoft.jasperserver.remote.exception.IllegalParameterValueException;
import com.jaspersoft.jasperserver.remote.resources.GenericParametersReflectionHelper;
import com.jaspersoft.jasperserver.war.cascade.InputControlValidationException;
import com.jaspersoft.jasperserver.war.cascade.handlers.converters.DataConverterService;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;

import static org.mockito.Mockito.when;
import static org.testng.Assert.*;

/**
 * <p></p>
 *
 * @author Yaroslav.Kovalchyk
 * @version $Id: DataTypeResourceConverterTest.java 29489 2013-03-11 15:59:46Z ztomchenco $
 */
public class DataTypeResourceConverterTest {
    @InjectMocks
    private DataTypeResourceConverter converter = new DataTypeResourceConverter();
    @Mock
    private DataConverterService dataConverterService;

    @BeforeMethod
    public void initMocks(){
        MockitoAnnotations.initMocks(this);
    }

    @Test
    public void correctClientServerResourceType(){
        assertEquals(converter.getClientResourceType(), GenericParametersReflectionHelper.extractClientType(ClientDataType.class));
        assertEquals(converter.getServerResourceType(), DataType.class.getName());
    }

    @Test
    public void resourceSpecificFieldsToServer_allSimpleFields() throws Exception {
        final String expectedPattern = "testPattern";
        final Boolean expectedStrictMax = true;
        final Boolean expectedStrictMin = true;
        final Integer expectedMaxLength = 125;
        final Integer expectedDecimals = 5;
        DataType expectedServerObject = new DataTypeImpl();
        ClientDataType clientObject = new ClientDataType();
        clientObject.setPattern(expectedPattern);
        clientObject.setStrictMax(expectedStrictMax);
        clientObject.setStrictMin(expectedStrictMin);
        clientObject.setMaxLength(expectedMaxLength);
        clientObject.setDecimals(expectedDecimals);
        final DataType result = converter.resourceSpecificFieldsToServer(clientObject, expectedServerObject);
        assertSame(result, expectedServerObject);
        assertEquals(result.getRegularExpr(), expectedPattern);
        assertEquals(result.isStrictMax(), expectedStrictMax.booleanValue());
        assertEquals(result.isStrictMin(), expectedStrictMin.booleanValue());
        assertEquals(result.getMaxLength(), expectedMaxLength);
        assertEquals(result.getDecimals(), expectedDecimals);
    }

    @Test
    public void resourceSpecificFieldsToServer_type() throws Exception {
        ClientDataType clientObject = new ClientDataType();
        DataType expectedServerObject = new DataTypeImpl();
        // number
        clientObject.setType(ClientDataType.TypeOfDataType.number);
        DataType result = converter.resourceSpecificFieldsToServer(clientObject, expectedServerObject);
        assertEquals(result.getType(), DataType.TYPE_NUMBER);
        // text
        clientObject.setType(ClientDataType.TypeOfDataType.text);
        result = converter.resourceSpecificFieldsToServer(clientObject, expectedServerObject);
        assertEquals(result.getType(), DataType.TYPE_TEXT);
        // date
        clientObject.setType(ClientDataType.TypeOfDataType.date);
        result = converter.resourceSpecificFieldsToServer(clientObject, expectedServerObject);
        assertEquals(result.getType(), DataType.TYPE_DATE);
        // datetime
        clientObject.setType(ClientDataType.TypeOfDataType.datetime);
        result = converter.resourceSpecificFieldsToServer(clientObject, expectedServerObject);
        assertEquals(result.getType(), DataType.TYPE_DATE_TIME);
        // time
        clientObject.setType(ClientDataType.TypeOfDataType.time);
        result = converter.resourceSpecificFieldsToServer(clientObject, expectedServerObject);
        assertEquals(result.getType(), DataType.TYPE_TIME);
        // default value is TYPE_TEXT
        clientObject.setType(null);
        result = converter.resourceSpecificFieldsToServer(clientObject, expectedServerObject);
        assertEquals(result.getType(), DataType.TYPE_TEXT);
    }

    @Test
    public void resourceSpecificFieldsToServer_maxValue_minValue() throws InputControlValidationException, IllegalParameterValueException {
        final String expectedMaxValue = "testMaxValue";
        final String expectedMinValue = "testMinValue";
        final String maxValueToConvert = "testMaxValueToConvert";
        final String minValueToConvert = "testMinValueToConvert";
        final String invalidMaxValue = "textInvalidMaxValue";
        final String invalidMinValue = "textInvalidMinValue";
        final DataType expectedServerObject = new DataTypeImpl();
        final ClientDataType clientObject = new ClientDataType();
        clientObject.setMaxValue(maxValueToConvert);
        clientObject.setMinValue(minValueToConvert);
        when(dataConverterService.convertSingleValue(maxValueToConvert, expectedServerObject)).thenReturn(expectedMaxValue);
        when(dataConverterService.convertSingleValue(minValueToConvert, expectedServerObject)).thenReturn(expectedMinValue);
        when(dataConverterService.convertSingleValue(invalidMaxValue,expectedServerObject)).thenThrow(new InputControlValidationException(null, null, null, null));
        when(dataConverterService.convertSingleValue(invalidMinValue,expectedServerObject)).thenThrow(new InputControlValidationException(null, null, null, null));
        final DataType result = converter.resourceSpecificFieldsToServer(clientObject, expectedServerObject);
        assertSame(result, expectedServerObject);
        assertEquals(result.getMaxValue(), expectedMaxValue);
        assertEquals(result.getMinValue(), expectedMinValue);
        IllegalParameterValueException exception = null;
        clientObject.setMaxValue(invalidMaxValue);
        try{
            converter.resourceSpecificFieldsToServer(clientObject, expectedServerObject);
        } catch (IllegalParameterValueException ex){
            exception = ex;
        }
        assertNotNull(exception);
        exception = null;
        clientObject.setMaxValue(null);
        clientObject.setMaxValue(invalidMinValue);
        try{
            converter.resourceSpecificFieldsToServer(clientObject, expectedServerObject);
        } catch (IllegalParameterValueException ex){
            exception = ex;
        }
        assertNotNull(exception);
    }

    @Test
    public void resourceSpecificFieldsToClient(){
        final String expectedPattern = "testPattern";
        final Boolean expectedStrictMax = true;
        final Boolean expectedStrictMin = true;
        final Integer expectedMaxLength = 125;
        final Integer expectedDecimals = 5;
        DataType serverObject = new DataTypeImpl();
        ClientDataType expectedClientObject = new ClientDataType();
        serverObject.setRegularExpr(expectedPattern);
        serverObject.setStrictMax(expectedStrictMax);
        serverObject.setStrictMin(expectedStrictMin);
        serverObject.setMaxLength(expectedMaxLength);
        serverObject.setDecimals(expectedDecimals);
        final ClientDataType result = converter.resourceSpecificFieldsToClient(expectedClientObject, serverObject);
        assertSame(result, expectedClientObject);
        assertEquals(result.getPattern(), expectedPattern);
        assertEquals(result.isStrictMax(), expectedStrictMax);
        assertEquals(result.isStrictMin(), expectedStrictMin);
        assertEquals(result.getMaxLength(), expectedMaxLength);
        assertEquals(result.getDecimals(), expectedDecimals);
    }

    @Test
    public void resourceSpecificFieldsToClient_emptyStringAsPatternIsIgnored(){
        DataType serverObject = new DataTypeImpl();
        ClientDataType expectedClientObject = new ClientDataType();
        serverObject.setRegularExpr("");
        final ClientDataType result = converter.resourceSpecificFieldsToClient(expectedClientObject, serverObject);
        assertSame(result, expectedClientObject);
        assertNull(result.getPattern());
    }

    @Test
    public void resourceSpecificFieldsToClient_type() throws Exception {
        ClientDataType clientObject = new ClientDataType();
        DataType serverObject = new DataTypeImpl();
        // number
        serverObject.setType(DataType.TYPE_NUMBER);
        ClientDataType result = converter.resourceSpecificFieldsToClient(clientObject, serverObject);
        assertEquals(result.getType(), ClientDataType.TypeOfDataType.number);
        // text
        serverObject.setType(DataType.TYPE_TEXT);
        result = converter.resourceSpecificFieldsToClient(clientObject, serverObject);
        assertEquals(result.getType(), ClientDataType.TypeOfDataType.text);
        // date
        serverObject.setType(DataType.TYPE_DATE);
        result = converter.resourceSpecificFieldsToClient(clientObject, serverObject);
        assertEquals(result.getType(), ClientDataType.TypeOfDataType.date);
        // datetime
        serverObject.setType(DataType.TYPE_DATE_TIME);
        result = converter.resourceSpecificFieldsToClient(clientObject, serverObject);
        assertEquals(result.getType(), ClientDataType.TypeOfDataType.datetime);
        // time
        serverObject.setType(DataType.TYPE_TIME);
        result = converter.resourceSpecificFieldsToClient(clientObject, serverObject);
        assertEquals(result.getType(), ClientDataType.TypeOfDataType.time);
        // invalid type
        Exception exception = null;
        serverObject.setType((byte)10);
        try{
            converter.resourceSpecificFieldsToClient(clientObject, serverObject);
        } catch (Exception e){
            exception = e;
        }
        assertNotNull(exception);
    }

    @Test
    public void resourceSpecificFieldsToClient_maxValue_minValue() throws Exception {
        final String expectedMaxValue = "testMaxValue";
        final String expectedMinValue = "testMinValue";
        final String maxValueToConvert = "testMaxValueToConvert";
        final String minValueToConvert = "testMinValueToConvert";
        final DataType serverObject = new DataTypeImpl();
        final ClientDataType expectedClientObject = new ClientDataType();
        serverObject.setMaxValue(maxValueToConvert);
        serverObject.setMinValue(minValueToConvert);
        when(dataConverterService.formatSingleValue(maxValueToConvert, serverObject, String.class)).thenReturn(expectedMaxValue);
        when(dataConverterService.formatSingleValue(minValueToConvert, serverObject, String.class)).thenReturn(expectedMinValue);
        final ClientDataType result = converter.resourceSpecificFieldsToClient(expectedClientObject, serverObject);
        assertSame(result, expectedClientObject);
        assertEquals(result.getMaxValue(), expectedMaxValue);
        assertEquals(result.getMinValue(), expectedMinValue);
    }

}
