/*
* Copyright (C) 2005 - 2009 Jaspersoft Corporation. All rights  reserved.
* http://www.jaspersoft.com.
*
* Unless you have purchased  a commercial license agreement from Jaspersoft,
* the following license terms  apply:
*
* This program is free software: you can redistribute it and/or  modify
* it under the terms of the GNU Affero General Public License  as
* published by the Free Software Foundation, either version 3 of  the
* License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU Affero  General Public License for more details.
*
* You should have received a copy of the GNU Affero General Public  License
* along with this program.&nbsp; If not, see <http://www.gnu.org/licenses/>.
*/
package com.jaspersoft.jasperserver.remote.services.impl;

import com.jaspersoft.jasperserver.api.common.domain.ExecutionContext;
import com.jaspersoft.jasperserver.api.common.domain.impl.ExecutionContextImpl;
import com.jaspersoft.jasperserver.api.engine.common.service.EngineService;
import com.jaspersoft.jasperserver.api.engine.jasperreports.domain.impl.ReportUnitRequest;
import com.jaspersoft.jasperserver.api.engine.jasperreports.domain.impl.ReportUnitResult;
import com.jaspersoft.jasperserver.api.engine.jasperreports.service.DataCacheProvider;
import com.jaspersoft.jasperserver.api.engine.jasperreports.service.DataSnapshotService;
import com.jaspersoft.jasperserver.api.metadata.common.domain.InputControlsContainer;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Resource;
import com.jaspersoft.jasperserver.api.metadata.jasperreports.domain.ReportUnit;
import com.jaspersoft.jasperserver.remote.ReportExporter;
import com.jaspersoft.jasperserver.remote.ServiceException;
import com.jaspersoft.jasperserver.remote.ServicesConfiguration;
import com.jaspersoft.jasperserver.remote.exception.RemoteException;
import com.jaspersoft.jasperserver.remote.exception.ResourceNotFoundException;
import com.jaspersoft.jasperserver.remote.exception.xml.ErrorDescriptor;
import com.jaspersoft.jasperserver.remote.services.ReportExecutionOptions;
import com.jaspersoft.jasperserver.remote.services.ReportExecutor;
import com.jaspersoft.jasperserver.remote.utils.AuditHelper;
import com.jaspersoft.jasperserver.remote.utils.RepositoryHelper;
import com.jaspersoft.jasperserver.war.cascade.CachedRepositoryService;
import com.jaspersoft.jasperserver.war.cascade.CascadeResourceNotFoundException;
import net.sf.jasperreports.engine.JRExporterParameter;
import net.sf.jasperreports.engine.JRParameter;
import net.sf.jasperreports.engine.JasperPrint;
import net.sf.jasperreports.engine.JasperReportsContext;
import net.sf.jasperreports.engine.ReportContext;
import net.sf.jasperreports.engine.SimpleReportContext;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.stereotype.Service;

import java.io.OutputStream;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.TimeZone;

/**
 * <p></p>
 *
 * @author Yaroslav.Kovalchyk
 * @version $Id: ReportExecutorImpl.java 26599 2012-12-10 13:04:23Z ykovalchyk $
 */
@Service
public class ReportExecutorImpl implements ReportExecutor {
    private final static Log log = LogFactory.getLog(ReportExecutorImpl.class);
    @javax.annotation.Resource
    private AuditHelper auditHelper;
    @javax.annotation.Resource(name = "engineService")
    private EngineService engine;
    @javax.annotation.Resource(name = "remoteServiceConfiguration")
    private ServicesConfiguration servicesConfiguration;
    @javax.annotation.Resource(name = "engineServiceDataCacheProvider")
    private DataCacheProvider dataCacheProvider;
    @javax.annotation.Resource(name = "dataSnapshotService")
    private DataSnapshotService dataSnapshotService;
    @javax.annotation.Resource
    private JasperReportsContext jasperReportsRemoteContext;
    @javax.annotation.Resource
    private CachedRepositoryService cachedRepositoryService;

    /**
     * Running of the report happens here.
     *
     * @param reportUnitUri         - the URI of a report to run
     * @param parameters             - input parameters
     * @param reportExecutionOptions - report execution options
     * @return report unit execution result
     * @throws com.jaspersoft.jasperserver.remote.exception.RemoteException if any error occurs
     */
    public ReportUnitResult runReport(String reportUnitUri, Map<String, Object> parameters,
            ReportExecutionOptions reportExecutionOptions) throws RemoteException {
        InputControlsContainer report = getResource(InputControlsContainer.class, reportUnitUri);
        long currentTime = System.currentTimeMillis();
        auditHelper.createAuditEvent("runReport");
        RunReportStrategy strategy = getStrategyForReport(report);
        if (strategy == null) {
            throw new RemoteException(new ErrorDescriptor.Builder()
                    .setErrorCode("webservices.error.errorExecutingReportUnit").setParameters(report.getURI()).getErrorDescriptor());
        }
        parameters.put(JRParameter.IS_IGNORE_PAGINATION, reportExecutionOptions.isIgnorePagination());
        // run the report
        ReportUnitResult reportUnitResult = strategy.runReport(report, parameters, engine, reportExecutionOptions);

        if (reportUnitResult == null) {
            throw new RemoteException(new ErrorDescriptor.Builder()
                    .setErrorCode("webservices.error.errorExecutingReportUnit").setParameters(report.getURI()).getErrorDescriptor());
        }

        auditHelper.addPropertyToAuditEvent("runReport", "reportExecutionStartTime", new Date(currentTime));
        auditHelper.addPropertyToAuditEvent("runReport", "reportExecutionTime", System.currentTimeMillis() - currentTime);
        return reportUnitResult;
    }

    protected <T extends Resource> T getResource(Class<T> resourceClass, String resourceUri) throws ResourceNotFoundException {
        try {
            return cachedRepositoryService.getResource(resourceClass, resourceUri);
        } catch (CascadeResourceNotFoundException e) {
            throw new ResourceNotFoundException(resourceUri);
        }
    }


    protected ExecutionContext createExecutionContext() {
        ExecutionContextImpl ctx = new ExecutionContextImpl();
        ctx.setLocale(LocaleContextHolder.getLocale());
        ctx.setTimeZone(TimeZone.getDefault());
        return ctx;
    }

    /**
     * Look for the ReportExporter configured for the named format and export
     * the report.
     *
     * @param reportUnitURI    - the report unit URI
     * @param jasperPrint      - jasperPring object
     * @param format           - format
     * @param output           - output stream to write to
     * @param exportParameters - parameters for export procedure
     * @return map with export results
     * @throws com.jaspersoft.jasperserver.remote.ServiceException
     */
    public Map<JRExporterParameter, Object> exportReport(String reportUnitURI, JasperPrint jasperPrint, String format, OutputStream output,
            HashMap exportParameters) throws ServiceException {
        ReportExporter exporter = servicesConfiguration.getExporter(format.toLowerCase());
        if (exporter == null) {
            throw new ServiceException(3, "Export format " + format.toLowerCase() + " not supported or misconfigured");
        }
        try {
            InputControlsContainer report = getResource(InputControlsContainer.class, reportUnitURI);
            final RunReportStrategy strategyForReport = getStrategyForReport(report);
            return exporter.exportReport(jasperPrint, output, engine, exportParameters, createExecutionContext(),
                    strategyForReport.getConcreteReportURI(report));
        } catch (Exception ex) {
            throw new ServiceException(3, ex.getMessage() + " (while exporting the report)");
        }
    }

    /**
     * @param report - the report resource from a repository
     * @return strategy for running of a report of concrete type
     */
    protected RunReportStrategy getStrategyForReport(Resource report) {
        return report instanceof ReportUnit ? new RunReportUnitStrategy() : null;
    }

    /**
     * Generic run report strategy. Contains generic run report functionality.
     *
     * @param <ReportType> - concrete type of report
     */
    protected abstract class GenericRunReportStrategy<ReportType extends Resource> implements RunReportStrategy {
        /**
         * Runs report of concrete type
         *
         * @param reportResource - the report resource from a repository
         * @param parameters     - input parameters
         * @param engine         - engine service
         * @return result of report execution
         */
        // report type should correspond to concrete type of strategy. getStrategyForReport() assure that.
        // So, unchecked cast is safe
        @SuppressWarnings("unchecked")
        public ReportUnitResult runReport(Resource reportResource, Map<String, Object> parameters, EngineService engine,
                ReportExecutionOptions options) {
            ReportType report = (ReportType) reportResource;
            Map<String, Object> convertedParameters = parameters != null ? RepositoryHelper.convertParameterValues(getConcreteReportURI(report), parameters, engine) : new HashMap<String, Object>();
            ReportUnitRequest request = getReportUnitRequest(report, convertedParameters, options);
            request.setAsynchronous(options.isAsync());
            ExecutionContext executionContext = createExecutionContext();
            ReportUnitResult result = (ReportUnitResult) engine.execute(executionContext, request);
            persistDataSnapshot(executionContext, options, reportResource, request.getReportContext());
            return result;
        }

        protected void persistDataSnapshot(ExecutionContext executionContext, ReportExecutionOptions options,
                Resource reportResource, ReportContext reportContext) {
            DataCacheProvider.SnapshotSaveStatus snapshotSaveStatus = dataCacheProvider.getSnapshotSaveStatus(reportContext);
            switch (snapshotSaveStatus) {
                case NEW:
                    // automatic save
                    if (log.isDebugEnabled()) {
                        log.debug("saving initial data snapshot for " + reportResource.getURIString());
                    }

                    saveAutoDataSnapshot(executionContext, reportResource, reportContext);
                    break;
                case UPDATED:
                    if (options.isSaveDataSnapshot()) {
                        // requested save
                        if (log.isDebugEnabled()) {
                            log.debug("saving updated data snapshot for " + reportResource.getURIString());
                        }

                        saveDataSnapshot(executionContext, reportResource, reportContext);
                    }
                    break;
                case NO_CHANGE:
                default:
                    //NOP
                    break;
            }
        }

        protected void saveAutoDataSnapshot(ExecutionContext executionContext, Resource reportResource,
                ReportContext reportContext) {
            ReportUnit reportUnit = getReportUnit(reportResource);
            try {
                dataSnapshotService.saveAutoReportDataSnapshot(executionContext, reportContext, reportUnit);
            } catch (Exception e) {
                // catching any exceptions for automatic and requested save
                log.error("Error while saving data snapshot for " + reportUnit.getURIString(), e);
            }
        }

        protected void saveDataSnapshot(ExecutionContext executionContext, Resource reportResource,
                ReportContext reportContext) {
            ReportUnit reportUnit = getReportUnit(reportResource);
            try {
                dataSnapshotService.saveReportDataSnapshot(executionContext, reportContext, reportUnit);
            } catch (Exception e) {
                // catching any exceptions for automatic and requested save
                log.error("Error while saving data snapshot for " + reportUnit.getURIString(), e);
            }
        }

        protected ReportUnitRequest getReportUnitRequest(ReportType reportResource, Map<String, Object> parameters,
                ReportExecutionOptions options) {
            Map<String, Object> requestParams = new HashMap<String, Object>();
            requestParams.putAll(parameters);

            // we need a report context for snapshots
            SimpleReportContext reportContext = new SimpleReportContext();
            requestParams.put(JRParameter.REPORT_CONTEXT, reportContext);

            ReportUnitRequest request = new ReportUnitRequest(getConcreteReportURI(reportResource), requestParams);
            request.setReportContext(reportContext);
            request.setJasperReportsContext(getJasperReportsContext(options.isInteractive()));

            // recording is enabled for first-time saves
            request.setRecordDataSnapshot(dataSnapshotService.isSnapshotPersistenceEnabled());

            // fresh data if requested or saving
            request.setUseDataSnapshot(!(options.isFreshData() || options.isSaveDataSnapshot()));

            return request;
        }
    }

    public JasperReportsContext getJasperReportsContext(Boolean interactive) {
        return jasperReportsRemoteContext;
    }

    /**
     * Get content type for resource type.
     *
     * @param outputFormat - resource output format
     * @return content type
     */
    public String getContentType(String outputFormat) {
        return servicesConfiguration.getExporter(outputFormat.toLowerCase()).getContentType();
    }

    /**
     * Strategy to run report of ReportUnit type
     */
    protected class RunReportUnitStrategy extends GenericRunReportStrategy<ReportUnit> {

        public String getConcreteReportURI(Resource reportResource) {
            return reportResource.getURI();
        }

        // report type is ReportUnit for this strategy. Therefore cast is safe.
        @SuppressWarnings("unchecked")
        public ReportUnit getReportUnit(Resource report) {
            return (ReportUnit) report;
        }
    }

    protected interface RunReportStrategy {
        ReportUnitResult runReport(Resource reportResource, Map<String, Object> parameters, EngineService engine,
                ReportExecutionOptions options);

        ReportUnit getReportUnit(Resource reportResource);

        String getConcreteReportURI(Resource reportResource);
    }
}
