/*
* Copyright (C) 2005 - 2009 Jaspersoft Corporation. All rights  reserved.
* http://www.jaspersoft.com.
*
* Unless you have purchased  a commercial license agreement from Jaspersoft,
* the following license terms  apply:
*
* This program is free software: you can redistribute it and/or  modify
* it under the terms of the GNU Affero General Public License  as
* published by the Free Software Foundation, either version 3 of  the
* License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU Affero  General Public License for more details.
*
* You should have received a copy of the GNU Affero General Public  License
* along with this program.&nbsp; If not, see <http://www.gnu.org/licenses/>.
*/
package com.jaspersoft.jasperserver.remote.services;

import com.jaspersoft.jasperserver.api.JSValidationException;
import com.jaspersoft.jasperserver.api.engine.common.service.ReportExecutionStatusInformation;
import com.jaspersoft.jasperserver.api.engine.common.service.SchedulerReportExecutionStatusSearchCriteria;
import com.jaspersoft.jasperserver.remote.exception.RemoteException;
import com.jaspersoft.jasperserver.remote.exception.ResourceNotFoundException;

import java.util.Map;
import java.util.Set;

/**
 * Facade service to run reports
 *
 * @author Yaroslav.Kovalchyk
 * @version $Id: RunReportService.java 28947 2013-02-26 15:02:08Z vsabadosh $
 */
public interface RunReportService {

    public static final String CONTEXT_PATH_ATTACHMENTS_PREFIX_TEMPLATE_PLACEHOLDER = "{contextPath}";
    public static final String REPORT_EXECUTION_ID_ATTACHMENTS_PREFIX_TEMPLATE_PLACEHOLDER = "{reportExecutionId}";
    public static final String EXPORT_OPTIONS_ATTACHMENTS_PREFIX_TEMPLATE_PLACEHOLDER = "{exportOptions}";

    /**
     * Possibility to run report and get its output resource via the only call
     *
     *
     * @param reportUnitURI - URI of the report to run
     * @param outputFormat - output format
     * @param ignorePagination - if true, then pagination is ignored
     * @param attachmentsPrefix   - URL prefix for report attachments (i.e. images prefix)
     * @param page - page number, if null - all pages
     * @param transformerKey - transformer key
     * @param rawParameters - map with parameters in raw state(raw mean, that all parameters are strings or arrays of strings)
     * @param avoidCache - if true, then report will be regenerated     @return resource in corresponding output format
     * @param freshData - if true, ignore existing data snapshots
     * @param saveDataSnapshot - if true, save a data snapshot with the current data
     * @param interactive - if true and report contains highcharts element, then report output is interactive, otherwise static image is generated.
     * @throws RemoteException is thrown in case if error occurs during report execution or export
     * @deprecated use getReportOutputFromRawParameters(String reportUnitURI, Map<String, String[]> rawParameters, ReportExecutionOptions options, ExportExecutionOptions exportExecutionOptions)
     */
    ReportOutputResource getReportOutputFromRawParameters(String reportUnitURI, String outputFormat, Boolean ignorePagination, String attachmentsPrefix, Integer page, String transformerKey,
                                                          Map<String, String[]> rawParameters, Boolean avoidCache, 
                                                          Boolean freshData, Boolean saveDataSnapshot, Boolean interactive) throws RemoteException;

    /**
     *
     * @param reportUnitURI - URI of the report to run
     * @param rawParameters - map with parameters in raw state(raw mean, that all parameters are strings or arrays of strings)
     * @param options - report execution options
     * @return report output resource
     * @throws RemoteException is thrown in case if error occurs during report execution or export
     */
    ReportOutputResource getReportOutputFromRawParameters(String reportUnitURI, Map<String, String[]> rawParameters, ReportExecutionOptions options, ExportExecutionOptions exportExecutionOptions) throws RemoteException;

    /**
     * Run report and generate report execution metadata.
     *
     * @param reportUnitURI - URI of the report unit
     * @param rawParameters - input parameters in a raw representation (Map<String, String[]>)
     * @param executionOptions - options for a report execution procedure
     * @param exportOptions - options for a report export
     * @return report execution metadata
     * @throws RemoteException in case if report execution metadata generation fails.
     *                          See RemoteException.getErrorDesctiptor() to get detailed error descriptions.
     */
    ReportExecution getReportExecutionFromRawParameters(String reportUnitURI, Map<String, String[]> rawParameters, ReportExecutionOptions executionOptions, ExportExecutionOptions exportOptions) throws RemoteException, JSValidationException;

    /**
     * Get current state of report execution by report execution ID
     *
     * @param executionId - report execution ID
     * @return report execution metadata
     * @throws ResourceNotFoundException if required resource isn't found
     */
    ReportExecution getReportExecution(String executionId) throws ResourceNotFoundException;

    /**
     * Get report output resource. Synchronous export.
     *
     * @param executionId - report execution ID
     * @param exportExecutionOptions - options for export
     * @return export output resource.
     * @throws RemoteException in case if report export fails. See RemoteException.getErrorDesctiptor() to get detailed error descriptions.
     */
    ReportOutputResource getOutputResource(String executionId, ExportExecutionOptions exportExecutionOptions) throws RemoteException;

    /**
     * Asynchronous start of export procedure.
     *
     * @param executionId - report execution ID
     * @param exportOptions - options for export
     * @return export metadata
     * @throws RemoteException in case if report export fails. See RemoteException.getErrorDesctiptor() to get detailed error descriptions.
     */
    ExportExecution executeExport(String executionId, ExportExecutionOptions exportOptions) throws RemoteException;

    /**
     * Get report output attachment.
     *
     * @param executionId - report execution ID
     * @param exportOptions - export options
     * @param attachmentName - name of required attachment
     * @return report attachment (e.g. image)
     * @throws ResourceNotFoundException in case if required resource is not found
     */
    ReportOutputResource getAttachment(String executionId, ExportExecutionOptions exportOptions, String attachmentName) throws ResourceNotFoundException;

    /**
     * Get export execution object.
     *
     * @param executionId - report execution ID
     * @param exportOptions - export options
     * @return export execution object
     * @throws ResourceNotFoundException in case if either report execution or export execution isn't found
     */
    ExportExecution getExportExecution(String executionId, ExportExecutionOptions exportOptions) throws ResourceNotFoundException;

    /**
     * Search for currently running reports
     *
     * @param searchCriteria - search criteria
     * @return set of currently running report's information
     * @throws RemoteException if any error occurs
     */
    Set<ReportExecutionStatusInformation> getCurrentlyRunningReports(SchedulerReportExecutionStatusSearchCriteria searchCriteria)
            throws RemoteException;

    /**
     * Report execution cancellation.
     *
     * @param executionId - report execution ID
     * @throws RemoteException if any error occurs
     */
    Boolean cancelReportExecution(String executionId) throws RemoteException;

}
