/*
* Copyright (C) 2005 - 2009 Jaspersoft Corporation. All rights  reserved.
* http://www.jaspersoft.com.
*
* Unless you have purchased  a commercial license agreement from Jaspersoft,
* the following license terms  apply:
*
* This program is free software: you can redistribute it and/or  modify
* it under the terms of the GNU Affero General Public License  as
* published by the Free Software Foundation, either version 3 of  the
* License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU Affero  General Public License for more details.
*
* You should have received a copy of the GNU Affero General Public  License
* along with this program.&nbsp; If not, see <http://www.gnu.org/licenses/>.
*/
package com.jaspersoft.jasperserver.jaxrs.resources;

import com.jaspersoft.jasperserver.api.metadata.common.domain.DataType;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Folder;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Resource;
import com.jaspersoft.jasperserver.api.metadata.common.domain.client.DataTypeImpl;
import com.jaspersoft.jasperserver.api.metadata.common.domain.client.FolderImpl;
import com.jaspersoft.jasperserver.api.metadata.common.domain.client.QueryImpl;
import com.jaspersoft.jasperserver.dto.resources.ClientDataType;
import com.jaspersoft.jasperserver.dto.resources.ClientQuery;
import com.jaspersoft.jasperserver.dto.resources.ClientResourceLookup;
import com.jaspersoft.jasperserver.dto.resources.ResourceMediaType;
import com.jaspersoft.jasperserver.remote.exception.AccessDeniedException;
import com.jaspersoft.jasperserver.remote.exception.ResourceNotFoundException;
import com.jaspersoft.jasperserver.remote.resources.converters.ResourceConverterProvider;
import com.jaspersoft.jasperserver.remote.resources.converters.ToClientConverter;
import com.jaspersoft.jasperserver.remote.resources.converters.ToServerConverter;
import com.jaspersoft.jasperserver.remote.services.SingleRepositoryService;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;

import javax.servlet.http.HttpServletResponse;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import java.util.List;

import static org.mockito.Matchers.eq;
import static org.mockito.Mockito.*;
import static org.testng.Assert.*;

/**
 * <p></p>
 *
 * @author Yaroslav.Kovalchyk
 * @version $Id: ResourceDetailsJaxrsServiceTest.java 29363 2013-03-07 14:13:40Z ztomchenco $
 */
public class ResourceDetailsJaxrsServiceTest {
    @InjectMocks
    private ResourceDetailsJaxrsService service = new ResourceDetailsJaxrsService();
    @Mock
    private ResourceConverterProvider resourceConverterProvider;
    @Mock
    private SingleRepositoryService repositoryService;
    @Mock
    private ToServerConverter serverConverter;
    @Mock
    private ToClientConverter clientConverter;

    final private String uri = "test";
    final private Resource folder = new FolderImpl();
    final private Resource res = new QueryImpl();
    final private ClientResourceLookup clientRes = new ClientQuery();

    @BeforeClass
    public void init() {
        MockitoAnnotations.initMocks(this);
    }

    @BeforeMethod
    public void cleanUp() throws Exception{
        Mockito.reset(repositoryService, resourceConverterProvider, serverConverter);
        when(resourceConverterProvider.getToServerConverter(any(ClientResourceLookup.class))).thenReturn(serverConverter);
        when(resourceConverterProvider.getToClientConverter(any(Resource.class))).thenReturn(clientConverter);

        when(serverConverter.toServer(any(ClientResourceLookup.class))).thenReturn(res);
        when(clientConverter.toClient(any(Resource.class))).thenReturn(clientRes);

        folder.setURIString("/");
        clientRes.setUri("/test/folder");
        res.setURIString("/test/folder");
    }

    @Test(groups = "GET")
    public void getResourceDetails_xml()  throws Exception{
        final DataType serverObject = new DataTypeImpl();
        final ClientDataType clientDataType = new ClientDataType();
        final String uri = "/test/resource/uri";
        final String clientResourceType = "testClientResourceType";
        when(repositoryService.getResource(eq(Folder.SEPARATOR + uri))).thenReturn(serverObject);
        when(resourceConverterProvider.getToClientConverter(serverObject)).thenReturn((ToClientConverter)new ToClientConverter<Resource, ClientResourceLookup>() {
            @Override
            public ClientResourceLookup toClient(Resource serverObject) {
                return clientDataType;
            }

            @Override
            public String getClientResourceType() {
                return clientResourceType;
            }
        });
        final Response response = service.getResourceDetails(uri, MediaType.APPLICATION_XML);
        assertNotNull(response);
        assertSame(response.getEntity(), clientDataType);
        final List<Object> contentTypeHeaders = response.getMetadata().get(HttpHeaders.CONTENT_TYPE);
        assertNotNull(contentTypeHeaders);
        assertFalse(contentTypeHeaders.isEmpty());
        assertEquals(contentTypeHeaders.get(0), ResourceMediaType.RESOURCE_XML_TEMPLATE.replace(ResourceMediaType.RESOURCE_TYPE_PLACEHOLDER, clientResourceType));
        assertEquals(response.getStatus(), Response.Status.OK.getStatusCode());
    }

    @Test(groups = "GET")
    public void getResourceDetails_json() throws Exception{
        final DataType serverObject = new DataTypeImpl();
        final ClientDataType clientDataType = new ClientDataType();
        final String uri = "test/resource/uri";
        final String clientResourceType = "testClientResourceType";
        when(repositoryService.getResource(eq(Folder.SEPARATOR + uri))).thenReturn(serverObject);
        when(resourceConverterProvider.getToClientConverter(serverObject)).thenReturn((ToClientConverter)new ToClientConverter<Resource, ClientResourceLookup>() {
            @Override
            public ClientResourceLookup toClient(Resource serverObject) {
                return clientDataType;
            }

            @Override
            public String getClientResourceType() {
                return clientResourceType;
            }
        });
        final Response response = service.getResourceDetails(uri, MediaType.APPLICATION_JSON);
        assertNotNull(response);
        assertSame(response.getEntity(), clientDataType);
        final List<Object> contentTypeHeaders = response.getMetadata().get(HttpHeaders.CONTENT_TYPE);
        assertNotNull(contentTypeHeaders);
        assertFalse(contentTypeHeaders.isEmpty());
        assertEquals(contentTypeHeaders.get(0), ResourceMediaType.RESOURCE_JSON_TEMPLATE.replace(ResourceMediaType.RESOURCE_TYPE_PLACEHOLDER, clientResourceType));
        assertEquals(response.getStatus(), Response.Status.OK.getStatusCode());
    }

    @Test(expectedExceptions = {ResourceNotFoundException.class})
    public void getResourceDetails_null() throws Exception {
        final String uri = "test/resource/uri";
        when(repositoryService.getResource(Folder.SEPARATOR + uri)).thenThrow(ResourceNotFoundException.class);
        final Response response = service.getResourceDetails(uri, MediaType.APPLICATION_JSON);
    }

    @Test(groups = "DELETE")
    public void testDeleteResource_delegatilon() throws Exception{
        Response response = service.deleteResource(uri);

        verify(repositoryService).deleteResource(Folder.SEPARATOR + uri);
    }

    @Test(groups = "DELETE")
    public void testDeleteResource_Status() throws Exception{
        Response response = service.deleteResource(uri);

        assertEquals(response.getStatus(), HttpServletResponse.SC_NO_CONTENT);
    }

    @Test(groups = "POST")
    public void testCreateResource_convertsToServer() throws Exception {
        Resource created = new DataTypeImpl();
        Mockito.when(repositoryService.getResource(anyString())).thenReturn(res);
        Mockito.when(repositoryService.createResource(any(Resource.class), anyString(), anyString(), anyBoolean())).thenReturn(created);

        Response response = service.createResourceInternal(clientRes, uri, null, true);

        Mockito.verify(serverConverter).toServer(clientRes);
        Mockito.verify(clientConverter).toClient(created);
    }

    @Test(groups = "POST")
    public void testCreateResource_status() throws Exception {
        Mockito.when(repositoryService.getResource(anyString())).thenReturn((Folder) folder);
        Response response = service.createResourceInternal(clientRes, uri, null, true);

        assertEquals(response.getStatus(), HttpServletResponse.SC_CREATED);

    }

    @Test(groups = "COPY")
    public void testCopyResource_copy() throws Exception {
        Response response = service.copyResource(uri + uri, uri, true, true);

        verify(repositoryService).copyResource(uri, Folder.SEPARATOR + uri + uri, true, true);
    }

    @Test(groups = "COPY")
    public void testCopyResource_status() throws Exception {
        Response response = service.copyResource(uri, uri + uri, true, true);

        assertEquals(response.getStatus(), HttpServletResponse.SC_NO_CONTENT);
    }

    @Test(groups = "COPY", expectedExceptions = {ResourceNotFoundException.class}, dependsOnMethods = {"testCopyResource_copy"})
    public void testCopyResource_notFound() throws Exception {
        doThrow(ResourceNotFoundException.class).when(repositoryService).copyResource(anyString(), anyString(), anyBoolean(), anyBoolean());

        Response response = service.copyResource(uri, uri + uri, true, true);
    }

    @Test(groups = "COPY", expectedExceptions = {AccessDeniedException.class}, dependsOnMethods = {"testCopyResource_copy"})
    public void testCopyResource_accessDenied() throws Exception {
        doThrow(AccessDeniedException.class).when(repositoryService).copyResource(anyString(), anyString(), anyBoolean(), anyBoolean());

        Response response = service.copyResource(uri, uri + uri, true, true);
    }

    @Test(groups = "MOVE")
    public void testMoveResource_move() throws Exception {
        Response response = service.copyResource(uri + uri, uri, true, true);

        verify(repositoryService).copyResource(uri, Folder.SEPARATOR + uri + uri, true, true);
    }

    @Test(groups = "MOVE")
    public void testMoveResource_status() throws Exception {
        Response response = service.copyResource(uri, uri + uri, true, true);

        assertEquals(response.getStatus(), HttpServletResponse.SC_NO_CONTENT);
    }

    @Test(groups = "MOVE", expectedExceptions = {ResourceNotFoundException.class}, dependsOnMethods = {"testCopyResource_copy"})
    public void testMoveResource_notFound() throws Exception {
        doThrow(ResourceNotFoundException.class).when(repositoryService).copyResource(anyString(), anyString(), anyBoolean(), anyBoolean());

        Response response = service.copyResource(uri, uri + uri, true, true);
    }

    @Test(groups = "MOVE", expectedExceptions = {AccessDeniedException.class}, dependsOnMethods = {"testCopyResource_copy"})
    public void testMoveResource_accessDenied() throws Exception {
        doThrow(AccessDeniedException.class).when(repositoryService).copyResource(anyString(), anyString(), anyBoolean(), anyBoolean());

        Response response = service.copyResource(uri, uri + uri, true, true);
    }
}
