package com.jaspersoft.jasperserver.war.action;

import org.springframework.webflow.execution.Event;
import org.springframework.webflow.execution.RequestContext;
import org.springframework.context.MessageSource;
import org.springframework.context.i18n.LocaleContextHolder;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.json.JSONObject;
import org.json.JSONArray;
import com.jaspersoft.jasperserver.api.metadata.user.service.UserAuthorityService;
import com.jaspersoft.jasperserver.api.metadata.user.service.TenantService;
import com.jaspersoft.jasperserver.api.metadata.user.service.RoleManagerService;

import com.jaspersoft.jasperserver.api.metadata.user.domain.Role;
import com.jaspersoft.jasperserver.api.metadata.user.domain.Tenant;
import com.jaspersoft.jasperserver.api.metadata.common.service.PaginatedOperationResult;
import com.jaspersoft.jasperserver.api.metadata.common.service.impl.PaginationHelper;

import com.jaspersoft.jasperserver.war.common.ConfigurationBean;
import com.jaspersoft.jasperserver.war.common.UsersOperationResult;
import com.jaspersoft.jasperserver.war.helper.RoleJsonHelper;
import com.jaspersoft.jasperserver.war.helper.UserJsonHelper;

import java.util.*;
import java.io.Serializable;

/**
 * @author schubar
 */
public class RoleManagerAction extends BaseManagerAction {

    public static final String RM_DEFAULT_ROLE = "defaultRole";
    public static final String UM_CONFIGURATION = "configuration";
    public static final String UM_TENANT_ID = "tenantId";
    public static final String UM_ROLE_DETAILS = "roleDetails";
    public static final String UM_USER_NAME = "userName";
    public static final String UM_ROLE_NAME = "roleName";
    public static final String UM_USER_ROLES = "userRoles";
    public static final String UM_FIRST_RESULT = "firstResult";
    public static final String RM_ASSIGNED_USER_LIST = "assignedUserList";
    public static final String RM_AVAILABLE_USER_LIST = "availableUserList";
    public static final String RM_ASSIGN_USERS = "assignUsers";
    public static final String RM_UNASSIGN_USERS = "unassignUsers";
    public static final String RM_ROLE_USERS = "rmRoleUsers";
    public static final String RM_ROLE_USERS_RESTORE = "rmRoleUsersRestore";
    public static final String AJAX_RESPONSE_MODEL = "ajaxResponseModel";

    public static final String AJAX_RESPONSE_SUCCESS= "success";
    public static final String USER_NAME_SEPARATOR = "userNameSeparator";
    public static final String ROLE_NAME_NOT_SUPPORTED_SYMBOLS = "roleNameNotSupportedSymbols";
    public static final String USER_DEFAULT_ROLE = "userDefaultRole";
    public static final String USER_PASSWORD_MASK = "passwordMask";

    protected final Log log = LogFactory.getLog(this.getClass());


    private UserAuthorityService userService;
    private RoleManagerService roleManagerService;
    private MessageSource messages;

    private RoleJsonHelper roleHelper;
    private UserJsonHelper userHelper;

    public void setUserService(UserAuthorityService userService) {
        this.userService = userService;
    }

    public void setRoleManagerService(RoleManagerService roleManagerService) {
        this.roleManagerService = roleManagerService;
    }

    public void setMessages(MessageSource messages) {
        this.messages = messages;
    }

    public void setRoleHelper(RoleJsonHelper roleHelper) {
        this.roleHelper = roleHelper;
    }

    public void setUserHelper(UserJsonHelper userHelper) {
        this.userHelper = userHelper;
    }

    private String getTenantId(RequestContext context) {
        String tenantId = getDecodedRequestParameter(context, UM_TENANT_ID);

        return (tenantId != null && tenantId.length() > 0) ? tenantId : null;
    }

    private String getRoleJson(RequestContext context) {
        String json = context.getRequestParameters().get(UM_ROLE_DETAILS);

        return json;
    }

    private String getUserName(RequestContext context) {
        String name = getDecodedRequestParameter(context, UM_USER_NAME);

        return (name != null) ? name : "";
    }

    private String getRoleName(RequestContext context) {

        return getRoleName(context, true);
    }

    private String getRoleName(RequestContext context, boolean decode) {
        String name;
        if (decode) {
            name = getDecodedRequestParameter(context, UM_ROLE_NAME);
        } else {
            name = context.getRequestParameters().get(UM_ROLE_NAME);
        }

        return (name != null) ? name : "";
    }

    private Set getAssignUsers(RequestContext context) throws Exception {
        String assignedUsersParam = context.getRequestParameters().get(RM_ASSIGN_USERS);

        Set assignedUsers = new HashSet();

        if (assignedUsersParam != null && assignedUsersParam.length() > 0) {

            JSONArray assignedUsersArray = new JSONArray(assignedUsersParam);

            for (int i = 0; i < assignedUsersArray.length(); i ++) {

                assignedUsers.add(assignedUsersArray.getString(i));
            }
        }

        return assignedUsers;
    }

    private Set getUnassignUsers(RequestContext context) throws Exception {
        String unassignUsersParam = context.getRequestParameters().get(RM_UNASSIGN_USERS);

        Set unassignUsers = new HashSet();

        if (unassignUsersParam != null && unassignUsersParam.length() > 0) {

            JSONArray assignedUsersArray = new JSONArray(unassignUsersParam);

            for (int i = 0; i < assignedUsersArray.length(); i ++) {

                unassignUsers.add(assignedUsersArray.getString(i));
            }
        }

        return unassignUsers;
    }

    private JSONObject getAvailableUserList(RequestContext context) throws Exception {
        String availableUserListParam = context.getRequestParameters().get(RM_AVAILABLE_USER_LIST);

        JSONObject availableUserList;
        if (availableUserListParam != null && availableUserListParam.length() > 0) {

            availableUserList = new JSONObject(availableUserListParam);
        } else {
            availableUserList = new JSONObject();

            availableUserList.put(UM_FIRST_RESULT, 0);
            availableUserList.put(UM_USER_NAME, "");
        }

        return availableUserList;
    }

    private JSONObject getAssignedUserList(RequestContext context) throws Exception {
        String assignedUserListParam = context.getRequestParameters().get(RM_ASSIGNED_USER_LIST);

        JSONObject assignedUserList;
        if (assignedUserListParam != null && assignedUserListParam.length() > 0) {

            assignedUserList = new JSONObject(assignedUserListParam);
        } else {
            assignedUserList = new JSONObject();

            assignedUserList.put(UM_FIRST_RESULT, 0);
            assignedUserList.put(UM_USER_NAME, "");
        }

        return assignedUserList;
    }

    private UsersOperationResult getUsersOperationResult(RequestContext context, String roleNmae) throws Exception {
        UsersOperationResult usersOperationResult = (UsersOperationResult) getSession(context).get(RM_ROLE_USERS);

        if (usersOperationResult == null || !usersOperationResult.getRoleName().equals(roleNmae)) {

            usersOperationResult = new UsersOperationResult(roleNmae);
            setUsersOperationResult(context, usersOperationResult);
        }

        return usersOperationResult;
    }

    private void setUsersOperationResult(RequestContext context, UsersOperationResult usersOperationResult) throws Exception {

        if (usersOperationResult != null) {

            getSession(context).put(RM_ROLE_USERS, usersOperationResult);
        }

    }

    private int getFirstResult(RequestContext context) {
        String firstResult = getDecodedRequestParameter(context, UM_FIRST_RESULT);

        return (firstResult != null) ? Integer.valueOf(firstResult).intValue() : 0;
    }

    public Event initEvent(RequestContext context) throws Exception {

        JSONObject conf = new JSONObject();
        conf.put(ROLE_NAME_NOT_SUPPORTED_SYMBOLS, configuration.getRoleNameNotSupportedSymbols());
        conf.put(USER_NAME_SEPARATOR, configuration.getUserNameSeparator());
        conf.put(USER_DEFAULT_ROLE, configuration.getDefaultRole());
        conf.put(USER_PASSWORD_MASK, configuration.getPasswordMask());
        conf.put(SUPERUSER_ROLE, ROLE_SUPERUSER);
        conf.put(ADMIN_ROLE, ROLE_ADMINISTRATOR);

        context.getFlowScope().put(UM_CONFIGURATION, conf.toString());
        context.getFlowScope().put(ROLE_NAME_NOT_SUPPORTED_SYMBOLS, configuration.getRoleNameNotSupportedSymbols());

        context.getFlowScope().put(RM_DEFAULT_ROLE, getRoleName(context));
        context.getFlowScope().put(UM_CURRENT_USER, getCurrentUser(context));
        context.getFlowScope().put(UM_CURRENT_USER_ROLES,
                roleHelper.convertRoleListToJson(getCurrentUserRoles(context), null).toString());

        return success();
    }

    public Event loadRoles(RequestContext context) throws Exception {
        final String roleName = getRoleName(context);
        final String tenantId = getTenantId(context);
        int firstResult = getFirstResult(context);
        int maxResults = this.configuration.getRoleItemsPerPage();
//        int maxResult = 5;

        String responseModel;

        try {
            final Set tenantIdSet = getTenantsCriteriaSet(tenantId);

            PaginatedOperationResult result = PaginationHelper.paginatedGetOperationResult(firstResult, maxResults,
                    new PaginationHelper.JasperService() {
                            public List getResultList(int firstResult, int maxResults) {
                                return userService.getTenantVisibleRoles(null, tenantIdSet, roleName, firstResult, maxResults);
                            }

                            public int getResultCount() {
                                return userService.getTenantVisibleRolesCount(null, tenantIdSet, roleName);
                            }
            });

            List roles = result.getResult();
            firstResult = result.getFirstResult();

            JSONObject rolesJson;
            if (roles != null && !roles.isEmpty()) {

                rolesJson = roleHelper.createRolesResponseJson(roles, firstResult, maxResults, result.getTotalResults());
            } else {

                rolesJson = roleHelper.createEmptyRoleListResponseJson();
            }

            responseModel = roleHelper.createDataResponseModel(rolesJson);
        } catch (Exception e) {

            responseModel = createUnexpectedExceptionResponseModel(e.getMessage());
        }

        context.getRequestScope().put(AJAX_RESPONSE_MODEL, responseModel);

        return success();
    }

    public Event getRoleDetails(RequestContext context) throws Exception {
        String roleName = getRoleName(context);

        String responseModel;

        Role role;
        try {

            role = this.userService.getRole(null, roleName);

            if (roleName != null) {

                JSONObject usersJson = roleHelper.convertRoleToJson(role, null);
                responseModel = roleHelper.createDataResponseModel(usersJson);
            } else {

                throw new IllegalArgumentException("Cannot find role with name : " + roleName);
            }
        } catch (Exception e) {

            responseModel = createUnexpectedExceptionResponseModel(e.getMessage());
        }

        context.getRequestScope().put(AJAX_RESPONSE_MODEL, responseModel);

        return success();
    }

    public Event deleteRole(RequestContext context) throws Exception {
        String roleName = getRoleName(context, false);

        String responseModel = roleHelper.createSuccessResponseModel();

        try {

            if (roleName.length() > 0) {

                this.userService.deleteRole(null, roleName);
                setUsersOperationResult(context, new UsersOperationResult(roleName));                
            } else {

                throw new IllegalArgumentException("Role name is empty.");
            }

        } catch (Exception e) {

            responseModel = createUnexpectedExceptionResponseModel(e.getMessage());
        }

        context.getRequestScope().put(AJAX_RESPONSE_MODEL, responseModel);

        return success();
    }

    public Event isRoleExist(RequestContext context) throws Exception {
        String roleName = getRoleName(context);

        String responseModel;

        Role role;
        try {

            role = this.userService.getRole(null, roleName);

            JSONObject isRoleExistJson = roleHelper.createIsRoleExistJson((role != null));
            responseModel = roleHelper.createDataResponseModel(isRoleExistJson);
        } catch (Exception e) {

            responseModel = createUnexpectedExceptionResponseModel(e.getMessage());
        }

        context.getRequestScope().put(AJAX_RESPONSE_MODEL, responseModel);

        return success();
    }

    public Event createRole(RequestContext context) throws Exception {
        String roleJson = getRoleJson(context);

        String responseModel = roleHelper.createSuccessResponseModel();

        try {

            if (roleJson != null && roleJson.length() > 0) {

                Role role = roleHelper.convertJsonToRole(roleJson);

                if (role.getTenantId() != null) {

                    Tenant tenant = this.getTenantService().getTenant(null, role.getTenantId());

                    if (tenant == null) {

                        throw new IllegalArgumentException("Cannot find organization with id : " + role.getTenantId());
                    }
                }

                this.userService.putRole(null, role);

                setUsersOperationResult(context, new UsersOperationResult(role.getRoleName()));                                
            } else {

                throw new IllegalAccessException("Error when creating role");
            }

        } catch (Exception e) {

            responseModel = createUnexpectedExceptionResponseModel(e.getMessage());
        }

        context.getRequestScope().put(AJAX_RESPONSE_MODEL, responseModel);

        return success();
    }

    public Event initEdit(RequestContext context) throws Exception {
        final String roleName = getRoleName(context, false);

        setUsersOperationResult(context, new UsersOperationResult(roleName));

        log.info("Init edit for role : " + roleName);
        context.getRequestScope().put(AJAX_RESPONSE_MODEL, roleHelper.createSuccessResponseModel());

        return success();
    }

    public Event updateRole(RequestContext context) throws Exception {
        String roleName = getRoleName(context, false);
        String roleJson = getRoleJson(context);

        String responseModel = roleHelper.createSuccessResponseModel();

        try {

            Role role = roleHelper.convertJsonToRole(roleJson);

            if (role != null && role.getRoleName().trim().length() > 0) {
                final Set assignedUserNames = getUsersOperationResult(context, roleName).getAssignedUsers();
                final Set unassignedUserNames = getUsersOperationResult(context, roleName).getUnassignedUsers();

                this.roleManagerService.updateRole(null, roleName, role, assignedUserNames, unassignedUserNames);
//                this.userService.assignUsers(null, roleName, assignedUserNames);
//                this.userService.unassignUsers(null, roleName, unassignedUserNames);
//
//                this.userService.updateRole(null, roleName, role);
            } else {

                throw new IllegalArgumentException("Error when updating role details");
            }

        } catch (Exception e) {

            responseModel = createUnexpectedExceptionResponseModel(e.getMessage());
        }

        context.getRequestScope().put(AJAX_RESPONSE_MODEL, responseModel);

        return success();
    }
    
    public Event cancelEdit(RequestContext context) throws Exception {
        final String roleName = getRoleName(context, false);

        setUsersOperationResult(context, new UsersOperationResult(roleName));

        log.info("Cancel edit role : " + roleName);
        context.getRequestScope().put(AJAX_RESPONSE_MODEL, roleHelper.createSuccessResponseModel());

        return success();
    }

    public Event initChangeUsers(RequestContext context) throws Exception {
        final String roleName = getRoleName(context, false);

        UsersOperationResult usersOperationResult = getUsersOperationResult(context, roleName);
        getSession(context).put(RM_ROLE_USERS_RESTORE, usersOperationResult.clone());

        log.info("Init change users for role : " + roleName);
        context.getRequestScope().put(AJAX_RESPONSE_MODEL, roleHelper.createSuccessResponseModel());

        return success();
    }

    public Event revertUsersChanges(RequestContext context) throws Exception {
        final String roleName = getRoleName(context, false);

        UsersOperationResult usersOperationResultRestore =
                (UsersOperationResult) getSession(context).get(RM_ROLE_USERS_RESTORE);
        setUsersOperationResult(context, usersOperationResultRestore);

        log.info("Revert user changes for role : " + roleName);
        context.getRequestScope().put(AJAX_RESPONSE_MODEL, roleHelper.createSuccessResponseModel());

        return success();
    }

    public Event loadAvailableUsers(RequestContext context) throws Exception {
        final String roleName = getRoleName(context);
        final String userName = getUserName(context);
        int firstResult = getFirstResult(context);
        final String tenantId = getTenantId(context);

        String responseModel;

        try {
//            final Set tenantIdSet = getAllSubTenantsIdSet(tenantId);

            JSONObject availableUsers =
                    getUnassignedUsers(context, roleName, userName, firstResult);

            responseModel = roleHelper.createDataResponseModel(availableUsers);
        } catch (Exception e) {

            responseModel = createUnexpectedExceptionResponseModel(e.getMessage());
        }

        context.getRequestScope().put(AJAX_RESPONSE_MODEL, responseModel);

        return success();
    }

    public Event loadAssignedUsers(RequestContext context) throws Exception {
        final String roleName = getRoleName(context);
        final String userName = getUserName(context);
        int firstResult = getFirstResult(context);
        final String tenantId = getTenantId(context);

        String responseModel;

        try {
//            final Set tenantIdSet = getAllSubTenantsIdSet(tenantId);

            JSONObject assignedUsers =
                    getAssignedUsers(context, roleName, userName, firstResult);

            responseModel = roleHelper.createDataResponseModel(assignedUsers);
        } catch (Exception e) {

            responseModel = createUnexpectedExceptionResponseModel(e.getMessage());
        }

        context.getRequestScope().put(AJAX_RESPONSE_MODEL, responseModel);

        return success();
    }

    public Event updateRoleUsers(RequestContext context) throws Exception {
        final String roleName = getRoleName(context, false);
        final String tenantId = getTenantId(context);

        String responseModel;

        try {
            final Set tenantIdSet = null; //getAllSubTenantsIdSet(tenantId);

            assignUsers(context, roleName, getAssignUsers(context));
            unassignUsers(context, roleName, getUnassignUsers(context));

            final String availableListUserName = getUserName(getAvailableUserList(context));
            final int availableListFirstResult = getFirstResult(getAvailableUserList(context));

            final String assignedListUserName = getUserName(getAssignedUserList(context));
            final int assignedListFirstResult = getFirstResult(getAssignedUserList(context));

            JSONObject userChanges = new JSONObject();

            userChanges.put(RM_AVAILABLE_USER_LIST,
                   getUnassignedUsers(context, roleName, availableListUserName, availableListFirstResult));

            userChanges.put(RM_ASSIGNED_USER_LIST,
                    getAssignedUsers(context, roleName, assignedListUserName, assignedListFirstResult));

            responseModel = roleHelper.createDataResponseModel(userChanges);
        } catch (Exception e) {

            responseModel = createUnexpectedExceptionResponseModel(e.getMessage());
        }

        context.getRequestScope().put(AJAX_RESPONSE_MODEL, responseModel);

        return success();
    }

    private JSONObject getUnassignedUsers(RequestContext context, final String roleName, final String userName,
                                         final int firstResult) throws Exception {

        int maxResults = 10;

//        final String userName = getUserName(getAssignedUserList(context));
//        final int firstResult = getFirstResult(getAssignedUserList(context));
        final Set assignedUserNames = getUsersOperationResult(context, roleName).getAssignedUsers();
        final Set unassignedUserNames = getUsersOperationResult(context, roleName).getUnassignedUsers();

        PaginatedOperationResult result = roleManagerService.getUsersWithoutRole(null, roleName, userName,
                        assignedUserNames, unassignedUserNames,
                        firstResult, maxResults);

        JSONObject availableUserListJson;
        if (result.getResult() != null && !result.getResult().isEmpty()) {

            availableUserListJson = userHelper.createUsersResponseJson(
                    result.getResult(), result.getFirstResult(), maxResults, result.getTotalResults());
        } else {

            availableUserListJson = userHelper.createEmptyUserListResponseJson();
        }

        return availableUserListJson;
    }

    private JSONObject getAssignedUsers(RequestContext context, final String roleName, final String userName,
                                        final int firstResult) throws Exception {

        int maxResults = 10;

//        final String userName = getUserName(getAssignedUserList(context));
//        final int firstResult = getFirstResult(getAssignedUserList(context));
        final Set assignedUserNames = getUsersOperationResult(context, roleName).getAssignedUsers();
        final Set unassignedUserNames = getUsersOperationResult(context, roleName).getUnassignedUsers();

        PaginatedOperationResult result = roleManagerService.getUsersWithRole(null, roleName, userName,
                        assignedUserNames, unassignedUserNames,
                        firstResult, maxResults);

        JSONObject assignedUserListJson;
        if (result.getResult() != null && !result.getResult().isEmpty()) {

            assignedUserListJson = userHelper.createUsersResponseJson(
                    result.getResult(), result.getFirstResult(), maxResults, result.getTotalResults());
        } else {

            assignedUserListJson = userHelper.createEmptyUserListResponseJson();
        }

        return assignedUserListJson;
    }

    private int getFirstResult(JSONObject json) throws Exception{
        int firstResult = 0;

        if (json != null && json.has(UM_FIRST_RESULT)) {

            firstResult = json.getInt(UM_FIRST_RESULT);
        }

        return firstResult;
    }

    private String getUserName(JSONObject json) throws Exception{
        String userName = "";

        if (json != null && json.has(UM_USER_NAME)) {

            userName = json.getString(UM_USER_NAME);
        }

        return userName;
    }

    private void assignUsers(RequestContext context, String roleName, Set userNames) throws Exception{

        UsersOperationResult result = getUsersOperationResult(context, roleName);

        Set assigned = result.getAssignedUsers();

        assigned.addAll(userNames);

        Set unassigned = result.getUnassignedUsers();

        unassigned.removeAll(userNames);
    }

    private void unassignUsers(RequestContext context, String roleName, Set userNames) throws Exception{

        UsersOperationResult result = getUsersOperationResult(context, roleName);

        Set unassigned = result.getUnassignedUsers();

        unassigned.addAll(userNames);

        Set assigned = result.getAssignedUsers();

        assigned.removeAll(userNames);

    }

    private String createUnexpectedExceptionResponseModel(String desc) throws Exception {

        String message = messages.getMessage("jsp.userManager.unexpectedException", new Object[0],
                LocaleContextHolder.getLocale());

        JSONObject exceptionJson = roleHelper.createUnexpectedExceptionJson("", message, desc);

        return roleHelper.createErrorResponseModel(exceptionJson);
    }

}