/*
 * Copyright (C) 2005 - 2007 JasperSoft Corporation.  All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased a commercial license agreement from JasperSoft,
 * the following license terms apply:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published by
 * the Free Software Foundation.
 *
 * This program is distributed WITHOUT ANY WARRANTY; and without the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see http://www.gnu.org/licenses/gpl.txt
 * or write to:
 *
 * Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330,
 * Boston, MA  USA  02111-1307
 */
package com.jaspersoft.jasperserver.api.metadata.view.service.impl;

import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;

import javax.naming.NamingException;

import junit.textui.TestRunner;

import org.acegisecurity.AccessDeniedException;
import org.acegisecurity.Authentication;
import org.acegisecurity.acl.AclEntry;
import org.acegisecurity.acl.AclProvider;
import org.acegisecurity.acl.basic.BasicAclEntry;
import org.acegisecurity.acl.basic.SimpleAclEntry;
import org.acegisecurity.context.SecurityContextHolder;
import org.acegisecurity.providers.TestingAuthenticationToken;
import org.acegisecurity.userdetails.UserDetails;
import org.acegisecurity.userdetails.UserDetailsService;
import org.springframework.test.AbstractDependencyInjectionSpringContextTests;

import com.jaspersoft.jasperserver.api.metadata.common.domain.Folder;
import com.jaspersoft.jasperserver.api.metadata.common.domain.InternalURI;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Resource;
import com.jaspersoft.jasperserver.api.metadata.common.domain.ResourceLookup;
import com.jaspersoft.jasperserver.api.metadata.common.service.RepositoryService;
import com.jaspersoft.jasperserver.api.metadata.jasperreports.domain.ReportUnit;
import com.jaspersoft.jasperserver.api.metadata.user.domain.ObjectPermission;
import com.jaspersoft.jasperserver.api.metadata.user.domain.Role;
import com.jaspersoft.jasperserver.api.metadata.user.domain.User;
import com.jaspersoft.jasperserver.api.metadata.user.service.ObjectPermissionService;
import com.jaspersoft.jasperserver.api.metadata.user.service.UserAuthorityService;
import com.jaspersoft.jasperserver.api.metadata.user.service.impl.AclService;
import com.jaspersoft.jasperserver.api.metadata.view.domain.FilterCriteria;

/**
 * @author swood
 *
 */
public class ObjectPermissionServiceTest extends AbstractDependencyInjectionSpringContextTests {

    public static String reportsFolderPath = Folder.SEPARATOR + "reports";
    public static String reportUnitFolder = reportsFolderPath + Folder.SEPARATOR + "samples";
    public static String reportUnitName = "AllAccounts";
    public static String reportUnitPath = reportUnitFolder + Folder.SEPARATOR + reportUnitName;

    public static String testUserName = "TestUser";
    public static String adminUserName = "admin";

    public static String userRoleName = "ROLE_USER";
    public static String testRoleName = "ROLE_TEST";
    public static String administratorRoleName = "ROLE_ADMINISTRATOR";
    public static String portletRoleName = "ROLE_PORTLET";
    public static int readPermissionMask = SimpleAclEntry.READ;
    public static int adminPermissionMask = SimpleAclEntry.ADMINISTRATION;

    private Properties jdbcProps;

    RepositoryService repositoryService, checkSecurityRepositoryService, unsecuredRepoService;
    UserAuthorityService userAuthorityService;
    ObjectPermissionService objectPermissionService;

    ReportUnit unit;
    Role userRole, adminRole, testRole, portletRole;
    User adminUser, testUser;

    Folder root, reportsFolder;

    public RepositoryService getCheckSecurityRepositoryService() {
		return checkSecurityRepositoryService;
	}

	public void setCheckSecurityRepositoryService(
			RepositoryService checkSecurityRepositoryService) {
		this.checkSecurityRepositoryService = checkSecurityRepositoryService;
	}

	public RepositoryService getUnsecuredRepoService() {
		return unsecuredRepoService;
	}

	public void setUnsecuredRepoService(RepositoryService unsecuredRepoService) {
		this.unsecuredRepoService = unsecuredRepoService;
	}

	public RepositoryService getRepositoryService() {
		return repositoryService;
	}

	public void setRepositoryService(RepositoryService repositoryService) {
		this.repositoryService = repositoryService;
	}

	public ObjectPermissionService getObjectPermissionService() {
		return objectPermissionService;
	}

	public void setObjectPermissionService(
			ObjectPermissionService objectPermissionService) {
		this.objectPermissionService = objectPermissionService;
	}

	public UserAuthorityService getUserAuthorityService() {
		return userAuthorityService;
	}

	public void setUserAuthorityService(UserAuthorityService userAuthorityService) {
		this.userAuthorityService = userAuthorityService;
	}

	public ObjectPermissionServiceTest(String name) {
        super(name);
        setAutowireMode(AUTOWIRE_BY_NAME);
    }

    public static void main(String[] args) {
        TestRunner.run(UserAuthorityServiceTest.class);
    }


    protected String[] getConfigLocations() {
		try {
			loadJdbcProps();
		} catch (Exception e) {
			throw new RuntimeException(e);
		}

		// metadata.additionalSettings=hibernateRepositoryAdditionalSettings.xml

		if (jdbcProps.getProperty("metadata.additionalSettings") == null) {
			return
					new String[] {"hibernateConfig.xml", "userAuthorityService.xml", "repoService-Security.xml", "methodAndObjectSecurity.xml"};
		} else {
			return
					new String[] {"hibernateConfig.xml", jdbcProps.getProperty("metadata.additionalSettings"), "userAuthorityService.xml", "repoService-Security.xml", "methodAndObjectSecurity.xml"};
		}
    }

    protected Properties loadJdbcProps() throws IOException, FileNotFoundException, NamingException {
        jdbcProps = new Properties();
        String jdbcPropFile = System.getProperty("test.hibernate.jdbc.properties");
        BufferedInputStream is = new BufferedInputStream(new FileInputStream(jdbcPropFile));
        jdbcProps.load(is);
        is.close();
        return jdbcProps;
    }

	protected void onSetUp() throws Exception {
        System.out.println("ObjectPermissionServiceTest Setup");

        setUpRoles();
        setUpUsers();

        setAuthenticatedUser(adminUserName);

        root = unsecuredRepoService.getFolder(null, Folder.SEPARATOR);

        assertTrue("null root", root != null);

        reportsFolder = unsecuredRepoService.getFolder(null, reportsFolderPath);

        assertTrue("null reportsFolder", reportsFolder != null);

        // THESE ARE DEFAULTS FOR THE SYSTEM: not deleted

        // Create default permission of ADMINISTRATION at the root level

        createObjectPermission(root, adminRole, SimpleAclEntry.ADMINISTRATION);

        /*
         * Just for testing: make sure that userRole does not have access to root
         */
        deleteObjectPermission(root, userRole);

        // Set up some data

        unit = (ReportUnit) unsecuredRepoService.getResource(null, reportUnitPath);
        assertNotNull("Null ReportUnit", unit);
        assertEquals("Unmatched ReportUnit name", reportUnitName, unit.getName());

    }

    /**
     * ROLE_USER
     * ROLE_ADMINISTRATOR
     *
     */
    private void setUpRoles() {

        userRole = getOrCreateRole(userRoleName);

        adminRole = getOrCreateRole(administratorRoleName);

        testRole = getOrCreateRole(testRoleName);

        portletRole = getOrCreateRole(portletRoleName);
    }

    private Role getOrCreateRole(String roleName) {
        Role r = userAuthorityService.getRole(null, roleName);
        if (r == null) {
            r = userAuthorityService.newRole(null);
            r.setRoleName(roleName);
            r.setExternallyDefined(false);
            userAuthorityService.putRole(null, r);
        }
        return r;
    }

    /**
     * testUserName: TestUser has role: ROLE_TEST
     * adminUserName: admin has role: ROLE_ADMINISTRATOR
     *
     */
    private void setUpUsers() {
        testUser = findOrCreateUser(testUserName);

        userAuthorityService.addRole(null, testUser, testRole);
        adminUser = findOrCreateUser(adminUserName);

        userAuthorityService.addRole(null, adminUser, adminRole);

    }

    private User findOrCreateUser(String username) {
        User workingUser = userAuthorityService.getUser(null, username);
        if (workingUser == null) {
            workingUser = userAuthorityService.newUser(null);
            workingUser.setUsername(username);
            workingUser.setPassword(username);
            workingUser.setFullName(username + " user");
            workingUser.setEnabled(true);

            userAuthorityService.putUser(null, workingUser);
        }

        return workingUser;
    }

    public void onTearDown() {
        System.out.println("ObjectPermissionServiceTest Tear down");

        // Remove read permissions for the user at the report unit level
        User u = userAuthorityService.getUser(null, testUserName);
        Resource r = unsecuredRepoService.getResource(null, reportUnitPath);
        deleteObjectPermission(r, u);

        // Remove block permissions at the folder level
        System.out.println("Deleting permission for " + userRoleName + " and " + administratorRoleName + " for " + reportUnitFolder);

        Folder f = unsecuredRepoService.getFolder(null, reportUnitFolder);
        deleteObjectPermission(f, userRole);
        deleteObjectPermission(f, adminRole);

        // Remove block permissions at the folder level
        System.out.println("Deleting permission for " + userRoleName + " and " + Folder.SEPARATOR);
        f = unsecuredRepoService.getFolder(null, Folder.SEPARATOR);
        deleteObjectPermission(f, userRole);

        System.out.println("Removing role: " + adminRole.getRoleName() + " from user: " + testUser.getUsername());
        userAuthorityService.removeRole(null, testUser, adminRole);

        u = userAuthorityService.getUser(null, testUser.getUsername());

        System.out.println("test user assigned roles are: ");
        for (Iterator it = u.getRoles().iterator(); it.hasNext(); ) {
            Role role = (Role) it.next();
            System.out.println("\t" + role.getRoleName());
        }

        // remove anything else still hanging around we don't want
        userAuthorityService.deleteUser(null, testUserName);
        userAuthorityService.deleteRole(null, testRoleName);


        /*
         *  Leave entries in the database:
         *          root level ADMINISTRATION permissions for ROLE_ADMINISTRATOR
         */

        /*
         * Set read permissions at the root level for ROLE_USER
         */

        createObjectPermission(root, userRole, SimpleAclEntry.READ);
    }

    /**
     * For our report unit without a parent:
     *      ROLE_ADMINISTRATOR: SimpleAclEntry.ADMINISTRATION
     *
     *   and
     *
     * testUserName: TestUser has role: ROLE_TEST
     * adminUserName: admin has role: ROLE_ADMINISTRATOR
     */
    public void testObjectPermissionSetup() {

        setAuthenticatedUser(adminUserName);

        List l = objectPermissionService.getObjectPermissionsForObjectAndRecipient(null, unit, adminRole);

        assertTrue("getObjectPermissionsForObjectAndRecipient size not 0: " +
                    (l == null ? "null" : (new Integer(l.size())).toString()), l == null || l.size() == 0);
/*
        ObjectPermission op = (ObjectPermission) l.get(0);

        System.out.println("ObjectPermission for unit and adminRole: " + op);
*/
        ReportUnit ru = (ReportUnit) repositoryService.getResource(null, unit.getURI());

        assertTrue("Null report unit for " + unit.getURI(), ru != null);

        assertTrue(ru.getName().equals(reportUnitName));
/*
        assertTrue("op has a non-null recipient: " + op.getPermissionRecipient(), op.getPermissionRecipient() == null);
*/
        String p = ru.getParentFolder();

        Folder f = repositoryService.getFolder(null, p);

        assertTrue("Invalid folder: " + p, f != null);

        l = objectPermissionService.getObjectPermissionsForObject(null, f);

        System.out.println("ObjectPermission for unit folder: " + l);

        assertTrue("getObjectPermissionsForObject size not 0: " +
                    (l == null ? "null" : (new Integer(l.size())).toString()), l == null || l.size() == 0);
    }

    public void testAclAccess() {

        Authentication aUser = setAuthenticatedUser(adminUserName);

        BasicAclEntry[] aclEntries = ((AclService) objectPermissionService).getAcls(unit);

        assertTrue("aclEntries = null", aclEntries != null);

        printAclEntries("getAcls(unit)", aclEntries);

        assertTrue("aclEntries.length = " + aclEntries.length + " not 2", aclEntries.length == 2);

        AclEntry[] userEntries = ((AclProvider) objectPermissionService).getAcls(unit, aUser);
        assertTrue("userEntries = null", userEntries != null);

        printAclEntries("getAcls(unit, aUser)", (BasicAclEntry[]) userEntries);

        // 1 entry from the root for ROLE_ADMINISTRATOR
        assertTrue("userEntries.length = " + userEntries.length + " not 1", userEntries.length == 1);

        AclEntry found = null;
        for (int i = 0; i < userEntries.length && found == null; i++ ) {
            Object entry = ((BasicAclEntry) userEntries[i]).getRecipient();
            found = (entry instanceof Role && ((Role) entry).getRoleName().equals(administratorRoleName)) ?
                    userEntries[i] : null;
        }

        assertTrue("Role recipient not found", found != null);
        //assertTrue("Wrong role: " + ((BasicAclEntry) found).getRecipient(), ((Role) ((BasicAclEntry) found).getRecipient()).getRoleName().equals(administratorRoleName));
    }

    public void testMethodAndObjectAccess() {

        setAuthenticatedUser(adminUserName);

        /*
         * Now test access:
         *      admin is the user, that has a ROLE_ADMINISTRATOR

                HibernateRepositoryService.loadResourcesList=ROLE_ADMINISTRATOR,AFTER_ACL_COLLECTION_READ

         */
        assertNotNull("null testUserName", testUserName);

        Resource r = repositoryService.getResource(null, reportUnitPath); // should be OK - admin has access

        assertNotNull("null Resource", r);

        exerciseResourcesListAccess(adminUserName, true,
                "ERROR: Administrator should have access to report unit via role at root",
                "Administrator has access to report unit");

        //System.out.println("Administrator has access to report unit");

        /*
         * The TestUser has ROLE_TEST role, but
         * does not have ROLE_ADMINISTRATOR access to the report unit, so it will not appear in the list
         */

        exerciseResourcesListAccess(testUserName, false,
                "ERROR: ROLE_TEST should not have access to report unit because there is no permissions for ROLE_TEST or this user",
                "ROLE_TEST does not have access to report unit");

        //System.out.println("ROLE_TEST does not have access to report unit");

        /*
         * Now add permissions for the testUserName to the report unit
         */
        ObjectPermission op = createObjectPermission(r, testUser, SimpleAclEntry.READ);

        /*
         * Should be able to access now
         */

        exerciseResourcesListAccess(testUserName, true,
                "ERROR: Test user should have specific access to report unit",
                "Test user has access to report unit");

        //System.out.println("Test user has access to report unit");

        objectPermissionService.deleteObjectPermission(null, op);

        // Give the testUserName a ROLE_ADMINISTRATOR role

        userAuthorityService.addRole(null, testUser, adminRole);

        /*
         * Should still be able to access
         */
        exerciseResourcesListAccess(testUserName, true,
                "ERROR: ROLE_ADMINISTRATOR should have access to root level",
                "ROLE_ADMINISTRATOR given access to root level");

        //System.out.println("ROLE_ADMINISTRATOR given access to root level");

        Folder f = repositoryService.getFolder(null, reportUnitFolder);

        // Block permissions at the folder level

        op = createObjectPermission(f, adminRole, SimpleAclEntry.NOTHING);

        /*
         * Should not be able to access
         */
        exerciseResourcesListAccess(testUserName, false,
                "ERROR: ROLE_ADMINISTRATOR should have no access to report unit folder level",
                "ROLE_ADMINISTRATOR given no access to report unit folder level");

        //System.out.println("ROLE_USER given no access to report unit folder level");

        // Should not be able to access: throw exception

        try {
            checkSecurityRepositoryService.getResource(null, reportUnitFolder);
            fail("ERROR: had access to " + reportUnitFolder);
        } catch (Exception e) {
            System.out.println("OK: had no access to " + reportUnitFolder + ": " + e.getMessage());
        }

        // Allow permissions at the folder level

        op = createObjectPermission(f, adminRole, SimpleAclEntry.ADMINISTRATION);

        // Check again

        try {
            checkSecurityRepositoryService.getResource(null, reportUnitFolder);
            System.out.println("OK: had access to " + reportUnitFolder);
        } catch (Exception e) {
            fail("ERROR: had no access to " + reportUnitFolder + ": " + e.getMessage());
            e.printStackTrace();
        }
    }

    private Authentication setAuthenticatedUser(String username) {
        UserDetails userDetails = ((UserDetailsService)userAuthorityService).loadUserByUsername(username);
        Authentication aUser = new TestingAuthenticationToken(userDetails, userDetails.getPassword(), userDetails.getAuthorities());
        aUser.setAuthenticated(true);

        SecurityContextHolder.getContext().setAuthentication(aUser);
        return aUser;
    }

    private void printAclEntries(String name, BasicAclEntry[] aclEntries) {

        System.out.println("AclEntry set: " + name);

        for (int i = 0; aclEntries != null && aclEntries.length > 0 && i < aclEntries.length; i++) {
            System.out.println("aclEntries[" + i + "]: " + aclEntries[i].getAclObjectIdentity() +
                    "\n\tparent: " + aclEntries[i].getAclObjectParentIdentity() +
                    "\n\trecipient: " + aclEntries[i].getRecipient() +
                    "\n\tmask: " + aclEntries[i].getMask());
        }

    }

    private void exerciseResourcesListAccess(String userName, boolean expectToFind,
            String testDescription, String successMessage) {
        FilterCriteria criteria = FilterCriteria.createFilter(ReportUnit.class);
        List l;

        setAuthenticatedUser(userName);

        Iterator it;
        boolean found = false;

        try {
            l = repositoryService.loadResourcesList(criteria);
            System.out.println(testDescription +  "\nloadResourcesList size: " + l.size());
            it = l.iterator();
            found = false;
            while (it.hasNext() && !found) {
                ResourceLookup rlu = (ResourceLookup) it.next();
                found = rlu.getURIString().equals(reportUnitPath);
            }

            // If we found and did not expect to, or we did not find and we expected to

            if (found != expectToFind) {
                fail(userName + ": loadResourcesList " + (expectToFind ? "did not contain " : "contained un") + "expected report unit for " + reportUnitName);
            }
        } catch (AccessDeniedException ex) {
            fail(userName + ": loadResourcesList AccessDeniedException unexpected");
        }

        System.out.println(successMessage);

    }

    private ObjectPermission createObjectPermission(InternalURI target, Object recipient, int permissionMask) {
        ObjectPermission permission = objectPermissionService.newObjectPermission(null);
        permission.setURI(target.getURI());
        permission.setPermissionRecipient(recipient);
        permission.setPermissionMask(permissionMask);
        objectPermissionService.putObjectPermission(null, permission);
        return permission;
    }

    private void deleteObjectPermission(InternalURI target, Object recipient) {

        ObjectPermission op = objectPermissionService.newObjectPermission(null);

        String targetURI = target.getURI();
        op.setURI(targetURI);
        op.setPermissionRecipient(recipient);

        System.out.println("deleteObjectPermission: about to get: " + targetURI + ", recipient: " + recipient);
        ObjectPermission op2 = objectPermissionService.getObjectPermission(null, op);
        if (op2 != null) {
            System.out.println("deleteObjectPermission: got " + op2 + ". about to delete: " + targetURI + ", recipient: " + recipient);
            objectPermissionService.deleteObjectPermission(null, op);
            System.out.println("deleted permission for uri: " + targetURI + ", recipient: " + recipient);
        } else {
            System.out.println("Can't delete permission for uri: " + targetURI + ", recipient: " + recipient + " because it does not exist");
        }

    }

}
