/*
 * Copyright (C) 2005 - 2007 JasperSoft Corporation.  All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased a commercial license agreement from JasperSoft,
 * the following license terms apply:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published by
 * the Free Software Foundation.
 *
 * This program is distributed WITHOUT ANY WARRANTY; and without the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see http://www.gnu.org/licenses/gpl.txt
 * or write to:
 *
 * Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330,
 * Boston, MA  USA  02111-1307
 */
package com.jaspersoft.jasperserver.api.metadata.view.service.impl;

import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import javax.naming.NamingException;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;

import junit.framework.TestCase;
import junit.textui.TestRunner;

import org.acegisecurity.Authentication;
import org.acegisecurity.GrantedAuthority;
import org.acegisecurity.GrantedAuthorityImpl;
import org.acegisecurity.context.SecurityContextHolder;
import org.acegisecurity.ui.webapp.AuthenticationProcessingFilter;
import org.acegisecurity.util.FilterChainProxy;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.mock.web.MockHttpServletRequest;
import org.springframework.mock.web.MockHttpServletResponse;
import org.springframework.test.AbstractDependencyInjectionSpringContextTests;

import com.jaspersoft.jasperserver.api.metadata.user.domain.Role;
import com.jaspersoft.jasperserver.api.metadata.user.domain.User;
import com.jaspersoft.jasperserver.api.metadata.user.service.UserAuthorityService;

/**
 * Test authentication with principals that are strings and User Details. We need to
 * exercise the MetadataAuthenticationProcessingFilter via the Acegi FilterChainProxy.
 *
 * Test using an external service and synchronizing roles.
 *
 * Create 2 dummy external services
 *      - principal is a string
 *      - principal is a userDetails
 *
 * users will be indicated as being external. Set their roles.
 *
 * Use each of these via a ProviderManager, and the MetadataAuthenticationProcessingFilter.
 *
 * Reauthenticate with changed roles and check that the role synch happens correctly.
 *
 * Add a JS internal role and check that a login does not remove it
 *
 * @author swood
 * @version $Id: AuthenticationTest.java 15184 2009-02-28 20:46:26Z afomin $
 */
public class AuthenticationTest extends AbstractDependencyInjectionSpringContextTests {

    private static Log log = LogFactory.getLog(AuthenticationTest.class);

    public static final String testExternalUserName = "TestExternalUser";
    public static final String testExternalPassword = "password";
    public static final String testExternalRoleNamePrefix = "TestExternalRole";
    public static final String testInternalRoleNamePrefix = "ROLE_TestInternalRole";

    public static final String testUserDetailsUserName = "tomcat";
    public static final String testUserDetailsPassword = "tomcat";

	Properties jdbcProps;

    UserAuthorityService userAuthorityService;

    public UserAuthorityService getUserAuthorityService() {
		return userAuthorityService;
	}

	public void setUserAuthorityService(UserAuthorityService userAuthorityService) {
		this.userAuthorityService = userAuthorityService;
	}

	public AuthenticationTest(String name) {
        super(name);
        setAutowireMode(AUTOWIRE_BY_NAME);
    }

    public static void main(String[] args) {
        TestRunner.run(AuthenticationTest.class);
    }

    protected String[] getConfigLocations() {
		try {
			loadJdbcProps();
		} catch (Exception e) {
			throw new RuntimeException(e);
		}

		// metadata.additionalSettings=hibernateRepositoryAdditionalSettings.xml

		if (jdbcProps.getProperty("metadata.additionalSettings") == null) {
			return
					new String[] {"hibernateConfig.xml", "userAuthorityService.xml", "viewService.xml",
                        "applicationContext-testProviders.xml"};
		} else {
			return
					new String[] {"hibernateConfig.xml", jdbcProps.getProperty("metadata.additionalSettings"), "userAuthorityService.xml", "viewService.xml",
                        "applicationContext-testProviders.xml"};
		}
    }

    protected Properties loadJdbcProps() throws IOException, FileNotFoundException, NamingException {
        jdbcProps = new Properties();
        String jdbcPropFile = System.getProperty("test.hibernate.jdbc.properties");
        BufferedInputStream is = new BufferedInputStream(new FileInputStream(jdbcPropFile));
        jdbcProps.load(is);
        is.close();
        return jdbcProps;
    }

    public void onSetUp() throws Exception {
        log.debug("Setup");
    }

    public void onTearDown() {
        log.debug("Tear down");

    	//if (jdbcProps.getProperty("test.databaseFlavor") != null &&
    	//		(jdbcProps.getProperty("test.databaseFlavor").startsWith("postgres") ||
    	//		 jdbcProps.getProperty("test.databaseFlavor").startsWith("oracle") ||
    	//		 jdbcProps.getProperty("test.databaseFlavor").startsWith("sqlserver"))) {   // TODO: fix and remove
    	//	log.warn("DATABASE FLAVOR IS POSTGRESQL, ORACLE, SQLSERVER: Skipping test: " + "tearDown");
		//	return;
    	//}
        
        // get rid of objects we don't want in samples

        userAuthorityService.deleteUser(null, testExternalUserName);

        User workingUser = userAuthorityService.getUser(null, testUserDetailsUserName);
        Role workRole = userAuthorityService.getRole(null, "ROLE_" + testExternalRoleNamePrefix + "0");
        userAuthorityService.removeRole(null, workingUser, workRole);

        for (int i = 0; i < 3; i++) {
            userAuthorityService.deleteRole(null, "ROLE_" + testExternalRoleNamePrefix + i);
            userAuthorityService.deleteRole(null, testInternalRoleNamePrefix + i);
        }
    }

    public Role addRole(User user, String roleName) {
        log.debug("addRole");

        Role role = userAuthorityService.getRole(null, roleName);

        if (role == null) {
                role = userAuthorityService.newRole(null);
                //System.out.println("role class: " + role.getClass().getName());
                role.setRoleName(roleName);
                role.setExternallyDefined(false);

                userAuthorityService.putRole(null, role);
        }

        userAuthorityService.addRole(null, user, role);
        return role;
    }

    private User findOrCreateUser(String username, String password) {
        User workingUser = userAuthorityService.getUser(null, username);
        if (workingUser == null) {
                workingUser = userAuthorityService.newUser(null);
                workingUser.setUsername(username);
                workingUser.setPassword((password == null ? username : password));
                workingUser.setFullName(username + " user");
                workingUser.setEnabled(true);
                workingUser.setExternallyDefined(false);

                userAuthorityService.putUser(null, workingUser);
        }

        return workingUser;
    }

    public void loginUser(String userName, String password, int noOfExternalRoles) throws Exception {

       FilterChainProxy filterChainProxy = (FilterChainProxy) applicationContext.getBean("filterChainProxy",
            FilterChainProxy.class);

        filterChainProxy.init(new MockFilterConfig());

        GrantedAuthority[] grantedAuthorities = generateGrantedAuthorities(testExternalRoleNamePrefix, noOfExternalRoles);

        MockStringAuthenticationProvider mockAuthenticationProvider = (MockStringAuthenticationProvider) applicationContext.getBean("stringPrincipalAuthenticationProvider",
            MockStringAuthenticationProvider.class);

        // Do the authentication
        mockAuthenticationProvider.setGrantedAuthorities(grantedAuthorities);

        AuthenticationProcessingFilter authenticationProcessingFilter = (AuthenticationProcessingFilter) applicationContext.getBean("authenticationProcessingFilter",
            AuthenticationProcessingFilter.class);

        MockHttpServletRequest request = new MockHttpServletRequest();
        request.setServletPath(authenticationProcessingFilter.getFilterProcessesUrl());
        request.setRequestURI(authenticationProcessingFilter.getFilterProcessesUrl());

        log.debug("request.getContextPath(): '" + request.getContextPath() + "', request.getRequestURI(): '" + request.getRequestURI() + "'");

        request.setParameter(AuthenticationProcessingFilter.ACEGI_SECURITY_FORM_USERNAME_KEY, userName);
        request.setParameter(AuthenticationProcessingFilter.ACEGI_SECURITY_FORM_PASSWORD_KEY, password);

        MockHttpServletResponse response = new MockHttpServletResponse();
        MockFilterChain chain = new MockFilterChain(true);

        // Do the authentication with the AuthenticationProcessingFilter
        filterChainProxy.doFilter(request, response, chain);

        Authentication auth = SecurityContextHolder.getContext().getAuthentication();

        assertTrue("login not successful", auth != null);

        // Now we are authenticated, run the filter again to get to the MetadataAuthenticationProcessingFilter.
        // This is what happens with the normal login process

        request = new MockHttpServletRequest();
        request.setServletPath("/a/random/url");
        response = new MockHttpServletResponse();
        chain = new MockFilterChain(true);

        filterChainProxy.doFilter(request, response, chain);

    }

    public void testRolesForExternalStringPrincipal() throws Exception {

    	//if (jdbcProps.getProperty("test.databaseFlavor") != null &&
    	//		(jdbcProps.getProperty("test.databaseFlavor").startsWith("postgres") ||
    	//		 jdbcProps.getProperty("test.databaseFlavor").startsWith("oracle") ||
    	//		 jdbcProps.getProperty("test.databaseFlavor").startsWith("sqlserver"))) {   // TODO: fix and remove
    	//	log.warn("DATABASE FLAVOR IS POSTGRESQL, ORACLE, SQLSERVER: Skipping test: " + "testRolesForExternalStringPrincipal");
		//	return;
    	//}
    	
        loginUser(testExternalUserName, testExternalPassword, 3);

        User workingUser = userAuthorityService.getUser(null, testExternalUserName);

        // Because we are adding the default ROLE_USER, there will be 4 roles
        checkUserAndRoles(workingUser, "first login", 3, 1);

        // Logout and login with less roles

        SecurityContextHolder.getContext().setAuthentication(null);

        loginUser(testExternalUserName, testExternalPassword, 2);

        workingUser = userAuthorityService.getUser(null, testExternalUserName);

        checkUserAndRoles(workingUser, "second login", 2, 1);

        // Add an internal role to the user and see what happens when they login again

        addRole(workingUser, testInternalRoleNamePrefix + "0");

        SecurityContextHolder.getContext().setAuthentication(null);

        loginUser(testExternalUserName, testExternalPassword, 2);

        workingUser = userAuthorityService.getUser(null, testExternalUserName);

        checkUserAndRoles(workingUser, "third login", 2, 2);
    }

    public void testRolesForExternalUserDetailsPrincipal() throws Exception {

    	//if (jdbcProps.getProperty("test.databaseFlavor") != null &&
    	//		(jdbcProps.getProperty("test.databaseFlavor").startsWith("postgres") ||
    	//		 jdbcProps.getProperty("test.databaseFlavor").startsWith("oracle") ||
    	//		 jdbcProps.getProperty("test.databaseFlavor").startsWith("sqlserver"))) {   // TODO: fix and remove
    	//	log.warn("DATABASE FLAVOR IS POSTGRESQL, ORACLE, SQLSERVER: Skipping test: " + "testRolesForExternalUserDetailsPrincipal");
		//	return;
    	//}
    	
        loginUser(testUserDetailsUserName, testUserDetailsPassword, 0);

        User workingUser = userAuthorityService.getUser(null, testUserDetailsUserName);

        // Because we are adding the default ROLE_USER, there will be 2 roles
        checkUserAndRoles(workingUser, "Userdetails first login", 1, 1);

        // Logout and login

        SecurityContextHolder.getContext().setAuthentication(null);

        loginUser(testUserDetailsUserName, testUserDetailsPassword, 0);

        workingUser = userAuthorityService.getUser(null, testUserDetailsUserName);

        checkUserAndRoles(workingUser, "Userdetails second login", 1, 1);

        // Add an internal role to the user and see what happens when they login again

        addRole(workingUser, testInternalRoleNamePrefix + "0");

        SecurityContextHolder.getContext().setAuthentication(null);

        loginUser(testUserDetailsUserName, testUserDetailsPassword, 0);

        workingUser = userAuthorityService.getUser(null, testUserDetailsUserName);

        checkUserAndRoles(workingUser, "Userdetails third login", 1, 2);
    }

    private void checkUserAndRoles(User user, String prefix, int expectedExternalCount, int expectedInternalCount) {

        assertTrue(prefix + ": missing user", user != null);
        assertTrue(prefix + ": not external", user.isExternallyDefined());

        Set roles = user.getRoles();

        assertTrue(prefix + ": not " + (expectedExternalCount + expectedInternalCount) + " roles, got " + roles.size(),
                expectedExternalCount + expectedInternalCount == roles.size());

        int externalCount = 0, internalCount = 0;

        for (Iterator it = roles.iterator(); it.hasNext(); ) {
            Role r = (Role) it.next();

            if (r.isExternallyDefined()) {
                externalCount++;
            } else {
                internalCount++;
            }
        }
        assertTrue(prefix + ": saw " + externalCount + " external roles, not " + expectedExternalCount, externalCount == expectedExternalCount);
        assertTrue(prefix + ": saw " + internalCount + " internal roles, not " + expectedInternalCount, internalCount == expectedInternalCount);

    }

    private static GrantedAuthority[] generateGrantedAuthorities(String prefix, int length) {
        GrantedAuthority[] newAuthorities = new GrantedAuthority[length];

        for (int i = 0; i < length; i++) {
            newAuthorities[i] = new GrantedAuthorityImpl(prefix + i);
        }
        return newAuthorities;
    }

    /**
     * From Acegi test cases
     *
     * @author Ben Alex
     * @version $Id: AuthenticationTest.java 15184 2009-02-28 20:46:26Z afomin $
     */
    public class MockFilterConfig implements FilterConfig {
        //~ Instance fields ========================================================

        private Map map = new HashMap();

        //~ Methods ================================================================

        public String getFilterName() {
            throw new UnsupportedOperationException("mock method not implemented");
        }

        public String getInitParameter(String arg0) {
            Object result = map.get(arg0);

            if (result != null) {
                return (String) result;
            } else {
                return null;
            }
        }

        public Enumeration getInitParameterNames() {
            throw new UnsupportedOperationException("mock method not implemented");
        }

        public void setInitParmeter(String parameter, String value) {
            map.put(parameter, value);
        }

        public ServletContext getServletContext() {
            throw new UnsupportedOperationException("mock method not implemented");
        }
    }

    /**
     * A mock <code>FilterChain</code>.
     *
     * From Acegi test cases
     *
     * @author Ben Alex
     * @version $Id: AuthenticationTest.java 15184 2009-02-28 20:46:26Z afomin $
     */
    public class MockFilterChain implements FilterChain {
        //~ Instance fields ========================================================

        private boolean expectToProceed;

        //~ Constructors ===========================================================

        public MockFilterChain(boolean expectToProceed) {
            this.expectToProceed = expectToProceed;
        }

        //~ Methods ================================================================

        public void doFilter(ServletRequest request, ServletResponse response)
                throws IOException, ServletException {
            if (expectToProceed) {
                TestCase.assertTrue(true);
            } else {
                TestCase.fail("Did not expect filter chain to proceed");
            }
        }
    }

}
