/*
// $Id: //open/mondrian/testsrc/main/mondrian/test/DialectTest.java#21 $
// This software is subject to the terms of the Common Public License
// Agreement, available at the following URL:
// http://www.opensource.org/licenses/cpl.html.
// Copyright (C) 2007-2009 Julian Hyde
// All Rights Reserved.
// You must accept the terms of that agreement to use this software.
*/
package mondrian.test;

import junit.framework.TestCase;
import junit.framework.AssertionFailedError;

import java.sql.*;
import java.util.*;

import mondrian.olap.Util;
import mondrian.spi.Dialect;
import mondrian.spi.DialectManager;
import mondrian.spi.impl.MySqlDialect;
import mondrian.spi.impl.InfobrightDialect;

import javax.sql.DataSource;
import mondrian.spi.impl.NetezzaDialect;
import mondrian.spi.impl.PostgreSqlDialect;

/**
 * Unit test which checks that {@link mondrian.spi.Dialect}
 * accurately represents the capabilities of the underlying database.
 *
 * <p>The existing mondrian tests, when run on various databases and drivers,
 * make sure that Dialect never over-states the capabilities of a particular
 * database. But sometimes they under-state a database's capabilities: for
 * example, MySQL version 3 did not allow subqueries in the FROM clause, but
 * version 4 does. This test helps ensure that mondrian is using the full
 * capabilities of each database.
 *
 * <p><strong>NOTE: If you see failures in this test, let the mondrian
 * developers know!</strong>
 * You may be running a version of a database which no one has
 * tried before, and which has more capabilities than we expect. If you tell us
 * about them, we can change mondrian to use those features.</p>
 *
 * @author jhyde
 * @version $Id: //open/mondrian/testsrc/main/mondrian/test/DialectTest.java#21 $
 * @since May 18, 2007
 */
public class DialectTest extends TestCase {
    private Connection connection;
    private Dialect dialect;
    private static final String INFOBRIGHT_UNSUPPORTED =
        "The query includes syntax that is not supported by the Infobright"
        + " Optimizer. Either restructure the query with supported syntax, or"
        + " enable the MySQL Query Path in the brighthouse.ini file to execute"
        + " the query with reduced performance.";

    /**
     * Creates a DialectTest.
     *
     * @param name Test case name
     */
    public DialectTest(String name) {
        super(name);
    }

    protected DataSource getDataSource() {
        TestContext.instance().clearConnection();
        return TestContext.instance().getConnection().getDataSource();
    }

    protected void tearDown() throws Exception {
        if (connection != null) {
            try {
                connection.close();
            } catch (SQLException e) {
                // ignore
            } finally {
                connection = null;
            }
        }
        super.tearDown();
    }

    protected Dialect getDialect() {
        if (dialect == null) {
            dialect = DialectManager.createDialect(getDataSource(), null);
        }
        return dialect;
    }

    protected Connection getConnection() {
        if (connection == null) {
            try {
                connection = getDataSource().getConnection();
            } catch (SQLException e) {
                throw Util.newInternal(e, "while creating connection");
            }
        }
        return connection;
    }

    public void testDialectVsDatabaseProduct() throws SQLException {
        final Dialect dialect = getDialect();
        final Dialect.DatabaseProduct databaseProduct =
            dialect.getDatabaseProduct();
        final DatabaseMetaData databaseMetaData = getConnection().getMetaData();
        switch (databaseProduct) {
        case MYSQL:
            // Dialect has identified that it is MySQL.
            assertTrue(dialect instanceof MySqlDialect);
            assertFalse(dialect instanceof InfobrightDialect);
            assertFalse(MySqlDialect.isInfobright(databaseMetaData));
            assertEquals("MySQL", databaseMetaData.getDatabaseProductName());
            break;
        case INFOBRIGHT:
            // Dialect has identified that it is MySQL.
            assertTrue(dialect instanceof MySqlDialect);
            assertTrue(dialect instanceof InfobrightDialect);
            assertTrue(MySqlDialect.isInfobright(databaseMetaData));
            assertEquals("MySQL", databaseMetaData.getDatabaseProductName());
            break;
        case POSTGRESQL:
            // Dialect has identified that it is PostgreSQL.
            assertTrue(dialect instanceof PostgreSqlDialect);
            assertFalse(dialect instanceof NetezzaDialect);
            assertTrue(databaseMetaData.getDatabaseProductName()
                    .indexOf("PostgreSQL") >= 0);
            break;
        case NETEZZA:
            // Dialect has identified that it is Netezza and a sub class of
            // PostgreSql.
            assertTrue(dialect instanceof PostgreSqlDialect);
            assertTrue(dialect instanceof NetezzaDialect);
            assertTrue(databaseMetaData.getDatabaseProductName()
                    .indexOf("Netezza") >= 0);
            break;
        default:
            // Neither MySQL nor Infobright.
            assertFalse(dialect instanceof MySqlDialect);
            assertFalse(dialect instanceof InfobrightDialect);
            assertNotSame("MySQL", databaseMetaData.getDatabaseProductName());
            break;
        }
    }

    public void testAllowsCompoundCountDistinct() {
        String sql =
            dialectize(
                "select count(distinct [customer_id], [product_id])\n"
                + "from [sales_fact_1997]");
        if (getDialect().allowsCompoundCountDistinct()) {
            assertQuerySucceeds(sql);
        } else {
            String[] errs = {
                // oracle
                "(?s)ORA-00909: invalid number of arguments.*",
                // derby
                "Syntax error: Encountered \",\" at line 1, column 36.",
                // access
                "\\[Microsoft\\]\\[ODBC Microsoft Access Driver\\] Syntax error \\(missing operator\\) in query expression '.*'.",
                // infobright
                INFOBRIGHT_UNSUPPORTED,
                // postgres
                "ERROR: function count\\(integer, integer\\) does not exist",
                // LucidDb
                ".*Invalid number of arguments to function 'COUNT'. Was expecting 1 arguments",
                // teradata
                ".*Syntax error: expected something between the word 'customer_id' and ','\\..*",
                // netezza
                "(?s).*ERROR:  Function 'COUNT', number of parameters greater than the maximum \\(1\\).*",
            };
            assertQueryFails(sql, errs);
        }
    }

    public void testAllowsCountDistinct() {
        String sql1 =
            dialectize(
                "select count(distinct [customer_id]) from [sales_fact_1997]");
        // one distinct-count and one nondistinct-agg
        String sql2 =
            dialectize(
                "select count(distinct [customer_id]),\n"
                + " sum([time_id])\n"
                + "from [sales_fact_1997]");
        if (getDialect().allowsCountDistinct()) {
            assertQuerySucceeds(sql1);
            assertQuerySucceeds(sql2);
        } else {
            String[] errs = {
                // access
                "\\[Microsoft\\]\\[ODBC Microsoft Access Driver\\] Syntax error \\(missing operator\\) in query expression '.*'."
            };
            assertQueryFails(sql1, errs);
            assertQueryFails(sql2, errs);
        }
    }

    public void testAllowsMultipleCountDistinct() {
        // multiple distinct-counts
        String sql1 =
            dialectize(
                "select count(distinct [customer_id]),\n"
                + " count(distinct [time_id])\n"
                + "from [sales_fact_1997]");
        // multiple distinct-counts with group by and other aggs
        String sql3 =
            dialectize(
                "select [unit_sales],\n"
                + " count(distinct [customer_id]),\n"
                + " count(distinct [product_id])\n"
                + "from [sales_fact_1997]\n"
                + "where [time_id] in (371, 372)\n"
                + "group by [unit_sales]");
        if (getDialect().allowsMultipleCountDistinct()) {
            assertQuerySucceeds(sql1);
            assertQuerySucceeds(sql3);
            assertTrue(getDialect().allowsCountDistinct());
        } else {
            String[] errs = {
                // derby
                "Multiple DISTINCT aggregates are not supported at this time.",
                // access
                "\\[Microsoft\\]\\[ODBC Microsoft Access Driver\\] Syntax error \\(missing operator\\) in query expression '.*'."
            };
            assertQueryFails(sql1, errs);
            assertQueryFails(sql3, errs);
        }
    }

    public void testAllowsDdl() {
        int phase = 0;
        SQLException e = null;
        Statement stmt = null;
        try {
            String dropSql = dialectize("drop table [foo]");
            String createSql = dialectize("create table [foo] ([i] integer)");
            stmt = getConnection().createStatement();

            // drop previously existing table, and ignore any errors
            try {
                stmt.execute(dropSql);
            } catch (SQLException e3) {
                // ignore
            }
            // now create and drop a dummy table
            phase = 1;
            assertFalse(stmt.execute(createSql));
            phase = 2;
            assertFalse(stmt.execute(dropSql));
            phase = 3;
        } catch (SQLException e2) {
            e = e2;
        } finally {
            if (stmt != null) {
                try {
                    stmt.close();
                } catch (SQLException e1) {
                    // ignore
                }
            }
        }
        if (getDialect().allowsDdl()) {
            assertNull(e == null ? null : e.getMessage(), e);
            assertEquals(3, phase);
        } else {
            assertEquals(1, phase);
            assertNotNull(e);
        }
    }

    public void testAllowsFromQuery() {
        String sql =
            dialectize(
                "select * from (select * from [sales_fact_1997]) as [x]");
        if (getDialect().allowsFromQuery()) {
            assertQuerySucceeds(sql);
        } else {
            assertQueryFails(sql, new String[] {});
        }
    }

    public void testRequiresFromQueryAlias() {
        if (getDialect().requiresAliasForFromQuery()) {
            assertTrue(getDialect().allowsFromQuery());
        }
        if (!getDialect().allowsFromQuery()) {
            return;
        }

        String sql =
            dialectize("select * from (select * from [sales_fact_1997])");
        if (getDialect().requiresAliasForFromQuery()) {
            String[] errs = {
                // mysql
                "Every derived table must have its own alias",
                // derby
                "Syntax error: Encountered \"<EOF>\" at line 1, column 47.",
                // postgres
                "ERROR: subquery in FROM must have an alias",
                // teradata
                ".*Syntax error, expected something like a name or a Unicode delimited identifier or an 'UDFCALLNAME' keyword between '\\)' and ';'\\.",
                // netezza
                "(?s).*ERROR:  sub-SELECT in FROM must have an alias.*",
            };
            assertQueryFails(sql, errs);
        } else {
            assertQuerySucceeds(sql);
        }
    }

    public void testRequiresOrderByAlias() {
        String sql =
            dialectize(
                "SELECT [unit_sales]\n"
                + "FROM [sales_fact_1997]\n"
                + "ORDER BY [unit_sales] + [store_id]");
        if (getDialect().requiresOrderByAlias()) {
            final String[] errs = {
                // infobright
                INFOBRIGHT_UNSUPPORTED,
            };
            assertQueryFails(sql, errs);
        } else {
            assertQuerySucceeds(sql);
        }
    }

    public void testAllowsOrderByAlias() {
        String sql =
            dialectize(
                "SELECT [unit_sales] as [x],\n"
                + " [unit_sales] + [store_id] as [y]\n"
                + "FROM [sales_fact_1997]\n"
                + "ORDER BY [y]");
        if (getDialect().allowsOrderByAlias()) {
            assertQuerySucceeds(sql);
        } else {
            String[] errs = {
                // oracle
                "(?s)ORA-03001: unimplemented feature.*",
                // access
                "\\[Microsoft\\]\\[ODBC Microsoft Access Driver\\] Too few parameters. Expected 1.",
                // infobright
                INFOBRIGHT_UNSUPPORTED,
            };
            assertQueryFails(sql, errs);
        }
    }

    public void testSupportsGroupByExpressions() {
        String sql =
            dialectize(
                "SELECT sum([unit_sales] + 3) + 8\n"
                + "FROM [sales_fact_1997]\n"
                + "GROUP BY [unit_sales] + [store_id]");
        if (getDialect().supportsGroupByExpressions()) {
            assertQuerySucceeds(sql);
        } else {
            assertQueryFails(
                sql,
                new String[] {
                    "'sum\\(`unit_sales` \\+ 3\\) \\+ 8' isn't in GROUP BY"
                });
        }
    }

    /**
     * Tests that the
     * {@link mondrian.spi.Dialect#supportsGroupingSets()}
     * dialect property is accurate.
     */
    public void testAllowsGroupingSets() {
        String sql =
            dialectize(
                "SELECT [customer_id],\n"
                + " SUM([store_sales]),\n"
                + " GROUPING([unit_sales]),\n"
                + " GROUPING([customer_id])\n"
                + "FROM [sales_fact_1997]\n"
                + "GROUP BY GROUPING SETS(\n"
                + " ([customer_id], [unit_sales]),\n"
                + " [customer_id],\n"
                + " ())");
        if (getDialect().supportsGroupingSets()) {
            assertQuerySucceeds(sql);
        } else {
            String[] errs = {
                // derby
                "Syntax error: Encountered \"SETS\" at line 6, column 19.",
                // mysql
                "(?s)You have an error in your SQL syntax; check .*",
                // access
                "(?s)\\[Microsoft\\]\\[ODBC Microsoft Access Driver\\] Syntax error \\(missing operator\\) in query expression 'GROUPING SETS.*",
                // postgres
                "ERROR: syntax error at or near \"SETS\"",
                // netezza
                "(?s).*found \"SETS\" \\(at char 135\\) expecting `EXCEPT' or `FOR' or `INTERSECT' or `ORDER' or `UNION'.*",
            };
            assertQueryFails(sql, errs);
        }
    }

    public void testSupportsMultiValueInExpr() {
        String sql =
            dialectize(
                "SELECT [unit_sales]\n"
                + "FROM [sales_fact_1997]\n"
                + "WHERE ([unit_sales], [time_id]) IN ((1, 371), (2, 394))");

        if (getDialect().supportsMultiValueInExpr()) {
            assertQuerySucceeds(sql);
        } else {
            String[] errs = {
                // derby
                "Syntax error: Encountered \",\" at line 3, column 20.",
                // access
                "\\[Microsoft\\]\\[ODBC Microsoft Access Driver\\] Syntax error \\(comma\\) in query expression '.*'.",
                // infobright
                INFOBRIGHT_UNSUPPORTED,
                // teradata
                ".*Syntax error, expected something like a 'SELECT' keyword or '\\(' between '\\(' and the integer '1'\\.",
                // netezza
                "(?s).*found \"1\" \\(at char 81\\) expecting `SELECT' or `'\\(''.*",
            };
            assertQueryFails(sql, errs);
        }
    }

    public void testResultSetConcurrency() {
        int[] Types = {
            ResultSet.TYPE_FORWARD_ONLY,
            ResultSet.TYPE_SCROLL_INSENSITIVE,
            ResultSet.TYPE_SCROLL_SENSITIVE
        };
        int[] Concurs = {
            ResultSet.CONCUR_READ_ONLY,
            ResultSet.CONCUR_UPDATABLE
        };
        String sql =
            dialectize("SELECT [unit_sales] FROM [sales_fact_1997]");
        for (int type : Types) {
            for (int concur : Concurs) {
                boolean b =
                    getDialect().supportsResultSetConcurrency(type, concur);
                Statement stmt = null;
                try {
                    stmt = getConnection().createStatement(type, concur);
                    ResultSet resultSet = stmt.executeQuery(sql);
                    assertTrue(resultSet.next());
                    Object col1 = resultSet.getObject(1);
                    Util.discard(col1);
                    if (!b) {
                        // It's a little surprising that the driver said it
                        // didn't support this type/concurrency combination,
                        // but allowed the statement to be executed anyway.
                        // But don't fail.
                        Util.discard(
                            "expected to fail for type=" + type +
                            ", concur=" + concur);
                    }
                } catch (SQLException e) {
                    if (b) {
                        fail("expected to succeed for type=" + type +
                            ", concur=" + concur);
                        throw Util.newInternal(e, "query [" + sql + "] failed");
                    }
                } finally {
                    if (stmt != null) {
                        try {
                            stmt.close();
                        } catch (SQLException e) {
                            // ignore
                        }
                    }
                }
            }
        }
    }

    public void testGenerateInline() throws SQLException {
        final List<String> typeList = Arrays.asList("String", "Numeric");
        final List<String> nameList = Arrays.asList("x", "y");
        assertInline(
            nameList, typeList,
            new String[]{"a", "1"});

        assertInline(
            nameList, typeList,
            new String[]{"a", "1"}, new String[]{"bb", "2"});

        // date value
        final List<String> typeList2 = Arrays.asList("String", "Date");
        assertInline(
            nameList, typeList2,
            new String[]{"a", "2008-04-29"}, new String[]{"b", "2007-01-02"});
    }

    /**
     * Tests that the method {@link mondrian.spi.Dialect#getNullCollation()}
     * is accurate.
     */
    public void testNullCollation() throws SQLException {
        Dialect dialect = getDialect();
        String ascQuery =
            "select "
            + dialect.quoteIdentifier("grocery_sqft")
            + " from "
            + dialect.quoteIdentifier("store")
            + " order by "
            + dialect.quoteIdentifier("grocery_sqft");
        String descQuery = ascQuery + " DESC";
        Dialect.NullCollation nullCollation = getDialect().getNullCollation();
        switch (nullCollation) {
        case NEGINF:
            assertFirstLast(ascQuery, null, 30351);
            assertFirstLast(descQuery, 30351, null);
            break;
        case POSINF:
            assertFirstLast(ascQuery, 13305, null);
            assertFirstLast(descQuery, null, 13305);
            break;
        default:
            fail("unexpected value " + nullCollation);
        }
    }

    private void assertFirstLast(
        String query,
        Integer expectedFirst,
        Integer expectedLast) throws SQLException
    {
        ResultSet resultSet =
            getConnection().createStatement().executeQuery(query);
        List<Integer> values = new ArrayList<Integer>();
        while (resultSet.next()) {
            values.add(resultSet.getInt(1));
            if (resultSet.wasNull()) {
                values.set(values.size() - 1, null);
            }
        }
        resultSet.close();
        Integer actualFirst = values.get(0);
        Integer actualLast = values.get(values.size() - 1);
        assertEquals(expectedFirst, actualFirst);
        assertEquals(expectedLast, actualLast);
    }

    private void assertInline(
        List<String> nameList,
        List<String> typeList,
        String[]... valueList) throws SQLException {
        String sql =
            getDialect().generateInline(
                nameList,
                typeList,
                Arrays.asList(valueList));
        Statement stmt = null;
        try {
            stmt = getConnection().createStatement();
            ResultSet resultSet = stmt.executeQuery(sql);
            Set<List<String>> actualValues = new HashSet<List<String>>();
            while (resultSet.next()) {
                final List<String> row = new ArrayList<String>();
                for (int i = 0; i < typeList.size(); i++) {
                    final String s;
                    final String type = typeList.get(i);
                    if (type.equals("String")) {
                        s = resultSet.getString(i + 1);
                    } else if (type.equals("Date")) {
                        s = String.valueOf(resultSet.getDate(i + 1));
                    } else if (type.equals("Numeric")) {
                        s = String.valueOf(resultSet.getInt(i + 1));
                    } else {
                        throw new RuntimeException("unknown type " + type);
                    }
                    row.add(s);
                }
                actualValues.add(row);
            }
            Set<List<String>> expectedRows = new HashSet<List<String>>();
            for (String[] strings : valueList) {
                expectedRows.add(Arrays.asList(strings));
            }
            assertEquals(expectedRows, actualValues);
            stmt.close();
            stmt = null;
        } finally {
            if (stmt != null) {
                try {
                    stmt.close();
                } catch (SQLException e) {
                    // ignore
                }
            }
        }
    }

    /**
     * Converts query or DDL statement into this dialect.
     *
     * @param s SQL query or DDL statement
     * @return Query or DDL statement translated into this dialect
     */
    private String dialectize(String s) {
        final Dialect.DatabaseProduct databaseProduct =
            getDialect().getDatabaseProduct();
        switch (databaseProduct) {
        case ACCESS:
            break;
        case MYSQL:
        case INFOBRIGHT:
            s = s.replace('[', '`');
            s = s.replace(']', '`');
            break;
        case ORACLE:
            s = s.replace('[', '"');
            s = s.replace(']', '"');
            s = s.replaceAll(" as ", "");
            break;
        default:
            s = s.replace('[', '"');
            s = s.replace(']', '"');
            break;
        }
        return s;
    }

    /**
     * Asserts that a query succeeds and produces at least one row.
     *
     * @param sql SQL query in current dialect
     */
    protected void assertQuerySucceeds(String sql) {
        Statement stmt = null;
        try {
            stmt = getConnection().createStatement();
            ResultSet resultSet = stmt.executeQuery(sql);
            assertTrue(resultSet.next());
            Object col1 = resultSet.getObject(1);
            Util.discard(col1);
        } catch (SQLException e) {
            throw Util.newInternal(e, "query [" + sql + "] failed");
        } finally {
            if (stmt != null) {
                try {
                    stmt.close();
                } catch (SQLException e) {
                    // ignore
                }
            }
        }
    }

    /**
     * Asserts that a query fails.
     *
     * @param sql SQL query
     * @param patterns Array of expected patterns, generally one for each
     *   SQL dialect for which the test is expected to fail
     */
    protected void assertQueryFails(String sql, String[] patterns) {
        Statement stmt = null;
        try {
            stmt = getConnection().createStatement();
            ResultSet resultSet;
            try {
            resultSet = stmt.executeQuery(sql);
            } catch (SQLException e2) {
                // execution failed - good
                String message = e2.getMessage();
                for (String pattern : patterns) {
                    if (message.matches(pattern)) {
                        return;
                    }
                }
                throw new AssertionFailedError(
                    "error [" + message
                        + "] did not match any of the supplied patterns");
            }
            assertTrue(resultSet.next());
            Object col1 = resultSet.getObject(1);
            Util.discard(col1);
        } catch (SQLException e) {
            throw Util.newInternal(e, "failed in wrong place");
        } finally {
            if (stmt != null) {
                try {
                    stmt.close();
                } catch (SQLException e) {
                    // ignore
                }
            }
        }
    }
}

// End DialectTest.java
