/*
// $Id: //open/mondrian-release/3.2/src/main/mondrian/rolap/agg/DenseIntSegmentDataset.java#1 $
// This software is subject to the terms of the Eclipse Public License v1.0
// Agreement, available at the following URL:
// http://www.eclipse.org/legal/epl-v10.html.
// Copyright (C) 2002-2002 Kana Software, Inc.
// Copyright (C) 2002-2010 Julian Hyde and others
// All Rights Reserved.
// You must accept the terms of that agreement to use this software.
//
// jhyde, 21 March, 2002
*/
package mondrian.rolap.agg;

import mondrian.rolap.CellKey;
import mondrian.rolap.SqlStatement;

/**
 * Implementation of {@link DenseSegmentDataset} that stores
 * values of type {@link Object}.
 *
 * <p>The storage requirements are as follows. Table requires 1 word per
 * cell.</p>
 *
 * @author jhyde
 * @since 21 March, 2002
 * @version $Id: //open/mondrian-release/3.2/src/main/mondrian/rolap/agg/DenseIntSegmentDataset.java#1 $
 */
class DenseIntSegmentDataset extends DenseNativeSegmentDataset {
    private final int[] values; // length == m[0] * ... * m[axes.length-1]

    /**
     * Creates a DenseIntSegmentDataset.
     *
     * @param segment Segment
     * @param size Number of coordinates
     */
    DenseIntSegmentDataset(Segment segment, int size) {
        super(segment, size);
        this.values = new int[size];
    }

    public int getInt(CellKey key) {
        int offset = key.getOffset(axisMultipliers);
        return values[offset];
    }

    public Object getObject(CellKey pos) {
        int offset = pos.getOffset(axisMultipliers);
        return getObject(offset);
    }

    protected Integer getObject(int offset) {
        final int value = values[offset];
        if (value == 0 && isNull(offset)) {
            return null;
        }
        return value;
    }

    public boolean exists(CellKey pos) {
        return true;
    }

    public void populateFrom(int[] pos, SegmentDataset data, CellKey key) {
        final int offset = getOffset(pos);
        final int value = values[offset] = data.getInt(key);
        if (value == 0) {
            nullIndicators.set(offset, !data.isNull(key));
        }
    }

    public void populateFrom(
        int[] pos, SegmentLoader.RowList rowList, int column)
    {
        int offset = getOffset(pos);
        final int value = values[offset] = rowList.getInt(column);
        if (value == 0) {
            nullIndicators.set(offset, !rowList.isNull(column));
        }
    }

    public SqlStatement.Type getType() {
        return SqlStatement.Type.INT;
    }

    public void put(CellKey key, int value) {
        int offset = key.getOffset(axisMultipliers);
        values[offset] = value;
    }

    public void put(int[] ordinals, int value) {
        int offset = getOffset(ordinals);
        values[offset] = value;
    }

    void set(int k, int o) {
        values[k] = o;
    }

    protected int size() {
        return values.length;
    }
}

// End DenseIntSegmentDataset.java
