/*
 * iReport - Visual Designer for JasperReports.
 * Copyright (C) 2002 - 2009 Jaspersoft Corporation. All rights reserved.
 * http://www.jaspersoft.com
 *
 * Unless you have purchased a commercial license agreement from Jaspersoft,
 * the following license terms apply:
 *
 * This program is part of iReport.
 *
 * iReport is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * iReport is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with iReport. If not, see <http://www.gnu.org/licenses/>.
 */


package com.jaspersoft.jasperserver.rest.services;

import com.jaspersoft.jasperserver.api.metadata.xml.domain.impl.Argument;
import com.jaspersoft.jasperserver.api.metadata.xml.domain.impl.ResourceDescriptor;
import com.jaspersoft.jasperserver.core.util.XMLUtil;
import com.jaspersoft.jasperserver.remote.AbstractService;
import com.jaspersoft.jasperserver.remote.ServiceException;
import com.jaspersoft.jasperserver.remote.services.DeleteService;
import com.jaspersoft.jasperserver.remote.services.GetService;
import com.jaspersoft.jasperserver.remote.services.PostService;
import com.jaspersoft.jasperserver.remote.services.PutService;
import com.jaspersoft.jasperserver.rest.RESTAbstractService;
import com.jaspersoft.jasperserver.rest.RESTUtils;
import com.jaspersoft.jasperserver.ws.xml.Marshaller;
import com.jaspersoft.jasperserver.ws.xml.Unmarshaller;
import java.io.IOException;
import java.io.StringReader;
import java.util.HashMap;
import java.util.Map;
import javax.activation.DataSource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.security.AccessDeniedException;
import org.springframework.web.multipart.MultipartHttpServletRequest;
import org.w3c.dom.Document;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * This class implements the logic to manage the repository resources.
 * It provides support for the HTTP verbs GET, POST, PUT and DELETE.
 *
 * @author gtoffoli
 */
public class RESTResource extends RESTAbstractService {

    private final static Log log = LogFactory.getLog(RESTResource.class);

    /**
     * Get a resource based of the requested URI.
     * If the parameter file is set, it will be used to retrieve the content of the file.
     * 
     * @param req
     * @param resp
     */
    @Override
    protected void doGet(HttpServletRequest req, HttpServletResponse resp) throws ServiceException {

            // Get the uri of the resource
            String uri = RESTUtils.extractRepositoryUri(req.getPathInfo());

            // by default get the root...
            if (uri == null || uri.length() == 0) uri="/";

            // get the resources....
            GetService service = new GetService(context);

            // Add all the options...
            Map<String,Object> options = new HashMap<String, Object>();

            // This options allow to get the informations about an input control
            // including the data to fill the input control....
            if (req.getParameter(Argument.IC_GET_QUERY_DATA) != null)
            {
                options.put(Argument.IC_GET_QUERY_DATA, req.getParameter(Argument.IC_GET_QUERY_DATA));

                // Extract parameters
                Map<String, Object> parameters = RESTUtils.extractParameters(req);

                // Add the parsed parameters to the options map (only if it makes sense)
                if (parameters.size() > 0)
                {
                    options.put(Argument.PARAMS_ARG, parameters);
                }
            }

            ResourceDescriptor rd = null;
            try {
               rd = service.getResource(uri, options);
            } catch (ServiceException ex)
            {
                throw ex;
            } catch (Exception ex)
            {
               RESTUtils.setStatusAndBody(HttpServletResponse.SC_INTERNAL_SERVER_ERROR, resp, ex.getLocalizedMessage());
               return;
            }

            // This check should not be useful, since an execption should have been thrown by service.getReporse
            if (rd == null)
            {
                RESTUtils.setStatusAndBody(HttpServletResponse.SC_NOT_FOUND, resp, "");
                return;
            }

            // if we are here, rd is not null and ready to be sent to the client...but

            String file = req.getParameter("file");

            // If the client specified a specific file, return the file
            if (file != null)
            {
                Map<String, DataSource> attachments = service.getOutputAttachments();
                
                if (attachments.containsKey(file))
                {
                    DataSource ds = attachments.get(file);
                    RESTUtils.sendFile(ds, resp);
                    return;
                }
                else
                {
                   RESTUtils.setStatusAndBody(HttpServletResponse.SC_NOT_FOUND, resp, "The specified file is not present in this resource");
                   return;
                }
            }
            else // else return the resource descriptor
            {

                Marshaller m = new Marshaller();
                String xml = m.writeResourceDescriptor(rd);
                // send the xml...
                RESTUtils.setStatusAndBody(HttpServletResponse.SC_OK, resp, xml);
            }
            return;
        }

    /**
     * The PUT service is used to create a new resource in the repository...
     *
     * @param req
     * @param resp
     * @throws ServiceException
     */
    @Override
    protected void doPut(HttpServletRequest req, HttpServletResponse resp) throws ServiceException {

        PutService service = new PutService(context);

        HttpServletRequest mreq = RESTUtils.extractAttachments(service, req);

        String resourceDescriptorXml = null;

        // get the resource descriptor...
        if (mreq instanceof MultipartHttpServletRequest)
        {
            resourceDescriptorXml = mreq.getParameter(RESTUtils.REQUEST_PARAMENTER_RD);
        }
        else
        {
            try {
                resourceDescriptorXml = IOUtils.toString(req.getInputStream());
            } catch (IOException ex) {
                throw new ServiceException(ServiceException.INTERNAL_SERVER_ERROR, ex.getLocalizedMessage());
            }
        }


        if (resourceDescriptorXml == null)
        {
            RESTUtils.setStatusAndBody(HttpServletResponse.SC_BAD_REQUEST, resp, "Missing parameter " + RESTUtils.REQUEST_PARAMENTER_RD + " " + service.getInputAttachments());
            return;
        }


        // Parse the resource descriptor...
        InputSource is = new InputSource(new StringReader(resourceDescriptorXml));
        Document doc = null;
        ResourceDescriptor rd = null;
        try {
            doc = XMLUtil.getNewDocumentBuilder().parse(is);
            rd = Unmarshaller.readResourceDescriptor(doc.getDocumentElement());

            // we force the rd to be new...
            rd.setIsNew(true);

            ResourceDescriptor createdRd = service.putResource(rd, true);

            Marshaller m = new Marshaller();
            String xml = m.writeResourceDescriptor(createdRd);
            // send the xml...
            RESTUtils.setStatusAndBody(HttpServletResponse.SC_CREATED, resp, xml);

        } catch (SAXException ex) {
            RESTUtils.setStatusAndBody(HttpServletResponse.SC_BAD_REQUEST, resp, "Invalid resource descriptor");
            return;
        } catch (ServiceException ex) {
            throw ex;
        } catch (Exception ex) {
            log.error("Unexpected error during resource save: " + ex.getMessage(), ex);
            throw new ServiceException(ServiceException.INTERNAL_SERVER_ERROR, ex.getLocalizedMessage());
        }

    }

    /**
     * POST can be used to modify a resource or to copy/move it.
     *
     * @param req
     * @param resp
     * @throws ServiceException
     */
    @Override
    protected void doPost(HttpServletRequest req, HttpServletResponse resp) throws ServiceException {
        String sourceURI = RESTUtils.extractRepositoryUri(req.getPathInfo());
        String destURI = RESTUtils.getDetinationUri(req);
        if (destURI!=null)
        {
            PostService service = new PostService(context);
            if (req.getParameterMap().containsKey(RESTUtils.REQUEST_PARAMENTER_COPY_TO))
                service.copyResource(sourceURI, destURI);
            else
                service.moveResource(sourceURI, destURI);
        }
        else // Modify the resource...
        {
            PutService service = new PutService(context);

            HttpServletRequest mreq = RESTUtils.extractAttachments(service, req);

            String resourceDescriptorXml = null;

            // get the resource descriptor...
            if (mreq instanceof MultipartHttpServletRequest)
            {
                resourceDescriptorXml = mreq.getParameter(RESTUtils.REQUEST_PARAMENTER_RD);
            }
            else
            {
                try {
                    resourceDescriptorXml = IOUtils.toString(req.getInputStream());
                } catch (IOException ex) {
                    throw new ServiceException(ServiceException.INTERNAL_SERVER_ERROR, ex.getLocalizedMessage());
                }
            }
            if (resourceDescriptorXml == null)
            {
                RESTUtils.setStatusAndBody(HttpServletResponse.SC_BAD_REQUEST, resp, "Missing parameter " + RESTUtils.REQUEST_PARAMENTER_RD + " " + service.getInputAttachments());
                return;
            }


            // Parse the resource descriptor...
            InputSource is = new InputSource(new StringReader(resourceDescriptorXml));
            Document doc = null;
            ResourceDescriptor rd = null;
            try {
                doc = XMLUtil.getNewDocumentBuilder().parse(is);
                rd = Unmarshaller.readResourceDescriptor(doc.getDocumentElement());

                // we force the rd to be new...
                rd.setIsNew(false);
                
                if (rd.getUriString() == null || !rd.getUriString().equals(sourceURI))
                {
                    RESTUtils.setStatusAndBody(HttpServletResponse.SC_BAD_REQUEST, resp, "Request and descriptor uri are not equals");
                }

                
                ResourceDescriptor createdRd = service.putResource(rd, true);

                Marshaller m = new Marshaller();
                String xml = m.writeResourceDescriptor(createdRd);
                // send the xml...
                RESTUtils.setStatusAndBody(HttpServletResponse.SC_OK, resp, xml);

            } catch (SAXException ex) {
                log.error("error parsing...", ex);
                RESTUtils.setStatusAndBody(HttpServletResponse.SC_BAD_REQUEST, resp, "Invalid resource descriptor");
                return;
            } catch (ServiceException ex) {
                log.error("error executing the service...", ex);
                throw ex;
            } catch (Exception ex) {
                log.error("error executing the service...", ex);
                throw new ServiceException(ServiceException.INTERNAL_SERVER_ERROR, ex.getLocalizedMessage());
            }
        }
    }

    @Override
    protected void doDelete(HttpServletRequest req, HttpServletResponse resp) throws ServiceException {

        String uri = RESTUtils.extractRepositoryUri(req.getPathInfo());
        DeleteService service = new DeleteService(context);
        service.deleteResource(uri);
        resp.setStatus(HttpServletResponse.SC_OK);
    }





}
