/*
 * Copyright (C) 2005 - 2011 Jaspersoft Corporation. All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased  a commercial license agreement from Jaspersoft,
 * the following license terms  apply:
 *
 * This program is free software: you can redistribute it and/or  modify
 * it under the terms of the GNU Affero General Public License  as
 * published by the Free Software Foundation, either version 3 of  the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero  General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public  License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */


package com.jaspersoft.jasperserver.remote.services;

import com.jaspersoft.jasperserver.api.common.domain.ExecutionContext;
import com.jaspersoft.jasperserver.api.common.domain.impl.ExecutionContextImpl;
import com.jaspersoft.jasperserver.api.engine.jasperreports.domain.impl.ReportUnitRequest;
import com.jaspersoft.jasperserver.api.engine.jasperreports.domain.impl.ReportUnitResult;
import com.jaspersoft.jasperserver.api.engine.jasperreports.util.HtmlExportUtil;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Resource;
import com.jaspersoft.jasperserver.api.metadata.jasperreports.domain.ReportUnit;
import com.jaspersoft.jasperserver.api.metadata.xml.domain.impl.Argument;
import com.jaspersoft.jasperserver.api.metadata.xml.domain.impl.OperationResult;

import com.jaspersoft.jasperserver.remote.AbstractService;
import com.jaspersoft.jasperserver.remote.ReportExporter;
import com.jaspersoft.jasperserver.remote.ServiceException;
import com.jaspersoft.jasperserver.ws.xml.ByteArrayDataSource;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Arrays;
import java.util.Date;

import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.TimeZone;
import net.sf.jasperreports.engine.JRExporter;
import net.sf.jasperreports.engine.JRExporterParameter;
import net.sf.jasperreports.engine.JasperPrint;
import net.sf.jasperreports.engine.export.GenericElementReportTransformer;
import net.sf.jasperreports.engine.export.JExcelApiExporter;
import net.sf.jasperreports.engine.export.JRCsvExporter;
import net.sf.jasperreports.engine.export.JRHtmlExporterParameter;
import net.sf.jasperreports.engine.export.JRRtfExporter;
import net.sf.jasperreports.engine.export.JRXlsAbstractExporterParameter;
import net.sf.jasperreports.engine.export.JRXmlExporter;
import net.sf.jasperreports.engine.util.JRSaver;
import net.sf.jasperreports.engine.util.JRTypeSniffer;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.context.ApplicationContext;

/**
 * Run Report service
 * Run a report unit using the passing in parameters and options
 * @author gtoffoli
 */
public class RunReportService extends AbstractService {

    private final static Log log = LogFactory.getLog(GetService.class);


    public RunReportService(ApplicationContext context)
    {
        super(context);
    }

    /**
     * Return a response.
     * Generated files (one or more) are put in the output attachments map of this context
     *
     * @param reportUnitURI
     * @param parameters
     * @param arguments
     * @return
     * @throws ServiceException
     */
    public OperationResult runReport(String reportUnitURI, Map<String, Object> parameters, Map<String, String> arguments) throws ServiceException {

        long currentTime = System.currentTimeMillis();
        getManagementServices().getAuditHelper().createAuditEvent("runReport");

        OperationResult or = getServicesUtils().createOperationResult(OperationResult.SUCCESS , null);

        try {


                String format = arguments.get(Argument.RUN_OUTPUT_FORMAT);
                if (format == null) format = Argument.RUN_OUTPUT_FORMAT_PDF;
                format = format.toUpperCase();

                String transformerKey = arguments.get(Argument.RUN_TRANSFORMER_KEY);
                
                Resource reportResource = getManagementServices().getRepository().getResource(null, reportUnitURI);


                if (reportResource == null || !(reportResource instanceof ReportUnit)) {
                	throw new ServiceException(2,
                			getMessage("webservices.error.notValidReportUnit", new Object[]{reportUnitURI})
                                );
                }

                // convert parameters parameters
                parameters = getManagementServices().getRepositoryHelper().convertParameterValues(reportUnitURI, parameters);
                
                // run the report
                ReportUnitResult result = (ReportUnitResult)getManagementServices().getRunReportEngine().execute(
                                createExecutionContext(), new ReportUnitRequest(reportUnitURI, parameters));


                if (result == null) {

                		throw new ServiceException(ServiceException.FILL_ERROR,
                				getMessage("webservices.error.errorExecutingReportUnit",
                                            new Object[]{reportUnitURI})

                                        );

                } else {

                    JasperPrint jasperPrint = result.getJasperPrint();

                    or = exportReport(reportUnitURI, jasperPrint, arguments);

                    // The jasperprint may have been transformed during export...
                    this.getAttributes().put("jasperPrint", jasperPrint);
                }

                if (or.getReturnCode() != 0) {
                    getManagementServices().getAuditHelper().addExceptionToAllAuditEvents(new Exception(or.getMessage()));
                }

            } catch (ServiceException e) {
                log.error("caught exception: " + e.getMessage(), e);
                or.setReturnCode( e.getErrorCode() );
                or.setMessage(e.getMessage());
                getManagementServices().getAuditHelper().addExceptionToAllAuditEvents(e);

            } catch (Throwable e) {
            	log.error("caught Throwable exception: " + e.getMessage(), e);
                e.printStackTrace(System.out);
                System.out.flush();
                or.setReturnCode( 1 );
                or.setMessage(e.getMessage());
                getManagementServices().getAuditHelper().addExceptionToAllAuditEvents(e);
            }

            getManagementServices().getAuditHelper().addPropertyToAuditEvent("runReport", "reportExecutionStartTime", new Date(currentTime));
            getManagementServices().getAuditHelper().addPropertyToAuditEvent("runReport", "reportExecutionTime", System.currentTimeMillis() - currentTime);
            return or;
    }


    /**
     * Export the report in a specific format using the specified arguments
     * Generated files (one or more) are put in the output attachments map of this context
     *
     * @param jasperPrint JasperPrint
     * @param arguments - indicates the final file format, starting/ending pages, etc...
     * @return OperationResult
     * @throws ServiceException
     */
    public OperationResult exportReport(String reportUnitURI, JasperPrint jasperPrint, Map<String, String> arguments) throws ServiceException {

        long currentTime = System.currentTimeMillis();
        getManagementServices().getAuditHelper().createAuditEvent("runReport");

        OperationResult or = getServicesUtils().createOperationResult(OperationResult.SUCCESS , null);

        try {
                String format = arguments.get(Argument.RUN_OUTPUT_FORMAT);
                if (format == null) format = Argument.RUN_OUTPUT_FORMAT_PDF;
                format = format.toUpperCase();

                String transformerKey = arguments.get(Argument.RUN_TRANSFORMER_KEY);

                // Export...
        	ByteArrayOutputStream bos = new ByteArrayOutputStream();
                ByteArrayDataSource bads = null;

                if (format.equals(Argument.RUN_OUTPUT_FORMAT_JRPRINT))
                {
                    log.debug("Returning JasperPrint");


                    if (transformerKey != null)
                    {
                            if (log.isDebugEnabled())
                            {
                                    log.debug("Transforming JasperPrint generic element for key "+ transformerKey);
                            }

                            GenericElementReportTransformer.transformGenericElements(jasperPrint, transformerKey);
                    }

                    JRSaver.saveObject(jasperPrint, bos);
                    bads = new ByteArrayDataSource(bos.toByteArray());
                    getOutputAttachments().put("jasperPrint", bads);
                }
                else
                {
                    //List jasperPrintList = new ArrayList();
                    //jasperPrintList.add(jasperPrint);

                    HashMap exportParameters = new HashMap();

                    String value = arguments.get(Argument.RUN_OUTPUT_PAGE);
                    if (value != null)
                    {
                        exportParameters.put(Argument.RUN_OUTPUT_PAGE, value);
                    }

                    value = arguments.get(Argument.RUN_OUTPUT_IMAGES_URI);
                    if (value != null) exportParameters.put(Argument.RUN_OUTPUT_IMAGES_URI, value);

                    Map exporterParams;
                    try {
                            exporterParams = exportReport(reportUnitURI, jasperPrint, format, bos, exportParameters);

                            log.error("Exporter params: " + Arrays.asList(exporterParams.keySet().toArray()));
                   } catch (Exception e) {
                                    log.error("Error exporting report", e);
                                    throw new ServiceException(ServiceException.EXPORT_ERROR,
                                                    getMessage("webservices.error.errorExportingReportUnit",
                                                                    new Object[] { e.getMessage() }));
                    } finally {
                                    if (bos != null) {
                                            try {
                                                    bos.close();
                                            } catch (IOException ex) {
                                            }
                                    }
                    }

                    bads = new ByteArrayDataSource(bos.toByteArray(), getContentType(format));
                    getOutputAttachments().put("report", bads);
                    addAdditionalAttachmentsForReport(jasperPrint, format, exporterParams);
                }

                if (or.getReturnCode() != 0) {
                    getManagementServices().getAuditHelper().addExceptionToAllAuditEvents(new Exception(or.getMessage()));
                }

            } catch (ServiceException e) {
                log.error("caught exception: " + e.getMessage(), e);
                or.setReturnCode( e.getErrorCode() );
                or.setMessage(e.getMessage());
                getManagementServices().getAuditHelper().addExceptionToAllAuditEvents(e);

            } catch (Throwable e) {
            	log.error("caught Throwable exception: " + e.getMessage(), e);
                e.printStackTrace(System.out);
                System.out.flush();
                or.setReturnCode( 1 );
                or.setMessage(e.getMessage());
                getManagementServices().getAuditHelper().addExceptionToAllAuditEvents(e);
            }

            getManagementServices().getAuditHelper().addPropertyToAuditEvent("runReport", "reportExecutionStartTime", new Date(currentTime));
            getManagementServices().getAuditHelper().addPropertyToAuditEvent("runReport", "reportExecutionTime", System.currentTimeMillis() - currentTime);
            return or;
    }




    protected ExecutionContext createExecutionContext() {
            ExecutionContextImpl ctx = new ExecutionContextImpl();
            ctx.setLocale(getLocale());
            ctx.setTimeZone(TimeZone.getDefault());
            return ctx;
    }



    public JRExporter getExporter(String type, Map exportParameters) {
		JRExporter exporter = null;
		if (type.equals(Argument.RUN_OUTPUT_FORMAT_HTML)) {
			exporter = HtmlExportUtil.getHtmlExporter();
                if (exportParameters.get(Argument.RUN_OUTPUT_IMAGES_URI) != null)
                {
                     exporter.setParameter(JRHtmlExporterParameter.IMAGES_URI,"" + exportParameters.get(Argument.RUN_OUTPUT_IMAGES_URI));
                }
                else
                    exporter.setParameter(JRHtmlExporterParameter.IMAGES_URI, "images/");
	        exporter.setParameter(JRHtmlExporterParameter.IS_USING_IMAGES_TO_ALIGN, Boolean.TRUE);
	        //exporter.setParameter(JRExporterParameter.OUTPUT_STRING_BUFFER, reportContent);

	        // collecting the images into a map
	        exporter.setParameter(JRHtmlExporterParameter.IMAGES_MAP, new LinkedHashMap());
		} else if (type.equals(Argument.RUN_OUTPUT_FORMAT_XLS)) {
			exporter =  new JExcelApiExporter();
			exporter.setParameter(JRXlsAbstractExporterParameter .IS_ONE_PAGE_PER_SHEET, Boolean.FALSE);
			exporter.setParameter(JRXlsAbstractExporterParameter .IS_WHITE_PAGE_BACKGROUND, Boolean.FALSE);
		} else if (type.equals(Argument.RUN_OUTPUT_FORMAT_CSV)) {
			exporter =  new JRCsvExporter();
		} else if (type.equals(Argument.RUN_OUTPUT_FORMAT_XML)) {
			exporter =  new JRXmlExporter();
		} else if (type.equals(Argument.RUN_OUTPUT_FORMAT_RTF)) {
			exporter =  new JRRtfExporter();
		}
		if (exporter != null &&(exportParameters.get(Argument.RUN_OUTPUT_PAGE) != null)) {
                        exporter.setParameter(JRExporterParameter.PAGE_INDEX, new Integer("" + exportParameters.get(Argument.RUN_OUTPUT_PAGE)));
		}

		return exporter;

	}

	public String getContentType(String type) {
		String contentType = null;
		if (type.equals(Argument.RUN_OUTPUT_FORMAT_PDF)) {
			contentType = "application/pdf";
		} else if (type.equals(Argument.RUN_OUTPUT_FORMAT_HTML)) {
			contentType = "text/html";
		} else if (type.equals(Argument.RUN_OUTPUT_FORMAT_XLS)) {
			contentType = "application/xls";
		} else if (type.equals(Argument.RUN_OUTPUT_FORMAT_CSV)) {
			contentType = "application/vnd.ms-excel";
		} else if (type.equals(Argument.RUN_OUTPUT_FORMAT_XML)) {
			contentType = "text/xml";
		} else if (type.equals(Argument.RUN_OUTPUT_FORMAT_RTF)) {
			contentType =  "application/rtf";
		}

		return contentType;

	}

	/**
	 * Create additional Web Services attachments for the content. At this stage, HTML reports
	 * have their images as attachments
	 *
	 * @param jasperPrint
	 * @param format
	 * @param format
	 * @param exportParameters
	 * @throws ServiceException
	 */
	private void addAdditionalAttachmentsForReport(JasperPrint jasperPrint, String format, Map exportParameters) throws ServiceException {

            log.error("Format requested: " + format + "  " + Argument.RUN_OUTPUT_FORMAT_HTML);
		if (!format.equals(Argument.RUN_OUTPUT_FORMAT_HTML)) {
			return;
		}

		try {
			Map imagesMap = (Map)exportParameters.get(JRHtmlExporterParameter.IMAGES_MAP);

                        log.error("imagesMap : " +  Arrays.asList(imagesMap.keySet().toArray()));



			for (Iterator it = imagesMap.entrySet().iterator(); it.hasNext();) {
				Map.Entry entry = (Map.Entry) it.next();
				String name = (String) entry.getKey();
				byte[] data = (byte[]) entry.getValue();
				byte imageType = JRTypeSniffer.getImageType(data);
				String mimeType = JRTypeSniffer.getImageMimeType(imageType);

                                log.error("Adding image for HTML: " + name
							+ ", type: " + mimeType);
				if (log.isDebugEnabled()) {
					log.debug("Adding image for HTML: " + name
							+ ", type: " + mimeType);
				}

                ByteArrayDataSource bads = new ByteArrayDataSource(data, mimeType);
                getOutputAttachments().put(name, bads);
			}
		} catch (Throwable e) {
			log.error(e);
			throw new ServiceException(ServiceException.EXPORT_ERROR,
                                   getMessage("webservices.error.errorAddingImage",
                                            new Object[]{e.getMessage()})
                                );
		}
	}


        /**
         * Look for the ReportExporter configured for the named format and export
         * the report.
         *
         *
         * @param reportUnitURI
         * @param jasperPrint
         * @param format
         * @param output
         * @param exportParameters
         * @return
         * @throws Exception
         */
        protected Map exportReport(
    		String reportUnitURI,
    		JasperPrint jasperPrint,
                String format,
                OutputStream output,
                HashMap exportParameters) throws ServiceException
	{
            ReportExporter exporter = getManagementServices().getServiceConfiguration().getExporter(format.toLowerCase());

            if (exporter == null)
            {
                throw new ServiceException(3, "Export format " + format.toLowerCase() +" not supported or misconfigured");
            }

            try {
            
                return exporter.exportReport(
                            jasperPrint,
                            output,
                            getManagementServices().getEngine(),
                            exportParameters,
                            createExecutionContext(),
                            reportUnitURI);
                
            } catch (Exception ex)
            {
                throw new ServiceException(3, ex.getMessage() +" (while exporting the report)");
            }
        }



    

}
