/*
 * Copyright (C) 2005 - 2011 Jaspersoft Corporation. All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased  a commercial license agreement from Jaspersoft,
 * the following license terms  apply:
 *
 * This program is free software: you can redistribute it and/or  modify
 * it under the terms of the GNU Affero General Public License  as
 * published by the Free Software Foundation, either version 3 of  the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero  General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public  License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package com.jaspersoft.jasperserver.remote.services;

import com.jaspersoft.jasperserver.api.JSException;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Folder;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Resource;
import com.jaspersoft.jasperserver.api.metadata.jasperreports.domain.ReportUnit;
import com.jaspersoft.jasperserver.api.metadata.view.domain.FilterCriteria;
import com.jaspersoft.jasperserver.api.metadata.view.domain.FilterElement;
import com.jaspersoft.jasperserver.api.metadata.view.domain.FilterElementOr;
import com.jaspersoft.jasperserver.api.metadata.xml.domain.impl.Argument;
import com.jaspersoft.jasperserver.api.metadata.xml.domain.impl.ResourceDescriptor;
import com.jaspersoft.jasperserver.remote.AbstractService;
import com.jaspersoft.jasperserver.remote.ResourceHandler;
import com.jaspersoft.jasperserver.remote.ServiceException;
import com.jaspersoft.jasperserver.remote.handlers.ReportUnitHandler;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.security.Authentication;
import org.springframework.security.GrantedAuthority;
import org.springframework.security.context.SecurityContextHolder;
import org.springframework.security.userdetails.UserDetails;

/**
 * Login REST service
 * The dirty job of loggin a user and sending out an error is done by the RESTLoginAuthenticationFilter.
 * This service just return a succesful login message
 * @author gtoffoli
 */
public class ListService extends AbstractService {

    private final static Log log = LogFactory.getLog(ListService.class);


    public ListService(ApplicationContext context)
    {
        super(context);
    }

    
    /**
     * Return a list of ResourceDescriptor(s)
     * @throws JSException
     */
    public List listResources(String uri) throws ServiceException
    {
        return listResources(uri, 0);
    }
    /**
     * 
     * @param uri
     * @param maxItems - The maximum number of items (or 0 to get all the items)
     * @return
     * @throws ServiceException
     */
    public List listResources(String uri, int maxItems) throws ServiceException
    {
	log.debug("list for uri: " + uri);
        log.debug("Max items: " + maxItems);

        // The result of our list action
        List listOfResources = new ArrayList();

        // The resource to list based on the uri
        Resource resource = getManagementServices().locateResource(uri);

        // If the uri indicates the root, just list the root directory
        if (resource == null)
        {
            log.warn("No resource " + uri + " found");
            throw new ServiceException(ServiceException.RESOURCE_NOT_FOUND,"Invalid uri or not existing resource");
        }

        if (resource instanceof Folder)
        {
            List folders = getManagementServices().getRepository().getSubFolders(null, uri);
            // This filters with object level security.
            // Will only get folders the user has access to
            filterFolderList(folders);

            if (folders == null) return listOfResources;

            // We avoid to read or cache the attachments if we are just listing our resources...
            Map options = new HashMap();
            options.put(Argument.NO_RESOURCE_DATA_ATTACHMENT, Boolean.TRUE);

            for (int i=0; i <folders.size(); ++i)
            {
                Resource folderRes = (Resource)folders.get(i);
                listOfResources.add( getManagementServices().createResourceDescriptor(folderRes) );
            }

            // create a criteria for finding things with a common parent folder.
            FilterCriteria filterCriteria = new FilterCriteria();
            filterCriteria.addFilterElement( FilterCriteria.createParentFolderFilter(uri) );

            // This filters with object level security
            // Will only get resources the user has access to

            List units = getManagementServices().getRepository().loadClientResources(filterCriteria);

            if (units != null)
            {
                for (Iterator it = units.iterator(); units != null && it.hasNext(); )
                {
                    Resource fileRes = (Resource) it.next();
                    try {
                        listOfResources.add( getManagementServices().createResourceDescriptor(fileRes, options));
                    } catch (Exception ex)
                    {
                        log.error(ex);
                    }
                }
            }
        }
        else if (resource instanceof ReportUnit)
        {
            // The list inside a report unit should return the whole content of the report unit.
            // For this porpuse, we force some special parameters to look into the report unit...
            Map options = new HashMap();
            options.put(ReportUnitHandler.OPTION_REPORT_UNIT_CONTENTS, Boolean.TRUE);
            listOfResources = getManagementServices().createResourceDescriptor( uri ).getChildren();
        }

        // check for max resources...
        if (maxItems > 0 && maxItems < listOfResources.size())
        {
            log.debug("There are " + listOfResources.size() + " found. Getting indexes from 0 to " + maxItems);
            listOfResources = listOfResources.subList(0, maxItems);
        }

        return listOfResources;
    }



    /**
     * Return a list of ResourceDescriptor(s)
     * @throws JSException
     */
    public List getResources(FilterCriteria criteria) throws ServiceException
    {
        return getResources(criteria,0, null);
    }


    /**
     * Get resources which satisfy the given criteria.
     * Since there is not a good or precise way to filter the type of resource,
     * this method allows to provide a specific wsType (that can be null).
     *
     * @param criteria
     * @param maxItems number of maximum items returned (0 for not set a limit)
     * @param wsType - Can be null. It is the list of wsType allowed in the returned resources.
     * @return
     * @throws ServiceException
     */
    public List getResources(FilterCriteria criteria, int maxItems, List<String> wsTypes) throws ServiceException
    {


        // The result of our list action
        List listOfResources = new ArrayList();

        if (criteria == null) return listOfResources;

        List lookups =  getManagementServices().getRepository().loadClientResources(criteria);
        if (lookups != null && !lookups.isEmpty()) {

            for (Iterator it = lookups.iterator(); it.hasNext(); ) {

                    ResourceDescriptor rd = getManagementServices().createResourceDescriptor( (Resource) it.next());
                    if (wsTypes == null || wsTypes.contains(rd.getWsType()))
                    {
                        listOfResources.add(rd);
                        if (maxItems > 0 && listOfResources.size() == maxItems)
                        {
                            break;
                        }
                    }
            }
        }

        return listOfResources;
    }


    /**
     *
     *
     * @param uri - Must be a valid uri, it must exists and the user must have access to it.
     * @param queryString
     * @param wsTypes
     * @param recursive
     * @param maxItems
     * @return
     * @throws ServiceException
     */
    public List getResources(String uri, String queryString, List<String> wsTypes, boolean recursive, int maxItems) throws ServiceException
    {

        // 1. look is the uri to search in is valid:
        // This will rise an exception if the uri is not found...
        Resource resource = getManagementServices().locateResource(uri);

        if (resource instanceof Folder)
        {
            List<Folder> folders = new ArrayList<Folder>();
            folders.add((Folder) resource);
            filterFolderList(folders);
            if (folders.isEmpty())
            {
                throw new ServiceException(ServiceException.FORBIDDEN, "Folder not accessible");
            }
        }

        FilterCriteria criteria = FilterCriteria.createFilter();
        // Prepare the list of filters based on the request
        List<FilterElement> filters = new ArrayList<FilterElement>();

        if (recursive)
        {
            // As default, we set as ancestor filter the current uri
            FilterElement fe = FilterCriteria.createAncestorFolderFilter(uri);
            criteria.addFilterElement(fe);
        }
        else
        {
            // Filter resources that are children of this uri (folder?)
            FilterElement fe = FilterCriteria.createParentFolderFilter(uri);
            criteria.addFilterElement(fe);
        }

        // process q
        if (queryString != null && queryString.length() > 0)
        {
             // TODO: need to escape something?!!
             FilterElement feOr_left = new FilterElementOr(FilterCriteria.createPropertyLikeFilter("name", "%" + queryString + "%"),
                                                           FilterCriteria.createPropertyLikeFilter("label", "%" + queryString + "%"));

             FilterElementOr feOr = new FilterElementOr( feOr_left, FilterCriteria.createPropertyLikeFilter("description", "%" + queryString + "%"));

             criteria.addFilterElement(feOr);
        }

        // process type
        if (wsTypes != null && !wsTypes.isEmpty())
        {
            for (String resourceType : wsTypes)
            {
                 // The resource type is the one defined as wsType, so we need to translate it in a real class...
                 ResourceHandler resHandler = getManagementServices().getHandlerRegistry().getHandler(resourceType);

                 if (resHandler == null)
                 {
                    throw new ServiceException(ServiceException.RESOURCE_BAD_REQUEST, "Invalid resource type: " + resourceType);
                 }
                 criteria.setFilterClass(resHandler.getResourceType());
            }
        }

        return getResources(criteria, maxItems, wsTypes);
    }


    private void filterFolderList(List folderList) {
        if (folderList == null || folderList.isEmpty()) {
            return;
        }

        Set<String> roles = getCurrentUserRoles();
        for (Iterator i = folderList.iterator(); i.hasNext(); ) {
            Folder folder = (Folder)i.next();
            if (getManagementServices().getServiceConfiguration().getTempFolder().equals(folder.getURIString())) {
                boolean accessDenied = true;
                if (roles != null && roles.size() > 0) {
                    for (String role: roles) {
                        if (getManagementServices().getServiceConfiguration().getRoleToAccessTempFolder().equals(role)) {
                            accessDenied = false;
                            break;
                        }
                    }
                }

                if (accessDenied) {
                    i.remove();
                }
            }
        }
    }


    private Set<String> getCurrentUserRoles() {
        Set<String> roleNames = new HashSet<String>();

        Authentication authenticationToken = SecurityContextHolder.getContext().getAuthentication();
        if (authenticationToken == null) {
            return roleNames;
        }

        if (authenticationToken.getPrincipal() instanceof UserDetails) {
            UserDetails contextUserDetails = (UserDetails) authenticationToken.getPrincipal();
            for (GrantedAuthority authority: contextUserDetails.getAuthorities()) {
                roleNames.add(authority.getAuthority());
            }

        }

        return roleNames;
    }
}
