/*
 * Copyright (C) 2005 - 2011 Jaspersoft Corporation. All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased  a commercial license agreement from Jaspersoft,
 * the following license terms  apply:
 *
 * This program is free software: you can redistribute it and/or  modify
 * it under the terms of the GNU Affero General Public License  as
 * published by the Free Software Foundation, either version 3 of  the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero  General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public  License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package com.jaspersoft.jasperserver.api.engine.scheduling.service;

import java.util.List;

import com.jaspersoft.jasperserver.api.JasperServerAPI;
import com.jaspersoft.jasperserver.api.common.domain.ExecutionContext;
import com.jaspersoft.jasperserver.api.common.domain.ValidationErrors;
import com.jaspersoft.jasperserver.api.engine.scheduling.domain.ReportJob;


/**
 * The central report scheduling service.
 *
 * <p>
 * This interface contains report job management methods that should be used by
 * code which deals with report scheduling.
 * </p>
 *
 * @author Lucian Chirita (lucianc@users.sourceforge.net)
 * @version $Id: ReportSchedulingService.java 19921 2010-12-11 14:52:49Z tmatyashovsky $
 * @since 1.0
 */
@JasperServerAPI
public interface ReportSchedulingService {

    /**
     * Schedules a new report job.
     *
     * <p>
     * The job contains details on which report to execute, when to execute it
     * and what to do with the job output.
     * This method will persisted the job details and will register a trigger
     * in the scheduler to fire at the required points in time.
     * </p>
	 * 
	 * <p>
	 * An ID for the job will be automatically generated and returned as part of
	 * the scheduled job.
	 * </p>
     *
     * @param context the caller execution context
     * @param job the details of the job to schedule
     * @return the job details as scheduled
     */
	ReportJob scheduleJob(ExecutionContext context, ReportJob job);
	
    /**
     * Updates an existing report job.
     *
     * <p>
     * The existing job is determined by the ID of the job argument.
     * The details of the existing job will be updated according to the report
     * job object passed as argument.
     * </p>
     *
     * <p>
     * If any attribute of the job trigger has changed, the scheduler will drop
     * and recreate the trigger used to fire the job.
     * </p>
     *
     * @param context the caller execution context
     * @param job the job details to update
     * @throws ReportJobNotFoundException when a job with the specified ID is 
     * not found
     * @see ReportJob#getId()
     * @see ReportJob#getTrigger()
     * @see ReportJob#getVersion()
     */
	void updateScheduledJob(ExecutionContext context, ReportJob job)
			throws ReportJobNotFoundException;
	
    /**
     * Returns a list of active jobs defined for a report.
     *
     * <p>
     * The method will return a list of {@link ReportJobSummary} objects which
     * contain summary information about jobs along with runtime information
     * related to the execution status of the jobs.
     * </p>
     *
     * <p>
     * Only active jobs will be listed, no information is available for jobs
     * that have completed.
     * </p>
     *
     * @param context the caller execution context
     * @param reportUnitURI the repository URI/path of the report for which jobs
     * should be listed
     * @return a list of {@link ReportJobSummary} objects
     */
	List getScheduledJobs(ExecutionContext context, String reportUnitURI);
	
    /**
     * Returns the list of all active report jobs defined in the scheduler.
     *
     * <p>
     * All jobs that are accessible in the current context will be returned by
     * this method. 
     * </p>
     *
     * @param context the caller execution context
     * @return a list of {@link ReportJobSummary} objects
     * @since 2.1.0
     * @see #getScheduledJobs(ExecutionContext, String)
     */
	List getScheduledJobs(ExecutionContext context);

	/**
	 * Removes a scheduled job.
	 * 
	 * <p>
	 * This method deletes the persisted job details and unregisters the trigger
	 * from the scheduler.
	 * If a job with the specified ID is not found, no operation is performed.
	 * </p>
	 * 
     * @param context the caller execution context
	 * @param jobId the ID of the job to delete
	 * @see ReportJob#getId()
	 */
	void removeScheduledJob(ExecutionContext context, long jobId);

	/**
	 * Removes a list of scheduled jobs.
	 * 
	 * <p>
	 * The IDs which do not correspond to jobs will be ignored.
	 * </p>
	 * 
     * @param context the caller execution context
	 * @param jobIds the IDs of the jobs to delete
	 * @see #removeScheduledJob(ExecutionContext, long)
	 */
	void removeScheduledJobs(ExecutionContext context, long[] jobIds);
	
	/**
	 * Loads the details of a report job.
	 * 
     * @param context the caller execution context
	 * @param jobId the ID of the job to load
	 * @return the details of the requested job, or <code>null</code> if no job
	 * with the specified ID was found
	 * @see ReportJob#getId()
	 */
	ReportJob getScheduledJob(ExecutionContext context, long jobId);

	/**
	 * Checks a report job for validation errors.
	 * 
	 * <p>
	 * This method can be used to validate a report job prior to saving it.
	 * The job will be automatically validated on save, and the operation will
	 * fail if any validation errors are encountered.
	 * </p>
	 * 
     * @param context the caller execution context
	 * @param job the job details
	 * @return a validation object which contains errors if any found
	 * @see ValidationErrors#isError()
	 */
	ValidationErrors validateJob(ExecutionContext context, ReportJob job);

}
