/*
 * Copyright (C) 2005 - 2011 Jaspersoft Corporation. All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased  a commercial license agreement from Jaspersoft,
 * the following license terms  apply:
 *
 * This program is free software: you can redistribute it and/or  modify
 * it under the terms of the GNU Affero General Public License  as
 * published by the Free Software Foundation, either version 3 of  the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero  General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public  License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package com.jaspersoft.jasperserver.rest;


import com.jaspersoft.jasperserver.remote.ServiceException;
import com.jaspersoft.jasperserver.remote.ServicesUtils;

import java.io.IOException;
import java.io.PrintWriter;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.springframework.context.ApplicationContext;
import org.springframework.web.context.support.WebApplicationContextUtils;
/**
 *
 *
 * @author gtoffoli
 */
public class RESTServlet extends HttpServlet {

    private final static Log log = LogFactory.getLog(RESTServlet.class);

    private final static String BEAN_NAME_REST_SERVICE_REGISTRY = "restServiceRegistry";

    private static ServicesUtils servicesUtils = null;
    private static ApplicationContext applicationContext = null;
    private static RESTServicRegistry registry = null;


    /**
     * our initialize routine; subclasses should call this if they override it
     */
    @Override
    public void init() throws javax.servlet.ServletException {

        applicationContext = WebApplicationContextUtils.getWebApplicationContext(getServletContext());
        servicesUtils = new ServicesUtils(applicationContext);
        registry = (RESTServicRegistry)applicationContext.getBean(BEAN_NAME_REST_SERVICE_REGISTRY);
    }

    /**
     * The REST servlet check for the resource type that has been requested.<br>
     * The resource is represented by the url i.e. /rest/<b>resource</b> or /rest/<b>report</b>
     * The resource logic is implemented by extending a RESTService.<br>
     * The RESTService handled for a specific request is configured in the file shared-config\applicationContext-rest-services.xml. It is a map which
     * binds urls to RESTServices. The RESTService implementation is in charge to accept or not the specific method type.
     *
     * @param req  HttpServletRequest
     * @param resp  HttpServletResponse
     * @throws ServletException
     * @throws IOException
     */
    @Override
    protected void service(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {

        // This print writer is used only in case of exception, but in general exception should be always
        // handled inside the RESTService implementation, if available.
        PrintWriter pw = null;

        try {

            String serviceName = null;

            // Get the name of the service requested by the user
            String path = req.getPathInfo();
            if (path == null)
            {
               // error...
               RESTUtils.setStatusAndBody( HttpServletResponse.SC_NOT_FOUND, resp, "Service not valid");
               return;
            }

            serviceName = RESTUtils.extractServiceName(path);

            if (log.isDebugEnabled())
                log.debug("REST Service requested: " + serviceName);

            // If the service name is null or empty, return an error.
            if (serviceName == null || serviceName.length() == 0)
            {
                // Send unknow service error...
                RESTUtils.setStatusAndBody( HttpServletResponse.SC_NOT_FOUND, resp, "Service not valid");
                return;
            }

            RESTService service = registry.getService(serviceName);
            if (service == null)
            {
                RESTUtils.setStatusAndBody( HttpServletResponse.SC_NOT_FOUND, resp, "Service not valid");
                return;
            }

            // Lookup the service in the services registry
            service.execute(req, resp);
        }
        catch (ServiceException ex)
        {
            if (log.isDebugEnabled())
                log.debug("Error executing a REST service: " + ex.getErrorCode(), ex);
            // The servlet is able to map common ServiceException errors with most appropriate REST (HTTP) error codes...
            // This should be really just the latest case, when a more specific error is not available.
            // In general the service implementation should take care to send to the client the proper response.
            switch (ex.getErrorCode())
            {
                case ServiceException.RESOURCE_NOT_FOUND:
                {
                    RESTUtils.setStatusAndBody( HttpServletResponse.SC_NOT_FOUND , resp, ex.getLocalizedMessage());
                    break;
                }
                case ServiceException.RESOURCE_BAD_REQUEST:
                {
                    RESTUtils.setStatusAndBody( HttpServletResponse.SC_BAD_REQUEST , resp, ex.getLocalizedMessage());
                    break;
                }
                case ServiceException.FORBIDDEN:
                {
                    RESTUtils.setStatusAndBody( HttpServletResponse.SC_FORBIDDEN, resp, ex.getLocalizedMessage());
                    break;
                }
                default:
                {
                    RESTUtils.setStatusAndBody( HttpServletResponse.SC_INTERNAL_SERVER_ERROR , resp, ex.getLocalizedMessage());
                }
            }
        }

        catch (UnsupportedOperationException ex)
        {
            // This should Never happen. If it does, it's probably an implementation bug, or the remote service
            // implementation did not provide a better or specific error.
            log.error("Error executing a REST service", ex);
            RESTUtils.setStatusAndBody( HttpServletResponse.SC_NOT_IMPLEMENTED, resp, ex.getLocalizedMessage());
        }

        catch (Exception ex)
        {
            // This should Never happen. If it does, it's probably an implementation bug, or the remote service
            // implementation did not provide a better or specific error.
            log.error("Error executing a REST service", ex);
            RESTUtils.setStatusAndBody( HttpServletResponse.SC_INTERNAL_SERVER_ERROR , resp, ex.getLocalizedMessage());
        }
    }

    
}
