/*
 * iReport - Visual Designer for JasperReports.
 * Copyright (C) 2002 - 2009 Jaspersoft Corporation. All rights reserved.
 * http://www.jaspersoft.com
 *
 * Unless you have purchased a commercial license agreement from Jaspersoft,
 * the following license terms apply:
 *
 * This program is part of iReport.
 *
 * iReport is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * iReport is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with iReport. If not, see <http://www.gnu.org/licenses/>.
 */
package com.jaspersoft.jasperserver.remote;

import com.jaspersoft.jasperserver.api.engine.common.service.EngineService;
import com.jaspersoft.jasperserver.api.logging.audit.context.AuditContext;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Folder;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Resource;
import com.jaspersoft.jasperserver.api.metadata.common.domain.ResourceLookup;
import com.jaspersoft.jasperserver.api.metadata.common.service.RepositoryService;
import com.jaspersoft.jasperserver.api.metadata.view.domain.FilterCriteria;
import com.jaspersoft.jasperserver.api.metadata.xml.domain.impl.ResourceDescriptor;
import com.jaspersoft.jasperserver.remote.utils.AuditHelper;
import com.jaspersoft.jasperserver.remote.utils.RepositoryHelper;
import com.jaspersoft.jasperserver.ws.axis2.WSException;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import com.jaspersoft.jasperserver.ws.axis2.scheduling.ReportSchedulerService;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.context.MessageSource;
import org.springframework.security.AccessDeniedException;

/**
 * This class provides access to all the services used to manage JasperReports server.
 * Each service implementation is set in the applicationContext-remote-services.xml
 *
 *
 * @author gtoffoli
 */
public class ManagementServices {

    protected static final Log log = LogFactory.getLog(ManagementServices.class);
    private RepositoryService repository;
    private EngineService engine;
    private EngineService runReportEngine;
    private RepositoryHelper repositoryHelper;
    private ResourceHandlerRegistry handlerRegistry;
    private ServicesConfiguration servicesConfiguration;
    private AuditContext auditContext;
    private AuditHelper auditHelper;
    private MessageSource messageSource;
    private BeanFactory beanFactory;
    private ResourceActionResolver resourceActionResolver;
    private ReportSchedulerService reportSchedulerService;


    /**
     * Returns a message using the configured MessageSource
     *
     * @param messageCode
     * @param args
     * @param locale
     * @return
     */
    public String getMessage(String messageCode, Object[] args, Locale locale) {
        return getMessageSource().getMessage(messageCode, args, locale);
    }

    /**
     *
     *
     * @return RepositoryService
     */
    public RepositoryService getRepository() {
        return repository;
    }

    public ReportSchedulerService getSchedulerService(){
        return reportSchedulerService;
    }

    public EngineService getEngine() {
        return engine;
    }

    public EngineService getRunReportEngine() {
        return runReportEngine;
    }

    public ResourceHandlerRegistry getHandlerRegistry() {
        return handlerRegistry;
    }

    public RepositoryHelper getRepositoryHelper() {
        if (repositoryHelper == null) {
            repositoryHelper = new RepositoryHelper(engine);
        }
        return repositoryHelper;
    }

    public AuditHelper getAuditHelper() {
        if (auditHelper == null) {
            auditHelper = new AuditHelper(auditContext);
        }
        return auditHelper;
    }

    public ServicesConfiguration getServiceConfiguration() {
        return servicesConfiguration;
    }

    public void setAuditContext(AuditContext auditContext) {
        this.auditContext = auditContext;
    }

    public void setEngine(EngineService engine) {
        this.engine = engine;
    }

    public void setHandlerRegistry(ResourceHandlerRegistry handlerRegistry) {
        this.handlerRegistry = handlerRegistry;
    }

    public void setRepository(RepositoryService repository) {
        this.repository = repository;
    }

    public void setRunReportEngine(EngineService runReportEngine) {
        this.runReportEngine = runReportEngine;
    }

    public void setServiceConfiguration(ServicesConfiguration servicesConfiguration) {
        this.servicesConfiguration = servicesConfiguration;
    }

    public void setMessageSource(MessageSource messageSource) {
        this.messageSource = messageSource;
    }

    public void setBeanFactory(BeanFactory beanFactory) throws BeansException {
        this.beanFactory = beanFactory;
    }

    public void setReportSchedulerService(ReportSchedulerService reportSchedulerService) throws BeansException {
        this.reportSchedulerService = reportSchedulerService;
    }

    /**
     * Find a resource from an uri. This method is exensively used by the services to
     * manage the repository.
     * 
     * @param uri
     * @return
     * @throws ServiceException
     */
    public Resource locateResource(String uri) throws ServiceException {

        try {

            Resource res = null;

            String name;
            String folderName;

            int sep = uri.lastIndexOf(Folder.SEPARATOR);
            if (sep >= 0) {
                name = uri.substring(sep + Folder.SEPARATOR_LENGTH);
                folderName = uri.substring(0, sep);
            } else {
                // No separator: error
                throw new ServiceException("jsexception.invalid.uri", new Object[]{uri});
            }

            // Check if the folder is a RU first...
            if (folderName.endsWith("_files")) {
                String parentUri = folderName.substring(0, folderName.length() - "_files".length());
                log.warn("Loading uri: " + parentUri);
                Resource parentRes = getRepository().getResource(null, parentUri);
                if (parentRes != null) {
                    // The parent folder is a RU...
                    // Get the resource (quick way to check accessibility....)
                    ResourceDescriptor ruRd = createResourceDescriptor(folderName);

                    log.warn("Loaded RU " + res);
                    if (ruRd == null) {
                        // The user can not access to this RU...
                        return null;
                    }

                    res = getRepository().getResource(null, uri);
                    log.warn("Loaded resource " + uri + " " + res);
                }
            }

            if (res == null) {
                if (folderName.length() == 0) {
                    folderName = "/";
                }
                // why it does not work when i give it http://localhost:8080/jasperserver/rest/resources/ContentFiles/pdf
                FilterCriteria filterCriteria = new FilterCriteria();
                filterCriteria.addFilterElement(FilterCriteria.createParentFolderFilter(folderName));
                filterCriteria.addFilterElement(FilterCriteria.createPropertyEqualsFilter("name", name));

                // This filters with object level security
                // Will only get resources the user has access to

                List resources = getRepository().loadClientResources(filterCriteria);
                if (resources != null && !resources.isEmpty()) {
                    res = (Resource) resources.get(0);
                }
            }

            if (res == null) // try to look for a folder...
            {
                Folder folder = getRepository().getFolder(null, uri);
                if (folder != null) {
                    res = folder;
                } else {
                    throw new ServiceException(ServiceException.RESOURCE_NOT_FOUND, "Could not locate resource");
                }
            }
            return res;

        }
        catch (ServiceException ex)
        {
            throw ex;
        }
        catch (Exception ex)
        {
            if (ex instanceof AccessDeniedException)
            {
                throw new ServiceException( ServiceException.FORBIDDEN, ex.getLocalizedMessage() );
            }
            throw new ServiceException(ex);
        }
    }



    /**
     * the same as createResourceDescriptor( resource, null)
     *
     * @param resource
     * @return
     * @throws ServiceException
     */
    public ResourceDescriptor createResourceDescriptor( Resource resource) throws ServiceException
    {
        return createResourceDescriptor( resource, null);
    }

    /**
     * Create a ResourceDescriptor from a Resource.
     * The real type of this resource is saved in WsType
     *
     * @param resource
     * @param options
     * @return
     * @throws ServiceException
     */
    public ResourceDescriptor createResourceDescriptor( Resource resource, Map options) throws ServiceException
    {
        if (resource == null)
        {
            return null;
        }
        
    	if (resource instanceof ResourceLookup) {
    		throw new ServiceException("jsexception.resourcelookup.not.a.resource", new Object[] {resource.getClass().getName()});
    	}

    	ResourceHandler resourceHandler = getHandlerRegistry().getHandler(resource);
    	if (resourceHandler == null)
    	{
    		throw new ServiceException("No resource handler found for class " + resource.getClass().getName());
    	}

        return resourceHandler.get(resource, options);
    }


    /**
     * Same as createResourceDescriptor(uri, null)
     * @param uri
     * @return
     * @throws WSException
     */
    public ResourceDescriptor createResourceDescriptor(String uri) throws ServiceException
    {
        return createResourceDescriptor(uri, null);
    }


    /**
     * Creates a descriptor checking the special options in the map.
     *
     * @param uri
     * @param options
     * @return
     * @throws ServiceException
     */
    public ResourceDescriptor createResourceDescriptor(String uri, Map options) throws ServiceException
    {
    	Resource res = locateResource(uri);
    	return createResourceDescriptor( res, options);
        
    }

    /**
     * @return the resourceActionResolver
     */
    public ResourceActionResolver getResourceActionResolver() {
        return resourceActionResolver;
    }

    /**
     * @param resourceActionResolver the resourceActionResolver to set
     */
    public void setResourceActionResolver(ResourceActionResolver resourceActionResolver) {
        this.resourceActionResolver = resourceActionResolver;
    }

    /**
     * @return the messageSource
     */
    public MessageSource getMessageSource() {
        return messageSource;
    }

}
