/*
 * Copyright (C) 2005 - 2011 Jaspersoft Corporation. All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased  a commercial license agreement from Jaspersoft,
 * the following license terms  apply:
 *
 * This program is free software: you can redistribute it and/or  modify
 * it under the terms of the GNU Affero General Public License  as
 * published by the Free Software Foundation, either version 3 of  the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero  General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public  License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package com.jaspersoft.jasperserver.api.metadata.view.service.impl;

import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import javax.naming.NamingException;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;

import junit.framework.TestCase;
import junit.textui.TestRunner;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.mock.web.MockHttpServletRequest;
import org.springframework.mock.web.MockHttpServletResponse;
import org.springframework.security.Authentication;
import org.springframework.security.GrantedAuthority;
import org.springframework.security.GrantedAuthorityImpl;
import org.springframework.security.context.SecurityContextHolder;
import org.springframework.security.ui.webapp.AuthenticationProcessingFilter;
import org.springframework.security.util.FilterChainProxy;

import com.jaspersoft.jasperserver.api.metadata.user.domain.Role;
import com.jaspersoft.jasperserver.api.metadata.user.domain.User;
import com.jaspersoft.jasperserver.util.test.BaseJasperServerTest;

/**
 * Test authentication with principals that are strings and User Details. We need to
 * exercise the MetadataAuthenticationProcessingFilter via the Acegi FilterChainProxy.
 *
 * Test using an external service and synchronizing roles.
 *
 * Create 2 dummy external services
 *      - principal is a string
 *      - principal is a userDetails
 *
 * users will be indicated as being external. Set their roles.
 *
 * Use each of these via a ProviderManager, and the MetadataAuthenticationProcessingFilter.
 *
 * Reauthenticate with changed roles and check that the role synch happens correctly.
 *
 * Add a JS internal role and check that a login does not remove it
 *
 * @author swood
 * @version $Id: AuthenticationTest.java 19930 2010-12-11 15:17:19Z tmatyashovsky $
 */
public class AuthenticationTest extends BaseJasperServerTest {

    private static Log log = LogFactory.getLog(AuthenticationTest.class);

    public static final String testExternalUserName = "TestExternalUser";
    public static final String testExternalPassword = "password";
    public static final String testExternalRoleNamePrefix = "TestExternalRole";
    public static final String testInternalRoleNamePrefix = "ROLE_TestInternalRole";

    public static final String testUserDetailsUserName = "tomcat";
    public static final String testUserDetailsPassword = "tomcat";

	Properties jdbcProps;

	public AuthenticationTest(String name) {
        super(name);
        setAutowireMode(AUTOWIRE_BY_NAME);
    }

    public static void main(String[] args) {
        TestRunner.run(AuthenticationTest.class);
    }

    public void onSetUp() throws Exception {
        log.debug("Setup");
    }

    public void onTearDown() {
        log.debug("Tear down");

    	//if (jdbcProps.getProperty("test.databaseFlavor") != null &&
    	//		(jdbcProps.getProperty("test.databaseFlavor").startsWith("postgres") ||
    	//		 jdbcProps.getProperty("test.databaseFlavor").startsWith("oracle") ||
    	//		 jdbcProps.getProperty("test.databaseFlavor").startsWith("sqlserver"))) {   // TODO: fix and remove
    	//	log.warn("DATABASE FLAVOR IS POSTGRESQL, ORACLE, SQLSERVER: Skipping test: " + "tearDown");
		//	return;
    	//}
        
        // get rid of objects we don't want in samples

        getUserAuthorityService().deleteUser(null, testExternalUserName);

        User workingUser = getUserAuthorityService().getUser(null, testUserDetailsUserName);
        Role workRole = getUserAuthorityService().getRole(null, "ROLE_" + testExternalRoleNamePrefix + "0");
        getUserAuthorityService().removeRole(null, workingUser, workRole);

        for (int i = 0; i < 3; i++) {
        	getUserAuthorityService().deleteRole(null, "ROLE_" + testExternalRoleNamePrefix + i);
        	getUserAuthorityService().deleteRole(null, testInternalRoleNamePrefix + i);
        }
    }

    public Role addRole(User user, String roleName) {
        log.debug("addRole");

        Role role = getUserAuthorityService().getRole(null, roleName);

        if (role == null) {
                role = getUserAuthorityService().newRole(null);
                //System.out.println("role class: " + role.getClass().getName());
                role.setRoleName(roleName);
                role.setExternallyDefined(false);

                getUserAuthorityService().putRole(null, role);
        }

        getUserAuthorityService().addRole(null, user, role);
        return role;
    }

    public void loginUser(String userName, String password, int noOfExternalRoles) throws Exception {

       FilterChainProxy filterChainProxy = (FilterChainProxy) applicationContext.getBean("filterChainProxy",
            FilterChainProxy.class);

        filterChainProxy.init(new MockFilterConfig());

        GrantedAuthority[] grantedAuthorities = generateGrantedAuthorities(testExternalRoleNamePrefix, noOfExternalRoles);

        MockStringAuthenticationProvider mockAuthenticationProvider = (MockStringAuthenticationProvider) applicationContext.getBean("stringPrincipalAuthenticationProvider",
            MockStringAuthenticationProvider.class);

        // Do the authentication
        mockAuthenticationProvider.setGrantedAuthorities(grantedAuthorities);

        AuthenticationProcessingFilter authenticationProcessingFilter = (AuthenticationProcessingFilter) applicationContext.getBean("authenticationProcessingFilter",
            AuthenticationProcessingFilter.class);

        MockHttpServletRequest request = new MockHttpServletRequest();
        request.setServletPath(authenticationProcessingFilter.getFilterProcessesUrl());
        request.setRequestURI(authenticationProcessingFilter.getFilterProcessesUrl());

        log.debug("request.getContextPath(): '" + request.getContextPath() + "', request.getRequestURI(): '" + request.getRequestURI() + "'");

        request.setParameter(AuthenticationProcessingFilter.SPRING_SECURITY_FORM_USERNAME_KEY, userName);
        request.setParameter(AuthenticationProcessingFilter.SPRING_SECURITY_FORM_PASSWORD_KEY, password);

        MockHttpServletResponse response = new MockHttpServletResponse();
        MockFilterChain chain = new MockFilterChain(true);

        // Do the authentication with the AuthenticationProcessingFilter
        filterChainProxy.doFilter(request, response, chain);

        Authentication auth = SecurityContextHolder.getContext().getAuthentication();

        assertTrue("login not successful", auth != null);

        // Now we are authenticated, run the filter again to get to the MetadataAuthenticationProcessingFilter.
        // This is what happens with the normal login process

        request = new MockHttpServletRequest();
        request.setServletPath("/a/random/url");
        response = new MockHttpServletResponse();
        chain = new MockFilterChain(true);

        filterChainProxy.doFilter(request, response, chain);

    }

    public void testRolesForExternalStringPrincipal() throws Exception {

    	//if (jdbcProps.getProperty("test.databaseFlavor") != null &&
    	//		(jdbcProps.getProperty("test.databaseFlavor").startsWith("postgres") ||
    	//		 jdbcProps.getProperty("test.databaseFlavor").startsWith("oracle") ||
    	//		 jdbcProps.getProperty("test.databaseFlavor").startsWith("sqlserver"))) {   // TODO: fix and remove
    	//	log.warn("DATABASE FLAVOR IS POSTGRESQL, ORACLE, SQLSERVER: Skipping test: " + "testRolesForExternalStringPrincipal");
		//	return;
    	//}
    	
        loginUser(testExternalUserName, testExternalPassword, 3);

        User workingUser = getUserAuthorityService().getUser(null, testExternalUserName);

        // Because we are adding the default ROLE_USER, there will be 4 roles
        checkUserAndRoles(workingUser, "first login", 3, 1);

        // Logout and login with less roles

        SecurityContextHolder.getContext().setAuthentication(null);

        loginUser(testExternalUserName, testExternalPassword, 2);

        workingUser = getUserAuthorityService().getUser(null, testExternalUserName);

        checkUserAndRoles(workingUser, "second login", 2, 1);

        // Add an internal role to the user and see what happens when they login again

        addRole(workingUser, testInternalRoleNamePrefix + "0");

        SecurityContextHolder.getContext().setAuthentication(null);

        loginUser(testExternalUserName, testExternalPassword, 2);

        workingUser = getUserAuthorityService().getUser(null, testExternalUserName);

        checkUserAndRoles(workingUser, "third login", 2, 2);
    }

    public void testRolesForExternalUserDetailsPrincipal() throws Exception {

    	//if (jdbcProps.getProperty("test.databaseFlavor") != null &&
    	//		(jdbcProps.getProperty("test.databaseFlavor").startsWith("postgres") ||
    	//		 jdbcProps.getProperty("test.databaseFlavor").startsWith("oracle") ||
    	//		 jdbcProps.getProperty("test.databaseFlavor").startsWith("sqlserver"))) {   // TODO: fix and remove
    	//	log.warn("DATABASE FLAVOR IS POSTGRESQL, ORACLE, SQLSERVER: Skipping test: " + "testRolesForExternalUserDetailsPrincipal");
		//	return;
    	//}
    	
        loginUser(testUserDetailsUserName, testUserDetailsPassword, 0);

        User workingUser = getUserAuthorityService().getUser(null, testUserDetailsUserName);

        // Because we are adding the default ROLE_USER, there will be 2 roles
        checkUserAndRoles(workingUser, "Userdetails first login", 1, 1);

        // Logout and login

        SecurityContextHolder.getContext().setAuthentication(null);

        loginUser(testUserDetailsUserName, testUserDetailsPassword, 0);

        workingUser = getUserAuthorityService().getUser(null, testUserDetailsUserName);

        checkUserAndRoles(workingUser, "Userdetails second login", 1, 1);

        // Add an internal role to the user and see what happens when they login again

        addRole(workingUser, testInternalRoleNamePrefix + "0");

        SecurityContextHolder.getContext().setAuthentication(null);

        loginUser(testUserDetailsUserName, testUserDetailsPassword, 0);

        workingUser = getUserAuthorityService().getUser(null, testUserDetailsUserName);

        checkUserAndRoles(workingUser, "Userdetails third login", 1, 2);
    }

    private void checkUserAndRoles(User user, String prefix, int expectedExternalCount, int expectedInternalCount) {

        assertTrue(prefix + ": missing user", user != null);
        assertTrue(prefix + ": not external", user.isExternallyDefined());

        Set roles = user.getRoles();

        assertTrue(prefix + ": not " + (expectedExternalCount + expectedInternalCount) + " roles, got " + roles.size(),
                expectedExternalCount + expectedInternalCount == roles.size());

        int externalCount = 0, internalCount = 0;

        for (Iterator it = roles.iterator(); it.hasNext(); ) {
            Role r = (Role) it.next();

            if (r.isExternallyDefined()) {
                externalCount++;
            } else {
                internalCount++;
            }
        }
        assertTrue(prefix + ": saw " + externalCount + " external roles, not " + expectedExternalCount, externalCount == expectedExternalCount);
        assertTrue(prefix + ": saw " + internalCount + " internal roles, not " + expectedInternalCount, internalCount == expectedInternalCount);

    }

    private static GrantedAuthority[] generateGrantedAuthorities(String prefix, int length) {
        GrantedAuthority[] newAuthorities = new GrantedAuthority[length];

        for (int i = 0; i < length; i++) {
            newAuthorities[i] = new GrantedAuthorityImpl(prefix + i);
        }
        return newAuthorities;
    }

    /**
     * From Acegi test cases
     *
     * @author Ben Alex
     * @version $Id: AuthenticationTest.java 19930 2010-12-11 15:17:19Z tmatyashovsky $
     */
    public class MockFilterConfig implements FilterConfig {
        //~ Instance fields ========================================================

        private Map map = new HashMap();

        //~ Methods ================================================================

        public String getFilterName() {
            throw new UnsupportedOperationException("mock method not implemented");
        }

        public String getInitParameter(String arg0) {
            Object result = map.get(arg0);

            if (result != null) {
                return (String) result;
            } else {
                return null;
            }
        }

        public Enumeration getInitParameterNames() {
            throw new UnsupportedOperationException("mock method not implemented");
        }

        public void setInitParmeter(String parameter, String value) {
            map.put(parameter, value);
        }

        public ServletContext getServletContext() {
            throw new UnsupportedOperationException("mock method not implemented");
        }
    }

    /**
     * A mock <code>FilterChain</code>.
     *
     * From Acegi test cases
     *
     * @author Ben Alex
     * @version $Id: AuthenticationTest.java 19930 2010-12-11 15:17:19Z tmatyashovsky $
     */
    public class MockFilterChain implements FilterChain {
        //~ Instance fields ========================================================

        private boolean expectToProceed;

        //~ Constructors ===========================================================

        public MockFilterChain(boolean expectToProceed) {
            this.expectToProceed = expectToProceed;
        }

        //~ Methods ================================================================

        public void doFilter(ServletRequest request, ServletResponse response)
                throws IOException, ServletException {
            if (expectToProceed) {
                TestCase.assertTrue(true);
            } else {
                TestCase.fail("Did not expect filter chain to proceed");
            }
        }
    }

}
