/*
 * Copyright (C) 2005 - 2011 Jaspersoft Corporation. All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased  a commercial license agreement from Jaspersoft,
 * the following license terms  apply:
 *
 * This program is free software: you can redistribute it and/or  modify
 * it under the terms of the GNU Affero General Public License  as
 * published by the Free Software Foundation, either version 3 of  the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero  General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public  License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package com.jaspersoft.jasperserver.api.engine.scheduling.service;

import com.jaspersoft.jasperserver.api.JasperServerAPI;
import com.jaspersoft.jasperserver.api.common.domain.ExecutionContext;
import com.jaspersoft.jasperserver.api.common.domain.ValidationErrors;
import com.jaspersoft.jasperserver.api.engine.scheduling.domain.ReportJob;
import com.jaspersoft.jasperserver.api.engine.scheduling.domain.ReportJobRuntimeInformation;
import com.jaspersoft.jasperserver.api.engine.scheduling.domain.ReportJobSummary;

/**
 * A service that provides methods related to the execution of report jobs.
 *  
 * <p>
 * This service would not be directly called by code that does report scheduling
 * in regular scenarios, such code would normally use {@link ReportSchedulingService}.
 * This report job execution service is used internal by the central scheduling service.
 * </p>
 * 
 * @author Lucian Chirita (lucianc@users.sourceforge.net)
 * @version $Id: ReportJobsScheduler.java 19921 2010-12-11 14:52:49Z tmatyashovsky $
 * @since 1.0
 * @see ReportSchedulingService
 * @see ReportJobsPersistenceService
 */
@JasperServerAPI
public interface ReportJobsScheduler {

	/**
	 * Registers a trigger for a new report job.
	 * 
     * @param context the caller execution context
	 * @param job the report job details
	 * @see ReportSchedulingService#scheduleJob(ExecutionContext, ReportJob)
	 */
	void scheduleJob(ExecutionContext context, ReportJob job);

	/**
	 * Recreates the trigger for an existing report job.
	 * 
	 * <p>
	 * This action is performed when the schedule part of a report job is
	 * updated.
	 * </p>
	 * 
     * @param context the caller execution context
	 * @param job the updated report job details
	 * @see ReportSchedulingService#updateScheduledJob(ExecutionContext, ReportJob)
	 * @see ReportJob#getTrigger()
	 */
	void rescheduleJob(ExecutionContext context, ReportJob job);

	/**
	 * Removes the trigger for a scheduled job.
	 * 
     * @param context the caller execution context
	 * @param jobId the ID of the job whose trigger is to be removed
	 * @see ReportSchedulingService#removeScheduledJob(ExecutionContext, long)
	 * @see ReportSchedulingService#removeScheduledJobs(ExecutionContext, long[])
	 */
	void removeScheduledJob(ExecutionContext context, long jobId);
	
	/**
	 * Registers a scheduler listener.
	 * 
	 * The listener callback methods will be invoked when the corresponding
	 * actions are performed by this services.
	 * 
	 * @param listener the listener to register with the service
	 * @see #removeReportSchedulerListener(ReportSchedulerListener)
	 */
	void addReportSchedulerListener(ReportSchedulerListener listener);
	
	/**
	 * Unregisters a scheduler listener.
	 * 
	 * <p>
	 * If the listener was not previously registered, no action is performed.
	 * </p>
	 * 
	 * @param listener the listener to unregister
	 */
	void removeReportSchedulerListener(ReportSchedulerListener listener);
	
	/**
	 * Retrieves runtime information for a list of jobs.
	 * 
	 * <p>
	 * The services locates the trigger for each job in the list, and uses it
	 * to fill information related to the execution state of the job.
	 * </p>
	 * 
     * @param context the caller execution context
	 * @param jobIds the job IDs
	 * @return runtime information for the jobs
	 * @see ReportSchedulingService#getScheduledJobs(ExecutionContext, String)
	 * @see ReportSchedulingService#getScheduledJobs(ExecutionContext)
	 * @see ReportJobSummary#getRuntimeInformation()
	 */
	ReportJobRuntimeInformation[] getJobsRuntimeInformation(ExecutionContext context, long[] jobIds);
	
	/**
	 * Performs validation checks related to the job trigger for a report job.
	 * 
	 * @param job the report job to validate
	 * @param errors the validation object in which errors are collected
	 * @see ReportSchedulingService#validateJob(ExecutionContext, ReportJob)
	 * @since 1.2.0
	 */
	void validate(ReportJob job, ValidationErrors errors);

}
