/*
 * Copyright (C) 2005 - 2011 Jaspersoft Corporation. All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased  a commercial license agreement from Jaspersoft,
 * the following license terms  apply:
 *
 * This program is free software: you can redistribute it and/or  modify
 * it under the terms of the GNU Affero General Public License  as
 * published by the Free Software Foundation, either version 3 of  the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero  General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public  License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package com.jaspersoft.jasperserver.api.logging.access.service.impl;

import com.jaspersoft.jasperserver.api.logging.context.LoggableEvent;
import com.jaspersoft.jasperserver.api.logging.access.domain.hibernate.RepoAccessEvent;
import com.jaspersoft.jasperserver.api.logging.access.domain.AccessEvent;
import com.jaspersoft.jasperserver.api.metadata.common.service.impl.HibernateDaoImpl;
import com.jaspersoft.jasperserver.api.metadata.common.service.impl.hibernate.PersistentObjectResolver;
import com.jaspersoft.jasperserver.api.metadata.common.service.impl.hibernate.HibernateRepositoryService;
import com.jaspersoft.jasperserver.api.metadata.common.service.ResourceFactory;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Resource;
import com.jaspersoft.jasperserver.api.metadata.common.domain.impl.IdedObject;
import com.jaspersoft.jasperserver.api.metadata.user.domain.User;

import java.util.List;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.ArrayList;

import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;

/**
 * @author Sergey Prilukin
 * @version $Id$
 */
public class AccessServiceImpl extends HibernateDaoImpl implements AccessService, PersistentObjectResolver {

    private HibernateRepositoryService hibernateRepositoryService;
    private PersistentObjectResolver persistentUserResolver;
    private int maxAccessEventAge;
    private ResourceFactory persistentClassFactory;
    private ResourceFactory clientClassFactory;

    public void setMaxAccessEventAge(int maxAccessEventAge) {
        this.maxAccessEventAge = maxAccessEventAge;
    }

    public void setHibernateRepositoryService(HibernateRepositoryService hibernateRepositoryService) {
        this.hibernateRepositoryService = hibernateRepositoryService;
    }

    public void setPersistentUserResolver(PersistentObjectResolver persistentUserResolver) {
        this.persistentUserResolver = persistentUserResolver;
    }

    public void setPersistentClassFactory(ResourceFactory persistentClassFactory) {
        this.persistentClassFactory = persistentClassFactory;
    }

    public void setClientClassFactory(ResourceFactory clientClassFactory) {
        this.clientClassFactory = clientClassFactory;
    }

    public Object getPersistentObject(Object clientObject) {
        if (clientObject == null) {
            return null;
        } else if (clientObject instanceof Resource) {
            return hibernateRepositoryService.getRepoResource((Resource)clientObject);
        } else if (clientObject instanceof User) {
            return persistentUserResolver.getPersistentObject(clientObject);
        }

        return null;
    }

    public void saveEvent(LoggableEvent loggableEvent) {
        AccessEvent accessEvent = (AccessEvent)loggableEvent;
        if (hibernateRepositoryService.resourceExists(null, accessEvent.getResource().getURI())) {
            RepoAccessEvent repoAccessEvent = new RepoAccessEvent();
            repoAccessEvent.copyFromClient(loggableEvent, this);
            getHibernateTemplate().save(repoAccessEvent);            
        }
    }

    public void saveEvents(List<LoggableEvent> loggableEvents) {
        if (loggableEvents == null || loggableEvents.isEmpty()) {
            return;
        }

        for (LoggableEvent loggableEvent: loggableEvents) {
            saveEvent(loggableEvent);
        }
    }

    public List<AccessEvent> getAllEvents(int firstResult, int maxResults) {
        DetachedCriteria criteria =
                DetachedCriteria.forClass(persistentClassFactory.getImplementationClass(AccessEvent.class));
        criteria.addOrder(Order.asc("eventDate"));

        List results = getHibernateTemplate().findByCriteria(criteria, firstResult, maxResults);

        if (results != null && !results.isEmpty()) {
            List<AccessEvent> clientEventsList = new ArrayList<AccessEvent>(results.size());
            for (Object event: results) {
                AccessEvent accessEvent = (AccessEvent) ((IdedObject) event).toClient(clientClassFactory);
                clientEventsList.add(accessEvent);
            }

            return clientEventsList;
        } else {
            return Collections.emptyList();
        }
    }

    public int getAccessEventsCount() {
        DetachedCriteria criteria =
                DetachedCriteria.forClass(persistentClassFactory.getImplementationClass(AccessEvent.class));
        criteria.setProjection(Projections.rowCount());
        return (Integer) getHibernateTemplate().findByCriteria(criteria).get(0);
    }

    public void purgeAccessEvents() {
        Calendar cal = Calendar.getInstance();
        cal.add(Calendar.DATE, -maxAccessEventAge);
        Date last = cal.getTime();
        
        getHibernateTemplate().bulkUpdate("delete RepoAccessEvent e where e.eventDate < ?", last);
    }
}
