/*
 * JaLingo, http://jalingo.sourceforge.net/
 *
 * Copyright (c) 2002-2006 Oleksandr Shyshko
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package ja.lingo.setup;

import java.io.*;

public abstract class OsDependent {
    protected String defaultDestination;
    protected String defaultDirName;
    protected boolean canCreateShortcuts;
    protected String javaExe = "java";

    public static OsDependent create() {
        String osName = System.getProperty( "os.name" ).toLowerCase();
        if ( osName == null ) {
            osName = "";
        }

        if ( osName.contains( "windows" ) ) {
            return new Windows();
        } else if ( osName.contains( "mac" ) ) {
            return new Mac();
        } else {
            return new Linux();
        }
    }

    public String getDefaultDestination() {
        return defaultDestination;
    }
    public String getDefaultDirName() {
        return defaultDirName;
    }
    public boolean canCreateShortcuts() {
        return canCreateShortcuts;
    }

    public void doPostinstall( String destination, String jar ) throws IOException {
        // by default, do nothing
    }
    public String getJavaCmd() {
        return System.getProperty( "java.home" ) + File.separator + "bin" + File.separator + javaExe;
    }

    public String getJavaCmdArgs( String destination, String jar ) {
        return "-jar \"" + calculateDestination( destination, jar ) + "\"";
    }
    public String calculateDestination( String destination, String fileName ) {
        return destination + File.separator + fileName;
    }

    private static class Windows extends OsDependent {
        {
            // TODO detect Windows system disk and use it instead "C:" (look in env)
            defaultDestination = "C:\\Program Files";
            defaultDirName = "JaLingo";
            canCreateShortcuts = true;
            javaExe = "javaw";
        }
    }

    private static class Mac extends OsDependent {
        {
            defaultDestination = "/Applications";
            defaultDirName = "jalingo";
        }
    }

    private static class Linux extends OsDependent {
        {
            defaultDestination = "/usr/local";
            defaultDirName = "jalingo";
        }

        public void doPostinstall( String destination, String jar ) throws IOException {
            String script = destination + File.separator + "jalingo.bash";

            PrintStream ps = null;
            try {
                ps = new PrintStream( new FileOutputStream( script ) );
                ps.println( "#!/bin/bash" );
                ps.println();
                ps.print( getJavaCmd() + " " + getJavaCmdArgs( destination, jar ));
            } finally {
                if ( ps != null ) {
                    ps.close();
                }
            }

            // TODO set executable flag
        }
    }
}