/*
 * JaLingo, http://jalingo.sourceforge.net/
 *
 * Copyright (c) 2002-2006 Oleksandr Shyshko
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package ja.lingo.setup;

import ja.centre.gui.mediator.WindowDraggerMediator;
import ja.lingo.application.util.JaLingoColors;
import ja.lingo.application.util.progress.ProgressGlassPane;
import ja.lingo.application.util.starlabel.StarLabel;
import ja.lingo.application.JaLingoInfo;
import ja.lingo.setup.panels.Panel0Welcome;
import ja.lingo.setup.panels.Panel1License;
import ja.lingo.setup.panels.Panel2Destination;
import ja.lingo.setup.panels.Panel3Finish;

import javax.swing.*;
import javax.swing.plaf.ColorUIResource;
import javax.swing.plaf.FontUIResource;
import javax.swing.plaf.metal.MetalLookAndFeel;
import javax.swing.plaf.metal.OceanTheme;
import java.awt.*;
import static java.awt.BorderLayout.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.Arrays;

public class JaLingoSetup {
    private JButton backButton;
    private JButton nextButton;
    private JButton cancelButton;
    private JLabel cancelButtonReplacement;

    private Panel2Destination panel2Destination;
    private Panel3Finish panel3Finish;

    private JPanel cardPanel;
    private JPanel[] panels;
    private int step;

    private String[] nextButtonTexts = new String[] {
            "Next >",
            "I Agree >",
            "Install >",
            "Finish"
    };
    private JFrame frame;

    private int previousPanelIndex;
    private StarLabel starLabel;

    public JaLingoSetup() {
        initializeLookAndFeel();
        initializeButtons();
        initializePanels();

        starLabel = new StarLabel( GuiHelper.TITLE, false );
        starLabel.start();

        JPanel buttonPanel = createButtonPanel();

        cardPanel.setBorder( BorderFactory.createEmptyBorder( 0, GuiHelper.GAP, 0, GuiHelper.GAP ) );
        buttonPanel.setBorder( BorderFactory.createEmptyBorder( 0, GuiHelper.GAP, GuiHelper.GAP, GuiHelper.GAP ) );

        // NOTE: wow... lots of spagetti ;) size of the installer is the matter
        JPanel panel = new JPanel( new BorderLayout( GuiHelper.GAP, GuiHelper.GAP ) );
        panel.add( starLabel, NORTH );
        panel.add( cardPanel, CENTER );
        panel.add( GuiHelper.wrap( buttonPanel, EAST ), SOUTH );
        panel.setBackground( Color.WHITE );
        panel.setBorder( BorderFactory.createLineBorder( JaLingoColors.COLOR ) );

        frame = new JFrame();
        frame.setContentPane( panel );
        frame.setTitle( GuiHelper.TITLE );
        frame.setIconImage( new ImageIcon( GuiHelper.getResource( GuiHelper.RESOURCE_FRAME_ICON ) ).getImage() );
        frame.setDefaultCloseOperation( JFrame.DO_NOTHING_ON_CLOSE );
        frame.setResizable( false );
        frame.addWindowListener( new WindowAdapter() {
            public void windowClosing( WindowEvent e ) {
                cancel();
            }
        } );
        frame.setUndecorated( true );

        new WindowDraggerMediator( starLabel );

        fireNavigated();

        frame.setSize( 500, 400 );
        frame.setLocationRelativeTo( null );
        frame.setVisible( true );
    }

    public static void initializeLookAndFeel() {
        try {
            MetalLookAndFeel laf = new MetalLookAndFeel();
            MetalLookAndFeel.setCurrentTheme( new OceanTheme() {
                private final FontUIResource FONT = new FontUIResource( new Font( "Arial", Font.PLAIN, 12 ) );

                protected ColorUIResource getSecondary2() {
                    return new ColorUIResource( JaLingoColors.COLOR );
                }

                public FontUIResource getControlTextFont() {
                    return FONT;
                }

                public FontUIResource getMenuTextFont() {
                    return FONT;
                }

                public void addCustomEntriesToTable( UIDefaults table ) {
                    super.addCustomEntriesToTable( table );

                    java.util.List buttonGradient = Arrays.asList(
                            new Object[] { .7f, 0f,
                                    new ColorUIResource( 0xDDE8F3 ), getWhite(), getSecondary2() } );

                    table.put( "Button.gradient", buttonGradient );
                }
            } );
            UIManager.setLookAndFeel( laf );
        } catch ( UnsupportedLookAndFeelException e ) {
            e.printStackTrace();// NOTE: just log it and continue
        }

        UIDefaults defaults = UIManager.getDefaults();
        defaults.put( "OptionPane.background", Color.WHITE );
        defaults.put( "Panel.background", Color.WHITE );
        defaults.put( "CheckBox.background", Color.WHITE );
    }

    private void initializeButtons() {
        backButton = new JButton( "< Back" );
        backButton.addActionListener( new ActionListener() {
            public void actionPerformed( ActionEvent e ) {
                back();
            }
        } );

        nextButton = new JButton();
        nextButton.addActionListener( new ActionListener() {
            public void actionPerformed( ActionEvent e ) {
                next();
            }
        } );

        cancelButton = new JButton( "Cancel" );
        cancelButton.addActionListener( new ActionListener() {
            public void actionPerformed( ActionEvent e ) {
                cancel();
            }
        } );

        cancelButtonReplacement = new JLabel();
        cancelButtonReplacement.setVisible( false );

        backButton.setFocusable( false );
        nextButton.setFocusable( false );
        cancelButton.setFocusable( false );

        Dimension preferredSize = new JButton( nextButtonTexts[1] ).getPreferredSize();
        backButton.setPreferredSize( preferredSize );
        nextButton.setPreferredSize( preferredSize );
        cancelButton.setPreferredSize( preferredSize );
        cancelButtonReplacement.setPreferredSize( preferredSize );
    }

    private void initializePanels() {
        panel2Destination = new Panel2Destination(
                JaLingoSetupHelper.getDefaultDestination(),
                JaLingoSetupHelper.getDefaultDirName(),
                JaLingoSetupHelper.canCreateShortcuts() );

        panel3Finish = new Panel3Finish();

        panels = new JPanel[] {
                new Panel0Welcome(),
                new Panel1License(),
                panel2Destination,
                panel3Finish
        };

        cardPanel = new JPanel( new CardLayout() );
        for ( JPanel panel : panels ) {
            cardPanel.add( panel, panel.getClass().getName() );
        }
    }

    private JPanel createButtonPanel() {
        JPanel buttonPanel = new JPanel();
        new BoxLayout( buttonPanel, BoxLayout.X_AXIS );

        buttonPanel.add( backButton );
        buttonPanel.add( nextButton );
        buttonPanel.add( cancelButton );
        buttonPanel.add( cancelButtonReplacement );
        return buttonPanel;
    }

    private void fireNavigated() {
        if ( step == panels.length ) {
            finish();
            return;
        }

        Class panelClass = panels[step].getClass();

        // if install was just clicked
        if ( panelClass == Panel3Finish.class ) {
            install();
        } else {
            goTo( panelClass );
        }

    }
    private void goTo( Class panelClass ) {
        ((CardLayout) cardPanel.getLayout()).show( cardPanel, panelClass.getName() );
        nextButton.setText( nextButtonTexts[step] );

        if ( panelClass == Panel0Welcome.class || panelClass == Panel3Finish.class ) {
            backButton.setVisible( false );
        } else {
            backButton.setVisible( true );
        }

        if ( panelClass == Panel3Finish.class ) {
            cancelButton.setVisible( false );
            cancelButtonReplacement.setVisible( true );
        }
    }

    private void cancelNavigation() {
        step = previousPanelIndex;
    }

    private void showError( Exception e ) {
        e.printStackTrace();
        JOptionPane.showMessageDialog( frame, e.getMessage(),
                GuiHelper.TITLE, JOptionPane.ERROR_MESSAGE );
    }

    private void back() {
        previousPanelIndex = step;
        step--;

        fireNavigated();
    }

    private void next() {
        previousPanelIndex = step;
        step++;

        fireNavigated();
    }

    private void cancel() {
        if ( JOptionPane.showConfirmDialog( cardPanel,
                "Are you sure you want to quit " + GuiHelper.TITLE + "?",
                GuiHelper.TITLE,
                JOptionPane.YES_NO_OPTION,
                JOptionPane.QUESTION_MESSAGE ) == JOptionPane.YES_OPTION ) {
            System.exit( 0 );
        }
    }

    private void install() {
        ProgressGlassPane.install( frame, new JLabel( "Installing..." ) );

        new Thread() {
            public void run() {
                try {
                    // install in background
                    JaLingoSetupHelper.install(
                            panel2Destination.getDestination(),
                            panel2Destination.isDesktopSelected(),
                            panel2Destination.isStartMenuSelected() );

                    // switch to final step
                    SwingUtilities.invokeLater( new Runnable() {
                        public void run() {
                            ProgressGlassPane.restore( frame );

                            goTo( Panel3Finish.class );
                        }
                    } );
                } catch ( final Exception e ) {
                    // if error occurred
                    SwingUtilities.invokeLater( new Runnable() {
                        public void run() {
                            ProgressGlassPane.restore( frame );

                            showError( e );
                            cancelNavigation();
                        }

                    } );
                }
            }
        }.start();
    }

    private void finish() {
        // TODO implement run
        //if ( panel3Finish.isRunSelected() ) {
        //    try {
        //        JaLingoSetupHelper.run( panel2Destination.getDestination() );
        //
        //        frame.setVisible( false );
        //        starLabel.stop();
        //
        //        return;
        //    } catch ( Exception e ) {
        //        e.printStackTrace();
        //        showError( e );
        //    }
        //}

        System.exit( 0 );
    }

    private static boolean isJvmVersionOk() {
        String javaVersion = System.getProperty( "java.version" );

        if ( javaVersion.compareTo( JaLingoInfo.MIN_JAVA_VERSION ) < 0 ) {

            String OPTION_CONTINUE = "Skip";
            String OPTION_EXIT = "Exit";

            return JOptionPane.showOptionDialog(
                    null,
                    "<html>"
                            + "The required version of JRE is <font color=red>" + JaLingoInfo.MIN_JAVA_VERSION + "</font> or later.<br>"
                            + "The installed version of JRE is <font color=blue>" + javaVersion + "</font>.<br>"
                            + "<br>"

                            + "It is HIGHLY recomended to exit JaLingo Setup now and<br>"
                            + "update JRE to the latest available at http://java.sun.com/</font>."
                    ,
                    "Java Version Check",
                    JOptionPane.YES_NO_OPTION,
                    JOptionPane.WARNING_MESSAGE,
                    null,
                    new String[] { OPTION_CONTINUE, OPTION_EXIT },
                    OPTION_EXIT ) == JOptionPane.YES_OPTION;
        }

        return true;
    }

    public static void main( String[] args ) {
        SwingUtilities.invokeLater( new Runnable() {
            public void run() {
                if ( isJvmVersionOk() ) {
                    try {
                        new JaLingoSetup();
                    } catch ( Exception e ) {
                        System.exit( 1 );
                    }
                }
            }
        } );
    }
}
