#ifndef _ZBUFFER_H
#define _ZBUFFER_H

#include "triangle.h"
#include "scene.h"
#include "fstalloc.h"
#include "stencil.h"

#define ZBUFFER_ACCESS_FILTERED 0
#define ZBUFFER_ACCESS_RAW 1

typedef enum ZBufferDataType {
    ZBUFFER_DATA_COLOR,
    ZBUFFER_DATA_ZDEPTH,
    ZBUFFER_DATA_STENCIL
} ZBufferDataType;

typedef struct FilterType{
    int SuperSize;
    RealType **FilterData;
    char *Name;
    RealType TotalWeight;
} FilterType;

typedef float ZTranspType;

/* Every scan line is represented in z-buffer by array of z-slots, mainly    */
/* formed by the linked list of z-points set up by different Triangles       */
/* projection.                                                               */
typedef struct ZPointStruct {
    struct ZPointStruct *Next; /* Link to the next z-point at same location. */
    IRndrPixelType Color;
    IRndrZDepthType z;
    ZTranspType Transp;                              /* Transparancy factor. */
} ZPointStruct;

typedef struct ZListStruct {
    ZPointStruct First;    /* Head of linked list, contains nearest z-point. */
    int Stencil;
} ZListStruct;

typedef struct ZBufferStruct {
    ZListStruct **z;
    int SizeX;
    int SizeY;
    int TargetSizeX;
    int TargetSizeY;

    FastAllocType PointsAlloc;
    IRndrBoolType ColorsValid;
    int AccessMode;
    struct FilterType *Filter;
    int UseTransparency;
    IRndrColorType BackgroundColor;
    SceneStruct *Scene;
    int ColorQuantization;

    /* Temporary line calculation buffers. */
    IRndrColorType *LineColors;
    ByteType *LineAlpha;
    IrtImgPixelStruct *LinePixels;

    /* Callbacks. */
    IRndrZCmpPolicyType ZPol;
    IRndrPixelClbkType PreZCmpClbk;
    IRndrPixelClbkType ZPassClbk;
    IRndrPixelClbkType ZFailClbk;

    StencilConfigStruct StencilCfg;
} ZBufferStruct;

int  ZBufferInit(ZBufferStruct *Buffer,
                 SceneStruct *Scene,
                 int SuperSize,
		 int ColorQuantization);

void ZBufferPutPixel(ZBufferStruct *Buffer,
                     int x,
                     int y,
                     RealType z,
                     RealType Transparency,
                     IRndrColorType Color);

void ZBufferSetFilter(ZBufferStruct *Buffer,
                      char *FilterName);

void ZBufferSetBackgrZ(ZBufferStruct *Buffer,
                       RealType BackgrZ);

void ZBufferScanTri(ZBufferStruct *Buffer,
                    TriangleStruct *f);

int  ZBufferGetLineDepth(ZBufferStruct *Buffer,
                         int x0, int x1, int y,
                         RealType *Result);

void ZBufferGetLineColor(ZBufferStruct *Buffer,
                         int x0, int x1, int y,
                         IRndrColorType *Result);

int  ZBufferGetLineStencil(ZBufferStruct *Buffer,
                           int x0, int x1, int y,
                           int *Result);

void ZBufferSaveFile(ZBufferStruct *Buffer,
                     char *BaseDirectory,
                     char *OutFileName,
                     char *FileType,
                     ZBufferDataType DataType);

void ZBufferClearColor(ZBufferStruct *Buffer);

void ZBufferClearDepth(ZBufferStruct *Buffer, IRndrZDepthType ClearZ);

void ZBufferClearStencil(ZBufferStruct *Buffer);

void ZBufferRelease(ZBufferStruct *Buffer);

#endif
