/*****************************************************************************
*    Rendering library implementation.                                       *
******************************************************************************
* (C) Gershon Elber, Technion, Israel Institute of Technology                *
******************************************************************************
* Written by:  David Shafrir & Alex Reicher       Ver 0.3, Sep. 2003         *
*****************************************************************************/

#include "rndr_loc.h"
#include "rndr_lib.h"
#include "zbuffer.h"

typedef enum ModeType {
    MODE_NONE,
    MODE_OBJ,
    MODE_PLL,
    MODE_LIGHT
} ModeType;

typedef struct IRndrStruct {
    ZBufferStruct ZBuf;
    SceneStruct Scene;
    ObjectStruct Obj;
    TriangleStruct Tri;
    LineSegmentStruct Seg;
    ModeType Mode;
} IRndrStruct;

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Creates a new Rendering context, and returns a handle to it.             M
*                                                                            *
* PARAMETERS:                                                                M
*   SizeX:              IN, the width of the z-buffer.                       M
*   SizeY:              IN, the hight of the z-buffer.                       M
*   SuperSampSize:      IN, the super-sample size.                           M
*   ColorQuantization:  IN, non zero to quantize the generated colors to     M
*		        ColorQuantization levels of colors.		     M
*   UseTransparency:    IN, whether tarnsparency is on.                      M
*   BackfaceCulling:    IN, whether to use back-face culling.                M
*   BackgrCol:          IN, the background color.                            M
*   AmbientLight:       IN, the abient light factor.                         M
*                                                                            *
* RETURN VALUE:                                                              *
*   IRndrPtrType: a handle to the newly created z-buffer.                    M
*                                                                            *
* KEYWORDS:                                                                  M
*   IRndrInitialize, create, initialize, z-buffer                            M
*****************************************************************************/
IRndrPtrType IRndrInitialize(int SizeX,
                             int SizeY,
                             int SuperSampSize,
			     int ColorQuantization,
                             IRndrBoolType UseTransparency,
                             IRndrBoolType BackfaceCulling,
                             IRndrColorType BackgrCol,
                             RealType AmbientLight)
{
    IRndrPtrType Rend;

    Rend = MALLOC(IRndrStruct, 1);
    Rend -> Scene.SizeX = SizeX * SuperSampSize;
    Rend -> Scene.SizeY = SizeY * SuperSampSize;
    Rend -> Scene.ShadeModel = IRNDR_SHADING_PHONG;
    Rend -> Scene.BackFace = BackfaceCulling;
    Rend -> Scene.Ambient = AmbientLight;
    PT_COPY(&Rend -> Scene.BackgroundColor, BackgrCol);
    SceneSetMatrices(&Rend -> Scene, NULL, NULL, NULL);

    /* All the initalizations are done once and for all. */
    ZBufferInit(&Rend -> ZBuf, &Rend -> Scene, SuperSampSize,
		ColorQuantization);
    Rend -> ZBuf.UseTransparency = UseTransparency;

    LineSegmentInit(&Rend -> Seg, NULL);
    LightListInitEmpty(&Rend -> Scene.Lights);
    TriangleInit(&Rend -> Tri);
    ObjectInit(&Rend -> Obj);
    Rend -> Mode = MODE_NONE;

    IRndrStencilFunc(Rend, IRNDR_ALWAYS, 0, 0);
    IRndrStencilOp(Rend, IRNDR_KEEP, IRNDR_INCR, IRNDR_INCR);

    IRndrClearDepth(Rend, (IRndrZDepthType) FAREST_Z);

    return Rend;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Dispose of a the rendering context.                                      M
*                                                                            *
* PARAMETERS:                                                                M
*   Rend:  IN,OUT, the rendering context.                                    M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IRndrDestroy, destroy, dispose, free, release                            M
*****************************************************************************/
void IRndrDestroy(IRndrPtrType Rend)
{
    ZBufferRelease(&Rend -> ZBuf);
    LineSegmentRelease(&Rend -> Seg);
    SceneRelease(&Rend -> Scene);
    TriangleRelease(&Rend -> Tri);
    ObjectRelease(&Rend -> Obj);

    FREE(Rend);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Clear depth information in the rendering context.                        M
*                                                                            *
* PARAMETERS:                                                                M
*   Rend:    IN,OUT, the rendering context.                                  M
*   ClearZ:  IN, Depth to clear the ZBuffer to.                              M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IRndrClearDepth, clear, reset, depth, Z coordinate                       M
*****************************************************************************/
void IRndrClearDepth(IRndrPtrType Rend, IRndrZDepthType ClearZ)
{
    ZBufferClearDepth(&Rend -> ZBuf, ClearZ);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Clear stencil information in the rendering context.                      M
*                                                                            *
* PARAMETERS:                                                                M
*   Rend:  IN,OUT, the rendering context.                                    M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IRndrClearStencil, clear, reset, stencil                                 M
*****************************************************************************/
void IRndrClearStencil(IRndrPtrType Rend)
{
    ZBufferClearStencil(&Rend -> ZBuf);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Reset color information to the registered background color.              M
*                                                                            *
* PARAMETERS:                                                                M
*   Rend:  IN,OUT, the rendering context.                                    M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IRndrClearColor, clear, reset, background, color                         M
*****************************************************************************/
void IRndrClearColor(IRndrPtrType Rend)
{
    ZBufferClearColor(&Rend -> ZBuf);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Adds a new light source.                                                 M
*                                                                            *
* PARAMETERS:                                                                M
*   Rend:   IN, OUT the rendering context.                                   M
*   Type:   IN, the light type ( POINT, VECTOR )                             M
*   Where:  IN, the light position.                                          M
*   Color:  IN, the light's color.                                           M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IRndrAddLightSource, add light source, list                              M
*****************************************************************************/
void IRndrAddLightSource(IRndrPtrType Rend,
                         IRndrLightType Type,
                         PointType Where,
                         IRndrColorType Color)
{
    LightStruct
        *Light = MALLOC(LightStruct, 1);
    Light -> Type = Type;
    PT_COPY(Light -> Where, Where);
    PT_COPY(Light -> Color, Color);
    LightListAdd(&Rend -> Scene.Lights, Light);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Changes the filter, used for antialiasing.                               M
*                                                                            *
* PARAMETERS:                                                                M
*   Rend:       IN,OUT, the rendering context.                               M
*   FilterName: IN, the filter's name.                                       M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IRndrSetFilter, Filter, anti-aliasing                                    M
*****************************************************************************/
void IRndrSetFilter(IRndrPtrType Rend, char *FilterName)
{
    ZBufferSetFilter(&Rend -> ZBuf, FilterName);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Changes the shading model.                                               M
*                                                                            *
* PARAMETERS:                                                                M
*   Rend:        IN, OUT, the rendering context.                             M
*   ShadeModel:  IN, the new shading model (FLAT,GOURAUD,PHONG,NONE).        M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IRndrSetShadeModel, shading, light model                                 M
*****************************************************************************/
void IRndrSetShadeModel(IRndrPtrType Rend, IRndrShadingType ShadeModel)
{
    if (ShadeModel < IRNDR_SHADING_LAST) {
        Rend -> Scene.ShadeModel = ShadeModel;
    }
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Sets the view and perspective matrices.                                  M
*                                                                            *
* PARAMETERS:                                                                M
*   Rend:     IN,OUT, the rendering context.                                 M
*   ViewMat:  IN, the view matrix.                                           M
*   PrspMat:  IN, the perspective matrix.                                    M
*   ScrnMat:  IN, the mapping to the screen or NULL if scale [-1,+1] to      M
*	      image size.						     M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IRndrSetViewPrsp, view, perspective, matrix context                      M
*****************************************************************************/
void IRndrSetViewPrsp(IRndrPtrType Rend,
		      MatrixType ViewMat,
		      MatrixType PrspMat,
		      MatrixType ScrnMat)
{
    SceneSetMatrices(&Rend -> Scene, ViewMat, PrspMat, ScrnMat);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Retrives the 6 clipping planes, defining the viewing frastrum.           M
*                                                                            *
* PARAMETERS:                                                                M
*   Rend:        IN, the rendering context.                                  M
*   ClipPlanes:  OUT, a pointer to the 6 user allocated planes.              M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IRndrGetClippingPlanes, clipping, viewing frastrum                       M
*****************************************************************************/
void IRndrGetClippingPlanes(IRndrPtrType Rend, PlaneType *ClipPlanes)
{
    int i, j;

    for (i = 0; i < 3; i++)
        for (j = 0; j < 2; j++)
            SceneGetClippingPlane(&Rend -> Scene, i, j, ClipPlanes[2 * i + j]);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Sets the near and far XY clipping planes, defining the viewing frustum.  M
*                                                                            *
* PARAMETERS:                                                                M
*   Rend:  IN, OUT, the rendering context.                                   M
*   ZNear: IN, the (negation of the) z-coordinate of the near clipping plane.M
*   ZFar:  IN, the (negation of the) z-coordinate of the far clipping plane. M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IRndrSetZBounds, clipping, viewing frastrum                              M
*****************************************************************************/
void IRndrSetZBounds(IRndrPtrType Rend, RealType ZNear, RealType ZFar)
{
    SceneSetZClippingPlanes(&Rend -> Scene, ZNear, ZFar);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Sets the Polyline parameters, used for line drawing.                     M
*                                                                            *
* PARAMETERS:                                                                M
*   Rend:         IN,OUT, the rendering context.                             M
*   MinWidth:     IN, the width of the line at Z = Znear.                    M
*   MaxWidth:     IN, the width of the line at Z = Zfar.                     M
*   ZNear, ZFar:  IN, as stated above.usually the expected scene width.      M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IRndrSetPllParams, polyline line depth cue                               M
*****************************************************************************/
void IRndrSetPllParams(IRndrPtrType Rend,
                        RealType MinWidth,
                        RealType MaxWidth,
                        RealType ZNear,
                        RealType ZFar)
{
    RealType
        ScaleFactor = Rend -> Scene.SizeX * 0.5;
    PolylineOptionsStruct PllOptions;

    PllOptions.MinWidth = MinWidth * ScaleFactor;
    PllOptions.MaxWidth = MaxWidth * ScaleFactor;
    PllOptions.ZNear = ZNear;
    PllOptions.ZFar = ZFar;

    LineSegmentSetOptions(&Rend -> Seg, &PllOptions);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Sets the z-buffer access mode (original super smapled data or filtered). M
*                                                                            *
* PARAMETERS:                                                                M
*   Rend:        IN, OUT, the rendering context.                             M
*   UseRawMode:  IN, whether the access mode is RAW (otherwise filtered).    M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IRndrSetRawMode, antialising access mode raw                             M
*****************************************************************************/
void IRndrSetRawMode(IRndrPtrType Rend, IRndrBoolType UseRawMode)
{
    Rend -> ZBuf.AccessMode = UseRawMode ?
        ZBUFFER_ACCESS_RAW : ZBUFFER_ACCESS_FILTERED;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Sets the z-buffer comparison function.                                   M
*                                                                            *
* PARAMETERS:                                                                M
*   Rend:    IN,OUT, the rendering context.                                  M
*   ZCmpPol: IN, the comparison function (linear order).                     M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IRndrSetZCmpPolicy, comparison, z-buffer, sort                           M
*****************************************************************************/
void IRndrSetZCmpPolicy(IRndrPtrType Rend, IRndrZCmpPolicyType ZCmpPol)
{
    Rend -> ZBuf.ZPol = ZCmpPol;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Sets the z-buffer pre comparison function callback function.             M
*                                                                            *
* PARAMETERS:                                                                M
*   Rend:      IN, OUT, the rendering context.                               M
*   PixelClbk: IN, the callback function.                                    M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IRndrSetPreZCmpClbk, comparison pre  callback z-buffer                   M
*****************************************************************************/
void IRndrSetPreZCmpClbk(IRndrPtrType Rend, IRndrPixelClbkType PixelClbk)
{
    Rend -> ZBuf.PreZCmpClbk = PixelClbk;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Sets the z-buffer post comparison function callback function.            M
*                                                                            *
* PARAMETERS:                                                                M
*   Rend:                   IN,OUT, the rendering context.                   M
*   ZPassClbk, ZFailClbk:   IN, the callback functions called on             M
*                           success/failure.                                 M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IRndrSetPostZCmpClbk, comparison, z-buffer, post                         M
*****************************************************************************/
void IRndrSetPostZCmpClbk(IRndrPtrType Rend,
                            IRndrPixelClbkType ZPassClbk,
                            IRndrPixelClbkType ZFailClbk)
{
    Rend -> ZBuf.ZPassClbk = ZPassClbk;
    Rend -> ZBuf.ZFailClbk = ZFailClbk;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Sets the z-buffer stencil test function.                                 M
*                                                                            *
* PARAMETERS:                                                                M
*   Rend:  IN, OUT, the rendering context.                                   M
*   Func:  IN, stencil test function type.                                   M
*   Ref:   IN, stencil test refernce value.                                  M
*   Mask:  IN, stencil test mask.                                            M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IRndrStencilFunc, stencil buffer, OpenGL, glStencilFunc                  M
*****************************************************************************/
void IRndrStencilFunc(IRndrPtrType Rend,
                      IRndrFuncType Func,
                      int Ref,
                      unsigned Mask)
{
    StencilConfigStruct
        *SCfg = &Rend -> ZBuf.StencilCfg;

    SCfg -> Func = Func;
    SCfg -> Ref = Ref;
    SCfg -> Mask = Mask;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Sets the z-buffer stencil operations.                                    M
*                                                                            *
* PARAMETERS:                                                                M
*   Rend:   IN, OUT, the rendering context.                                  M
*   Fail:   IN, stencil test failure operation.                              M
*   ZFail:  IN, Z-test failure operation.                                    M
*   ZPass:  IN, Z-test pass operation.                                       M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IRndrStencilOp, stencil buffer, OpenGL, glStencilOp                      M
*****************************************************************************/
void IRndrStencilOp(IRndrPtrType Rend,
                    IRndrOpType Fail,
                    IRndrOpType ZFail,
                    IRndrOpType ZPass)
{
    StencilConfigStruct
        *SCfg = &Rend -> ZBuf.StencilCfg;

    SCfg -> OpFail = Fail;
    SCfg -> OpZFail = ZFail;
    SCfg -> OpZPass = ZPass;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Sets the Irit object to be scan converted.                               M
*                                                                            *
* PARAMETERS:                                                                M
*   Rend:       IN, OUT, the rendering context.                              M
*   Object:     IN, the object to be scanned.                                M
*   NoShading:  IN, if TRUE, ignore shading on this one.                     M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IRndrBeginObject, scan irit object z-buffer                              M
*****************************************************************************/
void IRndrBeginObject(IRndrPtrType Rend,
		      IPObjectStruct *Object,
		      int NoShading)
{
    Rend -> Obj.noShade = NoShading;

    if (IP_IS_POLYLINE_OBJ(Object)) {
        AttrSetObjectIntAttrib(Object, "_TRANSFORMED", TRUE);
        Rend -> Obj.Transformed = TRUE;
        Rend -> Mode = MODE_PLL;
    }
    else {
        Rend -> Mode = MODE_OBJ;
    }
    ObjectSet(&Rend -> Obj, Object, &Rend -> Scene);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Scan converts a triangle polygon.                                        M
*                                                                            *
* PARAMETERS:                                                                M
*   Rend:      IN, OUT, the rendering context.                               M
*   Triangle:  IN, the triangle to be scanned.                               M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IRndrPutTriangle,scan triangle  polygon z-buffer                         M
*****************************************************************************/
void IRndrPutTriangle(IRndrPtrType Rend, IPPolygonStruct *Triangle)
{
    if (IPVrtxListLen(Triangle -> PVertex) != 3)
	return;

    if (Rend -> Mode != MODE_OBJ) {
        _IRndrReportError(IRIT_EXP_STR("IRndrPutTriangle() not during object scan"));
    }
    if (TriangleSet(&Rend -> Tri, Triangle, &Rend -> Obj, &Rend -> Scene)) {
        ZBufferScanTri(&Rend -> ZBuf, &Rend -> Tri);
    }
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Marks the end of  the object scaning.                                    M
*                                                                            *
* PARAMETERS:                                                                M
*   Rend: IN, OUT, the rendering context.                                    M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IRndrEndObject, scan, irit object, z-buffer                              M
*****************************************************************************/
void IRndrEndObject(IRndrPtrType Rend)
{
    Rend -> Mode = MODE_NONE;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Begin drawing a line.                                                    M
*                                                                            *
* PARAMETERS:                                                                M
*   Rend:  IN, OUT, the rendering context.                                   M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IRndrBeginPll, new line, start                                           M
*****************************************************************************/
void IRndrBeginPll(IRndrPtrType Rend)
{
    LineSegmentStart(&Rend -> Seg);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Sets the next vertex of the line.                                        M
*                                                                            *
* PARAMETERS:                                                                M
*   Rend:    IN, OUT, the rendering context.                                 M
*   Vertex:  IN, the next vertex of the line.                                M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IRndrPutPllVertex, line vertex lineto                                    M
*****************************************************************************/
void IRndrPutPllVertex(IRndrPtrType Rend, IPVertexStruct *Vertex)
{
    int i;
    RealType Coord[4];

    VertexTransform(Vertex, &Rend -> Scene.Matrices, &Rend -> Obj, Coord);
    LineSegmentSet(&Rend -> Seg, Coord);
    for (i = 0; i < Rend -> Seg.TrianglesNum; i++) {
        if (TriangleSet(&Rend -> Tri, LineSegmentGetTri(&Rend -> Seg, i),
                        &Rend -> Obj, &Rend -> Scene)) {
            ZBufferScanTri(&Rend -> ZBuf, &Rend -> Tri);
        }
    }
}
/*****************************************************************************
* DESCRIPTION:                                                               M
*   Marks the end of the line .                                              M
*                                                                            *
* PARAMETERS:                                                                M
*   Rend:  IN, OUT, the rendering context.                                   M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IRndrEndPll, line end                                                    M
*****************************************************************************/
void IRndrEndPll(IRndrPtrType Rend)
{
    int i;

    Rend -> Mode = MODE_NONE;
    LineSegmentEnd(&Rend -> Seg);
    for (i = 0; i < Rend -> Seg.TrianglesNum; i++) {
        if (TriangleSet(&Rend -> Tri, LineSegmentGetTri(&Rend -> Seg, i),
                        &Rend -> Obj, &Rend -> Scene)) {
            ZBufferScanTri(&Rend -> ZBuf, &Rend -> Tri);
        }
    }
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Manually adds a single pixel.                                            M
*                                                                            *
* PARAMETERS:                                                                M
*   Rend:  IN, OUT, the rendering context.                                   M
*   x:     IN, the column number.                                            M
*   y:     IN, the line number.                                              M
*   z:     IN, the pixel's depth.                                            M
*   Transparency: IN, the pixel's transparency value.                        M
*   Color: IN, the new color of pixel at (x, y).                             M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IRndrPutPixel, put pixel                                                 M
*****************************************************************************/
void IRndrPutPixel(IRndrPtrType Rend,
                   int x,
                   int y,
                   RealType z,
                   RealType Transparency,
                   IRndrColorType Color)
{
    ZBufferPutPixel(&Rend -> ZBuf, x, y, z, Transparency, Color);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Retrieve a pixel's color from the z-buffer.                              M
*                                                                            *
* PARAMETERS:                                                                M
*   Rend:  IN, OUT, the rendering context.                                   M
*   x:     IN, the column number.                                            M
*   y:     IN, the line number.                                              M
*   Result:  OUT, the user allocated buffer to hold the result.              M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IRndrGetPixelColor, z-buffer, line color information                     M
*****************************************************************************/
void IRndrGetPixelColor(IRndrPtrType Rend,
                        int x,
                        int y,
                        IRndrColorType *Result)
{
    ZBufferGetLineColor(&Rend -> ZBuf, x, x + 1, y, Result);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Retrieve a pixel's depth from the z-buffer.                              M
*                                                                            *
* PARAMETERS:                                                                M
*   Rend: IN,OUT, the rendering context.                                     M
*   x:    IN, the column number.                                             M
*   y:    IN, the line number.                                               M
*   Result: OUT, the user allocated buffer to hold the result.               M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IRndrGetPixelDepth, z-buffer, line depth                                 M
*****************************************************************************/
void IRndrGetPixelDepth(IRndrPtrType Rend,
                        int x,
                        int y,
                        RealType *Result)
{
    ZBufferGetLineDepth(&Rend -> ZBuf, x, x + 1, y, Result);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Retrieve a pixel's stencil from the z-buffer.                            M
*                                                                            *
* PARAMETERS:                                                                M
*   Rend:  IN, OUT, the rendering context.                                   M
*   x:     IN, the column number.                                            M
*   y:     IN, the line number.                                              M
*   Result:  OUT, the user allocated buffer to hold the result.              M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IRndrGetPixelStencil, z-buffer line stencil                              M
*****************************************************************************/
void IRndrGetPixelStencil(IRndrPtrType Rend,
                          int x,
                          int y,
                          int *Result)
{
    ZBufferGetLineStencil(&Rend -> ZBuf, x, x + 1, y, Result);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Retrieve color data from the z-buffer.                                   M
*                                                                            *
* PARAMETERS:                                                                M
*   Rend:   IN, OUT, the rendering context.                                  M
*   y:      IN, the line number.                                             M
*   Result: OUT, the user allocated buffer to hold the result.               M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IRndrGetLineColor, z-buffer, line color information                      M
*****************************************************************************/
void IRndrGetLineColor(IRndrPtrType Rend, int y, IRndrColorType *Result)
{
    ZBufferGetLineColor(&Rend -> ZBuf, 0,
			Rend -> ZBuf.AccessMode == ZBUFFER_ACCESS_RAW ?
			    Rend -> ZBuf.SizeX :
			    Rend -> ZBuf.TargetSizeX, y, Result);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Retrieve z-coordinate data from the z-buffer.                            M
*                                                                            *
* PARAMETERS:                                                                M
*   Rend:   IN, OUT, the rendering context.                                  M
*   y:      IN, the line number.                                             M
*   Result: OUT, the user allocated buffer to hold the result.               M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IRndrGetLineDepth, z-buffer, line Z information                          M
*****************************************************************************/
void IRndrGetLineDepth(IRndrPtrType Rend, int y, RealType *Result)
{
    ZBufferGetLineDepth(&Rend -> ZBuf, 0,
			Rend -> ZBuf.AccessMode == ZBUFFER_ACCESS_RAW ?
			    Rend -> ZBuf.SizeX :
			    Rend -> ZBuf.TargetSizeX, y, Result);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Retrieve stencil data from the z-buffer.                                 M
*                                                                            *
* PARAMETERS:                                                                M
*   Rend:   IN, OUT, the rendering context.                                  M
*   y:      IN, the line number.                                             M
*   Result: OUT, the user allocated buffer to hold the result.               M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IRndrGetLineStencil, z-buffer, line stencil information                  M
*****************************************************************************/
void IRndrGetLineStencil(IRndrPtrType Rend, int y, int *Result)
{
    ZBufferGetLineStencil(&Rend -> ZBuf, 0,
			  Rend -> ZBuf.AccessMode == ZBUFFER_ACCESS_RAW ?
			      Rend -> ZBuf.SizeX :
			      Rend -> ZBuf.TargetSizeX, y, Result);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Save  the color info of the z-buffer into a file .                       M
*                                                                            *
* PARAMETERS:                                                                M
*   Rend:           IN, OUT, the rendering context.                          M
*   BaseDirectory:  IN, the directory to save the file in.                   M
*   OutFileName:    IN, the file name.                                       M
*   Type:           IN, the file type.                                       M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IRndrSaveFile, z-buffer save dump                                        M
*****************************************************************************/
void IRndrSaveFile(IRndrPtrType Rend,
		   char *BaseDirectory,
		   char *OutFileName,
		   char *Type)
{
    ZBufferSaveFile(&Rend -> ZBuf, BaseDirectory,
		    OutFileName, Type, ZBUFFER_DATA_COLOR);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Save the z coordinate values of the z-buffer into a file.                M
*                                                                            *
* PARAMETERS:                                                                M
*   Rend:           IN, OUT, the rendering context.                          M
*   BaseDirectory:  IN, the directory to save the file in.                   M
*   OutFileName:    IN, the file name.                                       M
*   Type:           IN, the file type.                                       M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IRndrSaveFileDepth, z-buffer save dump                                   M
*****************************************************************************/
void IRndrSaveFileDepth(IRndrPtrType Rend,
			char *BaseDirectory,
			char *OutFileName,
			char *Type)
{
    ZBufferSaveFile(&Rend -> ZBuf, BaseDirectory,
		    OutFileName, Type, ZBUFFER_DATA_ZDEPTH);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Save  the context of the z-buffer into a file .                          M
*                                                                            *
* PARAMETERS:                                                                M
*   Rend:          IN, OUT, the rendering context.                           M
*   BaseDirectory: IN, the directory to save the file in.                    M
*   OutFileName:   IN, the file name.                                        M
*   Type:          IN, the file type.                                        M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IRndrSaveFileStencil, z-buffer save dump                                 M
*****************************************************************************/
void IRndrSaveFileStencil(IRndrPtrType Rend,
			  char *BaseDirectory,
			  char *OutFileName,
			  char *Type)
{
    ZBufferSaveFile(&Rend -> ZBuf, BaseDirectory,
		    OutFileName, Type, ZBUFFER_DATA_STENCIL);
}
