/*****************************************************************************
* Color of the scan-line pixel evaluation algorithms.                        *
******************************************************************************
* (C) Gershon Elber, Technion, Israel Institute of Technology                *
******************************************************************************
* Written by:  Bassarab Dmitri & Plavnik Michael       Ver 0.2, Apr. 1995    *
* Modified by:  David Shafrir & Alex Reicher            Aug. 2003            *
*****************************************************************************/

#include "rndr_loc.h"
#include "color.h"

GLOBAL_DATA IRndrColorType Colors[] =
{
    { 0,    0,    0 },                                  /*  0. Black.        */
    { 0,    0,    1 },                                  /*  1. Blue.         */
    { 0,    1,    0 },                                  /*  2. Green.        */
    { 0,    1,    1 },                                  /*  3. Cyan.         */
    { 1,    0,    0 },                                  /*  4. Red.          */
    { 1,    0,    1 },                                  /*  5. Magenta.      */
    { 0.5,  0.5,  0 },                                  /*  6. Brown.        */
    { 0.5,  0.5,  0.5 },                                /*  7. Lightgrey.    */
    { 0.25, 0.25, 0.25 },                               /*  8. Darkgray.     */
    { 0.25, 0.25, 1 },                                  /*  9. Lightblue.    */
    { 0.25, 1,    0.25 },                               /* 10. Lightgreen.   */
    { 0.25, 1,    1 },                                  /* 11. Lightcyan.    */
    { 1,    0.25, 0.25 },                               /* 12. Lightred.     */
    { 1,    0.25, 1 },                                  /* 13. Lightmagenta. */
    { 1,    1,    0.25 },                               /* 14. Yellow.       */
    { 1,    1,    1 }                                   /* 15. White.        */
};

static void ColorAdd(IRndrColorType c,
                     const IRndrColorType l,
                     const IRndrColorType o,
                     const IntensivityStruct *i);

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Computes intensivity diffuse and specular values using specular model of M
*   illumination (see Foily, Van Dam). Differs between point and vector      M
*   viewer, point and vector light sources, which is defined in Light object M
*   and by calling IS_VIEWER_POINT() function.                               M
*                                                                            *
* PARAMETERS:                                                                M
*   l:       IN, pointer to the light source object.                         M
*   p:       IN, point for which intensivity is computing.                   M
*   n:       IN, normal to the surface in the point "p".                     M
*   o:       IN, pointer to the object with surface characteristics.         M
*   Scene:   IN, pointer to the scene the object belongs.                    M
*   i:       OUT, pointer to resulting intensivity object.                   M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   LightIntensivity, specular, shading                                      M
*****************************************************************************/
void LightIntensivity(LightStruct *l,
                      const PointType p,
                      const NormalType n,
                      const ObjectStruct *o,
                      SceneStruct *Scene,
		      IntensivityStruct *i)
{
    RealType CosTheta, CosAlpha, *Light, *Sight;  /* For efficiency reasons. */
    STATIC_DATA NormalType Normal;

    if (l -> Type == POINT_LIGHT) {
        STATIC_DATA PointType LightStorage;

        Light = &LightStorage[0];                  /* We elliminate copying. */
        PT_SUB(Light, l -> Where, p);
        PT_NORMALIZE(Light);
    }
    else
        Light = l -> Where;

    if (!Scene -> Matrices.ParallelProjection ) {
        STATIC_DATA PointType SightStorage;

        Sight = &SightStorage[0];
        PT_SUB(Sight, Scene -> Matrices.Viewer, p);

        PT_NORMALIZE(Sight);

    }
    else
        Sight = Scene -> Matrices.Viewer;

    i -> Diff = i -> Spec = 0.0;

    /* Minus corrects assumed direction - "to the center of sphere". */
    PT_COPY(Normal, n);

    PT_SCALE(Normal, -1.0);

    CosTheta = DOT_PROD(Light, Normal);

    if (CosTheta > IRIT_EPS) {      /* Light passes from behind of the poly. */
        STATIC_DATA PointType Mirrored;

        i -> Diff = o -> KDiffuse * CosTheta;
        PT_SCALE(Normal, 2 * CosTheta);
        PT_SUB(Mirrored, Normal, Light);
        CosAlpha = DOT_PROD(Sight, Mirrored);
        if (CosAlpha > IRIT_EPS){
            i -> Spec = o -> KSpecular * pow(CosAlpha, (RealType) o -> Power);
        }
    }
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Aux. function which adds to resulting color light source color scaled by *
*   diffuse-specualr composition.                                            *
*                                                                            *
* PARAMETERS:                                                                *
*   c:      IN, OUT, resulting color.                                        *
*   l:      IN, light source color.                                          *
*   o:      IN, object color.                                                *
*   i:      IN, pointer to an intensivity object.                            *
*                                                                            *
* RETURN VALUE:                                                              *
*   void                                                                     *
*****************************************************************************/
static void ColorAdd(IRndrColorType c,
                     const IRndrColorType l,
                     const IRndrColorType o,
                     const IntensivityStruct *i)
{
    c[RED_CLR]   += l[RED_CLR]   * (o[RED_CLR]   * i -> Diff + i -> Spec);
    c[GREEN_CLR] += l[GREEN_CLR] * (o[GREEN_CLR] * i -> Diff + i -> Spec);
    c[BLUE_CLR]  += l[BLUE_CLR]  * (o[BLUE_CLR]  * i -> Diff + i -> Spec);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   For scan liny "y" and pixel "x" in it computes color value in [0, 1]     M
*   RGB format.                                                              M
*                                                                            *
* PARAMETERS:                                                                M
*   Poly:    IN, the polygon the pixel belongs.                              M
*   x:       IN, scan line pixel position.                                   M
*   y:       IN, scan line number.                                           M
*   o:       IN, the object of the triangle.                                 M
*   Scene:   IN, the scene context.                                          M
*   Value:   IN, OUT, interpolation value.                                   M
*   r:       OUT, resulting color.                                           M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   TriangleColorEval, RGB, intensivity, scan line, shading                  M
*****************************************************************************/
void TriangleColorEval(IPPolygonStruct *Poly,
		       int x,
		       int y,
		       ObjectStruct *o,
		       SceneStruct *Scene,
		       InterpolStruct *Value,
		       IRndrColorType r)
{
    IRndrColorType Orig;
    PointType p;
    int j,
        ShadeModel = Scene -> ShadeModel;
    RealType w = 1.0 / Value -> w;

    if (o -> noShade) {        /* Take care of special objects (pure color). */
        PT_COPY(r, o -> Color);
        return;
    }

    /* Initialize current image point and transform to object space. */
    p[X_AXIS] = x;
    p[Y_AXIS] = y;
    p[Z_AXIS] = Value -> z;
    MatMultPtby4by4(p, p, Scene -> Matrices.ViewInvMat);

    /* Transform normal to object space, using homogeneous coordinates     */
    /* interpolation.					                   */
    PT_SCALE(Value -> n, w);

    /* Fill resulting color with (texture) color, or object color. */
    if (Value -> HasColor)
	PT_COPY(r, Value -> c);
    else
        PT_COPY(r, o -> Color);

    /* If no shading is request, abort now. */
    if (ShadeModel == SHADING_NONE)
        return;

    switch (o -> Txtr.Type) {
        case TEXTURE_TYPE_RSTR:
            {
		IrtImgPixelStruct
		    *Pxl = TextureImageGetPixel(&o -> Txtr,
						o -> Txtr.PrmImage,
						p,
						Value -> v * w,
						Value -> u * w,
						Poly);

		r[RED_CLR]   = Pxl -> r;
		r[GREEN_CLR] = Pxl -> g;
		r[BLUE_CLR]  = Pxl -> b;
		PT_SCALE(r, 1. / 0xff);
	    }
	    break;
	case TEXTURE_TYPE_PROC:
	    {
		CagdRType Uv[2];

		Uv[0] = Value -> u * w;
		Uv[1] = Value -> v * w;

		PT_NORMALIZE(Value -> n);
		p[0] *= o -> Txtr.tScale[0];
		p[1] *= o -> Txtr.tScale[1];
		p[2] *= o -> Txtr.tScale[2];
		PT_COPY(Orig, r);
		(*o -> Txtr.vTexture)(&o -> Txtr, p, Value -> n, Uv, r);
		r[0] *= Orig[0];
		r[1] *= Orig[1];
		r[2] *= Orig[2];
	    }
	    break;
	case TEXTURE_TYPE_SRF:
	    {
		CagdRType t,
		    u = o -> Txtr.SrfParamDomain[0][2] *
			    (Value -> u *  w) +
				o -> Txtr.SrfParamDomain[0][0],
		    v = o -> Txtr.SrfParamDomain[1][2] *
			    (Value -> v * w) +
				o -> Txtr.SrfParamDomain[1][0],
	            *Pt = CagdSrfEval(o -> Txtr.Srf, u, v);

		switch (o -> Txtr.Srf -> PType) {
		    case CAGD_PT_E1_TYPE:
		    case CAGD_PT_P1_TYPE:
		        t = Pt[1];
			break;
		    case CAGD_PT_E2_TYPE:
		    case CAGD_PT_P2_TYPE:
			t = sqrt(SQR(Pt[1]) + SQR(Pt[2]));
			break;
		    case CAGD_PT_E3_TYPE:
		    case CAGD_PT_P3_TYPE:
			t = sqrt(SQR(Pt[1]) + SQR(Pt[2]) + SQR(Pt[3]));
			break;
		}
		t = CAGD_IS_RATIONAL_SRF(o -> Txtr.Srf) ? t / Pt[0] : t;

		switch (o -> Txtr.SrfFunc) {
		    case STEXTURE_FUNC_SQRT:
		        t = sqrt(FABS(t));
			break;
		    case STEXTURE_FUNC_ABS:
			t = FABS(t);
			break;
		    }

		t = (t - o -> Txtr.SrfScaleMinMax[0]) /
		    (o -> Txtr.SrfScaleMinMax[1] -
		     o -> Txtr.SrfScaleMinMax[0]);
		t = BOUND(t, 0.0, 1.0);

		if (o -> Txtr.SrfScale != NULL) {
		    IrtImgPixelStruct
			*Pxl = TextureImageGetPixel(&o -> Txtr,
						    o -> Txtr.SrfScale,
						    p, t, 0, Poly);

		    r[RED_CLR]   = Pxl -> r;
		    r[GREEN_CLR] = Pxl -> g;
		    r[BLUE_CLR]  = Pxl -> b;
		    PT_SCALE(r, 1. / 0xff);
		}
		else {
		    RealType
			t1 = 1.0 - t;
		    
		    /* Use our own scale: */
		    r[RED_CLR]   = t * t;
		    r[GREEN_CLR] = 2 * t * t1;
		    r[BLUE_CLR]  = t1 * t1;
		}
	    }
	    break;
    }

    PT_COPY(Orig, r);
    PT_SCALE(r, Scene -> Ambient);  /* Compose ambient color in resulting.  */
    if (ShadeModel == SHADING_PHONG)         /* Ensure nrml is a real nrml. */
        PT_NORMALIZE(Value -> n);

    /* Add to the resulting color diffuse and specular components from all  */
    /* light sources, check shadows by the way.                             */
    for (j = 0; j < Scene -> Lights.n ; j++) {
        if (ShadeModel == SHADING_PHONG) {
            IntensivityStruct i;

            LightIntensivity(&Scene -> Lights.Src[j],
			     p, Value -> n, o, Scene, &i);
            ColorAdd(r, Scene -> Lights.Src[j].Color, Orig, &i);
        }
	else {
            if (ShadeModel == SHADING_GOURAUD) {
		Value -> i[j].Diff *= w;
		Value -> i[j].Spec *= w;
            }
            ColorAdd(r, Scene -> Lights.Src[j].Color, Orig, &Value -> i[j]);
        }
    }

    MINM(r[RED_CLR], 1);         /* Ensure that color is in [0, 1] interval. */
    MINM(r[GREEN_CLR], 1);
    MINM(r[BLUE_CLR], 1);
}
