/******************************************************************************
* MvarCoer.c - Handle point coercions/conversions.			      *
*******************************************************************************
* (C) Gershon Elber, Technion, Israel Institute of Technology                 *
*******************************************************************************
* Written by Gershon Elber, May. 97.					      *
******************************************************************************/

#include <assert.h>
#include "mvar_loc.h"

/*****************************************************************************
* DESCRIPTION:                                                               M
* Coerces a multi-variate to point type PType.				     M
*                                                                            *
* PARAMETERS:                                                                M
*   MV:       To coerce to a new point type PType.                           M
*   PType:    New point type for MV.                                         M
*                                                                            *
* RETURN VALUE:                                                              M
*   MvarMVStruct *:   A new multi-variate with PType as its point type.      M
*                                                                            *
* KEYWORDS:                                                                  M
*   MvarCoerceMVTo, coercion                                                 M
*****************************************************************************/
MvarMVStruct *MvarCoerceMVTo(MvarMVStruct *MV, CagdPointType PType)
{
#ifdef MVAR_MALLOC_STRUCT_ONCE
    CagdBType
	IsRational = CAGD_IS_RATIONAL_PT(PType);
    int i, *q,
	Dim = MV -> Dim,
	MaxAxis = CAGD_NUM_OF_PT_COORD(PType),
	Len = MVAR_CTL_MESH_LENGTH(MV),
	ExtraMem = sizeof(int) * Dim + 8 +          /* Lengths. */
		   sizeof(int) * (Dim + 1) + 8 +  /* SubSpaces. */
		   sizeof(int) * Dim + 8 +           /* Orders. */
		   sizeof(CagdBType) * Dim + 8 +  /* Periodics. */
		   sizeof(CagdRType **) * Dim + 8;      /* KVs. */
    MvarMVStruct *TMV;

    TMV = CagdStructOnceCoercePointsTo(MV -> Points, MV,
				       sizeof(MvarMVStruct) + 8,
				       ExtraMem, Len,
				       MV -> PType, PType);

    /* Find address beyond the struct and beyond the control points. */
    q = (int *) &TMV -> Points[MaxAxis][Len];

    TMV -> Lengths = q;
    q += Dim;

    TMV -> SubSpaces = q;
    q += Dim + 1;

    TMV -> Orders = q;
    q += Dim;

    TMV -> Periodic = q; /* Assume CagdBType same as int. */
    q += Dim;

    /* Align it to 8 bytes. */
    q = (int *) ((((IritIntPtrSizeType) q) + 7) & ~0x07);

    TMV -> KnotVectors = (CagdRType **) q;

    TMV -> GType = MV -> GType;
    TMV -> Dim = Dim;
    TMV -> Pnext = NULL;
    TMV -> PAux = TMV -> PAux2 = NULL;

    TMV -> Attr = IP_ATTR_COPY_ATTRS(MV -> Attr);

    CAGD_GEN_COPY(TMV -> Lengths, MV -> Lengths, Dim * sizeof(int));
    CAGD_GEN_COPY(TMV -> SubSpaces, MV -> SubSpaces, (Dim + 1) * sizeof(int));
    CAGD_GEN_COPY(TMV -> Orders, MV -> Orders, Dim * sizeof(int));
    CAGD_GEN_COPY(TMV -> Periodic, MV -> Periodic, Dim * sizeof(int));

    if (TMV -> GType == MVAR_BSPLINE_TYPE) {
        for (i = 0; i < Dim; i++) {
	    assert(MV -> KnotVectors[i] != NULL);
	    TMV -> KnotVectors[i] =
	        BspKnotCopy(NULL, MV -> KnotVectors[i],
			    MVAR_MVAR_ITH_PT_LST_LEN(MV, i) + MV -> Orders[i]);
	}
    }
    else {
        ZAP_MEM(TMV -> KnotVectors, Dim * sizeof(CagdRType *));
    }

    TMV -> Pnext = NULL;
    TMV -> Attr = NULL;
    CAGD_PROPAGATE_ATTR(TMV, MV);
    MV = TMV;
#else
    MV = MvarMVCopy(MV);
    CagdCoercePointsTo(MV -> Points, MVAR_CTL_MESH_LENGTH(MV),
		       MV -> PType, PType);
#endif /* MVAR_MALLOC_STRUCT_ONCE */

    MV -> PType = PType;

    return MV;
}
