/*****************************************************************************
*   "Irit" - the 3d (not only polygonal) solid modeller.		     *
*									     *
* Written by:  Gershon Elber				Ver 0.2, Mar. 1990   *
******************************************************************************
* (C) Gershon Elber, Technion, Israel Institute of Technology                *
******************************************************************************
*   Module to evaluate the binary tree generated by the InptPrsr module.     *
*   All the objects are handled the same but the numerical one, which is     *
* moved as a RealType and not as an object (only internally within this	     *
* module) as it is frequently used and consumes much less memory this way.   *
*****************************************************************************/

#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include <string.h>
#include "program.h"
#include "allocate.h"
#include "attribut.h"
#include "obj_dpnd.h"
#include "ctrl-brk.h"
#include "dosintr.h"
#include "freeform.h"
#include "geom_lib.h"
#include "inptprsg.h"
#include "inptprsl.h"
#include "objects.h"
#include "overload.h"
#include "iritgrap.h"

/* #define IRIT_PRSR_DEBUG_DPNDNCY */
#ifdef DEBUG
IRIT_SET_DEBUG_PARAMETER(_DebugIritPrsrDepndncy, FALSE);
IRIT_SET_DEBUG_PARAMETER(_DebugIritPrsrDispNesting, FALSE);
#endif /* DEBUG */

GLOBAL_DATA InptPrsrEvalErrType
    IPGlblEvalError = IPE_NO_ERR;		 /* Global used by EvalTree. */

static void LocalPrintTree(ParseTree *Root, int Level, char *Str, int StrLen);
static void InptPrsrPropagateDependencyObjects(IPObjectStruct *PObj,
					       ParseTree *Root);
static void InptPrsrPropagateUpdatedObjects(IPObjectStruct *PObj,
					    int UpdateIndex);

/*****************************************************************************
* DESCRIPTION:                                                               M
* Returns a string representing the given type.				     M
*                                                                            *
* PARAMETERS:                                                                M
*   Type:     Type to get a string representation for.                       M
*                                                                            *
* RETURN VALUE:                                                              M
*   char *:   A string describing type Type.                                 M
*                                                                            *
* KEYWORDS:                                                                  M
*   InptPrsrTypeToStr                                                        M
*****************************************************************************/
char *InptPrsrTypeToStr(IritExprType Type)
{
    switch (Type) {
	case POLY_EXPR:
	    return "Poly";
	case NUMERIC_EXPR:
	    return "Numeric";
	case POINT_EXPR:
	    return "Point";
	case VECTOR_EXPR:
	    return "Vector";
	case PLANE_EXPR:
	    return "Plane";
	case CTLPT_EXPR:
	    return "Control Point";
	case MATRIX_EXPR:
	    return "Matrix";
	case STRING_EXPR:
	    return "String";
	case OLST_EXPR:
	    return "List Object";
	case CURVE_EXPR:
	    return "Curve";
	case SURFACE_EXPR:
	    return "Surface";
	case TRIMSRF_EXPR:
	    return "Trimmed Surface";
	case TRIVAR_EXPR:
	    return "Trivar";
	case TRISRF_EXPR:
	    return "Triangular Srf";
	default:
	    return "Undefined/Mixed";
    }

}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Counts the number of parameters in Root. Parameters are defined as         M
* subtrees seperated by commas, i.e.: the infix form 1, 2, 3, 4 is 	     M
* represented as [1, [2, [3, 4]]] in the tree supplied to this function and  M
* 4 (number of parameters) is returned.					     M
*                                                                            *
* PARAMETERS:                                                                M
*   Root:      To count the number of its parameters.                        M
*                                                                            *
* RETURN VALUE:                                                              M
*   int:       Number of parameters encountered.                             M
*                                                                            *
* KEYWORDS:                                                                  M
*   InptEvalCountNumParameters                                               M
*****************************************************************************/
int InptEvalCountNumParameters(ParseTree *Root)
{
    int i = 1;

    if (Root == NULL)
	return 0;

    while (Root -> NodeKind == COMMA) {
	i++;
	Root = Root -> Right;
    }
    return i;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Fetches the i'th paramter out of a tree represent n parameters             M
* (0 <= i < n). See InptEvalCountNumParameters for more description of	     M
* structure.								     M
* Note it is assumed the tree HAS n parameters and 0<=i<n (No input error).  M
*                                                                            *
* PARAMETERS:                                                                M
*   Root:     To fetch its i'th parameter.                                   M
*   i:        The parameter wanted.                                          M
*   n:        Total number of parameters in Root.                            M
*                                                                            *
* RETURN VALUE:                                                              M
*   ParseTree *:    The requested parameter.                                 M
*                                                                            *
* KEYWORDS:                                                                  M
*   InptEvalFetchParameter                                                   M
*****************************************************************************/
ParseTree *InptEvalFetchParameter(ParseTree *Root, int i, int n)
{
    int j;

    for (j = 0; j < i; j++)
	Root = Root -> Right;

    if (i == n - 1)
	return Root;
    else
	return Root -> Left;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Fetches the parameters of a function from the parsed tree.		     M
* Returns TRUE iff fetching was succesfull.				     M
*                                                                            *
* PARAMETERS:                                                                M
*   Root:          Where the parameters should be searched for.              M
*   FuncTable:     The table's entry of the function we deal with.           M
*   NumParams:     Number of parameters of function.                         M
*   Level:         Level of recursion. To identify top levels.               M
*   Params:        Where to put the parameters as ParseTree.                 M
*   ParamPtrs:     And as pointers to RealType/PointType/VectorType etc.     M
*                                                                            *
* RETURN VALUE:                                                              M
*   int:           Number of matches or zero if error. 			     M
*                                                                            *
* KEYWORDS:                                                                  M
*   InptEvalFetchParameters                                                  M
*****************************************************************************/
int InptEvalFetchParameters(ParseTree *Root,
			    FuncTableType *FuncTable,
			    int NumParams,
			    int Level,
			    ParseTree *Params[],
			    VoidPtr ParamPtrs[])
{
    int i;

    Level++;
    for (i = 0; i < NumParams; i++) {
	ParseTree *Param;

	if ((Param = InptEvalFetchParameter(Root -> Right, i, NumParams))
								== NULL ||
	    (Params[i] = InptPrsrEvalTree(Param, Level)) == NULL)
	    return i;

	if (FuncTable != NULL) {
	    IritExprType
		ParamObjExpr = Params[i] -> PObj ?
		    InptPrsrObjType2Expr(Root, Params[i] -> PObj) : NO_EXPR;

	    if ((ParamObjExpr & FuncTable -> ParamObjType[i]) == 0) {
		IRIT_NON_FATAL_ERROR5(
			"Eval Error: Parameter type mismatch - Func %s, parameter %d\n\texpected \"%s\", found \"%s\".",
			FuncTable -> FuncName, i + 1,
			InptPrsrTypeToStr(FuncTable -> ParamObjType[i]),
			InptPrsrTypeToStr(ParamObjExpr));
		return 0;
	    }
	    
	    switch (FuncTable -> ParamObjType[i]) {
		case NUMERIC_EXPR:
		    ParamPtrs[i] = &Params[i] -> PObj -> U.R;
		    break;
		case POINT_EXPR | VECTOR_EXPR:/* Assuming point/vector same. */
		case POINT_EXPR:
		case VECTOR_EXPR:
		    ParamPtrs[i] = Params[i] -> PObj -> U.Vec;
		    break;
		case PLANE_EXPR:
		    ParamPtrs[i] = Params[i] -> PObj -> U.Plane;
		    break;
		case CTLPT_EXPR:
		    ParamPtrs[i] = &Params[i] -> PObj -> U.CtlPt;
		    break;
		case STRING_EXPR:
		    ParamPtrs[i] = Params[i] -> PObj -> U.Str;
		    break;
		default:
		    ParamPtrs[i] = Params[i] -> PObj;
		    break;
	    }
	}
    }

    return i;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Tests number of parameters and type of them against what is defined in the M
* global tables Num/Obj/GenFuncTable. return TRUE if mismatch was detected.  M
*                                                                            *
* PARAMETERS:                                                                M
*   Root:     The function entry node.                                       M
*                                                                            *
* RETURN VALUE:                                                              M
*   int:      TRUE if a mismatch wasdetected, FALSE of o.k.		     N
*                                                                            *
* KEYWORDS:                                                                  M
*   IritEvalFuncParamMismatch                                                M
*****************************************************************************/
int IritEvalFuncParamMismatch(ParseTree *Root)
{
    int FuncOffset, Count,
	i = Root -> NodeKind / 100;
    FuncTableType *FuncTable;

    switch (i * 100) {
	case NUM_FUNC:		       /* Numeric (real returned) functions. */
	    FuncOffset = Root -> NodeKind - NUM_FUNC_OFFSET;
	    FuncTable = (FuncTableType *) NumFuncTable;
	    break;
	case OBJ_FUNC1:			     /* Object (returned) functions. */
	case OBJ_FUNC2:			     /* Object (returned) functions. */
	case OBJ_FUNC3:			     /* Object (returned) functions. */
	case OBJ_FUNC4:			     /* Object (returned) functions. */
	    FuncOffset = Root -> NodeKind - OBJ_FUNC_OFFSET;
	    FuncTable = (FuncTableType *) ObjFuncTable;
	    break;
	case GEN_FUNC:
	    FuncOffset = Root -> NodeKind - GEN_FUNC_OFFSET;
	    FuncTable = (FuncTableType *) GenFuncTable;
	    break;
	default:
	    IPGlblEvalError = IE_ERR_FATAL_ERROR;
	    UpdateCharError(IRIT_EXP_STR("Undefined function - "),
			    Root -> NodeKind, Root);
	    return TRUE;
    }

    Count = InptEvalCountNumParameters(Root -> Right);

    if (FuncTable[FuncOffset].NumOfParam == ANY_PARAM_NUM) {
	for (i = 0; i < Count; i++) {
	    IritExprType EType;

	    /* Special cases - FORLOOP and IF - top level considerations. */
	    if ((Root -> NodeKind == FORLOOP && i == 3) ||
		(Root -> NodeKind == IFCOND && i > 0))
		EType = InptPrsrTypeCheck(InptEvalFetchParameter(Root -> Right,
								 i, Count), 0);
	    else
		EType = InptPrsrTypeCheck(InptEvalFetchParameter(Root -> Right,
								 i, Count), 1);

	    /* Check only for consistency of input. */
	    if (EType == ERROR_EXPR)
		return TRUE;
	}
	return FALSE;
    }
    else { /* Number of parameter is well known and is in Count. */
	/* See if number of parameters is ok: */
	if (Count != FuncTable[FuncOffset].NumOfParam) {
	    IPGlblEvalError = IE_ERR_NUM_PRM_MISMATCH;
	    sprintf(IPGlblCharData,
		    IRIT_EXP_STR("%d expected in function \"%s\", %d found"),
		    FuncTable[FuncOffset].NumOfParam,
		    FuncTable[FuncOffset].FuncName,
		    Count);
	    return TRUE;
	}

	/* See if type of parameters are consistent: */
	for (i = 0; i < Count; i++) {
	    IritExprType EType;

	    /* Special cases - FORLOOP and IF - top level considerations. */
	    if ((Root -> NodeKind == FORLOOP && i == 3) ||
		(Root -> NodeKind == IFCOND && i > 0))
		EType = InptPrsrTypeCheck(InptEvalFetchParameter(Root -> Right,
								 i, Count), 0);
	    else
		EType = InptPrsrTypeCheck(InptEvalFetchParameter(Root -> Right,
								 i, Count), 1);

	    if (EType == ERROR_EXPR)
		return TRUE;
	    if (FuncTable[FuncOffset].ParamObjType[i] != ANY_EXPR &&
		!(FuncTable[FuncOffset].ParamObjType[i] & EType)) {
		sprintf(IPGlblCharData, IRIT_EXP_STR("Func %s,%sparameter %d"),
			FuncTable[FuncOffset].FuncName,
			IPGlblEvalError == IE_ERR_IP_OBJ_UNDEFINED ?
				    IRIT_EXP_STR(" undefined ") : " ", i + 1);
		IPGlblEvalError = IE_ERR_TYPE_MISMATCH;
		return TRUE;
	    }
	}
	return FALSE;
    }
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Deallocates a parsed tree - release all memory allocated by it.	     M
*                                                                            *
* PARAMETERS:                                                                M
*   Root:      Of tree to release.                                           M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   InptPrsrFreeTree                                                         M
*****************************************************************************/
void InptPrsrFreeTree(ParseTree *Root)
{
    char s[LINE_LEN];

    if (!Root)
	return;

    if (IS_FUNCTION(Root -> NodeKind)) {
	if (!IS_NO_PARAM_FUNC(Root -> NodeKind))
	    InptPrsrFreeTree(Root -> Right);
	if (Root -> PObj != NULL) {
	    IPFreeObject(Root -> PObj);
	}
	ExprFree(Root);

	return;
    }

    switch (Root -> NodeKind) {
	case DIV:
	case MINUS:
	case MULT:
	case PLUS:
	case POWER:
	    InptPrsrFreeTree(Root -> Right);
	    InptPrsrFreeTree(Root -> Left);
	    if (Root -> PObj != NULL) {
		IPFreeObject(Root -> PObj);
	    }
	    ExprFree(Root);
	    break;

	case UNARMINUS:
	case BOOL_NOT:
	    InptPrsrFreeTree(Root -> Right);
	    if (Root -> PObj != NULL) {
		IPFreeObject(Root -> PObj);
	    }
	    ExprFree(Root);
	    break;

	case COMMA:
	    /* Long lists of objects (polygons!?) could result in deap level */
	    /* of recursion - handle this by iterating through the list.     */
	    while (Root != NULL && Root -> NodeKind == COMMA) {
	        ParseTree
		    *Right = Root -> Right;

		InptPrsrFreeTree(Root -> Left);
		if (Root -> PObj != NULL) {
		    IPFreeObject(Root -> PObj);
		}
		ExprFree(Root);
		Root = Right;
	    }
	    if (Root != NULL)
	        InptPrsrFreeTree(Root);
	    break;
	case COLON:
	case EQUAL:
	case CMP_EQUAL:
	case CMP_NOTEQUAL:
	case CMP_LSEQUAL:
	case CMP_GTEQUAL:
	case CMP_LESS:
	case CMP_GREAT:
	case BOOL_OR:
	case BOOL_AND:
  	    InptPrsrFreeTree(Root -> Right);
	    InptPrsrFreeTree(Root -> Left);
	    if (Root -> PObj != NULL) {
		IPFreeObject(Root -> PObj);
	    }
	    ExprFree(Root);
	    break;

	case PARAMETER:
	    if (Root -> PObj) {
		IPFreeObject(Root -> PObj);
	    }
	    ExprFree(Root);
	    break;

	case NUMBER:
	case STRING:
	    if (Root -> PObj) {
		IPFreeObject(Root -> PObj);
	    }
	    ExprFree(Root);
	    break;

	case TOKENSTART:
	    ExprFree(Root);
	    break;

	case OPENPARA:
	case CLOSPARA:
	    ExprFree(Root);
	    break;

	default:
	    /*   We might free partially build (by InptPrsr) tree when error */
	    /* is detected, and such tree may have nodes with                */
	    /* NodeKind >= IP_PRSR_MAX_TOKEN.				     */
	    if (Root -> NodeKind >= IP_PRSR_MAX_TOKEN) {
		ExprFree(Root);
	    }
	    else {
		sprintf(s, IRIT_EXP_STR("%s (%d).\n"),
		    IRIT_EXP_STR("InptPrsrFreeTree: Undefined ParseTree type to free"),
		    Root -> NodeKind);
		IRIT_FATAL_ERROR(s);
	    }
	    break;
    }
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Prints the content of ROOT (using inorder traversal).			     M
* If *Str = NULL prints on stderr, else on given string Str.		     *
*                                                                            *
* PARAMETERS:                                                                M
*   Root:      Of tree to print.                                             M
*   Str:       String to write on the inorder of Root, or stderr if NULL.    M
*   StrLen:    Length of string Str.					     M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   InptPrsrPrintTree                                                        M
*****************************************************************************/
void InptPrsrPrintTree(ParseTree *Root, char *Str, int StrLen)
{
    if (Str == NULL) {
	IPGlblCharData[0] = 0;			   /* Make the string empty. */
	/* Copy to a local strings and then send to stderr. */
	LocalPrintTree(Root, 0, IPGlblCharData, INPUT_LINE_LEN);
	fprintf(stderr, IPGlblCharData);		     /* and print... */
    }
    else {
	Str[0] = 0;				   /* Make the string empty. */
	/* Dont print to stderr - copy to Str directly. */
	LocalPrintTree(Root, 0, Str, StrLen);
    }
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Auxiliary function of InptPrsrPrintTree.				     *
* expression.								     *
*                                                                            *
* PARAMETERS:                                                                *
*   Root:      Of tree to print.                                             *
*   Level:     Of recursion.                                                 *
*   Str:       String to write on the inorder of Root, or stderr if NULL.    *
*   StrLen:    Length of string Str.					     *
*                                                                            *
* RETURN VALUE:                                                              *
*   void                                                                     *
*****************************************************************************/
static void LocalPrintTree(ParseTree *Root, int Level, char *Str, int StrLen)
{
#ifndef IRIT_QUIET_STRINGS
    int Len, i,
	CloseFlag = FALSE;

    if (!Root)
	return;
    i = (Root -> NodeKind % IP_PRSR_MAX_TOKEN) / 100;

    if ((Len = (int) strlen(Str)) > StrLen - 100) {	/* Prevent overflow. */
	if (Str[Len - 1] == '.')
	    return;			 /* "..." was allready concatenated. */
	else {
	    strcat(Str, "...");
	    return;
	}
    }

#   ifdef DEBUG
    IRIT_IF_DEBUG_ON_PARAMETER(_DebugIritPrsrDispNesting)
	strcat(Str, "[");    /* Useful to see ALL nestings - no preceedings. */
#   endif /* DEBUG */

    switch (i * 100) {
	case USER_FUNC:
	    switch (Root -> NodeKind) {
		case USERFUNCDEF:
		    Level = 0;
		    CloseFlag = TRUE;
		    strcat(Str, "function(");
		    break;
		case USERPROCDEF:
		    Level = 0;
		    CloseFlag = TRUE;
		    strcat(Str, "procedure(");
		    break;
		case USERINSTDEF:
		    Level = 0;
		    CloseFlag = TRUE;
		    strcat(Str, Root -> UserFunc -> FuncName);
		    strcat(Str, "(");
		    break;
	    }
	    break;

	case NUM_FUNC:
	    Level = 0;
	    CloseFlag = TRUE;
            strcat(Str,
		   NumFuncTable[Root -> NodeKind - NUM_FUNC_OFFSET].FuncName);
	    strcat(Str, "(");
	    break;

	case OBJ_FUNC1:
	case OBJ_FUNC2:
	case OBJ_FUNC3:
	case OBJ_FUNC4:
	    Level = 0;
	    CloseFlag = TRUE;
            strcat(Str,
		   ObjFuncTable[Root -> NodeKind - OBJ_FUNC_OFFSET].FuncName);
	    strcat(Str, "(");
	    break;

	case GEN_FUNC:
	    Level = 0;
	    CloseFlag = TRUE;
            strcat(Str,
		   GenFuncTable[Root -> NodeKind - GEN_FUNC_OFFSET].FuncName);
	    strcat(Str, "(");
	    break;

	case OPERATORS:
	    switch (Root -> NodeKind % IP_PRSR_MAX_TOKEN) {
		case DIV:
		    if (Level > 1) {
			strcat(Str, "(");
		        CloseFlag = TRUE;
		    }
		    Level = 1;				       /* Div Level. */
		    LocalPrintTree(Root -> Left, Level, Str, StrLen);
		    strcat(Str, "/");
		    break;

		case MINUS:
		    if (Level > 0) {
			strcat(Str, "(");
	        	CloseFlag = TRUE;
		    }
		    Level = 0;				     /* Minus Level. */
		    LocalPrintTree(Root -> Left, Level, Str, StrLen);
		    strcat(Str, "-");
		    break;

		case MULT:
		    if (Level > 1) {
			strcat(Str, "(");
		        CloseFlag = TRUE;
		    }
		    Level = 1;				       /* Mul Level. */
		    LocalPrintTree(Root -> Left, Level, Str, StrLen);
		    strcat(Str, "*");
		    break;

		case PLUS:
		    if (Level > 0) {
			strcat(Str, "(");
	        	CloseFlag = TRUE;
		    }
		    Level = 0;				      /* Plus Level. */
		    LocalPrintTree(Root -> Left, Level, Str, StrLen);
		    strcat(Str, "+");
		    break;

		case POWER:
		    Level = 2;				     /* Power Level. */
		    LocalPrintTree(Root -> Left, Level, Str, StrLen);
		    strcat(Str, "^");
		    break;

		case UNARMINUS:
		    strcat(Str, "(-");
		    Level = 0;
		    CloseFlag = TRUE;
		    break;

		case COMMA:
		    LocalPrintTree(Root -> Left, Level, Str, StrLen);
		    strcat(Str, ",");
		    break;

		case COLON:
		    LocalPrintTree(Root -> Left, Level, Str, StrLen);
		    strcat(Str, ":");
		    break;

		case EQUAL:
		    LocalPrintTree(Root -> Left, Level, Str, StrLen);
		    strcat(Str, "=");
		    break;

		case CMP_EQUAL:
		    LocalPrintTree(Root -> Left, Level, Str, StrLen);
		    strcat(Str, "==");
		    break;

		case CMP_NOTEQUAL:
		    LocalPrintTree(Root -> Left, Level, Str, StrLen);
		    strcat(Str, "!=");
		    break;

		case CMP_LSEQUAL:
		    LocalPrintTree(Root -> Left, Level, Str, StrLen);
		    strcat(Str, "<=");
		    break;

		case CMP_GTEQUAL:
		    LocalPrintTree(Root -> Left, Level, Str, StrLen);
		    strcat(Str, ">=");
		    break;

		case CMP_LESS:
		    LocalPrintTree(Root -> Left, Level, Str, StrLen);
		    strcat(Str, "<");
		    break;

		case CMP_GREAT:
		    LocalPrintTree(Root -> Left, Level, Str, StrLen);
		    strcat(Str, ">");
		    break;

		case BOOL_OR:
		    LocalPrintTree(Root -> Left, Level, Str, StrLen);
		    strcat(Str, "||");
		    break;

		case BOOL_AND:
		    LocalPrintTree(Root -> Left, Level, Str, StrLen);
		    strcat(Str, "&&");
		    break;

		case BOOL_NOT:
		    LocalPrintTree(Root -> Left, Level, Str, StrLen);
		    strcat(Str, "!");
		    break;

		case NUMBER:
		    sprintf(&Str[strlen(Str)], "%g", Root -> PObj -> U.R);
		    break;

		case PARAMETER:
		    sprintf(&Str[strlen(Str)], "%s", Root -> PObj -> ObjName);
		    break;

		case STRING:
		    sprintf(&Str[strlen(Str)], "\"%s\"",
			    Root -> PObj -> U.Str);
		    break;

		case OPENPARA:
		    strcat(Str, "(");
		    break;

		case CLOSPARA:
		    strcat(Str, ")");
		    break;

		case TOKENSTART:
		    break;

		default:
		    IRIT_FATAL_ERROR("LocalPrintTree: Undefined ParseTree token to print, exit");
            }
            break;

	default:
	    IRIT_FATAL_ERROR("LocalPrintTree: Undefined ParseTree type to print, exit");
    }
    LocalPrintTree(Root -> Right, Level, Str, StrLen);
    if (CloseFlag)
	strcat(Str, ")");

#   ifdef DEBUG
    IRIT_IF_DEBUG_ON_PARAMETER(_DebugIritPrsrDispNesting)
	strcat(Str, "]");    /* Useful to see ALL nestings - no preceedings. */
#   endif /* DEBUG */
#endif /* !IRIT_QUIET_STRINGS */
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Duplicates a parse tree - Generates brand new ParseTree structure but      M
* binds to non-temp variables if they are exists - their Name is not NULL.   M
*   This means that updating these objects in the copied tree, will affect   M
* these objects in the original tree.					     M
*                                                                            *
* PARAMETERS:                                                                M
*   Root:     To duplicate.                                                  M
*                                                                            *
* RETURN VALUE:                                                              M
*   ParseTree *:   Duplicated parse tree                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   InptPrsrCopyTree                                                         M
*****************************************************************************/
ParseTree *InptPrsrCopyTree(ParseTree *Root)
{
    ParseTree *NewRoot;

    if (Root == NULL)
	return NULL;

    NewRoot = ExprMalloc();

    if (IS_FUNCTION(Root -> NodeKind)) {	       /* All the functions. */
	NewRoot -> NodeKind = Root -> NodeKind;
	NewRoot -> Right = InptPrsrCopyTree(Root -> Right);
	if (IS_USER_FUNCTION(Root -> NodeKind))
	    NewRoot -> UserFunc = Root -> UserFunc;
	return NewRoot;
    }

    switch (Root -> NodeKind) {
	case DIV:
	case MINUS:
	case MULT:
	case PLUS:
	case POWER:

	case COMMA:
	case COLON:
	case EQUAL:
	case CMP_EQUAL:
	case CMP_NOTEQUAL:
	case CMP_LSEQUAL:
	case CMP_GTEQUAL:
	case CMP_LESS:
	case CMP_GREAT:
        case BOOL_OR:
        case BOOL_AND:
	    NewRoot -> NodeKind = Root -> NodeKind;
	    NewRoot -> Right = InptPrsrCopyTree(Root -> Right);
	    NewRoot -> Left  = InptPrsrCopyTree(Root -> Left);
	    return NewRoot;

	case UNARMINUS:
        case BOOL_NOT:
	    NewRoot -> NodeKind = Root -> NodeKind;
	    NewRoot -> Right = InptPrsrCopyTree(Root -> Right);
	    NewRoot -> Left  = NULL;
	    return NewRoot;

	case NUMBER:
	case PARAMETER:
	case STRING:
	    NewRoot -> NodeKind = Root -> NodeKind;
	    NewRoot -> PObj = Root -> PObj;	    /* Point on SAME object. */
	    NewRoot -> PObj -> Count++;      /* But increase its ref. count. */
	    return NewRoot;

	case TOKENSTART:
	    NewRoot -> NodeKind = Root -> NodeKind;
	    return NewRoot;

	default:
	    IRIT_FATAL_ERROR("InptPrsrCopyTree: Undefined ParseTree type to copy, exit");
    }
    return NULL;				    /* Makes warning silent. */
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Routine to return evaluation error if happen one, zero elsewhere	     M
*                                                                            *
* PARAMETERS:                                                                M
*   Message:    Place here a message describing an error, if was one.        M
*                                                                            *
* RETURN VALUE:                                                              M
*   InptPrsrEvalErrType:  Type of evaluation error.                          M
*                                                                            *
* KEYWORDS:                                                                  M
*   InptPrsrEvalError                                                        M
*****************************************************************************/
InptPrsrEvalErrType InptPrsrEvalError(char **Message)
{
    InptPrsrEvalErrType Temp;

    *Message = IPGlblCharData;
    Temp = IPGlblEvalError;
    IPGlblEvalError = IPE_NO_ERR;

    return Temp;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Given an object and a parse tree, updates the dependencies and	     M
* parameters of object PObj according to parse tree Root.		     M
*   Root is assumed to be of the form "PObjName = ..."			     M
*                                                                            *
* PARAMETERS:                                                                M
*   PObj:      To update its parameter dependency list.                      M
*   Root:      Expression that created PObj as "PObjName = ...".             M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   InptEvalPropagateDependencies                                            M
*****************************************************************************/
void InptEvalPropagateDependencies(IPObjectStruct *PObj, ParseTree *Root)
{
    STATIC_DATA unsigned int
	UpdateIndex = 1;
    char Str[IP_EXPR_MAX_STRING_LEN];
    IPODObjectDpndncyStruct
	*Dpnds = PObj -> Dpnds;

    if (Dpnds != NULL) {
	IPODParamsStruct *p;

	/* Free the parameters of this object as we are going to redo them. */
	/*   Note we first remove this object's name from all other objects */
	/* this object depends upon.					    */
	for (p = Dpnds -> ObjParams; p != NULL; p = p -> Pnext) {
	    IPObjectStruct
		*PObjParam = IPGetObjectByName(p -> Name,
					       GlblObjList,
					       FALSE);

	    if (PObjParam != NULL)
		IPODDelDependencyFromObj(PObjParam -> Dpnds, PObj -> ObjName);
	}
	IPODFreeParametersOfObj(Dpnds -> ObjParams);
	Dpnds -> ObjParams = NULL;
	Dpnds -> NumParams = 0;

	if (Dpnds -> EvalExpr != NULL)
	    IritFree(Dpnds -> EvalExpr);
    }
    else
	PObj -> Dpnds = Dpnds = IPODNewDependencies();

    /* Now parse the parse tree and search for other objects and insert them */
    /* into the parameter list and insert this object into their dependency  */
    /* lists.								     */
    InptPrsrPrintTree(Root, Str, IP_EXPR_MAX_STRING_LEN);
    strcat(Str, ";");
    Dpnds -> EvalExpr = IritStrdup(Str);

#   ifdef DEBUG
    IRIT_IF_DEBUG_ON_PARAMETER(_DebugIritPrsrDepndncy)
        fprintf(stderr,	"Object \"%s\" eval expression:\n\t%s\n",
		PObj -> ObjName, Str);
#   endif /* DEBUG */

    InptPrsrPropagateDependencyObjects(PObj, Root);
    InptPrsrPropagateUpdatedObjects(PObj, UpdateIndex);

    /* Increment the propagation index only at top recursion level. */
    if (IritInputSource != IRIT_INPUT_SOURCE_LINE_QUEUE)
        UpdateIndex++;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Scans the given tree for object names and set up dependencies upon.	     *
*                                                                            *
* PARAMETERS:                                                                *
*   PObj:      Object whose dependencies are to be updated.                  *
*   Root:      Of tree to scan for parameters.                               *
*                                                                            *
* RETURN VALUE:                                                              *
*   void                                                                     *
*****************************************************************************/
static void InptPrsrPropagateDependencyObjects(IPObjectStruct *PObj,
					       ParseTree *Root)
{
    if (!Root)
	return;

    if (IS_FUNCTION(Root -> NodeKind)) {
	InptPrsrPropagateDependencyObjects(PObj, Root -> Right);
	return;
    }

    switch (Root -> NodeKind) {
	case DIV:
	case MINUS:
	case MULT:
	case PLUS:
	case POWER:
	case COMMA:
	case COLON:
	case EQUAL:
	case CMP_EQUAL:
	case CMP_NOTEQUAL:
	case CMP_LSEQUAL:
	case CMP_GTEQUAL:
	case CMP_LESS:
	case CMP_GREAT:
	case BOOL_OR:
	case BOOL_AND:
	    InptPrsrPropagateDependencyObjects(PObj, Root -> Right);
	    InptPrsrPropagateDependencyObjects(PObj, Root -> Left);
	    break;

	case UNARMINUS:
	case BOOL_NOT:
	    InptPrsrPropagateDependencyObjects(PObj, Root -> Right);
	    break;

	case PARAMETER:
	    if (Root -> PObj) {
		IPObjectStruct
		    *PObjParam = IPGetObjectByName(Root -> PObj -> ObjName,
						   GlblObjList,
						   FALSE);

		/* Let the other object know we depend on it. */
		if (PObjParam != NULL &&
		    strcmp(PObj -> ObjName, PObjParam -> ObjName) != 0) {
		    IPODAddDependencyToObj(&PObjParam -> Dpnds,
					   PObj -> ObjName);

		    /* Place the other object's name as our parameter. */
		    IPODAddParameterToObj(&PObj -> Dpnds,
					  PObjParam -> ObjName);

#		    ifdef DEBUG
		    IRIT_IF_DEBUG_ON_PARAMETER(_DebugIritPrsrDepndncy)
			fprintf(stderr,
				"Object \"%s\" - add \"%s\" as a parameter\n",
				PObj -> ObjName, PObjParam -> ObjName);
#		    endif /* DEBUG */
		}
	    }
	    break;

	default:
	    break;
    }
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Recursively update all objects that depend on PObj.                      *
*                                                                            *
* PARAMETERS:                                                                *
*   PObj:      An object that was modified to propagate.                     *
*   UpdateIndex:  Index of this update/propagation session.	             *
*                                                                            *
* RETURN VALUE:                                                              *
*   void                                                                     *
*****************************************************************************/
static void InptPrsrPropagateUpdatedObjects(IPObjectStruct *PObj,
					    int UpdateIndex)
{
    IPODObjectDpndncyStruct
	*Dpnds = PObj -> Dpnds;
    IPODDependsStruct *p;

#   ifdef DEBUG
    IRIT_IF_DEBUG_ON_PARAMETER(_DebugIritPrsrDepndncy)
        fprintf(stderr,
		"Object %s with index %d and visit #%d, Global index %d\n",
		PObj -> ObjName, Dpnds -> EvalIndex,
		Dpnds -> NumVisits, UpdateIndex);
#   endif /* DEBUG */

    if (Dpnds -> EvalIndex == UpdateIndex) {
        if (++Dpnds -> NumVisits > Dpnds -> NumParams)
	    IRIT_NON_FATAL_ERROR("Cycles detected in dependencies; aborted");
	return;
    }
    else {
	Dpnds -> EvalIndex = UpdateIndex;
	Dpnds -> NumVisits = 1;
    }

    /* Scan all objects that depends on this object and reevaluate them. */
    for (p = Dpnds -> ObjDepends; p != NULL; p = p -> Pnext) {
        IPObjectStruct
	    *PObjDepend = IPGetObjectByName(p -> Name, GlblObjList, FALSE);

	if (PObjDepend != NULL) {
	    IPODObjectDpndncyStruct
	        *Dpnds2 = PObjDepend -> Dpnds;

	    if (Dpnds2 != NULL) {
	        if (Dpnds2 -> EvalExpr != NULL) {
		    /* Evaluate expression that created object PObjDepend. */
		    InptPrsrQueueInputLine(Dpnds2 -> EvalExpr);
		    if (GlblHandleDependencies == 2)
		        fprintf(stderr, IRIT_EXP_STR("Reeval: %s\n"),
				Dpnds2 -> EvalExpr);
		}
	    }
	}
    }
}
