/*****************************************************************************
*   "Irit" - the 3d (not only polygonal) solid modeller.		     *
*									     *
* Written by:  Gershon Elber				Ver 0.2, Mar. 1990   *
******************************************************************************
* (C) Gershon Elber, Technion, Israel Institute of Technology                *
******************************************************************************
*   Module to provide the required interfact for the cagd library for the    *
* free form surfaces and curves.					     *
*****************************************************************************/

#include <stdio.h>
#include <math.h>
#include "program.h"
#include "allocate.h"
#include "attribut.h"
#include "objects.h"
#include "geom_lib.h"
#include "ip_cnvrt.h"
#include "bool_lib.h"
#include "freeform.h"

STATIC_DATA IPObjectStruct
    *GlblGeom2PolyObj = NULL;
STATIC_DATA int
    GlblGeom2PolyNormals = FALSE,
    GlblGeom2PolyOptimal = FALSE;

static IPObjectStruct *ComputeCurveIsoLines(IPObjectStruct *PObj,
					    int Optimal);
static IPObjectStruct *ComputeSurfaceIsoLines(IPObjectStruct *PObj,
					      int Optimal);
static IPObjectStruct *ComputeTrimSrfIsoLines(IPObjectStruct *PObj,
					      int Optimal);
static IPObjectStruct *ComputeTrivarIsoLines(IPObjectStruct *PObj,
					     int Optimal);
static IPObjectStruct *ComputeTriSrfIsoLines(IPObjectStruct *PObj,
					     int Optimal);
static void Geometry2PolygonsAux(IPObjectStruct *Obj, MatrixType Mat);
static void Geometry2PolylinesAux(IPObjectStruct *Obj, MatrixType Mat);

/*****************************************************************************
* DESCRIPTION:                                                               M
* Routine to reverse a surface by flipping the U and V directions.	     M
*                                                                            *
* PARAMETERS:                                                                M
*   SrfObj:     Surface to reverse.                                          M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:    Reversed surface. The normal of the reversed        M
*                        surface is flipped with respected to the original   M
*                        surface, SrfObj, by 180 degrees.		     M
*                                                                            *
* KEYWORDS:                                                                  M
*   SurfaceReverse                                                           M
*****************************************************************************/
IPObjectStruct *SurfaceReverse(IPObjectStruct *SrfObj)
{
    IPObjectStruct
	*RetVal = NULL;

    if (IP_IS_SRF_OBJ(SrfObj)) {
	CagdSrfStruct
	    *RevSrf = CagdSrfReverse2(SrfObj -> U.Srfs);

	if (RevSrf != NULL)
	    RetVal = IPGenSRFObject(RevSrf);
    }
    else if (IP_IS_TRIMSRF_OBJ(SrfObj)) {
	TrimSrfStruct
	    *RevSrf = TrimSrfReverse2(SrfObj -> U.TrimSrfs);

	if (RevSrf != NULL)
	    RetVal = IPGenTRIMSRFObject(RevSrf);
    }

    return RetVal;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Routine to convert curve(s) to a piecewise linear polyline approximation   *
* and convert its control polygon into polyline as well.		     *
*                                                                            *
* PARAMETERS:                                                                *
*   PObj:       Curves to approximate using piecewise linear approximation.  *
*   Optimal:    Do we want an optimal sampling but expensive approach?       *
*                                                                            *
* RETURN VALUE:                                                              *
*   IPObjectStruct *:   A polyline approximation to PObj.                    *
*****************************************************************************/
static IPObjectStruct *ComputeCurveIsoLines(IPObjectStruct *PObj, int Optimal)
{
    RealType
	Resolution = GetResolutionReal();
    IPObjectStruct *PObjPoly;
    CagdCrvStruct *Crv;

    if (!IP_IS_CRV_OBJ(PObj))
	IRIT_FATAL_ERROR("Curve was expected.");

    if (Optimal == 0 && Resolution < MIN_FREE_FORM_RES)
	Resolution = MIN_FREE_FORM_RES;

    PObjPoly = IPGenPOLYObject(NULL);
    IP_ATTR_SAFECOPY_ATTRS(PObjPoly -> Attr, PObj -> Attr);
    IP_SET_POLYLINE_OBJ(PObjPoly);
    for (Crv = PObj -> U.Crvs; Crv != NULL; Crv = Crv -> Pnext) {
	IPPolygonStruct *Poly;

	Poly = IPCurve2Polylines(Crv, Resolution,
				 (SymbCrvApproxMethodType) Optimal);

	PObjPoly -> U.Pl = IPAppendPolyLists(Poly, PObjPoly -> U.Pl);
    }

    return PObjPoly;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Routine to convert a surface to a set of piecewise linear polyline	     *
* approximation and convert its control mesh into set of polyline as well.   *
*                                                                            *
* PARAMETERS:                                                                *
*   PObj:        Surfaces to apporximate as a mesh of piecewise linear       *
*		 isocurves.                                                  *
*   Optimal:     Do we want an optimal sampling but expensive approach?      *
*                                                                            *
* RETURN VALUE:                                                              *
*   IPObjectStruct *:  A polyline object approximating PObj.                 *
*****************************************************************************/
static IPObjectStruct *ComputeSurfaceIsoLines(IPObjectStruct *PObj,
					      int Optimal)
{
    RealType
	Resolution = GetResolutionReal();
    IPObjectStruct *PObjPolys;
    CagdSrfStruct *Srf;

    if (!IP_IS_SRF_OBJ(PObj))
	IRIT_FATAL_ERROR("Surface was expected.");

    if (Optimal == 0 && Resolution < MIN_FREE_FORM_RES)
	Resolution = MIN_FREE_FORM_RES;

    PObjPolys = IPGenPOLYObject(NULL);
    IP_ATTR_SAFECOPY_ATTRS(PObjPolys -> Attr, PObj -> Attr);
    IP_SET_POLYLINE_OBJ(PObjPolys);
    for (Srf = PObj -> U.Srfs; Srf != NULL; Srf = Srf -> Pnext) {
	int NumOfIso[2];
	IPPolygonStruct *Polys, *PolyTmp;

	NumOfIso[0] = NumOfIso[1] = (int) -Resolution;
	Polys = IPSurface2Polylines(Srf, NumOfIso, Resolution,
				    (SymbCrvApproxMethodType) Optimal);

	for (PolyTmp = Polys;
	     PolyTmp -> Pnext != NULL;
	     PolyTmp = PolyTmp -> Pnext);
	PolyTmp -> Pnext = PObjPolys -> U.Pl;
	PObjPolys -> U.Pl = Polys;
    }

    return PObjPolys;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Routine to convert a trimmed surface to a set of piecewise linear polyline *
* approximation and convert its control mesh into set of polyline as well.   *
*                                                                            *
* PARAMETERS:                                                                *
*   PObj:        Trimmed surfaces to apporximate as a mesh of piecewise      *
*		  linear isocurves.                                          *
*   Optimal:     Do we want an optimal sampling but expensive approach?      *
*                                                                            *
* RETURN VALUE:                                                              *
*   IPObjectStruct *:  A polyline object approximating PObj.                 *
*****************************************************************************/
static IPObjectStruct *ComputeTrimSrfIsoLines(IPObjectStruct *PObj,
					      int Optimal)
{
    RealType
	Resolution = GetResolutionReal();
    IPObjectStruct *PObjPolys;
    TrimSrfStruct *TrimSrf;

    if (!IP_IS_TRIMSRF_OBJ(PObj))
	IRIT_FATAL_ERROR("Trimmed surface was expected.");

    if (Optimal == 0 && Resolution < MIN_FREE_FORM_RES)
	Resolution = MIN_FREE_FORM_RES;

    PObjPolys = IPGenPOLYObject(NULL);
    IP_ATTR_SAFECOPY_ATTRS(PObjPolys -> Attr, PObj -> Attr);
    IP_SET_POLYLINE_OBJ(PObjPolys);
    for (TrimSrf = PObj -> U.TrimSrfs;
	 TrimSrf != NULL;
	 TrimSrf = TrimSrf -> Pnext) {
	int NumOfIso[2];
	IPPolygonStruct *Polys, *PolyTmp;

	NumOfIso[0] = NumOfIso[1] = (int) -Resolution;
	Polys = IPTrimSrf2Polylines(TrimSrf, NumOfIso, Resolution,
				    (SymbCrvApproxMethodType) Optimal,
				    TRUE, TRUE);

	for (PolyTmp = Polys;
	     PolyTmp -> Pnext != NULL;
	     PolyTmp = PolyTmp -> Pnext);
	PolyTmp -> Pnext = PObjPolys -> U.Pl;
	PObjPolys -> U.Pl = Polys;
    }

    return PObjPolys;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Routine to convert a trimmed surface to a set of piecewise linear polyline *
* approximation and convert its control mesh into set of polyline as well.   *
*                                                                            *
* PARAMETERS:                                                                *
*   PObj:        Trimmed surfaces to apporximate as a mesh of piecewise      *
*		  linear isocurves.                                          *
*   Optimal:     Do we want an optimal sampling but expensive approach?      *
*                                                                            *
* RETURN VALUE:                                                              *
*   IPObjectStruct *:  A polyline object approximating PObj.                 *
*****************************************************************************/
static IPObjectStruct *ComputeTrivarIsoLines(IPObjectStruct *PObj,
					     int Optimal)
{
    RealType
	Resolution = GetResolutionReal();
    IPObjectStruct *PObjPolys;
    TrivTVStruct *Trivar;

    if (!IP_IS_TRIVAR_OBJ(PObj))
	IRIT_FATAL_ERROR("Trivariate function was expected.");

    if (Optimal == 0 && Resolution < MIN_FREE_FORM_RES)
	Resolution = MIN_FREE_FORM_RES;

    PObjPolys = IPGenPOLYObject(NULL);
    IP_ATTR_SAFECOPY_ATTRS(PObjPolys -> Attr, PObj -> Attr);
    IP_SET_POLYLINE_OBJ(PObjPolys);
    for (Trivar = PObj -> U.Trivars;
	 Trivar != NULL;
	 Trivar = Trivar -> Pnext) {
	int NumOfIso[3],
	    Res = (int) MAX(Resolution * 0.5, 2);
	IPPolygonStruct *Polys, *PolyTmp;

	NumOfIso[0] = NumOfIso[1] = NumOfIso[2] = -Res;
	Polys = IPTrivar2Polylines(Trivar, NumOfIso, Resolution,
				   (SymbCrvApproxMethodType) Optimal);

	for (PolyTmp = Polys;
	     PolyTmp -> Pnext != NULL;
	     PolyTmp = PolyTmp -> Pnext);
	PolyTmp -> Pnext = PObjPolys -> U.Pl;
	PObjPolys -> U.Pl = Polys;
    }

    return PObjPolys;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Routine to convert a triangular surface to a set of piecewise linear	     *
* polyline approximation and convert its control mesh into set of polyline   *
* as well.								     *
*                                                                            *
* PARAMETERS:                                                                *
*   PObj:        Triangular surfaces to apporximate as a mesh of piecewise   *
*		 linear isocurves.                                           *
*   Optimal:     Do we want an optimal sampling but expensive approach?      *
*                                                                            *
* RETURN VALUE:                                                              *
*   IPObjectStruct *:  A polyline object approximating PObj.                 *
*****************************************************************************/
static IPObjectStruct *ComputeTriSrfIsoLines(IPObjectStruct *PObj,
					     int Optimal)
{
    RealType
	Resolution = GetResolutionReal();
    IPObjectStruct *PObjPolys;
    TrngTriangSrfStruct *TriSrf;

    if (!IP_IS_TRISRF_OBJ(PObj))
	IRIT_FATAL_ERROR("Triangular surface was expected.");

    if (Optimal == 0 && Resolution < MIN_FREE_FORM_RES)
	Resolution = MIN_FREE_FORM_RES;

    PObjPolys = IPGenPOLYObject(NULL);
    IP_ATTR_SAFECOPY_ATTRS(PObjPolys -> Attr, PObj -> Attr);
    IP_SET_POLYLINE_OBJ(PObjPolys);
    for (TriSrf = PObj -> U.TriSrfs;
	 TriSrf != NULL;
	 TriSrf = TriSrf -> Pnext) {
	int NumOfIso[3];
	IPPolygonStruct *Polys, *PolyTmp;

	NumOfIso[0] = NumOfIso[1] = NumOfIso[2] = (int) -Resolution;
	Polys = IPTriSrf2Polylines(TriSrf, NumOfIso, Resolution,
				   (SymbCrvApproxMethodType) Optimal);

	for (PolyTmp = Polys;
	     PolyTmp -> Pnext != NULL;
	     PolyTmp = PolyTmp -> Pnext);
	PolyTmp -> Pnext = PObjPolys -> U.Pl;
	PObjPolys -> U.Pl = Polys;
    }

    return PObjPolys;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Routine to convert a surface to a set of polygons approximating it.	     M
*   Result is saved as an attribute on the surface.			     M
*   If, however, approximation already exists, no computation is performed.  M
*                                                                            *
* PARAMETERS:                                                                M
*   PObj:      Surface to approximate using polygons.                        M
*   Normals:   Compute normals as well, if TRUE.	                     M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   ComputeSurfacePolygons                                                   M
*****************************************************************************/
void ComputeSurfacePolygons(IPObjectStruct *PObj, int Normals)
{
  int	OldOnlyTri, OldMergeCoplanarPolys,
	Resolution = GetResolution(FALSE),
	FourPerFlat = GetFourPerFlat(),
	PolyApproxOpt = GetPolyApproxOptimal(),
	PolyApproxUV = GetPolyApproxUV(),
	PolyApproxTri = GetPolyApproxTri(),
	PolyMergeCoplanarPolys = GetPolyMergeCoplanar(),
        BoolUVBooleanState = BoolSetParamSurfaceUVVals(TRUE);
    RealType t,
	RelResolution = AttrGetObjectRealAttrib(PObj, "resolution"),
	Tolerance = GetPolyApproxTol();
    IPPolygonStruct *Polys;
    IPObjectStruct *PObjPoly;

    BoolSetParamSurfaceUVVals(BoolUVBooleanState);/* Restore bool_lib state. */

    if (AttrGetObjectObjAttrib(PObj, "_polygons") != NULL)
	return;

    if (Resolution < MIN_FREE_FORM_RES)
	Resolution = MIN_FREE_FORM_RES;
    if (!IP_ATTR_IS_BAD_REAL(RelResolution))
	Resolution = REAL_TO_INT(Resolution * RelResolution);

#ifndef __MSDOS__
    /* Make the resolution more reasonable (very slow on MSDOS). */
    Resolution *= 2;
#endif /* __MSDOS__ */

    if (AttrGetObjectStrAttrib(PObj, "twoperflat") != NULL)
	FourPerFlat = FALSE;
    if (AttrGetObjectStrAttrib(PObj, "fourperflat") != NULL)
	FourPerFlat = TRUE;

    t = AttrGetObjectRealAttrib(PObj, "u_resolution");
    if (!IP_ATTR_IS_BAD_REAL(t))
        AttrSetRealAttrib(&PObj -> U.Srfs -> Attr, "u_resolution", t);
    t = AttrGetObjectRealAttrib(PObj, "v_resolution");
    if (!IP_ATTR_IS_BAD_REAL(t))
        AttrSetRealAttrib(&PObj -> U.Srfs -> Attr, "v_resolution", t);

    OldOnlyTri = IPSurface2PolygonsGenTriOnly(PolyApproxTri);
    CagdSrfSetMakeOnlyTri(PolyApproxTri);
    OldMergeCoplanarPolys = CagdSrf2PolygonMergeCoplanar(PolyMergeCoplanarPolys);

    Polys = IPSurface2Polygons(PObj -> U.Srfs, FourPerFlat,
			       PolyApproxOpt ? Tolerance : Resolution,
			       PolyApproxUV || BoolUVBooleanState,
			       Normals, PolyApproxOpt);

    IPSurface2PolygonsGenTriOnly(OldOnlyTri);
    CagdSrfSetMakeOnlyTri(OldOnlyTri);
    CagdSrf2PolygonMergeCoplanar(OldMergeCoplanarPolys);

    PObjPoly = IPGenPolyObject("", Polys, NULL);
    IP_ATTR_SAFECOPY_ATTRS(PObjPoly -> Attr, PObj -> Attr);
    AttrSetObjectObjAttrib(PObj, "_polygons", PObjPoly, FALSE);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Routine to convert a triangular surface to a set of polygons		     M
* approximating it.							     M
*   Result is saved as an attribute on the surface.			     M
*   If, however, approximation already exists, no computation is performed.  M
*                                                                            *
* PARAMETERS:                                                                M
*   PObj:      Triangular surface to approximate using polygons.             M
*   Normals:   Compute normals as well, if TRUE.	                     M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   ComputeTriSrfPolygons                                                    M
*****************************************************************************/
void ComputeTriSrfPolygons(IPObjectStruct *PObj, int Normals)
{
    int	OldOnlyTri,
	Resolution = GetResolution(FALSE),
	PolyApproxOpt = GetPolyApproxOptimal(),
	PolyApproxUV = GetPolyApproxUV(),
	PolyApproxTri = GetPolyApproxTri(),
        BoolUVBooleanState = BoolSetParamSurfaceUVVals(TRUE);
    RealType
	RelResolution = AttrGetObjectRealAttrib(PObj, "resolution"),
	Tolerance =  GetPolyApproxTol();
    IPPolygonStruct *Polys;
    IPObjectStruct *PObjPoly;

    BoolSetParamSurfaceUVVals(BoolUVBooleanState);/* Restore bool_lib state. */

    if (AttrGetObjectObjAttrib(PObj, "_polygons") != NULL)
	return;

    if (Resolution < MIN_FREE_FORM_RES)
	Resolution = MIN_FREE_FORM_RES;
    if (!IP_ATTR_IS_BAD_REAL(RelResolution))
	Resolution = REAL_TO_INT(Resolution * RelResolution);

#ifndef __MSDOS__
    /* Make the resolution more reasonable (very slow on MSDOS). */
    Resolution *= 2;
#endif /* __MSDOS__ */

    OldOnlyTri = IPSurface2PolygonsGenTriOnly(PolyApproxTri);

    /* No optimal tesselation support so comment that out here: */
    Polys = IPTriSrf2Polygons(PObj -> U.TriSrfs,
			      /* PolyApproxOpt ? Tolerance : */ Resolution,
			      PolyApproxUV || BoolUVBooleanState,
			      Normals, PolyApproxOpt);

    CagdSrfSetMakeOnlyTri(OldOnlyTri);

    PObjPoly = IPGenPolyObject("", Polys, NULL);
    IP_ATTR_SAFECOPY_ATTRS(PObjPoly -> Attr, PObj -> Attr);
    AttrSetObjectObjAttrib(PObj, "_polygons", PObjPoly, FALSE);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Routine to convert a trimmed surface to a set of polygons approximating it.M
*   Result is saved as an attribute on the trimmed surface.		     M
*   If, however, approximation already exists, no computation is performed.  M
*                                                                            *
* PARAMETERS:                                                                M
*   PObj:      Trimmed surface to approximate using polygons.                M
*   Normals:   Compute normals as well, if TRUE.	                     M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   ComputeTrimSrfPolygons                                                   M
*****************************************************************************/
void ComputeTrimSrfPolygons(IPObjectStruct *PObj, int Normals)
{
    int	OldOnlyTri,
	Resolution = GetResolution(FALSE),
	FourPerFlat = GetFourPerFlat(),
	PolyApproxOpt = GetPolyApproxOptimal(),
	PolyApproxUV = GetPolyApproxUV(),
	PolyApproxTri = GetPolyApproxTri();
    RealType t,
	RelResolution = AttrGetObjectRealAttrib(PObj, "resolution"),
	Tolerance =  GetPolyApproxTol();
    IPPolygonStruct *Polys;
    IPObjectStruct *PObjPoly;

    if (AttrGetObjectObjAttrib(PObj, "_polygons") != NULL)
	return;

    if (Resolution < MIN_FREE_FORM_RES)
	Resolution = MIN_FREE_FORM_RES;
    if (!IP_ATTR_IS_BAD_REAL(RelResolution))
	Resolution = REAL_TO_INT(Resolution * RelResolution);

#ifndef __MSDOS__
    /* Make the resolution more reasonable (very slow on MSDOS). */
    Resolution *= 2;
#endif /* __MSDOS__ */

    if (AttrGetObjectStrAttrib(PObj, "twoperflat") != NULL)
	FourPerFlat = FALSE;
    if (AttrGetObjectStrAttrib(PObj, "fourperflat") != NULL)
	FourPerFlat = TRUE;

    t = AttrGetObjectRealAttrib(PObj, "u_resolution");
    if (!IP_ATTR_IS_BAD_REAL(t))
        AttrSetRealAttrib(&PObj -> U.TrimSrfs -> Attr, "u_resolution", t);
    t = AttrGetObjectRealAttrib(PObj, "v_resolution");
    if (!IP_ATTR_IS_BAD_REAL(t))
        AttrSetRealAttrib(&PObj -> U.TrimSrfs -> Attr, "v_resolution", t);

    OldOnlyTri = IPSurface2PolygonsGenTriOnly(PolyApproxTri);

    Polys = IPTrimSrf2Polygons(PObj -> U.TrimSrfs, FourPerFlat,
			       PolyApproxOpt ? Tolerance : Resolution,
			       PolyApproxUV, Normals, PolyApproxOpt);

    CagdSrfSetMakeOnlyTri(OldOnlyTri);

    PObjPoly = IPGenPolyObject("", Polys, NULL);
    IP_ATTR_SAFECOPY_ATTRS(PObjPoly -> Attr, PObj -> Attr);
    AttrSetObjectObjAttrib(PObj, "_polygons", PObjPoly, FALSE);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Routine to convert a trivariate function to a set of polygons            M
* approximating it.							     M
*   Result is saved as an attribute on the trimmed surface.		     M
*   If, however, approximation already exists, no computation is performed.  M
*                                                                            *
* PARAMETERS:                                                                M
*   PObj:      Trivariate function surface to approximate using polygons.    M
*   Normals:   Compute normals as well, if TRUE.	                     M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   ComputeTrivarPolygons                                                    M
*****************************************************************************/
void ComputeTrivarPolygons(IPObjectStruct *PObj, int Normals)
{
    int	OldOnlyTri,
	Resolution = GetResolution(FALSE),
	FourPerFlat = GetFourPerFlat(),
	PolyApproxOpt = GetPolyApproxOptimal(),
	PolyApproxUV = GetPolyApproxUV(),
	PolyApproxTri = GetPolyApproxTri();
    RealType t,
	RelResolution = AttrGetObjectRealAttrib(PObj, "resolution"),
	Tolerance =  GetPolyApproxTol();
    IPPolygonStruct *Polys;
    IPObjectStruct *PObjPoly;

    if (AttrGetObjectObjAttrib(PObj, "_polygons") != NULL)
	return;

    if (Resolution < MIN_FREE_FORM_RES)
	Resolution = MIN_FREE_FORM_RES;
    if (!IP_ATTR_IS_BAD_REAL(RelResolution))
	Resolution = REAL_TO_INT(Resolution * RelResolution);

#ifndef __MSDOS__
    /* Make the resolution more reasonable (very slow on MSDOS). */
    Resolution *= 2;
#endif /* __MSDOS__ */

    if (AttrGetObjectStrAttrib(PObj, "twoperflat") != NULL)
	FourPerFlat = FALSE;
    if (AttrGetObjectStrAttrib(PObj, "fourperflat") != NULL)
	FourPerFlat = TRUE;

    t = AttrGetObjectRealAttrib(PObj, "u_resolution");
    if (!IP_ATTR_IS_BAD_REAL(t))
        AttrSetRealAttrib(&PObj -> U.TrimSrfs -> Attr, "u_resolution", t);
    t = AttrGetObjectRealAttrib(PObj, "v_resolution");
    if (!IP_ATTR_IS_BAD_REAL(t))
        AttrSetRealAttrib(&PObj -> U.TrimSrfs -> Attr, "v_resolution", t);

    OldOnlyTri = IPSurface2PolygonsGenTriOnly(PolyApproxTri);

    Polys = IPTrivar2Polygons(PObj -> U.Trivars, FourPerFlat,
			       PolyApproxOpt ? Tolerance : Resolution,
			       PolyApproxUV, Normals, PolyApproxOpt);

    CagdSrfSetMakeOnlyTri(OldOnlyTri);

    PObjPoly = IPGenPolyObject("", Polys, NULL);
    PObjPoly -> Attr = IP_ATTR_COPY_ATTRS(PObj -> Attr);
    AttrSetObjectObjAttrib(PObj, "_polygons", PObjPoly, FALSE);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Routine to convert a surface/list of surfaces into set of polygons.	     M
*                                                                            *
* PARAMETERS:                                                                M
*   Obj:       A geometry to convert and approximate using polygons.	     M
*   RNormals:   Do we want normals as well (at vertices)?                    M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:   A polygonal object approximating Obj.                M
*                                                                            *
* KEYWORDS:                                                                  M
*   Geometry2Polygons                                                        M
*****************************************************************************/
IPObjectStruct *Geometry2Polygons(IPObjectStruct *Obj, RealType *RNormals)
{
    MatrixType Mat;
    int OldTraverseCopyObjState = IPTraverseObjectCopy(TRUE);

    GlblGeom2PolyObj = NULL;
    GlblGeom2PolyNormals= REAL_PTR_TO_INT(RNormals);

    MatGenUnitMat(Mat);

    if (Obj -> Pnext == NULL && IPGetLastObj(GlblObjList) != Obj) {
	/* Obj is not in display list - put it first and search it first. */
	Obj -> Pnext = GlblObjList;
	IPTraverseObjHierarchy(Obj, Obj, Geometry2PolygonsAux, Mat, FALSE);
	Obj -> Pnext = NULL;
    }
    else 
	IPTraverseObjHierarchy(Obj, GlblObjList, Geometry2PolygonsAux,
			       Mat, FALSE);

    IPTraverseObjectCopy(OldTraverseCopyObjState);

    return GlblGeom2PolyObj;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Auxiliary function of Geometry2Polygons. Do the leaves.                  *
*****************************************************************************/
static void Geometry2PolygonsAux(IPObjectStruct *Obj, MatrixType Mat)
{
    IPObjectStruct
	*RetVal = NULL;

    if (IP_IS_POLY_OBJ(Obj)) {
	if (IP_IS_POLYGON_OBJ(Obj))
	    RetVal = IPCopyObject(NULL, Obj, FALSE);
    }
    else if (IP_IS_SRF_OBJ(Obj) ||
	     IP_IS_TRISRF_OBJ(Obj) ||
	     IP_IS_TRIMSRF_OBJ(Obj)) {
	if (AttrGetObjectObjAttrib(Obj, "_polygons") != NULL)
	    AttrFreeOneAttribute(&Obj -> Attr, "_polygons");

	if (IP_IS_SRF_OBJ(Obj))
	    ComputeSurfacePolygons(Obj, GlblGeom2PolyNormals);
	else if (IP_IS_TRISRF_OBJ(Obj))
	    ComputeTriSrfPolygons(Obj, GlblGeom2PolyNormals);
	else if (IP_IS_TRIMSRF_OBJ(Obj))
	    ComputeTrimSrfPolygons(Obj, GlblGeom2PolyNormals);

	RetVal = IPCopyObject(NULL, AttrGetObjectObjAttrib(Obj, "_polygons"),
									FALSE);

	if (!GlblGeom2PolyNormals) {
	    IPPolygonStruct *Pl;

	    for (Pl = RetVal -> U.Pl; Pl != NULL; Pl = Pl -> Pnext) {
		IPVertexStruct
		    *V = Pl -> PVertex;

		do {
		    IP_RST_NORMAL_VRTX(V);

		    V = V -> Pnext;
		}
		while (V != NULL && V != Pl -> PVertex);
	    }
	}
    }
    else if (IP_IS_TRIVAR_OBJ(Obj)) {
	if (AttrGetObjectObjAttrib(Obj, "_polygons") != NULL)
	    AttrFreeOneAttribute(&Obj -> Attr, "_polygons");

	ComputeTrivarPolygons(Obj, GlblGeom2PolyNormals);

	RetVal = IPCopyObject(NULL, AttrGetObjectObjAttrib(Obj, "_polygons"),
									FALSE);

	if (!GlblGeom2PolyNormals) {
	    IPPolygonStruct *Pl;

	    for (Pl = RetVal -> U.Pl; Pl != NULL; Pl = Pl -> Pnext) {
		IPVertexStruct
		    *V = Pl -> PVertex;

		do {
		    IP_RST_NORMAL_VRTX(V);

		    V = V -> Pnext;
		}
		while (V != NULL && V != Pl -> PVertex);
	    }
	}
    }
    else {
	IRIT_WNDW_PUT_STR("Unconvertable to polygons object ignored");
	return;
    }

    if (RetVal != NULL && RetVal -> U.Pl != NULL) {
        IPObjectStruct
	    *PObjTmp = GMTransformObject(RetVal, Mat);

	IPFreeObject(RetVal);
	RetVal = PObjTmp;

	if (GlblGeom2PolyObj != NULL) {
	    IPGetLastPoly(RetVal -> U.Pl) -> Pnext = GlblGeom2PolyObj -> U.Pl;
	    GlblGeom2PolyObj -> U.Pl = RetVal -> U.Pl;
	    RetVal -> U.Pl = NULL;
	    IPFreeObject(RetVal);
	}
	else {
	    GlblGeom2PolyObj = RetVal;
	    IP_ATTR_FREE_ATTRS(GlblGeom2PolyObj -> Attr);
	}
    }
    else if (RetVal != NULL)
	IPFreeObject(RetVal);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Routine to convert a surface/curve/list of these into set of polylines.    M
*                                                                            *
* PARAMETERS:                                                                M
*   Obj:       A geometry to convert and approximate using polygons.	     M
*   Optimal:   Do we want an optimal sampling but expensive approach?        M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:   A polyline object approximating Obj.                 M
*                                                                            *
* KEYWORDS:                                                                  M
*   Geometry2Polylines                                                       M
*****************************************************************************/
IPObjectStruct *Geometry2Polylines(IPObjectStruct *Obj, RealType *Optimal)
{
    MatrixType Mat;
    int OldTraverseCopyObjState = IPTraverseObjectCopy(TRUE);

    GlblGeom2PolyObj = NULL;
    GlblGeom2PolyOptimal = REAL_PTR_TO_INT(Optimal) ?
			SYMB_CRV_APPROX_TOLERANCE : SYMB_CRV_APPROX_UNIFORM;

    MatGenUnitMat(Mat);

    if (Obj -> Pnext == NULL && IPGetLastObj(GlblObjList) != Obj) {
	/* Obj is not in display list - put it first and search it first. */
	Obj -> Pnext = GlblObjList;
	IPTraverseObjHierarchy(Obj, Obj, Geometry2PolylinesAux, Mat, FALSE);
	Obj -> Pnext = NULL;
    }
    else 
	IPTraverseObjHierarchy(Obj, GlblObjList, Geometry2PolylinesAux,
			       Mat, FALSE);
    IPTraverseObjectCopy(OldTraverseCopyObjState);

    return GlblGeom2PolyObj;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Auxiliary function of Geometry2Polylines. Do the leaves.                 *
*****************************************************************************/
static void Geometry2PolylinesAux(IPObjectStruct *Obj, MatrixType Mat)
{
    IPObjectStruct
	*RetVal = NULL;

    if (IP_IS_POLY_OBJ(Obj)) {
	if (IP_IS_POLYGON_OBJ(Obj)) {
	    RetVal = IPCopyObject(NULL, Obj, FALSE);
	    IP_SET_POLYLINE_OBJ(RetVal);
	}
	else
	    RetVal = IPCopyObject(NULL, Obj, FALSE);
    }
    else if (IP_IS_CRV_OBJ(Obj)) {
	RetVal = ComputeCurveIsoLines(Obj, GlblGeom2PolyOptimal);
    }
    else if (IP_IS_SRF_OBJ(Obj)) {
	RetVal = ComputeSurfaceIsoLines(Obj, GlblGeom2PolyOptimal);
    }
    else if (IP_IS_TRIMSRF_OBJ(Obj)) {
	RetVal = ComputeTrimSrfIsoLines(Obj, GlblGeom2PolyOptimal);
    }
    else if (IP_IS_TRIVAR_OBJ(Obj)) {
	RetVal = ComputeTrivarIsoLines(Obj, GlblGeom2PolyOptimal);
    }
    else if (IP_IS_TRISRF_OBJ(Obj)) {
	RetVal = ComputeTriSrfIsoLines(Obj, GlblGeom2PolyOptimal);
    }
    else {
	IRIT_WNDW_PUT_STR("Unconvertable to polylines object ignored");
    }

    if (RetVal != NULL) {
        IPObjectStruct
	    *PObjTmp = GMTransformObject(RetVal, Mat);

	IPFreeObject(RetVal);
	RetVal = PObjTmp;

	if (GlblGeom2PolyObj != NULL) {
	    IPGetLastPoly(RetVal -> U.Pl) -> Pnext = GlblGeom2PolyObj -> U.Pl;
	    GlblGeom2PolyObj -> U.Pl = RetVal -> U.Pl;
	    RetVal -> U.Pl = NULL;
	    IPFreeObject(RetVal);
	}
	else {
	    GlblGeom2PolyObj = RetVal;
	    IP_ATTR_FREE_ATTRS(GlblGeom2PolyObj -> Attr);
	}
    }    
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Routine to return extremum value of control mesh/polygon.		     M
*                                                                            *
* PARAMETERS:                                                                M
*   Obj:      Object to bound itsextremum possible values.                   M
*   Min:      TRUE for minimum, FALSE for maximum.                           M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:   A control points with extremum values. Rational are  M
*                       Correctly porjected back to Euclidean space.         M
*                                                                            *
* KEYWORDS:                                                                  M
*   ExtremumControlPointVals                                                 M
*****************************************************************************/
IPObjectStruct *ExtremumControlPointVals(IPObjectStruct *Obj, CagdRType *Min)
{
    CagdBType
	FindMin = REAL_PTR_TO_INT(Min);
    CagdRType *Extremum,
	**Points = NULL;
    int Len = 0;
    CagdPointType
	PType  = CAGD_PT_E1_TYPE;

    if (IP_IS_SRF_OBJ(Obj)) {
	Points = Obj -> U.Srfs -> Points;
	Len = Obj -> U.Srfs -> ULength * Obj -> U.Srfs -> VLength;
	PType = Obj -> U.Srfs -> PType;
    }
    else if (IP_IS_CRV_OBJ(Obj)) {
	Points = Obj -> U.Crvs -> Points;
	Len = Obj -> U.Crvs -> Length;
	PType = Obj -> U.Crvs -> PType;
    }
    else {
	IRIT_NON_FATAL_ERROR("Extremum allowed on curves/surfaces only");
	return NULL;
    }

    Extremum = SymbExtremumCntPtVals(Points, Len, FindMin);

    return IPGenCTLPTObject(PType, Extremum, NULL);
}
 
/*****************************************************************************
* DESCRIPTION:                                                               M
* Creates an exact rational quadratic circle parallel to the XY plane.	     M
*                                                                            *
* PARAMETERS:                                                                M
*   Position:   Center of circle.                                            M
*   Radius:     Radius of circle.                                            M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:   A rational quadratic Bspline curve object 	     M
*			representing a circle.				     M
*                                                                            *
* KEYWORDS:                                                                  M
*   GenCircleCurveObject                                                     M
*****************************************************************************/
IPObjectStruct *GenCircleCurveObject(VectorType Position, RealType *Radius)
{
    int i;
    CagdPtStruct Pos;
    CagdCrvStruct *CircCrv;
    IPObjectStruct *CrvObj;

    for (i = 0; i < 3; i++)
	Pos.Pt[i] = Position[i];
    CircCrv = BspCrvCreateCircle(&Pos, *Radius);

    if (CircCrv == NULL)
	return NULL;

    CrvObj = IPGenCRVObject(CircCrv);

    return CrvObj;
}
 
/*****************************************************************************
* DESCRIPTION:                                                               M
* Creates an approximated cubic polynomial circle parallel to the XY plane.  M
*                                                                            *
* PARAMETERS:                                                                M
*   Position:   Center of circle.                                            M
*   Radius:     Radius of circle.                                            M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:   A cubic polynomial Bspline curve object 	     M
*			representing a circle.				     M
*                                                                            *
* KEYWORDS:                                                                  M
*   GenPCircleCurveObject                                                    M
*****************************************************************************/
IPObjectStruct *GenPCircleCurveObject(VectorType Position, RealType *Radius)
{
    int i;
    CagdPtStruct Pos;
    CagdCrvStruct *CircCrv;
    IPObjectStruct *CrvObj;

    for (i = 0; i < 3; i++)
	Pos.Pt[i] = Position[i];
    CircCrv = BspCrvCreatePCircle(&Pos, *Radius);

    if (CircCrv == NULL)
	return NULL;

    CrvObj = IPGenCRVObject(CircCrv);

    return CrvObj;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Creates an approximated spiral curve.					     M
*                                                                            *
* PARAMETERS:                                                                M
*   NumOfLoops:    Number of loops in the spiral - can be fractional.        M
*   Pitch:         Essentially the size of the spiral.  A Pitch of one will  M
*		   construct a roughly size-one spiral curve.		     M
*   RSampling:     Number of samples to compute on the spiral.  Should be    M
*		   several hundreds for a reasonable result.		     M
*   RCtlPtsPerLoop: Number of control points to use per loop. Use at least 5 M
*                  for a reasonable approximation.			     M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:   A cubic polynomial Bspline curve object 	     M
*			representing a circle.				     M
*                                                                            *
* KEYWORDS:                                                                  M
*   GenSpiralCurveObject                                                     M
*****************************************************************************/
IPObjectStruct *GenSpiralCurveObject(CagdRType *NumOfLoops,
				     CagdRType *Pitch,
				     CagdRType *RSampling,
				     CagdRType *RCtlPtsPerLoop)
{
    IPObjectStruct *CrvObj;
    CagdCrvStruct
	*SpiralCrv = BspCrvCreateApproxSpiral(*NumOfLoops, *Pitch,
					      REAL_PTR_TO_INT(RSampling),
					      REAL_PTR_TO_INT(RCtlPtsPerLoop));

    if (SpiralCrv == NULL)
	return NULL;

    CrvObj = IPGenCRVObject(SpiralCrv);

    return CrvObj;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Creates an arbitrary arc specified by Two end points and Center. Arc must  M
* be less than 180 degree.						     M
*                                                                            *
* PARAMETERS:                                                                M
*   Start:    Location of arc.                                               M
*   Center:   Location of arc.                                               M
*   End:      Location of arc.                                               M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:  A curve representing the requested arc.               M
*                                                                            *
* KEYWORDS:                                                                  M
*   GenArcCurveObject                                                        M
*****************************************************************************/
IPObjectStruct *GenArcCurveObject(VectorType Start,
				  VectorType Center,
				  VectorType End)
{
    int i;
    CagdPtStruct StartPt, CenterPt, EndPt;
    CagdCrvStruct *ArcCrv;
    IPObjectStruct *CrvObj;

    for (i = 0; i < 3; i++) {
	StartPt.Pt[i] = Start[i];
	CenterPt.Pt[i] = Center[i];
	EndPt.Pt[i] = End[i];
    }
    ArcCrv = BzrCrvCreateArc(&StartPt, &CenterPt, &EndPt);

    if (ArcCrv == NULL)
	return NULL;

    CrvObj = IPGenCRVObject(ArcCrv);

    return CrvObj;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Creates an arbitrary arc specified by Two end points and Center. Arc must  M
* be less than 360 degree.						     M
*                                                                            *
* PARAMETERS:                                                                M
*   Center:      Location of center of arc.                                  M
*   Radius:      Of arc.                                                     M
*   StartAngle:  Of arc.                                                     M
*   EndAngle:    Of arc.                                                     M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:  A curve representing the requested arc.               M
*                                                                            *
* KEYWORDS:                                                                  M
*   GenArc2CurveObject                                                       M
*****************************************************************************/
IPObjectStruct *GenArc2CurveObject(VectorType Center,
				   RealType *Radius,
				   RealType *StartAngle,
				   RealType *EndAngle)
{
    int i;
    CagdPtStruct CenterPt;
    CagdCrvStruct *ArcCrv;
    IPObjectStruct *CrvObj;

    for (i = 0; i < 3; i++)
	CenterPt.Pt[i] = Center[i];

    ArcCrv = CagdCrvCreateArc(&CenterPt, *Radius, *StartAngle, *EndAngle);

    if (ArcCrv == NULL)
	return NULL;

    CrvObj = IPGenCRVObject(ArcCrv);

    return CrvObj;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Constructs a ruled surface out of the two provided curves.		     M
*                                                                            *
* PARAMETERS:                                                                M
*   Obj1, Obj2:  Two polys/curves to rule a surface between.                 M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *: A ruled surface object.                                M
*                                                                            *
* KEYWORDS:                                                                  M
*   GenRuledSrfObject                                                        M
*****************************************************************************/
IPObjectStruct *GenRuledSrfObject(IPObjectStruct *Obj1, IPObjectStruct *Obj2)
{
    IPObjectStruct
	*SrfObj = NULL;

    if (IP_IS_POLY_OBJ(Obj1) && IP_IS_POLY_OBJ(Obj2)) {
	SrfObj = PrimGenRULEDObject(Obj1, Obj2);
    }
    else if (IP_IS_CRV_OBJ(Obj1) && IP_IS_CRV_OBJ(Obj2)) {
	CagdSrfStruct
	    *RuledSrf = CagdRuledSrf(Obj1 -> U.Crvs, Obj2 -> U.Crvs, 2, 2);

	if (RuledSrf == NULL)
	    return NULL;

	SrfObj = IPGenSRFObject(RuledSrf);
    }
    else {
	IRIT_FATAL_ERROR("RuledSrf: improper input geometry to ruled surface!");
    }

    return SrfObj;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Constructs a ruled trivariate out of the two provided surfaces.	     M
*                                                                            *
* PARAMETERS:                                                                M
*   Obj1, Obj2:  Two surfaces to rule into a trivariate between.             M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *: A ruled trivariate object.                             M
*                                                                            *
* KEYWORDS:                                                                  M
*   GenRuledTVObject                                                         M
*****************************************************************************/
IPObjectStruct *GenRuledTVObject(IPObjectStruct *Obj1, IPObjectStruct *Obj2)
{
    IPObjectStruct
	*TVObj = NULL;

    if (IP_IS_SRF_OBJ(Obj1) && IP_IS_SRF_OBJ(Obj2)) {
	TrivTVStruct
	    *RuledTV = TrivRuledTV(Obj1 -> U.Srfs, Obj2 -> U.Srfs, 2, 2);

	if (RuledTV == NULL)
	    return NULL;

	TVObj = IPGenTRIVARObject(RuledTV);
    }
    else {
	IRIT_FATAL_ERROR("RuledTV: improper input geometry to ruled trivariate!");
    }

    return TVObj;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Construct a boolean sum surface out of a single boundary curve.	     M
*                                                                            *
* PARAMETERS:                                                                M
*   BndryCrv:  A curve to fill its interior with a surface. Better be close. M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *: A filling surface object.                              M
*                                                                            *
* KEYWORDS:                                                                  M
*   GenBoolOneSrfObject                                                      M
*****************************************************************************/
IPObjectStruct *GenBoolOneSrfObject(IPObjectStruct *BndryCrv)
{
    IPObjectStruct *SrfObj;
    CagdSrfStruct
	*BoolSumSrf = CagdOneBoolSumSrf(BndryCrv -> U.Crvs);

    if (BoolSumSrf == NULL)
	return NULL;

    SrfObj = IPGenSRFObject(BoolSumSrf);

    return SrfObj;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Construct a boolean sum surface out of the four provided curves.	     M
*                                                                            *
* PARAMETERS:                                                                M
*   Crv1, Crv2, Crv3, Crv4:  Four curves to construct a Boolean sum between. M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:  The Boolean sum surface.                              M
*                                                                            *
* KEYWORDS:                                                                  M
*   GenBoolSumSrfObject                                                      M
*****************************************************************************/
IPObjectStruct *GenBoolSumSrfObject(IPObjectStruct *Crv1,
				    IPObjectStruct *Crv2,
				    IPObjectStruct *Crv3,
				    IPObjectStruct *Crv4)
{
    IPObjectStruct *SrfObj;
    CagdSrfStruct
	*BoolSumSrf = CagdBoolSumSrf(Crv1 -> U.Crvs,
				     Crv2 -> U.Crvs,
				     Crv3 -> U.Crvs,
				     Crv4 -> U.Crvs);

    if (BoolSumSrf == NULL)
	return NULL;

    SrfObj = IPGenSRFObject(BoolSumSrf);

    return SrfObj;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Construct a surface out of the provided curve list.			     M
*                                                                            *
* PARAMETERS:                                                                M
*   CrvList:     A list of curves to approximate a surface through.          M
*   OtherOrder:  Other order of surface.                                     M
*   OtherEC:	 End conditions of other direction.                          M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:   A surface approximately traversing through the given M
*                       curves.                                              M
*                                                                            *
* KEYWORDS:                                                                  M
*   GenSrfFromCrvsObject                                                     M
*****************************************************************************/
IPObjectStruct *GenSrfFromCrvsObject(IPObjectStruct *CrvList,
				     RealType *OtherOrder,
				     RealType *OtherEC)
{
    int i,
	NumCrvs = 0;
    IPObjectStruct *SrfObj, *CrvObj;
    CagdSrfStruct *Srf;
    CagdCrvStruct *Crv,
	*Crvs = NULL;

    if (!IP_IS_OLST_OBJ(CrvList))
	IRIT_FATAL_ERROR("SURFACE: Not object list object!");

    while ((CrvObj = IPListObjectGet(CrvList, NumCrvs)) != NULL) {
	if (!IP_IS_CRV_OBJ(CrvObj)) {
	    IRIT_NON_FATAL_ERROR("SURFACE: List contains non curve object(s).");
	    return NULL;
	}
	if (CrvObj -> U.Crvs -> Pnext != NULL) {
	    IRIT_NON_FATAL_ERROR("SURFACE: nested curve lists are disallowed.");
	    return NULL;
	}
	NumCrvs++;
    }

    /* Chain all curves into a single list and invoke the srf constructor: */
    for (i = 0; i < NumCrvs; i++) {
	CrvObj = IPListObjectGet(CrvList, i);
	Crv = CagdCrvCopy(CrvObj -> U.Crvs);
	LIST_PUSH(Crv, Crvs);
    }

    Crvs = CagdListReverse(Crvs);
    switch (REAL_PTR_TO_INT(OtherEC)) {
	case KV_UNIFORM_OPEN:
	    Srf = CagdSrfFromCrvs(Crvs, REAL_PTR_TO_INT(OtherOrder),
				  CAGD_END_COND_OPEN);
	    break;
	case KV_UNIFORM_FLOAT:
	    Srf = CagdSrfFromCrvs(Crvs, REAL_PTR_TO_INT(OtherOrder),
				  CAGD_END_COND_FLOAT);
	    break;
	case KV_UNIFORM_PERIODIC:
	    Srf = CagdSrfFromCrvs(Crvs, REAL_PTR_TO_INT(OtherOrder),
				  CAGD_END_COND_PERIODIC);
	    break;
    }

    CagdCrvFreeList(Crvs);

    if (Srf == NULL)
	return NULL;

    SrfObj = IPGenSRFObject(Srf);

    return SrfObj;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Constructs a Sweep surface out of the CrossSection curve, Axis curve and   M
* optional Scaling curve and Scaler which scales the CrossSection.	     M
*                                                                            *
* PARAMETERS:                                                                M
*   CrossSection:  Cross section(s) of sweep.                                M
*   Axis:          Axis curve of sweep.                                      M
*   Frame:         Orientation specification. Either an orientation curve    M
*		   Or a vector specification.				     M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:  A sweep surface.                                      M
*                                                                            *
* KEYWORDS:                                                                  M
*   GenSweepSrfObject                                                        M
*****************************************************************************/
IPObjectStruct *GenSweepSrfObject(IPObjectStruct *CrossSection,
				  IPObjectStruct *Axis,
				  IPObjectStruct *Frame)
{
    RealType
	R = 0.0;

    return GenSweepScaleSrfObject(CrossSection, Axis, NULL, Frame, &R);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Constructs a Sweep surface out of the CrossSection curve, Axis curve and   M
* optional Scaling curve and Scaler which scales the CrossSection.	     M
*                                                                            *
* PARAMETERS:                                                                M
*   CrossSection:  Cross section of sweep.                                   M
*   Axis:          Axis curve of sweep.                                      M
*   Scale:         Either a numeric constant scale, or a scaling scalar      M
*		   curve, or NULL for no scaling.			     M
*   Frame:         Orientation specification. Either an orientation curve    M
*		   Or a vector specification.				     M
*   RRefine:       Refinement control.		                             M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:  A sweep surface.                                      M
*                                                                            *
* KEYWORDS:                                                                  M
*   GenSweepScaleSrfObject                                                   M
*****************************************************************************/
IPObjectStruct *GenSweepScaleSrfObject(IPObjectStruct *CrossSection,
				       IPObjectStruct *Axis,
				       IPObjectStruct *Scale,
				       IPObjectStruct *Frame,
				       RealType *RRefine)
{
    int FrameIsCrv = IP_IS_CRV_OBJ(Frame),
	HasFrame = FrameIsCrv || IP_IS_VEC_OBJ(Frame),
	Refine = REAL_PTR_TO_INT(RRefine);
    CagdCrvStruct
	*AxisCrv = Axis -> U.Crvs,
	*CrossSecCrvs = NULL;
    CagdSrfStruct *SweepSrf;

    if (IP_IS_OLST_OBJ(CrossSection)) {
	int i = 0;
	IPObjectStruct *CrossSecObj;
	CagdCrvStruct *TCrv;

	/* Extract all the cross sections into a linked list. */
        while ((CrossSecObj = IPListObjectGet(CrossSection, i++)) != NULL) {
	    if (!IP_IS_CRV_OBJ(CrossSecObj)) {
		IRIT_NON_FATAL_ERROR("SWEEPSRF: Cross section list contains non curve object(s).");
		return NULL;
	    }
	    TCrv = CagdCrvCopy(CrossSecObj -> U.Crvs);
	    LIST_PUSH(TCrv, CrossSecCrvs);
	}

	CrossSecCrvs = (CagdCrvStruct *) CagdListReverse(CrossSecCrvs);
    }
    else
	CrossSecCrvs = CagdCrvCopy(CrossSection -> U.Crvs);

    if (Refine > 0) {
	CagdCrvStruct
	    *TCrv = CagdSweepAxisRefine(AxisCrv, NULL, Refine);

	AxisCrv = TCrv;
    }

    SweepSrf = CagdSweepSrf(CrossSecCrvs, AxisCrv,
			    (Scale != NULL &&
			        IP_IS_CRV_OBJ(Scale)) ? Scale -> U.Crvs : NULL,
			    (Scale != NULL &&
			        IP_IS_NUM_OBJ(Scale)) ? Scale -> U.R : 1.0,
			    HasFrame ? (FrameIsCrv
					        ? (VoidPtr) Frame -> U.Crvs
					        : (VoidPtr) Frame -> U.Vec)
				     : NULL,
			    FrameIsCrv);

    CagdCrvFreeList(CrossSecCrvs);

    if (AxisCrv != Axis -> U.Crvs)
	CagdCrvFree(AxisCrv);

    return SweepSrf == NULL ? NULL : IPGenSRFObject(SweepSrf);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Computes an approximation to the offset of a (planar) curve or a surface.  M
*                                                                            *
* PARAMETERS:                                                                M
*   Obj:        Curve to approximate its offset by Offset amount.            M
*   Offset:     Amount of offset, either a number or a scalar distance crv.  M
*   Tolerance:  Accuracy of offset.                                          M
*   BezInterp:  Do we want Bezier interpolation or approximation?            M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:    An offset approximating to Obj.                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   GenOffsetObject                                                          M
*****************************************************************************/
IPObjectStruct *GenOffsetObject(IPObjectStruct *Obj,
				IPObjectStruct *Offset,
				RealType *Tolerance,
				RealType *BezInterp)
{
    if (IP_IS_POLY_OBJ(Obj)) {
	IPPolygonStruct
	    *Pl = GMPolyOffset(Obj -> U.Pl, IP_IS_POLYGON_OBJ(Obj),
			       Offset -> U.R, NULL);
	IPObjectStruct 
	    *PlObj = IPGenPOLYObject(Pl);

    if (IP_IS_POLYGON_OBJ(Obj))
        IP_SET_POLYGON_OBJ(PlObj);
    else
        IP_SET_POLYLINE_OBJ(PlObj);

	return PlObj;
    }
    else if (IP_IS_SRF_OBJ(Obj)) {
    	IPObjectStruct *SrfObj;
    	CagdSrfStruct
	    *OffsetSrf = SymbSrfSubdivOffset(Obj -> U.Srfs, Offset -> U.R,
					     *Tolerance);

    	if (OffsetSrf == NULL)
	    return NULL;

    	SrfObj = IPGenSRFObject(OffsetSrf);

    	return SrfObj;
    }
    else if (IP_IS_TRIMSRF_OBJ(Obj)) {
    	IPObjectStruct *TSrfObj;
    	CagdSrfStruct
	    *OffsetSrf = SymbSrfSubdivOffset(Obj -> U.TrimSrfs -> Srf,
					     Offset -> U.R, *Tolerance);

    	if (OffsetSrf == NULL)
	    return NULL;

    	TSrfObj = IPGenTRIMSRFObject(TrimSrfNew(OffsetSrf,
			     TrimCrvCopyList(Obj -> U.TrimSrfs -> TrimCrvList),
					      TRUE));

    	return TSrfObj;
    }
    else if (IP_IS_CRV_OBJ(Obj)) {
    	IPObjectStruct *CrvObj;
    	CagdCrvStruct *OffsetCrv;

	if (IP_IS_CRV_OBJ(Offset))
	    OffsetCrv = SymbCrvVarOffset(Obj -> U.Crvs, Offset -> U.Crvs,
					 REAL_PTR_TO_INT(BezInterp));
	else
	    OffsetCrv = SymbCrvSubdivOffset(Obj -> U.Crvs, Offset -> U.R,
					    *Tolerance,
					    REAL_PTR_TO_INT(BezInterp));

    	if (OffsetCrv == NULL)
	    return NULL;

    	CrvObj = IPGenCRVObject(OffsetCrv);

    	return CrvObj;
    }
    else {
	IRIT_NON_FATAL_ERROR("Offset allowed on polys/curves/surfaces only");
	return NULL;
    }
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Computes an approximation to the offset of a (planar) curve.		     M
*   This offset is computed and approximated to with a given tolerance       M
* Epsilon that is related to the square of the distance between the original M
* curve and its offset approximation.					     M
*   If Trim then regions in the curve with curvature that is larger than     M
* offset distance required will be trimmed out.				     M
*                                                                            *
* PARAMETERS:                                                                M
*   Obj:        Curve to approximate its offset by Offset amount.            M
*   Offset:     Amount of offset, either a number or a scalar distance crv.  M
*   Epsilon:    Accuracy of offset.                                          M
*   Trim:       Should we deal with and trim self intersecting loops?        M
*   BezInterp:  Do we want Bezier interpolation or approximation?            M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:    An offset approximating to Obj.                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   GenAOffsetObject                                                         M
*****************************************************************************/
IPObjectStruct *GenAOffsetObject(IPObjectStruct *Obj,
				 IPObjectStruct *Offset,
				 RealType *Epsilon,
				 RealType *Trim,
				 RealType *BezInterp)
{
    if (IP_IS_SRF_OBJ(Obj)) {
	IRIT_NON_FATAL_ERROR("Adaptive offset is not supported for surfaces");
	return NULL;
    }
    else if (IP_IS_CRV_OBJ(Obj)) {
    	IPObjectStruct *CrvObj;
    	CagdCrvStruct *OffsetCrv;

	if (APX_EQ(*Trim, 0.0)) {
	    if (IP_IS_CRV_OBJ(Offset))
	        OffsetCrv = SymbCrvAdapVarOffset(Obj -> U.Crvs,
						 Offset -> U.Crvs,
						 *Epsilon, NULL,
						 REAL_PTR_TO_INT(BezInterp));
	    else
	        OffsetCrv = SymbCrvAdapOffset(Obj -> U.Crvs, Offset -> U.R,
					      *Epsilon, NULL,
					      REAL_PTR_TO_INT(BezInterp));
	}
	else				    
	    OffsetCrv = SymbCrvAdapOffsetTrim(Obj -> U.Crvs, Offset -> U.R,
					      *Epsilon, NULL,
					      REAL_PTR_TO_INT(BezInterp));

    	if (OffsetCrv == NULL)
	    return NULL;
	else
	    CrvObj = IPGenCRVObject(OffsetCrv);

    	return CrvObj;
    }
    else {
	IRIT_NON_FATAL_ERROR("Offset allowed on curves/surfaces only");
	return NULL;
    }
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Computes an approximation to the offset of a (planar) curve.		     M
*   This offset is computed and approximated using a least sqaure fit.	     M
*                                                                            *
* PARAMETERS:                                                                M
*   Obj:          Curve to approximate its offset by Offset amount.          M
*   Offset:       Amount of offset.                                          M
*   NumOfSamples: To sample the offset curve.                                M
*   NumOfDOF:     Number of control points in resulting approximation.       M
*   Order:        Of resulting approximation.                                M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:    An offset approximating to Obj.                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   GenLeastSqrOffsetObject                                                  M
*****************************************************************************/
IPObjectStruct *GenLeastSqrOffsetObject(IPObjectStruct *Obj,
					RealType *Offset,
					RealType *NumOfSamples,
					RealType *NumOfDOF,
					RealType *Order)
{
    CagdRType Tolerance;

    if (IP_IS_CRV_OBJ(Obj)) {
    	CagdCrvStruct
	    *OffsetCrv = SymbCrvLeastSquarOffset(Obj -> U.Crvs,
						 *Offset,
						 REAL_PTR_TO_INT(NumOfSamples),
						 REAL_PTR_TO_INT(NumOfDOF),
						 REAL_PTR_TO_INT(Order),
						 &Tolerance);

    	if (OffsetCrv == NULL)
	    return NULL;
	else
	    return IPGenCRVObject(OffsetCrv);
    }
    else {
	IRIT_NON_FATAL_ERROR("LOffset allowed on curves only");
	return NULL;
    }
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Computes an approximation to the offset of a (planar) curve.		     M
*   This offset is computed and approximated using a tangent field matcing.  M
*                                                                            *
* PARAMETERS:                                                                M
*   Obj:          Curve to approximate its offset by Offset amount.          M
*   Offset:       Amount of offset.                                          M
*   Tolerance:    Accuracy of offset, measured in offset direction (normal   M
*		  of curve) angular error, in radians.                       M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:    An offset approximating to Obj.                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   GenMatchingOffsetObject                                                  M
*****************************************************************************/
IPObjectStruct *GenMatchingOffsetObject(IPObjectStruct *Obj,
					RealType *Offset,
					RealType *Tolerance)
{
    if (IP_IS_CRV_OBJ(Obj)) {
    	CagdCrvStruct
	    *OffsetCrv = SymbCrvMatchingOffset(Obj -> U.Crvs,
					       *Offset,
					       *Tolerance);

    	if (OffsetCrv == NULL)
	    return NULL;
	else
	    return IPGenCRVObject(OffsetCrv);
    }
    else {
	IRIT_NON_FATAL_ERROR("MOffset allowed on curves only");
	return NULL;
    }
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Trim OffCrv in all locations it is closer than TrimAmount to Crv.  Tol   M
* controls the accuracy of the solution.                                     M
*                                                                            *
* PARAMETERS:                                                                M
*   Crv:        Original curve.                                              M
*   OffCrv:     Approximated offset curve.                                   M
*   Tol:        Tolerance of trimming computation.                           M
*   TrimAmount: Amount to trim OffCrv, if closer than that to Crv.           M
*   NumerTol:   If Positive, a numerical marching improvement step is        M
*		applied with NumerTol tolerance to the derived               M
*		intersection/clipped regions.				     M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:  List of curve segments, after the trimming.           M
*                                                                            *
* SEE ALSO:                                                                  M
*   SymbCrvTrimGlblOffsetSelfInter                                           M
*                                                                            *
* KEYWORDS:                                                                  M
*   TrimOffsetObject                                                         M
*****************************************************************************/
IPObjectStruct *TrimOffsetObject(IPObjectStruct *Crv,
				 IPObjectStruct *OffCrv,
				 RealType *Tol,
				 RealType *TrimAmount,
				 RealType *NumerTol)
{
    CagdCrvStruct *TCrv,
	*TrimCrvs = SymbCrvTrimGlblOffsetSelfInter(Crv -> U.Crvs,
						   OffCrv -> U.Crvs,
						   *Tol, *TrimAmount,
						   *NumerTol);

    if (TrimCrvs != NULL) {
        int i = 0;
	IPObjectStruct
	    *NewPObj = IPGenLISTObject(NULL);

	while (TrimCrvs != NULL) {
	    TCrv = TrimCrvs;
	    TrimCrvs = TrimCrvs -> Pnext;
	    TCrv -> Pnext = NULL;

	    IPListObjectInsert(NewPObj, i++, IPGenCRVObject(TCrv));
	}

	IPListObjectInsert(NewPObj, i, NULL);

	return NewPObj;
    }
    else
        return NULL;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Affine reparametrization, effectively changing the domain of the curve.    M
*                                                                            *
* PARAMETERS:                                                                M
*   Obj:         A curve to change its parametric domain.                    M
*   TMin, TMax:  New domain for Obj.					     M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:  A curve identical to Obj but with parametric domain   M
*                      from TMin to TMax.                                    M
*                                                                            *
* KEYWORDS:                                                                  M
*   CrvReparametrization                                                     M
*****************************************************************************/
IPObjectStruct *CrvReparametrization(IPObjectStruct *Obj,
				     RealType *TMin,
				     RealType *TMax)
{
    IPObjectStruct *CrvObj;
    CagdCrvStruct
	*Crv = CAGD_IS_BEZIER_CRV(Obj -> U.Crvs) ?
			  CnvrtBezier2BsplineCrv(Obj -> U.Crvs) :
			  CagdCrvCopy(Obj -> U.Crvs);
    int ParamType1 = REAL_PTR_TO_INT(TMin),
	ParamType2 = REAL_PTR_TO_INT(TMax),
	Len = CAGD_CRV_PT_LST_LEN(Crv) + Crv -> Order;
    CagdRType MinDomain, MaxDomain,
	*KV = Crv -> KnotVector;

    switch (ParamType1) {
	case CAGD_NIELSON_FOLEY_PARAM:
	case CAGD_CHORD_LEN_PARAM:
	case CAGD_CENTRIPETAL_PARAM:
	case CAGD_UNIFORM_PARAM:
	    if (ParamType1 != ParamType2) {
		IRIT_NON_FATAL_ERROR("SREPARAM: Wrong parametrization spec.");
		return NULL;
	    }
	    BspReparameterizeCrv(Crv, ParamType1);
	    break;
	default:
	    CagdCrvDomain(Crv, &MinDomain, &MaxDomain);

	    /* Translate to 0.0, scale, and translate to new location. */
	    BspKnotAffineTrans(KV, Len, -MinDomain, 1.0);
	    BspKnotScale(KV, Len, (*TMax - *TMin) / (MaxDomain - MinDomain));
	    BspKnotAffineTrans(KV, Len, *TMin, 1.0);
	    break;
    }

    CrvObj = IPGenCRVObject(Crv);

    return CrvObj;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Affine reparametrization, effectively changing the domain of the surface.  M
*                                                                            *
* PARAMETERS:                                                                M
*   Obj:         A surface to change its parametric domain.                  M
*   RDir:        Direction of reparametrization. Either U or V.		     M
*   TMin, TMax:  New domain for Obj.					     M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:  A surface identical to Obj but with parametric        M
*                      domain from TMin to TMax in direction RDir.           M
*                                                                            *
* KEYWORDS:                                                                  M
*   SrfReparametrization                                                     M
*****************************************************************************/
IPObjectStruct *SrfReparametrization(IPObjectStruct *Obj,
				     RealType *RDir,
				     RealType *TMin,
				     RealType *TMax)
{
    IPObjectStruct
	*SrfObj = NULL;
    CagdSrfDirType
	Dir = (CagdSrfDirType) REAL_PTR_TO_INT(RDir);
    int ParamType1 = REAL_PTR_TO_INT(TMin),
	ParamType2 = REAL_PTR_TO_INT(TMax);

    if (IP_IS_SRF_OBJ(Obj)) {
        CagdSrfStruct
	    *Srf = CAGD_IS_BEZIER_SRF(Obj -> U.Srfs) ?
			  CnvrtBezier2BsplineSrf(Obj -> U.Srfs) :
			  CagdSrfCopy(Obj -> U.Srfs);

	switch (ParamType1) {
	    case CAGD_NIELSON_FOLEY_PARAM:
	    case CAGD_CHORD_LEN_PARAM:
	    case CAGD_CENTRIPETAL_PARAM:
	    case CAGD_UNIFORM_PARAM:
		if (ParamType1 != ParamType2) {
		    IRIT_NON_FATAL_ERROR("SREPARAM: Wrong parametrization spec.");
		    return NULL;
	        }
		BspReparameterizeSrf(Srf, Dir, ParamType1);
		break;
	    default:
		switch (Dir) {
		    case CAGD_CONST_U_DIR:
			BspKnotAffineTransOrder2(Srf -> UKnotVector,
						 Srf -> UOrder,
						 CAGD_SRF_UPT_LST_LEN(Srf) +
								 Srf -> UOrder,
						 *TMin, *TMax);
		        break;
		    case CAGD_CONST_V_DIR:
			BspKnotAffineTransOrder2(Srf -> VKnotVector,
						 Srf -> VOrder,
						 CAGD_SRF_VPT_LST_LEN(Srf) +
								 Srf -> VOrder,
						 *TMin, *TMax);
		        break;
		    default:
			IRIT_NON_FATAL_ERROR("SREPARAM: Wrong parametric direction.");
			return NULL;
		}
		break;
	}

	SrfObj = IPGenSRFObject(Srf);
    }
    else if (IP_IS_TRIMSRF_OBJ(Obj)) {
        CagdRType UMin, UMax, VMin, VMax;
        TrimSrfStruct
	    *TrimSrf = Obj -> U.TrimSrfs;

	switch (ParamType1) {
	    case CAGD_NIELSON_FOLEY_PARAM:
	    case CAGD_CHORD_LEN_PARAM:
	    case CAGD_CENTRIPETAL_PARAM:
	    case CAGD_UNIFORM_PARAM:
		if (ParamType1 != ParamType2) {
		    IRIT_NON_FATAL_ERROR("SREPARAM: Wrong parametrization spec.");
		    return NULL;
	        }
		TrimSrf = TrimSrfCopy(TrimSrf);
		BspReparameterizeSrf(TrimSrf -> Srf, Dir, ParamType1);
		break;
	    default:
		TrimSrfDomain(TrimSrf, &UMin, &UMax, &VMin, &VMax);
	
		switch (Dir) {
		    case CAGD_CONST_U_DIR:
			TrimSrf = TrimAffineTransTrimSrf(TrimSrf,
						    *TMin, *TMax, VMin, VMax);
		        break;
		    case CAGD_CONST_V_DIR:
			TrimSrf = TrimAffineTransTrimSrf(TrimSrf,
						    UMin, UMax, *TMin, *TMax);
		        break;
		}
		break;
	}

	SrfObj = IPGenTRIMSRFObject(TrimSrf);
    }

    return SrfObj;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Affine reparametrization, effectively changing the domain of the trivar.   M
*                                                                            *
* PARAMETERS:                                                                M
*   Obj:         A trivariate to change its parametric domain.               M
*   RDir:        Direction of reparametrization. Either U, V or W.	     M
*   TMin, TMax:  New domain for Obj.					     M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:  A trivar identical to Obj but with parametric         M
*                      domain from TMin to TMax in direction RDir.           M
*                                                                            *
* KEYWORDS:                                                                  M
*   TrivReparametrization                                                    M
*****************************************************************************/
IPObjectStruct *TrivReparametrization(IPObjectStruct *Obj,
				      RealType *RDir,
				      RealType *TMin,
				      RealType *TMax)
{
    IPObjectStruct *TVObj;
    TrivTVDirType
	Dir = (TrivTVDirType) REAL_PTR_TO_INT(RDir);
    TrivTVStruct
	*TV = TRIV_IS_BEZIER_TV(Obj -> U.Trivars) ?
			  TrivCnvrtBezier2BsplineTV(Obj -> U.Trivars) :
			  TrivTVCopy(Obj -> U.Trivars);
    int Len;
    CagdRType *KV, MinDomain, MaxDomain, MinUDomain, MaxUDomain,
	MinVDomain, MaxVDomain, MinWDomain, MaxWDomain;

    TrivTVDomain(TV, &MinUDomain, &MaxUDomain,
		     &MinVDomain, &MaxVDomain,
		     &MinWDomain, &MaxWDomain);

    switch (Dir) {
        case TRIV_CONST_U_DIR:
            Len = TRIV_TV_UPT_LST_LEN(TV) + TV -> UOrder;
	    KV = TV -> UKnotVector;
	    MinDomain = MinUDomain;
	    MaxDomain = MaxUDomain;
	    break;
        case TRIV_CONST_V_DIR:
            Len = TRIV_TV_VPT_LST_LEN(TV) + TV -> VOrder;
	    KV = TV -> VKnotVector;
	    MinDomain = MinVDomain;
	    MaxDomain = MaxVDomain;
	    break;
        case TRIV_CONST_W_DIR:
            Len = TRIV_TV_WPT_LST_LEN(TV) + TV -> WOrder;
	    KV = TV -> WKnotVector;
	    MinDomain = MinWDomain;
	    MaxDomain = MaxWDomain;
	    break;
	default:
	    IRIT_NON_FATAL_ERROR("TREPARAM: Wrong parametric direction.");
	    return NULL;
    }

    /* Translate to 0.0, scale, and translate to new location. */
    BspKnotAffineTrans(KV, Len, -MinDomain, 1.0);
    BspKnotScale(KV, Len, (*TMax - *TMin) / (MaxDomain - MinDomain));
    BspKnotAffineTrans(KV, Len, *TMin, 1.0);

    TVObj = IPGenTRIVARObject(TV);

    return TVObj;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Affine reparametrization, effectively changing the domain of the multivar. M
*                                                                            *
* PARAMETERS:                                                                M
*   Obj:         A multivariate to change its parametric domain.             M
*   RDir:        Direction of reparametrization. 			     M
*   TMin, TMax:  New domain for Obj.					     M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:  A multivariate identical to Obj but with parametric   M
*                      domain from TMin to TMax in direction RDir.           M
*                                                                            *
* KEYWORDS:                                                                  M
*   MvarReparametrization                                                    M
*****************************************************************************/
IPObjectStruct *MvarReparametrization(IPObjectStruct *Obj,
				      RealType *RDir,
				      RealType *TMin,
				      RealType *TMax)
{
    IPObjectStruct *MVObj;
    MvarMVDirType
	Dir = (MvarMVDirType) REAL_PTR_TO_INT(RDir);
    MvarMVStruct
	*MV = MVAR_IS_BEZIER_MV(Obj -> U.MultiVars) ?
			  MvarCnvrtBezier2BsplineMV(Obj -> U.MultiVars) :
			  MvarMVCopy(Obj -> U.MultiVars);
    int Len;
    CagdRType *KV, MinDomain, MaxDomain;

    if (Dir < 0 || Dir >= MV -> Dim) {
	IRIT_WNDW_PUT_STR("Direction is out of domain");
	return NULL;
    }

    MvarMVDomain(MV, &MinDomain, &MaxDomain, Dir);
    Len = MV -> Lengths[Dir] + MV -> Orders[Dir];
    KV = MV -> KnotVectors[Dir];

    /* Translate to 0.0, scale, and translate to new location. */
    BspKnotAffineTrans(KV, Len, -MinDomain, 1.0);
    BspKnotScale(KV, Len, (*TMax - *TMin) / (MaxDomain - MinDomain));
    BspKnotAffineTrans(KV, Len, *TMin, 1.0);

    MVObj = IPGenMULTIVARObject(MV);

    return MVObj;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Computes curvature properties of the given curve.			     M
*                                                                            *
* PARAMETERS:                                                                M
*   PObj:       Curve to evaluate its curvature properties.                  M
*   Eps:        Accuracy of computation.                                     M
*   Operation:  Action to take:		                                     M
*		1 - to compute the curvature square field as the Y axis of   M
*		    a 2D curve (X axis being the original parametrization).  M
*		2 - to compute the parameter values of the inflection points.M
*		3 - to subdivide the curve at the inflection points.	     M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:  Either extremum curvature locations if Eps > 0, or    M
*                      curvature field square curve if Eps < 0.              M
*                                                                            *
* KEYWORDS:                                                                  M
*   CrvCurvaturePts                                                          M
*****************************************************************************/
IPObjectStruct *CrvCurvaturePts(IPObjectStruct *PObj,
				RealType *Eps,
				 RealType *Operation)
{ 
    IPObjectStruct *NewPObj;

    if (CAGD_NUM_OF_PT_COORD(PObj -> U.Crvs -> PType) < 2 ||
	CAGD_NUM_OF_PT_COORD(PObj -> U.Crvs -> PType) > 3) {
	IRIT_NON_FATAL_ERROR(
	    "CCRVTR: Only 2 or 3 dimensional curves are supported.");
	return NULL;
    }

    if (REAL_PTR_TO_INT(Operation) == 1) {
	CagdRType TMin, TMax;
	CagdCrvStruct *CrvtrCrv2D,
	    *CrvtrCrv = SymbCrv3DCurvatureSqr(PObj -> U.Crvs);

	CagdCrvDomain(PObj -> U.Crvs, &TMin, &TMax);
	CrvtrCrv2D = SymbPrmtSclrCrvTo2D(CrvtrCrv, TMin, TMax);
	CagdCrvFree(CrvtrCrv);

	NewPObj = IPGenCRVObject(CrvtrCrv2D);
    }
    else if (REAL_PTR_TO_INT(Operation) == 2) {
	int i;
	CagdPtStruct *IPtsTmp,
	    *IPts = SymbCrvExtremCrvtrPts(PObj -> U.Crvs, *Eps);

	NewPObj = IPGenLISTObject(NULL);

	for (IPtsTmp = IPts, i = 0;
	     IPtsTmp != NULL;
	     IPtsTmp = IPtsTmp -> Pnext, i++) {
	    IPListObjectInsert(NewPObj, i,
			       IPGenNUMValObject(IPtsTmp -> Pt[0]));
	}

	CagdPtFreeList(IPts);

	IPListObjectInsert(NewPObj, i, NULL);
    }
    else { /* Operation == 3. */
	CagdPtStruct
	    *IPts = SymbCrvExtremCrvtrPts(PObj -> U.Crvs, *Eps);
	CagdCrvStruct
	    *Crvs = CagdCrvSubdivAtParams(PObj -> U.Crvs, IPts, *Eps * 10);

	CagdPtFreeList(IPts);

	NewPObj = IPLnkListToListObject(Crvs, IP_OBJ_CURVE);
    }

    return NewPObj;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Compute the inflection points of a curve.				     M
*                                                                            *
* PARAMETERS:                                                                M
*   PObj:       Curve to compute inflection points for.                      M
*   Eps:        Accuracy of computation.                                     M
*   Operation:  Action to take:		                                     M
*		1 - to compute the curve whose zero set is the inflection    M
*		    points set.						     M
*		2 - to compute the parameter values of the inflection points.M
*		3 - to subdivide the curve at the inflection points.	     M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:  Either inflection locations if Operation == 1,	     M
*                      curvature field sign curve if Operation == 2, or      M
*		       curve segments with no interior inflection points if  M
*		       Operation == 3.					     M
*                                                                            *
* KEYWORDS:                                                                  M
*   CrvInflectionPts                                                         M
*****************************************************************************/
IPObjectStruct *CrvInflectionPts(IPObjectStruct *PObj,
				 RealType *Eps,
				 RealType *Operation)
{
    IPObjectStruct *NewPObj;

    if (REAL_PTR_TO_INT(Operation) == 1) {
	CagdRType TMin, TMax;
	CagdCrvStruct *InflectCrv2D,
	    *InflectCrv = SymbCrv2DCurvatureSign(PObj -> U.Crvs);

	CagdCrvDomain(PObj -> U.Crvs, &TMin, &TMax);
	InflectCrv2D = SymbPrmtSclrCrvTo2D(InflectCrv, TMin, TMax);
	CagdCrvFree(InflectCrv);

	NewPObj = IPGenCRVObject(InflectCrv2D);
    }
    else if (REAL_PTR_TO_INT(Operation) == 2) {
	int i;
	CagdPtStruct *IPtsTmp,
	    *IPts = SymbCrv2DInflectionPts(PObj -> U.Crvs, *Eps);

	NewPObj = IPGenLISTObject(NULL);

	for (IPtsTmp = IPts, i = 0;
	     IPtsTmp != NULL;
	     IPtsTmp = IPtsTmp -> Pnext, i++) {
	    IPListObjectInsert(NewPObj, i,
			       IPGenNUMValObject(IPtsTmp -> Pt[0]));
	}

	CagdPtFreeList(IPts);

	IPListObjectInsert(NewPObj, i, NULL);
    }
    else { /* Operation == 3. */
	CagdPtStruct
	    *IPts = SymbCrv2DInflectionPts(PObj -> U.Crvs, *Eps);
	CagdCrvStruct
	    *Crvs = CagdCrvSubdivAtParams(PObj -> U.Crvs, IPts, *Eps * 10);

	CagdPtFreeList(IPts);

	NewPObj = IPLnkListToListObject(Crvs, IP_OBJ_CURVE);
    }

    return NewPObj;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Computes curvature properties of the given curve.  If the given curve is a M
* planar vector curve, its signed curvature is evaluated.  Otherwise, if the M
* given curve is scalar, it is assumed to be a signed curvature field and a  M
* planar curve is reconstructed from it.				     M
*                                                                            *
* PARAMETERS:                                                                M
*   PObj:      Curve to evaluate its curvature properties, or evaluate a crv M
*	       from curvature signatures.		                     M
*   RSamples:  Samples in case of Curve->Crvtr, accuracy of computation in   M
*	       case of Crvtr->Curve.                                         M
*   ROrder:    Order of reconstructed approximation.                         M
*   RArcLenPeriodic:   If sampling curvature field of a curve, parameterize  M
*	       the samples according to arc-length, if TRUE.  If a curve is  M
*	       reconstructed from a curvature field, a closed curve is made  M
*	       (periodic) if TRUE.				             M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:  Either extremum curvature locations if Eps > 0, or    M
*                      curvature field square curve if Eps < 0.              M
*                                                                            *
* KEYWORDS:                                                                  M
*   CrvCurvatureFunction                                                     M
*****************************************************************************/
IPObjectStruct *CrvCurvatureFunction(IPObjectStruct *PObj,
				     RealType *RSamples,
				     RealType *ROrder,
				     RealType *RArcLenPeriodic)
{
    CagdCrvStruct *RetCrv,
	*Crv = PObj -> U.Crvs;

    if (CAGD_NUM_OF_PT_COORD(Crv -> PType) == 1) {
	RetCrv = SymbSignedCrvtrGenCrv(Crv,
				       *RSamples,
				       REAL_PTR_TO_INT(ROrder),
				       REAL_PTR_TO_INT(RArcLenPeriodic));
    }
    else
        RetCrv = SymbCrvGenSignedCrvtr(Crv,
				       REAL_PTR_TO_INT(RSamples),
				       REAL_PTR_TO_INT(ROrder),
				       REAL_PTR_TO_INT(RArcLenPeriodic));

    return IPGenCRVObject(RetCrv);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Computes curvature properties of the given surface.			     M
*                                                                            *
* PARAMETERS:                                                                M
*   PObj:     Surface to compute curvature properties for.                   M
*   RPtType:  Point type of output.                                          M
*   RDir:     Either curvature in U or V or total upper bound.               M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:   A curvature bound field for surface PObj.            M
*                                                                            *
* KEYWORDS:                                                                  M
*   SrfCurvatureBounds                                                       M
*****************************************************************************/
IPObjectStruct *SrfCurvatureBounds(IPObjectStruct *PObj,
				   RealType *RPtType,
				   RealType *RDir)
{ 
    IPObjectStruct *NewPObj;
    CagdSrfStruct *TSrf, *CrvtrSrfBound;
    CagdPointType
	PtType = (CagdPointType) REAL_PTR_TO_INT(RPtType);
    CagdSrfDirType
	Dir = (CagdSrfDirType)  REAL_PTR_TO_INT(RDir);

    if (CAGD_NUM_OF_PT_COORD(PObj -> U.Srfs -> PType) < 2 ||
	CAGD_NUM_OF_PT_COORD(PObj -> U.Srfs -> PType) > 3) {
	IRIT_NON_FATAL_ERROR(
	    "SCRVTR: Only 2 or 3 dimensional curves are supported.");
	return NULL;
    }

    switch (Dir) {
	case CAGD_CONST_U_DIR:
	case CAGD_CONST_V_DIR:
	    CrvtrSrfBound = SymbSrfIsoDirNormalCurvatureBound(PObj -> U.Srfs,
									Dir);
	    break;
	default:
	    CrvtrSrfBound = SymbSrfCurvatureUpperBound(PObj -> U.Srfs);
	    break;
    }

    switch (PtType) {
	case CAGD_PT_P1_TYPE:
	    break;
	case CAGD_PT_E1_TYPE:
	case CAGD_PT_E3_TYPE:
	case CAGD_PT_P3_TYPE:
	    TSrf = CagdCoerceSrfTo(CrvtrSrfBound, PtType);
	    CagdSrfFree(CrvtrSrfBound);
	    CrvtrSrfBound = TSrf;
	    break;
	default:
	    CagdSrfFree(CrvtrSrfBound);
	    IRIT_NON_FATAL_ERROR("SCRVTR: Wrong point type coercion.");
	    return NULL;
    }

    NewPObj = IPGenSRFObject(CrvtrSrfBound);

    return NewPObj;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Computes radial curvatures linesfor the given surface and direction.	     M
*                                                                            *
* PARAMETERS:                                                                M
*   PObj:                 Surface to compute radial curvature for.	     M
*   ViewDir:		  Direction from which to compute the radial crvtr.  M
*   SubdivTol, NumerTol:  Tolerances of computation.	                     M
*   MergeTol:		  Tolerance of point to polyline merge, negative for M
*			  performing no merge into polyline.		     M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:   A curvature bound field for surface PObj.            M
*                                                                            *
* KEYWORDS:                                                                  M
*   SrfRadialCurvature                                                       M
*****************************************************************************/
IPObjectStruct *SrfRadialCurvature(IPObjectStruct *PObj,
				   VectorType ViewDir,
				   RealType *SubdivTol,
				   RealType *NumerTol,
				   RealType *MergeTol)
{ 
    IPObjectStruct *ObjPtList;
    MvarPtStruct
	*MVPts = MvarSrfRadialCurvature(PObj -> U.Srfs, ViewDir,
					*SubdivTol, *NumerTol);

    if (MVPts == NULL)
	return NULL;

    ObjPtList = ConvertMVPtsToCtlPts(MVPts, *MergeTol);

    MvarPtFreeList(MVPts);

    return ObjPtList;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Computes U/V-extreme implicit curve locations.                             M
*                                                                            *
* PARAMETERS:                                                                M
*   PSrf:             Surface to compute radial curvature for.		     M
*   Dir:	      U/V direction from which to compute the extreme pts.   M
*   SubdivTol, NumerTol:  Tolerances of computation.	                     M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:   Extreme locations along U or Vurface PObj, in the    M
*			parametric space of the surface.	             M
*                                                                            *
* KEYWORDS:                                                                  M
*   ImplicitCrvExtreme                                                       M
*****************************************************************************/
IPObjectStruct *ImplicitCrvExtreme(IPObjectStruct *PSrf,
				   RealType *Dir,
				   RealType *SubdivTol,
				   RealType *NumerTol)
{ 
    IPObjectStruct *ObjPtList;
    MvarPtStruct
	*MVPts = MvarImplicitCrvExtreme(PSrf -> U.Srfs, REAL_PTR_TO_INT(Dir),
					*SubdivTol, *NumerTol);

    if (MVPts == NULL)
	return NULL;

    ObjPtList = ConvertMVPtsToCtlPts(MVPts, -1.0);

    MvarPtFreeList(MVPts);

    return ObjPtList;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Computes the Gauss curvature surface of the given surface.		     M
*                                                                            *
* PARAMETERS:                                                                M
*   PObj:      Surface to compute its gauss curvature surface for.           M
*   NumerOnly: If TRUE, only the numerator component of K is returned.       M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:   A Gauss curvature surface of PObj.                   M
*                                                                            *
* KEYWORDS:                                                                  M
*   SrfGaussCurvature                                                        M
*****************************************************************************/
IPObjectStruct *SrfGaussCurvature(IPObjectStruct *PObj, RealType *NumerOnly)
{ 
    IPObjectStruct *NewPObj;
    CagdSrfStruct *GaussSrf;

    if (CAGD_NUM_OF_PT_COORD(PObj -> U.Srfs -> PType) < 2 ||
	CAGD_NUM_OF_PT_COORD(PObj -> U.Srfs -> PType) > 3) {
	IRIT_NON_FATAL_ERROR(
	    "SGauss: Only 2 or 3 dimensional surfaces are supported.");
	return NULL;
    }
    
    GaussSrf = SymbSrfGaussCurvature(PObj -> U.Srfs,
				     REAL_PTR_TO_INT(NumerOnly));

    NewPObj = IPGenSRFObject(GaussSrf);

    return NewPObj;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Computes the Mean curvature square surface of the given surface.	     M
*                                                                            *
* PARAMETERS:                                                                M
*   PObj:      Surface to compute its mean curvature square surface for.     M
*   NumerOnly: If TRUE, computes only the (rational) numerator of the mean   M
*	       curvature (note it is not the square of the mean curvature).  M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:   A Mean curvature square surface of PObj.             M
*                                                                            *
* KEYWORDS:                                                                  M
*   SrfMeanCurvature                                                         M
*****************************************************************************/
IPObjectStruct *SrfMeanCurvature(IPObjectStruct *PObj, RealType *NumerOnly)
{ 
    IPObjectStruct *NewPObj;
    CagdSrfStruct *MeanSrf;

    if (CAGD_NUM_OF_PT_COORD(PObj -> U.Srfs -> PType) < 2 ||
	CAGD_NUM_OF_PT_COORD(PObj -> U.Srfs -> PType) > 3) {
	IRIT_NON_FATAL_ERROR(
	    "SMean: Only 2 or 3 dimensional surfaces are supported.");
	return NULL;
    }

    if (APX_EQ(REAL_PTR_TO_INT(NumerOnly), 0.0))
	MeanSrf = SymbSrfMeanCurvatureSqr(PObj -> U.Srfs);
    else
        MeanSrf = SymbSrfMeanNumer(PObj -> U.Srfs);

    NewPObj = IPGenSRFObject(MeanSrf);

    return NewPObj;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Computes the isoparametric focal surface of the given surface.	     M
*                                                                            *
* PARAMETERS:                                                                M
*   PObj:     Surface to compute its focal surface using isoparametric	     M
*	      normal curvature direction.				     M
*   RDir:     Either iso focal surface in U or V.		             M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:   An iso focal surface of surface PObj.                M
*                                                                            *
* KEYWORDS:                                                                  M
*   SrfIsoFocalSrf                                                        M
*****************************************************************************/
IPObjectStruct *SrfIsoFocalSrf(IPObjectStruct *PObj, RealType *RDir)
{ 
    IPObjectStruct *NewPObj;
    CagdSrfStruct *IsoFocalSrf;
    CagdSrfDirType
	Dir = (CagdSrfDirType) REAL_PTR_TO_INT(RDir);

    if (CAGD_NUM_OF_PT_COORD(PObj -> U.Srfs -> PType) < 2 ||
	CAGD_NUM_OF_PT_COORD(PObj -> U.Srfs -> PType) > 3) {
	IRIT_NON_FATAL_ERROR(
	    "IsoFocal: Only 2 or 3 dimensional surfaces are supported.");
	return NULL;
    }

    IsoFocalSrf = SymbSrfIsoFocalSrf(PObj -> U.Srfs, Dir);

    NewPObj = IPGenSRFObject(IsoFocalSrf);

    return NewPObj;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Computes the (mean) evolute curve/surface of the given curve/surface.	     M
*                                                                            *
* PARAMETERS:                                                                M
*   PObj:     Curve/Surface to compute its (mean) evolute for.               M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:   A (mean) evolute curve/surface of PObj.              M
*                                                                            *
* KEYWORDS:                                                                  M
*   FreeformEvolute                                                          M
*****************************************************************************/
IPObjectStruct *FreeformEvolute(IPObjectStruct *PObj)
{ 
    IPObjectStruct *NewPObj;

    if (IP_IS_CRV_OBJ(PObj)) {
	CagdCrvStruct *EvoluteCrv, *EvoluteCrvAux;

	if (CAGD_NUM_OF_PT_COORD(PObj -> U.Crvs -> PType) < 2 ||
	    CAGD_NUM_OF_PT_COORD(PObj -> U.Crvs -> PType) > 3) {
	    IRIT_NON_FATAL_ERROR(
		"EVOLUTE: Only 2 or 3 dimensional curves are supported.");
	    return NULL;
	}

	EvoluteCrvAux = SymbCrv3DRadiusNormal(PObj -> U.Crvs);
	EvoluteCrv = SymbCrvAdd(PObj -> U.Crvs, EvoluteCrvAux);
	CagdCrvFree(EvoluteCrvAux);

	NewPObj = IPGenCRVObject(EvoluteCrv);
    }
    else if (IP_IS_SRF_OBJ(PObj)) {
	CagdSrfStruct *EvoluteSrf, *EvoluteSrfAux;

	if (CAGD_NUM_OF_PT_COORD(PObj -> U.Srfs -> PType) < 2 ||
	    CAGD_NUM_OF_PT_COORD(PObj -> U.Srfs -> PType) > 3) {
	    IRIT_NON_FATAL_ERROR(
		"EVOLUTE: Only 2 or 3 dimensional surfaces are supported.");
	    return NULL;
	}

	EvoluteSrfAux = SymbSrfMeanEvolute(PObj -> U.Srfs);
	EvoluteSrf = SymbSrfAdd(PObj -> U.Srfs, EvoluteSrfAux);
	CagdSrfFree(EvoluteSrfAux);

	NewPObj = IPGenSRFObject(EvoluteSrf);
    }
    else
	NewPObj = NULL;

    return NewPObj;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Merges two surfaces into one in specified Dir and SameEdge flag. 	     M
*                                                                            *
* PARAMETERS:                                                                M
*   Srf1, Srf2:  Two surfaces to merge.                                      M
*   Dir:         Direction of merge. Either U or V.                          M
*   SameEdge:    Do Srf1 and Srf2 share a common edge?                       M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:  A surface result of the merge.                        M
*                                                                            *
* KEYWORDS:                                                                  M
*   MergeSrfSrf                                                              M
*****************************************************************************/
IPObjectStruct *MergeSrfSrf(IPObjectStruct *Srf1,
			    IPObjectStruct *Srf2,
			    RealType *Dir,
			    RealType *SameEdge)
{
    IPObjectStruct *SrfObj;
    CagdSrfStruct
	*Srf = CagdMergeSrfSrf(Srf1 -> U.Srfs, Srf2 -> U.Srfs,
			       (CagdSrfDirType) REAL_PTR_TO_INT(Dir),
			       REAL_PTR_TO_INT(SameEdge), TRUE);

    SrfObj = IPGenSRFObject(Srf);

    return SrfObj;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Merge two curves/ctl points into one curve by adding a linear segment      M
* between the first end point to second start point.			     M
*                                                                            *
* PARAMETERS:                                                                M
*   PObj1, PObj2:  Either curve or a control point to merge.                 M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:   Merged curve.                                        M
*                                                                            *
* KEYWORDS:                                                                  M
*   MergeCurvesAndCtlPoints                                                  M
*****************************************************************************/
IPObjectStruct *MergeCurvesAndCtlPoints(IPObjectStruct *PObj1,
					IPObjectStruct *PObj2)
{
    IPObjectStruct *CrvObj;
    CagdCrvStruct
	*Crv = NULL;
    CagdPtStruct Pt1, Pt2;

    if (IP_IS_CRV_OBJ(PObj1)) {
	if (IP_IS_CRV_OBJ(PObj2)) {
	    Crv = CagdMergeCrvCrv(PObj1 -> U.Crvs, PObj2 -> U.Crvs, TRUE);
	}
	else if (IP_IS_CTLPT_OBJ(PObj2)) {
	    CagdRType
		*Coords2 = PObj2 -> U.CtlPt.Coords;

	    CagdCoerceToE3(Pt2.Pt, &Coords2, -1, PObj2 -> U.CtlPt.PtType);
	    Crv = CagdMergeCrvPt(PObj1 -> U.Crvs, &Pt2);
	}
	else
	    IRIT_FATAL_ERROR("Curve/CtlPt was expected.");
    }
    else if (IP_IS_CTLPT_OBJ(PObj1)) {
	CagdRType
	    *Coords1 = PObj1 -> U.CtlPt.Coords;

	CagdCoerceToE3(Pt1.Pt, &Coords1, -1, PObj1 -> U.CtlPt.PtType);

	if (IP_IS_CRV_OBJ(PObj2)) {
	    Crv = CagdMergePtCrv(&Pt1, PObj2 -> U.Crvs);
	}
	else if (IP_IS_CTLPT_OBJ(PObj2)) {
	    CagdRType
		*Coords2 = PObj2 -> U.CtlPt.Coords;

	    CagdCoerceToE3(Pt2.Pt, &Coords2, -1, PObj2 -> U.CtlPt.PtType);
	    Crv = CagdMergePtPt(&Pt1, &Pt2);
	}
	else
	    IRIT_FATAL_ERROR("Curve/CtlPt was expected.");
    }
    else
	IRIT_FATAL_ERROR("Curve/CtlPt was expected.");

    if (Crv == NULL)
	return NULL;

    CrvObj = IPGenCRVObject(Crv);

    return CrvObj;
}
