/*****************************************************************************
*  Open GL using CG (programmable hardware) drawing functions.		     *
******************************************************************************
* (C) Gershon Elber, Technion, Israel Institute of Technology                *
******************************************************************************
* Written by:  Eran Karpen & Sagi Schein	     Ver 0.1, May 2005.	     *
*****************************************************************************/

#include <stdio.h>
#include <windows.h>
#include <GL/gl.h>
#include <GL/glu.h>

#include "iritprsr.h"
#include "allocate.h"
#include "grap_loc.h"
#include "triv_lib.h"
#include "geom_lib.h"
#include "wntdrvs.h"
#include "editmanp.h"

#ifdef HAVE_OGL_CG_LIB

#include <cg/cg.h>
#include <cg/cgGL.h>
/* This needs to be include into the CG SDK where the NV extensions reside */
#include <gl/glext.h>
#include <gl/wglext.h>

#include "opngl_cg.h"

STATIC_DATA int GlblRenderTrivState;

#define PBUFFER_SIZE (1024)
#define POINT_SIZE (1)

/* Static data for CG shaders. */
enum { FIRST = 1, LAST = 2, OTHER = 4};
enum { NONE, NORMALOFFSET, NORMALTRIG };

STATIC_DATA int VertexCount;
STATIC_DATA CGcontext 
    Context = NULL;
STATIC_DATA CGprogram
    FfdVertexProgram = NULL,
    FfdFragmentProgram = NULL,
    FfdVertexProgramPhase1 = NULL,
    FfdVertexProgramPhase2 = NULL,
    FfdFragmentProgramPhase2 = NULL,
    DmFfdFragProgram = NULL;

typedef struct CGparams {
    CGparameter  ModelViewProjMatrix; 
    CGparameter  ModelViewMatrix; 
    CGparameter  Position;
    CGparameter  Color;
    CGparameter  Normal;
    CGparameter  TexCoord;
    CGparameter  TexDim; 
    CGparameter  TrivDim;
    CGparameter  UBases;
    CGparameter  VBases;
    CGparameter  WBases;
    CGparameter  TvSpan;
    CGparameter  Range;
    CGparameter  MinLimit;
    CGparameter  AnimVal;
    CGparameter  TexObject; 
    CGparameter  AmbientFactor;
    CGparameter  DiffuseFactor; 
    CGparameter  SpecularFactor; 
    CGparameter  Shininess;
    CGparameter  LightPos0;
    CGparameter  LightPos1;
    CGparameter  FlippedNormals;
    CGparameter  NormalMethod;
    CGparameter  Texture;
} CGparamsStruct;

typedef struct {
    CGparameter TargetCoord;
    CGparameter SourceCoord;
    CGparameter SourcePdCoord;
    CGparameter TexPosObj;
    CGparameter TexIdxObj;
} CGDmParamStruct; 

typedef struct PBuffer {
    HPBUFFERARB HBuffer;
    HGLRC PBufferRC;
    HDC PBufferDC;
    HGLRC OrigRC;
    HDC OrigDC;
    int Width;
    int Height;
    GLuint PBufferTexId;
} PBufferStruct;

CGDmParamStruct CGParamDmFfd;

STATIC_DATA CGparamsStruct
    CGParamSinglePhase,
    CGParamDoublePhase1,
    CGParamDoublePhase2,
    CGParamDoublePhase2Fragment;

STATIC_DATA CGprofile 
    VertexProfile, 
    FragmentProfile;
STATIC_DATA int 
    FfdShaderType,
    FfdTexDimU,
    FfdTexDimV,
    MinParam[3],
    RangeParam[3];

PBufferStruct PBuffer[2];

STATIC_DATA GLfloat *VertexPhase1Texture = NULL;
STATIC_DATA GLuint VertexPhase1TextureHandle;

STATIC_DATA BBoxType ObjBBox; 
STATIC_DATA TrivTVStruct *Tv;
STATIC_DATA RealType
    AnimVal[3],
    TvSpan[3];
STATIC_DATA int AnimGlobalOn = FALSE,
    NrmlDir = 1,
    Cx = 0,
    Cy = 1, 
    Nx = PBUFFER_SIZE - POINT_SIZE,
    Ny = PBUFFER_SIZE - POINT_SIZE,
    Dx,Dy,
    GlblNormalCalcMethod = 0,
    VertexPhase1TextureSize = 4096,
    PBuffCurrent = -1,
    FirstInList = TRUE,
    ObjColor[4];

STATIC_DATA PFNGLACTIVETEXTUREARBPROC
	glActiveTextureARB = NULL;
STATIC_DATA PFNWGLGETPIXELFORMATATTRIBIVARBPROC 
	wglGetPixelFormatAttribivARB = NULL;
STATIC_DATA PFNWGLGETPIXELFORMATATTRIBFVARBPROC 
	wglGetPixelFormatAttribfvARB = NULL;
STATIC_DATA PFNWGLCHOOSEPIXELFORMATARBPROC 
	wglChoosePixelFormatARB = NULL;
STATIC_DATA PFNWGLCREATEPBUFFERARBPROC 
	wglCreatePbufferARB = NULL;
STATIC_DATA PFNWGLGETPBUFFERDCARBPROC 
	wglGetPbufferDCARB = NULL;
STATIC_DATA PFNWGLRELEASEPBUFFERDCARBPROC 
	wglReleasePbufferDCARB = NULL;
STATIC_DATA PFNWGLDESTROYPBUFFERARBPROC
	wglDestroyPbufferARB = NULL;
STATIC_DATA PFNWGLQUERYPBUFFERARBPROC 
	wglQueryPbufferARB = NULL;
STATIC_DATA PFNWGLBINDTEXIMAGEARBPROC 
	wglBindTexImageARB = NULL;
STATIC_DATA PFNWGLRELEASETEXIMAGEARBPROC 
	wglReleaseTexImageARB = NULL;
STATIC_DATA PFNWGLSETPBUFFERATTRIBARBPROC
	wglSetPbufferAttribARB = NULL;
STATIC_DATA PFNGLMULTITEXCOORD3FARBPROC 
	glMultiTexCoord3f = NULL;
STATIC_DATA PFNGLMULTITEXCOORD2FARBPROC 
	glMultiTexCoord2f = NULL;

static int IGLoadShaders(void);
static int IrtParseFfdTextureString(char *DTexture,
				    char *FName,
				    int* ShaderType,
				    RealType TvSpan[3],
				    int * TrivRenderState,
				    int * AnimSamples,
				    PointType * ConstOffset,
				    int *NormalMethod);

static int ApplyDeformedObjectRender(IPObjectStruct * PObj);
static void IGGetAnimationFactors(IPObjectStruct *PObj, RealType t);
static int LoadFFDShadersSinglePhase(void);
static int LoadFFDShadersDoublePhase(void);
static int RenderShaderSinglePhase(IPObjectStruct *PObj);
static int RenderShaderDoublePhase(IPObjectStruct *PObj);
static void DrawModelAsGrid(IPObjectStruct * PObj);
static int IsArbExtensionSupported(const char *extension);
static void DrawTexturedQuad();
static void SetGlbObjColor(IPObjectStruct *PObj);
static void UpdateShadingParameters(int algorithm);
static int IGCheckSysStatus(IPObjectStruct *PObj,int ForceRecompute);
static void RecomputeLists(IPObjectStruct *PObj);
static void SetControlPoint(IPObjectStruct *PObj);
static int PreProcessAnimation(IPObjectStruct * PObj);
static void RecomputeCacheGeom(IPObjectStruct * PObj,
			       int FreePolygons,
			       int FreeIsolines,
			       int FreeSketches,
			       int FreeCtlMesh);
static IPObjectStruct * SingleStepAnimation(IPObjectStruct * PObj);
void IncrementInstanceCounter(IPObjectStruct * PObj);
static void ApplyDeformedVertices(IPObjectStruct *PObj,
				  int XLen, 
				  int YLen);

/*****************************************************************************
* DESCRIPTION:                                                               *
*    Draw to the proper CG interface instead of the usual opengl             * 
*    Used for the single phase algorithm                                     *
*									     *
* PARAMETERS:                                                                *
*   Vertex:      Trivariate from which to create texture.                    *
*                                                                            *
* RETURN VALUE:                                                              *
*   int:       TRUE if the a texture object is created, FALSE otherwise.     *
*****************************************************************************/
int IGCgVertexHanlingSinglePhase(IPVertexStruct *V)
{
    char *Str;
    RealType TColor[4], vx, vy, vz, t1, t2, t3;

    if (V->Coord[0] < 0 || V->Coord[0] > 1 ||
	V->Coord[1] < 0 || V->Coord[1] > 1 ||
	V->Coord[2] < 0 || V->Coord[2] > 1) {
	    if (IGGlblMore)
		printf("Error: Model not in unit box\n");
	    return FALSE;
	}

    if ((Str = AttrGetStrAttrib(V -> Attr, "RGB")) != NULL &&
	sscanf(Str, "%lf,%lf,%lf",
	    &TColor[0], &TColor[1], &TColor[2]) == 3) {
        TColor[0] /= 255.0;
        TColor[1] /= 255.0;
        TColor[2] /= 255.0;
	TColor[3]=IGGlblOpacity;
	glColor4f((GLfloat) TColor[0],
		  (GLfloat) TColor[1],
		  (GLfloat) TColor[2],
		  (GLfloat) TColor[3]);
    }
    
    vx = (V -> Coord[0] + AnimVal[0]) * 
	(RangeParam[0] * TvSpan[0]) + MinParam[0];
    vy = (V -> Coord[1] + AnimVal[1]) * 
	(RangeParam[1] * TvSpan[1]) + MinParam[1];
    vz = (V -> Coord[2] + AnimVal[2]) * 
	(RangeParam[2] * TvSpan[2]) + MinParam[2];
    
    t1 = vx - floor(vx);
    glMultiTexCoord3f(0, (GLfloat) (0.5 - t1 + 0.5 * t1 * t1),
		      (GLfloat)(0.5 + t1 - t1 * t1),
		      (GLfloat)(0.5 * t1 * t1));

    t2 = vy - floor(vy);
    glMultiTexCoord3f(1, (GLfloat) (0.5 - t2 + 0.5 * t2 * t2),
		      (GLfloat)(0.5 + t2 - t2 * t2),
		      (GLfloat)(0.5 * t2 * t2));

    t3 = vz - floor(vz);
    glMultiTexCoord3f(2, (GLfloat) (0.5 - t3 + 0.5 * t3 * t3),
		      (GLfloat)(0.5 + t3 - t3 * t3),
		      (GLfloat)(0.5 * t3 * t3));

    glNormal3f(NrmlDir * (GLfloat) V -> Normal[0],
	       NrmlDir * (GLfloat) V -> Normal[1],
	       NrmlDir * (GLfloat) V -> Normal[2]);

    glVertex4f((GLfloat) floor(vx),
	       (GLfloat) floor(vy),
	       (GLfloat) floor(vz), 1);

    return TRUE;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*    Draw to the proper CG interface instead of the usual opengl             * 
*    Used for the first pass in the  double phase algorithm                  *
*	    								     *
* PARAMETERS:                                                                *
*   Vertex:      Trivariate from which to create texture.                    *
*                                                                            *
* RETURN VALUE:                                                              *
*   int:       TRUE if the a texture object is created, FALSE otherwise.     *
*****************************************************************************/
int IGCgVertexHanlingFirstPass(IPVertexStruct * V)
{
    PointType VP1, VP2, T1, T2, T3, P1, P2,
	X = { 1, 0, 0 },
	Y = { 0, 1, 0 },
	Z = { 0, 0, 1 };
    GLfloat Px = ((GLfloat) (Cx)) / ((GLfloat) (Nx));
    GLfloat Py = ((GLfloat) (Cy)) / ((GLfloat) (Ny));

    if (V->Coord[0] < 0 || V -> Coord[0] > 1 ||
	V -> Coord[1] < 0 || V -> Coord[1] > 1 ||
	V -> Coord[2] < 0 || V -> Coord[2] > 1) {
	if (IGGlblMore)
	    printf("Error: Model not in unit box\n");
	return FALSE;
    }

    glTexCoord3f((GLfloat) (V->Coord[0]), 
	(GLfloat) (V->Coord[1]), (GLfloat) (V->Coord[2]));
    glVertex4f(Px, Py, 0, 1);
    AttrSetUVAttrib(&V -> Attr, "xyvals", Px, Py);
    /* Update the global counters of the grid */
    Cx += POINT_SIZE;
    if (Cx >= Nx) {
	Cy += POINT_SIZE;
	Cx = 0;
    }
    VertexCount++;

    if (GlblNormalCalcMethod == 1) {
	GLfloat Offset = 0.5f;
	float epsilon = 0.0001f;
	PT_COPY (P1, V -> Normal);
	PT_SCALE(P1, epsilon);
	PT_ADD(VP1, P1, V -> Coord);
	glTexCoord3f((GLfloat) VP1[0], (GLfloat) VP1[1], (GLfloat) VP1[2]);
	glVertex4f(Px, Py + Offset, 0, 1);
    }    
    /* Calculate Orthogonal Vectors */
    else if (GlblNormalCalcMethod == 2) {
	GLfloat Offset = 0.333f;
	RealType epsilon = 0.001;
	CROSS_PROD(T1, X, V -> Normal);
	CROSS_PROD(T2, Y, V -> Normal);
	CROSS_PROD(T3, Z, V -> Normal);
	if (PT_LENGTH(T1) > PT_LENGTH(T2)) 
	    PT_COPY (P1, T1);
	else PT_COPY (P1, T2);
	if (PT_LENGTH(T3) > PT_LENGTH(P1))
	    PT_COPY (P1, T3);

	PT_NORMALIZE(P1);
	CROSS_PROD(P2, P1, V -> Normal);
	PT_NORMALIZE(P2);
	PT_SCALE(P1, epsilon);
	PT_SCALE(P2, epsilon);
	PT_ADD(VP1, (V -> Coord), P1);
	PT_ADD(VP2, (V -> Coord), P2);

	glTexCoord3f((GLfloat) VP1[0], (GLfloat) VP1[1], (GLfloat) VP1[2]);
	glVertex4f(Px, Py + Offset, 0, 1);
	glTexCoord3f((GLfloat) VP2[0], (GLfloat) VP2[1], (GLfloat) VP2[2]);
	glVertex4f(Px, Py + 2 * Offset, 0, 1);
    }
    return TRUE;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*    Draw to the proper CG interface instead of the usual opengl             * 
*    Used for the single phase algorithm                                     *
*									     *
* PARAMETERS:                                                                *
*   Vertex:      Trivariate from which to create texture.                    *
*                                                                            *
* RETURN VALUE:                                                              *
*   int:       TRUE if the a texture object is created, FALSE otherwise.     *
*****************************************************************************/
int IGCgVertexHandlingSecondPass(IPVertexStruct *V)
{
    char *Str;
    RealType TColor[4];
    float *XY;
    	
    if ((Str = AttrGetStrAttrib(V -> Attr, "RGB")) != NULL &&
		sscanf(Str, "%lf,%lf,%lf",
		&TColor[0], &TColor[1], &TColor[2]) == 3) {
            TColor[0] /= 255.0;
            TColor[1] /= 255.0;
            TColor[2] /= 255.0;
	    TColor[3] = IGGlblOpacity;
	    glColor4f((GLfloat) TColor[0],
	      (GLfloat) TColor[1],
	      (GLfloat) TColor[2],
	      (GLfloat) TColor[3]);
	}

    glNormal3f(NrmlDir* (GLfloat) V -> Normal[0],
                NrmlDir* (GLfloat) V -> Normal[1],
                NrmlDir* (GLfloat) V -> Normal[2]);
    XY = AttrGetUVAttrib(V -> Attr, "xyvals");    
    if (XY)
	glVertex4f(XY[0], XY[1], 0, 1);
    
    return TRUE;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* a trivariate B-spline - assumed to have a uniform knot                     * 
* Stores its control points in a texture. The texture is rounded up to the   *
* nearest power of two sizes - a limitation of the vertex texture HW ?       *
* PARAMETERS:                                                                *
*   PObj:      Trivariate from which to create texture.                      *
*                                                                            *
* RETURN VALUE:                                                              *
*   int:       TRUE if the a texture object is created, FALSE otherwise.     *
*****************************************************************************/
static int IGCreateFFDTexture(IPObjectStruct *DefObj, IPObjectStruct *TvObj)
{
    int k, j, i, 
	TexIdx = 0, 
	RowIdx = 0;
    
    GLfloat* ffdImg, *ffdIdxImg;
    GLuint ffdTexture, ffdIdxTexture;
    TrivTVStruct *tv = TvObj -> U.Trivars;
   
    FfdTexDimU = tv -> WLength * tv -> ULength;
    FfdTexDimV = tv -> VLength;
    for (k = 2; k < FfdTexDimU; k *= 2);
	FfdTexDimU = k;
    for (k = 2; k < FfdTexDimV; k *= 2);
	FfdTexDimV = k;
    
    ffdImg = 
	(GLfloat*) IritMalloc(FfdTexDimV * FfdTexDimU * sizeof(GLfloat) * 4);
    ffdIdxImg = 
	(GLfloat*) IritMalloc(FfdTexDimV * FfdTexDimU * sizeof(GLfloat) * 4);
    VertexPhase1Texture = 
	(GLfloat*) IritMalloc(VertexPhase1TextureSize * sizeof(GLfloat) * 4);

    ZAP_MEM(ffdImg, sizeof(ffdImg));
    ZAP_MEM(ffdIdxImg, sizeof(ffdIdxImg));
    
    for (k = 0; k < tv->WLength; k++) {
        for (j = 0; j < tv -> ULength; j++) {
            for (i = 0; i < tv -> VLength; i++) {
                int idx = TRIV_MESH_UVW(tv, j, i, k);
                if (TexIdx >= FfdTexDimU * FfdTexDimV)
		    break;
              
                *(ffdImg + 4 * TexIdx) = (GLfloat) tv -> Points[1][idx];
                *(ffdImg + 4 * TexIdx+1) = (GLfloat) tv -> Points[2][idx];
                *(ffdImg + 4 * TexIdx+2) = (GLfloat) tv -> Points[3][idx];
                *(ffdImg + 4 * TexIdx+3) = (GLfloat) 1;

		*(ffdIdxImg + 4 * TexIdx) = (GLfloat) j;
                *(ffdIdxImg + 4 * TexIdx + 1) = (GLfloat) i;
                *(ffdIdxImg + 4 * TexIdx + 2) = (GLfloat) k;
                *(ffdIdxImg + 4 * TexIdx + 3) = (GLfloat) 1;
                
                TexIdx++;
                RowIdx++;
                if (RowIdx % tv -> VLength == 0) {
                    TexIdx += (FfdTexDimV - tv -> VLength);
                    RowIdx =0;
                }
               
            }
        }
    }

    for (i = 0; i < VertexPhase1TextureSize*4; i++)
	VertexPhase1Texture[i] = 0;

    for (i = 0; i < VertexPhase1TextureSize; i++) {
	RealType t1 = (RealType) i / (VertexPhase1TextureSize - 1);

	VertexPhase1Texture[4 * i] = (GLfloat) (0.5 - t1 + 0.5 * t1 * t1);
	VertexPhase1Texture[4 * i + 1] = (GLfloat) (0.5 + t1 - t1 * t1);
	VertexPhase1Texture[4 * i + 2] = (GLfloat) (0.5 * t1 * t1);
	VertexPhase1Texture[4 * i + 3] = 0;

    }

    glGenTextures(1, &ffdTexture);
    glBindTexture(GL_TEXTURE_2D, ffdTexture);
    glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA_FLOAT32_ATI, 
	FfdTexDimV,FfdTexDimU, 0, GL_RGBA, GL_FLOAT, ffdImg);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
    IritFree(ffdImg);

   
    glGenTextures(1, &ffdIdxTexture);
    glBindTexture(GL_TEXTURE_2D, ffdIdxTexture);
    glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA_FLOAT32_ATI, 
	FfdTexDimV,FfdTexDimU, 0, GL_RGBA, GL_FLOAT, ffdIdxImg);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
    IritFree(ffdIdxImg);

    glGenTextures(1, &VertexPhase1TextureHandle);
    glBindTexture(GL_TEXTURE_2D, VertexPhase1TextureHandle);
    /* We must use GL_RGBA_FLOAT32_ATI for texture in shaders */
    glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA_FLOAT32_ATI,
		VertexPhase1TextureSize, 1,
		0, GL_RGBA, GL_FLOAT, VertexPhase1Texture);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER,
        GL_NEAREST);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, 
        GL_NEAREST);
    IritFree(VertexPhase1Texture);


    AttrSetObjectIntAttrib(TvObj, "_ffd_tex_id", ffdTexture);
    AttrSetObjectIntAttrib(TvObj, "_ffd_idx_tex_id", ffdIdxTexture);
    AttrSetObjectPtrAttrib(TvObj, "_ffd_def_obj", DefObj); 

   return TRUE;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Checks if machine hardware supports VP40 & FP40 cg profiles.               * 
* Loading the shaders only if there is support.                              * 
*                                                                            *
* PARAMETERS:                                                                *
*   void                                                                     *
*                                                                            *
* RETURN VALUE:                                                              *
*   int:       TRUE if loading was successful, FALSE otherwise.              *
*****************************************************************************/
static int IGLoadShaders(void)
{   
    switch (FfdShaderType) {
        case 0:
            return LoadFFDShadersSinglePhase();
        case 1:
            return LoadFFDShadersDoublePhase();     
        default:
            return FALSE;
    }
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Load shaders and assign parameter handles for the single phase algorithm   * 
* Loading the shaders only if there is support.                              * 
*                                                                            *
* PARAMETERS:                                                                *
*   void                                                                     *
*                                                                            *
* RETURN VALUE:                                                              *
*   int:       TRUE if loading was successful, FALSE otherwise.              *
*****************************************************************************/
static int LoadFFDShadersSinglePhase(void)
{
    char ShaderPath[255];

    if (FfdVertexProgram != NULL) 
    	return TRUE; /* Valid shaders are already loaded. */

    /* CG_PROFILE_VP40 supports the NV_vertex_program3 extension */
    
    if (cgGLIsProfileSupported(CG_PROFILE_VP40) &&
    	cgGLIsProfileSupported(CG_PROFILE_FP40)) {
		VertexProfile = CG_PROFILE_VP40;
		cgGLSetOptimalOptions(VertexProfile);
		FragmentProfile = CG_PROFILE_FP40;
		cgGLSetOptimalOptions(FragmentProfile);
    }
    else
        return FALSE; 

    /* Test cgContext creation. */
    Context = cgCreateContext();
    if (CheckCgError())
        return FALSE;
    
	SearchPath(NULL, "ffd_shdr.cg", NULL, 255, ShaderPath, NULL);

    FfdVertexProgram = cgCreateProgramFromFile(Context, 
					CG_SOURCE, ShaderPath,
		                        VertexProfile, "FFDSinglePhase", NULL);
    if (CheckCgError() && 
        FfdVertexProgram != NULL)
	return FALSE;

    if (!cgIsProgramCompiled(FfdVertexProgram))
        cgCompileProgram(FfdVertexProgram);

    cgGLLoadProgram(FfdVertexProgram);
    if (CheckCgError())
	    return FALSE;
    /* Bind the handles to semantics */  

    CG_GET_NAMED_PARAMETER(FfdVertexProgram, "vIn.position",
	CGParamSinglePhase.Position);
    CG_GET_NAMED_PARAMETER(FfdVertexProgram, "vIn.normal",
	CGParamSinglePhase.Normal);
    CG_GET_NAMED_PARAMETER(FfdVertexProgram, "vIn.color",
	CGParamSinglePhase.Color);
    CG_GET_NAMED_PARAMETER(FfdVertexProgram, "ModelViewProj",
	CGParamSinglePhase.ModelViewProjMatrix);
    CG_GET_NAMED_PARAMETER(FfdVertexProgram, "ModelView", 
	CGParamSinglePhase.ModelViewMatrix);
    CG_GET_NAMED_PARAMETER(FfdVertexProgram, "TrivDim",
	CGParamSinglePhase.TrivDim);    
    CG_GET_NAMED_PARAMETER(FfdVertexProgram, "TextureDim",
	CGParamSinglePhase.TexDim);
    CG_GET_NAMED_PARAMETER(FfdVertexProgram, "vIn.UBases",
	CGParamSinglePhase.UBases);
    CG_GET_NAMED_PARAMETER(FfdVertexProgram, "vIn.VBases", 
	CGParamSinglePhase.VBases);
    CG_GET_NAMED_PARAMETER(FfdVertexProgram, "vIn.WBases",
	CGParamSinglePhase.WBases);
    CG_GET_NAMED_PARAMETER(FfdVertexProgram, "texture",
	CGParamSinglePhase.TexObject);
    CG_GET_NAMED_PARAMETER(FfdVertexProgram, "AmbientFactor", 
	CGParamSinglePhase.AmbientFactor);
    CG_GET_NAMED_PARAMETER(FfdVertexProgram, "DiffuseFactor",
	CGParamSinglePhase.DiffuseFactor);
    CG_GET_NAMED_PARAMETER(FfdVertexProgram, "SpecularFactor",
	CGParamSinglePhase.SpecularFactor);
    CG_GET_NAMED_PARAMETER(FfdVertexProgram, "Shininess",
	CGParamSinglePhase.Shininess);
    CG_GET_NAMED_PARAMETER(FfdVertexProgram, "LightPos0",
	CGParamSinglePhase.LightPos0);
    CG_GET_NAMED_PARAMETER(FfdVertexProgram, "LightPos1",
	CGParamSinglePhase.LightPos1);
    
    return TRUE;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Load shaders and assign parameter handles for the single phase algorithm   * 
* Loading the shaders only if there is support.                              * 
*                                                                            *
* PARAMETERS:                                                                *
*   void                                                                     *
*                                                                            *
* RETURN VALUE:                                                              *
*   int:       TRUE if loading was successful, FALSE otherwise.              *
*****************************************************************************/
static int LoadFFDShadersDoublePhase(void)
{
    char ShaderPath[255];

    if (FfdVertexProgramPhase1 != NULL &&
	    FfdVertexProgramPhase2 &&
	    FfdFragmentProgram != NULL &&
	    FfdFragmentProgramPhase2 != NULL) 
        return TRUE; /* Valid shaders are already loaded. */

    /* CG_PROFILE_VP40 supports the NV_vertex_program3 extension */
    if (cgGLIsProfileSupported(CG_PROFILE_VP40) &&
    	    cgGLIsProfileSupported(CG_PROFILE_FP40)) {
	VertexProfile = CG_PROFILE_VP40;
	cgGLSetOptimalOptions(VertexProfile);
	FragmentProfile = CG_PROFILE_FP40;
	cgGLSetOptimalOptions(FragmentProfile);
    }
    else
        return FALSE; 

    Context = cgCreateContext();
    if (CheckCgError())
        return FALSE;
    SearchPath(NULL, "ffd_shdr.cg", NULL, 255, ShaderPath, NULL);
    /* Phase 1 vertex shader */
    FfdVertexProgramPhase1 = cgCreateProgramFromFile(Context, 
                                CG_SOURCE, ShaderPath,
	                        VertexProfile, "FFDDoublePhaseVertex1", NULL);
    if (CheckCgError() || 
        FfdVertexProgramPhase1 == NULL)
	    return FALSE;
    
    if (!cgIsProgramCompiled(FfdVertexProgramPhase1))
        cgCompileProgram(FfdVertexProgramPhase1);
    
    cgGLLoadProgram(FfdVertexProgramPhase1);
    if (CheckCgError())
        return FALSE;

    CG_GET_NAMED_PARAMETER(FfdVertexProgramPhase1,
	"vIn.position", CGParamDoublePhase1.Position);
    CG_GET_NAMED_PARAMETER(FfdVertexProgramPhase1,
	"TvSpan", CGParamDoublePhase1.TvSpan);
    CG_GET_NAMED_PARAMETER(FfdVertexProgramPhase1,
	"Range", CGParamDoublePhase1.Range);
    CG_GET_NAMED_PARAMETER(FfdVertexProgramPhase1,
	"MinLimit", CGParamDoublePhase1.MinLimit);
    CG_GET_NAMED_PARAMETER(FfdVertexProgramPhase1,
	"AnimVal", CGParamDoublePhase1.AnimVal);
    CG_GET_NAMED_PARAMETER(FfdVertexProgramPhase1, 
	"vIn.texCoord", CGParamDoublePhase1.TexCoord);
    CG_GET_NAMED_PARAMETER(FfdVertexProgramPhase1,
	"ModelViewProj", CGParamDoublePhase1.ModelViewProjMatrix);
    CG_GET_NAMED_PARAMETER(FfdVertexProgramPhase1,
	"Texture", CGParamDoublePhase1.Texture);

    
    /* Phase 1 fragment shader */
    FfdFragmentProgram = cgCreateProgramFromFile(Context, 
                                CG_SOURCE, ShaderPath,
	                        FragmentProfile, "FFDDoublePhasePixel", NULL);
    if (CheckCgError() || FfdFragmentProgram == NULL)
	return FALSE;
    
    if (!cgIsProgramCompiled(FfdFragmentProgram))
        cgCompileProgram(FfdFragmentProgram);

    cgGLLoadProgram(FfdFragmentProgram);
    if (CheckCgError())
	return FALSE;
      
    CG_GET_NAMED_PARAMETER(FfdFragmentProgram, 
	"TrivDim", CGParamDoublePhase1.TrivDim);
    CG_GET_NAMED_PARAMETER(FfdFragmentProgram,
	"TextureDim", CGParamDoublePhase1.TexDim);
    CG_GET_NAMED_PARAMETER(FfdFragmentProgram, 
	"texture", CGParamDoublePhase1.TexObject);

    /* Phase 2 vertex shader */
    FfdVertexProgramPhase2 = cgCreateProgramFromFile(Context, 
                                CG_SOURCE, ShaderPath,
		                VertexProfile, "FFDDoublePhaseVertex2", NULL);
    if (CheckCgError() || FfdVertexProgramPhase2 == NULL)
	return FALSE;

    if (!cgIsProgramCompiled(FfdVertexProgramPhase2))
        cgCompileProgram(FfdVertexProgramPhase2);    
 
    cgGLLoadProgram(FfdVertexProgramPhase2);
    if (CheckCgError())
	return FALSE;

    CG_GET_NAMED_PARAMETER(FfdVertexProgramPhase2,
	"vIn.position", CGParamDoublePhase2.Position);
    CG_GET_NAMED_PARAMETER(FfdVertexProgramPhase2,
	"vIn.color", CGParamDoublePhase2.Color);    
    CG_GET_NAMED_PARAMETER(FfdVertexProgramPhase2, 
	"AmbientFactor", CGParamDoublePhase2.AmbientFactor);
    CG_GET_NAMED_PARAMETER(FfdVertexProgramPhase2,
	"DiffuseFactor", CGParamDoublePhase2.DiffuseFactor);
    CG_GET_NAMED_PARAMETER(FfdVertexProgramPhase2,
	"SpecularFactor", CGParamDoublePhase2.SpecularFactor);
    CG_GET_NAMED_PARAMETER(FfdVertexProgramPhase2,
	"Shininess", CGParamDoublePhase2.Shininess);
    CG_GET_NAMED_PARAMETER(FfdVertexProgramPhase2,
	"LightPos0", CGParamDoublePhase2.LightPos0);
    CG_GET_NAMED_PARAMETER(FfdVertexProgramPhase2, 
	"LightPos1", CGParamDoublePhase2.LightPos1);
    CG_GET_NAMED_PARAMETER(FfdVertexProgramPhase2,
	"ModelView", CGParamDoublePhase2.ModelViewMatrix);
    CG_GET_NAMED_PARAMETER(FfdVertexProgramPhase2,
	"ModelViewProj", CGParamDoublePhase2.ModelViewProjMatrix);
    CG_GET_NAMED_PARAMETER(FfdVertexProgramPhase2,
	"texture", CGParamDoublePhase2.TexObject);
    CG_GET_NAMED_PARAMETER(FfdVertexProgramPhase2,
	"FlippedNormals", CGParamDoublePhase2.FlippedNormals);
    CG_GET_NAMED_PARAMETER(FfdVertexProgramPhase2, 
	"NormalMethod", CGParamDoublePhase2.NormalMethod);

    /* Phase 2 fragment shader */
    FfdFragmentProgramPhase2 = cgCreateProgramFromFile(Context, 
			    CG_SOURCE, ShaderPath,
			    FragmentProfile, "FFDDoublePhaseFragment2", NULL);
    if (CheckCgError() || FfdVertexProgramPhase2 == NULL)
	return FALSE;

    if (!cgIsProgramCompiled(FfdFragmentProgramPhase2))
        cgCompileProgram(FfdFragmentProgramPhase2);    
 
    cgGLLoadProgram(FfdFragmentProgramPhase2);
    if (CheckCgError())
	return FALSE;

    CG_GET_NAMED_PARAMETER(FfdFragmentProgramPhase2, "AmbientFactor", 
	CGParamDoublePhase2Fragment.AmbientFactor);
    CG_GET_NAMED_PARAMETER(FfdFragmentProgramPhase2, "DiffuseFactor",
	CGParamDoublePhase2Fragment.DiffuseFactor);
    CG_GET_NAMED_PARAMETER(FfdFragmentProgramPhase2, "SpecularFactor",
	CGParamDoublePhase2Fragment.SpecularFactor);
    CG_GET_NAMED_PARAMETER(FfdFragmentProgramPhase2, "Shininess",
	CGParamDoublePhase2Fragment.Shininess);
    CG_GET_NAMED_PARAMETER(FfdFragmentProgramPhase2, "LightPos0",
	CGParamDoublePhase2Fragment.LightPos0);
    CG_GET_NAMED_PARAMETER(FfdFragmentProgramPhase2, "LightPos1", 
	CGParamDoublePhase2Fragment.LightPos1);
    CG_GET_NAMED_PARAMETER(FfdFragmentProgramPhase2, "ModelView",
	CGParamDoublePhase2Fragment.ModelViewMatrix);
    CG_GET_NAMED_PARAMETER(FfdFragmentProgramPhase2, "FlippedNormals",
	CGParamDoublePhase2Fragment.FlippedNormals);
    CG_GET_NAMED_PARAMETER(FfdFragmentProgramPhase2, "NormalMethod", 
	CGParamDoublePhase2Fragment.NormalMethod);

    return TRUE;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Parses the string of the "ffd_texture" attribute	      	             *
*                                                                            *
* PARAMETERS:                                                                *
*   DTexture:       The string of the "dtexture" attribute.                  *
*   FName:          The texture file name will be placed here.               *
*   ShaderType:     The algorithm to use                                     *
*   NormalMethod:   Normals Calculation method                               *
*                                                                            *
* RETURN VALUE:                                                              *
*   int:         TRUE if parsed succesfully, FALSE otherwise.                *
*                                                                            *
* KEYWORDS:                                                                  *
*   IrtParseDTextureString                                                   *
*****************************************************************************/ 
static int IrtParseFfdTextureString(char *DTexture,
				    char *FName,
				    int *ShaderType,
				    RealType TvSpan[3],
				    int * TrivRenderState,
				    int * AnimSamples,
				    PointType * ConstOffset,
				    int *NormalMethod)
{
    char *p;

    if (DTexture == NULL)
    	return FALSE;

    TvSpan[0] = 1.0;
    TvSpan[1] = 1.0;
    TvSpan[2] = 1.0;

    strncpy(FName, DTexture, LINE_LEN_LONG - 1);
    if ((p = strchr(FName, ',')) != NULL) {
	*p++ = 0; /* Mark the end of the regular string. */
    	if (sscanf(p, "%d,%d,%lf,%lf,%lf,%d,%lf,%lf,%lf,%d", ShaderType, 
							TrivRenderState, 
							&TvSpan[0], 
							&TvSpan[1], 
							&TvSpan[2],
							AnimSamples,
							&((*ConstOffset)[0]),
							&((*ConstOffset)[1]),
							&((*ConstOffset)[2]),
							NormalMethod
							) != 10)
	return FALSE;
    }
    SearchPath(NULL, FName, NULL, LINE_LEN_LONG - 1, FName, NULL);
	return TRUE;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Evaluate the scl_z animation curve, if any, to set the scaling factor.   *
*                                                                            *
* PARAMETERS:                                                                *
*   PObj:   The DDM texture object.                                          *
*   t:      Current animation time.                                          *
*                                                                            *
* RETURN VALUE:                                                              *
*   RealType:    Scale factor computed, one by default.                      *
*****************************************************************************/
static void IGGetAnimationFactors(IPObjectStruct *PObj, RealType t)
{
    int i;
    RealType
        Factor = 1.0;
    IPObjectStruct
	*AnimCrv = AttrGetObjectObjAttrib(PObj, "ffd_anim"),
	*Crvs[3] = {NULL}, *TCrv;

    if (AnimCrv != NULL) {
        if (IP_IS_OLST_OBJ(AnimCrv)) {
	    int ListLen = IPListObjectLength(AnimCrv);
	    for (i = 0; i < ListLen; i++) {
		TCrv = IPListObjectGet(AnimCrv, i);
		if (IP_IS_CRV_OBJ(TCrv)) {
		    if (strnicmp(TCrv -> ObjName, "trans_u", 7) == 0)
			Crvs[0] = TCrv;
		    else if (strnicmp(TCrv -> ObjName, "trans_v", 7) == 0)
			Crvs[1] = TCrv;
		    else if (strnicmp(TCrv -> ObjName, "trans_w", 7) == 0)
			Crvs[2] = TCrv;
		}
	    }
    
	}
	else {
	    TCrv = AnimCrv;
	    if (IP_IS_CRV_OBJ(TCrv)) {
		if (strnicmp(TCrv -> ObjName,"trans_u", 7) == 0)
		    Crvs[0] = TCrv;
		else if (strnicmp(TCrv -> ObjName, "trans_v", 7) == 0)
		    Crvs[1] = TCrv;
		else if (strnicmp(TCrv -> ObjName, "trans_w", 7) == 0)
		    Crvs[2] = TCrv;
	    }
	}
    
	for (i = 0; i < 3; i++) {
	    RealType *R, TMin, TMax;
	    CagdCrvStruct *Crv;
	    
	    if (Crvs[i] == NULL)
		continue;
	    Crv = Crvs[i] -> U.Crvs;
	    CagdCrvDomain(Crv, &TMin, &TMax);
	    if (t < TMin)
		t = TMin;
	    else if (t > TMax)
		t = TMax;
	    R = CagdCrvEval(Crv, t);
	    AnimVal[i] = CAGD_IS_RATIONAL_CRV(Crv) ? R[1] / R[0] : R[1];
	}
    }
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Read the trivariate file name form the model definition                  *
* set up the texture for further processing.                                 *
*                                                                            *
* PARA*ETERS:                                                                *
*   PObj:      Object to extract its texture mapping function if has one.    *
*                                                                            *
* RETURN VALUE:                                                              *
*   int:       TRUE if object has ddm texture and process was successful     *
*                                                                            *
* KEYWORDS:                                                                  *
*                                                              *
*****************************************************************************/
static int IGInitFfdDraw(IPObjectStruct *PObj)
{
    IPObjectStruct* DefObj;
    int NormalMethod,
	AnimSamples = 1;
    PointType * ConstOffset;
    char FfdTextureFile[LINE_LEN_LONG], *Texture, *ModelFileName;

    /* Store the BBox for use in the vertex callback */
    if (glMultiTexCoord3f == NULL)
        glMultiTexCoord3f = 
	    (PFNGLMULTITEXCOORD3FARBPROC)wglGetProcAddress("glMultiTexCoord3f");
    if (glMultiTexCoord2f == NULL)
        glMultiTexCoord2f = 
	    (PFNGLMULTITEXCOORD2FARBPROC)wglGetProcAddress("glMultiTexCoord2f");
    if (glActiveTextureARB == NULL)
        glActiveTextureARB = 
	    (PFNGLACTIVETEXTUREARBPROC)wglGetProcAddress("glActiveTextureARB");
    
    if (!IP_IS_TRIVAR_OBJ(PObj))
	return FALSE;
    /* Texture already created */
    if (AttrGetObjectIntAttrib(PObj, "_ffd_tex_id") != IP_ATTR_BAD_INT)
        return TRUE;

    Texture = AttrGetObjectStrAttrib(PObj, "ffd_texture");
    if (Texture == NULL)
    	return FALSE;
    ConstOffset = (PointType*) (IritMalloc(sizeof(PointType)));
    (*ConstOffset)[0] = (*ConstOffset)[1] = (*ConstOffset)[2] = 0;
    if (!IrtParseFfdTextureString(Texture,
				FfdTextureFile,
				&FfdShaderType,
				TvSpan,
				&GlblRenderTrivState,
				&AnimSamples,
				ConstOffset,
				&NormalMethod)) 
				return FALSE;
    ModelFileName = FfdTextureFile;
    if ((DefObj= IPGetDataFiles(&ModelFileName, 1, TRUE, TRUE)) == NULL)
	return FALSE;
    
    if (IGCreateFFDTexture(DefObj,PObj) == FALSE)
	return FALSE;
    
    if (FfdShaderType == 1)
	if (InitPBuffer(0, PBUFFER_SIZE, PBUFFER_SIZE) == FALSE) {
	    if (IGGlblMore) {
		printf("Failed To init PBuffer\n");
	    }
	    return FALSE;
	}

    AttrSetObjectIntAttrib(PObj, "_anim_samples", AnimSamples);
    AttrSetObjectIntAttrib(PObj, "_normal_method", NormalMethod);
    AttrSetObjectPtrAttrib(PObj, "_const_samples", ConstOffset);
#ifdef DEBUG_CG_FFD
    printf("const sample [%lf,%lf,%lf]\n",
	(*ConstOffset)[0], (*ConstOffset)[1], (*ConstOffset)[2]);
#endif

    return TRUE;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Init a single PBuffer as rendering surface. of type float32 RGBA         M
*                                                                            *
* PARAMETERS:                                                                M
*   PbIdx: Buffer Index							     M
*   XLen: X Size of the buffer						     M
*   YLen: Y Size of the buffer						     M 
*                                                                            *
* RETURN VALUE:                                                              M
*   int:  TRUE if successful, FALSE if no go.				     M
*                                                                            *
* KEYWORDS:                                                                  M
*   InitPBuffer                                                              M
*****************************************************************************/
int InitPBuffer(int PbIdx, int XLen, int YLen)
{
    GLuint NumFormats;
    GLint PixelFormat;
    const int PBufferFlags[] = {
        WGL_TEXTURE_FORMAT_ARB, WGL_TEXTURE_RGBA_ARB,
        WGL_TEXTURE_TARGET_ARB, WGL_TEXTURE_2D_ARB,
	0, 0, 0};
    const int IAttribList[]={
        WGL_BIND_TO_TEXTURE_RGBA_ARB, 1, 
        WGL_DRAW_TO_PBUFFER_ARB, 1,
        WGL_SUPPORT_OPENGL_ARB, 1,
        WGL_PIXEL_TYPE_ARB,WGL_TYPE_RGBA_FLOAT_ATI,
	WGL_RED_BITS_ARB, 32,
        WGL_GREEN_BITS_ARB, 32,
        WGL_BLUE_BITS_ARB, 32,
	WGL_ALPHA_BITS_ARB, 32,
        0 };			
    const float FAttribList[] = { 0 };

    if (PBuffer[PbIdx].HBuffer) /*PBuffer already exists*/
        return TRUE;
    
    if (!IsArbExtensionSupported("WGL_ARB_extensions_string") ||
	!IsArbExtensionSupported("WGL_ARB_pixel_format") ||
	!IsArbExtensionSupported("WGL_ARB_pbuffer"))
        return FALSE;
	
    glGenTextures(1, &PBuffer[PbIdx].PBufferTexId);
    glBindTexture(GL_TEXTURE_2D, PBuffer[PbIdx].PBufferTexId);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);

    PBuffer[PbIdx].OrigDC = wglGetCurrentDC();
    PBuffer[PbIdx].OrigRC = wglGetCurrentContext();
    
    wglGetPixelFormatAttribivARB = (PFNWGLGETPIXELFORMATATTRIBIVARBPROC)
        wglGetProcAddress("wglGetPixelFormatAttribivARB");
    wglGetPixelFormatAttribfvARB = (PFNWGLGETPIXELFORMATATTRIBFVARBPROC)
        wglGetProcAddress("wglGetPixelFormatAttribfvARB");
    wglChoosePixelFormatARB = (PFNWGLCHOOSEPIXELFORMATARBPROC)
        wglGetProcAddress("wglChoosePixelFormatARB");
    wglCreatePbufferARB = (PFNWGLCREATEPBUFFERARBPROC)
        wglGetProcAddress("wglCreatePbufferARB");
    wglGetPbufferDCARB=(PFNWGLGETPBUFFERDCARBPROC)
	wglGetProcAddress("wglGetPbufferDCARB");
    wglReleasePbufferDCARB = (PFNWGLRELEASEPBUFFERDCARBPROC)
    	wglGetProcAddress("wglReleasePbufferDCARB");
    wglDestroyPbufferARB = (PFNWGLDESTROYPBUFFERARBPROC)
	wglGetProcAddress("wglDestroyPbufferARB");
    wglQueryPbufferARB = (PFNWGLQUERYPBUFFERARBPROC)
	wglGetProcAddress("wglQueryPbufferARB");
    wglBindTexImageARB = (PFNWGLBINDTEXIMAGEARBPROC)
        wglGetProcAddress("wglBindTexImageARB");
    wglReleaseTexImageARB = (PFNWGLRELEASETEXIMAGEARBPROC)
        wglGetProcAddress("wglReleaseTexImageARB");
    wglSetPbufferAttribARB = (PFNWGLSETPBUFFERATTRIBARBPROC)
        wglGetProcAddress("wglSetPbufferAttribARB");
    
    if (!wglChoosePixelFormatARB(PBuffer[PbIdx].OrigDC, 
				 IAttribList, 
				 FAttribList, 
				 1,
				 &PixelFormat, 
				 &NumFormats))
				 return FALSE;
    
    PBuffer[PbIdx].Width =  XLen;
    PBuffer[PbIdx].Height = YLen;
    PBuffer[PbIdx].HBuffer=wglCreatePbufferARB(PBuffer[PbIdx].OrigDC, 
					       PixelFormat, 
					       PBuffer[PbIdx].Width, 
					       PBuffer[PbIdx].Height, 
					       PBufferFlags);
    if (!PBuffer[PbIdx].HBuffer)
	return FALSE;
    PBuffer[PbIdx].PBufferDC=wglGetPbufferDCARB(PBuffer[PbIdx].HBuffer);
    if (!PBuffer[PbIdx].PBufferDC)
	return FALSE;

    PBuffer[PbIdx].PBufferRC =wglCreateContext(PBuffer[PbIdx].PBufferDC);
    if (!PBuffer[PbIdx].PBufferRC)
        return FALSE;
    wglQueryPbufferARB(PBuffer[PbIdx].HBuffer,
	WGL_PBUFFER_WIDTH_ARB, &PBuffer[PbIdx].Width);
    wglQueryPbufferARB(PBuffer[PbIdx].HBuffer, 
	WGL_PBUFFER_HEIGHT_ARB, &PBuffer[PbIdx].Height);
    wglShareLists(PBuffer[PbIdx].OrigRC, PBuffer[PbIdx].PBufferRC);

#ifdef DEBUG_CG_FFD    
    printf("Create pbuffer size = (%dX%d)\n", 
	PBuffer[PbIdx].Width,PBuffer[PbIdx].Height);
#endif
    return TRUE;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Close the PBuffer.						             M
*                                                                            *
* PARAMETERS:                                                                M
*   PbIdx: Buffer Index							     M
*                                                                            *
* RETURN VALUE:                                                              M
*   void.				    				     M
*                                                                            *
* KEYWORDS:                                                                  M
*   ClosePBuffer, InitPBuffer                                                M
*****************************************************************************/
void ClosePBuffer(int PbIdx)
{
    if (PBuffer[PbIdx].HBuffer == NULL)
	return;
    if (PBuffer[PbIdx].PBufferRC) {
	if (!wglDeleteContext(PBuffer[PbIdx].PBufferRC))	{
#ifdef DEBUG_CG_FFD
	    printf("Fail to release Pbuffer Rendering Context.\n");
#endif
       }
       PBuffer[PbIdx].PBufferRC = NULL;
    }
   if (PBuffer[PbIdx].PBufferDC &&
       !wglReleasePbufferDCARB(PBuffer[PbIdx].HBuffer, PBuffer[PbIdx].PBufferDC)) {
#ifdef DEBUG_CG_FFD
	    printf("Fail to release Pbuffer DC\n"); 
#endif
	    PBuffer[PbIdx].PBufferDC = NULL;
       }
	
    if (!wglDestroyPbufferARB(PBuffer[PbIdx].HBuffer)) {
#ifdef DEBUG_CG_FFD
	printf("Fail to destroy pbuffer\n");
#endif
    }
    PBuffer[PbIdx].HBuffer = NULL;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Draws a Black rectangle to the PBuffer to make sure it is in init state  M
*                                                                            *
* PARAMETERS:                                                                M
*   PbIdx: Buffer to close                                                   M
*                                                                            *
* RETURN VALUE:                                                              M
*   void.                                        			     M
*                                                                            *
* KEYWORDS:                                                                  M
*   ErasePBuffer                                                             M
*****************************************************************************/
void ErasePBuffer(int PbIdx)
{
    glDisable(GL_TEXTURE_2D);
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glOrtho(0, 1, 0, 1, -1, 1);
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    glBegin(GL_POLYGON);
    glColor4f(0, 0, 0, 1.0);
    glVertex3f(0, 0, 0);
    glVertex3f(0, 1, 0);
    glVertex3f(1, 1, 0);
    glVertex3f(1, 0, 0);
    glEnd();
    glPopMatrix();
    glMatrixMode(GL_PROJECTION);
    glPopMatrix();
    glMatrixMode(GL_MODELVIEW);
    
    wglReleaseTexImageARB(PBuffer[PbIdx].HBuffer, WGL_FRONT_LEFT_ARB);
    glDisable(GL_TEXTURE_2D);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Changes the valid pbuffer index					     M
*                                                                            *
* PARAMETERS:                                                                M
*   PbIdx: Buffer index to make current                                      M
*                                                                            *
* RETURN VALUE:                                                              M
*   int: TRUE for success, FALSE otherwise.                      	     M
*                                                                            *
* KEYWORDS:                                                                  M
*   MakePBufferCurrent                                                       M
*****************************************************************************/
int MakePBufferCurrent(int PbIdx)
{
    if (PBuffCurrent == 1)
	return TRUE;
    if (!wglMakeCurrent(PBuffer[PbIdx].PBufferDC, PBuffer[PbIdx].PBufferRC))
        return FALSE;
    PBuffCurrent = 1;
    return TRUE;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Changes the valid pbuffer index					     M
*                                                                            *
* PARAMETERS:                                                                M
*   PbIdx: Buffer index to make current                                      M
*                                                                            *
* RETURN VALUE:                                                              M
*   int: TRUE for success, FALSE otherwise.                            	     M
*                                                                            *
* KEYWORDS:                                                                  M
*   MakeCurrent                                                              M
*****************************************************************************/
int MakeCurrent(int PbIdx)
{
    if (PBuffCurrent == 0)
	return TRUE;
    if (!wglMakeCurrent(PBuffer[PbIdx].OrigDC, PBuffer[PbIdx].OrigRC))
        return FALSE;
    PBuffCurrent = 0;
    return TRUE;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Creates and allocate an array from pbuffer data			     M
*                                                                            *
* PARAMETERS:                                                                M
*   PbIdx: Buffer index			                                     M
*   XLen:  X length of the buffer					     M
*   YLen:  Y length of the buffer					     M
*                                                                            *
* RETURN VALUE:                                                              M
*   GLfloat *: pointer to the allocated array                         	     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IGGetPBufferToHost                                                       M
*****************************************************************************/
GLfloat *IGGetPBufferToHost(int PbIdx,int XLen, int YLen)
{
    GLfloat *image;
    MakePBufferCurrent(PbIdx);
    image = (GLfloat*) IritMalloc(XLen * YLen * 4 * sizeof(GLfloat));
    glReadPixels(0, 0, XLen, YLen, GL_RGBA, GL_FLOAT, image);
    MakeCurrent(PbIdx);
    return image;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Draws a single object with DTexture attributes using current modes       M
*   and transformations.	                                             M
*                                                                            *
* PARAMETERS:                                                                M
*   PObj:      Object to draw.                                               M
*                                                                            *
* RETURN VALUE:                                                              M
*   int:  TRUE if successful, FALSE if no go.				     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IGCGFfdDraw                                                              M
*****************************************************************************/
int IGCGFfdDraw(IPObjectStruct *PObj)
{
    int Ret = FALSE;
    STATIC_DATA int LocInList = 0;

    if (!IGInitFfdDraw(PObj) ||
	!IGLoadShaders())
        return FALSE;
    
    glPushAttrib(GL_CURRENT_BIT | GL_COLOR_BUFFER_BIT | GL_ENABLE_BIT);
    glDisable(GL_BLEND);
    glDepthMask(GL_TRUE);

    switch (FfdShaderType) {
	case 0:
	    Ret = RenderShaderSinglePhase(PObj);
	    break;
	case 1:
	    Ret = RenderShaderDoublePhase(PObj);
	    break;
    }
    glPopAttrib();
    if (GlblRenderTrivState) {
	return FALSE; /* Force the triv to be rendered */
    }
    else
	return TRUE; /* Finish */
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Draw FFD using single pass rendering algorithm on the vertex processor   *
*   assume to load the proper shaders                                        *
*                                                                            *
* PARAMETERS:                                                                *
*   PObj:      Object to draw.                                               *
*                                                                            *
* RETURN VALUE:                                                              *
*   int:  TRUE if successful, FALSE if no go.				     *
*                                                                            *
* KEYWORDS:                                                                  *
*   IGFfdDraw,RenderShaderSinglePhase                                        *
*****************************************************************************/
static int RenderShaderSinglePhase(IPObjectStruct *PObj)
{
    IGVertexHandleExtraFuncType OldFunc;
    TrivTVStruct* Tv;
    IPObjectStruct *DefObj;
    GLuint ModelDisplayList;
    int TexId;
    STATIC_DATA int NumPolys;

    SetGlbObjColor(PObj);
    IGCheckSysStatus(PObj,0);
    OldFunc = IGSetHandleVertexProcessingFunc(IGCgVertexHanlingSinglePhase); 
    
    if (IGGlblAnimation) {
        IGGetAnimationFactors(PObj,IGAnimation.RunTime);
	RecomputeLists(PObj);			      
    }
    else{
        AnimVal[0] = 0.0;
        AnimVal[1] = 0.0;
        AnimVal[2] = 0.0;
    }
    DefObj = SingleStepAnimation(PObj);
    if (DefObj == NULL)
	DefObj = (IPObjectStruct*)AttrGetObjectPtrAttrib(PObj, "_ffd_def_obj");
    
    Tv = PObj->U.Trivars;
    MinParam[0] = Tv -> UOrder - 1;
    MinParam[1] = Tv -> VOrder - 1;
    MinParam[2] = Tv -> WOrder - 1;
    RangeParam[0] =  Tv -> ULength - MinParam[0];
    RangeParam[1] =  Tv -> VLength - MinParam[1];
    RangeParam[2] =  Tv -> WLength - MinParam[2];
    
    memcpy(ObjBBox, DefObj -> BBox, sizeof(ObjBBox)); 
    
     /* Enabling shaders & Textures. */
    cgGLEnableProfile(VertexProfile);
    CheckCgError();

    cgGLBindProgram(FfdVertexProgram);
    CheckCgError();
/*
    cgGLEnableProfile(FragmentProfile);
    CheckCgError();

    cgGLBindProgram(FfdFragmentProgram);
    CheckCgError();
*/  
    CG_GL_SET_PARAMETER_2F(CGParamSinglePhase.TexDim, 
	(GLfloat) 1.0 / FfdTexDimV, (GLfloat) 1.0 / FfdTexDimU);
    CG_GL_SET_PARAMETER_3F(CGParamSinglePhase.TrivDim,
        (GLfloat) Tv -> ULength, 
        (GLfloat) Tv -> VLength,
        (GLfloat) Tv -> WLength);
    
    CG_GL_SET_STATE_MATRIX_PARAMETER(CGParamSinglePhase.ModelViewProjMatrix,
    	CG_GL_MODELVIEW_PROJECTION_MATRIX, CG_GL_MATRIX_IDENTITY);
    
    CG_GL_SET_STATE_MATRIX_PARAMETER(CGParamSinglePhase.ModelViewMatrix,
    	CG_GL_MODELVIEW_MATRIX, CG_GL_MATRIX_IDENTITY);
    
    UpdateShadingParameters(1);

    TexId = AttrGetObjectIntAttrib(PObj, "_ffd_tex_id");
    glBindTexture(GL_TEXTURE_2D, TexId);
    cgGLSetTextureParameter(CGParamSinglePhase.TexObject, TexId);	
    CheckCgError();
    cgGLEnableTextureParameter(CGParamSinglePhase.TexObject);
	
    /* Render loop - for this version we have to traverse here */   
    /* Recursive call after attr set - 
       force normal render accept for attribute setting */
    if ((ModelDisplayList = AttrGetObjectIntAttrib(PObj, "_ffd_model_disp_lst"))
        != IP_ATTR_BAD_INT) {
            glCallList(ModelDisplayList);
	    IGGlblNumPolys = NumPolys;
        }
    else{
	NumPolys = IGGlblNumPolys;
	ModelDisplayList  = glGenLists(1);
        AttrSetObjectIntAttrib(PObj, "_ffd_model_disp_lst", ModelDisplayList);
        glNewList(ModelDisplayList, GL_COMPILE);
	IGDisplayObject(DefObj);
	glEndList();
	NumPolys = IGGlblNumPolys - NumPolys; 
	glCallList(ModelDisplayList);
    }
    
    cgGLDisableTextureParameter(CGParamSinglePhase.TexObject);
    cgGLDisableProfile(VertexProfile);
/*  
    cgGLDisableProfile(FragmentProfile);
*/
    IGSetHandleVertexProcessingFunc(OldFunc);

    return TRUE;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Draw FFD using double pass rendering algorithm on the pixel and vertex   *
*   processos. First pass will compute deformation on the vertex and pixel   *
*   processors and output the result to a pbuffer. The second pass will      *
*   apply the deformation and compute the shading                            *   
* PARAMETERS:                                                                *
*   PObj:      Object to draw.                                               *
*                                                                            *
* RETURN VALUE:                                                              *
*   int:  TRUE if successful, FALSE if no go.				     *
*                                                                            *
* KEYWORDS:                                                                  *
*   IGFfdDraw,RenderShaderDoublePhase                                        *
*****************************************************************************/
static int RenderShaderDoublePhase(IPObjectStruct *PObj)
{
    int TexId;
    PointType *ConstOffset,AnimValLocal;
    IPObjectStruct *DefObj = SingleStepAnimation(PObj);
    if (DefObj == NULL)
	DefObj = (IPObjectStruct*) AttrGetObjectPtrAttrib(PObj, "_ffd_def_obj");

    IGCheckSysStatus(DefObj, 0);

    if (IGGlblAnimation) {
	IGGetAnimationFactors(PObj, IGAnimation.RunTime);
	if (AttrGetObjectIntAttrib(PObj, "_anim_samples") > 0)
	    PreProcessAnimation(PObj);
    }

    AnimValLocal[0] = AnimVal[0];
    AnimValLocal[1] = AnimVal[1];
    AnimValLocal[2] = AnimVal[2];
    
    ConstOffset = AttrGetObjectPtrAttrib(PObj, "_const_samples");
    if (ConstOffset) {
	AnimValLocal[0] += (*ConstOffset)[0];
	AnimValLocal[1] += (*ConstOffset)[1];
	AnimValLocal[2] += (*ConstOffset)[2];
    
    }

    MakePBufferCurrent(0);
    
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    /* Enable phase1 vertex shader and pixel shader */
    cgGLEnableProfile(VertexProfile);
    CheckCgError();
    cgGLBindProgram(FfdVertexProgramPhase1);
    CheckCgError();

    cgGLEnableProfile(FragmentProfile);
    CheckCgError();
    cgGLBindProgram(FfdFragmentProgram);
    CheckCgError();

    CG_GET_NAMED_PARAMETER(FfdVertexProgramPhase1, 
	"vIn.position", CGParamDoublePhase1.Position);
    CG_GET_NAMED_PARAMETER(FfdVertexProgramPhase1, 
	"TvSpan", CGParamDoublePhase1.TvSpan);
    CG_GET_NAMED_PARAMETER(FfdVertexProgramPhase1,
	"Range", CGParamDoublePhase1.Range);
    CG_GET_NAMED_PARAMETER(FfdVertexProgramPhase1,
	"MinLimit", CGParamDoublePhase1.MinLimit);
    CG_GET_NAMED_PARAMETER(FfdVertexProgramPhase1,
	"AnimVal", CGParamDoublePhase1.AnimVal);
    CG_GET_NAMED_PARAMETER(FfdVertexProgramPhase1,
	"ModelViewProj", CGParamDoublePhase1.ModelViewProjMatrix);
    CG_GET_NAMED_PARAMETER(FfdVertexProgramPhase1, 
	"Texture", CGParamDoublePhase1.Texture);
    CG_GET_NAMED_PARAMETER(FfdFragmentProgram,
	"texture", CGParamDoublePhase1.TexObject);

    TexId = AttrGetObjectIntAttrib(PObj, "_ffd_tex_id");

    glActiveTextureARB(GL_TEXTURE0_ARB);  
    glBindTexture(GL_TEXTURE_2D, TexId);
    cgGLSetTextureParameter(CGParamDoublePhase1.TexObject, TexId);	
    CheckCgError();
    cgGLEnableTextureParameter(CGParamDoublePhase1.TexObject);
    CG_GL_SET_PARAMETER_2D(CGParamDoublePhase1.TexDim,
			   1.0 / FfdTexDimV, 1.0 / FfdTexDimU);

    glActiveTextureARB(GL_TEXTURE1_ARB); 
    glBindTexture(GL_TEXTURE_2D, VertexPhase1TextureHandle);
    cgGLSetTextureParameter(CGParamDoublePhase1.Texture,
			    VertexPhase1TextureHandle);
    CheckCgError();
    cgGLEnableTextureParameter(CGParamDoublePhase1.Texture);
    CheckCgError();

    /* Set globals for this model */  
    Tv = PObj->U.Trivars;
    MinParam[0] = Tv -> UOrder - 1;
    MinParam[1] = Tv -> VOrder - 1;
    MinParam[2] = Tv -> WOrder - 1;
    RangeParam[0] =  Tv -> ULength - MinParam[0];
    RangeParam[1] =  Tv -> VLength - MinParam[1];
    RangeParam[2] =  Tv -> WLength - MinParam[2];
   
    CG_GL_SET_PARAMETER_3F(CGParamDoublePhase1.TvSpan,
		    (GLfloat)TvSpan[0],
		    (GLfloat)TvSpan[1],
		    (GLfloat)TvSpan[2]);
    
    CG_GL_SET_PARAMETER_3F(CGParamDoublePhase1.Range,
		    (GLfloat)RangeParam[0],
		    (GLfloat)RangeParam[1],
		    (GLfloat)RangeParam[2]);

    CG_GL_SET_PARAMETER_3F(CGParamDoublePhase1.MinLimit,
		    (GLfloat)MinParam[0],
		    (GLfloat)MinParam[1],
		    (GLfloat)MinParam[2]);

    CG_GL_SET_PARAMETER_3F(CGParamDoublePhase1.AnimVal,
		    (GLfloat)AnimValLocal[0],
		    (GLfloat)AnimValLocal[1],
		    (GLfloat)AnimValLocal[2]);
#ifdef DEBUG_CG_FFD
    printf("Anim val = [%lf %lf %lf]\n", AnimValLocal[0], 
	AnimValLocal[1], AnimValLocal[2]);
#endif
    CG_GL_SET_PARAMETER_3F(CGParamDoublePhase1.TrivDim,
		    (GLfloat)Tv -> ULength,
		    (GLfloat)Tv -> VLength,
		    (GLfloat)Tv -> WLength);
    
    GlblNormalCalcMethod = 
	AttrGetObjectIntAttrib(PObj, "_normal_method");

    DrawModelAsGrid(DefObj);
   
    cgGLDisableTextureParameter(CGParamDoublePhase1.TexObject);
    cgGLDisableTextureParameter(CGParamDoublePhase1.Texture);
    glDisable(GL_TEXTURE_2D);
    cgGLDisableProfile(VertexProfile);
    CheckCgError();
    cgGLDisableProfile(FragmentProfile);
    CheckCgError();
    ApplyDeformedObjectRender(DefObj);
    
    IncrementInstanceCounter(PObj);
    return TRUE;
}

/* Phase 2 */
/*****************************************************************************
* DESCRIPTION:                                                               *
*   Apply the deformed vertex positions that are stored.		     *	
*   The second pass will						     *
*   apply the deformation and compute the shading                            *  
*   The function is called after the whole scene finished drawing            *	
* PARAMETERS:                                                                *
*									     *
*                                                                            *
* RETURN VALUE:                                                              *
*   int:  TRUE if successful, FALSE if no go.				     *
*                                                                            *
* KEYWORDS:                                                                  *
*   IGFfdDraw, RenderShaderDoublePhase                                       *
*****************************************************************************/
static int ApplyDeformedObjectRender(IPObjectStruct *PObj)
{
    GLuint ModelDisplayList;
    IGVertexHandleExtraFuncType OldFunc;
    STATIC_DATA int NumPolys;

    MakeCurrent(0);
    
    cgGLEnableProfile(VertexProfile);
    CheckCgError();
    cgGLBindProgram(FfdVertexProgramPhase2);
    CheckCgError();

    cgGLEnableProfile(FragmentProfile);
    CheckCgError();
    cgGLBindProgram(FfdFragmentProgramPhase2);
    CheckCgError();

    glDisable(GL_TEXTURE_2D);
    glActiveTextureARB(GL_TEXTURE0_ARB);
    glBindTexture(GL_TEXTURE_2D, PBuffer[0].PBufferTexId);
    wglBindTexImageARB(PBuffer[0].HBuffer, WGL_FRONT_LEFT_ARB);
    
    UpdateShadingParameters(2);
    
    CG_GL_SET_STATE_MATRIX_PARAMETER(CGParamDoublePhase2.ModelViewProjMatrix,
    CG_GL_MODELVIEW_PROJECTION_MATRIX, CG_GL_MATRIX_IDENTITY);
    CG_GL_SET_STATE_MATRIX_PARAMETER(CGParamDoublePhase2.ModelViewMatrix,
	CG_GL_MODELVIEW_MATRIX, CG_GL_MATRIX_IDENTITY);

    CG_GL_SET_PARAMETER_1F(CGParamDoublePhase2.FlippedNormals,
	IGGlblFlipNormalOrient ? 1.0f : -1.0f);
    CG_GL_SET_PARAMETER_1F(CGParamDoublePhase2.NormalMethod,
	(float) GlblNormalCalcMethod);
    CG_GL_SET_STATE_MATRIX_PARAMETER(CGParamDoublePhase2Fragment.ModelViewMatrix,
    CG_GL_MODELVIEW_MATRIX, CG_GL_MATRIX_IDENTITY);
    CG_GL_SET_PARAMETER_1F(CGParamDoublePhase2Fragment.FlippedNormals,
	IGGlblFlipNormalOrient ? 1.0f : -1.0f);
    CG_GL_SET_PARAMETER_1F(CGParamDoublePhase2Fragment.NormalMethod,
	(float) GlblNormalCalcMethod);

    OldFunc = IGSetHandleVertexProcessingFunc(IGCgVertexHandlingSecondPass); 
    VertexCount = 0;
    
    SetGlbObjColor(PObj);
    /* Render loop - for this version we have to traverse here* /   
    /* Recursive call after attr set -
       force normal render accept for attribute setting */
    if ((ModelDisplayList = 
	AttrGetObjectIntAttrib(PObj, "_ffd_model_disp_lst"))
	!= IP_ATTR_BAD_INT) {
	    glCallList(ModelDisplayList);
	    IGGlblNumPolys += AttrGetObjectIntAttrib(PObj, "_num_polys");
	}
    else{
	GLuint texId = 0;
	MakeCurrent(1);
	NumPolys = IGGlblNumPolys;
	ModelDisplayList  = glGenLists(1);
	AttrSetObjectIntAttrib(PObj, "_ffd_model_disp_lst", ModelDisplayList);
	glNewList(ModelDisplayList, GL_COMPILE);
	IGDisplayObject(PObj);
	glEndList();
	NumPolys = IGGlblNumPolys - NumPolys;
	AttrSetObjectIntAttrib(PObj, "_num_polys", NumPolys);    
    }

    wglReleaseTexImageARB(PBuffer[0].HBuffer, WGL_FRONT_LEFT_ARB);

    cgGLDisableProfile(VertexProfile);
    cgGLDisableProfile(FragmentProfile);
    
    /* Reset the parameters for the second pass */  
    Cx = 0;
    Cy = 1;

    IGSetHandleVertexProcessingFunc(NULL);
    return TRUE;    
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*  In the case where an object has animation-steps > 0 this mean that there  *	
*  is an active irit animation on the object, the animation is processed and *
*  the objects are stored as a list of display lists			     *
* PARAMETERS:                                                                *
*   Dummy object for rendering                                               *
*                                                                            *
* RETURN VALUE:                                                              *
*   TRUE for success, FALSE otherwise					     *
*                                                                            *
* KEYWORDS:                                                                  *
*   IGFfdDraw			                                             *
******************************************************************************/
static int PreProcessAnimation(IPObjectStruct * PObj)
{
    IPObjectStruct *DefObj, *AnimObj,
	 *AnimObjList = NULL;
    PointType CenterFirst, CenterLast;
    GMAnimationStruct Animation;
    RealType AnimTime;
    PointType *AnimOffset;
    int i, 
	AnimSteps = 0,
	NumInstances = 1;
    
    AnimSteps = AttrGetObjectIntAttrib(PObj, "_anim_samples");
    DefObj = AttrGetObjectPtrAttrib(PObj, "_ffd_def_obj");
    if (DefObj == NULL)
	return FALSE;
    
    if ((AnimObjList = (IPObjectStruct*) 
	AttrGetPtrAttrib(PObj -> Attr, "_anim_object_list")) == NULL) {
	
	GMAnimFindAnimationTime(&Animation, DefObj);
	AnimTime = Animation.FinalT - Animation.StartT;
	if (IRIT_APX_EQ(AnimTime, 0))
	    return FALSE;
	AnimOffset = IritMalloc(sizeof(PointType));
	(*AnimOffset)[0] = (*AnimOffset)[1] = (*AnimOffset)[2] = 0;
	for (i = 0; i < AnimSteps; i++) {
	    if (AnimSteps > 0) {
    		GMBBBboxStruct *BBox;
		AnimObj = GMAnimEvalObjAtTime((float) i * AnimTime / 
		    AnimSteps, DefObj);
#ifdef DEBUG_CG_FFD
		printf("load animation sample - %d\n", i);
#endif
		if (i == 0) {

			BBox = GMBBComputeBboxObject(AnimObj);
		    CenterFirst[0] = (BBox -> Max[0] + BBox -> Min[0]) / 2;
		    CenterFirst[1] = (BBox -> Max[1] + BBox -> Min[1]) / 2;
		    CenterFirst[2] = (BBox -> Max[2] + BBox -> Min[2]) / 2;
		}
		else if (i == AnimSteps - 1) {
		    BBox = GMBBComputeBboxObject(AnimObj);
		    CenterLast[0] = (BBox -> Max[0] + BBox -> Min[0]) / 2;
		    CenterLast[1] = (BBox -> Max[1] + BBox -> Min[1]) / 2;
		    CenterLast[2] = (BBox -> Max[2] + BBox -> Min[2]) / 2;
		    (*AnimOffset)[0] =  CenterLast[0] - CenterFirst[0];
		    (*AnimOffset)[1] =  CenterLast[1] - CenterFirst[1];
		    (*AnimOffset)[2] =  CenterLast[2] - CenterFirst[2];
		}
		AnimObj -> Pnext = NULL;
		AnimObjList = IPAppendObjLists(AnimObjList, AnimObj);
	    }
	}
	if ((*AnimOffset)[0] > (*AnimOffset)[1] && 
	    (*AnimOffset)[0] > (*AnimOffset)[2]) {
	    (*AnimOffset)[1] = 0; (*AnimOffset)[2] =0;
	}else if ((*AnimOffset)[1] > (*AnimOffset)[0] && 
	    (*AnimOffset)[1] > (*AnimOffset)[2]) {
	    (*AnimOffset)[0] = 0; (*AnimOffset)[2] =0;
	}
	else if ((*AnimOffset)[2] > (*AnimOffset)[0] &&
	    (*AnimOffset)[2] > (*AnimOffset)[1]) {
	    (*AnimOffset)[0] = 0; (*AnimOffset)[1] =0;
	}
	AttrSetPtrAttrib(&PObj -> Attr, "_anim_object_list", AnimObjList);
	AttrSetPtrAttrib(&PObj -> Attr, "_anim_offset", AnimOffset);

    }
    return TRUE;	
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Given an object get the next instance from the animation series.         *
*						   			     *
* PARAMETERS:                                                                *
*									     *
*                                                                            *
* RETURN VALUE:                                                              *
*   IPObjectStruct: 			    				     *
*                                                                            *
* KEYWORDS:                                                                  *
*   SingleStepAnimation		                                             *
******************************************************************************/
static IPObjectStruct *SingleStepAnimation(IPObjectStruct * PObj)
{
    IPObjectStruct *AnimObjList;
    PointType *AnimOffset;
    int StepDiv, StepMod, StepCounter, i, AnimSteps, NumInstances;
    
    if ((AnimSteps = AttrGetObjectIntAttrib(PObj, "_anim_samples"))
	== IP_ATTR_BAD_INT || AnimSteps <= 0) {
	return NULL;
    }
    if ((AnimObjList = 
	AttrGetObjectPtrAttrib(PObj, "_anim_object_list")) == NULL) {
	return NULL;
    }
    if ((AnimOffset = 
	AttrGetObjectPtrAttrib(PObj, "_anim_offset")) == NULL) {
	return NULL;
    }
    
    StepCounter = (int) (((IGAnimation.RunTime )) / IGAnimation.Dt);
    StepMod = StepCounter % AnimSteps;
    StepDiv = StepCounter / AnimSteps;
    NumInstances = (int) (((IGAnimation.FinalT - IGAnimation.StartT)
	/ IGAnimation.Dt) / AnimSteps);
    i = 0;
    while (AnimObjList) {
	if (i++ == StepMod) {   
	    AnimVal[0] = ((*AnimOffset)[0]) * (StepDiv);
	    AnimVal[1] = ((*AnimOffset)[1]) * (StepDiv);
	    AnimVal[2] = ((*AnimOffset)[2]) * (StepDiv);
	    
	    return AnimObjList;
	}
	AnimObjList = AnimObjList -> Pnext;
    }
    return NULL;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Increment the animation instnace counter. If an object has stored series *
*   of animation objects, the function will increment th counter	     *
* PARAMETERS:                                                                *
*									     *
*                                                                            *
* RETURN VALUE:                                                              *
*   void.	    			    				     *
*                                                                            *
* KEYWORDS:                                                                  *
*   IncrementInstanceCounter	                                             *
******************************************************************************/
static void IncrementInstanceCounter(IPObjectStruct * PObj)
{
    int AnimSteps, InstanceCounter, NumInstances;
    IPObjectStruct *DefObj;
    RealType RunTime;
    DefObj = SingleStepAnimation(PObj);

    if (DefObj == NULL)
	return;
    if ((AnimSteps = 
	AttrGetObjectIntAttrib(PObj, "_anim_samples")) == IP_ATTR_BAD_INT)
	return;
    if ((RunTime = 
	AttrGetObjectRealAttrib(DefObj, "_anim_runtime")) == IP_ATTR_BAD_REAL)
	AttrSetObjectRealAttrib(DefObj, "_anim_runtime", IGAnimation.RunTime);
    else if (IGAnimation.RunTime == 
	AttrGetObjectRealAttrib(DefObj, "_anim_runtime"))
	return;
    else AttrSetObjectRealAttrib(DefObj, "_anim_runtime", IGAnimation.RunTime);
	NumInstances = (int)(((IGAnimation.FinalT - IGAnimation.StartT)
	    / IGAnimation.Dt) / AnimSteps);
    if ((InstanceCounter = 
	AttrGetObjectIntAttrib(DefObj, "_anim_phase")) == IP_ATTR_BAD_INT)
		AttrSetObjectIntAttrib(DefObj, "_anim_phase", 0);
    else if (InstanceCounter < NumInstances)
	AttrSetObjectIntAttrib(DefObj, "_anim_phase", InstanceCounter + 1);
    else
	AttrSetObjectIntAttrib(DefObj, "_anim_phase",0);
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Calls display list for the second phase of the FFD algorithm	     *
*						   			     *
* PARAMETERS:                                                                *
*   PObj: Object to Draw						     *
*                                                                            *
* RETURN VALUE:                                                              *
*   void.	    			    				     *
*                                                                            *
* KEYWORDS:                                                                  *
*   DrawModelAsGrid						             *
******************************************************************************/
static void DrawModelAsGrid(IPObjectStruct * PObj)
{
    GLuint DisplayListHandle;
    
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glOrtho(0, 1, 0, 1, -1, 1);
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    
    gluLookAt(0, 0, 0,
              0, 0, -10,
              0, 1, 0);
	
    CG_GL_SET_STATE_MATRIX_PARAMETER(CGParamDoublePhase1.ModelViewProjMatrix,
				CG_GL_MODELVIEW_PROJECTION_MATRIX,
				CG_GL_MATRIX_IDENTITY);
    glPointSize(POINT_SIZE);
    
    if ((DisplayListHandle = AttrGetObjectIntAttrib(PObj, "_ffd_grid_disp_lst"))
        != IP_ATTR_BAD_INT) {
	glCallList(DisplayListHandle);
    }
    else {
        IGVertexHandleExtraFuncType OldFunc;
	int OldDrawStyle = IGGlblDrawStyle;

	IGGlblDrawStyle = IG_STATE_DRAW_STYLE_POINTS;
	OldFunc = IGSetHandleVertexProcessingFunc(IGCgVertexHanlingFirstPass); 

	DisplayListHandle  = glGenLists(1);
	AttrSetObjectIntAttrib(PObj, "_ffd_grid_disp_lst", DisplayListHandle);
	glNewList(DisplayListHandle, GL_COMPILE);
    	IGDisplayObject(PObj);
	glEndList();
	
	/* Restore state. */
	IGGlblDrawStyle = OldDrawStyle;
	IGSetHandleVertexProcessingFunc(OldFunc); 
    }
	
    glPopMatrix();
    glMatrixMode(GL_PROJECTION);
    glPopMatrix();
    glMatrixMode(GL_MODELVIEW);
    glPointSize(1);
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Check if ARB is supported on Gpu					     *
* PARAMETERS:                                                                *
*   expresion: ARB expresion to check				    	     *
*                                                                            *
* RETURN VALUE:                                                              *
*   TRUE if supported, FALSE otherwise.					     *
*                                                                            *
* KEYWORDS:                                                                  *
*   IsArbExtensionSupported						     *
*****************************************************************************/
static int IsArbExtensionSupported(const char *extension)
{
    HDC hdc=wglGetCurrentDC();
    const char *extensions = NULL;
    char *start;
    char *where, *terminator;

    /* Get function pointer */
    PFNWGLGETEXTENSIONSSTRINGARBPROC wglGetExtensionsStringARB=
	(PFNWGLGETEXTENSIONSSTRINGARBPROC)
	wglGetProcAddress("wglGetExtensionsStringARB");
    if (wglGetExtensionsStringARB == NULL)
        return FALSE;
    extensions = wglGetExtensionsStringARB(hdc);
    start = (char*) extensions;
    for ( ; ; ) {
        where =  strstr((const char *) start, extension);
        if (!where)
        break;
        terminator = where + strlen(extension);
        if (where == start || *(where - 1) == ' ')
        if (*terminator == ' ' || *terminator == '\0')
            return TRUE;
        start = terminator;
    }
    return FALSE;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Get the color for index on a local context	   			     *
*						   			     *
* PARAMETERS:                                                                *
*	color index								     *
*                                                                            *
* RETURN VALUE:                                                              *
*   void.	 			    				     *
*                                                                            *
* KEYWORDS:                                                                  *
*   SetColorIndex		                                             *
******************************************************************************/
static void SetColorIndex(int color)
{
    STATIC_DATA short Colors[IG_MAX_COLOR + 1][3] =
    {
	{ 0,   0,   0   },  /* 0. BLACK */
	{ 0,   0,   170 },  /* 1. BLUE */
	{ 0,   170, 0   },  /* 2. GREEN */
	{ 0,   170, 170 },  /* 3. CYAN */
	{ 170, 0,   0   },  /* 4. RED */
	{ 170, 0,   170 },  /* 5. MAGENTA */
	{ 170, 170, 0   },  /* 6. BROWN */
	{ 170, 170, 170 },  /* 7. LIGHTGREY */
	{ 85,  85,  85  },  /* 8. DARKGRAY */
	{ 85,  85,  255 },  /* 9. LIGHTBLUE */
	{ 85,  255, 85  },  /* 10. LIGHTGREEN */
	{ 85,  255, 255 },  /* 11. LIGHTCYAN */
	{ 255, 85,  85  },  /* 12. LIGHTRED */
	{ 255, 85,  255 },  /* 13. LIGHTMAGENTA */
	{ 255, 255, 85  },  /* 14. YELLOW */
	{ 255, 255, 255 }   /* 15. WHITE */
    };

    if (color < 0 || color > IG_MAX_COLOR)
        color = IG_IRIT_WHITE;

    ObjColor[0] = Colors[color][0];
    ObjColor[1] = Colors[color][1];
    ObjColor[2] = Colors[color][2];
    ObjColor[3] = 255;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Update the global obj color in a local context  			     *
*						   			     *
* PARAMETERS:                                                                *
*	Object 								     *
*                                                                            *
* RETURN VALUE:                                                              *
*   void.	 			    				     *
*                                                                            *
* KEYWORDS:                                                                  *
*   SetGlbObjColor		                                             *
******************************************************************************/
static void SetGlbObjColor(IPObjectStruct *PObj)
{
    int Color[4] = { 255, 255, 255, 255 };
    int c;

    if (AttrGetObjectRGBColor(PObj, &Color[0], &Color[1], &Color[2])) {
	 memcpy(ObjColor, Color, sizeof(ObjColor));
    }
    else if ((c = AttrGetObjectColor(PObj)) != IP_ATTR_NO_COLOR) {
	SetColorIndex(c);
    }
    else {
	/* Use white as default color: */
	 SetColorIndex(IG_IRIT_WHITE);
    }
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Update parameters for the shader					     *
* PARAMETERS:                                                                *
*   Algorithm: Which algorithm to use				    	     *
*                                                                            *
* RETURN VALUE:                                                              *
*   void.								     *
*                                                                            *
* KEYWORDS:                                                                  *
*   SetControlPoint							     *
*****************************************************************************/
static void UpdateShadingParameters(int Algorithm)
{
    NrmlDir =IGGlblFlipNormalOrient == TRUE ? -1 : 1;
    if (Algorithm == 1) {
	CG_GL_SET_PARAMETER_1F(CGParamSinglePhase.AmbientFactor,
	    IGShadeParam.LightAmbient[0]);
	CG_GL_SET_PARAMETER_1F(CGParamSinglePhase.DiffuseFactor, 
	    IGShadeParam.LightDiffuse[0]);
	CG_GL_SET_PARAMETER_1F(CGParamSinglePhase.SpecularFactor, 
	    IGShadeParam.LightSpecular[0]);
	CG_GL_SET_PARAMETER_1F(CGParamSinglePhase.Shininess,
	    IGShadeParam.Shininess);
	CG_GL_SET_PARAMETER_4F(CGParamSinglePhase.LightPos0 ,
					IGShadeParam.LightPos[0][0],
					IGShadeParam.LightPos[0][1],
					IGShadeParam.LightPos[0][2],
					IGShadeParam.LightPos[0][3]);
	CG_GL_SET_PARAMETER_4F(CGParamSinglePhase.LightPos1 ,
					IGShadeParam.LightPos[1][0],
					IGShadeParam.LightPos[1][1],
					IGShadeParam.LightPos[1][2],
					IGShadeParam.LightPos[1][3]);
    }
    else if (Algorithm == 2)
    {
	
	CG_GL_SET_PARAMETER_1F(CGParamDoublePhase2.AmbientFactor, 
	    IGShadeParam.LightAmbient[0]);
	CG_GL_SET_PARAMETER_1F(CGParamDoublePhase2.DiffuseFactor, 
	    IGShadeParam.LightDiffuse[0]);
	CG_GL_SET_PARAMETER_1F(CGParamDoublePhase2.SpecularFactor,
	    IGShadeParam.LightSpecular[0]);
	CG_GL_SET_PARAMETER_1F(CGParamDoublePhase2.Shininess,
	    IGShadeParam.Shininess); 
	CG_GL_SET_PARAMETER_4F(CGParamDoublePhase2.LightPos0 ,
					IGShadeParam.LightPos[0][0],
					IGShadeParam.LightPos[0][1],
					IGShadeParam.LightPos[0][2],
					IGShadeParam.LightPos[0][3]); 
	CG_GL_SET_PARAMETER_4F(CGParamDoublePhase2.LightPos1 ,
					IGShadeParam.LightPos[1][0],
					IGShadeParam.LightPos[1][1],
					IGShadeParam.LightPos[1][2],
					IGShadeParam.LightPos[1][3]);

	CG_GL_SET_PARAMETER_1F(CGParamDoublePhase2Fragment.AmbientFactor,
	    IGShadeParam.LightAmbient[0]);
	CG_GL_SET_PARAMETER_1F(CGParamDoublePhase2Fragment.DiffuseFactor,
	    IGShadeParam.LightDiffuse[0]);
	CG_GL_SET_PARAMETER_1F(CGParamDoublePhase2Fragment.SpecularFactor,
	    IGShadeParam.LightSpecular[0]);
	CG_GL_SET_PARAMETER_1F(CGParamDoublePhase2Fragment.Shininess,
	    IGShadeParam.Shininess); 
	CG_GL_SET_PARAMETER_4F(CGParamDoublePhase2Fragment.LightPos0 ,
					IGShadeParam.LightPos[0][0],
					IGShadeParam.LightPos[0][1],
					IGShadeParam.LightPos[0][2],
					IGShadeParam.LightPos[0][3]); 
	CG_GL_SET_PARAMETER_4F(CGParamDoublePhase2Fragment.LightPos1 ,
					IGShadeParam.LightPos[1][0],
					IGShadeParam.LightPos[1][1],
					IGShadeParam.LightPos[1][2],
					IGShadeParam.LightPos[1][3]);
    }
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Check if the system global variables have changed, and update Open GL    *
*   tiles display lists if nesseccery.                                       *
*                                                                            *
* PARAMETERS:                                                                *
*   PObj: Objects needs to recompute.	                                     *
*   ForceRecompute: Force Recompute without checking the status		     *
*                                                                            *
* RETURN VALUE:                                                              *
*   TRUE if need to recomputer, FALSE otherwise	                             *
*****************************************************************************/
static int IGCheckSysStatus(IPObjectStruct *PObj, int ForceRecompute)
{
    IPObjectStruct * DefObj;
    STATIC_DATA int DrawStyle,
		    DrawSurfacePoly,
		    DrawSurfaceWire,
		    ShadingModel,
		    FourPerFlat,
		    FlipNormalOrient,
		    PolygonOptiApprox,
		    CountNumPolys,
		    State = 0,
		    Init = 1;
    int	CurrState;
    STATIC_DATA RealType PlgnFineness;

    if ((CurrState = AttrGetObjectIntAttrib(PObj, "State")) != State ||
	(CountNumPolys != IGGlblCountNumPolys) ||
	(DrawStyle != IGGlblDrawStyle) ||
	(DrawSurfacePoly != IGGlblDrawSurfacePoly) ||
	(DrawSurfaceWire != IGGlblDrawSurfaceWire) ||
	(ShadingModel != IGGlblShadingModel) ||
	(FourPerFlat != IGGlblFourPerFlat) ||
	(FlipNormalOrient != IGGlblFlipNormalOrient) ||
	(PolygonOptiApprox != IGGlblPolygonOptiApprox) ||
	(PlgnFineness != IGGlblPlgnFineness) ||
	ForceRecompute) {
	if (CurrState == State || Init) {
	    State ^= 1;
	    Init = 0;
	    if ((PolygonOptiApprox != IGGlblPolygonOptiApprox) || 
		(PlgnFineness != IGGlblPlgnFineness)) {
		RecomputeCacheGeom(PObj, 1, 1, 1, 1);
	    }
	    IGActiveFreeNamedAttribute(PObj, "State");
	    AttrSetObjectIntAttrib(PObj, "State", State);
	    CountNumPolys = IGGlblCountNumPolys;
	    DrawStyle = IGGlblDrawStyle;
	    DrawSurfacePoly = IGGlblDrawSurfacePoly;
	    DrawSurfaceWire = IGGlblDrawSurfaceWire;
	    ShadingModel = IGGlblShadingModel;
	    FourPerFlat = IGGlblFourPerFlat;
	    FlipNormalOrient = IGGlblFlipNormalOrient;
	    PolygonOptiApprox = IGGlblPolygonOptiApprox;
	    PlgnFineness = IGGlblPlgnFineness;
	    RecomputeLists(PObj);	
	    return TRUE;
	}
	else {
	    IGActiveFreeNamedAttribute(PObj, "State");
	    AttrSetObjectIntAttrib(PObj, "State", State);
	    RecomputeLists(PObj);
	    if ((DefObj = 
		AttrGetObjectPtrAttrib(PObj, "_ffd_def_obj")) != NULL)
		RecomputeCacheGeom(DefObj, 1, 1, 1, 1);
	}
    }
    return FALSE;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Deletes FFD attributes, frees open gl lists				     *	
*                                                                            *
* PARAMETERS:                                                                *
*   PObj: Objects needs to recompute.	                                     *
*                                                                            *
* RETURN VALUE:                                                              *
*   void.				   	                             *
*****************************************************************************/
static void RecomputeLists(IPObjectStruct *PObj)
{
    GLuint DisplayListHandle;

    if ((DisplayListHandle = AttrGetObjectIntAttrib(PObj, "_ffd_model_disp_lst"))	
	!= IP_ATTR_BAD_INT) {
	glDeleteLists(DisplayListHandle, 1);
	AttrFreeOneAttribute(&PObj->Attr, "_ffd_model_disp_lst");
    }
    if ((DisplayListHandle = AttrGetObjectIntAttrib(PObj, "_ffd_grid_disp_lst"))	
	!= IP_ATTR_BAD_INT) {
	glDeleteLists(DisplayListHandle, 1);
	AttrFreeOneAttribute(&PObj -> Attr, "_ffd_grid_disp_lst");
    }	    
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   recompute a tesselation for a freeform object on state changes	     *
* PARAMETERS:                                                                *
*								    	     *
*                                                                            *
* RETURN VALUE:                                                              *
*   void.								     *
*                                                                            *
* KEYWORDS:                                                                  *
*   RecomputeCacheGeom							     *
*****************************************************************************/
static void RecomputeCacheGeom(IPObjectStruct * PObj,
			       int FreePolygons,
			       int FreeIsolines,
			       int FreeSketches,
			       int FreeCtlMesh)
{
    if (IP_IS_OLST_OBJ(PObj)) {
	IPObjectStruct *PTmp;
	int i = 0;
	/* Search in its list. */
	while ((PTmp = IPListObjectGet(PObj, i++)) != NULL)
	    RecomputeCacheGeom(PTmp, FreePolygons,
		FreeIsolines, FreeSketches, FreeCtlMesh);
    }
    else if (IP_IS_FFGEOM_OBJ(PObj))
	IGActiveFreePolyIsoAttribute(PObj, FreePolygons,
		FreeIsolines, FreeSketches, FreeCtlMesh);
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   An interface to manipulate the position of a selected group of control   *
*   points. The control points are visually represented as point objects     *
*   with index attributes.						     *
* PARAMETERS:                                                                *
*								    	     *
*                                                                            *
* RETURN VALUE:                                                              *
*   void.								     *
*                                                                            *
* KEYWORDS:                                                                  *
*   SetControlPoint							     *
*****************************************************************************/
static void SetControlPoint(IPObjectStruct *PObj)
{
    int TexId, objCounter;
    GLfloat NewPoint[4] = { 0 };
    TrivTVStruct *Tv = NULL;
    IPObjectStruct *TvObj;
    IPObjectStruct *TObj;

    TexId = AttrGetObjectIntAttrib(PObj, "_ffd_tex_id");
    TvObj = (IPObjectStruct*)AttrGetObjectPtrAttrib(PObj, "_ffd_triv");
    Tv = TvObj -> U.Trivars;
    if (Tv == NULL || IP_ATTR_BAD_INT == TexId)
	return;

    for (objCounter = 0; objCounter < IGObjManipNumActiveObjs; 
	objCounter++) {
	int i, j, k, x, y;
	TObj = IGObjManipCurrentObjs[objCounter];
	i = AttrGetObjectIntAttrib(TObj, "TV_I");
	j = AttrGetObjectIntAttrib(TObj, "TV_J");
	k = AttrGetObjectIntAttrib(TObj, "TV_K");
	if (i == IP_ATTR_BAD_INT || 
	    j == IP_ATTR_BAD_INT || 
	    k == IP_ATTR_BAD_INT)
	    continue;
	x = j;
	y = k * Tv -> ULength + i;
	NewPoint[0] = (GLfloat)TObj -> U.Pt[0];
	NewPoint[1] = (GLfloat)TObj -> U.Pt[1];
	NewPoint[2] = (GLfloat)TObj -> U.Pt[2];
	glBindTexture(GL_TEXTURE_2D, TexId);
	glTexSubImage2D(GL_TEXTURE_2D, 0, x, y, 1, 1, GL_RGBA, 
	    GL_FLOAT, NewPoint);
    }
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   copy the values of deformed positions from the PBuffer to an array	     *	
*   Compute the bounding box and regenerate the trivariate		     *
* PARAMETERS:                                                                *
*    XLen, YLen the number of pixels in the x,y-direction		     *
*                                                                            *
* RETURN VALUE:                                                              *
*   int:  TRUE if successful, FALSE if no go.				     *
*                                                                            *
* KEYWORDS:                                                                  *
*   ApplyDeformedVertices                                                    *
*****************************************************************************/
static void ApplyDeformedVertices(IPObjectStruct *PObj,
				  int XLen, 
				  int YLen)
{
    int i, j;
    GLfloat *DefPositions;

    DefPositions = IGGetPBufferToHost(0, XLen, YLen);
    ObjBBox[1][0] = ObjBBox[0][0] = DefPositions[0];
    ObjBBox[1][1] = ObjBBox[0][1] = DefPositions[1];
    ObjBBox[1][2] = ObjBBox[0][2] = DefPositions[2];
    for (i = 0; i < YLen; i++) {
	for (j = 0; j < XLen; j++) {
	    int idx = 4 * (XLen * i + j);
	    if (ObjBBox[0][0] > DefPositions[idx])
		ObjBBox[0][0] = DefPositions[idx];
	    if (ObjBBox[1][0] < DefPositions[idx])
		ObjBBox[1][0] = DefPositions[idx];

	    if (ObjBBox[0][1] > DefPositions[idx + 1])
		ObjBBox[0][1] = DefPositions[idx + 1];
	    if (ObjBBox[1][1] < DefPositions[idx + 1])
		ObjBBox[1][1] = DefPositions[idx + 1];

	    if (ObjBBox[0][2] > DefPositions[idx + 2])
		ObjBBox[0][2] = DefPositions[idx + 2];
	    if (ObjBBox[1][2] < DefPositions[idx + 2])
		ObjBBox[1][2] = DefPositions[idx + 2];
	}
    }
    IritFree(DefPositions);
    /* Store the new bbox of the deformed object */
    memcpy(PObj -> BBox, ObjBBox, sizeof(ObjBBox));
}

#else

/* Dummy functions to link to if no Open GL CG graphics is available. */
int IGFfdDraw(IPObjectStruct *PObj)
{
    return FALSE;
}

#endif /* HAVE_OGL_CG_LIB */
