/********************************************************************

   Module Name:     SimpleProducerBean.java
   Creation Date:   6/20/98
   Description:     A basic InfoBusDataConsumer that publishes a single
                    MonetaryDataItem on the InfoBus

*********************************************************************/

/*
 * Copyright (c) 1997-1998 Lotus Development Corporation. All Rights Reserved.
 *
 *
 * LOTUS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. LOTUS SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 */

import java.beans.*;
import java.applet.*;
import java.awt.*;
import java.awt.event.*;

import javax.infobus.*;

/*
*   Remember that being an InfoBus participant exposes a portion of your
*   applet to the world -- a SECURITY issue -- and any code reachable by
*   your IB interfaces must be THREADSAFE
*/
public class SimpleProducerBean extends Applet
    implements  InfoBusMember, InfoBusDataProducer,
                PropertyChangeListener, ActionListener
{
    //  SECURITY: use a proxy subobject to implement our InfoBusDataProducer
    //   This interface is handed to other IB participants with events and
    //   data, and the proxy subobject minimizes unwanted introspection
    //
    private transient InfoBusDataProducer     m_producerProxy;

    // use an InfoBusMemberSupport to hold our InfoBus
    private  InfoBusBeanSupport    m_IBHolder;
    
    // Our actual data is in m_stringData. It gets updated when data entry 
    // in the TextField is concluded with the Return key, and also when the 
    // "Send Available" button is pushed.
    // Note m_stringData will be serialized and restored, therefore its 
    // value in start() will be different from the following if we've 
    // been deserialized
    private Float                  m_dollarValue = new Float(0f);

    // For publication on the InfoBus, we put the stringData into a
    // wrapper object that implements DataItem, DataItemChangeManager,
    // and ImmediateAccess
    // As a general rule (especially when publishing complex data)
    // we don't need to instantiate the wrapper object until a consumer 
    // on the InfoBus requests it
    // After its creation and publication, the MonetaryDataItem object will
    // automatically send DataItemChangeEvents if its contents are changed
    private transient MonetaryDataItem          m_dataItem;

    // the name we use to publish our data
    private String                  m_dataName = "DATA1";
    
    // following used to synchronize Available and Revoke events
    private transient Object                  m_AvailRevokeInterlock;
    // and this one to synchronize creation/modification of data/dataItem
    // (use a separate object because m_dollarValue changes its reference
    //  often, and m_dataItem is non-existent until requested)
    private transient Object        m_dataInterlock;


	public void init() 
    {
		super.init();


        // following code generated by a layout tool
		//{{INIT_CONTROLS
		setLayout(null);
		setSize(375,200);
		setBackground(new Color(16711680));
		label1 = new java.awt.Label("SimpleProducerBean");
		label1.setBounds(12,12,228,36);
		label1.setFont(new Font("Dialog", Font.BOLD, 18));
		add(label1);
		textField1 = new java.awt.TextField();
		textField1.setBounds(48,84,312,44);
		textField1.setFont(new Font("Dialog", Font.BOLD, 20));
		textField1.setBackground(new Color(16777215));
		add(textField1);
		buttonAvail = new java.awt.Button();
		buttonAvail.setLabel("Send Available");
		buttonAvail.setBounds(36,144,132,28);
		buttonAvail.setFont(new Font("Dialog", Font.BOLD, 16));
		buttonAvail.setBackground(new Color(12632256));
		add(buttonAvail);
		buttonRevoke = new java.awt.Button();
		buttonRevoke.setLabel("Send Revoke");
		buttonRevoke.setBounds(216,144,132,28);
		buttonRevoke.setFont(new Font("Dialog", Font.BOLD, 16));
		buttonRevoke.setBackground(new Color(12632256));
		add(buttonRevoke);
		label2 = new java.awt.Label("Price in U.S. Dollars:");
		label2.setBounds(24,48,313,29);
		label2.setFont(new Font("Dialog", Font.PLAIN, 18));
		add(label2);
		label3 = new java.awt.Label("$",Label.RIGHT);
		label3.setBounds(12,84,32,47);
		label3.setFont(new Font("Dialog", Font.BOLD, 18));
		add(label3);
		//}}
        // end generated code

        // the "this" in the following constructor tells the 
        // support to use this object in the source field of all
        // events it issues
        m_IBHolder = new InfoBusBeanSupport ( this );

        constructTransients();
        
        // hook up AWT1.1 events
        buttonAvail.addActionListener ( this );
        buttonRevoke.addActionListener ( this );
        textField1.addActionListener ( this );

	}


    private void readObject(java.io.ObjectInputStream stream)
        throws java.io.IOException, ClassNotFoundException
    {
        stream.defaultReadObject();
        constructTransients();
    }
            

    /*
    * Build or rebuild the objects and connections that are not serialized
    */
    private void constructTransients()
    {
        // create or reconstruct transients (if we had been serialized)
        m_AvailRevokeInterlock = new Object();
        m_dataInterlock = new Object();
        m_producerProxy = new InfoBusDataProducerProxy ( this );

        // register as a PropertyChangeListener with our IBMSupport
        // so that we are alerted if an external agent successfully
        // changes our InfoBus via setInfoBus
        m_IBHolder.addInfoBusPropertyListener ( this );

    }
    
    /*
    * In our applet's start method we rebuild our InfoBus connection using
    * the transient m_IBHolder.  We request a named InfoBus for this applet
    * and join it, all via the IBMemberSupport.joinInfoBus method.  If the join
    * is successful, a side effect will be a PropertyChangeEvent showing our 
    * InfoBus property changing from null to X -- and this in turn causes us
    * to add our ProducerProxy to X in our propertyChange method
    *
    * On many applets (including this one) if a null infoBusName is detected
    * by the start() method the applet joins a default bus (this is a feature
    * of many applets but is not mandated by the InfoBus specification)
    */
    public void start ()
    {
        try
        {
            if ( m_IBHolder.getInfoBusName() == null || m_IBHolder.getInfoBusName().equals("") )
            {
                //gets default bus, will set both infoBus and infoBusName props
                m_IBHolder.joinInfoBus( this ); 
            }
            else
            {
                //tells the IBBSupport to reattach to the InfoBus it has a name for
                //  (no effect if already attached, but at this point in code we aren't)
                m_IBHolder.rejoinInfoBus( ); 
            }
        }
        catch (Exception e)
        {
            System.out.println ( "SimpleConsumerBean.init Error "+
                                    "intializing its InfoBus:  " + e);
            System.exit(1);
        }
    }


    /*
    * In our applet's stop method we tell our IBMemberSupport to leave the 
    * InfoBus it was holding.  If successful, a side effect will be a 
    * PropertyChangeEvent showing our InfoBus property changing from X to 
    * null, which in turn causes us to remove our ProducerProxy
    *
    * Note that we do NOT reset our infoBusName property!  This is so we 
    * rejoin the same bus if we are restarted -- including after 
    * deserialization.
    */
    public void stop ()
    {
        // tell the IBMSupport to leave its bus
        try
        {
            m_IBHolder.leaveInfoBus();
        }
        catch ( PropertyVetoException pve )
        {
            // do nothing -- our InfoBus property is managed by external bean
        }
        catch ( InfoBusMembershipException ibme )
        {
            // do nothing -- simply means we already had null for IB
        }
    }
    

    ////////////////////////////////////////////////
    //  InfoBusBean methods not in InfoBusMember
    ////////////////////////////////////////////////
    /**
    * Updates the value of InfoBus by setting a new infoBusName.  If the InfoBus
    * cannot be changed, the set will throw an Exception
    * Setting the infoBusName to null causes this bean to leave its current
    * bus, resulting in both the InfoBus and infoBusName properties being null.
    *
    * On many applets (including this one) if a null infoBusName is detected
    * by the start() method the applet joins a default bus (this is a feature
    * of many applets but is not mandated by the InfoBus specification)
    *
    * @exception InfoBusMembershipException if InfoBus cannot be changed
    */
    public void setInfoBusName ( String newName )
        throws InfoBusMembershipException
    {
        m_IBHolder.setInfoBusName( newName );
    }
    
    public String getInfoBusName()
    {
        return m_IBHolder.getInfoBusName();
    }



    ////////////////////////////////////////////
    //  InfoBusMember methods
    ////////////////////////////////////////////

    // Delegate all calls to our InfoBusMemberSupport object, m_IBHolder

    public InfoBus getInfoBus()
    {
        return m_IBHolder.getInfoBus();
    }

    public void setInfoBus (InfoBus b) throws PropertyVetoException
    {
        m_IBHolder.setInfoBus ( b );
    }

    public void addInfoBusVetoableListener(VetoableChangeListener l)
    {
        m_IBHolder.addInfoBusVetoableListener ( l );
    }

    public void removeInfoBusVetoableListener(VetoableChangeListener l)
    {
        m_IBHolder.removeInfoBusVetoableListener ( l );
    }
    
     public void addInfoBusPropertyListener(PropertyChangeListener l)
    {
        m_IBHolder.addInfoBusPropertyListener ( l );
    }

    public void removeInfoBusPropertyListener(PropertyChangeListener l)
    {
        m_IBHolder.removeInfoBusPropertyListener ( l );
    }


    /////////////////////////////////////////////////////////
    //  Reproduced versions of InfoBusDataProducer methods
    //    to be called by ProducerProxy
    /////////////////////////////////////////////////////////

    /*
    * A PropertyChangeEvent is issued by any InfoBusMember if it
    * is changing the InfoBus to which it is connected.  Since we
    * use an InfoBusMemberSupport object to hold and administer our
    * InfoBus connection, we take advantage of this fact to track
    * a change to our own InfoBus and move our Producer Proxy
    */
    public void propertyChange ( PropertyChangeEvent pce )
    {
        // Note: this implementation is written for a PropertyChangeListener
        //  that may be listening to multiple properties and/or sources

        String s = pce.getPropertyName();

        // check if the PCE refers to an InfoBus
        if ( s.equalsIgnoreCase("InfoBus") )
        {
            // we are only concerned with moving our ProducerProxy
            // if an InfoBus PCE came from our own IBMemberSupport!
            if ( pce.getSource() == this )
            {
                Object oldVal = pce.getOldValue();
                Object newVal = pce.getNewValue();

                //note that either old or new value may validly be NULL

                if ( oldVal != null && oldVal instanceof InfoBus )
                {
                    ((InfoBus) oldVal).removeDataProducer ( m_producerProxy );
                }

                if ( newVal != null && newVal instanceof InfoBus )
                {
                    ((InfoBus) newVal).addDataProducer ( m_producerProxy );
                }
            }
            // put code here if we're watching other objects' IB properties
        }
        // put code here if we're watching other properties in the system
    }


    /*
    * Event delivery method for Producers -- the event is a request
    * for named data made by a consumer
    * SECURITY:  The Producer is responsible for determining whether
    * the data may be released;  the calling consumer is in the call
    * stack
    */
    public void dataItemRequested ( InfoBusItemRequestedEvent ibe )
    {
        if ( ibe == null )
        {
            return;
        }

        String s = ibe.getDataItemName();
        if ( ( null != s  ) && s.equals( m_dataName ) )
        {
            // THREADSAFETY: make our activity on a positive match thread safe
            synchronized (m_dataInterlock) // NEVER USE OUR INFOBUS AS THE LOCK OBJECT
            {
                if ( m_dataItem == null )
                {
                    m_dataItem = new MonetaryDataItem( m_dollarValue, m_dataName,
                                                        m_producerProxy );
                }
                ibe.setDataItem( m_dataItem );
            }
        }
        
    }


    ////////////////////////////
    // ActionListener Methods
    ////////////////////////////

    public void actionPerformed ( ActionEvent e )
    {
        // java.awt.Buttons send ActionEvent with their label
        String s = e.getActionCommand();

        if ( s == null )
        {
            return;
        }

        if ( s.equals( "Send Available" ) )
        {
            availClicked();
        }
        else if ( s.equals( "Send Revoke" ) )
        {
            revokeClicked();
        }
        else
        {
            textChanged();
        }

    }
    
    private void parseTextField()
    {
        String s = textField1.getText();
        //save last value in case an error occurs
        Float oldVal = m_dollarValue;
        
        try
        {
            m_dollarValue = Float.valueOf( s );
        }
        catch ( NumberFormatException e )
        {
            m_dollarValue = oldVal;
            textField1.setText(m_dollarValue.toString());
        }
    }
    
    
    /*
    * Text changed in the TextField, so update our DataItem
    */
    private void textChanged ()
    {
        // THREADSAFETY:  make our data manipulations threadsafe
        synchronized ( m_dataInterlock )
        {
            // get the textField, store in m_dollarValue
            parseTextField();
            
            // calling setDollarValue on a MonetaryDataItem automatically sends
            //  a DataItemChangeEvent if any listeners are present
            if (m_dataItem !=null)
            {
                m_dataItem.setDollarValue( m_dollarValue );
            }
        }
    }

    /*
    * Send an InfoBusItemAvailableEvent for our data
    */
    private void availClicked ()
    {
        // THREADSAFETY:  make our data manipulations threadsafe
        synchronized ( m_dataInterlock )
        {
            // get the textField, store in m_dollarValue
            parseTextField();

            if ( m_dataItem != null )
            {
                m_dataItem.setDollarValue( m_dollarValue );
            }
        }
        
        // THREADSAFETY: create a lock to prevent a REVOKE from being sent 
        // before the AVAILABLE has completed
        synchronized ( m_AvailRevokeInterlock )
        {
            InfoBus ib = m_IBHolder.getInfoBus();
            ib.fireItemAvailable( m_dataName, null, m_producerProxy );
        }
            
    }

    private void revokeClicked ()
    {
        // THREADSAFETY: use interlock to prevent REVOKE if AVAIL outstanding
        synchronized ( m_AvailRevokeInterlock )
        {
            InfoBus ib = m_IBHolder.getInfoBus();
            ib.fireItemRevoked( m_dataName, m_producerProxy );
        }
    }


    public Dimension getMinimumSize()
    {
        return new Dimension( 350, 200 );
    }
    
    
    // following generated by layout tool
	//{{DECLARE_CONTROLS
	java.awt.Label label1;
	java.awt.TextField textField1;
	java.awt.Button buttonAvail;
	java.awt.Button buttonRevoke;
	java.awt.Label label2;
	java.awt.Label label3;
	//}}
}
