/********************************************************************

   Module Name:     SimpleDataItem.java
   Creation Date:   8/28/97
   Description:     ImmediateAccess wrapper for a generic Object

*********************************************************************/

/*
 * Copyright (c) 1997 Sun Microsystems, Inc. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 *
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 */




import javax.infobus.*;
import java.util.Vector;
import java.util.Enumeration;

public class SimpleDataItem implements ImmediateAccess, DataItem, 
    DataItemChangeManager, InfoBusPropertyMap
{

    protected Object                m_value;
    protected Vector            m_changedListeners = null;
    protected InfoBusEventListener  m_source;

    public SimpleDataItem( Object v, InfoBusEventListener source )
    {
        m_value = v;
        m_source = source;
    }

    /** 
    * Sets the Value property and sends a DataItemChangeEvent to 
    * all registered DataItemChangeListeners
    */
    public void setValue( Object v )
    {
        m_value = v;
        if ( null != m_changedListeners )
        {
            DataItemChangeListener listener;
            DataItemValueChangedEvent dice = 
                new DataItemValueChangedEvent(this, this, null);

            Vector listCopy = (Vector) m_changedListeners.clone();
            Enumeration enum = listCopy.elements();
            while ( enum.hasMoreElements() )
            {
                listener = (DataItemChangeListener) enum.nextElement();
                listener.dataItemValueChanged(dice);
            }
        }
    }


    /**
    * Copies the data from newValue as an Object, and sets
    * the data of this SimpleDataItem to that Object
    * @param newValue the new data to be copied
    */
    public void setValue ( ImmediateAccess newValue )
    {
        Object value = newValue.getValueAsObject();

        setValue ( value );
    }


    //Methods from ImmediateAccess interface:
    // getValueAsObject, getValueAsString, getPresentationString
    /**
    * Returns the data as an Object
    * @return the data encapsulated in this SimpleDataItem
    */
    public Object getValueAsObject()
    {
        return m_value;
    }

    /**
    * Returns the data in String form by calling toString on the wrapped Object
    * @return the data encapsulated in this SimpleDataItem
    */
    public String getValueAsString()
    {
        return m_value != null ? m_value.toString() : new String("");
    }

    /**
    * Returns the data in String form by calling toString on the wrapped Object
    * @return the data encapsulated in this SimpleDataItem
    */
    public String getPresentationString( java.util.Locale locale)
    {
        return  m_value != null ? m_value.toString() : new String("");
    }

    //methods from DataItemChangeManager:  add[remove]DataItemChangeListener, 
    /**
    * Registers a DataItemChagnedListener with the SimpleDataItem so that
    * future changes will send a DataItemChangeEvent to the Listener
    * @param l the Listener registering for future change notifications
    */
   	public void	addDataItemChangeListener( DataItemChangeListener l )
    {
		//m_changedListeners vector is not instanced until first addDICL call
		if ( null == m_changedListeners ) 
        {
			m_changedListeners = new Vector();
		}
		m_changedListeners.addElement(l);
	}

    /**
    * Unregisters l from the list of Listeners to which change notifications
    * are sent
    * @param l the Listener to be removed from the notifications list
    */
	public void	removeDataItemChangeListener(DataItemChangeListener l)
    {
		//m_changedListeners is null until first addDICL call
		if ( null != m_changedListeners )
        {
			m_changedListeners.removeElement(l);
		}
	}

    //methods from DataItem: getTransferable, getSource, getProperty
    /**
    * SimpleDataItem does not support clipboard transfers
    * @return null to indicate non-support
    */
    public java.awt.datatransfer.Transferable getTransferable ()
    {
        return null;
    }

    /**
    * Returns the InfoBusEventListener which offers this DataItem on the
    * InfoBus
    * @return the InfoBusEventListener registered as the source on 
    * this DataItem
    */
    public InfoBusEventListener getSource()
    {
        return m_source;
    }


    /**
    * SimpleDataItem does not support any properties, so returns null
    * regardless of propertyName
    * @param propertyName the name of the property desisred
    * @return always null because SimpleDataItem supports no properties
    */
    public Object getProperty( String propertyName )
    {
        return null;
    }

	/**
    * SimpleDataItem does not support any properties, so returns null
    * regardless of propertyName
    * @param propertyName the name of the property desisred
    * @return always null because SimpleDataItem supports no properties
    */
    public Object get( Object key )
    {
        return null;
    }

	/** Not Supported
	* @return null because not supported.
    */
    public java.awt.datatransfer.DataFlavor[] getDataFlavors()
	{
		return null;
	}


	/**
	* Not supported, no effect.
	*/
	public void release()
	{
	}


}

/*-------------------------------------------------------------------------
	Change History:

	$Log:   //reebok/xyzL/JavaComp/javax/infobus/examples/dbaccess/SimpleDataItem.java  $
 * 
 *    Rev 1.4   08 Feb 1998 16:57:38   ckarle
 * Update to TP3 InfoBus spec
  
     Rev 1.2.2.2   15 Jan 1998 17:48:12   tmcgary
  dataItemChangeEvent -> dataItemValueChangedEvent.
  
     Rev 1.2   23 Nov 1997 02:04:32   ckarle
  Revised to IBMember v1.3 and spec 0.06 DICE
  
     Rev 1.1   12 Nov 1997 16:23:56   ckarle
  Upgrade to 05 spec
  
     Rev 1.1   06 Aug 1997 16:56:10   ckarle
  Revised to IB Spec 0.04
  
     Rev 1.0   11 Jun 1997 15:56:20   rmorreal
  Initial Revision
  
     Rev 1.3   28 Feb 1997 15:54:50   cgrindel
  Kona Source Reconstruction [Charles Grindel 2/28/97]
  
     Rev 1.2   15 Jan 1997 16:00:48   ckarle
  Minor edits to curly braces to fix debugger bug
  
     Rev 1.1   11 Jan 1997 14:08:28   ckarle
  Fixed bug in setValue method
  
     Rev 1.0   08 Jan 1997 15:16:46   ckarle
  Initial Revision
  
     Rev 1.1   08 Nov 1996 15:37:14   dwilson
  Modified references to member variables that are also method names.
  
     Rev 1.0   04 Nov 1996 08:43:30   dwilson
  Initial Revision
---------------------------------------------------------------------------*/
