
#ifndef object_h
#define object_h

#include <Str.h>

#include <List.h>
#include <AVLTree.h>
#include "NibStr.h"

class SourceLine; // for the time of assembly, a Symbol may hold a reference
                  // to a source-line rather than a value.

class Symbol {
	
public:
	
	Str name;
	
	union {
		unsigned long long ivalue;
		double fvalue;
		SourceLine * reference;
	};
	
	bool resolved;
	bool numerical;
	bool accessed;
	bool external;
	bool isfloat;

	Symbol (const Symbol & o) : name(o.name), ivalue(o.ivalue),
	  resolved(o.resolved), numerical(o.numerical), accessed(o.accessed),
	  external(o.external), isfloat(o.isfloat) { };
	
	Symbol (void) : ivalue(0), resolved(false), numerical(false),
	                accessed(false), external(false), isfloat(false) { };

	Symbol ( const Str & n, double v ) :
					name(n), fvalue(v), resolved(true), accessed(false),
					external(false), isfloat(true) { };

	Symbol ( const Str & n,  unsigned long long v ) :
					name(n), ivalue(v), resolved(true), accessed(false),
					external(false), isfloat(false) { };
	
	Symbol (const Str & n) : name(n), resolved(false), accessed(false),
	                         external(false), isfloat(false) { };
	
	int compare(const Symbol & rv) const { return name.compare(rv.name); };
	
	~Symbol() { };
	
};

class BOstream;
class BIstream;

BOstream & operator<< (BOstream & out, const Symbol & sym);
BIstream & operator>> (BIstream & in, Symbol & sym);

// class representing one entry in the relocation table
class Reloc {

public:
	unsigned long offset;
	
	Reloc(void) : offset(0) { };
	Reloc(unsigned long o) : offset(o) { };
	Reloc(const Reloc & rv) : offset(rv.offset) { };
	~Reloc (void) { } ;
};

BOstream & operator<< (BOstream & out, const Reloc & r);
BIstream & operator>> (BIstream & in, Reloc & r);

class Import {

public:
	
	unsigned long offset;
	union {
		int nibsize;
		int targetlib; // this is used by clld to store the target-lib # of
		               // an import into a shared library. nibsize is in this
		               // case >= 5
	};
	Str name;
	bool reloc_demand;

	Import(void) : offset(0), nibsize(0), reloc_demand(false) { };
	
	Import (unsigned long o, int s, const Str & n, bool rd) : offset(o), nibsize(s),
	         name(n), reloc_demand(rd) { } ;

	Import (const Import & rv) : offset(rv.offset), nibsize(rv.nibsize),
	          name(rv.name), reloc_demand(rv.reloc_demand) { };

};

BOstream & operator<< (BOstream & out, const Import & i);
BIstream & operator>> (BIstream & in, Import & i);

extern template class ListItem<Reloc>;
extern template class List<Reloc>;

extern template class ListItem<Import>;
extern template class List<Import>;

extern template class AVLItem<Symbol>;
extern template class AVLTree<Symbol>;

class ObjectFile {

public:
	
	Str name;
	
	List<Reloc> relocs;
	List<Import> imports; // imports.length() == 0 => all imports are resolved
	AVLTree<Symbol> exports;
	
	NibStr body;
	
	bool shared;   // true => this is a shared object, do not link immediately
	bool used;     // true => need to link this object into the executable
	
	char fail;
	
	ObjectFile (void) : shared(false), used(false), fail(0) { };
	
	~ObjectFile (void) { };

};

BOstream & operator<< (BOstream & out, const ObjectFile & of);
BIstream & operator>> (BIstream & in, ObjectFile & of);

#endif object_h

