/**
 \brief
  This module implements the gnocl::aboutDialog widget.
 \authors
  Peter G. Baum, William J Giddings
 \date 2001-03:
*/

/** \page page1 gnocl::aboutDialog
  \section sec DESCRIPTION
  Implementation of...
  \subsection subsection1 Implemented Options
\verbatim
 *  -allowGrow
 *      type: boolean
 *      Whether the window may be greater than its children require.
 *  -allowShrink
 *      type: boolean
 *      Whether the window can be made smaller than the place that its children require.
 *  -buttons
 *      type: list (default: %#Ok)
 *      List of buttons to be displayed. An element of the list can be either the name of a stock
 *      button ("%#Cancel"), pure text, a list of a button ID created via the gnocl::button command
 *      and a value which shall be returned if this button is selected, or a list of text (either
 *      pure text or a PERCENTSTRING) and a stock icon ("%#Ok"), in which case the value of the stock
 *      icon is returned, if the button is selected.
 *  -child
 *      type: widget-ID
 *      The widget to be shown. This option or -text is required.
 *  -defaultButton
 *      type: integer
 *      Selects the default button which can normally be chosen by simply pressing return.
 *      The last button is selected per default (see HIG).
 *  -defaultHeight
 *      type: integer
 *      Default height of the window.
 *  -defaultWidth
 *      type: integer
 *      Default width of the window.
 *  -dragTargets
 *      type: list of strings
 *      List of source targets (e.g. text/plain or application/x-color) which are supported.
 *  -dropTargets
 *      type: list of strings
 *      List of destination targets (e.g. text/plain or application/x-color) which are supported.
 *  -icon
 *      type: percent-string (default: "")
 *      Sets the icon of the dialog which is shown, depending on the windowmanager,
 *      in the window decoration, the window list, and/or if the window is iconified. This must be
 *      either a file name (prefix "%/") or empty.
 *  -modal
 *      type: boolean (default: 1)
 *      Whether the window is modal, i.e. it grabs all GTK+ events.
 *  -name
 *      type: string
 *      Name of the widget, can be used to set options in an rc file.
 *  -onDelete
 *      type: string (default: "")
 *      Tcl command which is executed if the widget shall be deleted. If the command returns 0,
 *      the widget is not deleted. Before evaluation the following percent strings are
 *      substituated: TABLE %% | % %w | widget name. TABLE
 *  -onDestroy
 *      type: string (default: "")
 *      Tcl command which is executed if the widget is destroyed. Before evaluation the following
 *      percent strings are substituated: TABLE %% | % %w | widget name. TABLE
 *  -onDragData
 *      type: string (default: "")
 *      Tcl command which is executed if data is draged from this the widget to another.
 *      This command must return the data to be draged. Before evaluation the following percent
 *      strings are substituated: TABLE %% | % %w | widget name %t | time %T | type of data TABLE
 *  -onDropData
 *      type: string (default: "")
 *      Tcl command which is executed if data is dropped on the widget. Before evaluation the
 *      following percent strings are substituated:
 *      TABLE %% | % %w | widget name %d | data %l | length of data %t | time %T | type of data %x | x coordinate %y | y coordinate TABLE
 *  -onKeyPress
 *      type: string (default: "")
 *      Tcl command which is executed if a key is pressed while the widget is having the focus.
 *      Before evaluation the following percent strings are substituated:
 *      TABLE %% | % %w | widget name %k | key code as integer %K | key code as symbol %a | unicode unicode character,
 *      or the empty string if there is no corresponding character. %s | state of the buttons and modifiers (bitmask) TABLE
 *  -onKeyRelease
 *      type: string (default: "")
 *      Tcl command which is executed if a key is released while the widget is having the focus.
 *      Before evaluation the following percent strings are substituated:
 *      TABLE %% | % %w | widget name %k | key code as integer %K | key code as symbol %a | unicode unicode character,
 *      or the empty string if there is no corresponding character. %s | state of the buttons and modifiers (bitmask) TABLE
 *  -onPopupMenu
 *      type: string (default: "")
 *      Tcl command which is executed if the "popup-menu" signal is recieved, which is normally the
 *      case if the user presses Shift-F10. Before evaluation the following percent strings are substituated: %w by widget name.
 *  -onRealize
 *      type: string (default: "")
 *      Tcl command whih is executed in the global scope if the widget has been realized.
 *      Before evaluation the following percent strings are substituated: TABLE %% | % %w | widget name. TABLE
 *  -onResponse
 *      type: string (default: "break")
 *      Tcl command which is executed if a button is chosen.
 *      If the command returns the return code TCL_BREAK, as it is the default, the dialog is closed.
 *      onResponse returns TCL_BREAK if either the last command of onResponse is "break",
 *      or the tcl procedure which is called as last command in onResponse exits with "return -code break".
 *      Before evaluation the following percent strings are substituted:
 *      TABLE %% | % %v | text of the button chosen, or "#NONE" or "#DELETE" TABLE
 *  -onShowHelp
 *      type: string (default: "")
 *      Tcl command which is executed in the global scope if the "show-help" signal is recieved,
 *      which is normally the case if the user presses F1 or Ctrl-F1. Before evaluation the following percent
 *      strings are substituated TABLE %% | % %w | widget name %h | help type: either "whatsThis" or "tooltip" TABLE
 *  -resizable
 *      type: boolean (default: 1)
 *      Whether the user can change the size of the window.
 *  -text
 *      type: string
 *      The text to be shown. This option or -child is required.
 *  -title
 *      type: string (default: "")
 *      Title of the dialog.
 *  -tooltip
 *      type: string
 *      Message that appear next to this widget when the mouse pointer is held over it for a short amount of time.
 *  -type
 *      type: ONEOF info warning question error
 *      Type of the dialog. This determines the icon which is displayed. If the option -text is given,
 *      the default is info, if option -child is given, the default is to display no icon.
 \endverbatim
  \subsection subsection2 Implemented Commands
\verbatim
 *  id delete
 *   Deletes the widget and the associated tcl command.
\endverbatim
  \subsection subsection3 EXAMPLE
\code
 * proc responseProc { val } {
 *  if { "Yes" == $val } {
 *     return -code break
 *  } else {
 *     gnocl::dialog -type question -text "Are you sure?" -buttons %#No
 *  }
 *  }
 * gnocl::dialog -type question -text "Do you like Gnocl?" -buttons "%#No {{%_I _like it} %#Yes}" -onResponse "responseProc %v"
\endcode
*/
/*
 * $Id: aboutDialog.c,v 1.1 2005/08/16 20:57:45 baum Exp $
 *
 * This file implements the gtk dialog and messageDialog
 *
 * Copyright (c) 2001 - 2005 Peter G. Baum  http://www.dr-baum.net
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 */

/*
   History:
   2005-07: Begin of developement
 */

#include "gnocl.h"
#include <string.h>
#include <assert.h>

/* declarations */
static int optStrv ( Tcl_Interp *interp, GnoclOption *opt,
					 GObject *obj, Tcl_Obj **ret );

/****v* aboutDialg/dialogOptions
 * AUTHOR
 *  PGB
 * SOURCE
 */
static GnoclOption dialogOptions[] =
{
	{ "-logo", GNOCL_OBJ, NULL },
	{ "-artists", GNOCL_LIST, "artists", optStrv },
	{ "-authors", GNOCL_LIST, "authors", optStrv },
	{ "-comments", GNOCL_STRING, "comments" },
	{ "-copyright", GNOCL_STRING, "copyright" },
	{ "-documenters", GNOCL_LIST, "documenters", optStrv },
	{ "-license", GNOCL_STRING, "license" },
	{ "-version", GNOCL_STRING, "version" },
	{ "-website", GNOCL_STRING, "website" },

	{ "-allowGrow", GNOCL_BOOL, "allow-grow" },
	{ "-allowShrink", GNOCL_BOOL, "allow-shrink" },
	{ "-defaultHeight", GNOCL_INT, "default-height" },
	{ "-defaultWidth", GNOCL_INT, "default-width" },
	{ "-dragTargets", GNOCL_LIST, "s", gnoclOptDnDTargets },
	{ "-dropTargets", GNOCL_LIST, "t", gnoclOptDnDTargets },
	{ "-icon", GNOCL_OBJ, "", gnoclOptIcon },
	{ "-name", GNOCL_STRING, "name" },
	{ "-onDelete", GNOCL_OBJ, "", gnoclOptOnDelete },
	{ "-onDestroy", GNOCL_OBJ, "destroy", gnoclOptCommand },
	{ "-onDragData", GNOCL_OBJ, "", gnoclOptOnDragData },
	{ "-onDropData", GNOCL_OBJ, "", gnoclOptOnDropData },
	{ "-onKeyPress", GNOCL_OBJ, "", gnoclOptOnKeyPress },
	{ "-onKeyRelease", GNOCL_OBJ, "", gnoclOptOnKeyRelease },
	{ "-onPopupMenu", GNOCL_OBJ, "popup-menu", gnoclOptCommand },
	{ "-onRealize", GNOCL_OBJ, "realize", gnoclOptCommand },
	{ "-onResize", GNOCL_OBJ, "", gnoclOptOnConfigure },
	{ "-onShowHelp", GNOCL_OBJ, "", gnoclOptOnShowHelp },
	{ "-resizable", GNOCL_BOOL, "resizable" },
	{ "-sensitive", GNOCL_BOOL, "sensitive" },
	{ "-title", GNOCL_STRING, "title" },
	{ "-typeHint", GNOCL_OBJ, "", gnoclOptWindowTypeHint },
	{ "-tooltip", GNOCL_OBJ, "", gnoclOptTooltip },
	{ NULL }
};
/*****/

/****v* aboutDialg/constants
 * AUTHOR
 *  PGB
 * SOURCE
 */
const int logoIdx = 0;
/*****/

/****f* aboutDialg/static/optStrv
 * AUTHOR
 *  PGB
 * SOURCE
 */
static int optStrv ( Tcl_Interp *interp, GnoclOption *opt,
					 GObject *obj, Tcl_Obj **ret )
{
	if ( ret == NULL ) /* set value */
	{
		typedef char *charp;
		int  no;
		int  k;
		char **strv;

		Tcl_ListObjLength ( interp, opt->val.obj, &no );
		strv = g_new ( charp, no + 1 );

		for ( k = 0; k < no; ++k )
		{
			Tcl_Obj *pobj;

			if ( Tcl_ListObjIndex ( interp, opt->val.obj, k, &pobj ) != TCL_OK )
			{
				g_free ( strv );
				return TCL_ERROR;
			}

			strv[k] = Tcl_GetString ( pobj );
		}

		strv[no] = NULL;

		g_object_set ( obj, opt->propName, strv, NULL );
		g_free ( strv );
	}

	else /* get value */
	{
		gchar **strv;
		int   no;
		int   k;

		g_object_get ( obj, opt->propName, &strv, NULL );

		for ( no = 0; strv[no] != NULL; ++no )
			;

		*ret = Tcl_NewListObj ( 0, NULL );

		for ( k = 0; k < no; ++k )
			Tcl_ListObjAppendElement ( NULL, *ret,
									   Tcl_NewStringObj ( strv[k], -1 ) );

		g_strfreev ( strv );
	}

	return TCL_OK;
}

/*****/

/****f* static/configure
 * AUTHOR
 *  PGB
 * SOURCE
 */
static int configure ( Tcl_Interp *interp, GtkAboutDialog *dialog,
					   GnoclOption options[] )
{
	if ( options[logoIdx].status == GNOCL_STATUS_CHANGED )
	{
		GdkPixbuf *pix;

		if ( gnoclGetStringType ( options[logoIdx].val.obj ) != GNOCL_STR_FILE )
		{
			Tcl_SetResult ( interp, "Logo must be of file type", TCL_STATIC );
			return TCL_ERROR;
		}

		pix = gnoclPixbufFromObj ( interp, options + logoIdx );

		if ( pix == NULL )
			return TCL_ERROR;

		gtk_about_dialog_set_logo ( dialog, pix );
	}

	return TCL_OK;
}

/*****/

/****f* static/cget
 * AUTHOR
 *  PGB
 * SOURCE
 */
static int cget ( Tcl_Interp *interp, GtkLabel *label,
				  GnoclOption options[], int idx )
{
	Tcl_Obj *obj = NULL;

	return gnoclCgetNotImplemented ( interp, options + idx );
}

/*****/

/****f* function/dialogFunc
 * AUTHOR
 *  PGB
 * SOURCE
 */
static int dialogFunc ( ClientData data, Tcl_Interp *interp,
						int objc, Tcl_Obj * const objv[] )
{

	static const char *cmds[] = { "delete", "configure", "cget", NULL };
	enum cmdIdx { DeleteIdx, ConfigureIdx, CgetIdx };
	int idx;
	GtkLabel *dialog = ( GtkLabel * ) data;

	if ( objc < 2 )
	{
		Tcl_WrongNumArgs ( interp, 1, objv, "command" );
		return TCL_ERROR;
	}

	if ( Tcl_GetIndexFromObj ( interp, objv[1], cmds, "command",
							   TCL_EXACT, &idx ) != TCL_OK )
		return TCL_ERROR;

	switch ( idx )
	{
		case DeleteIdx:
			return gnoclDelete ( interp, GTK_WIDGET ( dialog ), objc, objv );

		case ConfigureIdx:
			{
				int ret = TCL_ERROR;

				if ( gnoclParseAndSetOptions ( interp, objc - 1, objv + 1,
											   dialogOptions, G_OBJECT ( dialog ) ) == TCL_OK )
				{
					ret = configure ( interp, dialog, dialogOptions );
				}

				gnoclClearOptions ( dialogOptions );

				return ret;
			}

			break;
		case CgetIdx:
			{
				int     idx;

				/* kill( 0, SIGINT ); */

				switch ( gnoclCget ( interp, objc, objv, G_OBJECT ( dialog ),
									 dialogOptions, &idx ) )
				{
					case GNOCL_CGET_ERROR:
						return TCL_ERROR;
					case GNOCL_CGET_HANDLED:
						return TCL_OK;
					case GNOCL_CGET_NOTHANDLED:
						return cget ( interp, dialog, dialogOptions, idx );
				}
			}
	}

	return TCL_OK;
}

/*****/

/****f* function/gnoclAboutDialogCmd
 * AUTHOR
 *  PGB
 * SOURCE
 */
int gnoclAboutDialogCmd ( ClientData data, Tcl_Interp *interp,
						  int objc, Tcl_Obj * const objv[] )
{
	int            ret;
	GtkAboutDialog *dialog;

	if ( gnoclParseOptions ( interp, objc, objv, dialogOptions ) != TCL_OK )
	{
		gnoclClearOptions ( dialogOptions );
		return TCL_ERROR;
	}

	dialog = GTK_ABOUT_DIALOG ( gtk_about_dialog_new( ) );

	ret = gnoclSetOptions ( interp, dialogOptions, G_OBJECT ( dialog ), -1 );

	if ( ret == TCL_OK )
		ret = configure ( interp, dialog, dialogOptions );

	gnoclClearOptions ( dialogOptions );

	if ( ret != TCL_OK )
	{
		gtk_widget_destroy ( GTK_WIDGET ( dialog ) );
		return TCL_ERROR;
	}

	gtk_widget_show ( GTK_WIDGET ( dialog ) );

	return gnoclRegisterWidget ( interp, GTK_WIDGET ( dialog ), dialogFunc );
}

/*****/
