package net.sourceforge.ganttproject.time;

import java.util.Date;
import java.util.LinkedList;
import java.util.List;

import net.sourceforge.ganttproject.calendar.CalendarActivityImpl;
import net.sourceforge.ganttproject.calendar.GPCalendarActivity;
import net.sourceforge.ganttproject.time.TimeUnitGraph.Composition;

/**
 * Created by IntelliJ IDEA.
 * 
 * @author bard Date: 01.02.2004
 */
public class TimeUnitImpl implements TimeUnit {
    private final String myName;

    private final TimeUnitGraph myGraph;

    private final TimeUnit myDirectAtomUnit;
    private final DateFrameable myFramer;
    private WorkingTimeCalendar myCalendar;
    
    private TextFormatter myTextFormatter;

    public TimeUnitImpl(String name, TimeUnitGraph graph,
            TimeUnit directAtomUnit, DateFrameable framer) {
        myName = name;
        myGraph = graph;
        myDirectAtomUnit = directAtomUnit;
        myFramer = framer;
    }

    public String getName() {
        return myName;
    }

    public boolean isConstructedFrom(TimeUnit atomUnit) {
        return myGraph.getComposition(this, atomUnit) != null;
    }

    public int getAtomCount(TimeUnit atomUnit) {
        Composition composition = myGraph.getComposition(this, atomUnit);
        if (composition == null) {
            throw new RuntimeException(
                    "Failed to find a composition of time unit=" + this
                            + " from time unit=" + atomUnit);
        }
        return composition.getAtomCount();
    }

    public TimeUnit getDirectAtomUnit() {
        return myDirectAtomUnit;
    }

    public String toString() {
        return getName() + " hash=" + hashCode();
    }

    void setWorkingTimeCalendar(WorkingTimeCalendar calendar) {
        myCalendar = calendar;
    }
    
    public void setTextFormatter(TextFormatter formatter) {
        myTextFormatter = formatter;
    }

    public TimeUnitText format(Date baseDate) {
        return myTextFormatter == null ? new TimeUnitText("") : myTextFormatter
                .format(this, baseDate);
    }

    protected TextFormatter getTextFormatter() {
        return myTextFormatter;
    }

    public Date adjustRight(Date baseDate) {
        return myFramer==null ? baseDate : myFramer.adjustRight(baseDate);
    }

    public Date adjustLeft(Date baseDate) {
        return myFramer==null ? baseDate : myFramer.adjustLeft(baseDate);
    }

    public Date jumpLeft(Date baseDate) {
        return myFramer==null ? baseDate : myFramer.jumpLeft(baseDate);
    }

    public Date shiftDate(Date base, long length) {
        if (length==0) {
            return base;
        }
        return length > 0 ? shiftDateForward(base, length) : shiftDateBackward(base, length);
    }
    
    private Date shiftDateBackward(Date base, long length) {
        Date intervalStart = adjustLeft(base);
        while (length < 0) {
            intervalStart = jumpLeft(intervalStart);
            if (isWorkingInterval(intervalStart)) {
                length++;
            }
        }
        return intervalStart;
    }

    private Date shiftDateForward(Date base, long length) {
        Date intervalStart = adjustLeft(base);
        while (length > 0) {
            boolean isWorking = isWorkingInterval(intervalStart);
            if (isWorking) {
                length--;
            }
            intervalStart = adjustRight(intervalStart);            
        }
        while (!isWorkingInterval(intervalStart)) {
            intervalStart = adjustRight(intervalStart);
        }
        return intervalStart;
    }

    public boolean isWorkingInterval(Date intervalStart) {
        return myCalendar==null ? true : myCalendar.isWorkingTime(intervalStart);
    }
    
    
}
