#!/usr/bin/env python
from __future__ import with_statement

import os
import sys
import string

from release import version

GAEOGEN_VERSION = version

def usage():
    return """Usage: gaeogen.py <generation type> [args]
GAEOGen command line tool, version %s.

Available generation types:

    * controller  - generates a controller class and actions (w/ templates)

        usage: gaeogen.py controller <controller_name> [<action1>, <action2>, ..., <actionN>]

        e.g.,
        gaeogen.py controller Say
        gaeogen.py controller Product new create edit delete


    * model       - generates a data model class

        usage: gaeogen.py model <model_name> [<property_name>:<property_type>, ...]

        e.g.,
        gaeogen.py model User
        
    * plugin      - generates a blank plugin template
    
        usage: gaeogen.py plugin <plugin_name>
        
        e.g.,
        gaeogen.py plugin Hello

    * scaffold    -

        usage: gaeogen.py scaffold <controller_name> [index, edit, new, show, destroy] [<property_name>:<property_type>, ...]

*NOTE* that you should use this tool under your project's directory root.""" % (GAEOGEN_VERSION)

def create_file(file_name, content):
    print 'Creating %s ...' % (file_name)
    if not os.path.exists(os.path.dirname(file_name)):
        os.makedirs(os.path.dirname(file_name), 0755)
    with open(file_name, 'w') as f:
        f.write('\n'.join(content))

class GenBase(object):
    def __init__(self, name):
        super(GenBase, self).__init__()
        self.name = name
        self.content = []

    def generate_content(self):
        raise NotImplementedError

    def save(self, file_name):
        self.generate_content()
        create_file(file_name, self.content)

class GenController(GenBase):
    def __init__(self, name):
        super(GenController, self).__init__(name)
        self.actions = {}
        self.custom_import = []

    def add_action(self, name, content=['pass']):
        if not content:
            content = ['pass']
        self.actions[name] = content

    def add_import(self, custom_import):
        self.custom_import.append(custom_import)

    def generate_content(self):
        self.content = [
            'from google.appengine.ext import db',
            '',
            'from gaeo.controller import BaseController',
            ''
        ] + self.custom_import + [
            '',
            'class %sController(BaseController):' % self.name.capitalize(),
        ]

        if not self.actions:
            self.content.append('    pass')

        for act in sorted(self.actions.keys()):
            self.content.append('%sdef %s(self):' % (' ' * 4, act))
            self.content += map(lambda f: ' ' * 8 + f, self.actions[act])
            self.content.append('')

class GenModel(GenBase):
    def __init__(self, name):
        super(GenModel, self).__init__(name)
        self.props = {}

    def add_property(self, arg):
        name, sep, prop = arg.partition(':')
        if name:
            if not prop:
                prop = 'db.StringProperty()'
            if not prop.startswith('db.'):
                prop = 'db.' + prop
            if not prop.endswith(')'):
                prop = prop + '()'
            self.props[name] = prop

    def generate_content(self):
        self.content = [
            'from google.appengine.ext import db',
            'from gaeo.model import BaseModel',
            '',
            'class %s(BaseModel):' % self.name.capitalize(),
        ]

        if not self.props:
            self.content.append('    pass')

        for name in sorted(self.props.keys()):
            self.content.append(' ' * 4 + '%s = %s' % (name, self.props[name]))
        self.content.append('')

class GenScaffold(object):
    def __init__(self, name, actions, properties):
        self.name = name
        self.actions = actions
        self.properties = []
        for prop in properties:
            n, s, p = prop.partition(':')
            self.properties.append((n, p))

    def create_action(self, action):
        ''' Create action content for controllers '''
        _content = []

        if action == 'create':
            if self.properties:
                _content.append('r = %s(' % self.name.capitalize())
                for prop, dtype in self.properties:
                    _content.append("    %s = self.params.get('%s', None)," % (prop, prop))
                _content.append(')')
                _content.append('r.put()')
                _content.append("self.redirect('/%s')" % (self.name))
        elif action in ['edit', 'show']:
            _content.append("rec = %s.get(self.params.get('key'))" % (self.name.capitalize()))

            if self.properties:
                _content.append('if rec:')
                for prop, dtype in self.properties:
                    _content.append("    setattr(self, '%s', rec.%s)" % (prop, prop))
                _content.append('else:')
                for prop, dtype in self.properties:
                    _content.append("    setattr(self, '%s', 'Error')" % (prop))

        elif action == 'index':
            _content.append('query = %s.all()' % (self.name.capitalize()))
            _content.append('self.result = query')

        elif action == 'update':
            if self.properties:
                _content.append("r = %s.get(self.params.get('key'))" % (self.name.capitalize()))
                for prop, dtype in self.properties:
                    _content.append("r.%s = self.params.get('%s', None)" % (prop, prop))
                _content.append('r.put()')
                _content.append("self.redirect('/%s')" % (self.name))

        elif action == 'destroy':
            _content.append("r = %s.get(self.params.get('key'))" % (self.name.capitalize()))
            _content.append("if r is not None:")
            _content.append("    r.delete()")
            _content.append("self.redirect('/%s')" % (self.name))

        else:
            _content.append('pass')

        # default
        return _content

    def create_page(self, action):
        ''' Create HTML page for template '''
        _content = [
            '<h1>%sController#%s</h1>' % (self.name.capitalize(), action),
        ]

        if action == 'index':
            if 'new' in self.actions:
                _content.append('<a href="/%s/new">New</a>' % self.name)
            _content.append('<ul>')
            _content.append('{% for rec in result %}')
            _content.append('    <li><a href="/%s/show?key={{ rec.key }}">{{ rec.key }}</a></li>' % (self.name))
            _content.append('{% endfor %}')
            _content.append('</ul>')
        elif action == 'show':
            if 'edit' in self.actions:
                _content.append('<a href="/%s/edit?key={{ params.key }}">Edit</a>' % self.name)
            if 'destroy' in self.actions:
                _content.append('<a href="/%s/destroy?key={{ params.key }}" onClick="if(confirm(\'Are you sure?\')){return true;}else{return false;}">Delete</a>' % self.name)
            for prop, dtype in self.properties:
                _content.append('<p>%s: {{ %s }}</p>' % (prop, prop))
        elif action in ['edit', 'new']:
            _content.append('<form action="/%s/%s" method="post">' %
                (self.name, {'edit':'update', 'new':'create'}[action]))
            _content.append('  <fieldset>')
            _content.append('  <legend>%s %s</legend>' % (action.capitalize(), self.name.capitalize()))

            for prop, dtype in self.properties:
                # Depends on user's input, dtype may be "TextProperty" or "TextProperty()".
                if dtype.startswith('TextProperty'):
                    _content.append('    <p>%s: <textarea name="%s" rows="8" cols="40">{{ %s }}</textarea></p>' % ((prop,) * 3))
                else:
                    _content.append('    <p>%s: <input type="text" name="%s" value="{{ %s }}"></p>' % ((prop,) * 3))

            _content.append('    <p><input type="submit" value="Submit"></p>')

            if action == 'edit':
                _content.append('    <p><input type="reset" value="Reset"></p>')
                _content.append('    <p><input type="button" value="Cancel" onClick="javascript: history.go(-1)"></p>')
                _content.append('    <p><input type="hidden" name="key" value="{{ params.key }}"></p>')

            _content.append('  </fieldset>')
            _content.append('</form>')
        else:
            _content = []

        # default
        return _content

def gen_controller(argv, custom_import=None, template_helper=None):
    cur_dir = os.getcwd()

    controller_name = argv[0].lower()
    ctrl = GenController(controller_name)

    application_dir = os.path.join(cur_dir, 'application')
    controller_dir = os.path.join(application_dir, 'controller')
    template_dir = os.path.join(application_dir, 'templates', controller_name)

    if not os.path.exists(template_dir):
        print 'Creating %s ...' % (template_dir)
        os.makedirs(template_dir, 0755)

    if custom_import:
        ctrl.add_import(custom_import)

    for arg in argv[1:]:
        if template_helper:
            ctrl.add_action(arg, template_helper.create_action(arg))
            page_content = template_helper.create_page(arg)
            if page_content:
                create_file(os.path.join(template_dir, '%s.html' % arg), page_content)
        else:
            ctrl.add_action(arg)
            create_file(os.path.join(template_dir, '%s.html' % arg), [
                '<h1>%sController#%s</h1>' % (controller_name.capitalize(), arg)
            ])

    print 'Creating Controller %s ...' % (controller_name)
    ctrl.save(os.path.join(controller_dir, '%s.py' % controller_name))
    return ctrl

def gen_model(argv):
    cur_dir = os.getcwd()

    model_name = argv[0].lower()
    application_dir = os.path.join(cur_dir, 'application')
    model_dir = os.path.join(application_dir, 'model')

    # check if the model directory had been created
    if not os.path.exists(os.path.join(model_dir, '__init__.py')):
        create_file(os.path.join(model_dir, '__init__.py'), [])

    mdl = GenModel(model_name)
    for arg in argv[1:]:
        mdl.add_property(arg)

    print 'Creating Model %s ...' % (model_name)
    mdl.save(os.path.join(model_dir, '%s.py' % model_name))
    return mdl

def gen_scaffold(argv):
    name = argv[0].lower()

    model_argv = [name]
    ctrlr_argv = [name]

    for arg in argv[1:]:
        if ':' in arg:
            model_argv.append(arg)
        else:
            ctrlr_argv.append(arg)

    dependency = {
        'edit' : ['update'],
        'new' : ['create'],
    }
    if 'index' not in ctrlr_argv:
        ctrlr_argv.append('index')
    for ctrl in ctrlr_argv:
        if ctrl in dependency:
            for depend in dependency[ctrl]:
                if depend not in ctrlr_argv:
                    ctrlr_argv.append(depend)

    gen_model(model_argv)
    scaffold = GenScaffold(name, ctrlr_argv[1:], model_argv[1:])
    gen_controller(ctrlr_argv, custom_import = 'from model.%s import %s' % (name.lower(), name.capitalize()), template_helper = scaffold)
    return scaffold


def gen_plugin(argv):
    """
    Generates a plugin template
    """
    plugin_name = argv[0]
    cur_dir = os.getcwd()
    plugin_home = os.path.join(cur_dir, 'plugins', plugin_name)
    if not os.path.exists(plugin_home):
        os.mkdir(plugin_home)
        init_script_file = os.path.join(plugin_home, '__init__.py')
        create_file(init_script_file, [
            'from gaeo.controller import BaseController',
            '',
            'class %s:' % plugin_name.capitalize(),
            '    pass',
            '',
            'BaseController.__bases__ += (%s,)' % plugin_name.capitalize()
        ])
        print "The '%s' plugin has been created." % plugin_name
        
    return True

def main(argv):
    gen_type = argv[0].lower()
    try:
        func = eval('gen_%s' % (gen_type))
    except NameError:
        print usage()
        return False

    if argv[1] is None:
        print "Usage: %s %s <%s name>" % ('gaeogen', gen_type, gen_type)
        return False

    try:
        return func(argv[1:])
    except:
        import traceback
        traceback.print_exc()
        return False
    return True

def commandline():
    if len(sys.argv) < 3 or '--help' in sys.argv or 'help' in sys.argv or not main(sys.argv[1:]):
        print usage()
        sys.exit(1)

#paster version of command, not used yet
try:
    from paste.script import command
    import optparse
    class GaeogenCommand(command.Command):
        """Generate GAEO utilities
    
        Example usage::
    
        $ paster gaeogen [controller|plugin|crud]
        """
        #max_args = 3
        min_args = 2
        summary = __doc__.splitlines()[0]
        usage = '\n' + __doc__
        group_name = "GAEO"
    
        parser = command.Command.standard_parser(verbose=True)
        parser = optparse.OptionParser(
                     usage="paster gaeogen [controller|plugin|crud]")

    def command(self):
        self.__dict__.update(self.options.__dict__)
        print self.args
        main(self.args)
except:
    pass
           
if __name__ == '__main__':
    commandline()
