// -*- mode: c++; c-set-style: "stroustrup"; tab-width: 4; -*-
//
// CImage.c
//
// Copyright (C) 2004 Koji Nakamaru
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#ifndef _CImage_h_
#define _CImage_h_

template <typename T, int C>
class CImage {
public:
	CImage();
	virtual ~CImage();
	bool initialize(int w, int h);
	void finalize();
	void clear();
	T *pixel(int x, int y);
	int w();
	int h();
	void lock();
	void unlock();
	bool isChanged();
	void setChanged(bool p);

protected:
	int _w;
	int _h;
	T *_pixels;
	pthread_mutex_t _mutex;
	bool _is_mutex_initialized;
	bool _is_changed;
};

// inline functions

template <class T, int C>
CImage<T, C>::CImage()
: _w(0),
  _h(0),
  _pixels(NULL),
  _is_mutex_initialized(false)
{
	memset(&_mutex, 0, sizeof(_mutex));
}

template <class T, int C>
CImage<T, C>::~CImage()
{
	finalize();
}

template <class T, int C>
bool CImage<T, C>::initialize(
	int w,
	int h)
{
	if (w <= 0 || h <= 0) {
		return false;
	}
	finalize();
	if ((_pixels = new T[w * h * C]) == NULL) {
		return false;
	}
	_w = w;
	_h = h;
	pthread_mutex_init(&_mutex, NULL);
	_is_mutex_initialized = true;
	return true;
}

template <class T, int C>
void CImage<T, C>::finalize()
{
	if (_is_mutex_initialized) {
		pthread_mutex_destroy(&_mutex);
		memset(&_mutex, 0, sizeof(_mutex));
		_is_mutex_initialized = false;
	}
	forgetArray(&_pixels);
}

template <class T, int C>
inline void CImage<T, C>::clear()
{
	memset(_pixels, 0, sizeof(T) * _w * _h * C);
}

template <class T, int C>
inline T *CImage<T, C>::pixel(
	int x,
	int y)
{
	return &_pixels[(y * _w + x) * C];
}

template <class T, int C>
inline int CImage<T, C>::w()
{
	return _w;
}

template <class T, int C>
inline int CImage<T, C>::h()
{
	return _h;
}

template <class T, int C>
inline void CImage<T, C>::lock()
{
	pthread_mutex_lock(&_mutex);
}

template <class T, int C>
inline void CImage<T, C>::unlock()
{
	pthread_mutex_unlock(&_mutex);
}

template <class T, int C>
inline bool CImage<T, C>::isChanged()
{
	return _is_changed;
}

template <class T, int C>
inline void CImage<T, C>::setChanged(bool p)
{
	_is_changed = p;
}

#endif
