/*
Copyright (C) 1997-2001 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

//
// cg_local.h
// CGame local header
//

#ifndef __CGAME_LOCAL_H__
#define __CGAME_LOCAL_H__

#include "../shared/shared.h"
#include "cg_shared.h"
#include "cg_api.h"
#include "cg_entities.h"
#include "cg_effects.h"
#include "cg_keys.h"
#include "cg_media.h"
#include "ui/ui_local.h"
#include "menu/menu.h"

extern cgImport_t cgi;

// Local tags
enum {
	CGTAG_ANY,
	CGTAG_LOCATION,
	CGTAG_MAPFX,
	CGTAG_MENU,
};

// Generic memory allocation
#define CG_MemAlloc(size,zeroFill)		cgi.Mem_Alloc((size),(zeroFill),CGTAG_ANY,__FILE__,__LINE__)
#define CG_MemFree(ptr)					cgi.Mem_Free((ptr),__FILE__,__LINE__)
#define CG_StrDup(in)					cgi.Mem_StrDup((in),CGTAG_ANY,__FILE__,__LINE__)

// Allocates to a tagged location
// Useful for releasing entire blocks of memory at a time
#define CG_AllocTag(size,zeroFill,tag)	cgi.Mem_Alloc((size),(zeroFill),(tag),__FILE__,__LINE__)
#define CG_FreeTag(tag)					cgi.Mem_FreeTag((tag),__FILE__,__LINE__)
#define CG_TagStrDup(in,tag)			cgi.Mem_StrDup((in),(tag),__FILE__,__LINE__)
#define CG_MemTagSize(tag)				cgi.Mem_TagSize((tag))
#define CG_ChangeTag(tagFrom,tagTo)		cgi.Mem_ChangeTag((tagFrom),(tagTo))

// Filesystem
#define CG_FS_FreeFile(buffer)			cgi.FS_FreeFile((buffer),__FILE__,__LINE__)
#define CG_FS_FreeFileList(list,num)	cgi.FS_FreeFileList((list),(num),__FILE__,__LINE__)

/*
=============================================================================

	CGAME STATE

=============================================================================
*/

enum {
	GAME_MOD_DEFAULT,
	GAME_MOD_DDAY,
	GAME_MOD_GIEX,
	GAME_MOD_GLOOM,
	GAME_MOD_LOX,
	GAME_MOD_ROGUE,
	GAME_MOD_XATRIX
};

typedef struct clientInfo_s {
	char				name[MAX_CFGSTRLEN];
	char				cInfo[MAX_CFGSTRLEN];
	struct shader_s		*skin;
	struct shader_s		*icon;
	char				iconName[MAX_QPATH];
	struct refModel_s	*model;
	struct refModel_s	*weaponModels[MAX_CLIENTWEAPONMODELS];
} clientInfo_t;

typedef struct cgPredictionState_s {
	float				step;						// for stair up smoothing
	uint32				stepTime;
	int16				origins[CMD_BACKUP][3];		// for debug comparing against server
	vec3_t				origin;						// generated by CG_PredictMovement
	vec3_t				angles;
	vec3_t				velocity;
	vec3_t				error;
} cgPredictionState_t;

typedef struct cgDownloadInfo_s {
	qBool		downloading;

	char		*fileName;
	int			percent;
	float		bytesDown;
} cgDownloadInfo_t;

// ==========================================================================

typedef struct cgState_s {
	byte				currGameMod;

	// Time
	int					netTime;
	float				netFrameTime;

	int					refreshTime;
	float				refreshFrameTime;

	int					realTime;

	// View settings
	refDef_t			refDef;
	vec4_t				viewBlend;

	// the client maintains its own idea of view angles, which are
	// sent to the server each frame. It is cleared to 0 upon entering each level.
	// the server sends a delta each frame which is added to the locally
	// tracked view angles to account for standing on rotating objects,
	// and teleport direction changes
	vec3_t				viewAngles;

	refConfig_t			refConfig;

	qBool				forceRefDef;
	uint32				frameCount;

	// Gloom
	qBool				gloomCheckClass;
	int					gloomClassType;

	// Third person camera
	qBool				thirdPerson;
	byte				cameraTrans;

	// Prediction
	cgPredictionState_t	predicted;

	frame_t				frame;								// received from server
	frame_t				oldFrame;
	float				lerpFrac;							// between oldFrame and frame

	qBool				attractLoop;

	//
	// Network
	//
	int					playerNum;

	qBool				localServer;
	char				*serverMessage;
	char				*serverName;
	int					serverProtocol;
	int					connectCount;

	cgDownloadInfo_t	download;

	//
	// Config strings
	//
	char				configStrings[MAX_CFGSTRINGS][MAX_CFGSTRLEN];
	struct shader_s		*imageCfgStrings[MAX_CS_IMAGES];
	struct refModel_s	*modelCfgDraw[MAX_CS_MODELS];
	struct cBspModel_s	*modelCfgClip[MAX_CS_MODELS];
	struct sfx_s		*soundCfgStrings[MAX_CS_SOUNDS];

	//
	// Loading screen
	//
	qBool				mapLoaded;
	qBool				mapLoading;
	float				loadingPercent;

	char				loadingString[MAX_QPATH];
	char				loadFileName[MAX_QPATH];

	//
	// Menu
	//
	qBool				menuOpen;

	//
	// HUD and inventory
	//
	vec2_t				hudScale;

	char				layout[1024];				// general 2D overlay
	int					inventory[MAX_CS_ITEMS];

	//
	// Client info
	//
	int					maxClients;
	clientInfo_t		clientInfo[MAX_CS_CLIENTS];
	clientInfo_t		baseClientInfo;

	//
	// Enhanced protocol support
	//
	qBool				strafeHack;
} cgState_t;

extern cgState_t	cg;

/*
=============================================================================

	PLAYER MOVEMENT

=============================================================================
*/

enum {
	GLM_DEFAULT,

	GLM_OBSERVER,

	GLM_BREEDER,
	GLM_HATCHLING,
	GLM_DRONE,
	GLM_WRAITH,
	GLM_KAMIKAZE,
	GLM_STINGER,
	GLM_GUARDIAN,
	GLM_STALKER,

	GLM_ENGINEER,
	GLM_GRUNT,
	GLM_ST,
	GLM_BIOTECH,
	GLM_HT,
	GLM_COMMANDO,
	GLM_EXTERM,
	GLM_MECH
};

//
// pmove.c
// common between the client and server for consistancy
//
void	Pmove (pMoveNew_t *pMove, float airAcceleration);

//
// cg_predict.c
//
void	CG_CheckPredictionError (void);

void	CG_BuildSolidList (void);

void	CG_PMTrace (trace_t *out, vec3_t start, vec3_t mins, vec3_t maxs, vec3_t end, qBool entities);

void	CG_PredictMovement (void);

/*
=============================================================================

	SCREEN

=============================================================================
*/

//
// cg_screen.c
//

struct shader_s	*CG_RegisterPic (char *name);

float	palRed (int index);
float	palGreen (int index);
float	palBlue (int index);

float	palRedf (int index);
float	palGreenf (int index);
float	palBluef (int index);

void	CG_AddNetgraph (void);
void	CG_DebugGraph (float value, int color);
void	CG_DrawDebugGraph (void);

void	SCR_UpdatePING (void);

void	SCR_ParseCenterPrint (void);

void	SCR_Draw (void);

//
// cg_view.c
//

void	V_RenderView (int realTime, float netFrameTime, float refreshFrameTime, float stereoSeparation, qBool refreshPrepped);

void	V_Register (void);
void	V_Unregister (void);

/*
=============================================================================

	INVENTORY / HUD

=============================================================================
*/

//
// cg_hud.c
//

void	HUD_CopyLayout (void);
void	HUD_DrawLayout (void);
void	HUD_DrawStatusBar (void);

//
// cg_inventory.c
//

void	Inv_ParseInventory (void);
void	Inv_DrawInventory (void);

/*
=============================================================================

	NEAREST LOCATION SUPPORT

=============================================================================
*/

void	CG_LocationInit (void);
void	CG_LocationShutdown (void);

void	CG_LoadLocations (char *mapName);

void	CG_Say_Preprocessor (void);

/*
=============================================================================

	CONSOLE VARIABLES

=============================================================================
*/

extern cVar_t	*cg_advInfrared;
extern cVar_t	*cg_brassTime;
extern cVar_t	*cg_decals;
extern cVar_t	*cg_decalBurnLife;
extern cVar_t	*cg_decalFadeTime;
extern cVar_t	*cg_decalLife;
extern cVar_t	*cg_decalLOD;
extern cVar_t	*cg_decalMax;
extern cVar_t	*cg_mapEffects;
extern cVar_t	*cl_add_particles;
extern cVar_t	*cg_particleCulling;
extern cVar_t	*cg_particleGore;
extern cVar_t	*cg_particleMax;
extern cVar_t	*cg_particleShading;
extern cVar_t	*cg_particleSmokeLinger;
extern cVar_t	*cg_railCoreRed;
extern cVar_t	*cg_railCoreGreen;
extern cVar_t	*cg_railCoreBlue;
extern cVar_t	*cg_railSpiral;
extern cVar_t	*cg_railSpiralRed;
extern cVar_t	*cg_railSpiralGreen;
extern cVar_t	*cg_railSpiralBlue;
extern cVar_t	*cg_thirdPerson;
extern cVar_t	*cg_thirdPersonAngle;
extern cVar_t	*cg_thirdPersonClip;
extern cVar_t	*cg_thirdPersonDist;

extern cVar_t	*cl_explorattle;
extern cVar_t	*cl_explorattle_scale;
extern cVar_t	*cl_footsteps;
extern cVar_t	*cl_gun;
extern cVar_t	*cl_noskins;
extern cVar_t	*cl_predict;
extern cVar_t	*cl_showmiss;
extern cVar_t	*cl_vwep;

extern cVar_t	*crosshair;

extern cVar_t	*gender_auto;
extern cVar_t	*gender;
extern cVar_t	*hand;
extern cVar_t	*skin;

extern cVar_t	*glm_advgas;
extern cVar_t	*glm_advstingfire;
extern cVar_t	*glm_blobtype;
extern cVar_t	*glm_bluestingfire;
extern cVar_t	*glm_flashpred;
extern cVar_t	*glm_flwhite;
extern cVar_t	*glm_forcecache;
extern cVar_t	*glm_jumppred;
extern cVar_t	*glm_showclass;


extern cVar_t	*cl_testblend;
extern cVar_t	*cl_testentities;
extern cVar_t	*cl_testlights;
extern cVar_t	*cl_testparticles;

extern cVar_t	*r_hudScale;
extern cVar_t	*r_fontScale;

extern cVar_t	*crosshair;
extern cVar_t	*ch_alpha;
extern cVar_t	*ch_pulse;
extern cVar_t	*ch_scale;
extern cVar_t	*ch_red;
extern cVar_t	*ch_green;
extern cVar_t	*ch_blue;
extern cVar_t	*ch_xOffset;
extern cVar_t	*ch_yOffset;

extern cVar_t	*cl_showfps;
extern cVar_t	*cl_showping;
extern cVar_t	*cl_showtime;

extern cVar_t	*con_chatHud;
extern cVar_t	*con_chatHudLines;
extern cVar_t	*con_chatHudPosX;
extern cVar_t	*con_chatHudPosY;
extern cVar_t	*con_chatHudShadow;
extern cVar_t	*con_notifyfade;
extern cVar_t	*con_notifylarge;
extern cVar_t	*con_notifylines;
extern cVar_t	*con_notifytime;
extern cVar_t	*con_alpha;
extern cVar_t	*con_clock;
extern cVar_t	*con_drop;
extern cVar_t	*con_scroll;

extern cVar_t	*scr_conspeed;
extern cVar_t	*scr_centertime;
extern cVar_t	*scr_showpause;
extern cVar_t	*scr_hudalpha;

extern cVar_t	*scr_netgraph;
extern cVar_t	*scr_timegraph;
extern cVar_t	*scr_debuggraph;
extern cVar_t	*scr_graphheight;
extern cVar_t	*scr_graphscale;
extern cVar_t	*scr_graphshift;
extern cVar_t	*scr_graphalpha;

extern cVar_t	*viewsize;
extern cVar_t	*gl_polyblend;

/*
=============================================================================

	SUPPORTING FUNCTIONS

=============================================================================
*/

//
// cg_draw.c
//

void		CG_DrawFill (float x, float y, int w, int h, vec4_t color);

void		CG_DrawModel (int x, int y, int w, int h, struct refModel_s *model, struct shader_s *shader, vec3_t origin, vec3_t angles);

//
// cg_main.c
//

void		CG_SetRefConfig (refConfig_t *inConfig);

void		CG_UpdateCvars (void);

void		CG_LoadMap (int playerNum, int serverProtocol, qBool attractLoop, qBool strafeHack, refConfig_t *inConfig);

void		CG_Init (void);
void		CG_Shutdown (void);

//
// cg_muzzleflash.c
//

void		CG_ParseMuzzleFlash (void);
void		CG_ParseMuzzleFlash2 (void);

//
// cg_players.c
//

extern char	cg_weaponModels[MAX_CLIENTWEAPONMODELS][MAX_CFGSTRLEN];
extern int	cg_numWeaponModels;

int			CG_GloomClassForModel (char *model, char *skin);
void		CG_LoadClientinfo (clientInfo_t *ci, char *skin);

void		CG_StartSound (vec3_t origin, int entNum, entChannel_t entChannel, int soundNum, float volume, float attenuation, float timeOffset);
void		CG_FixUpGender (void);

//
// cg_parse.c
//

void		CG_ParseClientinfo (int player);

void		CG_ParseConfigString (int num, char *str);

void		CG_StartServerMessage (void);
void		CG_EndServerMessage (int realTime);
qBool		CG_ParseServerMessage (int command);

/*
=============================================================================

	INTERFACES

=============================================================================
*/

//
// cg_loadscreen.c
//

void	CG_IncLoadPercent (float increment);
void	CG_LoadingPercent (float percent);
void	CG_LoadingString (char *str);
void	CG_LoadingFilename (char *str);

void	CG_UpdateConnectInfo (char *serverName, char *serverMessage, int connectCount, char *dlFileName, int dlPercent, float bytesDownloaded);
void	CG_DrawConnectScreen (void);

#endif // __CGAME_LOCAL_H__
