<?php
/**

* @package efront
*/

/**
* Converts a time interval to human-readable format
*
* The argument to this function is an amount of seconds. The function converts the value 
* to a more human friendly format, in seconds, minutes and hours.
* <br/>Example:
* <code>
* print eF_timeFromDuration(12345);    //Prints: '3 hours 25 minutes 45 seconds'
* </code>
*
* @param int $duration The amount of time, in seconds, to convert
* @return string The human readable form of the interval
* @version 1.0
*/
function eF_timeFromDuration($total_seconds)
{
    $time = '';
    if ($total_seconds < 0) {
        $total_seconds = 0;
    }

    $seconds       = $total_seconds % 60;                                       //Calculate the seconds that do not form a complete minute
    $total_minutes = ($total_seconds - $seconds) / 60;                          //Calculate the total minutes that correspond to the total seconds

    if ($total_minutes > 60) {
        $minutes = $total_minutes % 60;                                         //Calculate the minutes that do not form a complete hour
        $hours   = ($total_minutes - $minutes) / 60;                            //Calculate the total hours that correspond to the total minutes
    }

    if (isset($hours)) {
        $time = $hours;
        $hours == '1' ? $time .= " "._HOUR." " : $time .= " "._HOURS." ";
    }

    if (isset($minutes)) {
        $time .= $minutes;
        $minutes == '1' ? $time .= " "._MINUTE." " : $time .= " "._MINUTES." ";
    }

    if (isset($seconds)) {
        $time .= $seconds;
        $seconds == '1' ? $time .= " "._SECOND." " : $time .= " "._SECONDS." ";
    }

    return $time;
}



/**
* Prints online time statistics for a user
*
* This function calculates statistics for the online time of the specified user.
* Specifically, it displays The last login date and time, The last session duration
* and the medium stay online time for the user. These data are put in a string and returned. 
* <br/>Example:
* <code>
* print_r(eF_averageDuration(time()-86400, time(), 'professor'));       //Calculates statistics for a month, for the user 'professor'
* </code>
* Outputs:
* <code>
* Last log-in 2005-05-17 18:56
* Last log-in duration    1 hour 31 minutes 47 seconds 
* Mean log-in duration    3 hours 24 minutes 48 seconds  (7 times this month)
* </code>
* 
* @param int $start_month The timestamp corresponding to the start time of the statistic period
* @param int $end_month The timestamp corresponding to the end time of the statistic period
* @param string $usrname The login name of the user to calculate statistics for
* @return string The HTML code containing the data (in table rows only, without '<table>' tags)
* @version 1.0
* @todo Should return array
*/
function eF_averageDuration($start_month, $end_month, $usrname)
{
    $count      = 0;
    $in         = $end_month;
    $duration   = 0;
    $last_visit = 0;
    $time       = "";
    $flag       = 0;
    
    while ($in > $start_month) {
        $res = eF_execute("select id,timestamp from logs where users_LOGIN='$usrname' and action='login' and timestamp<$in order by timestamp DESC LIMIT 0,1");
        list($last_visit_id, $last_visit) = mysql_fetch_row($res);

        if ($last_visit) {
            $res = eF_execute("select id,timestamp from logs where users_LOGIN='$usrname' and action not like 'login' and timestamp<$in order by timestamp DESC LIMIT 0,1");
            list($visit_id,$visit) = mysql_fetch_row($res);

            if ($visit > $last_visit) {
                $duration += $visit - $last_visit; 
            }
            $in = $last_visit;
        } else {
            break;
        }

        if (!$flag) {
            $flag            = 1;
            $lv              = $last_visit;
            $last_visit_time = $duration;
        }
        $count++;
    }

    if ($count) {
        $duration = intval($duration / $count);

        $avg_time  = eF_timeFromDuration($duration);
        $last_time = eF_timeFromDuration($last_visit_time);
        if ($lv) {
            $date  = strftime("%Y-%m-%d %H:%M", $lv);
            $data = "
                <tr><td>"._LASTLOGIN."</td><td>$date</td></tr>
                <tr><td>"._LASTLOGINTIME."</td><td>$last_time</td></tr>
                <tr><td>"._MEANLOGINTIME."</td><td>$avg_time";
            if($count > 1) {
                $data.= " ($count "._TIMESTHISMONTH.")</td></tr>";
            } else {
                $data.= "($count "._TIMETHISMONTH.")</td></tr>";
            }
        }
    }
    return $data;
}


/**
* Prints the number of posts for a user
*
* This function prints the number of messages the specified user has posted so far in the forum. If the $lessons_ID
* parameter is set, then only messages posted in the correspoding lesson category are counted.
* <br/>Example:
* <code>
* eF_numberOfPosts('professor');        //Returns a number, i.e. 16
* </code>
*
* @param string $login The login name of the user
* @param int $lessons_ID The lesson id
* @return int The forum messages number
* @version 2.0
* Changes from version 1.1 to 2.0 (24/10/05):
* - Added $lessons_ID parameter and functionality
* - Changed return type, from string to int
* Changes from version 1.0 to 1.1:
* - Got rid of forum
*/
function eF_numberOfPosts($login, $lessons_ID = false)
{
    if ($lessons_ID && eF_checkParameter($lessons_ID, 'id')) {
        $result = eF_getTableData("f_messages, f_topics, f_categories", "count(*)", "f_topics.id = f_messages.f_topics_ID and f_categories.id = f_topics.f_categories_ID and f_categories.lessons_ID = ".$lessons_ID." and f_messages.users_LOGIN='".$login."'");
    } else {
        $result = eF_getTableData("f_messages", "count(*)", "users_LOGIN='".$login."'");
    }
    
    return $result[0]['count(*)'];
}

/**
*
*/
function eF_userAverageScore($login, $lessons_ID = false)
{
    if ($lessons_ID && eF_checkParameter($lessons_ID, 'id')) {
        $result = eF_getTableData("done_tests, tests, content", "avg(score)", "done_tests.tests_ID = tests.id and tests.content_ID = content.id and content.lessons_ID = ".$lessons_ID." and users_LOGIN='".$login."'");
    } else {
        $result = eF_getTableData("done_tests", "avg(score)", "users_LOGIN='".$login."'");
    }  
    return $result[0]['avg(score)'];    
}

/**
*
*/
function eF_testAverageScore($tests_ID = false, $lessons_ID = false)
{
    if ($tests_ID && eF_checkParameter($tests_ID, 'id')) {                                      //Get the mean score for this test
        $result = eF_getTableData("done_tests", "avg(score)", "id = ".$tests_ID);
    } elseif ($lessons_ID && eF_checkParameter($lessons_ID, 'id')) {                            //If no tests id is specified, get the mean score of all the tests in this lesson
        $result = eF_getTableData("done_tests, tests, content", "avg(score)", "done_tests.tests_ID = tests.id and tests.content_ID = content.id and content.lessons_ID = ".$lessons_ID);
    } else {
        return array();
    }
    
    return $result[0]['avg(score)'];
}

/**
* Prints the number of comments for a user
*
* This function prints the number of comments the user $login has added so far in 
* the lesson content. If no lesson is specified, it returns all his comments
* <br/>Example:
* <code>
* echo eF_numberOfComments('john', 1);
* //prints i.e. 5
* </code>
*
* @param int $start_month The timestamp corresponding to the start time of the statistic period
* @param int $end_month The timestamp corresponding to the end time of the statistic period
* @param string $login The login name of the user
* @param int $lessons_ID The lesson id
* @return int The comments number.
* @version 2.1
* Changes from 2.0 to 2.1 (24/10/05):
* - Made lessons_ID parameter optional
* Changes from 1.0 to 2.0 (17/10/05):
* - Returns int instead of string
* - Changed handling of $start_month, $end_month
* - Added functionality
*/
function eF_numberOfComments($login, $lessons_ID = false, $start_month = false, $end_month = false)
{
    ($lessons_ID && eF_checkParameter($lessons_ID, 'id')) ? $lessons_str = " and content.lessons_ID=".$lessons_ID : $lessons_str = '';
    
    if (!$start_month && !$end_month) {
        $result = eF_getTableData("content, comments", "count(*)", "users_LOGIN = '".$login."' and comments.content_ID = content.id".$lessons_str);
    } else {
        !$start_month ? $start_month = time() : null;
        !$end_month   ? $end_month   = time() : null;

        $result = eF_getTableData("logs l, content", "count(*)", "l.users_LOGIN='$login' and l.action='comment' and l.timestamp>$start_month and l.timestamp<$end_month and l.comments=content.id".$lessons_str);
    }
    
    $comments_num = $result[0]['count(*)'];

    return $comments_num;
}



/**
* Prints the average test score of a user
*
* This function calculates and prints the average test score for a user.
* <br/>Example:
* <code>
* eF_averageTestScore(time()-86400, time(), 'professor', 1);        //Prints: 'Medium term: 40 %'
* </code>
*
* @param int $start_month The timestamp corresponding to the start time of the statistic period
* @param int $end_month The timestamp corresponding to the end time of the statistic period
* @param string $usrname The login name of the user
* @param int $lessons_ID The lesson id
* @return string The html code (in table rows '<tr>') of the data.
* @version 1.0
* @deprecated
* @todo Should return array
* @todo A bit useless, too few data...
*/
function eF_averageTestScore($start_month, $end_month, $usrname, $lessons_ID)
{
    $data = '';
    $res = eF_execute("select count(*),avg(score) from done_tests d,tests t,content c where d.users_LOGIN='$usrname' and d.timestamp>$start_month and d.timestamp<$end_month and d.tests_ID=t.id and t.content_ID=c.id and c.lessons_ID=$lessons_ID");
    list($num, $avg_score) = mysql_fetch_row($res);

    $avg_score *= 100;
    $avg_score  = round($avg_score, 2);
    if ($num) {
        $data.= "<tr><td colspan=2>"._MEAN.": <b>$avg_score %</b></td></tr>";
    }

    return $data;
}



/**
* Prints a user's last moves
*
* This function prints a list the user's last moves in the system
* <br/> Example:
* <code>
* eF_lastMoves(time()-86400, time(), 'professor', 1);
* </code>
* Outputs:
* <code>
* 1) Log-out at 17 May 2005, 18:52
* 2) <b>Introduction</b>(17 May 2005, 15:20)
* 3) Log-in at 17 May 2005, 14:59
* 4) Log-out at 16 May 2005, 23:33
* 5) Personal data change at 16 May 2005, 20:26
* 6) Log-in at 16 May 2005, 20:07
* 7) Log-out at 16 May 2005, 17:01
* 8) Unit test <b>?????????? ??????? ???????</b> at 16 May 2005, 16:48
* 9) Log-in at 16 May 2005, 16:48 
* </code>
*
* @param int $start_month The timestamp corresponding to the start time of the statistic period
* @param int $end_month The timestamp corresponding to the end time of the statistic period
* @param string $usrname The login name of the user
* @param int $lessons_ID The lesson id
* @return string The html code (in table rows '<tr>') of the data.
* @version 1.0
* @todo Should return array
* @todo It is not used anywhere!
* @todo _AT
*/
function eF_lastMoves($start_month, $end_month, $usrname, $lessons_ID)
{
    global $s_login;
    
    $letters = 150;
    $limit   = $letters;
    $seri    = "";
    $data    = "<tr><td colspan=2>";
    $count   = 1;

    $res     = eF_execute("select l.action,l.timestamp,l.comments from logs l,content c where ((c.id=l.comments and c.lessons_ID=$lessons_ID) or l.comments=0) and l.users_LOGIN='$usrname' and l.timestamp>$start_month and l.timestamp<$end_month group by l.id order by l.timestamp DESC");

    while (list($action, $timestamp, $comments) = mysql_fetch_row($res) ) {
         if ($tmp) {
            $data .= $count++.") ".$tmp; 
            unset($tmp);
         }
        
        if ($old_timestamp && $old_timestamp - $timestamp < 10  && $comments) {
            $old_timestamp = $timestamp; 
            continue;
        }
        
        if ($last_action != $action && (!$comments || $last_com!=$comments)) {  
            if($action=="login") {
                $data .= $count++.") "._LOGINAT." ".eF_timestampToTime($timestamp)."<br><br>"; 
                $seri  = "";
            } else if ($action=="logout") {
                $data .= $count++.") "._LOGOUTAT." ".eF_timestampToTime($timestamp)."<br><br>"; 
                $seri  = "";
            } else if ($action=="comment") {
                $res2 = eF_execute("select name from content where id=$comments");
                list($name) = mysql_fetch_row($res2);
                $tmp = _UNITCOMMENT." <b>$name</b> "._AT." ".eF_timestampToTime($timestamp)."<br><br>";
            } else if ($action=="tests") {
                $res2 = eF_execute("select name from content where id=$comments");
                list($name) = mysql_fetch_row($res2);
                $tmp = _UNITRATE." <b>$name</b> "._AT." ".eF_timestampToTime($timestamp)."<br><br>";
            } else if ($action=="personal") {
                $tmp = _PERSONALDATACHANGEAT." ".eF_timestampToTime($timestamp)."<br><br>"; 
            } else if ($action != "lastmove") {
                $res2 = eF_execute("select name from content where id=$comments");
                list($name) = mysql_fetch_row($res2);
                $tmp = "<b>$name</b>".$sign." (".eF_timestampToTime($timestamp).")<br><br>";
            }
        }
        $old_timestamp = $timestamp;
        $last_action   = $action;
        $last_com      = $comments;
        
        if ($count > 100) {
            return $data."</td></tr>";
        }
    }

    if ($tmp) {
        $data .= $count++.") ".$tmp; 
        unset($tmp);
    }
    $data .= "</td></tr>";
    
    return $data;
}


/**
* Prints the lesson's students
*
* This function prints the number of students that have enrolled in this lesson.
* <br/>Example:
* <code>
* print EfrontLessonStudents(1);           //Prints: 'Number of students: 4'
* </code>
*
* @param int $lesson The lesson id
* @return string The html code of the list
* @version 1.0
* @todo Should return array
*/
function EfrontLessonStudents($lesson)
{
    $res = eF_execute("select count(*) from users u,users_to_lessons l where u.login=l.users_LOGIN and u.user_type='student' and l.lessons_ID=$lesson");
    list($num) = mysql_fetch_row($res);
    $data .= "
        <tr><td colspan = 2>"._STUDENTNUMBER.": $num</td></tr>";
    return $data;
}

function eF_getLessonStudents($lesson)
{
    $res = eF_execute("select u.LOGIN from users u,users_to_lessons l where u.login=l.users_LOGIN and u.user_type='student' and l.lessons_ID=$lesson");
    $users = array();
    while ($k = mysql_fetch_row($res)){        
        $user['login'] = $k[0];
        $users[] = $user;        
    }
    return $users;
}

function eF_getStudentLessons($login)
{
    $res = eF_execute("select l.id, l.name from lessons l, users_to_lessons ul where ul.lessons_ID = l.ID and ul.users_LOGIN='$login'");
    $lessons = array();
    while ($k = mysql_fetch_row($res)){
        $lesson['id'] = $k[0];
        $lesson['name'] = $k[1];
        $lessons[] = $lesson;
    }
    return $lessons;
}

/**
* Prints the lesson's tests
*
* This function prints a list with the tests that exist in this lesson. Each test name
* is a link pointing to the statistics page, relevant to this test.
* <br/>Example:
* <code>
* print EfrontLessonTests(1);           
* </code>
* Outputs:
* <code>
* Test for unit 1
* Test for unit 2
* Test for unit 3
* </code>
*
* @param int $lesson The lesson id
* @return string The html code of the list
* @version 1.0
* @todo Should return array
*/
function EfrontLessonTests($lesson)
{
    $data = false;
    $res  = eF_execute("select t.id,c.name from tests t,content c where t.content_ID=c.id and c.lessons_id=$lesson LIMIT 10");
    while(list($id, $name) = mysql_fetch_row($res)) {
        $data.= "
            <tr><td colspan=2><a href=professor.php?ctg=personal&op=statistics&test_id=$id>$name</a></td></tr>";
    }
    
    return $data;
}


/**
* Calculates a lesson's traffic through a sepecific time period
*
* This function calculates, for a specific lesson, how many times the lesson's pages have been accessed.
* Additionally, it returns the access number for every user.
*
* @param int $lesson The lesson id
* @param int $from_time Time from
* @return int $to_time Time to
* @version 1.0
*/
function eF_getLessonTraffic($lesson, $from_time, $to_time)
{
    $sql = "SELECT users_LOGIN, count(id) as cnt FROM logs WHERE lessons_id='$lesson' and timestamp between $from_time and  $to_time group by users_LOGIN order by users_LOGIN";
    $res = eF_execute($sql);
    $traffic = array();
    $traffic['users'] = array();
    $traffic['user_cnt'] = array();
    $cnt_access = 0;
    $cnt_time = 0;
    while ($k=mysql_fetch_row($res)){
        $cnt_access += $k[1];
        $traffic['users'][]=$k[0]; 
        $traffic['users_cnt'][]=$k[1];
        $tm =  ef_getUserLessonTime($lesson, $k[0], $from_time, $to_time); 
        $traffic['users_time'][]= $tm;  
        $cnt_time = $cnt_time + $tm['minutes']*60+$tm['seconds'];
    }
    $traffic['total_access'] = $cnt_access;
    $traffic['total_time']['minutes'] = floor($cnt_time/60);
    $traffic['total_time']['seconds'] = $cnt_time%60;
    return $traffic;
}

function eF_getLessonAccess($lesson, $from_time, $to_time)
{
    $sql = "SELECT count(id) as cnt FROM logs WHERE lessons_id='$lesson' and timestamp between $from_time and  $to_time order by timestamp";
    $res = eF_execute($sql);
    $k = mysql_fetch_row($res);
    return $k[0];
}

function ef_getUserContentTime($log_id, $content_id, $login, $time_start){
    $sql = "SELECT action, comments, timestamp FROM logs where id > $log_id and users_LOGIN='$login' order by id";
    $res = ef_execute($sql);
    while ($k = mysql_fetch_row($res)){
        $action = $k[0];
        $comments = $k[1];
        $timestamp = $k[2];
        if ($action != "content"){
            $interval = ef_convertIntervalToTime($timestamp - $time_start);
            return $interval;
        }
        else{
            if ($comments != $content_id){
                $interval = ef_convertIntervalToTime($timestamp - $time_start);
                return $interval;
            }                
        }        
    }    
}


function ef_getUserLessonTime($lesson, $login, $from_time, $to_time){
    
    $sql = "SELECT id, timestamp, lessons_id FROM logs WHERE users_LOGIN='$login' and timestamp between $from_time and $to_time order by timestamp";
    $res = eF_execute($sql);
    $total_time = array();
    $total_time['minutes'] = 0;
    $total_time['seconds'] = 0;
    $total_time['hours'] = 0;
    $l_start = 0;
    $inlesson = 0;
    while ($k=mysql_fetch_row($res)){       
        $c_id = $k[0];
        $c_time = $k[1];
        $c_lesson = $k[2];
        if ($inlesson){
            if ($c_lesson != $lesson){
                $inlesson = 0;
                $interval = ef_convertIntervalToTime($c_time - $l_start);
                if ($interval['hours']==0 && $interval['minutes']<=30){
                    $total_time['minutes'] += $interval['minutes'];
                    $total_time['seconds'] += $interval['seconds'];
                }               
            }
            else{
                $interval = ef_convertIntervalToTime($c_time - $l_start);
                if ($interval['hours']==0 && $interval['minutes']<=30){
                    $total_time['minutes'] += $interval['minutes'];
                    $total_time['seconds'] += $interval['seconds'];
                }
                $l_start = $c_time;
            }
        }
        else{
            if ($c_lesson == $lesson){
                $inlesson = 1;
                $l_start = $c_time;
            }
            
        }
    }        
    
    $sec = $total_time['seconds'];
    
    if ($sec >=60){
        $nm = floor($sec / 60);
        $ns = $sec %60;
        $total_time['seconds'] = $ns;
        $total_time['minutes'] += $nm;
    }
    if ($total_time['minutes'] >=60){
        $nh = floor($total_time['minutes']/60);
        $nm = $total_time['minutes'] % 60;
        $total_time['hours'] = $nh;
        $total_time['minutes'] = $nm;
    }
    
    return $total_time;
}


/**
* Calculates pages seen by users
*
* This function calculates, for a specific unit, which users have seen at least
* $content pages, from $timestamp until now. It returns an array, where each element
* is another array, with fields:
* - 'name': The user login name
* - 'num' : The number of pages he has seen 
*
* @param int $lesson The lesson id
* @param int $content The threshold to check for a user if he has seen so many pages
* @param int $timestamp We check the time interval from this date until today.
* @return array The result list
* @version 1.0
*/
function eF_usersSeenContent($lesson, $content, $timestamp)
{
    $res = eF_execute("select l.users_LOGIN,count(*) as mycounter from logs l,content c,users_to_lessons u where (l.action='theory' or l.action='exercises' or l.action='tests' or l.action='examples' or l.action='lessons') and l.timestamp>$timestamp and l.comments=c.id and c.lessons_ID=$lesson and u.users_LOGIN=l.users_LOGIN and u.lessons_ID=$lesson group by l.users_LOGIN order by mycounter DESC");

    $count = 0;
    while (list($name, $num) = mysql_fetch_row($res)) {
        if ($num >= $content) {
            $result[$count]['name'] = $name;
            $result[$count]['num']  = $num;
            $count++;
        }
    }

    return $result;
}


/**
* Get online times for a user
*
* This function is used to calculate the times that a user logged into the system, and 
* for how long he stayed online. These data are put in an array, where each element is the timestamp
* corrresponding to each login date. The last element of the aray is another array, where each value
* corresponds to the online times (in seconds) of each login.
* <br/>Example:
* <code>
* print_r (eF_userTimes('professor'));
* </code>
* Outputs: 
* <code>
* Array
* (
*     [timestamps] => Array
*         (
*             [0] => 1097257088
*             [1] => 1097257123
*             [2] => 1097257128
*             [3] => 1097258654
*         )
* 
*     [0] => 19
*     [1] => 3
*     [2] => 115
*     [3] => 10
* )
* </code>
*
* @param string $usrname The login name of the user
* @return array The results array
* @version 1.0
* @todo Very time-consuming, should be much faster!
* @deprecated
*/
function eF_userTimes($usrname)
{

    $start_month = mktime(0, 0, 0, strftime("%m", time()), 1, strftime("%Y",time()));
    $end_month   = mktime(23, 59, 59, strftime("%m", time()), strftime("%d",time()), strftime("%Y",time()));
    $in          = 0;
    $count       = -1;
    
    while($in < $end_month) {
        $res = eF_execute("select id,timestamp from logs where users_LOGIN='$usrname' and action='login' and timestamp>$in order by timestamp LIMIT 0,1");
        list($last_visit_id, $last_visit) = mysql_fetch_row($res);
    
        if($last_visit) {
            $res = eF_execute("select id,timestamp from logs where users_LOGIN='$usrname' and action='login' and timestamp>$last_visit order by timestamp LIMIT 0,1");
            list($visit_id, $visit) = mysql_fetch_row($res);

            if (!$visit) {
                $visit = $end_month;
            }
            $res = eF_execute("select timestamp from logs where users_LOGIN='$usrname' and action not like 'login' and timestamp<$visit order by timestamp DESC LIMIT 0,1");
            list($last_action) = mysql_fetch_row($res);

            $last_action_date = date("md", $last_action);
            $last_visit_date  = date("md", $last_visit);

            $temp  = 0;
            if($last_action > $last_visit) {
                if ($last_visit_date > $temp) {
                    $count++;
                    $duration['timestamps'][$count] = $last_visit;
                    $duration[$count] = 0;
                }
                $duration[$count] += $last_action - $last_visit; 
                $temp = $last_visit_date;
            }

            $temp2 = $last_visit;
            $in    = $last_visit;
        }
        else break;
    }

    return $duration;
}

/**
* Get user online times
*
* This function calculates the time a user spent online. If the optional interval parameter is set, then 
* statistics are calculated only for this time period.
* <br/>Example:
* <code>
* $interval = array('from' => time()-86400, 'to' => time());        //Calculate statistics for the last 24 hours
* $times = eF_getUserTimes('john', $interval);
* print_r($times);
* //Returns:
*Array
*(
*    [duration] => Array
*        (
*            [0] => 19
*            [1] => 120
*            [2] => 63
*        )
*
*    [times] => Array
*        (
*           [0] => 1118770769
*           [1] => 1118824615
*           [2] => 1118824760
*        )
*)        
* </code>
*
* @param string $login The user login name
* @param array $interval The time interval to calculate statistics for
* @return array The login times and durations (in seconds)
* @version 1.0 27/10/2005
*/
function eF_getUserTimes($login, $interval = false) {
    $times = array('duration' => array(), 'time' => array(), 'session_ip' => array());

    if (isset($interval['from']) && eF_checkParameter($interval['from'], 'timestamp') && isset($interval['to']) && eF_checkParameter($interval['to'], 'timestamp')) {
        $result = eF_getTableDataFlat("logs", "timestamp, action, session_ip", "timestamp > ".$interval['from']." and timestamp < ".$interval['to']." and users_LOGIN='".$login."' and (action='login' or action = 'logout')", "timestamp");
    } else {
        $result = eF_getTableDataFlat("logs", "timestamp, action, session_ip", "users_LOGIN='".$login."' and (action='login' or action = 'logout')", "timestamp");
    }
    
    if (sizeof($result) > 0) {
        for ($i = 0; $i < sizeof($result['action']) - 1; $i++) {                                            //The algorithm goes like this: We search for the 'login' actions in the log. When one is found, then we search either for the next 'login' or 'logout' action, if there are no other actions, or the last non-login or logout action. This way, we calculate the true time spent inside the system. If we calculated only the logout-login times, then when a user had closed a window without logging out first, the online time would be reported falsely
            if ($result['action'][$i] == 'login') {
                $count      = $i + 1;
                $end_action = $result['timestamp'][$count];
                while ($result['action'][$count] != 'logout' && $result['action'][$count] != 'login' && $count < sizeof($result['action'])) {
                    $end_action = $result['timestamp'][$count];
                    $count++;
                }
                if ($end_action - $result['timestamp'][$i] <= 3600){    //only take into account intervals less than one hour
                    $times['duration'][]   = $end_action - $result['timestamp'][$i];
                    $times['time'][]       = $result['timestamp'][$i];
                    $times['session_ip'][] = eF_decodeIP($result['session_ip'][$i]);
                }
            }
        }
    }

    return $times;
}

function ef_getProfessorStudents($login){
    $sql = "select distinct(ul.users_LOGIN) from users_to_lessons ul, users u where ul.users_LOGIN = u.login and u.user_type = 'student' and exists (select l.lessons_ID from users_to_lessons l where ul.lessons_ID=l.lessons_ID and l.users_LOGIN='$login') order by ul.users_LOGIN";
    $res = ef_execute($sql);   
    $users = array();
    while ($k = mysql_fetch_row($res)){
        $user = array();
        $user['login']= $k[0];
        $users[] = $user;
    }    
    return $users;
}



function ef_getUserStatistics($login, $interval){
    $traffic = array();
    $traffic['lessons_name'] = array();
    $traffic['lessons_id'] = array();
    $traffic['lessons_access'] = array();
    $traffic['lessons_time'] = array();
    
    $sql = "select count(id) from logs where users_LOGIN='$login' and timestamp between ".$interval['from']." and ".$interval['to']." and action='login'";
    $result = ef_execute($sql);
    $k=mysql_fetch_row($result);
    $traffic['logins']=$k[0];
    
    $sql = "select s.name, l.lessons_ID, count(l.id) as cnt from logs l, lessons s where l.users_LOGIN='$login' and l.timestamp between ".$interval['from']." and ".$interval['to']." and l.lessons_ID=s.id  group by s.name, l.lessons_ID";
    $result = ef_execute($sql);
    $cnt = 0;
    while ($k = mysql_fetch_row($result)){
        $cnt += $k[2];
        $traffic['lessons_name'][]=$k[0];
        $traffic['lessons_id'][]=$k[1];
        $traffic['lessons_access'][]=$k[2];
        $traffic['lessons_time'][]=ef_getUserLessonTime($k[1],$login,$interval['from'],$interval['to']);
    }
    $traffic['total_access'] = $cnt;
    return $traffic;
}


function ef_getUserLoginTime($login, $interval){
    $from = $interval['from'];
    $to = $interval['to'];
    $sql = "select id, timestamp, action from logs where users_LOGIN='$login' and timestamp between $from and $to order by timestamp";
    $res = ef_execute($sql);
    $total_time = 0;
    $l_start = 0;
    $inlogin = 0;
    while ($k = mysql_fetch_row($res)){
        $c_time = $k[1];
        $c_action = $k[2];
        
        if ($inlogin){
            if ($c_action != 'logout' && $c_action != 'login'){
                $total_time += ($c_time - $l_start);
                $l_start = $c_time;
            }    
            else if ($c_action == 'logout'){
                if ($c_time < ($l_start + 3600))//if it is logged in more than an hour, we don't consider it
                    $total_time += ($c_time - $l_start);
                $inlogin = 0; 
            }
            else if ($c_action == 'login'){
                $inlogin = 1;
                $l_start = $c_time;
            }
        }
        else{
            if ($c_action == 'login'){
                $inlogin = 1;
                $l_start = $c_time;
            }
        }
    }       
    return ef_convertIntervalToTime($total_time);
}

function ef_getSystemStatistics($interval){
    $from = $interval['from'];
    $to = $interval['to'];
    $statistics = array();
    $sql = "select count(id) from logs where action='login' and timestamp between $from and $to ";
    $res = ef_execute($sql);
    $k = mysql_fetch_row($res);
    $statistics['total_logins']=$k[0];
    return $statistics;
}

function ef_getLessonAccessStatistics($interval, $lesson, $login)
{
    $from = $interval['from'];
    $to = $interval['to'];
    $statistics = array();
    $sql = "select count(id) from logs where lessons_ID=$lesson and timestamp between $from and $to and users_LOGIN='$login'";
    $res = ef_execute($sql);
    $k = mysql_fetch_row($res);
    $statistics['total_access']=$k[0];
    return $statistics;
}

function ef_getUserLogins($interval, $login)
{
    $from = $interval['from'];
    $to = $interval['to'];
    $statistics = array();
    $sql = "select count(id) from logs where action='login' and timestamp between $from and $to and users_LOGIN='$login'";
    $res = ef_execute($sql);
    $k = mysql_fetch_row($res);
    $statistics['total_logins']=$k[0];
    return $statistics;
}

function ef_getActiveUsers($interval){
    $from = $interval['from'];
    $to = $interval['to'];
    $sql = "select users_LOGIN as login, count(id) as cnt from logs where timestamp between $from and $to and action='login' group by users_LOGIN order by count(id) desc";
    $res = ef_execute($sql);
    $activeusers = array();
    $n = 0;
    $totalseconds = 0;
    while ( ($k = mysql_fetch_row($res)) && $n<20 ){
        $activeusers['login'][]=$k[0];
        $activeusers['access'][]=$k[1];
        $time = ef_getUserLoginTime($k[0],$interval);
        $secs = $time['hours']*3600+$time['minutes']*60+$time['seconds'];
        $totalseconds+=$secs;
        $activeusers['secs'][] = $secs;
        $activeusers['time'][]=$time;
        $n++;
    }   
    array_multisort($activeusers['secs'],SORT_DESC,SORT_NUMERIC,$activeusers['access'],SORT_DESC,SORT_NUMERIC,$activeusers['login'],SORT_DESC,SORT_STRING,$activeusers['time'] );
    $totalhours = floor($totalseconds/3600);
    $totmins = floor(($totalseconds-3600*$totalhours)/60);
    $totsecs = ($totalseconds-3600*$totalhours-60%$totmins) % 60;
    $activeusers['totaltime'] = array();
    $activeusers['totaltime']['hours']=$totalhours;
    $activeusers['totaltime']['minutes']=$totmins;
    $activeusers['totaltime']['seconds']=$totsecs;
    
    return $activeusers;    
}


function ef_getActiveLessons($interval){
    $from = $interval['from'];
    $to = $interval['to'];
    $sql = "select g.lessons_ID, l.name,  count(g.id) from logs g, lessons l where timestamp between $from and $to and g.lessons_ID=l.ID group by g.lessons_ID, l.name order by count(g.id) desc";
    $res = ef_execute($sql);
    $activelessons = array();
    while ( ($k = mysql_fetch_row($res))){
        $activelessons['id'][]=$k[0];
        $activelessons['name'][]=$k[1];
        $activelessons['access'][]=$k[2];
        $tr = ef_getLessonTraffic($k[0],$interval['from'],$interval['to']);
        $activelessons['time'][] = $tr['total_time'];   
    }   
    return $activelessons;  
}


function ef_getSystemLog($interval){
    $from = $interval['from'];
    $to = $interval['to'];
    $log = array();
    $sql = "select id, users_LOGIN, action, timestamp from logs where timestamp between $from and $to order by timestamp";
    $res = ef_execute($sql);
    while ($k=mysql_fetch_row($res)){
        $log['id'][] = $k[0];
        $log['login'][] = $k[1];
        $log['action'][] = $k[2];
        $log['time'][] = date("d/m/Y h:i:s",$k[3]);
    }
    return $log;
}

function ef_getLessonLog($lesson, $interval){
    $from = $interval['from'];
    $to = $interval['to'];
    $log = array();
    $sql = "select id, users_LOGIN, action, timestamp from logs where timestamp between $from and $to and lessons_id='$lesson' order by timestamp";
    $res = ef_execute($sql);
    while ($k=mysql_fetch_row($res)){
        $log['id'][] = $k[0];
        $log['login'][] = $k[1];
        $log['action'][] = $k[2];
        $log['time'][] = date("d/m/Y h:i:s",$k[3]);
    }
    return $log;
}

function ef_getUserLog($login, $interval){
    $from = $interval['from'];
    $to = $interval['to'];
    $log = array();
    $sql = "select id, users_LOGIN, action, timestamp from logs where timestamp between $from and $to and users_LOGIN='$login' order by timestamp";
    $res = ef_execute($sql);
    while ($k=mysql_fetch_row($res)){
        $log['id'][] = $k[0];
        $log['login'][] = $k[1];
        $log['action'][] = $k[2];
        $log['time'][] = date("d/m/Y h:i:s",$k[3]);
    }
    return $log;
}

/**
* Calculates the system users per day
*
* This function calculates how many users have entered the system. By default, the data concern 
* An interval of ten days, starting from today. Alternatively, the $dates_array may be set, were
* each element contains a timestamp, corresponding to a day for which we need information. It returns
* an array of equal size, or 10 if no arguments are specified, containing the number of users logged in
* each day.
* 
* @param array $dates_array An array of timestamps, corresponding to days we need information for
* @param int $limit The maximum number of entries
* @return array An array with the students per day that logged in the system
* @version 1.1
* Changes from 1.0 to 1.1 (6/10/05):
* - Rewritten many lines, in order to make it more compatible and shorter
*/
function eF_usersPerDay($dates_array, $limit = 1000)
{
    if ($dates_array == 0) {
        $today_end = mktime(23, 59, 59, strftime("%m", time()), strftime("%d", time()), strftime("%Y",time()));
        $time[0]   = $today_end;
        $interval  = 10;
        
        for($i = 0; $i < 12; $i++) {
            $time[$i+1] = mktime(0, 0, 0, strftime("%m", time()), strftime("%d", time()) - $i, strftime("%Y", time()));
        }
    } else {
        $time[0]   = $dates_array['to'] + 86400;
        $interval  = ($dates_array['to'] - $dates_array['from']) / 86400 + 1;

        for($i = 0; $i < $interval & $i <= $limit; $i++) {
            $time[$i+1] = $time[$i] - 86400;
        }
    }

    for($i = 0; $i < $interval && $i < $limit; $i++) {
        $result            = eF_getTableData("logs l,users u", "distinct(users_LOGIN)", "l.timestamp<".$time[$i]." and l.timestamp>".$time[$i+1]." and l.users_LOGIN=u.login and (u.user_type='student' or u.user_type='professor')");
        $users_per_day[$i] = sizeof($result);
    }    

    return $users_per_day;
}



/**
* Provides information on system usage
*
* This function retrieves the users that logged in to the system duriong the specified interval.
* If a user logged in many times in one day, then it reports only the last one. For full report 
* see eF_allUsersInDays(). The function returns an array, each element of which is another array 
* with the fields:
* - 'timestamp' : The time of the last login
* - 'usrname'   : The user login name
* - 'usertype'  : The user type
* <br/> Example:
* <code>
* $users_Array = eF_usersInDays(28, 3, 2005, 31, 3, 2005);            //Calculate information for the dates between 28/3/2005 and 31/3/2005
* </code>
* Outputs:
* <code>
* Array
* (
*     [0] => Array
*         (
*             [timestamp] => 1107273158
*             [usrname] => admin
*             [usertype] => administrator
*         )
* 
*     [1] => Array
*         (
*             [timestamp] => 1107297691
*             [usrname] => periklis
*             [usertype] => professor
*         )
* 
* )
* </code>
*
* @param int $from_day The day of the month from which to calculate statistics for
* @param int $from_month The month of the year from which to calculate statistics for 
* @param int $from_year The year from which to calculate statistics for
* @param int $to_day The day of the month until which to calculate statistics for
* @param int $to_month The month of the year until which to calculate statistics for 
* @param int $to_year The year until which to calculate statistics for
* @return array The results array
* @version 1.0
* @deprecated
*/
function eF_usersInDays($from_day, $from_month, $from_year, $to_day, $to_month, $to_year)
{
    $start = mktime(0, 0, 0, $from_month, $from_day, $from_year);
    $end   = mktime(23, 59, 59, $to_month, $to_day, $to_year);
    $res   = eF_execute("select distinct(users_LOGIN) from logs where timestamp>$start and timestamp<$end and action='login'");
    $users   = eF_getTableData("logs", "distinct(users_LOGIN)", "timestamp>$start and timestamp<$end and action='login'");
    foreach ($users as $user) {
        $result = eF_getTableData("logs l, users u", "min(l.timestamp), u.user_type", "l.users_LOGIN='$user' and l.timestamp>$start and l.timestamp<$end and l.action='login' and u.login = l.users_LOGIN", "", "(u.user_type)" );
    }
    //print_r($result);
    $count = 0;
    while (list($usrname) = mysql_fetch_row($res)) {
        $res2 = eF_execute("select min(timestamp) from logs where users_LOGIN='$usrname' and timestamp>$start and timestamp<$end and action='login'");
        list($timestamp) = mysql_fetch_row($res2);
        $res3 = eF_execute("select user_type from users where login='$usrname'");
        list($usertype) = mysql_fetch_row($res3);
        $result[$count]['timestamp'] = $timestamp;
        $result[$count]['usrname']   = $usrname;
        $result[$count]['usertype']  = $usertype;
        $count++;
    }

    return $result;
}


/**
* Provides complete information on system usage
*
* This function retrieves the users that logged in to the system during the specified interval.
* For each user, full data are displayed, that is exactly all the times that he logged in and out
* of the system. The function returns an array, each element of which is another array with the fields:
* - 'timestamp' : The time of the last login
* - 'action'    : The action that took place, 'login' or 'logout'
* - 'usrname'   : The user login name
* - 'usertype'  : The user type
* <br/> Example:
* <code>
* $users_Array = eF_usersInDays(28, 3, 2005, 31, 3, 2005);            //Calculate information for the dates between 28/3/2005 and 31/3/2005
* </code>
* Outputs:
* <code>
* Array
* (
*     [0] => Array
*         (
*             [usertype] => administrator
*             [action] => login
*             [timestamp] => 1107273158
*             [usrname] => admin
*         )
* 
*     [1] => Array
*         (
*             [usertype] => administrator
*             [action] => logout
*             [timestamp] => 1107273163
*             [usrname] => admin
*         )
* 
* )
* </code>
*
* @param int $from_day The day of the month from which to calculate statistics for
* @param int $from_month The month of the year from which to calculate statistics for 
* @param int $from_year The year from which to calculate statistics for
* @param int $to_day The day of the month until which to calculate statistics for
* @param int $to_month The month of the year until which to calculate statistics for 
* @param int $to_year The year until which to calculate statistics for
* @return array The results array
* @version 1.0
* @deprecated
*/
function eF_allUsersInDays($from_day, $from_month, $from_year, $to_day, $to_month, $to_year)
{
    $start = mktime(0,0,0,$from_month,$from_day,$from_year);
    $end   = mktime(23,59,59,$to_month,$to_day,$to_year);
    $res   = eF_execute("select distinct(users_LOGIN) from logs where timestamp>$start and timestamp<$end and action='login'");
    $count = 0;

    while (list($usrname) = mysql_fetch_row($res)) {
        $res2 = eF_execute("select timestamp,action from logs where users_LOGIN='$usrname' and timestamp>$start and timestamp<$end and (action='login' or action='logout')");
        $res3 = eF_execute("select user_type from users where login='$usrname'");
        list($usertype) = mysql_fetch_row($res3);
        while(list($timestamp, $action) = mysql_fetch_row($res2)) {
            $result[$count]['usertype']  =$usertype;
            $result[$count]['action']    =$action;
            $result[$count]['timestamp'] =$timestamp;
            $result[$count]['usrname']   =$usrname;
            $count++;
        }
    }

    return $result;
}


/**
* Retrieves the users that have done a test.
*
* This function retrieves a list of users that have done test with id $test_id,
* along with their score. The results are returned in an array with 2 fields, each
* being a separate array. The first of them is a list with the users that completed
* the test and the second a list with the corresponding scores.
* <br/>Example:
* <code>
* $test_data = eF_usersDoneTests(1);            
* </code>
* Outputs:
* <code>
* Array
* (
*     [0] => Array
*         (
*             [0] => niki
*             [1] => peris
*         )
* 
*     [1] => Array
*         (
*             [0] => 1
*             [1] => 0.4
*         )
* 
* )
* </code>
* 
* @param int $test_id The id of the test 
* @return array The results array
* @version 1.0
*/
function eF_usersDoneTests($test_id) {
    $res = eF_getTableData("done_tests", "users_LOGIN,score", "tests_ID=".$test_id);
    for ($i = 0; $i < sizeof($res); $i++) {
        $student[$i] = $res[$i]['users_LOGIN'];
        $score[$i]   = $res[$i]['score'];
    }
    return array($student,$score);
}


/**
* Calculates a list with the content that a user has seen
* 
* This function returns an array containing the total units of a lesson, and 
* the units that the user has seen until now. This information is grouped by 
* unit type, in 4 categories: Theory, Exercises, Examples and Tests.
* <br/>Example:
* <code>
* $content_done = eF_usersContentDone('student', 1);
* </code>
* Outputs:
* <code>
* Array
* (
*     [theory] => 25
*     [theory_seen] => 5
*     [exercises] => 2
*     [exercises_seen] => 0
*     [examples] => 6
*     [examples_seen] => 0
*     [tests] => 9
*     [tests_done] => 3
* )
* </code>
*
* @param string $usrname The login name of the user
* @param id $lesson The lesson id
* @return array The results array.
* @version 1.0
*/
/*
function eF_usersContentDone($usrname, $lesson) 
{
$periods = eF_getTableData("periods", "count(*)", "lessons_ID=".$_SESSION['s_lessons_ID']);
    if ($periods[0]['count(*)'] == 0) {  //makriria lines. gia tin ypostiriksi ton periodon sta statistika   
        $res = eF_execute("select count(*) from content where lessons_ID=$lesson and ctg_type='theory' and active=1");
        list($theory)         = mysql_fetch_row($res);    
        $res = eF_execute("select count(distinct l.comments) from logs l,content c where l.users_LOGIN='$usrname' and l.action='theory' and l.comments=c.id and c.lessons_ID=$lesson and c.active=1 ");
        list($theory_seen)    = mysql_fetch_row($res);
//print_r($theory_seen);    
        $res = eF_execute("select count(*) from content where lessons_ID=$lesson and ctg_type='exercises' and active=1");
        list($exercises)      = mysql_fetch_row($res);
        $res = eF_execute("select count(distinct l.comments) from logs l,content c where l.users_LOGIN='$usrname' and l.action='exercises' and l.comments=c.id and c.lessons_ID=$lesson and c.active=1 ");
        list($exercises_seen) = mysql_fetch_row($res);
        $res = eF_execute("select count(*) from content where lessons_ID=$lesson and ctg_type='examples' and active=1");
        list($examples)       = mysql_fetch_row($res);
        $res = eF_execute("select count(distinct l.comments) from logs l,content c where l.users_LOGIN='$usrname' and l.action='examples' and l.comments=c.id and c.lessons_ID=$lesson and c.active=1 ");
        list($examples_seen)  = mysql_fetch_row($res);
        $res = eF_execute("select count(*) from content where lessons_ID=$lesson and ctg_type='tests' and active=1");
        list($tests)          = mysql_fetch_row($res);
        $res = eF_execute("select count(*) from done_tests d,content c,tests t where d.users_LOGIN='$usrname' and d.tests_ID=t.id and t.content_ID=c.id and c.lessons_ID=$lesson and c.active=1 ");
        list($tests_done)     = mysql_fetch_row($res);
    } else {
        $res = eF_execute("select count(*) from content c, periods p, current_content cc where c.lessons_ID=$lesson and ctg_type='theory' and active=1 AND p.lessons_ID = c.lessons_ID AND p.id = cc.periods_ID AND cc.content_ID = c.id AND p.from_timestamp <".time()." AND p.to_timestamp >".time());
        list($theory)         = mysql_fetch_row($res);    
        $res = eF_execute("select count(distinct l.comments) from logs l,content c, periods p, current_content cc where l.users_LOGIN='$usrname' and l.action='theory' and l.comments=c.id and c.lessons_ID=$lesson and c.active=1 AND p.lessons_ID = c.lessons_ID AND p.id = cc.periods_ID AND cc.content_ID = c.id AND p.from_timestamp <".time()." AND p.to_timestamp >".time());
        list($theory_seen)    = mysql_fetch_row($res);
//print_r($theory_seen);            
        $res = eF_execute("select count(*) from content c, periods p, current_content cc where c.lessons_ID=$lesson and ctg_type='exercises' and active=1 AND p.lessons_ID = c.lessons_ID AND p.id = cc.periods_ID AND cc.content_ID = c.id AND p.from_timestamp <".time()." AND p.to_timestamp >".time());
        list($exercises)      = mysql_fetch_row($res);      
        $res = eF_execute("select count(distinct l.comments) from logs l,content c, periods p, current_content cc where l.users_LOGIN='$usrname' and l.action='exercises' and l.comments=c.id and c.lessons_ID=$lesson and c.active=1 AND p.lessons_ID = c.lessons_ID AND p.id = cc.periods_ID AND cc.content_ID = c.id AND p.from_timestamp <".time()." AND p.to_timestamp >".time());
        list($exercises_seen) = mysql_fetch_row($res);
        $res = eF_execute("select count(*) from content c, periods p, current_content cc where c.lessons_ID=$lesson and ctg_type='examples' and active=1 AND p.lessons_ID = c.lessons_ID AND p.id = cc.periods_ID AND cc.content_ID = c.id AND p.from_timestamp <".time()." AND p.to_timestamp >".time());
        list($examples)       = mysql_fetch_row($res);
        $res = eF_execute("select count(distinct l.comments) from logs l,content c, periods p, current_content cc where l.users_LOGIN='$usrname' and l.action='examples' and l.comments=c.id and c.lessons_ID=$lesson and c.active=1 AND p.lessons_ID = c.lessons_ID AND p.id = cc.periods_ID AND cc.content_ID = c.id AND p.from_timestamp <".time()." AND p.to_timestamp >".time());
        list($examples_seen)  = mysql_fetch_row($res);
        $res = eF_execute("select count(*) from content c, periods p, current_content cc where c.lessons_ID=$lesson and ctg_type='tests' and active=1 AND p.lessons_ID = c.lessons_ID AND p.id = cc.periods_ID AND cc.content_ID = c.id AND p.from_timestamp <".time()." AND p.to_timestamp >".time());
        list($tests)          = mysql_fetch_row($res);
        
        $res = eF_execute("select count(*) from done_tests d,content c,tests t where d.users_LOGIN='$usrname' and d.tests_ID=t.id and t.content_ID=c.id and c.lessons_ID=$lesson and c.active=1 ");
        list($tests_done)     = mysql_fetch_row($res); //na to do



  
}



    $content_stats = array('theory'         => $theory, 
                           'theory_seen'    => $theory_seen,
                           'exercises'      => $exercises,
                           'exercises_seen' => $exercises_seen,
                           'examples'       => $examples,
                           'examples_seen'  => $examples_seen,
                           'tests'          => $tests,
                           'tests_done'     => $tests_done
        );
    return $content_stats;
}
*/
function eF_usersContentDone($usrname, $lesson) 
{
    $tmp = eF_getStudentProgress($usrname, $lesson);
    $theory_perc = $tmp[$lesson];
    $res = eF_execute("select count(*) from content where lessons_ID=$lesson and ctg_type='exercises' and active=1");
    list($exercises)      = mysql_fetch_row($res);
    $res = eF_execute("select count(distinct l.comments) from logs l,content c where l.users_LOGIN='$usrname' and l.action='exercises' and l.comments=c.id and c.lessons_ID=$lesson and c.active=1 ");
    list($exercises_seen) = mysql_fetch_row($res);
    if ($exercises > 0){
        $exercises_perc = round(($exercises_seen/$exercises)*100,2);
    }
    else
        $exercices_perc = 0;    
    $res = eF_execute("select count(*) from content where lessons_ID=$lesson and ctg_type='examples' and active=1");
    list($examples)       = mysql_fetch_row($res);
    $res = eF_execute("select count(distinct l.comments) from logs l,content c where l.users_LOGIN='$usrname' and l.action='examples' and l.comments=c.id and c.lessons_ID=$lesson and c.active=1 ");
    list($examples_seen)  = mysql_fetch_row($res);
    if ($examples > 0){
        $examples_perc = round(($examples_seen/$examples)*100,2);
    }
    else
        $examples_perc = 0;    
    $res = eF_execute("select count(*) from content where lessons_ID=$lesson and ctg_type='tests' and active=1");
    list($tests)          = mysql_fetch_row($res);
    $res = eF_execute("select count(*) from done_tests d,content c,tests t where d.users_LOGIN='$usrname' and d.tests_ID=t.id and t.content_ID=c.id and c.lessons_ID=$lesson and c.active=1 ");
    list($tests_done)     = mysql_fetch_row($res);
    if ($tests > 0){
        $tests_perc = round(($tests_done/$tests)*100,2);
    }
    else
        $tests_perc = 0;
    
    $res = eF_execute("select count(*) from users_to_projects u, projects p where u.users_LOGIN='$usrname' and u.projects_ID=p.id and p.lessons_ID=$lesson");
    list($projects) = mysql_fetch_row($res);
    $res = eF_execute("select count(*) from users_to_projects u, projects p where u.users_LOGIN='$usrname' and u.projects_ID=p.id and p.lessons_ID=$lesson and u.grade!=''");
    list($projects_done)     = mysql_fetch_row($res);
    if ($projects > 0){
        $projects_perc = round(($projects_done/$projects)*100,2);
    }
    else
        $projects_perc = 0;
    
    $content_stats = array('theory'         => $theory, 
                           'theory_seen'    => $theory_seen,
                           'theory_perc'    => $theory_perc,
                           'exercises'      => $exercises,
                           'exercises_seen' => $exercises_seen,
                           'exercises_perc' => $exercises_perc,
                           'examples'       => $examples,
                           'examples_seen'  => $examples_seen,
                           'examples_perc'  => $examples_perc,
                           'tests'          => $tests,
                           'tests_done'     => $tests_done,
                           'tests_perc'     => $tests_perc,
                           'projects'          => $projects,
                           'projects_done'     => $projects_done,
                           'projects_perc'     => $projects_perc
        );

    return $content_stats;
}

/**
* Calculates folder statistics.
*
* This function is used to calculate how many files and folders are contained inside a lesson folder. It 
* also calculates the total size of these files. The parameter given corresponds to the lesson id, which
* coincides with the lesson folder name inside the content/ directory. The function returns a 3-values array, 
* whith the first element being the total files number, the second the total folders and the third the total size.
* <br/>Example<br/>:
* <code>
* $files_array = EfrontLessonFiles(3);                                         //3 is the lesson id
* print_r($files_array);
* //Outputs:
*Array
*(
*    [0] => 10
*    [1] => 2
*    [2] => 1791160
*)
* </code>
* @param string $lesson_id The lesson id
* @return array An array with 3 elements: number of files, number of directories and total file size, in bytes. if the lesson does not exist it returns false.
* @version 1.0
*/
function EfrontLessonFiles($lesson_id) 
{
    $dir = G_LESSONSPATH.$lesson_id;

    if ($handle = @opendir($dir)) {                                         //Check if lesson directory exists
        $stats = eF_diveIntoDir($dir);                                        //Calculate statistics
    } else {
        $stats = false;
    }
    
    return $stats;
}

/**
* Calculates statistics for the designated folder.
*
* This function is used to recursively calculate statistcs for the designated folder.
* Statistics include number of files and folders as well as total file size.
* It returns a 3-valued array containing these values
*
* @param string $dir The directory name
* @return array An array with 3 elements, or false if the lesson does not exist
* @version 1.0
* @see EfrontLessonFiles
*/
function eF_diveIntoDirOld($dir)
{
    $file_count = 0;
    $dir_count  = 0;
    $file_size  = 0;                                                    //Initialize statistics
    
    if ($handle = @opendir($dir)) {
        while (false !== ($file = readdir($handle)))  {
            if (is_dir($dir.'/'.$file)) {
                if ($file != "." && $file != "..") {
                    $dirs[++$dir_count] = $dir.'/'.$file;
                    eF_diveIntoDir($dir.'/'.$file);                     //Recurse into directory
                }
            } else {
                $files[++$file_count] = $file;

                $file_stat  = stat($dir.'/'.$file);
                $file_size += $file_stat[7];
            }
        }
    }
    return array($file_count, $dir_count, $file_size);
}

/**
* Calculates statistics for the designated folder.
*
* This function is used to recursively calculate statistics for the designated folder.
* Statistics include number of files and folders as well as total file size.
* It returns a 3-valued array containing these values.
*
* @param string $dir The directory name
* @param int $files Total files to continue counting from
* @param int $directories Total directories to continue counting from
* @param int $file_size Total file size to continue counting from
* @return array An array with 3 elements, total files, directories and file size
* @version 1.2.1
* @see EfrontLessonFiles
* changes from 1.2 to 1.2.1:
* - Fixed bug in file size reporting
* changes from version 1.0 (renamed to eF_diveIntoDirOld) to 1.2:
* - Rewritten because it was buggy (it displayed only the root folder statistics)
*/
function eF_diveIntoDir($dir, $files = 0, $directories = 0, $file_size = 0)
{    
    if ($handle = @opendir($dir)) {
        $in_directories = 0;
        while (false !== ($file = readdir($handle)))  {
            if (is_dir($dir.'/'.$file)) {
                if ($file != "." && $file != "..") {
                    $new_dir = $dir.'/'.$file;
                    $directories++;
                    list($files, $directories, $file_size) = eF_diveIntoDir($new_dir, $files, $directories, $file_size);
                }
            } else {
                $files++;
                $file_stat = stat($dir.'/'.$file);
                $file_size += $file_stat[7];
            }
        }
    }

    return array($files, $directories, $file_size);
}

/*
*
*/
function eF_getUsersInfoOLD($lessons_ID = false, $sort = false, $order = false) {
    if ($lessons_ID && eF_checkParameter($lessons_ID, 'id')) {
        $lessons_sql = ' and ul.lessons_ID='.$lessons_ID;
    } else {
        $lessons_sql = '';
    }

    $sort_values = array("login", "name", "timestamp", "languages_NAME", "email");
    if (!in_array($sort, $sort_values)) {
        $sort = 'login';
    }
    
    if ($order && in_array($order, array('desc', 'asc'))) {
        $sort .= " $order";
    }
    //$users = eF_getTableData("users u, users_to_lessons ul", "u.login as bbb, ul.users_LOGIN as aaa", "ul.users_LOGIN=u.login".$lessons_sql, "u.user_type");
    $users_info['admins']     = eF_getTableData("users", "login, name, timestamp, languages_NAME, email, active", "user_type = 'administrator'", 'login');
    $users_info['professors'] = eF_getTableData("users", "login, name, timestamp, languages_NAME, email, active", "user_type = 'professor'", 'login');
    $users_info['students']   = eF_getTableData("users", "login, name, timestamp, languages_NAME, email, active", "user_type = 'student'", 'login');
    
    //print_r($user_info);
    return $users_info;
}

/**
*
*/
function eF_getUsersGeneralInfo($user_type, $lessons_ID = false) {

    if ($lessons_ID && eF_checkParameter($lessons_ID, 'id')) {
        switch ($user_type) {
            case 'administrator':    
                $users_info = array();
            break;
            case 'professor':
                $users_info = eF_getTableDataFlat("users u, users_to_lessons ul", "u.login, u.name, u.surname, u.timestamp, u.languages_NAME, u.email, u.active", "ul.lessons_ID = $lessons_ID and ul.users_LOGIN = u.login and user_type = 'professor'", 'login');
            break;
            case 'student':        
                $users_info = eF_getTableDataFlat("users u, users_to_lessons ul", "u.login, u.name, u.surname, u.timestamp, u.languages_NAME, u.email, u.active", "ul.lessons_ID = $lessons_ID and ul.users_LOGIN = u.login and user_type = 'student'", 'login');
            break;
            default: break;
        }
        
    } else {
        switch ($user_type) {
            case 'administrator':    
                $users_info = eF_getTableDataFlat("users", "login, name, surname, timestamp, languages_NAME, email, active", "user_type = 'administrator'", 'login');
            break;
            case 'professor':
                $users_info = eF_getTableDataFlat("users", "login, name, surname, timestamp, languages_NAME, email, active", "user_type = 'professor'", 'login');
            break;
            case 'student':        
                $users_info = eF_getTableDataFlat("users", "login, name, surname, timestamp, languages_NAME, email, active", "user_type = 'student'", 'login');
            break;
            default: break;
        }
    }
    
    return $users_info;
}


/**
*
*/
function eF_getUsersActivityInfo($user_type, $lessons_ID = false, $group_id = false) {

    switch ($user_type) {
        case 'administrator':    
            $admins = eF_getTableDataFlat("users", "login", "user_type = 'administrator'");
            $admins = $admins['login'];
        
            $users_info = array('logins' => $admins);
        break;
        
        case 'professor':
            if ($lessons_ID && eF_checkParameter($lessons_ID, 'id')) {
                if ($group_id && eF_checkParameter($group_id, 'id'))
                    $professors = eF_getTableDataFlat("users u, users_to_lessons ul, users_to_groups g", "u.login,u.name,u.surname", "g.users_LOGIN = u.login and g.groups_id=$group_id and  u.login=ul.users_LOGIN and ul.lessons_ID = $lessons_ID and u.user_type = 'professor'");
                else
                    $professors = eF_getTableDataFlat("users u, users_to_lessons ul", "u.login,u.name,u.surname", "u.login=ul.users_LOGIN and ul.lessons_ID = $lessons_ID and u.user_type = 'professor'");
            } else {
                $professors = eF_getTableDataFlat("users", "login", "user_type = 'professor'");
            }
        $names = $professors['name'];
        $surnames = $professors['surname'];
            $professors = $professors['login'];

            for ($i = 0; $i < sizeof($professors); $i++) {
                if ($lessons_ID && eF_checkParameter($lessons_ID, 'id')) {
                    $temp_user_lessons = eF_getTableDataFlat("users_to_lessons, lessons", "lessons.id, lessons.name", "lessons.id = users_to_lessons.lessons_ID and users_LOGIN='".$professors[$i]."' and lessons.id=$lessons_ID");
                } else {
                    $temp_user_lessons = eF_getTableDataFlat("users_to_lessons, lessons", "lessons.id, lessons.name", "lessons.id = users_to_lessons.lessons_ID and users_LOGIN='".$professors[$i]."'");
                }

                $temp_num_comments = array();
                $temp_num_posts    = array();
                if (sizeof($temp_user_lessons) > 0) {
                    $professors_user_lessons[$i] = $temp_user_lessons['name'];
                    foreach ($temp_user_lessons['id'] as $id) {
                        $temp_num_comments[] = eF_numberOfComments($professors[$i], $id);
                        $temp_num_posts[]    = eF_numberOfPosts($professors[$i], $id);
                        $temp_time            = eF_getUserLessonTime($id, $professors[$i],"0000000", time());          
                    }
                } else {
                    $professors_user_lessons[$i] = array();
                }

                $professors_num_comments[$i] = $temp_num_comments;
                $professors_num_posts[$i]    = $temp_num_posts;
                $professors_lesson_time[$i]   = $temp_time;
            }
            $users_info = array('logins'       => $professors,
                                'name'         => $names,
                                'surname'      => $surnames,
                                'lessons'      => $professors_user_lessons, 
                                'time'         => $professors_lesson_time,
                                'num_comments' => $professors_num_comments, 
                                'num_posts'    => $professors_num_posts);
        break;
        
        case 'student':
            if ($lessons_ID && eF_checkParameter($lessons_ID, 'id')) {
                if ($group_id && ef_checkParameter($group_id, 'id')){
                    $students = eF_getTableDataFlat("users u, users_to_lessons ul, users_to_groups g", "u.login,u.name,u.surname", "g.users_LOGIN = u.login and g.groups_ID = $group_id and u.login=ul.users_LOGIN and ul.lessons_ID = $lessons_ID and u.user_type = 'student'");
                }
                else{
                    $students = eF_getTableDataFlat("users u, users_to_lessons ul", "u.login,u.name,u.surname", "u.login=ul.users_LOGIN and ul.lessons_ID = $lessons_ID and u.user_type = 'student'");
                }
            } else {
                $students = eF_getTableDataFlat("users", "login", "user_type = 'student'");
            }
            $names = $students['name'];
            $surnames = $students['surname'];
            $students = $students['login'];

            for ($i = 0; $i < sizeof($students); $i++) {
                if ($lessons_ID && eF_checkParameter($lessons_ID, 'id')) {
                    $temp_user_lessons = eF_getTableDataFlat("users_to_lessons, lessons", "lessons.id, lessons.name", "lessons.id = users_to_lessons.lessons_ID and users_LOGIN='".$students[$i]."' and lessons.id=$lessons_ID");                    
                } else {
                    $temp_user_lessons = eF_getTableDataFlat("users_to_lessons, lessons", "lessons.id, lessons.name", "lessons.id = users_to_lessons.lessons_ID and users_LOGIN='".$students[$i]."'");                    
                    
                }
                $temp_num_comments  = array();
                $temp_content_done  = array();
                $temp_average_score = array();
                $temp_project_score = array();
                $temp_num_posts     = array();
                if (sizeof($temp_user_lessons) > 0) {
                    $students_user_lessons[$i] = $temp_user_lessons['name'];
                    foreach ($temp_user_lessons['id'] as $id) {
                        $temp_num_comments[]  = eF_numberOfComments($students[$i], $id);
                        $temp_content_done[]  = eF_usersContentDone($students[$i], $id);
                        $temp_num_posts[]     = eF_numberOfPosts($students[$i], $id);
                        $temp_time            = eF_getUserLessonTime($id, $students[$i],"0000000", time());          
                        $score                = eF_getTableData("done_tests, tests, content", "avg(score)", "done_tests.tests_ID = tests.id and tests.content_ID = content.id and content.lessons_ID = ".$id." and users_LOGIN='".$students[$i]."'");
                        $temp_average_score[] = round($score[0]['avg(score)'] * 100, 2);
                        $grades               = ef_getTableData("users_to_projects, projects", "avg(grade)", "users_to_projects.projects_ID = projects.id and projects.lessons_ID =".$id." and users_LOGIN='".$students[$i]."'");                        
                        $temp_project_score[] = sprintf("%01.2f", round($grades[0]['avg(grade)'], 2));
                       
                    }
                } else {
                    $students_user_lessons[$i] = array();
                }

                $students_num_comments[$i]  = $temp_num_comments;
                $students_content_done[$i]  = $temp_content_done;
                $students_average_score[$i] = $temp_average_score;
                $students_project_score[$i] = $temp_project_score;
                $students_num_posts[$i]     = $temp_num_posts;
                $students_lesson_time[$i]   = $temp_time;
            }
            $users_info = array('logins'       => $students,
                                'name'         => $names,
                                'surname'      => $surnames,
                                'time'         => $students_lesson_time,
                                'lessons'      => $students_user_lessons, 
                                'content_done' => $students_content_done, 
                                'avg_score'    => $students_average_score, 
                                'project_score'=> $students_project_score, 
                                'num_comments' => $students_num_comments, 
                                'num_posts'    => $students_num_posts);       
                                  
        break;
    }
    
    return $users_info;
}

/**
*
*/
function eF_getUsersCommunicationInfo($user_type, $lessons_ID = false) {

    if ($lessons_ID && eF_checkParameter($lessons_ID, 'id')) {
        switch ($user_type) {
            case 'administrator':    
                $users_info = array();
            break;
            case 'professor':
                $users_info = eF_getTableDataFlat("users u, users_to_lessons ul", "u.login", "ul.lessons_ID = $lessons_ID and ul.users_LOGIN = u.login and u.user_type = 'professor'", 'login');
            break;
            case 'student':        
                $users_info = eF_getTableDataFlat("users u, users_to_lessons ul", "u.login", "ul.lessons_ID = $lessons_ID and ul.users_LOGIN = u.login and u.user_type = 'student'", 'login');
            break;
            default: break;
        }
    } else {
        switch ($user_type) {
            case 'administrator':    
                $users_info = eF_getTableDataFlat("users", "login", "user_type = 'administrator'", 'login');
            break;
            case 'professor':
                $users_info = eF_getTableDataFlat("users", "login", "user_type = 'professor'", 'login');
            break;
            case 'student':        
                $users_info = eF_getTableDataFlat("users", "login", "user_type = 'student'", 'login');
            break;
            default: break;
        }
    }

    for ($i = 0; $i < sizeof($users_info['login']); $i++) {
        $login[$i]         = $users_info['login'][$i];
        $posts[$i]         = eF_numberOfPosts($users_info['login'][$i], $lessons_ID);
        $comments[$i]      = eF_numberOfComments($users_info['login'][$i], $lessons_ID);
        
        $temp              = eF_getTableData("f_personal_messages", "count(*)", "users_LOGIN='".$users_info['login'][$i]."'");
        $p_messages[$i]    = $temp[0]['count(*)'];
        $temp              = eF_diveIntoDir(G_UPLOADPATH.$users_info['login'][$i].'/message_attachments/');
        $files_space[$i]   = ceil($temp[2] / 1000);
        
        $temp              = eF_getTableData("chatmessages", "count(*)", "users_LOGIN='".$users_info['login'][$i]."'");
        $chat_messages[$i] = $temp[0]['count(*)'];
    }
    
    $users_info = array('login'         => $login,
                        'posts'         => $posts,
                        'comments'      => $comments,
                        'p_messages'    => $p_messages,
                        'files_space'   => $files_space,
                        'chat_messages' => $chat_messages);


    return $users_info;
}

/**
*
*/
function eF_getUsersUsageInfo($user_type, $lessons_ID = false)
{
    if ($lessons_ID && eF_checkParameter($lessons_ID, 'id')) {
        switch ($user_type) {
            case 'administrator':    
                $users_info = array();
            break;
            case 'professor':
                $users_info = eF_getTableDataFlat("users u, users_to_lessons ul", "u.login", "ul.lessons_ID = $lessons_ID and ul.users_LOGIN = u.login and u.user_type = 'professor'", 'login');
            break;
            case 'student':        
                $users_info = eF_getTableDataFlat("users u, users_to_lessons ul", "u.login", "ul.lessons_ID = $lessons_ID and ul.users_LOGIN = u.login and u.user_type = 'student'", 'login');
            break;
            default: break;
        }
    } else {
        switch ($user_type) {
            case 'administrator':    
                $users_info = eF_getTableDataFlat("users", "login", "user_type = 'administrator'", 'login');
            break;
            case 'professor':
                $users_info = eF_getTableDataFlat("users", "login", "user_type = 'professor'", 'login');
            break;
            case 'student':        
                $users_info = eF_getTableDataFlat("users", "login", "user_type = 'student'", 'login');
            break;
            default: break;
        }
    }

    for ($i = 0; $i < sizeof($users_info['login']); $i++) {
        $login[$i] = $users_info['login'][$i];
        
        $temp           = eF_getTableData("logs", "timestamp", "action='login' and users_LOGIN='".$users_info['login'][$i]."'", "timestamp desc");
        sizeof($temp) > 0 ? $last_login[$i] = $temp[0]['timestamp'] : $last_login[$i] = '-';

        $temp             = eF_getTableData("logs", "count(*)", "action='login' and users_LOGIN='".$users_info['login'][$i]."'");
        $total_logins[$i] = $temp[0]['count(*)'];
                
        $temp             = eF_getTableData("logs", "count(*)", "timestamp > ".(time() - 2592000)." and action='login' and users_LOGIN='".$users_info['login'][$i]."'");
        $month_logins[$i] = $temp[0]['count(*)'];

        $temp            = eF_getTableData("logs", "count(*)", "timestamp > ".(time() - 604800)." and action='login' and users_LOGIN='".$users_info['login'][$i]."'");
        $week_logins[$i] = $temp[0]['count(*)'];

        $temp = eF_getUserTimes($users_info['login'][$i]);
        sizeof($temp['duration']) > 0 ? $mean_duration[$i] = ceil((array_sum($temp['duration']) / sizeof($temp['duration'])) / 60) : $mean_duration[$i] = 0;

        $temp = eF_getUserTimes($users_info['login'][$i], array('from' => time() - 2592000, 'to' => time()));
        sizeof($temp['duration']) > 0 ? $month_mean_duration[$i] = ceil((array_sum($temp['duration']) / sizeof($temp['duration']) / 60)) : $month_mean_duration[$i] = 0;

        $temp = eF_getUserTimes($users_info['login'][$i], array('from' => time() - 604800, 'to' => time()));
        sizeof($temp['duration']) > 0 ? $week_mean_duration[$i] = ceil((array_sum($temp['duration']) / sizeof($temp['duration']) / 60)) : $week_mean_duration[$i] = 0;
    }
    
    $users_info = array('login'               => $login,
                        'last_login'          => $last_login,
                        'total_logins'        => $total_logins,
                        'month_logins'        => $month_logins,
                        'week_logins'         => $week_logins,
                        'mean_duration'       => $mean_duration,
                        'month_mean_duration' => $month_mean_duration,
                        'week_mean_duration'  => $week_mean_duration);

//$aaa = print_r(eF_getUserTimes('periklis'));
//exit;
//print_r($users_info);
//    exit;
    return $users_info;    
    
}



function eF_getUserTypesInfo($lesson_ID = false){
    if ($lesson_ID){
        $user_types = eF_getTableData("users","user_type, count(login) as cnt_total","","","user_type");
        for ($i=0;$i<sizeof($user_types);$i++){
            $user_current_type = $user_types[$i]['user_type'];
            $user_lesson = eF_getTableData("users u, users_to_lessons ul","count(u.login) as cnt_lesson","ul.users_LOGIN=u.login and u.user_type='$user_current_type' and ul.lessons_ID='$lesson_ID'");
            if (sizeof($user_lesson) > 0)
                $user_types[$i]['cnt_lesson'] = $user_lesson[0]['cnt_lesson'];
            else
                $user_types[$i]['cnt_lesson'] = 0;
        }
    }else
        $user_types = eF_getTableData("users","user_type, count(login) as cnt_total, count(login) as cnt_lesson","","","user_type");

    return $user_types;
    
}


/**
*
*/
function eF_getLessonsGeneralInfo($users_LOGIN = false) {
    if ($users_LOGIN && eF_checkParameter($users_LOGIN, 'login')) {
        $directions = eF_getTableData("directions d, lessons l, users_to_lessons ul", "distinct d.*", "l.directions_ID=d.id and ul.lessons_ID=l.id and ul.users_LOGIN='$users_LOGIN'");
    } else {
        $directions = eF_getTableData("directions", "*");
    }
    
    for ($i = 0; $i < sizeof($directions); $i++) {
        if ($users_LOGIN && eF_checkParameter($users_LOGIN, 'login')) {
            $directions[$i]['lessons'] = eF_getTableDataFlat("lessons l, users_to_lessons ul", "l.id, l.name, l.price, l.active, l.languages_NAME", "ul.users_LOGIN = '$users_LOGIN' and ul.lessons_ID = l.id and l.directions_ID=".$directions[$i]['id']);
        } else {
            $directions[$i]['lessons'] = eF_getTableDataFlat("lessons", "id, name, price, active, languages_NAME", "directions_ID=".$directions[$i]['id']);
        }
    }
    //print_r($directions);
    return $directions;
}

/**
*
*/
function eF_getLessonsParticipationInfo($users_LOGIN = false) 
{
    if ($users_LOGIN && eF_checkParameter($users_LOGIN, 'login')) {
        $directions = eF_getTableData("directions d, lessons l, users_to_lessons ul", "distinct d.*", "l.directions_ID=d.id and ul.lessons_ID=l.id and ul.users_LOGIN='$users_LOGIN'");
    } else {
        $directions = eF_getTableData("directions", "*");
    }

    for ($j = 0; $j < sizeof($directions); $j++) {
        if ($users_LOGIN && eF_checkParameter($users_LOGIN, 'login')) {
            $directions[$j]['lessons'] = eF_getTableDataFlat("lessons l, users_to_lessons ul", "l.id, l.name", "ul.users_LOGIN = '$users_LOGIN' and ul.lessons_ID = l.id and l.directions_ID=".$directions[$j]['id']);        
        } else {
            $directions[$j]['lessons'] = eF_getTableDataFlat("lessons", "id, name", "directions_ID=".$directions[$j]['id']);        
        }
        
        for ($i = 0; $i < sizeof($directions[$j]['lessons']['id']); $i++) {
            $temp   = eF_getTableData("users_to_lessons ul, users u", "count(*)", "u.user_type = 'student' and u.login = ul.users_LOGIN and ul.lessons_ID=".$directions[$j]['lessons']['id'][$i]);
            $directions[$j]['lessons']['num_students'][$i] = $temp[0]['count(*)'];

            $temp = eF_getTableData("users_to_lessons ul, users u", "count(*)", "u.user_type = 'professor' and u.login = ul.users_LOGIN and ul.lessons_ID=".$directions[$j]['lessons']['id'][$i]);
            $directions[$j]['lessons']['num_professors'][$i] = $temp[0]['count(*)'];

            $temp = eF_getTableData("f_messages, f_topics, f_categories", "count(*)", "f_topics.id = f_messages.f_topics_ID and f_categories.id = f_topics.f_categories_ID and f_categories.lessons_ID = ".$directions[$j]['lessons']['id'][$i]);
            $directions[$j]['lessons']['num_posts'][$i] = $temp[0]['count(*)'];
        }
    }
    //print_r($directions);
    return $directions;
}

/**
*
*/
function eF_getLessonsVolumeInfo($users_LOGIN = false) 
{
    if ($users_LOGIN && eF_checkParameter($users_LOGIN, 'login')) {
        $directions = eF_getTableData("directions d, lessons l, users_to_lessons ul", "distinct d.*", "l.directions_ID=d.id and ul.lessons_ID=l.id and ul.users_LOGIN='$users_LOGIN'");
    } else {
        $directions = eF_getTableData("directions", "*");
    }

    for ($j = 0; $j < sizeof($directions); $j++) {
        if ($users_LOGIN && eF_checkParameter($users_LOGIN, 'login')) {
            $directions[$j]['lessons'] = eF_getTableDataFlat("lessons l, users_to_lessons ul", "l.id, l.name", "ul.users_LOGIN = '$users_LOGIN' and ul.lessons_ID = l.id and l.directions_ID=".$directions[$j]['id']);        
        } else {
            $directions[$j]['lessons'] = eF_getTableDataFlat("lessons", "id, name", "directions_ID=".$directions[$j]['id']);        
        }

        for ($i = 0; $i < sizeof($directions[$j]['lessons']['id']); $i++) {
            $lesson_files = EfrontLessonFiles($directions[$j]['lessons']['id'][$i]);
            $directions[$j]['lessons']['files'][$i]       = $lesson_files[0];
            $directions[$j]['lessons']['directories'][$i] = $lesson_files[1];
            $directions[$j]['lessons']['filesize'][$i]    = ceil($lesson_files[2] / 1000);
            
            $temp = eF_getTableData("content", "count(*)", "lessons_ID=".$directions[$j]['lessons']['id'][$i]." and ctg_type='theory'");
            $directions[$j]['lessons']['theory'][$i] = $temp[0]['count(*)'];

            $temp = eF_getTableData("content", "count(*)", "lessons_ID=".$directions[$j]['lessons']['id'][$i]." and ctg_type='exercises'");
            $directions[$j]['lessons']['exercises'][$i] = $temp[0]['count(*)'];

            $temp = eF_getTableData("content", "count(*)", "lessons_ID=".$directions[$j]['lessons']['id'][$i]." and ctg_type='examples'");
            $directions[$j]['lessons']['examples'][$i] = $temp[0]['count(*)'];

            $temp = eF_getTableData("content", "count(*)", "lessons_ID=".$directions[$j]['lessons']['id'][$i]." and ctg_type='tests'");
            $directions[$j]['lessons']['tests'][$i] = $temp[0]['count(*)'];

            $directions[$j]['lessons']['total'][$i] = $directions[$j]['lessons']['theory'][$i] + $directions[$j]['lessons']['exercises'][$i] + $directions[$j]['lessons']['examples'][$i] + $directions[$j]['lessons']['tests'][$i];
        }
    }
//    print_r($directions);
//    print_r($lesson_files);
//    exit;
    return $directions;
}

/**
*
*/
function eF_getSystemInfo()
{
    
    $num_students                       = eF_getTableData("users", "count(*)", "user_type = 'student'");
    $general_info['num_students']       = $num_students[0]['count(*)'];
    
    $num_professors     = eF_getTableData("users", "count(*)", "user_type = 'professor'");
    $general_info['num_professors']     = $num_professors[0]['count(*)'];
    
    $num_administrators = eF_getTableData("users", "count(*)", "user_type = 'administrator'");
    $general_info['num_administrators'] = $num_administrators[0]['count(*)'];
    
    $general_info['total_users']        = $general_info['num_students'] + $general_info['num_professors'] + $general_info['num_administrators'];
    
    $num_directions     = eF_getTableData("directions", "count(*)");
    $general_info['num_directions']     = $num_directions[0]['count(*)'];
    
    $num_lessons                        = eF_getTableData("lessons", "count(*)");
    $general_info['num_lessons']        = $num_lessons[0]['count(*)'];
    
    $num_content                        = eF_getTableData("content", "count(*)");
    $general_info['num_content']        = $num_content[0]['count(*)'];

    $num_f_messages                     = eF_getTableData("f_messages", "count(*)");
    $general_info['num_f_messages']     = $num_f_messages[0]['count(*)'];

    $num_f_topics                       = eF_getTableData("f_topics", "count(*)");
    $general_info['num_f_topics']       = $num_f_topics[0]['count(*)'];

    $num_f_categories                   = eF_getTableData("f_categories", "count(*)");
    $general_info['num_f_categories']   = $num_f_categories[0]['count(*)'];

    $num_chatmessages                   = eF_getTableData("chatmessages", "count(*)");
    $general_info['num_chatmessages']   = $num_chatmessages[0]['count(*)'];

    $num_chatrooms                      = eF_getTableData("chatrooms", "count(*)");
    $general_info['num_chatrooms']      = $num_chatrooms[0]['count(*)'];
    
    $content_volume = eF_diveIntoDir('content');
    $general_info['content_files'] = $content_volume[0];
    $general_info['content_dirs']  = $content_volume[1];
    $general_info['content_size']  = ceil($content_volume[2]/1000);
    
    $messages_volume = eF_diveIntoDir('../upload');
    $general_info['messages_files'] = $messages_volume[0];
    $general_info['messages_dirs']  = $messages_volume[1];
    $general_info['messages_size']  = ceil($messages_volume[2]/1000);
    
    //print_r($general_info);
    return $general_info;
}
/**
*
*/
function eF_getUserInfo($login, $lessons_ID = false) 
{
    $general_info = eF_getTableData("users", "*", "login='".$login."'");
    $lesson_num = ef_getTableData("users_to_lessons","count(*)", "users_LOGIN='$login'");
    $interval = array(  'from'  => "000000",
                        'to'    => time());
    $user_info['user_type'] = $general_info[0]['user_type'];    
    $user_info['general_info'] = array(_LANGUAGE   => $general_info[0]['languages_NAME'],                                       
                                       _ACTIVE     => $general_info[0]['active'],                                       
                                       _JOINED     => '#filter:timestamp-'.$general_info[0]['timestamp'].'#');
    $user_info['common_info'] = array (_USERNAME   => $login." (".$general_info[0]['surname']." ".$general_info[0]['name'].")",                                                                                                                                                           
                                       _NAME       => $general_info[0]['name'],
                                       _SURNAME    => $general_info[0]['surname'],
                                       _USERTYPE   => $general_info[0]['user_type'],
                                       _LESSONS    => $lesson_num[0]['count(*)'],
                                       _TOTALLOGINTIME => ef_getUserLoginTime($login,$interval));        
    if (strlen($general_info[0]['avatar']) > 0)
        $user_info['avatar'] = $general_info[0]['avatar'];
    
    $res_profile = eF_getTableData("user_profile", "*", "active=1 AND languages_NAME='".$_SESSION['s_language']."'");
    
    for($l=0; $l<sizeof($res_profile); $l++) {      
        $user_info['general_info'][$res_profile[$l]['description']] = $general_info[0][$res_profile[$l]['name']];
      
    }   // afto einai gia na pairnei aytomata ta pedia meta tin prosthiki toy user profile


    $forum_info['messages_num'] = eF_numberOfPosts($login, $lessons_ID);
    if ($lessons_ID && eF_checkParameter($lessons_ID, 'id')) {
        $temp = eF_getTableData("f_messages fm, f_topics ft, f_categories fc", "fm.id, fm.f_topics_ID, fm.timestamp", "fm.users_LOGIN='$login' and fm.f_topics_ID = ft.id and ft.f_categories_ID = fc.id and fc.lessons_ID = $lessons_ID", "fm.timestamp desc limit 1");
    } else {
        $temp = eF_getTableData("f_messages", "id, f_topics_ID, timestamp", "users_LOGIN='$login'", "timestamp desc limit 1");
    }
    sizeof($temp) > 0 ? $forum_info['last_message'] = '#filter:timestamp_time-'.$temp[0]['timestamp'].'# <a href = "forum/forum_index.php?topic='.$temp[0]['f_topics_ID'].'&view_message='.$temp[0]['id'].'">&raquo</a>' : $forum_info['last_message'] = '';

    $p_messages_info['messages_num'] = eF_getTableData("f_personal_messages", "count(*)", "users_LOGIN='$login'");
    $p_messages_info['folders_num']  = eF_getTableData("f_folders", "count(*)", "users_LOGIN='$login'");
    $p_messages_info['volume']       = eF_diveIntoDir(G_UPLOADPATH.$login.'/message_attachments/');
    
    $temp = eF_getTableData("chatmessages", "count(*)", "users_LOGIN='$login'");
    sizeof($temp) > 0 ? $chat_info['messages_num'] = $temp[0]['count(*)'] : $chat_info['messages_num'] = '';
    $temp = eF_getTableData("chatmessages", "timestamp", "users_LOGIN='$login'", "timestamp desc limit 1");
    sizeof($temp) > 0 ? $chat_info['last_message'] = '#filter:timestamp_time-'.$temp[0]['timestamp'].'#' : $chat_info['last_message'] = '';

    $num_comments = eF_numberOfComments($login, $lessons_ID);

    $user_info['communication'] = array(_FORUMPOSTS       => $forum_info['messages_num'],
                                        _FORUMLASTMESSAGE => $forum_info['last_message'],
                                        _PERSONALMESSAGES => $p_messages_info['messages_num'][0]['count(*)'],
                                        _MESSAGESFOLDERS  => $p_messages_info['folders_num'][0]['count(*)'],
                                        _FILES            => $p_messages_info['volume'][0],
                                        _FOLDERS          => $p_messages_info['volume'][1],
                                        _TOTALSIZE        => ceil($p_messages_info['volume'][2] / 1000).' KB',
                                        _CHATMESSAGES     => $chat_info['messages_num'],
                                        _CHATLASTMESSAGE  => $chat_info['last_message'],
                                        _COMMENTS         => $num_comments);

    $last_login = eF_getTableData("logs", "timestamp", "action='login' and users_LOGIN='".$login."'", "timestamp desc");
    sizeof($last_login) > 0 ? $last_login = '#filter:timestamp_time-'.$last_login[0]['timestamp'].'#' : $last_login = '';

    $total_logins = eF_getTableData("logs", "count(*)", "action='login' and users_LOGIN='".$login."'");
    $total_logins = $total_logins[0]['count(*)'];

    $month_logins = eF_getTableData("logs", "count(*)", "timestamp > ".(time() - 2592000)." and action='login' and users_LOGIN='".$login."'");
    $month_logins = $month_logins[0]['count(*)'];

    $week_logins = eF_getTableData("logs", "count(*)", "timestamp > ".(time() - 604800)." and action='login' and users_LOGIN='".$login."'");
    $week_logins = $week_logins[0]['count(*)'];

    $temp = eF_getUserTimes($login);
    sizeof($temp['duration']) > 0 ? $mean_duration = ceil((array_sum($temp['duration']) / sizeof($temp['duration'])) / 60) : $mean_duration = 0;

    $temp = eF_getUserTimes($login, array('from' => time() - 2592000, 'to' => time()));
    sizeof($temp['duration']) > 0 ? $month_mean_duration = ceil((array_sum($temp['duration']) / sizeof($temp['duration']) / 60)) : $month_mean_duration = 0;

    $temp = eF_getUserTimes($login, array('from' => time() - 604800, 'to' => time()));
    sizeof($temp['duration']) > 0 ? $week_mean_duration = ceil((array_sum($temp['duration']) / sizeof($temp['duration']) / 60)) : $week_mean_duration = 0;

    
    $user_info['usage'] = array(_LASTLOGIN         => $last_login,
                                _TOTALLOGINS       => $total_logins,
                                _MONTHLOGINS       => $month_logins,
                                _WEEKLOGINS        => $week_logins,
                                _MEANDURATION      => $mean_duration."'",
                                _MONTHMEANDURATION => $month_mean_duration."'",
                                _WEEKMEANDURATION  => $week_mean_duration."'");
                                
    $user_info['tmp'] = array('last_login' => $last_login);
    
    
    switch($general_info[0]['user_type']) {
        case 'administrator':
            break;
        case 'professor':
            $lessons_info = array();
            
            if ($_SESSION['s_type'] == 'professor') {
                if ($lessons_ID)
                    $lessons = eF_getTableData("users_to_lessons ul, lessons l", "l.id, l.name", "ul.users_LOGIN = '".$login."' and ul.lessons_ID = l.id and ul.users_LOGIN='".$_SESSION['s_login']."' and l.id=$lessons_ID");
                else
                    $lessons = eF_getTableData("users_to_lessons ul, lessons l", "l.id, l.name", "ul.users_LOGIN = '".$login."' and ul.lessons_ID = l.id and ul.users_LOGIN='".$_SESSION['s_login']."'");
            } else {
                if ($lessons_ID)
                    $lessons = eF_getTableData("users_to_lessons ul, lessons l", "l.id, l.name", "ul.users_LOGIN = '".$login."' and ul.lessons_ID = l.id and l.id=$lessons_ID");
                else
                    $lessons = eF_getTableData("users_to_lessons ul, lessons l", "l.id, l.name", "ul.users_LOGIN = '".$login."' and ul.lessons_ID = l.id");
            }
            
            foreach ($lessons as $lesson) {
                $lessons_info['name'][] = $lesson['name'];
                $lessons_info['id'][]   = $lesson['id'];
                $lessons_info['time'][] = ef_getUserLessonTime($lesson['id'], $login,0,time());
            }
            $user_info['lessons_info'] = $lessons_info;
            break;
        case 'student':
            if ($_SESSION['s_type'] == 'professor') {
                if ($lessons_ID)
                    $lessons = eF_getTableData("users_to_lessons ul1, users_to_lessons ul2, lessons l", "l.id, l.name", "ul1.users_LOGIN = '".$login."' and ul2.users_LOGIN='".$_SESSION['s_login']."' and ul1.lessons_ID = l.id and ul1.lessons_ID = ul2.lessons_ID and l.id=$lessons_ID");
                else
                    $lessons = eF_getTableData("users_to_lessons ul1, users_to_lessons ul2, lessons l", "l.id, l.name", "ul1.users_LOGIN = '".$login."' and ul2.users_LOGIN='".$_SESSION['s_login']."' and ul1.lessons_ID = l.id and ul1.lessons_ID = ul2.lessons_ID");
                    
            } else {
                if ($lessons_ID)
                    $lessons = eF_getTableData("users_to_lessons ul, lessons l", "l.id, l.name", "ul.users_LOGIN = '".$login."' and ul.lessons_ID = l.id AND l.active=1 and l.id=$lessons_ID"); //makriria. ebala to active=1 oste na emfanizei mono ta energopoimena mathimata. etsi ki allios sta deactivated ta ebgaze ola 0.
                else
                    $lessons = eF_getTableData("users_to_lessons ul, lessons l", "l.id, l.name", "ul.users_LOGIN = '".$login."' and ul.lessons_ID = l.id AND l.active=1"); //makriria. ebala to active=1 oste na emfanizei mono ta energopoimena mathimata. etsi ki allios sta deactivated ta ebgaze ola 0.
            }
            foreach ($lessons as $lesson) {
                $lessons_info['id'][]           = $lesson['id'];
                $lessons_info['name'][]         = $lesson['name'];
                $lessons_info['content_done'][] = eF_usersContentDone($login, $lesson['id']);
                $lessons_info['done_tests'][]   = eF_getTableDataFlat("done_tests, tests, content", "content.name, done_tests.score, done_tests.timestamp, content.id as test_content_ID", "done_tests.tests_ID=tests.id and tests.content_ID=content.id and content.lessons_ID=".$lesson['id']." and done_tests.users_login='$login'");
                $lessons_info['done_projects'][] = eF_getTableDataFlat("users_to_projects, projects", "projects.title, users_to_projects.grade, users_to_projects.upload_timestamp", "users_to_projects.projects_ID=projects.id and users_to_projects.users_LOGIN='$login' and projects.lessons_ID=".$lesson['id']);
                $lessons_info['time'][] = ef_getUserLessonTime($lesson['id'], $login,0,time());
            }
            $user_info['lessons_info'] = $lessons_info;
            break;
        default: break;
    }       
    return $user_info;    
}

function eF_getUserLessonInfo($login, $lessons_ID) 
{
  
    $general_info = eF_getTableData("users", "*", "login='".$login."'");
    $user_info['general_info'] = array(_USERNAME   => $login,
                                       _LANGUAGE   => $general_info[0]['languages_NAME'],
                                       _ACTUALNAME => $general_info[0]['surname']." ".$general_info[0]['name'],                                       
                                       _ACTIVE     => $general_info[0]['active'],
                                       _USERTYPE   => $general_info[0]['user_type'],                                       
                                       _JOINED     => '#filter:timestamp-'.$general_info[0]['timestamp'].'#');
    if (strlen($general_info[0]['avatar']) > 0)
        $user_info['avatar'] = $general_info[0]['avatar'];
    
    $res_profile = eF_getTableData("user_profile", "*", "active=1 AND languages_NAME='".$_SESSION['s_language']."'");
    
    for ($l=0; $l<sizeof($res_profile); $l++) {      
        $user_info['general_info'][$res_profile[$l]['description']] = $general_info[0][$res_profile[$l]['name']];
      
    }   // afto einai gia na pairnei aytomata ta pedia meta tin prosthiki toy user profile


    $forum_info['messages_num'] = eF_numberOfPosts($login, $lessons_ID);
    if ($lessons_ID && eF_checkParameter($lessons_ID, 'id')) {
        $temp = eF_getTableData("f_messages fm, f_topics ft, f_categories fc", "fm.id, fm.f_topics_ID, fm.timestamp", "fm.users_LOGIN='$login' and fm.f_topics_ID = ft.id and ft.f_categories_ID = fc.id and fc.lessons_ID = $lessons_ID", "fm.timestamp desc limit 1");
    } else {
        $temp = eF_getTableData("f_messages", "id, f_topics_ID, timestamp", "users_LOGIN='$login'", "timestamp desc limit 1");
    }
    sizeof($temp) > 0 ? $forum_info['last_message'] = '#filter:timestamp_time-'.$temp[0]['timestamp'].'# <a href = "forum/forum_index.php?topic='.$temp[0]['f_topics_ID'].'&view_message='.$temp[0]['id'].'">&raquo</a>' : $forum_info['last_message'] = '';

    $p_messages_info['messages_num'] = eF_getTableData("f_personal_messages", "count(*)", "users_LOGIN='$login'");
    $p_messages_info['folders_num']  = eF_getTableData("f_folders", "count(*)", "users_LOGIN='$login'");
    $p_messages_info['volume']       = eF_diveIntoDir(G_UPLOADPATH.$login.'/message_attachments/');
    
    $temp = eF_getTableData("chatmessages", "count(*)", "users_LOGIN='$login'");
    sizeof($temp) > 0 ? $chat_info['messages_num'] = $temp[0]['count(*)'] : $chat_info['messages_num'] = '';
    $temp = eF_getTableData("chatmessages", "timestamp", "users_LOGIN='$login'", "timestamp desc limit 1");
    sizeof($temp) > 0 ? $chat_info['last_message'] = '#filter:timestamp_time-'.$temp[0]['timestamp'].'#' : $chat_info['last_message'] = '';

    $num_comments = eF_numberOfComments($login, $lessons_ID);

    $user_info['communication'] = array(_FORUMPOSTS       => $forum_info['messages_num'],
                                        _FORUMLASTMESSAGE => $forum_info['last_message'],
                                        _PERSONALMESSAGES => $p_messages_info['messages_num'][0]['count(*)'],
                                        _MESSAGESFOLDERS  => $p_messages_info['folders_num'][0]['count(*)'],
                                        _FILES            => $p_messages_info['volume'][0],
                                        _FOLDERS          => $p_messages_info['volume'][1],
                                        _TOTALSIZE        => ceil($p_messages_info['volume'][2] / 1000).' KB',
                                        _CHATMESSAGES     => $chat_info['messages_num'],
                                        _CHATLASTMESSAGE  => $chat_info['last_message'],
                                        _COMMENTS         => $num_comments);

    $last_login = eF_getTableData("logs", "timestamp", "action='login' and users_LOGIN='".$login."'", "timestamp desc");
    sizeof($last_login) > 0 ? $last_login = '#filter:timestamp_time-'.$last_login[0]['timestamp'].'#' : $last_login = '';

    $total_logins = eF_getTableData("logs", "count(*)", "action='login' and users_LOGIN='".$login."'");
    $total_logins = $total_logins[0]['count(*)'];

    $month_logins = eF_getTableData("logs", "count(*)", "timestamp > ".(time() - 2592000)." and action='login' and users_LOGIN='".$login."'");
    $month_logins = $month_logins[0]['count(*)'];

    $week_logins = eF_getTableData("logs", "count(*)", "timestamp > ".(time() - 604800)." and action='login' and users_LOGIN='".$login."'");
    $week_logins = $week_logins[0]['count(*)'];

    $temp = eF_getUserTimes($login);
    sizeof($temp['duration']) > 0 ? $mean_duration = ceil((array_sum($temp['duration']) / sizeof($temp['duration'])) / 60) : $mean_duration = 0;

    $temp = eF_getUserTimes($login, array('from' => time() - 2592000, 'to' => time()));
    sizeof($temp['duration']) > 0 ? $month_mean_duration = ceil((array_sum($temp['duration']) / sizeof($temp['duration']) / 60)) : $month_mean_duration = 0;

    $temp = eF_getUserTimes($login, array('from' => time() - 604800, 'to' => time()));
    sizeof($temp['duration']) > 0 ? $week_mean_duration = ceil((array_sum($temp['duration']) / sizeof($temp['duration']) / 60)) : $week_mean_duration = 0;

    
    $user_info['usage'] = array(_LASTLOGIN         => $last_login,
                                _TOTALLOGINS       => $total_logins,
                                _MONTHLOGINS       => $month_logins,
                                _WEEKLOGINS        => $week_logins,
                                _MEANDURATION      => $mean_duration."'",
                                _MONTHMEANDURATION => $month_mean_duration."'",
                                _WEEKMEANDURATION  => $week_mean_duration."'");
    
    switch($general_info[0]['user_type']) {
        case 'administrator':
            break;
        case 'professor':
            $lessons_info = array();
            if ($_SESSION['s_type'] == 'professor') {
                $lessons = eF_getTableData("users_to_lessons ul, lessons l", "l.id, l.name", "ul.users_LOGIN = '".$login."' and ul.lessons_ID = l.id and ul.users_LOGIN='".$_SESSION['s_login']."' and ul.lessons_ID=$lessons_ID");
            } else {
                $lessons = eF_getTableData("users_to_lessons ul, lessons l", "l.id, l.name", "ul.users_LOGIN = '".$login."' and ul.lessons_ID = l.id and ul.lessons_ID=$lessons_ID");
            }
            
            foreach ($lessons as $lesson) {
                $lessons_info['name'][] = $lesson['name'];
                $lessons_info['id'][]   = $lesson['id'];
            }
            $user_info['lessons_info'] = $lessons_info;
            break;
        case 'student':
            
            if ($_SESSION['s_type'] == 'professor') {
                $lessons = eF_getTableData("users_to_lessons ul1, users_to_lessons ul2, lessons l", "l.id, l.name", "ul1.users_LOGIN = '".$login."' and ul2.users_LOGIN='".$_SESSION['s_login']."' and ul1.lessons_ID = l.id and ul1.lessons_ID = ul2.lessons_ID and ul1.lessons_ID=$lessons_ID");
        
        } else {
                $lessons = eF_getTableData("users_to_lessons ul, lessons l", "l.id, l.name", "ul.users_LOGIN = '".$login."' and ul.lessons_ID = l.id AND l.active=1 and ul.lessons_ID=$lessons_ID"); //makriria. ebala to active=1 oste na emfanizei mono ta energopoimena mathimata. etsi ki allios sta deactivated ta ebgaze ola 0.
        
        }
        foreach ($lessons as $lesson) {
            $lessons_info['id'][]           = $lesson['id'];
            $lessons_info['name'][]         = $lesson['name'];
            $lessons_info['content_done'][] = eF_usersContentDone($login, $lesson['id']);
            $lessons_info['done_tests'][]   = eF_getTableDataFlat("done_tests, tests, content", "content.name, done_tests.score, done_tests.timestamp, content.id as test_content_ID", "done_tests.tests_ID=tests.id and tests.content_ID=content.id and content.lessons_ID=".$lesson['id']." and done_tests.users_login='$login'");
            $lessons_info['done_projects'][] = eF_getTableDataFlat("users_to_projects, projects", "projects.title, users_to_projects.grade, users_to_projects.upload_timestamp", "users_to_projects.projects_ID=projects.id and users_to_projects.users_LOGIN='$login' and projects.lessons_ID=".$lesson['id']);
        }
        $user_info['lessons_info'] = $lessons_info;
        break;
        default: break;
    }    
    
    return $user_info;    
}


/**
*
*/
function eF_getUserMoves($login, $interval, $lessons_ID = false) {
    
    if ($lessons_ID && eF_checkParameter($lessons_ID, 'id')) {
        $result = eF_getTableDataFlat("logs l, content c, lessons ls", "distinct l.timestamp, l.comments, l.action, l.session_ip, c.name as content_name, ls.name as lesson_name", "((c.id=l.comments and c.lessons_ID=$lessons_ID and ls.id = c.lessons_ID) or l.comments=0) and l.users_LOGIN='$login' and l.timestamp > ".$interval['from']." and l.timestamp < ".$interval['to']." group by timestamp");
    } else {
        $result = eF_getTableDataFlat("logs l, content c, lessons ls", "distinct l.timestamp, l.comments, l.action, l.session_ip, c.name as content_name, ls.name as lesson_name", "((c.id=l.comments and ls.id = c.lessons_ID) or l.comments=0) and l.users_LOGIN='$login' and l.timestamp > ".$interval['from']." and l.timestamp < ".$interval['to']." group by timestamp");
    }
    
    for ($i = 0; $i < sizeof($result['action']); $i++) {
        if ($result['action'][$i] == 'login') {
            $result['comments'][$i] = _FROM.' '.eF_decodeIP($result['session_ip'][$i]);
        } elseif ($result['action'][$i] == 'logout' || $result['action'][$i] == 'personal') {
            $result['comments'][$i] = '';
        } elseif ($result['action'][$i] == 'tests' || $result['action'][$i] == 'theory' || $result['action'][$i] == 'examples' || $result['action'][$i] == 'exercises' || $result['action'][$i] == 'lessons' || $result['action'][$i] == 'comment') {
            $result['comments'][$i] = $result['content_name'][$i]. " (".$result['lesson_name'][$i].")";
        }
    }

    return $result;
}


/**
*
*/
function eF_getLessonInfo($lessons_ID, $group_id = false) {

    $general_info   = eF_getTableData("lessons", "*", "id=$lessons_ID");
    $direction_name = eF_getTableData("directions", "name", "id=".$general_info[0]['directions_ID']);
    $lesson_info['general_info'] = array(_PRICE     => $general_info[0]['price'].' '.$GLOBALS['configuration']['currency'],
                                         _ACTIVE    => $general_info[0]['active'] ? _YES : _NO,
                                         _LANGUAGE  => $general_info[0]['languages_NAME']);
    
    if ($group_id){
        $num_students   = eF_getTableData("users_to_lessons ul, users u, users_to_groups g", "count(*)", "g.users_LOGIN = u.login and g.groups_ID = $group_id and u.user_type = 'student' and u.login = ul.users_LOGIN and ul.lessons_ID=".$lessons_ID);
        $num_professors = eF_getTableData("users_to_lessons ul, users u, users_to_groups g", "count(*)", "g.users_LOGIN = u.login and g.groups_ID = $group_id u.user_type = 'professor' and u.login = ul.users_LOGIN and ul.lessons_ID=".$lessons_ID);
    }
    else{
        $num_students   = eF_getTableData("users_to_lessons ul, users u", "count(*)", "u.user_type = 'student' and u.login = ul.users_LOGIN and ul.lessons_ID=".$lessons_ID);
        $num_professors = eF_getTableData("users_to_lessons ul, users u", "count(*)", "u.user_type = 'professor' and u.login = ul.users_LOGIN and ul.lessons_ID=".$lessons_ID);
    }
    
    $num_comments   = eF_getTableData("comments, content", "count(*)", "comments.content_ID = content.id and content.lessons_ID=".$lessons_ID);
    $num_posts      = eF_getTableData("f_messages, f_topics, f_categories", "count(*)", "f_topics.id = f_messages.f_topics_ID and f_categories.id = f_topics.f_categories_ID and f_categories.lessons_ID = ".$lessons_ID);
    $num_chatmsgs   = eF_getTableData("chatrooms, chatmessages", "count(*)", "chatmessages.chatrooms_ID = chatrooms.id and chatrooms.lessons_ID=".$lessons_ID);
    
    if ($num_comments[0]['count(*)'] > 0) {    
        $temp = eF_getTableData("f_messages, f_topics, f_categories", "f_messages.timestamp, f_messages.id, f_messages.f_topics_ID", "f_topics.id = f_messages.f_topics_ID and f_categories.id = f_topics.f_categories_ID and f_categories.lessons_ID = ".$lessons_ID, "timestamp desc limit 1");
        $last_post = $temp[0]['timestamp'].'-<a href = "forum/forum_index.php?topic='.$temp[0]['f_topics_ID'].'&view_message='.$temp[0]['id'].'">&raquo;</a>';
    } else {
        $last_post = '';
    }    
    
    $lesson_info['common_info'] =  array(_NAME      => $general_info[0]['name'],
                                         _DIRECTION => $direction_name[0]['name'],
                                         _STUDENTS     => $num_students[0]['count(*)'],
                                         _PROFESSORS   => $num_professors[0]['count(*)']);
                                         
    $lesson_info['participation'] = array(_COMMENTS     => $num_comments[0]['count(*)'],
                                          _FORUMPOSTS   => $num_posts[0]['count(*)'],
                                          _CHATMESSAGES => $num_chatmsgs[0]['count(*)'],
                                          _LASTPOST     => $last_post);
                                          
    $file_stats  = eF_diveIntoDir(G_LESSONSPATH.$lessons_ID);
    $files       = $file_stats[0];
    $directories = $file_stats[1];
    $file_size   = ceil($file_stats[2] / 1000);

    $lesson_info['volume'] = array(_FILES       => $files,
                                   _DIRECTORIES => $directories,
                                   _FILESIZE    => $file_size.' KB');

    $theory    = eF_getTableData("content", "count(*)", "lessons_ID=".$lessons_ID." and ctg_type='theory'");
    $exercises = eF_getTableData("content", "count(*)", "lessons_ID=".$lessons_ID." and ctg_type='exercises'");
    $examples  = eF_getTableData("content", "count(*)", "lessons_ID=".$lessons_ID." and ctg_type='examples'");
    $tests     = eF_getTableData("content", "count(*)", "lessons_ID=".$lessons_ID." and ctg_type='tests' and active='1'");
    $projects  = ef_getTableData("projects","count(*)", "lessons_ID=".$lessons_ID);
    $total     = $theory[0]['count(*)'] + $exercises[0]['count(*)'] + $examples[0]['count(*)'] + $tests[0]['count(*)'];
    
    $lesson_info['content'] = array(_THEORY   => $theory[0]['count(*)'],
                                    _PROJECTS => $projects[0]['count(*)'],
                                    _EXAMPLES => $examples[0]['count(*)'],
                                    _TESTS    => $tests[0]['count(*)'],
                                    _TOTAL    => $total);

    $tests = eF_getTableDataFlat("tests, content", "tests.*, content.name, content.timestamp", "tests.content_ID = content.id and tests.active = 1 and content.active = 1 and content.lessons_ID=".$lessons_ID);
       
    foreach ($tests['id'] as $key => $test_id) {
        $tests['duration'][$key] = round($tests['duration'][$key] / 60);
        $questions = eF_getTableDataFlat("questions q, tests_to_questions tq", "q.*, tq.weight", "q.id=tq.questions_ID and tq.tests_ID = ".$test_id);
        
        $tests['total_questions'][] = sizeof($questions['id']);
        
        $tests['raw_text'][$key]      = 0;
        $tests['multiple_one'][$key]  = 0;
        $tests['multiple_many'][$key] = 0;
        $tests['match'][$key]         = 0;
        $tests['empty_spaces'][$key]  = 0;
        $tests['true_false'][$key]    = 0;
        $tests['low'][$key]    = 0;
        $tests['medium'][$key] = 0;
        $tests['high'][$key]   = 0;

        for ($i = 0; $i < sizeof($questions['id']); $i++) {
            $tests[$questions['type'][$i]][$key]++; 
            $tests[$questions['difficulty'][$i]][$key]++; 
        }

        if ($group_id){
            $tests['done_test'][] = eF_getTableDataFlat("done_tests, users, users_to_groups g", "done_tests.*,users.name,users.surname", "g.users_LOGIN = users.login and g.groups_ID = $group_id and done_tests.users_LOGIN=users.login AND done_tests.tests_ID = ".$test_id);  //gia na parei kai to onomateponymo toy xristi ektos to login prosthesa kai ton pinaka users....
        }
        else{
            $tests['done_test'][] = eF_getTableDataFlat("done_tests,users", "done_tests.*,users.name,users.surname", "done_tests.users_LOGIN=users.login AND done_tests.tests_ID = ".$test_id);  //gia na parei kai to onomateponymo toy xristi ektos to login prosthesa kai ton pinaka users....
        }
    }
    $lesson_info['tests'] = $tests;
    
    $questions = ef_getTableDataFlat("questions q, content c","q.id, q.text, q.type, q.difficulty","q.content_ID = c.id and c.lessons_ID=$lessons_ID");
    
    
    for ($i=0; $i<sizeof($questions['id']); $i++){
        $question_id = $questions['id'][$i];
        if ($group_id){
            $score = eF_getTableData("done_questions q","sum(q.score), count(q.score)","q.questions_ID =$question_id");
        }
        else{
            $score = eF_getTableData("done_questions q","sum(q.score) as sum, count(q.score) as num","q.questions_ID =$question_id");
        }

        $questions['avg_score'][$i]  = round(100 * $score[0]['sum'] / $score[0]['num'], 2) / 100;
        $questions['times_done'][$i] = $score[0]['num'];
        mb_strlen(strip_tags($questions['text'][$i])) > 50 ? $questions['reduced_text'][$i] = mb_substr(strip_tags($questions['text'][$i]), 0, 50).'...' : $questions['reduced_text'][$i] = strip_tags($questions['text'][$i]);
    }

    
    $lesson_info['questions'] = $questions;
    
    $projects = ef_getTableDataFlat("projects", "*", "lessons_ID =".$lessons_ID);
    foreach ($projects['id'] as $key => $project_id) {       
        if ($group_id){
            $projects['done_project'][] = ef_getTableDataFlat("users_to_projects, users, users_to_groups g", "users_to_projects.*, users.name,users.surname", "g.users_LOGIN = users.login and g.groups_ID = $group_id and users_to_projects.users_LOGIN=users.login AND users_to_projects.projects_ID = ".$project_id);
        } 
        else{
            $projects['done_project'][] = ef_getTableDataFlat("users_to_projects, users", "users_to_projects.*, users.name,users.surname", "users_to_projects.users_LOGIN=users.login AND users_to_projects.projects_ID = ".$project_id);
        }
    }
    
    $lesson_info['projects'] = $projects;
//pr($lesson_info);
    return $lesson_info;

}

/**
 * This function returns an array which contains an array which holds
 * the information for the lessons and the tests a user have done for
 * the lessons his is register to.
 *
 * efront ver 2.2.
*/

function eF_getUserDoneTests($user){
    
    $lessons_id = eF_getTableData("lessons,users_to_lessons","lessons.id",'users_to_lessons.lessons_ID = lessons.id AND users_to_lessons.users_LOGIN="'.$user.'"');
    for($i = 0 ; $i < sizeof($lessons_id) ; $i ++){
        $done_test['done_tests'][] = eF_getTableDataFlat("done_tests, tests, content", "content.name, done_tests.score, done_tests.timestamp, content.id as test_content_ID", "done_tests.tests_ID=tests.id and tests.content_ID=content.id and content.lessons_ID=".$lessons_id[$i]['id']." and done_tests.users_login='$user'");
    }
    return $done_test['done_tests'];
    
}

/**
 * Returns all the done tests for a specified test
 * 
 * This function returns an array which holds the information for the done tests of a specific test
 * @param int $test_id The id of the specified test
 * @return array An array with the information for the done tests
 * @version 1.0 (10/6/2007)  
*/
function ef_getDoneTests($test_id){
    $done_tests = ef_getTableData("done_tests","*",'tests_ID="'.$test_id.'"');
    return $done_tests;
}

function ef_getTestInfo($test_id){
    $info = array();
    
    $common = ef_getTableData("tests t, content c, lessons l", "c.name as test_name, t.duration, l.name as lesson_name, l.id as lesson_id", "l.id = c.lessons_ID and t.content_id = c.id and t.id=$test_id");
    $questions = ef_getTableData("tests_to_questions t, questions q", "q.id, q.text, q.type, q.options", "t.tests_ID=$test_id AND t.questions_ID = q.id");
    $info['common_info'] = array();
    $info['common_info']['name'] = $common[0]['test_name'];
    $info['common_info']['lesson_name'] = $common[0]['lesson_name'];    
    $info['common_info']['duration'] = $common[0]['duration'];
    $info['common_info']['questions_number'] = sizeof($questions);
    
    
    $info['questions'] = array();
    for ($i=0; $i<sizeof($questions); $i++){
        $info['questions'][$i]['id'] = $questions[$i]['id'];
        mb_strlen(strip_tags($questions[$i]['text'])) > 50 ? $info['questions'][$i]['text'] = mb_substr(strip_tags($questions[$i]['text']), 0, 50).'...' : $info['questions'][$i]['text'] = strip_tags($questions[$i]['text']);
        $info['questions'][$i]['full_text'] = $questions[$i]['text'];
       
        switch ($questions[$i]['type']){
            case 'raw_text': $info['questions'][$i]['type'] = _RAWTEXT; break;
            case 'multiple_one': $info['questions'][$i]['type'] = _MULTIPLEONE; break;
            case 'multiple_many': $info['questions'][$i]['type'] = _MULTIPLEMANY; break;
            case 'true_false': $info['questions'][$i]['type'] = _TRUEFALSE; break;
            case 'empty_spaces': $info['questions'][$i]['type'] = _EMPTYSPACES; break;
            case 'match': $info['questions'][$i]['type'] = _MATCH; break;
        }
        $correct = ef_getTableData("done_questions q, done_tests t","*", "q.done_tests_id = t.id and t.tests_ID = $test_id and questions_ID='".$questions[$i]['id']."' and q.score = 1");
        $false = ef_getTableData("done_questions q, done_tests t","*", "q.done_tests_id = t.id and t.tests_ID = $test_id and q.questions_ID='".$questions[$i]['id']."' and q.score = 0");
        $info['questions'][$i]['correct'] = sizeof($correct);
        $info['questions'][$i]['false'] = sizeof($false);
    }
    
    return $info;
}
?>
