/**
 * <copyright>
 *
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   IBM - Initial API and implementation
 *
 * </copyright>
 *
 * $Id: UMLEnvironment.java,v 1.6 2007/04/23 21:13:54 cdamus Exp $
 */

package org.eclipse.ocl.uml;

import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.ocl.AbstractEnvironment;
import org.eclipse.ocl.Environment;
import org.eclipse.ocl.EnvironmentFactory;
import org.eclipse.ocl.TypeResolver;
import org.eclipse.ocl.expressions.Variable;
import org.eclipse.ocl.types.OCLStandardLibrary;
import org.eclipse.ocl.uml.internal.OCLFactoryImpl;
import org.eclipse.ocl.uml.internal.OCLStandardLibraryImpl;
import org.eclipse.ocl.uml.internal.UMLForeignMethods;
import org.eclipse.ocl.uml.util.OCLUMLUtil;
import org.eclipse.ocl.utilities.OCLFactory;
import org.eclipse.ocl.utilities.UMLReflection;
import org.eclipse.uml2.uml.Association;
import org.eclipse.uml2.uml.Behavior;
import org.eclipse.uml2.uml.BehavioredClassifier;
import org.eclipse.uml2.uml.CallOperationAction;
import org.eclipse.uml2.uml.Class;
import org.eclipse.uml2.uml.Classifier;
import org.eclipse.uml2.uml.Constraint;
import org.eclipse.uml2.uml.EnumerationLiteral;
import org.eclipse.uml2.uml.Feature;
import org.eclipse.uml2.uml.Namespace;
import org.eclipse.uml2.uml.Operation;
import org.eclipse.uml2.uml.Package;
import org.eclipse.uml2.uml.Parameter;
import org.eclipse.uml2.uml.ParameterDirectionKind;
import org.eclipse.uml2.uml.ParameterEffectKind;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Region;
import org.eclipse.uml2.uml.SendSignalAction;
import org.eclipse.uml2.uml.State;
import org.eclipse.uml2.uml.StateMachine;
import org.eclipse.uml2.uml.UMLFactory;
import org.eclipse.uml2.uml.UMLPackage;
import org.eclipse.uml2.uml.Vertex;

/**
 * Implementation of the {@link Environment} for parsing OCL expressions on UML
 * models. The <code>UMLEnvironment</code> uses a client-supplied resource set
 * to look up UML {@link Package}s in UML resources. It also uses an
 * {@link EPackage} registry to find the corresponding Ecore definitions of
 * packages when evaluating expressions on instances of the UML model (in the
 * case of evaluation on UML2-generated API objects).
 * 
 * @author Christian W. Damus (cdamus)
 */
public class UMLEnvironment
    extends
    AbstractEnvironment<Package, Classifier, Operation, Property, EnumerationLiteral, Parameter, State, CallOperationAction, SendSignalAction, Constraint, Class, EObject> {

    private static OCLStandardLibraryImpl standardLibrary;

    private UMLReflection<Package, Classifier, Operation, Property, EnumerationLiteral, Parameter, State, CallOperationAction, SendSignalAction, Constraint> reflection;

    /** The resource set for package lookups. */
    private ResourceSet resourceSet;

    /** The package registry for Ecore definition lookups. */
    private EPackage.Registry registry;

    private EnvironmentFactory<Package, Classifier, Operation, Property, EnumerationLiteral, Parameter, State, CallOperationAction, SendSignalAction, Constraint, Class, EObject> factory;

    private TypeResolver<Classifier, Operation, Property> typeResolver;

    private Package umlMetamodel;

    private Map<List<String>, Classifier> classifierCache = new java.util.HashMap<List<String>, Classifier>();

    private Map<List<String>, Package> packageCache = new java.util.HashMap<List<String>, Package>();

    /**
     * Initializes me with a package registry for looking up the Ecore
     * representations of UML packages and a resource set for looking up UML
     * packages in UML resources.
     * 
     * @param registry
     *            the Ecore package registry to use
     * @param rset
     *            the resource set to use
     */
    protected UMLEnvironment(EPackage.Registry registry, ResourceSet rset) {
        this.registry = registry;
        resourceSet = rset;
    }

    /**
     * Initializes me with a package registry for looking up the Ecore
     * representations of UML packages, a resource set for looking up UML
     * packages in UML resources, and a resource from which to load myself.
     * 
     * @param registry
     *            the Ecore package registry to use
     * @param rset
     *            the resource set to use
     * @param resource
     *            my resource for persistence
     */
    protected UMLEnvironment(EPackage.Registry registry, ResourceSet rset,
            Resource resource) {
        this(registry, rset);

        typeResolver = new TypeResolverImpl(this, resource);
    }

    /**
     * Initializes me with a parent environment. I inherit my package registry,
     * resource set, and resource from it.
     * 
     * @param parent
     *            my parent environment
     */
    protected UMLEnvironment(
            Environment<Package, Classifier, Operation, Property, EnumerationLiteral, Parameter, State, CallOperationAction, SendSignalAction, Constraint, Class, EObject> parent) {

        super((UMLEnvironment) parent);

        UMLEnvironment uparent = (UMLEnvironment) parent;

        if (uparent != null) {
            this.registry = uparent.registry;
            resourceSet = uparent.resourceSet;
            typeResolver = uparent.getTypeResolver();
        } else {
            this.registry = (EPackage.Registry.INSTANCE);
            resourceSet = new ResourceSetImpl();
        }
    }

    // implements the inherited specification
    public EnvironmentFactory<org.eclipse.uml2.uml.Package, Classifier, Operation, Property, EnumerationLiteral, Parameter, State, CallOperationAction, SendSignalAction, Constraint, org.eclipse.uml2.uml.Class, EObject> getFactory() {
        if (factory != null) {
            return factory;
        }

        if (getParent() != null) {
            factory = getParent().getFactory();
            if (factory != null) {
                return factory;
            }
        }

        // obtain a reasonable default factory
        factory = new UMLEnvironmentFactory(getResourceSet());

        return factory;
    }

    /**
     * Sets the factory that created me. This method should only be invoked by
     * that factory.
     * 
     * @param factory
     *            my originating factory
     */
    protected void setFactory(
            EnvironmentFactory<Package, Classifier, Operation, Property, EnumerationLiteral, Parameter, State, CallOperationAction, SendSignalAction, Constraint, Class, EObject> factory) {
        this.factory = factory;
    }

    // implements the inherited specification
    public void setParent(
            Environment<Package, Classifier, Operation, Property, EnumerationLiteral, Parameter, State, CallOperationAction, SendSignalAction, Constraint, Class, EObject> env) {
        super.setParent((UMLEnvironment) env);
    }

    // implements the inherited specification
    public OCLStandardLibrary<Classifier> getOCLStandardLibrary() {
        if (standardLibrary == null) {
            standardLibrary = OCLStandardLibraryImpl.INSTANCE;
        }

        return standardLibrary;
    }

    /**
     * Obtains the resource set in which I look for UML packages when resolving
     * package names.
     * 
     * @return my resource set
     */
    protected final ResourceSet getResourceSet() {
        return resourceSet;
    }

    /**
     * Obtains the UML metamodel library, loaded in my resource set.
     * 
     * @return the UML metamodel
     */
    protected Package getUMLMetamodel() {
        if (umlMetamodel == null) {
            if (getFactory() instanceof UMLEnvironmentFactory) {
                umlMetamodel = ((UMLEnvironmentFactory) getFactory())
                    .getUMLMetamodel();
            } else {
                umlMetamodel = OCLUMLUtil.getUMLMetamodel(getResourceSet());
            }
        }

        return umlMetamodel;
    }

    // implements the inherited specification
    public TypeResolver<Classifier, Operation, Property> getTypeResolver() {
        if (typeResolver == null) {
            typeResolver = createTypeResolver();
        }

        return typeResolver;
    }

    // implements the inherited specification
    public OCLFactory getOCLFactory() {
        return OCLFactoryImpl.INSTANCE;
    }

    // implements the inherited specification
    public UMLReflection<Package, Classifier, Operation, Property, EnumerationLiteral, Parameter, State, CallOperationAction, SendSignalAction, Constraint> getUMLReflection() {
        if (reflection == null) {
            if (getParent() != null) {
                reflection = getParent().getUMLReflection();
            } else {
                reflection = new UMLReflectionImpl(registry);
            }
        }

        return reflection;
    }

    /**
     * Creates a new type resolver for use with this environment.
     * 
     * @return a new type resolver
     */
    protected TypeResolver<Classifier, Operation, Property> createTypeResolver() {
        return new TypeResolverImpl(this);
    }

    /**
     * {@inheritDoc}
     * <p>
     * Implements the inherited specification by looking in my resource set for
     * a resource containing the specified package.
     * </p>
     */
    public org.eclipse.uml2.uml.Package lookupPackage(List<String> path) {
        Package tryCache = packageCache.get(path);
        if (tryCache != null) {
            return tryCache;
        }

        Package pkg = null;
        Package currPkg = getContextPackage();

        // Check whether this package is in the default package
        if (currPkg != null) {
            List<String> lookup = path;

            while (currPkg != null) {
                pkg = currPkg;

                for (int i = 0; i < lookup.size(); i++) {
                    String name = lookup.get(i);
                    pkg = UMLForeignMethods.getNestedPackage(pkg, name);

                    if (pkg == null) {
                        break;
                    }
                }

                if (pkg != null) {
                    packageCache.put(path, pkg);
                    return pkg;
                }

                if ((currPkg == getContextPackage()) && (lookup.size() > 0)
                    && UMLForeignMethods.isNamed(lookup.get(0), currPkg)) {
                    // handle the case where the first part of the qualified
                    // name matches the context package name
                    lookup = lookup.subList(1, lookup.size());
                } else {
                    lookup = path;
                    currPkg = currPkg.getNestingPackage();
                }
            }
        }

        // Check whether this package exists in the resource set
        Package result = OCLUMLUtil.findPackage(path, getResourceSet());
        packageCache.put(path, result);
        return result;
    }

    // implements the inherited specification
    public Classifier lookupClassifier(List<String> names) {
        Classifier tryCache = classifierCache.get(names);
        if (tryCache != null) {
            return tryCache;
        }

        Namespace ns = null;
        Namespace currNs = getContextPackage();

        if (names.size() > 1) {
            List<String> lookup = names;

            // Check whether this package is in the default package
            if (currNs != null) {
                while (currNs != null) {

                    ns = currNs;
                    int last = lookup.size() - 1;

                    for (int i = 0; i < last; i++) {
                        String name = lookup.get(i);
                        ns = (Namespace) UMLForeignMethods.getMember(ns, name,
                            UMLPackage.Literals.NAMESPACE);

                        if (ns == null) {
                            break;
                        }
                    }

                    if (ns != null) {
                        String name = lookup.get(last);

                        Classifier member = (Classifier) UMLForeignMethods
                            .getMember(ns, name, UMLPackage.Literals.CLASSIFIER);

                        if (member != null) {
                            classifierCache.put(names, member);
                            return member;
                        }
                    }

                    if ((currNs == getContextPackage()) && (lookup.size() > 1)
                        && UMLForeignMethods.isNamed(lookup.get(0), currNs)) {
                        // handle the case where the first part of the qualified
                        // name matches the context package name
                        lookup = lookup.subList(1, lookup.size());
                    } else {
                        lookup = names;
                        currNs = currNs.getNamespace();
                    }
                }
            }

            // Check whether this package exists
            List<String> newNames = names.subList(0, names.size() - 1);
            ns = OCLUMLUtil.findNamespace(newNames, getResourceSet());
            if (ns == null) {
                return null;
            }

            String name = names.get(names.size() - 1);
            Classifier member = (Classifier) UMLForeignMethods.getMember(ns,
                name, UMLPackage.Literals.CLASSIFIER);

            if (member != null) {
                classifierCache.put(names, member);
                return member;
            }

            return member;
        } else if (getContextPackage() != null) {
            String name = names.get(0);
            Classifier result = null;
            while (currNs != null) {
                result = (Classifier) UMLForeignMethods.getMember(currNs, name,
                    UMLPackage.Literals.CLASSIFIER);

                if (result != null) {
                    classifierCache.put(names, result);
                    return result;
                }

                currNs = currNs.getNamespace();
            }
        }

        return null;
    }

    @Override
    public Property lookupProperty(Classifier owner, String name) {
        Property result = super.lookupProperty(owner, name);

        if (result == null) {
            result = lookupNonNavigableProperty(owner, name);
        }

        return result;
    }

    /**
     * Looks up a non-navigable property owned by an association on behalf of
     * the specified <code>owner</code> classifier (which is at that end).
     * 
     * @param owner
     *            a classifier in the context of which the property is used
     * @param name
     *            the property name to look up
     * 
     * @return the non-navigable property, or <code>null</code> if it cannot
     *         be found
     */
    private Property lookupNonNavigableProperty(Classifier owner, String name) {
        Property result = null;

        if (owner == null) {
            Variable<Classifier, Parameter> vdcl = lookupImplicitSourceForProperty(name);

            if (vdcl == null) {
                return null;
            }

            owner = vdcl.getType();
        }

        EList<Association> associations = owner.getAssociations();

        for (Association next : associations) {
            result = next.getMemberEnd(name, null);

            if (result != null) {
                break;
            }
        }

        return result;
    }

    // implements the inherited specification
    public List<State> getStates(Classifier owner, List<String> pathPrefix) {
        EList<State> result = new BasicEList.FastCompare<State>();

        collectStates(owner, pathPrefix, result);

        // search supertypes
        for (Classifier general : owner.allParents()) {
            collectStates(general, pathPrefix, result);
        }

        // now, filter out redefinitions, in case our prefix match found
        // states that are redefined by other matches (as an instance of the
        // owner type cannot be in a state that is redefined by a more
        // specific state)
        Set<State> redefinitions = new java.util.HashSet<State>();
        for (State s : result) {
            State redef = s.getRedefinedState();

            while (redef != null) {
                redefinitions.add(redef);
                redef = redef.getRedefinedState();
            }
        }

        result.removeAll(redefinitions);

        return result;
    }

    /**
     * Finds all states in the specified owner type that match the given path
     * name prefix and add them to the accumulator list.
     * 
     * @param owner
     *            the owner type
     * @param pathPrefix
     *            partial qualified name, specifying the parent of the states to
     *            be collected
     * @param states
     *            a list of states directly owned by the namespace indicated by
     *            path prefix, within the owner type
     */
    private void collectStates(Classifier owner, List<String> pathPrefix,
            List<State> states) {
        if (owner instanceof BehavioredClassifier) {
            List<Behavior> behaviors = ((BehavioredClassifier) owner)
                .getOwnedBehaviors();

            for (Behavior b : behaviors) {
                if (b instanceof StateMachine) {
                    collectStates((StateMachine) b, pathPrefix, states);
                }
            }
        }
    }

    private void collectStates(StateMachine machine, List<String> pathPrefix,
            List<State> states) {
        if (pathPrefix.isEmpty()) {
            for (Region r : machine.getRegions()) {
                collectStates(r, pathPrefix, states);
            }
        } else {
            String firstName = pathPrefix.get(0);

            if (UMLForeignMethods.isNamed(firstName, machine)) {
                // we are allowed to qualify the states by machine name
                pathPrefix = pathPrefix.subList(1, pathPrefix.size());
            }

            for (Region r : machine.getRegions()) {
                collectStates(r, pathPrefix, states);
            }
        }
    }

    private void collectStates(Region region, List<String> pathPrefix,
            List<State> states) {
        if (pathPrefix.isEmpty()) {
            // terminus of the recursion: get all the states in this region
            for (Vertex v : region.getSubvertices()) {
                if (v instanceof State) {
                    states.add((State) v);
                }
            }
        } else {
            String firstName = pathPrefix.get(0);

            Vertex v = UMLForeignMethods.getSubvertex(region, firstName);
            if (v instanceof State) {
                State state = (State) v;

                if (state.isComposite()) {
                    // recursively search the regions of this composite state
                    pathPrefix = pathPrefix.subList(1, pathPrefix.size());

                    for (Region r : state.getRegions()) {
                        collectStates(r, pathPrefix, states);
                    }
                }
            }
        }
    }

    // implements the inherited specification
    public Property defineAttribute(Classifier owner,
            Variable<Classifier, Parameter> variable, Constraint constraint) {

        Property result;

        String name = variable.getName();
        Classifier type = variable.getType();

        result = UMLFactory.eINSTANCE.createProperty();

        result.addKeyword(UMLReflection.OCL_HELPER);

        result.setName(name);
        result.setType(type);

        owner.getOwnedRules().add(constraint);

        addProperty(owner, result);

        return result;
    }

    // implements the inherited specification
    public Operation defineOperation(Classifier owner, String name,
            Classifier type, List<Variable<Classifier, Parameter>> params,
            Constraint constraint) {
        Operation result = UMLFactory.eINSTANCE.createOperation();

        result.addKeyword(UMLReflection.OCL_HELPER);

        result.setName(name);
        result.setType(type == null ? getOCLStandardLibrary().getOclVoid()
            : type);
        result.setIsQuery(true); // OCL can only define queries

        for (Variable<Classifier, Parameter> next : params) {
            Parameter param = UMLFactory.eINSTANCE.createParameter();
            param.setName(next.getName());
            param.setType(next.getType() == null ? getOCLStandardLibrary()
                .getOclVoid()
                : next.getType());

            param.setDirection(ParameterDirectionKind.IN_LITERAL);
            param.setEffect(ParameterEffectKind.READ_LITERAL);

            result.getOwnedParameters().add(param);
        }

        owner.getOwnedRules().add(constraint);

        addOperation(owner, result);

        return result;
    }

    // implements the inherited specification
    public void undefine(Object feature) {
        Constraint definition = getDefinition(feature);

        if (definition == null) {
            throw new IllegalArgumentException(
                "not an additional feature: " + feature); //$NON-NLS-1$
        }

        EcoreUtil.remove((EObject) feature);
        EcoreUtil.remove(definition);

        definition.getConstrainedElements().clear();
    }

    // implements the inherited specification
    public Constraint getDefinition(Object feature) {
        Constraint result = null;
        Feature umlFeature = (Feature) feature;

        Classifier owner = (Classifier) umlFeature.getOwner();

        if (owner instanceof Class) {
            Classifier shadowed = ((TypeResolverImpl) getTypeResolver())
                .getShadowedClassifier(owner);

            if (shadowed != null) {
                owner = shadowed;
            }
        }

        if (owner != null) {
            for (Constraint ct : owner.getOwnedRules()) {
                if (ct.getKeywords().contains(UMLReflection.DEFINITION)
                    && ct.getConstrainedElements().contains(umlFeature)) {
                    result = ct;
                    break;
                }
            }
        }

        return result;
    }

    // implements the inherited specification
    public boolean isInPostcondition(
            org.eclipse.ocl.expressions.OCLExpression<Classifier> exp) {

        Constraint constraint = null;
        EObject parent = exp;
        while (parent != null) {
            if (parent instanceof Constraint) {
                constraint = (Constraint) parent;
                break;
            }

            parent = parent.eContainer();
        }

        return (constraint != null)
            && (!constraint.getKeywords().isEmpty())
            && UMLReflection.POSTCONDITION.equals(constraint.getKeywords().get(
                0));
    }
}
